/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <openssl/ssl.h>
#include <openssl/err.h>

#include <hulautil.h>
#include <msgapi.h>
#include <nmap.h>

#include "modwebp.h"

#if !defined(LIBC)	/* fixme - NetWare requirement */
#if defined(MODWEB_DEBUG_BUILD)
#define	ModuleDebugInit()		MWModuleScreen=XplCreateScreen("MW Module Debug Screen", 0)
#else
#define	ModuleDebugInit()		MWModuleScreen=XplCreateScreen("System Console", 0);
#endif
#else
#define	ModuleDebugInit()
#endif

unsigned char	*ModWebMVersion = "$Revision: 1.2 $";

int						TGid;
int						MWModuleScreen;

unsigned long			TModuleCount;
unsigned long			SModuleCount;
unsigned long			CModuleCount;

TemplateModuleStruct	*TModules	= NULL;
StreamModuleStruct	*SModules	= NULL;
ComposeModuleStruct	*CModules	= NULL;

typedef struct {
	unsigned char			*Name;
	XplPluginHandle		Handle;
	MWModuleShutdownFunc	Shutdown;
} ModuleStruct;

ModuleStruct			*Modules		= NULL;
unsigned long			ModuleCount	= 0;

#if defined(SOLARIS) || defined(LINUX) || defined(WIN32) || defined(S390RH)
	MWAPIStruct	MWAPI;
#else
	unsigned long	*MWAPI=NULL;
#endif

static void
AddTModule(TemplateModuleStruct *Functions)
{
	TModules=MemRealloc(TModules, (TModuleCount+1) * sizeof(TemplateModuleStruct));
	if (!TModules) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, ((TModuleCount+1) * sizeof(TemplateModuleStruct)), __LINE__, NULL, 0);
		return;
	}

	memcpy(&(TModules[TModuleCount]), Functions, sizeof(TemplateModuleStruct));
	TModuleCount++;
	ModuleDebug("Added T-module\n");
}

static void
FreeTModules(void)
{
	ModuleDebug("Unloading T-modules\n");

	if (TModules) {
		MemFree(TModules);
	}

	TModules=NULL;
	TModuleCount=0;
}


static void
AddSModule(TemplateModuleStruct *Functions)
{
	SModules=MemRealloc(SModules, (SModuleCount+1) * sizeof(StreamModuleStruct));
	if (!SModules) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, ((SModuleCount+1) * sizeof(StreamModuleStruct)), __LINE__, NULL, 0);
		return;
	}

	memcpy(&(SModules[SModuleCount]), Functions, sizeof(StreamModuleStruct));
	SModuleCount++;
	ModuleDebug("Added S-module\n");
}

static void
FreeSModules(void)
{
	ModuleDebug("Unloading S-modules\n");

	if (SModules) {
		MemFree(SModules);
	}

	SModules=NULL;
	SModuleCount=0;
}


static void
AddCModule(TemplateModuleStruct *Functions)
{
	CModules=MemRealloc(CModules, (CModuleCount+1) * sizeof(ComposeModuleStruct));
	if (!CModules) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, ((CModuleCount+1) * sizeof(ComposeModuleStruct)), __LINE__, NULL, 0);
		return;
	}

	memcpy(&(CModules[CModuleCount]), Functions, sizeof(ComposeModuleStruct));
	CModuleCount++;
	ModuleDebug("Added C-module\n");
}

static void
FreeCModules(void)
{
	ModuleDebug("Unloading C-modules\n");

	if (CModules) {
		MemFree(CModules);
	}

	CModules=NULL;
	CModuleCount=0;
}


void
AddModule(unsigned char *Name, XplPluginHandle Handle, MWModuleShutdownFunc Shutdown)
{
	Modules=MemRealloc(Modules, (ModuleCount+1) * sizeof(ModuleStruct));
	if (!Modules) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, ((ModuleCount+1) * sizeof(ModuleStruct)), __LINE__, NULL, 0);
		return;
	}

	Modules[ModuleCount].Handle=Handle;
	Modules[ModuleCount].Shutdown=Shutdown;
	Modules[ModuleCount].Name=MemStrdup(Name);
	if (!Modules[ModuleCount].Name) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OUT_OF_MEMORY, LOG_CRITICAL, 0, __FILE__, NULL, (strlen(Name)), __LINE__, NULL, 0);
		return;
	}
	ModuleCount++;
	ModuleDebug("Added module %d: %s\n", (int)ModuleCount, Name);
}

void
FreeModules(void)
{
	unsigned int i;

	ModuleDebug("Unloading all modules\n");

	FreeTModules();
	FreeSModules();
	FreeCModules();

	for (i=0; i<ModuleCount; i++) {
		ModuleDebug("Calling module %s shutdown code\n", Modules[i].Name);
		Modules[i].Shutdown();
		ModuleDebug("Unloading module %s, handle:%x\n", Modules[i].Name, (unsigned int)Modules[i].Handle);
		XplUnloadDLL(Modules[i].Name, Modules[i].Handle);
		MemFree(Modules[i].Name);
	}

	if (Modules) {
		MemFree(Modules);
	}

	Modules=NULL;
	ModuleCount=0;
}

static BOOL
LoadModwebModule(unsigned char *ModulePath, unsigned char *ModuleName, unsigned char *Arguments)
{
	unsigned char			Path[XPL_MAX_PATH+1];
	unsigned char			Name[XPL_MAX_PATH+1];
	XplPluginHandle		ModuleHandle;
	MWModuleInitFunc		ModuleInitFunc;
	MWModuleShutdownFunc	ModuleShutdownFunc;
	unsigned char			*ptr;

TryAgain:                                                                                                                           
   if ((strlen(ModuleName)<5) || (MWQuickNCmp(ModuleName+strlen(ModuleName)-strlen(MODULE_EXTENSION), MODULE_EXTENSION, strlen(MODULE_EXTENSION))==FALSE)) {                                                                                                            
      ptr=strrchr(ModuleName, '.');                                                                                                 
      if (ptr && MWQuickNCmp(ptr,".NLM", 4)) {                                                                                      
         strcpy(ptr, MODULE_EXTENSION);                                                                                             
         ptr=ModuleName;                                                                                                            
         while (*ptr) {                                                                                                             
            *ptr=tolower(*ptr);                                                                                                     
            ptr++;                                                                                                                  
         }                                                                                                                          
         goto TryAgain;                                                                                                             
      }                                                                                                                             
      ModuleDebug("Ignoring file %s, module extension must be %s\n", ModuleName, MODULE_EXTENSION);                                 
      return(FALSE);                                                                                                                
   }                                                                                                                                

	/* Load the module, then call it's init function */
#undef MODULE_MEMORY_DEBUG
#ifdef MODULE_MEMORY_DEBUG
	snprintf(Path, sizeof(Path), "-m %s\\%s\r", ModulePath, ModuleName);
#if !defined(LIBC)	/* fixme - NetWare requirement */
	SetCurrentScreen(XplCreateScreen("System Console", 0));
#endif
	ptr=Path;
	while (*ptr) {
		ungetch(*ptr);
		ptr++;
	}
	XplDelay(1000);
	snprintf(Path, sizeof(Path), "%s", ModuleName);

	ModuleHandle=(XplPluginHandle)FindNLMHandle(Path);
#else
	/* We want to make sure that any output from the loading is sent to the System Console */
#if !defined(LIBC)	/* fixme - NetWare requirement */
	XplSetCurrentScreen(XplCreateScreen("System Console", 0));
#endif
	
	snprintf(Path, sizeof(Path), "%s/%s", ModulePath, ModuleName);
	ModuleHandle=XplLoadDLL(Path);
	if (!ModuleHandle) {
		snprintf(Path, sizeof(Path), "%s/lib%s", ModulePath, ModuleName);

		ModuleHandle = XplLoadDLL(Path);
	}

#endif
	if (ModuleHandle) {
		XplConsolePrintf("Loaded module %s\n", Path);
	} else {
		XplConsolePrintf("Could not load module %s, errno:%d %s\n", Path, errno, strerror(errno));
		return(FALSE);
	}

	/* Give the module a chance to initialize */
	HulaStrNCpy(Name, ModuleName, sizeof(Name) - strlen("Init"));
	Name[strlen(Name)-strlen(MODULE_EXTENSION)]='\0';
	ptr=Name;
	while (*ptr) {
		*ptr=toupper(*ptr);
		ptr++;
	}

	strcat(Name, "Init");
	ModuleInitFunc=(MWModuleInitFunc)XplGetDLLFunction(ModuleName, Name, ModuleHandle);

	if (!ModuleInitFunc) {
		ModuleDebug("Could not load module %s, missing Init function '%s'\n", ModuleName, Name);
		return(FALSE);
	}

	/* Grab the shutdown function */
	HulaStrNCpy(Name, ModuleName, sizeof(Name) - strlen("Shutdown"));
	Name[strlen(Name)-strlen(MODULE_EXTENSION)]='\0';
	ptr=Name;
	while (*ptr) {
		*ptr=toupper(*ptr);
		ptr++;
	}

	strcat(Name, "Shutdown");
	ModuleShutdownFunc=(MWModuleShutdownFunc)XplGetDLLFunction(ModuleName, Name, ModuleHandle);
	if (!ModuleShutdownFunc) {
		ModuleDebug("Could not load module %s, missing Shutdown function '%s'\n", ModuleName, Name);
		return(FALSE);
	}

	if (!ModuleInitFunc(&MWAPI)) {
		ModuleDebug("Could not load module %s, Init returned FALSE\n", ModuleName);
		return(FALSE);
	}

	AddModule(ModuleName, ModuleHandle, ModuleShutdownFunc);

	return(TRUE);
}


BOOL
LoadModules(unsigned char *ModulePath, BOOL ScanDir)
{
	MDBValueStruct		*Modules;
	MDBValueStruct		*Modulename;
	MDBValueStruct		*Disabled;
	unsigned int					i;
	
	ModuleDebugInit();

	MWAPISetup;

	TGid=XplGetThreadGroupID();

	Modules = MDBCreateValueStruct(ModWebDirectoryHandle, MsgGetServerDN(NULL));
	Modulename = MDBCreateValueStruct(ModWebDirectoryHandle, MsgGetServerDN(NULL));
	Disabled = MDBCreateValueStruct(ModWebDirectoryHandle, MsgGetServerDN(NULL));

	if (MDBEnumerateObjects(MSGSRV_AGENT_MODWEB, NULL, NULL, Modules)<1) {
		LoggerEvent(LogHandle, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NOT_CONFIGURED, LOG_INFO, 0, "Modules", NULL, 0, 0, NULL, 0);
		ModuleDebug("No modules configured in DS\n");
		MDBDestroyValueStruct(Modules);
		MDBDestroyValueStruct(Modulename);
		MDBDestroyValueStruct(Disabled);
		return(FALSE);
	}

	ModuleDebug("%d modules configured\n", (int)Modules->Used);

	/* This reads all module names */
	for (i=0; i<Modules->Used; i++) {
		if (MDBRead(Modules->Value[i], MSGSRV_A_MODULE_NAME, Modulename)>0) {
			if (MDBRead(Modules->Value[i], MSGSRV_A_DISABLED, Disabled)==0) {
				MDBAddValue("0", Disabled);
			}
		}
	}

	for (i=0; i<Modulename->Used; i++) {
		if (Disabled->Value[i][0]!='1') {
			ModuleDebug("Loading %s/%s\n", ModulePath, Modulename->Value[i]);
			LoadModwebModule(ModulePath, Modulename->Value[i], NULL);
		}
	}

	if (ScanDir) {
		XplDir			*Directory;
		XplDir			*DirectoryEntry;
		unsigned char	Path[XPL_MAX_PATH+1];

		snprintf(Path, sizeof(Path), "%s", ModulePath);
		if ((Directory=XplOpenDir(Path))!=NULL) {
			while ((DirectoryEntry=XplReadDir(Directory))!=NULL) {
				for (i=0; i<Modules->Used; i++) {
					if (MWQuickCmp(DirectoryEntry->d_nameDOS, Modulename->Value[i])) {
						break;
					}
				}

				/* If we found the same module in the DS module list, i will be less than Used */
				if (i==Modules->Used) {
					LoadModwebModule(ModulePath, DirectoryEntry->d_nameDOS, NULL);
				}
			}
			XplCloseDir(Directory);
		}
	}

	MDBDestroyValueStruct(Modules);
	MDBDestroyValueStruct(Modulename);
	MDBDestroyValueStruct(Disabled);
	return(TRUE);
}


BOOL
MWRegisterModule(ModuleRegisterStruct *Register)
{
	int	OldTGid;

	OldTGid=XplGetThreadGroupID();
	XplSetThreadGroupID(TGid);

	ModuleDebug("Someone is trying to register a %d module\n", (int)Register->ModuleType);

	switch(Register->ModuleType) {
		case MODULE_TEMPLATE: {
			AddTModule((TemplateModuleStruct *)&(Register->Module.Template));
			break;
		}

		case MODULE_MIME: {
			AddSModule((TemplateModuleStruct *)&(Register->Module.Stream));
			break;
		}

		case MODULE_SEND: {
			AddCModule((TemplateModuleStruct *)&(Register->Module.Send));
			break;
		}
	}

	XplSetThreadGroupID(OldTGid);
	return(TRUE);
}
unsigned int NamedURLIDs[NM_MAX_NAMED_URLIDS] = { 0 } ; /* initialized to 0 */

/* This function registers URLID for given NamedID */
/* returns 0 on success and -1 on failure */
BOOL MWRegisterNamedURLs( NAMED_URL_ID *IDList, int IDCount)
{
	int i,j;

	/* for each element in IDList */
	for (i = 0; i < IDCount; i++) { 
		
		/* Make sure ID falls within range of Named IDs */
		/* and that slot is not already filled and that ID is sane */
		if ( (IDList[i].NamedID < NM_MAX_NAMED_URLIDS) && 
		     IDList[i].NamedID && !NamedURLIDs[i]) {

			NamedURLIDs[IDList[i].NamedID] = IDList[i].URLID;

		} else {
			/* bad news the ID is not valid */

			/* retore to original state */
			for (j = 0; j < i; j++) {
				NamedURLIDs[IDList[j].NamedID] = 0;
			}


			return(FALSE);

		}

	}

	return(TRUE);

}

