/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>

#include <xpl.h>
#include <logger.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>

#include "nmapdp.h"
#include "sql-lock.h"

#include <libical2.h>

#define MAX_THREAD_LOAD 50
#define PROTOCOL_VERSION "1.4"
#define DETAILED_HELP
#define PUSHCLIENTALLOC 20
#define MIME_STRUCT_ALLOC 20
#define MAX_HEADER 10240

static void SignalHandler(int sigtype);

#define QUEUE_WORK_TO_DO(c, id, r) \
        { \
            XplWaitOnLocalSemaphore(NMAP.client.semaphore); \
            if (XplSafeRead(NMAP.client.worker.idle)) { \
                (c)->queue.previous = NULL; \
                if (((c)->queue.next = NMAP.client.worker.head) != NULL) { \
                    (c)->queue.next->queue.previous = (c); \
                } else { \
                    NMAP.client.worker.tail = (c); \
                } \
                NMAP.client.worker.head = (c); \
                (r) = 0; \
            } else { \
                XplSafeIncrement(NMAP.client.worker.active); \
                XplSignalBlock(); \
                XplBeginThread(&(id), HandleConnection, STACKSPACE_Q, XplSafeRead(NMAP.client.worker.active), (r)); \
                XplSignalHandler(SignalHandler); \
                if (!(r)) { \
                    (c)->queue.previous = NULL; \
                    if (((c)->queue.next = NMAP.client.worker.head) != NULL) { \
                        (c)->queue.next->queue.previous = (c); \
                    } else { \
                        NMAP.client.worker.tail = (c); \
                    } \
                    NMAP.client.worker.head = (c); \
                } else { \
                    XplSafeDecrement(NMAP.client.worker.active); \
                    (r) = -1; \
                } \
            } \
            XplSignalLocalSemaphore(NMAP.client.semaphore); \
        }

NMAPGlobals NMAP;
unsigned char *MonthsOfTheYear[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" };

static ProtocolCommand NMAPProtocolCommands[] = {
    { NMAP_ADBK_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_ADBK_COMMAND) - 1, NmapCommandAdbk, NULL, NULL }, 
    { NMAP_AFLG_COMMAND, NMAP_AFLG_HELP, sizeof(NMAP_AFLG_COMMAND) - 1, NmapCommandAflg, NULL, NULL }, 
    { NMAP_AINFO_COMMAND, NMAP_AINFO_HELP, sizeof(NMAP_AINFO_COMMAND) - 1, NmapCommandAinfo, NULL, NULL }, 
    { NMAP_AUTH_COMMAND, NMAP_AUTH_HELP, sizeof(NMAP_AUTH_COMMAND) - 1, NmapCommandAuth, NULL, NULL }, 
    { NMAP_BODY_COMMAND, NMAP_BODY_HELP, sizeof(NMAP_BODY_COMMAND) - 1, NmapCommandBody, NULL, NULL }, 
    { NMAP_BRAW_COMMAND, NMAP_BRAW_HELP, sizeof(NMAP_BRAW_COMMAND) - 1, NmapCommandBraw, NULL, NULL }, 
    { NMAP_CAL_COMMAND, NMAP_CAL_HELP, sizeof(NMAP_CAL_COMMAND) - 1, NmapCommandCal, NULL, NULL }, 
    { NMAP_CAPA_COMMAND, NMAP_CAPA_HELP, sizeof(NMAP_CAPA_COMMAND) - 1, NmapCommandCapa, NULL, NULL }, 
    { NMAP_CHECK_COMMAND, NMAP_CHECK_HELP, sizeof(NMAP_CHECK_COMMAND) - 1, NmapCommandCheck, NULL, NULL }, 
    { NMAP_CLR_COMMAND, NMAP_CLR_HELP, sizeof(NMAP_CLR_COMMAND) - 1, NmapCommandClr, NULL, NULL }, 
    { NMAP_CONF_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CONF_COMMAND) - 1, NmapCommandConf, NULL, NULL }, 
    { NMAP_COPY_COMMAND, NMAP_COPY_HELP, sizeof(NMAP_COPY_COMMAND) - 1, NmapCommandCopy, NULL, NULL }, 
    { NMAP_CREA_COMMAND, NMAP_CREA_HELP, sizeof(NMAP_CREA_COMMAND) - 1, NmapCommandCrea, NULL, NULL }, 
    { NMAP_CSOPEN_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSOPEN_COMMAND) - 1, NmapCommandCsopen, NULL, NULL }, 
    { NMAP_CSORG_COMMAND, NMAP_CSORG_HELP, sizeof(NMAP_CSORG_COMMAND) - 1, NmapCommandCsorg, NULL, NULL }, 
    { NMAP_CSINFO_COMMAND, NMAP_CSINFO_HELP, sizeof(NMAP_CSINFO_COMMAND) - 1, NmapCommandCsinfo, NULL, NULL }, 
    { NMAP_CSFILT_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSFILT_COMMAND) - 1, NmapCommandCsfilt, NULL, NULL }, 
    { NMAP_CSFIND_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSFIND_COMMAND) - 1, NmapCommandCsfind, NULL, NULL }, 
    { NMAP_CSGFLG_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSGFLG_COMMAND) - 1, NmapCommandCsgflg, NULL, NULL }, 
    { NMAP_CSLIST_COMMAND, NMAP_CSLIST_HELP, sizeof(NMAP_CSLIST_COMMAND) - 1, NmapCommandCslist, NULL, NULL }, 
    { NMAP_CSCREA_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSCREA_COMMAND) - 1, NmapCommandCscrea, NULL, NULL }, 
    { NMAP_CSCHECK_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSCHECK_COMMAND) - 1, NmapCommandCscheck, NULL, NULL }, 
    { NMAP_CSCOMP_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSCOMP_COMMAND) - 1, NmapCommandCscomp, NULL, NULL }, 
    { NMAP_CSCOPY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSCOPY_COMMAND) - 1, NmapCommandCscopy, NULL, NULL }, 
    { NMAP_CSDELE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSDELE_COMMAND) - 1, NmapCommandCsdele, NULL, NULL }, 
    { NMAP_CSDFLG_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSDFLG_COMMAND) - 1, NmapCommandCsdflg, NULL, NULL }, 
    { NMAP_CSAFLG_COMMAND, NMAP_CSAFLG_HELP, sizeof(NMAP_CSAFLG_COMMAND) - 1, NmapCommandCsaflg, NULL, NULL }, 
    { NMAP_CSATND_COMMAND, NMAP_CSATND_HELP, sizeof(NMAP_CSATND_COMMAND) - 1, NmapCommandCsatnd, NULL, NULL }, 
    { NMAP_CSUPDA_COMMAND, NMAP_CSUPDA_HELP, sizeof(NMAP_CSUPDA_COMMAND) - 1, NmapCommandCsupda, NULL, NULL }, 
    { NMAP_CSUNSUBS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSUNSUBS_COMMAND) - 1, NmapCommandCsunsubs, NULL, NULL }, 
    { NMAP_CSRMOV_COMMAND, NMAP_CSRMOV_HELP, sizeof(NMAP_CSRMOV_COMMAND) - 1, NmapCommandCsrmov, NULL, NULL }, 
    { NMAP_CSRNAM_COMMAND, NMAP_CSRNAM_HELP, sizeof(NMAP_CSRNAM_COMMAND) - 1, NmapCommandCsrnam, NULL, NULL }, 
    { NMAP_CSSTRAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSTRAW_COMMAND) - 1, NmapCommandCsstraw, NULL, NULL }, 
    { NMAP_CSSTOR_COMMAND, NMAP_CSSTOR_HELP, sizeof(NMAP_CSSTOR_COMMAND) - 1, NmapCommandCsstor, NULL, NULL }, 
    { NMAP_CSSTAT_COMMAND, NMAP_CSSTAT_HELP, sizeof(NMAP_CSSTAT_COMMAND) - 1, NmapCommandCsstat, NULL, NULL }, 
    { NMAP_CSSHOW_COMMAND, NMAP_CSSHOW_HELP, sizeof(NMAP_CSSHOW_COMMAND) - 1, NmapCommandCsshow, NULL, NULL }, 
    { NMAP_CSSHOWSUB_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSHOWSUB_COMMAND) - 1, NmapCommandCsshowsub, NULL, NULL }, 
    { NMAP_CSSUBS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSUBS_COMMAND) - 1, NmapCommandCssubs, NULL, NULL }, 
    { NMAP_CSSALV_COMMAND, NMAP_CSSALV_HELP, sizeof(NMAP_CSSALV_COMMAND) - 1, NmapCommandCssalv, NULL, NULL }, 
    { NMAP_CSSFLG_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSFLG_COMMAND) - 1, NmapCommandCssflg, NULL, NULL }, 
    { NMAP_CSSINFO_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSINFO_COMMAND) - 1, NmapCommandCssinfo, NULL, NULL }, 
#if defined(NMAP_SQL_CALENDARS)
    { NMAP_CSSQL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSSQL_COMMAND) - 1, NmapCommandCssql, NULL, NULL }, 
#if defined(DEBUG)
    { "CSFILL", NMAP_HELP_NOT_DEFINED, sizeof("CSFILL") - 1, NmapCommandCsfill, NULL, NULL }, 
#endif
#endif
    { NMAP_CSPURG_COMMAND, NMAP_CSPURG_HELP, sizeof(NMAP_CSPURG_COMMAND) - 1, NmapCommandCspurg, NULL, NULL }, 
    { NMAP_CSPRGM_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_CSPRGM_COMMAND) - 1, NmapCommandCspurg, NULL, NULL }, 
    { NMAP_CSPRGV_COMMAND, NMAP_CSPRGV_HELP, sizeof(NMAP_CSPRGV_COMMAND) - 1, NmapCommandCspurg, NULL, NULL }, 
    { NMAP_DELE_COMMAND, NMAP_DELE_HELP, sizeof(NMAP_DELE_COMMAND) - 1, NmapCommandDele, NULL, NULL }, 
    { NMAP_DFLG_COMMAND, NMAP_DFLG_HELP, sizeof(NMAP_DFLG_COMMAND) - 1, NmapCommandDflg, NULL, NULL }, 
    { NMAP_ERRO_COMMAND, NMAP_ERRO_HELP, sizeof(NMAP_ERRO_COMMAND) - 1, NmapCommandErro, NULL, NULL }, 
    { NMAP_FDUMP_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_FDUMP_COMMAND) - 1, NmapCommandFdump, NULL, NULL }, 
    { NMAP_FEAT_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_FEAT_COMMAND) - 1, NmapCommandFeat, NULL, NULL }, 
    { NMAP_FLAG_COMMAND, NMAP_FLAG_HELP, sizeof(NMAP_FLAG_COMMAND) - 1, NmapCommandFlag, NULL, NULL }, 
    { NMAP_GFLG_COMMAND, NMAP_GFLG_HELP, sizeof(NMAP_GFLG_COMMAND) - 1, NmapCommandGflg, NULL, NULL }, 
    { NMAP_GINFO_COMMAND, NMAP_GINFO_HELP, sizeof(NMAP_GINFO_COMMAND) - 1, NmapCommandGinfo, NULL, NULL }, 
    { NMAP_GUID_COMMAND, NMAP_GUID_COMMAND_HELP, sizeof(NMAP_GUID_COMMAND) - 1, NmapCommandGuid, NULL, NULL }, 
    { NMAP_HEAD_COMMAND, NMAP_HEAD_HELP, sizeof(NMAP_HEAD_COMMAND) - 1, NmapCommandHead, NULL, NULL }, 
    { NMAP_HELP_COMMAND, NMAP_HELP_HELP, sizeof(NMAP_HELP_COMMAND) - 1, NmapCommandHelp, NULL, NULL }, 
    { NMAP_INFO_COMMAND, NMAP_INFO_HELP, sizeof(NMAP_INFO_COMMAND) - 1, NmapCommandInfo, NULL, NULL }, 
    { NMAP_LIST_COMMAND, NMAP_LIST_HELP, sizeof(NMAP_LIST_COMMAND) - 1, NmapCommandList, NULL, NULL }, 
    { NMAP_MBOX_COMMAND, NMAP_MBOX_HELP, sizeof(NMAP_MBOX_COMMAND) - 1, NmapCommandMbox, NULL, NULL }, 
    { NMAP_MIME_COMMAND, NMAP_MIME_HELP, sizeof(NMAP_MIME_COMMAND) - 1, NmapCommandMime, NULL, NULL }, 
    { NMAP_MIMED_COMMAND, NMAP_MIMED_HELP, sizeof(NMAP_MIMED_COMMAND) - 1, NmapCommandMimeD, NULL, NULL }, 
    { NMAP_NAMESPACE_COMMAND, NMAP_NAMESPACE_HELP, sizeof(NMAP_NAMESPACE_COMMAND) - 1, NmapCommandNamespace, NULL, NULL }, 
    { NMAP_NOOP_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_NOOP_COMMAND) - 1, NmapCommandNoop, NULL, NULL }, 
    { NMAP_NVER_COMMAND, NMAP_NVER_HELP, sizeof(NMAP_NVER_COMMAND) - 1, NmapCommandNver, NULL, NULL }, 
    { NMAP_OPER_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_OPER_COMMAND) - 1, NmapCommandOper, NULL, NULL }, 
    { NMAP_PURG_COMMAND, NMAP_PURG_HELP, sizeof(NMAP_PURG_COMMAND) - 1, NmapCommandPurg, NULL, NULL }, 
    { NMAP_PRGM_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_PRGM_COMMAND) - 1, NmapCommandPurg, NULL, NULL }, 
    { NMAP_PRGV_COMMAND, NMAP_PRGV_HELP, sizeof(NMAP_PRGV_COMMAND) - 1, NmapCommandPurg, NULL, NULL }, 
    { NMAP_PASS_COMMAND, NMAP_PASS_HELP, sizeof(NMAP_PASS_COMMAND) - 1, NmapCommandPass, NULL, NULL }, 
    { NMAP_PROXY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_PROXY_COMMAND) - 1, NmapCommandProxy, NULL, NULL }, 
    { NMAP_QADDM_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QADDM_COMMAND) - 1, NmapCommandQaddm, NULL, NULL }, 
    { NMAP_QADDQ_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QADDQ_COMMAND) - 1, NmapCommandQaddq, NULL, NULL }, 
    { NMAP_QABRT_COMMAND, NMAP_QABRT_HELP, sizeof(NMAP_QABRT_COMMAND) - 1, NmapCommandQabrt, NULL, NULL }, 
    { NMAP_QBODY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QBODY_COMMAND) - 1, NmapCommandQbody, NULL, NULL }, 
    { NMAP_QBRAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QBRAW_COMMAND) - 1, NmapCommandQbraw, NULL, NULL }, 
    { NMAP_QCOPY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QCOPY_COMMAND) - 1, NmapCommandQcopy, NULL, NULL }, 
    { NMAP_QCREA_COMMAND, NMAP_QCREA_HELP, sizeof(NMAP_QCREA_COMMAND) - 1, NmapCommandQcrea, NULL, NULL }, 
    { NMAP_QDELE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QDELE_COMMAND) - 1, NmapCommandQdele, NULL, NULL }, 
    { NMAP_QDONE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QDONE_COMMAND) - 1, NmapCommandQdone, NULL, NULL }, 
    { NMAP_QDSPC_COMMAND, NMAP_QDSPC_HELP, sizeof(NMAP_QDSPC_COMMAND) - 1, NmapCommandQdspc, NULL, NULL }, 
    { NMAP_QEND_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QEND_COMMAND) - 1, NmapCommandQend, NULL, NULL }, 
    { NMAP_QGREP_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QGREP_COMMAND) - 1, NmapCommandQgrep, NULL, NULL }, 
    { NMAP_QHEAD_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QHEAD_COMMAND) - 1, NmapCommandQhead, NULL, NULL }, 
    { NMAP_QINFO_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QINFO_COMMAND) - 1, NmapCommandQinfo, NULL, NULL }, 
    { NMAP_QMOD_FROM_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_FROM_COMMAND) - 1, NmapCommandQmodFrom, NULL, NULL }, 
    { NMAP_QMOD_FLAGS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_FLAGS_COMMAND) - 1, NmapCommandQmodFlags, NULL, NULL }, 
    { NMAP_QMOD_LOCAL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_LOCAL_COMMAND) - 1, NmapCommandQmodLocal, NULL, NULL }, 
    { NMAP_QMOD_MAILBOX_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_MAILBOX_COMMAND) - 1, NmapCommandQmodMailbox, NULL, NULL }, 
    { NMAP_QMOD_RAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_RAW_COMMAND) - 1, NmapCommandQmodRaw, NULL, NULL }, 
    { NMAP_QMOD_TO_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOD_TO_COMMAND) - 1, NmapCommandQmodTo, NULL, NULL }, 
    { NMAP_QMIME_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMIME_COMMAND) - 1, NmapCommandQmime, NULL, NULL }, 
    { NMAP_QMOVE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QMOVE_COMMAND) - 1, NmapCommandQmove, NULL, NULL }, 
    { NMAP_QRCP_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QRCP_COMMAND) - 1, NmapCommandQrcp, NULL, NULL }, 
    { NMAP_QRETR_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QRETR_COMMAND) - 1, NmapCommandQretr, NULL, NULL }, 
    { NMAP_QRTS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QRTS_COMMAND) - 1, NmapCommandQrts, NULL, NULL }, 
    { NMAP_QRUN_COMMAND, NMAP_QRUN_HELP, sizeof(NMAP_QRUN_COMMAND) - 1, NmapCommandQrun, NULL, NULL }, 
    { NMAP_QSQL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSQL_COMMAND) - 1, NmapCommandQsql, NULL, NULL }, 
    { NMAP_QSRCH_DOMAIN_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSRCH_DOMAIN_COMMAND) - 1, NmapCommandQsrchDomain, NULL, NULL }, 
    { NMAP_QSRCH_HEADER_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSRCH_HEADER_COMMAND) - 1, NmapCommandQsrchHeader, NULL, NULL }, 
    { NMAP_QSRCH_BODY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSRCH_BODY_COMMAND) - 1, NmapCommandQsrchBody, NULL, NULL }, 
    { NMAP_QSRCH_BRAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSRCH_BRAW_COMMAND) - 1, NmapCommandQsrchBraw, NULL, NULL }, 
    { NMAP_QSTOR_ADDRESS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_ADDRESS_COMMAND) - 1, NmapCommandQstorAddress, NULL, NULL }, 
    { NMAP_QSTOR_CAL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_CAL_COMMAND) - 1, NmapCommandQstorCal, NULL, NULL }, 
    { NMAP_QSTOR_FLAGS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_FLAGS_COMMAND) - 1, NmapCommandQstorFlags, NULL, NULL }, 
    { NMAP_QSTOR_FROM_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_FROM_COMMAND) - 1, NmapCommandQstorFrom, NULL, NULL }, 
    { NMAP_QSTOR_LOCAL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_LOCAL_COMMAND) - 1, NmapCommandQstorLocal, NULL, NULL }, 
    { NMAP_QSTOR_MESSAGE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_MESSAGE_COMMAND) - 1, NmapCommandQstorMessage, NULL, NULL }, 
    { NMAP_QSTOR_RAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_RAW_COMMAND) - 1, NmapCommandQstorRaw, NULL, NULL }, 
    { NMAP_QSTOR_TO_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_QSTOR_TO_COMMAND) - 1, NmapCommandQstorTo, NULL, NULL }, 
    { NMAP_QUIT_COMMAND, NMAP_QUIT_HELP, sizeof(NMAP_QUIT_COMMAND) - 1, NmapCommandQuit, NULL, NULL }, 
    { NMAP_QWAIT_COMMAND, NMAP_QWAIT_HELP, sizeof(NMAP_QWAIT_COMMAND) - 1, NmapCommandQwait, NULL, NULL }, 
    { NMAP_RNAM_COMMAND, NMAP_RNAM_HELP, sizeof(NMAP_RNAM_COMMAND) - 1, NmapCommandRnam, NULL, NULL }, 
    { NMAP_RSET_COMMAND, NMAP_RSET_HELP, sizeof(NMAP_RSET_COMMAND) - 1, NmapCommandRset, NULL, NULL }, 
    { NMAP_RIGHTS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_RIGHTS_COMMAND) - 1, NmapCommandRights, NULL, NULL }, 
    { NMAP_RMOV_COMMAND, NMAP_RMOV_HELP, sizeof(NMAP_RMOV_COMMAND) - 1, NmapCommandRmov, NULL, NULL }, 
    { NMAP_REPAIR_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_REPAIR_COMMAND) - 1, NmapCommandRepair, NULL, NULL }, 
    { NMAP_RESOURCE_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_RESOURCE_COMMAND) - 1, NmapCommandResource, NULL, NULL }, 
    { NMAP_SALV_COMMAND, NMAP_SALV_HELP, sizeof(NMAP_SALV_COMMAND) - 1, NmapCommandSalv, NULL, NULL }, 
    { NMAP_SEARCH_COMMAND, NMAP_SEARCH_HELP, sizeof(NMAP_SEARCH_COMMAND) - 1, NmapCommandSearch, NULL, NULL }, 
    { NMAP_SFLG_COMMAND, NMAP_SFLG_HELP, sizeof(NMAP_SFLG_COMMAND) - 1, NmapCommandSflg, NULL, NULL }, 
    { NMAP_SHOW_COMMAND, NMAP_SHOW_HELP, sizeof(NMAP_SHOW_COMMAND) - 1, NmapCommandShow, NULL, NULL }, 
    { NMAP_SHARE_ADBK_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SHARE_ADBK_COMMAND) - 1, NmapCommandShareAdbk, NULL, NULL }, 
    { NMAP_SHARE_CAL_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SHARE_CAL_COMMAND) - 1, NmapCommandShareCal, NULL, NULL }, 
    { NMAP_SHARE_MBOX_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SHARE_MBOX_COMMAND) - 1, NmapCommandShareMbox, NULL, NULL }, 
    { NMAP_SHOWPROXY_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SHOWPROXY_COMMAND) - 1, NmapCommandShowproxy, NULL, NULL }, 
    { NMAP_SHOWSH_COMMAND, NMAP_SHOWSH_HELP, sizeof(NMAP_SHOWSH_COMMAND) - 1, NmapCommandShowsh, NULL, NULL }, 
    { NMAP_SHOWSUB_COMMAND, NMAP_SHOWSUB_HELP, sizeof(NMAP_SHOWSUB_COMMAND) - 1, NmapCommandShowsub, NULL, NULL }, 
    { NMAP_SINFO_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SINFO_COMMAND) - 1, NmapCommandSinfo, NULL, NULL }, 
    { NMAP_SPACE_COMMAND, NMAP_SPACE_HELP, sizeof(NMAP_SPACE_COMMAND) - 1, NmapCommandSpace, NULL, NULL }, 
    { NMAP_STAT_COMMAND, NMAP_STAT_HELP, sizeof(NMAP_STAT_COMMAND) - 1, NmapCommandStat, NULL, NULL }, 
    { NMAP_STOR_COMMAND, NMAP_STOR_HELP, sizeof(NMAP_STOR_COMMAND) - 1, NmapCommandStor, NULL, NULL }, 
    { NMAP_STRAW_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_STRAW_COMMAND) - 1, NmapCommandStraw, NULL, NULL }, 
    { NMAP_SUBS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_SUBS_COMMAND) - 1, NmapCommandSubs, NULL, NULL }, 
    { NMAP_TLS_COMMAND, NMAP_TLS_HELP, sizeof(NMAP_TLS_COMMAND) - 1, NmapCommandTls, NULL, NULL }, 
    { NMAP_ULIST_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_ULIST_COMMAND) - 1, NmapCommandUlist, NULL, NULL }, 
    { NMAP_UNSUBS_COMMAND, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_UNSUBS_COMMAND) - 1, NmapCommandUnsubs, NULL, NULL }, 
    { NMAP_UPDA_COMMAND, NMAP_UPDA_HELP, sizeof(NMAP_UPDA_COMMAND) - 1, NmapCommandUpda, NULL, NULL }, 
    { NMAP_USER_COMMAND, NMAP_USER_HELP, sizeof(NMAP_USER_COMMAND) - 1, NmapCommandUser, NULL, NULL }, 
    { NMAP_VRFY_COMMAND, NMAP_VRFY_HELP, sizeof(NMAP_VRFY_COMMAND) - 1, NmapCommandVrfy, NULL, NULL }, 

    { NULL, NULL, 0, NULL, NULL, NULL }
};

/* fixme - improve protection on the NMAP.universalCounter!
   fixme - return MSG5001NOMEMORY on MDBValueStruct allocation failures!
   fixme - store user dn in client structure to eliminate subsequent lookups.
   fixme - uniform calendar locking
   fixme - uniform mailbox locking
   fixme - check sprintf and fwrite in place of fprintf
   fixme - cache user dn on entry and eliminate subsequent lookups for that dn as appro
   fixme - when restoring NetWare; reconfirm all handling of long/dos namespace issues
   fixme - mailbox repair
*/

static BOOL 
NMAPConnectionCloseCB(void *buffer, void *clientData)
{
    register NMAPClient *c = (NMAPClient *)buffer;

    if (c->conn) {
        ConnClose(c->conn, 1);
        ConnFree(c->conn);
        c->conn = NULL;
    }

    return(TRUE);
}

static BOOL 
NMAPConnectionAllocCB(void *buffer, void *clientData)
{
    register NMAPClient *c = (NMAPClient *)buffer;

    memset(c, 0, sizeof(NMAPClient));
    c->mailbox.newline = TRUE;

    return(TRUE);
}

void 
NMAPClientFree(NMAPClient *client)
{
    register NMAPClient *c = client;

    memset(c, 0, sizeof(NMAPClient));
    c->mailbox.newline = TRUE;

    MemPrivatePoolReturnEntry((void *)c);
}

BOOL
SetLongName(unsigned char *path, unsigned char *scratch)
{
    unsigned char *ptr;

    ptr = strrchr(path, '/');
    if (!ptr) {
        return(FALSE);
    }
    strcpy(scratch, ptr + 1);

    NWSetNameSpaceEntryName(path, NWOS2_NAME_SPACE, scratch);

    return(TRUE);
}

BOOL
SetLongPathName(unsigned char *path)
{
    unsigned char *ptr;
    unsigned char *start = path;

    ptr = strchr(start, ':');
    if (ptr) {
        start = ptr + 1;
        if (*start == '/') {
            start++;
        }
    }

    do {
        ptr = strchr(start, '/');
        if (ptr) {
            *ptr = '\0';
        }
        NWSetNameSpaceEntryName(path, NWOS2_NAME_SPACE, start);
        if (ptr) {
            *ptr = '/';
            start = ptr + 1;
        }
    } while (ptr);

    return(TRUE);
}

unsigned long
CalcUsedSpace(unsigned char *path, int pathLen, unsigned char *scratch)
{
    unsigned char *buf = NULL;
    unsigned long space;
    struct stat sb;
    XplDir *dir;
    XplDir *entry;

    if(!path || !scratch) {
        return(0L);
    }

    buf = MemMalloc(pathLen + 5);
    if(!buf) {
        return(0L);
    }

    /* fixme - what about calendars? */
    space = sprintf(buf, "%s", path);
    dir = XplOpenDir(buf);
    if(dir != NULL) {
        space = 0L;
        entry = XplReadDir(dir);
        while(entry != NULL) {
            if(entry->d_attr & XPL_A_SUBDIR && entry->d_name[0] != '.') {
                /* Got a directory */
                sprintf(scratch, "%s/%s", buf, entry->d_name);
                space += CalcUsedSpace(scratch, strlen(scratch), scratch);
            } else if(QuickCmp(entry->d_name + strlen(entry->d_name) - 4, ".box")) {
                /* Got a file */
                sprintf(scratch, "%s/%s", buf, entry->d_name);
                if(stat(scratch, &sb) == 0) {
                    space += sb.st_size;
                }
            }
            entry = XplReadDir(dir);
        }
        XplCloseDir(dir);
    }

    MemFree(buf);
    return(space);
}

void 
CollapseSubTree(const unsigned char *storePath, unsigned char *user, unsigned char *mailbox, unsigned char *buffer, unsigned int bufferLength)
{
    unsigned int pathLength;
    unsigned char *ptr;
    BOOL remove;
    XplDir *dirParent;
    XplDir *dirEntry;

    if (storePath && user && mailbox && buffer && storePath[0] && user[0] && mailbox[0]) {
        pathLength = strlen(storePath) + strlen(user) + 1;
        if ((pathLength + strlen(mailbox)) < bufferLength) {
            sprintf(buffer, "%s/%s/%s", storePath, user, mailbox);

            /*    Back up one directory entry to the mailbox parent.    */
            ptr = strrchr(buffer, '/');
            if (ptr) {
                *ptr = '\0';
            } else {
                return;
            }

            do {
                remove = TRUE;

                dirParent = XplOpenDir(buffer);
                if (dirParent) {
                    while ((dirEntry = XplReadDir(dirParent)) != NULL) {
                        if ((dirEntry->d_attr & XPL_A_SUBDIR) && (dirEntry->d_name[0] == '.')) {
                            continue;
                        }

                        remove = FALSE;
                        break;
                    }

                    XplCloseDir(dirParent);                            

                    if (remove) {
                        rmdir(buffer);
                    } else {
                        /*    No point in moving up; the sub-tree is used from here up.    */
                        return;
                    }
                } else {
                    break;
                }

                /*    Back up one directory entry.    */
                ptr = strrchr(buffer, '/');
                if (ptr) {
                    *ptr = '\0';
                } else {
                    break;
                }
            } while (strlen(buffer) > pathLength);
        }
    }

    return;
}

int 
NmapCommandCapa(void *param)
{
    int ccode;
    NMAPClient *client = (NMAPClient *)param;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (((ccode = ConnWrite(client->conn, MSG2001CAPA, sizeof(MSG2001CAPA) - 1)) != -1) 
        && (!NMAP.server.ssl.enable || ((ccode = ConnWrite(client->conn, MSG2001CAPATLS, sizeof(MSG2001CAPATLS) - 1)) != -1))) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandConf(void *param)
{
    int ccode;
    unsigned long i;
    NMAPClient *client = (NMAPClient *)param;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_AUTHORIZED) {
        if (((ccode = ConnWriteF(client->conn, "2001-Postmaster:%s\r\n", NMAP.postMaster)) != -1) 
                && ((ccode = ConnWriteF(client->conn, "2001-Server DN:%s\r\n", NMAP.server.dn)) != -1) 
                && ((ccode = ConnWriteF(client->conn, "2001-Official name:%s\r\n", NMAP.officialName)) != -1)) {
            ccode = ConnWriteF(client->conn, "2001-Keep space:%lu KB\r\n", NMAP.queue.minimumFree/1024);
        }
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    XplRWReadLockAcquire(&NMAP.lock.config);

    for (i = 0; (ccode != -1) && (i < NMAP.trusted.count); i++) {
        ccode = ConnWriteF(client->conn, "2001-Trusted host:%ld.%ld.%ld.%ld\r\n", NMAP.trusted.hosts[i] & 0xff, (NMAP.trusted.hosts[i] & 0xff00) >> 8, (NMAP.trusted.hosts[i] & 0xff0000) >> 16, (NMAP.trusted.hosts[i] & 0xff000000) >> 24);
    }

    XplRWReadLockRelease(&NMAP.lock.config);

    if ((ccode != -1) 
            && ((ccode = ConnWriteF(client->conn, "2001-Queue:%lu min. Interval\r\n", (NMAP.queue.sleep/60))) != -1) 
            && ((ccode = ConnWriteF(client->conn, "2001-Queue:%lu day(s) Timeout\r\n", (NMAP.queue.maxLinger/24)/60/60)) != -1) 
            && ((ccode = ConnWriteF(client->conn, "2001-SCMS: %u Users and %lu KB Threshold\r\n", NMAP.scms.userThreshold+1, NMAP.scms.sizeThreshold/1024)) != -1)) {
        if (NMAP.quota.useSystem) {
            ccode = ConnWriteF(client->conn, "2001-System-quota enabled:%lu KB\r\n", NMAP.quota.defaultValue);
        }

        if ((ccode != -1) && (NMAP.quota.useUser)) {
            ccode = ConnWrite(client->conn, "2001-User-quota enabled\r\n", 25);
        }

        if (ccode != -1) {
            ccode = ConnWriteF(client->conn, "2001-Tuning:%ld concurrent and %ld sequential threads for queue\r\n", NMAP.queue.limit.concurrent, NMAP.queue.limit.sequential);
        }

        for (i = 0; (ccode != -1) && (i < NMAP.queue.clients.count); i++) {
            ccode = ConnWriteF(client->conn, "2001-Queue client:%ld.%ld.%ld.%ld Port:%d Queue:%d (ID:%s)\r\n", 
                    NMAP.queue.clients.array[i].address & 0xff, 
                    (NMAP.queue.clients.array[i].address & 0xff00) >> 8, 
                    (NMAP.queue.clients.array[i].address & 0xff0000) >> 16, 
                    (NMAP.queue.clients.array[i].address & 0xff000000) >> 24, 
                    ntohs(NMAP.queue.clients.array[i].port), 
                    NMAP.queue.clients.array[i].queue, 
                    NMAP.queue.clients.array[i].identifier);
        }

        if ((ccode != -1) && ((ccode = ConnWriteF(client->conn, "2001-Default store:%s\r\n", NMAP.path.mail)) != -1)) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    }

    return(ccode);                        
}

int 
NmapCommandErro(void *param)
{
    NMAPClient *client = (NMAPClient *)param;

    return(ConnWrite(client->conn, MSG5000ERROR, sizeof(MSG5000ERROR) - 1));
}

int 
NmapCommandFdump(void *param)
{
/* fixme - handle XplDumpLock */
#if 0
    int ccode;
    unsigned long readCount;
    unsigned long writeCount;
    unsigned char *ptr;
    BOOL found;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_AUTHORIZED) {
        ptr = client->buffer + 5;
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    if (*ptr++ == ' ') {
        found = FALSE;

        XplDumpLock(ptr, readCount, writeCount, found);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (found) {
        ccode = ConnWriteF(client->conn, "1000-Flock %s has %lu readers and %lu writers\r\n", ptr, readCount, writeCount);
    } else {
        ccode = ConnWrite(client->conn, MSG4262NOTFOUND, sizeof(MSG4262NOTFOUND) - 1);
    }

#else
    int ccode;
    NMAPClient *client = (NMAPClient *)param;

    ccode = ConnWrite(client->conn, MSG1000NOTIMP, sizeof(MSG1000NOTIMP) - 1);
#endif

    return(ccode);
}

int 
NmapCommandFeat(void *param)
{
    int ccode;
    unsigned long column;
    unsigned char *ptr;
    unsigned char row;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* FEAT '<row>', <column> */
    if ((*ptr == ' ') && (ptr[0] == '\'') && (ptr[3] == ',') && (ptr[4] == ' ') && (isdigit(ptr[5]))) {
        row = ptr[1];
        column = atol(ptr + 5);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
        if (MsgGetUserFeature(client->dn, row, column, NULL, NULL)) {
            ccode = ConnWrite(client->conn, MSG1000FEATUREAVAIL, sizeof(MSG1000FEATUREAVAIL) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG2005FEATURENOTAVAIL, sizeof(MSG2005FEATURENOTAVAIL) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1);
    }

    return(ccode);;
}

int 
NmapCommandHelp(void *param)
{
    int ccode;
    unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;

    ptr = client->buffer + 4;
    if (*ptr == '\0') {
        ccode = ConnWrite(client->conn, MSG2001HELP, sizeof(MSG2001HELP) - 1);
    } else if (*ptr++ == ' ') {
        client->command = ProtocolCommandTreeSearch(&NMAP.commands, ptr);
        if (client->command) {
            if (client->command->help) {
                ccode = ConnWrite(client->conn, client->command->help, strlen(client->command->help));
            } else {
                ccode = ConnWrite(client->conn, NMAP_HELP_NOT_DEFINED, sizeof(NMAP_HELP_NOT_DEFINED) - 1);
            }
        } else {
            return(ConnWrite(client->conn, MSG3000UNKNOWN, sizeof(MSG3000UNKNOWN) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    return(ccode);
}

int 
NmapCommandNamespace(void *param)
{
    int ccode;
    NMAPClient *client = (NMAPClient *)param;

    /*
    "/"                 Heirarchical delimiter
    ""                  Personal namespace prefix
    "Shares/"           Shares from other users
    "Public Shares/"    Public shares    */
    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_USER) {
        ccode = ConnWriteF(client->conn, "1000 \"%c\" \"\" \"%s%c\" \"%s%c\"\r\n", NMAP.nameSpace.delimiter, NMAP.nameSpace.userPrefix, NMAP.nameSpace.delimiter, NMAP.nameSpace.publicPrefix, NMAP.nameSpace.delimiter);
    } else {
        ccode = ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1);
    }

    return(ccode);
}

int 
NmapCommandNver(void *param)
{
    NMAPClient *client = (NMAPClient *)param;

    return(ConnWriteF(client->conn, "1000 %s $Revision: 1.11 $\r\n", PROTOCOL_VERSION));
}

int 
NmapCommandOper(void *param)
{
    NMAPClient *client = (NMAPClient *)param;

    return(ConnWriteF(client->conn, "1000 %lu 0 0 0\r\n",clock()*100));
}

int 
NmapCommandProxy(void *param)
{
    int count;
    int ccode;
    unsigned long used;
    unsigned char *ptr;
    BOOL result;
    MDBValueStruct *vs;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 5;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        /* Load the owned proxies for client->user */
        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
        if (result) {
            vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
        } else {
            return(ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1));
        }

        if (vs) {
            MDBRead(client->dn, MSGSRV_A_OWNED_PROXIES, vs);
        } else {
            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }

        ccode = ConnWriteF(client->conn, "2002 %lu\r\n", vs->Used);
        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            count = strlen(vs->Value[used]);
            vs->Value[used][count - 11] = '\0';

            ccode = ConnWriteF(client->conn, "2002-%s %lu\r\n", vs->Value[used], atol(&(vs->Value[used][count - 10])));
        }

        MDBDestroyValueStruct(vs);

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else if (*ptr == ' ') {
        ccode = HandleProxyCreate(client);
    } else {
        ccode = ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);
    }

    return(ccode);
}

int 
NmapCommandQuit(void *param)
{
    NMAPClient *client = (NMAPClient *)param;

    /* QUIT */
    if (!(client->states & NMAP_CLIENT_QUEUE)) {
        FreeClientData(client);

        return(0);
    }

    return(ConnWrite(client->conn, MSG3013BADQSTATE, sizeof(MSG3013BADQSTATE) - 1));
}

int 
NmapCommandRset(void *param)
{
    int i;
    unsigned int used;
    unsigned char *ptr;
    sqlite3_stmt **stmt;
    NmapSqlCalComp *comp;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    if (*ptr == '\0') {
        /* RSET ALL */
        if (!(client->flags & NMAP_KEEPUSER)) {
            if (client->user[0] != '\0') {
                if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
                    StoreMaildrop(client);
                } else {
                    StoreSharedMaildrop(client);

                    NMAPQuit(client->share.mailbox.conn);
                    ConnFree(client->share.mailbox.conn);
                    client->share.mailbox.conn = NULL;

                    client->share.mailbox.permissions = 0;

                    client->share.flags &= ~NMAP_SHARE_MAILBOX;
                    client->share.flags &= ~NMAP_SHARE_MAILBOX_DENIED;
                }

                if (client->mailbox.lock) {
                    ReadNLockRelease(client->mailbox.lock);
                    client->mailbox.lock = NULL;
                }

#if !defined(NMAP_SQL_CALENDARS)
                if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
                    StoreCalendar(client);
                } else {
                    StoreSharedCalendar(client);

                    NMAPQuit(client->share.calendar.conn);
                    ConnFree(client->share.calendar.conn);
                    client->share.calendar.conn = NULL;

                    client->share.calendar.permissions = 0;

                    client->share.flags &= ~NMAP_SHARE_CALENDAR;
                    client->share.flags &= ~NMAP_SHARE_CALENDAR_DENIED;
                }

                if (client->calendar.lock) {
                    ReadNLockRelease(client->calendar.lock);
                    client->calendar.lock = NULL;
                }
#else
                /*
                    fixme for sql based calendaring
                */
#endif
            }

            client->states &= ~(NMAP_CLIENT_USER | NMAP_CLIENT_MBOX);
            client->user[0] = '\0';
            client->store = NULL;

            client->mailbox.name[0] = '\0';
            client->mailbox.message.used = 0;
            client->mailbox.message.count = 0;
            client->mailbox.newCount = 0;
            client->mailbox.size = 0;
            client->mailbox.newSize = 0;
            client->mailbox.id = 0;
            client->mailbox.nextUID = 0;
            client->mailbox.indexTime = 0;
            client->mailbox.newline = TRUE;

#if !defined(NMAP_SQL_CALENDARS)
            client->calendar.name[0] = '\0';
            client->calendar.entries.used = 0;
            client->calendar.entries.count = 0;
            client->calendar.size = 0;
            client->calendar.nextUID = 0;
            client->calendar.indexTime = 0;
            client->calendar.id = 0;
#else
            if (client->cal.objects) {
                used = client->cal.objects->used;
                comp = &(client->cal.objects->list[0]);

                while (used) {
                    if (comp->organizer) {
                        MemFree(comp->organizer);
                    }

                    if (comp->summary) {
                        MemFree(comp->summary);
                    }

                    used--;
                    comp++;
                }

                client->cal.objects->used = 0;
            }

            stmt = &(client->cal.sql.begin);
            for (i = 0; i < (sizeof(NMAPClientSQLStatements) / sizeof(sqlite3_stmt *)); i++, stmt++) {
                if (*stmt) {
                    sqlite3_finalize(*stmt);
                    *stmt = NULL;
                }
            }

            if (client->states & NMAP_CLIENT_CAL) {
                client->states &= ~(NMAP_CLIENT_CAL | NMAP_CLIENT_CALSEL);
                client->cal.id = 0;

                if (client->cal.handle) {
                    sqlite3_close(client->cal.handle);
                    client->cal.handle = NULL;
                }
            }
#endif

            return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
        }

        return(ConnWrite(client->conn, MSG4242NOTALLOWED, sizeof(MSG4242NOTALLOWED) - 1));
    } else if ((*ptr++ == ' ') && (!isspace(*ptr))) {
        if (toupper(*ptr) == 'M') {
            if (client->states & (NMAP_CLIENT_USER | NMAP_CLIENT_MBOX)) {
                if (client->user[0] != '\0') {
                    if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
                        StoreMaildrop(client);
                    } else {
                        StoreSharedMaildrop(client);

                        NMAPQuit(client->share.mailbox.conn);
                        ConnFree(client->share.mailbox.conn);
                        client->share.mailbox.conn = NULL;

                        client->share.mailbox.permissions = 0;

                        client->share.flags &= ~NMAP_SHARE_MAILBOX;
                        client->share.flags &= ~NMAP_SHARE_MAILBOX_DENIED;
                    }

                    if (client->mailbox.lock) {
                        ReadNLockRelease(client->mailbox.lock);
                        client->mailbox.lock = NULL;
                    }
                }

                client->states &= ~NMAP_CLIENT_MBOX;
                client->mailbox.name[0] = '\0';
                client->mailbox.message.used = 0;
                client->mailbox.message.count = 0;
                client->mailbox.newCount = 0;
                client->mailbox.size = 0;
                client->mailbox.newSize = 0;
                client->mailbox.id = 0;
                client->mailbox.nextUID = 0;
                client->mailbox.indexTime = 0;
                client->mailbox.newline = TRUE;

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }

            return(ConnWrite(client->conn, MSG3240BADSTATE, sizeof(MSG3240BADSTATE) - 1));
        } else if (toupper(*ptr) == 'C') {
#if !defined(NMAP_SQL_CALENDARS)
            if  (client->states & (NMAP_CLIENT_USER | NMAP_CLIENT_MBOX)) {
                if ((client->user[0] != '\0') && (client->calendar.name[0] != '\0')) {
                    if (!(client->share.flags & NMAP_SHARE_CALENDAR)) {
                        StoreCalendar(client);
                    } else {
                        StoreSharedCalendar(client);

                        NMAPQuit(client->share.calendar.conn);
                        ConnFree(client->share.calendar.conn);
                        client->share.calendar.conn = NULL;

                        client->share.calendar.permissions = 0;

                        client->share.flags &= ~NMAP_SHARE_CALENDAR;
                        client->share.flags &= ~NMAP_SHARE_CALENDAR_DENIED;
                    }

                    if (client->calendar.lock) {
                        ReadNLockRelease(client->calendar.lock);
                        client->calendar.lock = NULL;
                    }
                }

                client->calendar.name[0] = '\0';
                client->calendar.entries.used = 0;
                client->calendar.entries.count = 0;
                client->calendar.size = 0;
                client->calendar.nextUID = 0;
                client->calendar.indexTime = 0;
                client->calendar.id = 0;

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }
#else
            if (client->cal.objects) {
                used = client->cal.objects->used;
                comp = &(client->cal.objects->list[0]);

                while (used) {
                    if (comp->organizer) {
                        MemFree(comp->organizer);
                    }

                    if (comp->summary) {
                        MemFree(comp->summary);
                    }

                    used--;
                    comp++;
                }

                client->cal.objects->used = 0;
            }

            stmt = &(client->cal.sql.begin);
            for (i = 0; i < (sizeof(NMAPClientSQLStatements) / sizeof(sqlite3_stmt *)); i++, stmt++) {
                if (*stmt) {
                    sqlite3_finalize(*stmt);
                    *stmt = NULL;
                }
            }

            if (client->states & NMAP_CLIENT_CAL) {
                client->states &= ~(NMAP_CLIENT_CAL | NMAP_CLIENT_CALSEL);
                client->cal.id = 0;

                if (client->cal.handle) {
                    sqlite3_close(client->cal.handle);
                    client->cal.handle = NULL;
                }

                return(ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1));
            }
#endif

            return(ConnWrite(client->conn, MSG3240BADSTATE, sizeof(MSG3240BADSTATE) - 1));
        }
    }

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_UNHANDLED, LOGGER_EVENT_UNHANDLED_REQUEST, LOG_INFO, 0, client->user, client->buffer, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);

    return(ConnWrite(client->conn, MSG3000UNKNOWN, sizeof(MSG3000UNKNOWN) - 1));
}

int 
NmapCommandSpace(void *param)
{
    int ccode;
    long space;
    unsigned long used;
    unsigned char *ptr;
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    BOOL result = FALSE;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs = NULL;

    ptr = client->buffer + 5;

    if (*ptr == '\0') {
        if (client->states & NMAP_CLIENT_USER) {
            if (NMAP.quota.useUser) {
                vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
                if (vs) {
                    result = MsgFindObject(client->user, dn, NULL, NULL, vs);
                }
            }

            sprintf(client->path, "%s/%s", client->store, client->user);
        } else {
            return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
        }

    } else if ((*ptr++ == ' ') && (*ptr) && (!isspace(*ptr))) {
        if (client->states & (NMAP_CLIENT_QUEUE | NMAP_CLIENT_AUTHORIZED)) {
            if (NMAP.quota.useUser) {
                vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
                if (vs) {
                    result = MsgFindObject(ptr, dn, NULL, NULL, vs);
                }
            } 

            sprintf(client->path, "%s/%s", MsgFindUserStore(ptr, NMAP.path.mail), ptr);
        } else {
            return(ConnWrite(client->conn, MSG3012BADQSTATE, sizeof(MSG3012BADQSTATE) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    space = -1;
    if (result) {
        MDBFreeValues(vs);

        if ((MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_USE_QUOTA, vs) > 0) 
                && (vs->Value[0][0] == '1')) {
            MDBFreeValues(vs);

            if (MsgGetUserFeature(dn, FEATURE_NMAP_QUOTA, MSGSRV_A_QUOTA_VALUE, vs) > 0) {
                space = atol(vs->Value[0]);
            }
        }
    }

    if (vs) {
        MDBDestroyValueStruct(vs);
    }

    if (NMAP.quota.useSystem && (space == -1)) {
        space = NMAP.quota.defaultValue;
    }

    if ((used = XplGetDiskspaceUsed(client->path)) != 0) {
        if (space != -1) {
            ccode = ConnWriteF(client->conn, "1000 %lu %lu OK\r\n", used, space);
        } else {
            ccode = ConnWriteF(client->conn, "1000 %lu 0 No restrictions\r\n", used);
        }
    } else {
        ccode = ConnWriteF(client->conn, "1000 1 %lu OK\r\n", space);
    }

    return(ccode);
}

int 
NmapCommandTls(void *param)
{
    int ccode;
    BOOL result;
    NMAPClient *client = (NMAPClient *)param;

    if (NMAP.server.ssl.enable) {
        ccode = ConnWrite(client->conn, MSG1000STARTTLS, sizeof(MSG1000STARTTLS) - 1);
    } else {
        return(ConnWrite(client->conn, MSG4244NOTSUPPORTED, sizeof(MSG4244NOTSUPPORTED) - 1));
    }

    if (ccode != -1) {
        ccode = ConnFlush(client->conn);
    }

    if (ccode != -1) {
        client->conn->ssl.enable = TRUE;

        /* fixme - what do we want to do if encryption negotiation fails? */
        result = ConnNegotiate(client->conn, NMAP.server.ssl.context);
        if (!result) {
            client->conn->ssl.enable = FALSE;

            /* fixme - we need connection management flags for ConnClose to force the shutdown */
            ConnClose(client->conn, 1);
            ConnFree(client->conn);
            client->conn = NULL;
        }
    }

    return(ccode);
}

int 
NmapCommandUpda(void *param)
{
    unsigned long used;
    unsigned long totalSize = 0;
    unsigned long totalCount = 0;
    unsigned long purgedSize = 0;
    unsigned long purgedCount = 0;
    struct stat sb;
    NMAPClient *client = (NMAPClient *)param;
    MessageInfoStruct *msgInfo;

    /* fixme - verify command buffer; no arguments means no delimiter. */
    if (client->states & NMAP_CLIENT_MBOX) {
        if (!(client->share.flags & NMAP_SHARE_MAILBOX)) {
            sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, TAKE_PURGES, NOTIFY_CLIENT_SYNC);
        } else if (!(client->share.flags & NMAP_SHARE_MAILBOX_DENIED)) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            GET_CHANGED_FLAGS(client->path, TAKE_PURGES, NOTIFY_CLIENT_SYNC);

            if (SyncSharedMaildrop(client, NULL) == FALSE) {
                return(ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1));
            }
        } else {
            return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
        }
    } else if (!(client->states & NMAP_CLIENT_USER)) {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    } else {
        return(ConnWrite(client->conn, MSG4225NOMBOX, sizeof(MSG4225NOMBOX) - 1));
    }

    used = client->mailbox.message.used;
    msgInfo = client->mailbox.message.info;

    while (used-- != 0) {
        totalSize += msgInfo->SSize;
        totalCount++;

        if (msgInfo->State & MSG_STATE_PURGED) {
            purgedSize += msgInfo->SSize;
            purgedCount++;
        }

        msgInfo++;
    }

    return(ConnWriteF(client->conn, "1000 %s %lu %lu %lu %lu %lu %lu\r\n",
        client->user, totalCount, totalSize, client->mailbox.newCount,
        client->mailbox.newSize, purgedCount, purgedSize));
}

int 
NmapCommandVrfy(void *param)
{
    int ccode;
    unsigned long used;
    unsigned char *ptr;
    unsigned char type[MDB_MAX_ATTRIBUTE_CHARS + 1];
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_AUTHORIZED) {
        ptr = client->buffer + 4;
    } else {
        return(ConnWrite(client->conn, MSG3240NOAUTH, sizeof(MSG3240NOAUTH) - 1));
    }

    if ((*ptr++ == ' ') && (*ptr) && (!isspace(*ptr))) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (vs) {
        ccode = 0;
        if (MsgFindObject(ptr, dn, type, NULL, vs) && vs->Used) {
            for (used = 0; (ccode != -1) && (used < (vs->Used - 1)); used++) {
                ccode = ConnWriteF(client->conn, "2001-%s\r\n", vs->Value[used]);
            }

            if (ccode != -1) {
                ccode = ConnWriteF(client->conn, "2001 %s\r\n", vs->Value[vs->Used - 1]);
            }
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        MDBDestroyValueStruct(vs);
    } else {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
    }

    return(ccode);
}

int 
HandleCommand(NMAPClient *client)
{
    int ccode = 0;

    while ((ccode != -1) && (NMAP.state < NMAP_STOPPING)) {
        client->buffer[0] = '\0';

        ccode = ConnReadAnswer(client->conn, client->buffer, CONN_BUFSIZE);
        if ((ccode != -1) && (ccode < CONN_BUFSIZE)) {
#if 0
            XplConsolePrintf("NMAP command: [%d.%d.%d.%d] %s %s\r\n", 
                    client->conn->socketAddress.sin_addr.s_net, 
                    client->conn->socketAddress.sin_addr.s_host, 
                    client->conn->socketAddress.sin_addr.s_lh, 
                    client->conn->socketAddress.sin_addr.s_impno, 
                    client->buffer, 
                    (client->conn->ssl.enable)? "(Secure)": "");
#endif

            client->command = ProtocolCommandTreeSearch(&NMAP.commands, client->buffer);
            if (client->command) {
                ccode = client->command->handler(client);
            } else {
                ccode = ConnWrite(client->conn, MSG3000UNKNOWN, sizeof(MSG3000UNKNOWN) - 1);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_UNHANDLED, LOGGER_EVENT_UNHANDLED_REQUEST, LOG_INFO, 0, client->user, client->buffer, XplHostToLittle(client->conn->socketAddress.sin_addr.s_addr), 0, NULL, 0);
            }
        }

        if (client && client->conn) {
            ConnFlush(client->conn);
            continue;
        }

        break;
    }

    return(ccode);
}

static void 
HandleConnection(void *param)
{
    int ccode;
    long threadNumber = (long)param;
    unsigned long count;
    time_t sleep = time(NULL);
    time_t wokeup;
    NMAPClient *client;

    client = NMAPClientAlloc();
    if (!client) {
        XplSafeDecrement(NMAP.client.worker.active);

        return;
    }

    do {
        XplRenameThread(XplGetThreadID(), "NMAP Worker");

        XplSafeIncrement(NMAP.client.worker.idle);

        XplWaitOnLocalSemaphore(NMAP.client.worker.todo);

        XplSafeDecrement(NMAP.client.worker.idle);

        wokeup = time(NULL);

        XplWaitOnLocalSemaphore(NMAP.client.semaphore);

        client->conn = NMAP.client.worker.tail;
        if (client->conn) {
            NMAP.client.worker.tail = client->conn->queue.previous;
            if (NMAP.client.worker.tail) {
                NMAP.client.worker.tail->queue.next = NULL;
            } else {
                NMAP.client.worker.head = NULL;
            }
        }

        XplSignalLocalSemaphore(NMAP.client.semaphore);

        if (client->conn) {
            XplRWReadLockAcquire(&NMAP.lock.config);
            for (count = 0; count < NMAP.trusted.count; count++) {
                if (client->conn->socketAddress.sin_addr.s_addr == NMAP.trusted.hosts[count]) {
                    client->states |= NMAP_CLIENT_AUTHORIZED;
                    break;
                }
            }
            XplRWReadLockRelease(&NMAP.lock.config);

            ccode = -1;

            if ((!NMAP.server.ssl.enable && ConnNegotiate(client->conn, NULL)) 
                    || ConnNegotiate(client->conn, NMAP.server.ssl.context)) {
                if (!(client->states & NMAP_CLIENT_AUTHORIZED)) {
                    sprintf(client->user, "%x%s%x", XplGetThreadID(), NMAP.server.host, (unsigned int)time(NULL));

                    ccode = ConnWriteF(client->conn, MSG4242AUTHREQUIRED, client->user);
                } else {
                    ccode = ConnWriteF(client->conn, "1000 %s %s\r\n", NMAP.server.host, MSG1000READY);
                }

                SetCurrentNameSpace(NWOS2_NAME_SPACE);
            }

            if (ccode != -1) {
                ccode = ConnFlush(client->conn);
            }

            if (ccode != -1) {
                ccode = HandleCommand(client);
            }

            if (client->conn) {
                ConnClose(client->conn, 1);
                ConnFree(client->conn);
                client->conn = NULL;
            }
        } else {
        }

        /* Live or die? */
        if (threadNumber == XplSafeRead(NMAP.client.worker.active)) {
            if ((wokeup - sleep) > NMAP.client.sleepTime) {
                break;
            }
        }

        sleep = time(NULL);

        NMAPConnectionAllocCB(client, NULL);
    } while (NMAP.state == NMAP_RUNNING);

    NMAPClientFree(client);

    XplSafeDecrement(NMAP.client.worker.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static BOOL
InternalSetServerState(const unsigned char *state)
{
    MDBValueStruct *vs;

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (vs) {
        MDBAddValue(state, vs);

        MDBWrite(NMAP.server.dn, MSGSRV_A_SERVER_STATUS, vs);
        MDBDestroyValueStruct(vs);

        return(TRUE);
    }

    return(FALSE);
}

static int
ServerSocketInit(void)
{
    NMAP.server.conn = ConnAlloc(FALSE);
    if (NMAP.server.conn) {
        NMAP.server.conn->socketAddress.sin_family = AF_INET;
        NMAP.server.conn->socketAddress.sin_port = htons(NMAP_PORT);
        NMAP.server.conn->socketAddress.sin_addr.s_addr = MsgGetAgentBindIPAddress();	

	/* Get root privs back for the bind.  It's ok if this fails - 
	 * the user might not need to be root to bind to the port */
	XplSetEffectiveUserId(0);
	
        NMAP.server.conn->socket = ConnServerSocket(NMAP.server.conn, 2048);
	
	if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
	    XplConsolePrintf("hulanmap: Could not drop to unprivileged user '%s'\n", MsgGetUnprivilegedUser());
	    return -1;
	}


	if (NMAP.server.conn->socket == -1) {
	    int ret = NMAP.server.conn->socket;
	    XplConsolePrintf("hulanmap: Could not bind to port %d\n", NMAP_PORT);
	    ConnFree(NMAP.server.conn);
	    return ret;
	}
    } else {
	XplConsolePrintf("hulanmap: Could not allocate connection\n");
	return -1;
    }
    return 0;
}

static void 
NMAPServer(void *ignored)
{
    int i;
    int ccode;
    XplThreadID id;
    Connection *conn;

    XplSafeIncrement(NMAP.server.active);

    XplRenameThread(XplGetThreadID(), "NMAP Server");


    NMAP.state = NMAP_RUNNING;

    while (NMAP.state < NMAP_STOPPING) {
        if (ConnAccept(NMAP.server.conn, &conn) != -1) {
            if (NMAP.state < NMAP_STOPPING) {
                conn->ssl.enable = FALSE;

                QUEUE_WORK_TO_DO(conn, id, ccode);
                if (!ccode) {
                    XplSignalLocalSemaphore(NMAP.client.worker.todo);

                    continue;
                }

                ConnWrite(conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
            } else {
                ConnWrite(conn, MSG5002SHUTDOWN, sizeof(MSG5002SHUTDOWN) - 1);
            }

            ConnClose(conn, 0);

            ConnFree(conn);
            conn = NULL;

            continue;
        }
        switch (errno) {
            case ECONNABORTED:
#ifdef EPROTO
            case EPROTO: 
#endif
            case EINTR: {
                if (NMAP.state < NMAP_STOPPING) {
                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);
                }

                continue;
            }

            default: {
                if (NMAP.state < NMAP_STOPPING) {
                    XplConsolePrintf("NMAP: Exiting after an accept() failure; error %d\r\n", errno);

                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_ACCEPT_FAILURE, LOG_ERROR, 0, "Server", NULL, errno, 0, NULL, 0);

                    NMAP.state = NMAP_STOPPING;
                }

                break;
            }
        }

        break;
    }

    /* Shutting down */
    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_SERVER_SHUTDOWN, LOG_ERROR, 0, NULL, NULL, NMAP.state, errno, NULL, 0);
    NMAP.state = NMAP_STOPPING;

    InternalSetServerState("Shutting down");

    id = XplSetThreadGroupID(NMAP.id.group);

    if (NMAP.server.conn) {
        ConnClose(NMAP.server.conn, 1);
        NMAP.server.conn = NULL;
    }

    if (NMAP.server.ssl.enable) {
        NMAP.server.ssl.enable = FALSE;

        if (NMAP.server.ssl.conn) {
            ConnClose(NMAP.server.ssl.conn, 1);
            NMAP.server.ssl.conn = NULL;
        }

        if (NMAP.server.ssl.context) {
            ConnSSLContextFree(NMAP.server.ssl.context);
            NMAP.server.ssl.context = NULL;
        }
    }

    ConnCloseAll(1);

    if (ManagementState() == MANAGEMENT_RUNNING) {
        ManagementShutdown();
    }

    for (i = 0; (XplSafeRead(NMAP.server.active) > 1) && (i < 60); i++) {
        XplDelay(1000);
    }

    for (i = 0; (ManagementState() != MANAGEMENT_STOPPED) && (i < 60); i++) {
        XplDelay(1000);
    }

    XplConsolePrintf("NMAPD: Shutting down %d client threads\r\n", XplSafeRead(NMAP.client.worker.active) + XplSafeRead(NMAP.queue.worker.active));

    XplWaitOnLocalSemaphore(NMAP.client.semaphore);

    ccode = XplSafeRead(NMAP.client.worker.idle);
    while (ccode--) {
        XplSignalLocalSemaphore(NMAP.client.worker.todo);
    }

    XplSignalLocalSemaphore(NMAP.client.semaphore);

    for (i = 0; (XplSafeRead(NMAP.client.worker.active) + XplSafeRead(NMAP.queue.worker.active)) && (i < 60); i++) {
        XplDelay(1000);
    }

    if (XplSafeRead(NMAP.server.active) > 1) {
        XplConsolePrintf("NMAPD: %d server threads outstanding; attempting forceful unload.\r\n", XplSafeRead(NMAP.server.active) - 1);
    }

    if (XplSafeRead(NMAP.client.worker.active) + XplSafeRead(NMAP.queue.worker.active)) {
        XplConsolePrintf("NMAPD: %d threads outstanding; attempting forceful unload.\r\n", XplSafeRead(NMAP.client.worker.active) + XplSafeRead(NMAP.queue.worker.active));
    }

    LoggerClose(NMAP.handle.logging);
    NMAP.handle.logging = NULL;

    XplConsolePrintf("NMAPD: Writing queue agent list\r\n");
    if (NMAP.queue.clients.array) {
        RemoveAllPushClients();
    }

    XplCloseLocalSemaphore(NMAP.guid.semaphore);

    XplCloseLocalSemaphore(NMAP.queue.semaphore);

    for (i = 0; i <= MBOX_LOCK_SEM_ARRAY_SIZE; i++) {
        XplCloseLocalSemaphore(NMAP.lock.mailbox.sems[i]);
    }

    for (i = 0; i <= CSTORE_LOCK_SEM_ARRAY_SIZE; i++) {
        XplCloseLocalSemaphore(NMAP.lock.calendar.sems[i]);
    }

    XplCloseLocalSemaphore(NMAP.client.semaphore);

    XplCloseLocalSemaphore(NMAP.client.worker.todo);

    MemFree(NMAP.trusted.hosts);
    if (NMAP.quota.message) {
        MemFree(NMAP.quota.message);
    }

    if (NMAP.quota.warning) {
        MemFree(NMAP.quota.warning);
    }

    if (NMAP.newShareMessage) {
        MemFree(NMAP.newShareMessage);
    }

    XplConsolePrintf("NMAPD: Closing queue database\r\n");

    DeInitSpoolEntryIDLocks();

    MemPrivatePoolFree(NMAP.client.pool);

    QDBShutdown();

    XplRWLockDestroy(&NMAP.lock.config);
    NmapSqlLocksDestroy();
    NLockDestroy();

#if defined(NMAP_SQL_CALENDARS)
    CloseICal2Parser();
#endif

    InternalSetServerState("Shutdown");

    MsgShutdown();
/*  MDBShutdown(); */

    ConnShutdown();

    MemoryManagerClose(MSGSRV_AGENT_NMAP);

    XplConsolePrintf("NMAPD: Shutdown complete\r\n");

    XplSignalLocalSemaphore(NMAP.sem.main);
    XplWaitOnLocalSemaphore(NMAP.sem.shutdown);

    XplCloseLocalSemaphore(NMAP.sem.shutdown);
    XplCloseLocalSemaphore(NMAP.sem.main);

    XplSetThreadGroupID(id);

    return;
}

static void
CheckDiskspace(void)
{
    int i;
    unsigned long freeBlocks;

    XplRenameThread(XplGetThreadID(), "NMAP Disk Monitor");

    while (NMAP.state < NMAP_STOPPING) {
        /* Check every 10 minutes */
        freeBlocks = XplGetDiskspaceFree(NMAP.path.spool);        
        if (freeBlocks != 0x7f000000) {
            if (freeBlocks < ((NMAP.queue.minimumFree / NMAP.server.bytesPerBlock) + 1)) {
                /* BUG ALERT: */
                /* With a block size of 4KB, freeBlocks could */
                /* overflow if a volume had more than 16 TB. */

                NMAP.flags |= NMAP_FLAG_DISK_SPACE_LOW;

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_DISKSPACE_LOW, LOG_ERROR, 0, NMAP.path.spool, NULL, freeBlocks, (NMAP.queue.minimumFree/NMAP.server.bytesPerBlock)+1, NULL, 0);
            } else {
                NMAP.flags &= ~NMAP_FLAG_DISK_SPACE_LOW;
            }
        } else {
            XplConsolePrintf("NMAPD: The OS failed to respond to a disk space query\r\n");
        }

        for (i = 0; (i < 600) && (NMAP.state < NMAP_STOPPING); i++) {
            XplDelay(1000);
        }
    }

    XplConsolePrintf("NMAPD: Disk monitor thread done.\r\n");

    XplSafeDecrement(NMAP.server.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void
LoadMonitor(BOOL display)
{
    unsigned long counter;
    unsigned long current;
    unsigned long state = 0;
    unsigned long upTime = 0;
    unsigned long idleTime = 0;

    XplRenameThread(XplGetThreadID(), "NMAP Load Monitor");

    NMAP.id.load = XplGetThreadGroupID();

    /* We abuse the ThreadGroup for the load monitor in PrintDir, to have a thread group that uses DOS namespace    */
    /* therefore we set it here                                                                                                                */
    SetCurrentNameSpace(NWDOS_NAME_SPACE);
    SetTargetNameSpace(NWDOS_NAME_SPACE);

    /*  fixme - what to do whith this?  Don't kill it without handling the 
        fact that this thread provides a thread group using the DOS namespace.  */
    while (NMAP.state < NMAP_STOPPING) {
        if (!NMAP.loadMonitor.disabled) {
            current = XplGetServerUtilization(&upTime, &idleTime);

            if (current <= NMAP.loadMonitor.low) {
                if (state) {
                    if (NMAP.queue.limit.defaultConcurrent > NMAP.queue.limit.concurrent) {
                        NMAP.queue.limit.concurrent *= 2;
                    }

                    state = 0;
                } else {
                    if (NMAP.queue.limit.defaultConcurrent > NMAP.queue.limit.concurrent) {
                        NMAP.queue.limit.concurrent *= 2;
                    }

                    if (NMAP.queue.limit.defaultSequential > NMAP.queue.limit.sequential) {
                        NMAP.queue.limit.sequential *= 2;
                    }
                }
            } else if (current >= NMAP.loadMonitor.high) {
                if (XplSafeRead(NMAP.stats.queuedLocal.messages) >= NMAP.queue.limit.trigger) {
                    if (state) {
                        NMAP.queue.limit.sequential = max(NMAP.queue.limit.sequential / 2, 14);
                        state = 0;
                    } else {
                        NMAP.queue.limit.concurrent = max(NMAP.queue.limit.concurrent / 2, 7);
                        state = 1;
                    }
                }
            }
        }

#if 0
        if (NMAP.flags & NMAP_FLAG_DEBUG) {
            XplConsolePrintf("NMAP NMAP.client.worker.active: %d NMAP.queue.worker.active: %d\r\n", XplSafeRead(NMAPQueueThreads) + XplSafeRead(NMAPConnThreads), XplSafeRead(NMAPQueueThreads));
        }
#endif

        for (counter = 0; (counter < NMAP.loadMonitor.interval) && (NMAP.state < NMAP_STOPPING); counter += 5) {
            XplDelay(5000);
        }
    }

    XplConsolePrintf("NMAPD: Load monitor thread done.\r\n");    

    XplSafeDecrement(NMAP.server.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static void
ConfigMonitor(void)
{
    int i;
    int count;
    unsigned long used;
    unsigned long previous;
    MDBValueStruct *vs;

    XplRenameThread(GetThreadID(), "NMAP Config Monitor");

    previous = 0;

    vs = MDBCreateValueStruct(NMAP.handle.directory, MsgGetServerDN(NULL));
    if (vs) {
        if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_CONFIG_CHANGED, vs) > 0) {
            previous = atol(vs->Value[0]);
        }

        MDBDestroyValueStruct(vs);
    }

    while (NMAP.state < NMAP_STOPPING) {
        i = 0;
        do {
            NMAP.time.global = time(NULL);

            XplDelay(1000);
        } while ((++i < 300) && (NMAP.state < NMAP_STOPPING));

        vs = MDBCreateValueStruct(NMAP.handle.directory, MsgGetServerDN(NULL));
        if (!vs) {
            continue;
        }

        if ((NMAP.state < NMAP_STOPPING) 
                    && (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_CONFIG_CHANGED, vs) > 0) 
                    && ((unsigned long)atol(vs->Value[0]) != previous)) {
            /* Clear what we just read */
            previous = atol(vs->Value[0]);
            MDBFreeValues(vs);

            /* Acquire Write Lock */            
            XplRWWriteLockAcquire(&NMAP.lock.config);

            NMAP.trusted.count = MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_NMAP_TRUSTED_HOSTS, vs);

#if defined(NETWARE) || defined(LIBC)
            NMAP.trusted.count += 2;
#endif

            if (NMAP.trusted.hosts) {
                MemFree(NMAP.trusted.hosts);
            }

            if (NMAP.trusted.count) {
                NMAP.trusted.hosts = MemMalloc(NMAP.trusted.count * sizeof(unsigned long));
                if (NMAP.trusted.hosts) {
                    count = 0;

#if defined(NETWARE) || defined(LIBC)
                    NMAP.trusted.hosts[count++] = inet_addr("127.0.0.1");
                    NMAP.trusted.hosts[count++] = MsgGetHostIPAddress();
#endif

                    for (used = 0; used < vs->Used; used++) {
                        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "NMAP_TRUSTED_HOSTS", vs->Value[used], 0, 0, NULL, 0);

                        NMAP.trusted.hosts[count++] = inet_addr(vs->Value[used]);
                    }
                } else {
                    NMAP.trusted.count = 0;
                    previous--;
                }
            }

            MDBFreeValues(vs);

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_RTS_HANDLING, vs) > 0) { 
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_RTS_HANDLING", vs->Value[0], 0, 0, NULL, 0);
                NMAP.rts.handling = atol(vs->Value[0]);

                MDBFreeValues(vs);
            } else {
                NMAP.rts.handling = RTS_RETURN;
            }

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_FORWARD_UNDELIVERABLE, vs) > 0) {
                if (vs->Value[0][0] != '\0') {
                    strcpy(NMAP.queue.forwardUndeliverable.address, vs->Value[0]);
                    NMAP.queue.forwardUndeliverable.enabled = TRUE;
                } else {
                    NMAP.queue.forwardUndeliverable.address[0] = '\0';
                    NMAP.queue.forwardUndeliverable.enabled = FALSE;
                }

                MDBFreeValues(vs);
            } else {
                NMAP.queue.forwardUndeliverable.address[0] = '\0';
                NMAP.queue.forwardUndeliverable.enabled = FALSE;
            }

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUEUE_INTERVAL, vs) > 0) { 
                NMAP.queue.sleep = atol(vs->Value[0])*60;
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_QUEUE_INTERVAL", NULL, NMAP.queue.sleep, 0, NULL, 0);
                if (NMAP.queue.sleep < 1) {
                    NMAP.queue.sleep = 4 * 60;
                }

                MDBFreeValues(vs);
            } else {
                NMAP.queue.sleep = 4 * 60;
            }

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SCMS_USER_THRESHOLD, vs) > 0) { 
                NMAP.scms.userThreshold = atol(vs->Value[0]) - 1;
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_SCMS_USER_THRESHOLD", NULL, NMAP.scms.userThreshold, 0, NULL, 0);
                if (NMAP.scms.userThreshold < 4) {
                    NMAP.scms.userThreshold = 4;
                }

                MDBFreeValues(vs);
            } else {
                NMAP.scms.userThreshold = 4;
            }

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SCMS_SIZE_THRESHOLD, vs) > 0) { 
                NMAP.scms.sizeThreshold = atol(vs->Value[0]) * 1024;
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_SCMS_SIZE_THRESHOLD", NULL, NMAP.scms.sizeThreshold, 0, NULL, 0);
                if (NMAP.scms.sizeThreshold < 5120) {
                    NMAP.scms.sizeThreshold = 5120;
                }

                MDBFreeValues(vs);
            } else {
                NMAP.scms.sizeThreshold = 5120;
            }

            if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_RTS_ANTISPAM_CONFIG, vs) > 0) { 
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_RTS_ANTISPAM_CONFIG", vs->Value[0], 0, 0, NULL, 0);
                if (sscanf(vs->Value[0], "Enabled:%d Delay:%ld Threshhold:%lu", &NMAP.rts.blockSpam, &NMAP.queue.bounce.interval, &NMAP.queue.bounce.maxCount) != 3) {
                    NMAP.rts.blockSpam = FALSE;
                }

                if ((NMAP.queue.bounce.maxCount < 1) || (NMAP.queue.bounce.interval < 1)) {
                    NMAP.rts.blockSpam = FALSE;
                }
                MDBFreeValues(vs);
            } else {
                NMAP.rts.blockSpam = FALSE;
            }

            XplRWWriteLockRelease(&NMAP.lock.config);
        }

        MDBDestroyValueStruct(vs);
        vs = NULL;
    }

    XplConsolePrintf("NMAPD: Config monitor thread done.\r\n");    

    XplSafeDecrement(NMAP.server.active);

    XplExitThread(TSR_THREAD, 0);

    return;
}

static BOOL
ReadConfiguration(BOOL *recover)
{
    int count;
    unsigned long used;
    unsigned char *ptr;
    BOOL result = TRUE;
    MDBValueStruct *vs;
    MDBValueStruct *stores;

    tzset();
    NMAP.time.startup = NMAP.time.global = time(NULL);

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (vs && MDBRead(MSGSRV_ROOT, MSGSRV_A_ACL, vs)) {
        gethostname(NMAP.server.host, sizeof(NMAP.server.host));

        HashCredential(MsgGetServerDN(NULL), vs->Value[0], NMAP.server.hash);

        MDBFreeValues(vs);
    } else {
        return(FALSE);
    }

    if (recover) {
        if (MDBRead(NMAP.server.dn, MSGSRV_A_SERVER_STATUS, vs) > 0) {
            if (XplStrCaseCmp(vs->Value[0], "Shutdown") != 0) {
                *recover = TRUE;
            }
        }

        MDBFreeValues(vs);
    }

    if (MDBRead(NMAP.server.dn, MSGSRV_A_OFFICIAL_NAME, vs) > 0) {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_OFFICIAL_NAME", vs->Value[0], 0, 0, NULL, 0);

        strcpy(NMAP.server.host, vs->Value[0]);
        strcpy(NMAP.officialName, vs->Value[0]);

        MDBFreeValues(vs);
    }

    if (MDBRead(NMAP.server.dn, MSGSRV_A_POSTMASTER, vs) > 0) {
        if (vs->Value[0]) {
            if ((ptr = strrchr(vs->Value[0], '\\')) != NULL) {
                strcpy(NMAP.postMaster, ptr + 1);
            } else {
                strcpy(NMAP.postMaster, vs->Value[0]);
            }
        } else {
            strcpy(NMAP.postMaster, "admin");
        }

        MDBFreeValues(vs);
    }

    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_POSTMASTER", NMAP.postMaster, 0, 0, NULL, 0);

    MDBSetValueStructContext(NMAP.server.dn, vs);

    NMAP.server.ipAddress = MsgGetHostIPAddress();
    if (NMAP.server.ipAddress == MsgGetAgentBindIPAddress()) {
        NMAP.server.bound = TRUE;
    }

    NMAP.trusted.count = MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_NMAP_TRUSTED_HOSTS, vs);

#if defined(NETWARE) || defined(LIBC)
    NMAP.trusted.count += 2;
#endif

    if (NMAP.trusted.hosts) {
        MemFree(NMAP.trusted.hosts);
    }

    if (NMAP.trusted.count) {
        NMAP.trusted.hosts = MemMalloc(NMAP.trusted.count * sizeof(unsigned long));
        if (NMAP.trusted.hosts) {
            count = 0;

#if defined(NETWARE) || defined(LIBC)
            NMAP.trusted.hosts[count++] = inet_addr("127.0.0.1");
            NMAP.trusted.hosts[count++] = MsgGetHostIPAddress();
#endif

            for (used = 0; used < vs->Used; used++) {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "NMAP_TRUSTED_HOSTS", vs->Value[used], 0, 0, NULL, 0);

                NMAP.trusted.hosts[count++] = inet_addr(vs->Value[used]);
            }
        } else {
            NMAP.trusted.count = 0;
        }
    }

    MDBFreeValues(vs);

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_MINIMUM_SPACE, vs) > 0) { 
        NMAP.queue.minimumFree = atol(vs->Value[0]) * 1024;
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_MINIMUM_SPACE", NULL, NMAP.queue.minimumFree, 0, NULL, 0);

        MDBFreeValues(vs);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_UID, vs) > 0) { 
        NMAP.universalCounter = atol(vs->Value[0]);
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_UID", NULL, NMAP.universalCounter, 0, NULL, 0);

        MDBFreeValues(vs);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUEUE_TIMEOUT, vs) > 0) { 
        NMAP.queue.maxLinger = atoi(vs->Value[0]) * 24 * 60 * 60;
        if (!NMAP.queue.maxLinger) {
            NMAP.queue.maxLinger = 4 * 24 * 60 * 60;
        }

        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_QUEUE_TIMEOUT", NULL, NMAP.queue.maxLinger, 0, NULL, 0);

        MDBFreeValues(vs);
    } else {
        NMAP.queue.maxLinger = 4 * 24 * 60 * 60;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUEUE_INTERVAL, vs) > 0) { 
        NMAP.queue.sleep = atol(vs->Value[0]) * 60;
        if (NMAP.queue.sleep < 1) {
            NMAP.queue.sleep = 4 * 60;
        }
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_QUEUE_INTERVAL", NULL, NMAP.queue.sleep, 0, NULL, 0);

        MDBFreeValues(vs);
    } else {
        NMAP.queue.sleep = 4 * 60;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SCMS_USER_THRESHOLD, vs) > 0) { 
        NMAP.scms.userThreshold = atol(vs->Value[0]) - 1;
        if (NMAP.scms.userThreshold < 4) {
            NMAP.scms.userThreshold = 4;
        }

        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_SCMS_USER_THRESHOLD", NULL, NMAP.scms.userThreshold, 0, NULL, 0);

        MDBFreeValues(vs);
    } else {
        NMAP.scms.userThreshold = 4;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SCMS_SIZE_THRESHOLD, vs) > 0) { 
        NMAP.scms.sizeThreshold = atol(vs->Value[0]) * 1024;
        if (NMAP.scms.sizeThreshold < 5120) {
            NMAP.scms.sizeThreshold = 5120;
        }
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_SCMS_SIZE_THRESHOLD", NULL, NMAP.scms.sizeThreshold, 0, NULL, 0);

        MDBFreeValues(vs);
    } else {
        NMAP.scms.sizeThreshold = 5120;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUOTA_MESSAGE, vs) > 0) { 
        NMAP.quota.message = MemStrdup(vs->Value[0]);
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_QUOTA_MESSAGE", NMAP.quota.message, 0, 0, NULL, 0);

        MDBFreeValues(vs);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUOTA_WARNING, vs) > 0) { 
        NMAP.quota.warning = MemStrdup(vs->Value[0]);
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_QUOTA_WARNING", NMAP.quota.warning, 0, 0, NULL, 0);

        MDBFreeValues(vs);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_USE_QUOTA, vs) > 0) { 
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_USE_QUOTA", vs->Value[0], 0, 0, NULL, 0);
        if (vs->Value[0][0] == '3') {
            NMAP.quota.useUser = TRUE;
            NMAP.quota.useSystem = TRUE;
        } else if (vs->Value[0][0] == '1') {
            NMAP.quota.useUser = TRUE;
            NMAP.quota.useSystem = FALSE;
        } else if (vs->Value[0][0] == '2') {
            NMAP.quota.useUser = FALSE;
            NMAP.quota.useSystem = TRUE;
        } else {
            NMAP.quota.useUser = FALSE;
            NMAP.quota.useSystem = FALSE;
        }

        MDBFreeValues(vs);
    } else {
        NMAP.quota.useSystem = FALSE;
        NMAP.quota.useUser = FALSE;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUOTA_VALUE, vs) > 0) { 
        NMAP.quota.defaultValue = atol(vs->Value[0]);
        if (!NMAP.quota.defaultValue) {
            NMAP.quota.defaultValue = 0x7fffffff;
        }
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_NUMERIC, LOG_INFO, 0, "MSGSRV_A_QUOTA_VALUE", NULL, NMAP.quota.defaultValue, 0, NULL, 0);

        MDBFreeValues(vs);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_RTS_ANTISPAM_CONFIG, vs) > 0) { 
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_RTS_ANTISPAM_CONFIG", vs->Value[0], 0, 0, NULL, 0);

        if (sscanf(vs->Value[0], "Enabled:%d Delay:%ld Threshhold:%lu", &NMAP.rts.blockSpam, &NMAP.queue.bounce.interval, &NMAP.queue.bounce.maxCount) != 3) {
            NMAP.rts.blockSpam = FALSE;
        }

        if ((NMAP.queue.bounce.maxCount < 1) || (NMAP.queue.bounce.interval < 1)) {
            NMAP.rts.blockSpam = FALSE;
        }

        MDBFreeValues(vs);
    } else {
        NMAP.rts.blockSpam = FALSE;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_LIMIT_REMOTE_PROCESSING, vs) > 0) { 
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_LIMIT_REMOTE_PROCESSING", vs->Value[0], 0, 0, NULL, 0);

        if (atol(vs->Value[0])) {
            NMAP.defer.enabled = TRUE;
        } else {
            NMAP.defer.enabled = FALSE;
        }

        MDBFreeValues(vs);

        if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_LIMIT_REMOTE_START_WD, vs) > 0) { 
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_LIMIT_REMOTE_START_WD", vs->Value[0], 0, 0, NULL, 0);
            for (count = 1; count < 6; count++) {
                /* Counting weekdays */
                NMAP.defer.start[count] = (unsigned char)atoi(vs->Value[0]);
            }

            MDBFreeValues(vs);
        } else {
            NMAP.defer.enabled = FALSE;
        }

        if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_LIMIT_REMOTE_START_WE, vs) > 0) { 
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_LIMIT_REMOTE_START_WE", vs->Value[0], 0, 0, NULL, 0);

            NMAP.defer.start[0] = (unsigned char)atoi(vs->Value[0]);
            NMAP.defer.start[6] = (unsigned char)atoi(vs->Value[0]);

            MDBFreeValues(vs);
        } else {
            NMAP.defer.enabled = FALSE;
        }

        if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_LIMIT_REMOTE_END_WD, vs) > 0) { 
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_LIMIT_REMOTE_END_WD", vs->Value[0], 0, 0, NULL, 0);
            for (count = 1; count < 6; count++) {
                /* Counting weekdays */
                NMAP.defer.end[count] = (unsigned char)atoi(vs->Value[0]);
            }

            MDBFreeValues(vs);
        } else {
            NMAP.defer.enabled = FALSE;
        }

        if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_LIMIT_REMOTE_END_WE, vs) > 0) { 
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_LIMIT_REMOTE_END_WE", vs->Value[0], 0, 0, NULL, 0);

            NMAP.defer.end[0] = (unsigned char)atoi(vs->Value[0]);
            NMAP.defer.end[6] = (unsigned char)atoi(vs->Value[0]);

            MDBFreeValues(vs);
        } else {
            NMAP.defer.enabled = FALSE;
        }
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_NEW_SHARE_MESSAGE, vs) > 0) { 
        NMAP.newShareMessage = MemStrdup(vs->Value[0]);
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_NEW_SHARE_MESSAGE", NMAP.newShareMessage, 0, 0, NULL, 0);

        MDBFreeValues(vs);
    } else {
        NMAP.newShareMessage = (unsigned char *)MemMalloc(strlen(DEFAULT_NEW_SHARE_MESSAGE) + 1);
        strcpy(NMAP.newShareMessage, DEFAULT_NEW_SHARE_MESSAGE);
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SPOOL_DIRECTORY, vs) > 0) {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_SPOOL_DIRECTORY", vs->Value[0], 0, 0, NULL, 0);

        NMAP.server.bytesPerBlock = XplGetDiskBlocksize(vs->Value[0]);
        MsgCleanPath(vs->Value[0]);

        strcpy(NMAP.path.spool, vs->Value[0]);
        MsgMakePath(NMAP.path.spool);

        MDBFreeValues(vs);
    } else {
        result = FALSE;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_SCMS_DIRECTORY, vs) > 0) {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_SCMS_DIRECTORY", vs->Value[0], 0, 0, NULL, 0);

        MsgCleanPath(vs->Value[0]);

        strcpy(NMAP.path.scms, vs->Value[0]);
        MsgMakePath(NMAP.path.scms);

        /* Abuse qClients path to save stack space; set the real value a few lines down */
        for (count = 0; count < 16; count++) {
            sprintf(NMAP.path.qClients, "%s/%x", NMAP.path.scms, count);
            MsgMakePath(NMAP.path.qClients);
        }

        MDBFreeValues(vs);
    } else {
        result = FALSE;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_MESSAGE_STORE, vs) > 0) {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_MESSAGE_STORE", vs->Value[0], 0, 0, NULL, 0);

        MsgCleanPath(vs->Value[0]);

        strcpy(NMAP.path.mail, vs->Value[0]);
        MsgMakePath(NMAP.path.mail);

        MDBFreeValues(vs);
    } else {
        result = FALSE;
    }

    MsgGetDBFDir(NMAP.path.dbf);
    sprintf(NMAP.path.qClients, "%s/qclients", NMAP.path.dbf);

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_RTS_HANDLING, vs) > 0) { 
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_RTS_HANDLING", vs->Value[0], 0, 0, NULL, 0);
        NMAP.rts.handling = atol(vs->Value[0]);

        MDBFreeValues(vs);
    } else {
        NMAP.rts.handling=RTS_RETURN;
    }

    /* Default the values, in case the attribute exists but doesn't have what we're looking for */
    NMAP.loadMonitor.low = 70;
    NMAP.loadMonitor.high = 90;
    NMAP.queue.limit.trigger = 100;
    NMAP.loadMonitor.interval = 1;
    NMAP.flags &= ~NMAP_FLAG_DEBUG;
    NMAP.queue.limit.concurrent = QLIMIT_CONCURRENT;
    NMAP.queue.limit.sequential = QLIMIT_SEQUENTIAL;

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_QUEUE_TUNING, vs) > 0) { 
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_CONFIGURATION_STRING, LOG_INFO, 0, "MSGSRV_A_QUEUE_TUNING", vs->Value[0], 0, 0, NULL, 0);

        for (used = 0; used < vs->Used; used++) {
            if (XplStrNCaseCmp(vs->Value[used], "concurrent", 10) == 0) {
                sscanf(vs->Value[used], "Concurrent: %ld  Sequential: %ld", &NMAP.queue.limit.concurrent, &NMAP.queue.limit.sequential);
            } else if (XplStrNCaseCmp(vs->Value[used], "Load", 4) == 0) {
                sscanf(vs->Value[used], "Load high: %ld Load Low: %ld Queue Trigger: %ld Interval: %ld", &NMAP.loadMonitor.high, &NMAP.loadMonitor.low, &NMAP.queue.limit.trigger, &NMAP.loadMonitor.interval);
            } else if (XplStrNCaseCmp(vs->Value[used], "Debug", 5) == 0) {
                if (atol(vs->Value[used]) + 6) {
                    NMAP.flags |= NMAP_FLAG_DEBUG;
                } else {
                    NMAP.flags &= ~NMAP_FLAG_DEBUG;
                }
            }
        }

        MDBFreeValues(vs);
    }

    /* Set the globals for loadmonitor */
    NMAP.queue.limit.defaultConcurrent = NMAP.queue.limit.concurrent;
    NMAP.queue.limit.defaultSequential = NMAP.queue.limit.sequential;

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_FORWARD_UNDELIVERABLE, vs) > 0) {
        if (vs->Value[0][0] != '\0') {
            strcpy(NMAP.queue.forwardUndeliverable.address, vs->Value[0]);
            NMAP.queue.forwardUndeliverable.enabled = TRUE;
        } else {
            NMAP.queue.forwardUndeliverable.address[0] = '\0';
            NMAP.queue.forwardUndeliverable.enabled = FALSE;
        }

        MDBFreeValues(vs);
    } else {
        NMAP.queue.forwardUndeliverable.address[0] = '\0';
        NMAP.queue.forwardUndeliverable.enabled = FALSE;
    }

    if (MDBRead(MSGSRV_AGENT_NMAP, MSGSRV_A_CONFIGURATION, vs) > 0) {
        for (used = 0; used < vs->Used; used++) {
            if (XplStrNCaseCmp(vs->Value[used], "DiskSyncPoints=", 15) == 0) {
                NMAP.flushFlags = atol(vs->Value[used] + 15);
            }
        }

        MDBFreeValues(vs);
    }

    /* Some sanity checking on the QLimit stuff to prevent running out of memory */
    if (XplGetMemAvail() < (unsigned long)((STACKSPACE_Q + STACKSPACE_S) * NMAP.queue.limit.sequential)) {
        NMAP.queue.limit.sequential = (XplGetMemAvail() / (STACKSPACE_Q+STACKSPACE_S)) / 2;
        NMAP.queue.limit.concurrent = NMAP.queue.limit.sequential / 2;

        XplBell();
        XplConsolePrintf("NMAPD.NLM: WARNING - Tuning parameters adjusted to %ld par./%ld seq.\r\n", NMAP.queue.limit.concurrent, NMAP.queue.limit.sequential);
        XplBell();

        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_QLIMITS_ADJUSTED, LOG_WARNING, 0, NULL, NULL, NMAP.queue.limit.concurrent, NMAP.queue.limit.sequential, NULL, 0);
        NMAP.queue.limit.defaultConcurrent = NMAP.queue.limit.concurrent;
        NMAP.queue.limit.defaultSequential = NMAP.queue.limit.sequential;
    }

    /* Create Store Paths if they don't exist */
    if (MDBReadDN(NMAP.server.dn, MSGSRV_A_CONTEXT, vs) > 0) {
        stores = MDBCreateValueStruct(NMAP.handle.directory, NULL);
        if (stores) {
            for (used = 0; used < vs->Used; used++) {
                if (MDBRead(vs->Value[used], MSGSRV_A_MESSAGE_STORE, stores) > 0) {
                    MsgCleanPath(stores->Value[0]);
                    MsgMakePath(stores->Value[0]);

                    MDBFreeValues(stores);            
                }
            }

            MDBDestroyValueStruct(stores);
        } else {
            result = FALSE;
        }

        MDBFreeValues(vs);
    }

#if 1
    unlink(NMAP.path.qClients);
#else
    /* Read our lost clients */
    if (stat(NMAP.path.qClients, &sb) == 0) {
        if ((sb.st_size % sizeof(NMAPQueueClient))==0) {
            NMAP.queue.clients.count = sb.st_size / sizeof(NMAPQueueClient);
            NMAP.queue.clients.allocated = NMAP.queue.clients.count + PUSHCLIENTALLOC;
            NMAP.queue.clients.array = MemMalloc(NMAP.queue.clients.allocated * sizeof(NMAPQueueClient));
            if (NMAP.queue.clients.array) {
                fh = fopen(NMAP.path.qClients, "rb");
                if (fh) {
                    fread(NMAP.queue.clients.array, sizeof(NMAPQueueClient), NMAP.queue.clients.count, fh);
                    fclose(fh);
                }
                
                for (used = 0; used < NMAP.queue.clients.count; used++) {
                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_CONFIGURATION, LOGGER_EVENT_RELOADED_QUEUE_AGENT, LOG_WARNING, 0, NULL, NULL, NMAP.queue.clients.array[used].address, NMAP.queue.clients.array[used].queue, &(NMAP.queue.clients.array[used].port), sizeof(NMAP.queue.clients.array[used].port));

                    NMAP.queue.clients.array[used].usageCount = 0;
                    NMAP.queue.clients.array[used].errorCount = 0;
                }

                /* Clean up duplicate entries */
                for (used = 0; used < NMAP.queue.clients.count; used++) {
                    for (index = used + 1; index < NMAP.queue.clients.count; index++) {
                        if ((strcmp(NMAP.queue.clients.array[used].identifier, NMAP.queue.clients.array[index].identifier) == 0) 
                                || (((NMAP.queue.clients.array[used].address == NMAP.queue.clients.array[index].address) 
                                    && (NMAP.queue.clients.array[used].queue == NMAP.queue.clients.array[index].queue) 
                                    && (NMAP.queue.clients.array[used].port == NMAP.queue.clients.array[index].port)))) {
                            if (RemovePushClientIndex(used, TRUE)) {
                                if (used) {
                                    used--;
                                }
                            }

                            index = NMAP.queue.clients.count;
                        }
                    }
                }

                for (used = 0; used < NMAP.queue.clients.count;) {
                    if (NMAP.queue.clients.array[used].queue > 999) {
                        NMAP.queue.clients.array[used].usageCount = 1;
                        RemovePushClientIndex(used, TRUE);
                    } else {
                        used++;
                    }
                }
            } else {
                NMAP.queue.clients.count = 0;
                NMAP.queue.clients.allocated = 0;
            }
        }

        UPDATE_QUEUE_CLIENTS_REGISTERED;
    }
#endif

    MDBDestroyValueStruct(vs);

    return(result);
}

#if defined(NETWARE) || defined(LIBC) || defined(WIN32)
int _NonAppCheckUnload(void)
{
    static BOOL    checked = FALSE;
    XplThreadID    id;

    if (!checked) {
        NMAP.state = NMAP_STOPPING;
        checked = TRUE;

        XplWaitOnLocalSemaphore(NMAP.sem.shutdown);

        id = XplSetThreadGroupID(NMAP.id.group);
        if (NMAP.server.conn != NULL) {
            ConnFree(NMAP.server.conn);
            NMAP.server.conn = NULL;
        }
        XplSetThreadGroupID(id);

        XplWaitOnLocalSemaphore(NMAP.sem.main);
    }

    return(0);
}
#endif

static void 
SignalHandler(int sigtype)
{
    switch(sigtype) {
        case SIGHUP: {
            if (NMAP.state < NMAP_UNLOADING) {
                NMAP.state = NMAP_UNLOADING;
            }

            break;
        }
        case SIGINT:
        case SIGTERM: {
            if (NMAP.state == NMAP_STOPPING) {
                XplUnloadApp(getpid());
            } else if (NMAP.state < NMAP_STOPPING) {
                NMAP.state = NMAP_STOPPING;
            }

            break;
        }

        default: {
            break;
        }
    }

    return;
}

XplServiceCode(SignalHandler)

int
XplServiceMain(int argc, char *argv[])
{
    int ccode;
    XplThreadID id;
    BOOL display = FALSE;
    BOOL recover = FALSE;

    if (XplSetEffectiveUser(MsgGetUnprivilegedUser()) < 0) {
	XplConsolePrintf("hulanmap: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
	return 1;
    }

    XplSignalHandler(SignalHandler);

    NMAP.handle.logging = NULL;
    NMAP.handle.directory = NULL;
    NMAP.universalCounter = 1;
    NMAP.client.sleepTime = (5 * 60);
    NMAP.queue.maxLinger = 0;
    NMAP.queue.sleep = 0;
    NMAP.server.ipAddress = 0;
    NMAP.server.bound = FALSE;
    NMAP.trusted.hosts = NULL;
    NMAP.trusted.count = 0;
    NMAP.scms.userThreshold = 4;
    NMAP.scms.sizeThreshold = 5120;
    NMAP.quota.useSystem = FALSE;
    NMAP.quota.useUser = FALSE;
    NMAP.quota.defaultValue = 0x7FFFFFFF;
    NMAP.quota.message = NULL;
    NMAP.quota.warning = NULL;
    NMAP.queue.minimumFree = 5 * 1024 * 1024;
    NMAP.flags = 0;
    NMAP.time.startup = 0;
    NMAP.time.global = 0;
    NMAP.flushFlags = 0;
    NMAP.newShareMessage = NULL;
    NMAP.nameSpace.delimiter = '/';
    strcpy(NMAP.path.mail, XPL_DEFAULT_MAIL_DIR);
    strcpy(NMAP.path.scms, XPL_DEFAULT_SCMS_DIR);
    strcpy(NMAP.path.spool, XPL_DEFAULT_SPOOL_DIR);
    sprintf(NMAP.path.qClients, "%s/qclients", XPL_DEFAULT_DBF_DIR);
    strcpy(NMAP.path.dbf, XPL_DEFAULT_DBF_DIR);
    sprintf(NMAP.path.qDomains, "%s/qdomains.db", XPL_DEFAULT_DBF_DIR);
    strcpy(NMAP.nameSpace.userPrefix, "Shares");
    strcpy(NMAP.nameSpace.publicPrefix, "Public Shares");

    NMAP.rts.maxBodySize = 0;

    NMAP.queue.inhibitSpam = TRUE;
    NMAP.queue.forwardUndeliverable.enabled = FALSE;
    NMAP.queue.clients.count = 0;

    UPDATE_QUEUE_CLIENTS_REGISTERED;

    XplSafeWrite(NMAP.server.active, 0);
    XplSafeWrite(NMAP.client.worker.active, 0);
    XplSafeWrite(NMAP.queue.worker.active, 0);

    XplSafeWrite(NMAP.client.worker.idle, 0);
    XplSafeWrite(NMAP.client.worker.active, 0);
    XplSafeWrite(NMAP.client.worker.maximum, 100000);

    XplSafeWrite(NMAP.stats.nmap.toNMAP, 0);
    XplSafeWrite(NMAP.stats.storedLocal.bytes, 0);
    XplSafeWrite(NMAP.stats.storedLocal.count, 0);
    XplSafeWrite(NMAP.stats.storedLocal.recipients, 0);
    XplSafeWrite(NMAP.stats.queuedLocal.messages, 0);
    XplSafeWrite(NMAP.stats.spam, 0);
    XplSafeWrite(NMAP.stats.deliveryFailed.local, 0);
    XplSafeWrite(NMAP.stats.deliveryFailed.remote, 0);

    if (MemoryManagerOpen(MSGSRV_AGENT_NMAP)) {
        LoadProtocolCommandTree(&NMAP.commands, NMAPProtocolCommands);
    } else {
        XplBell();
        XplConsolePrintf("NMAPD: Memory manager failed to initialize; exiting!\r\n");
        XplBell();

        exit(-1);
    }

    XplOpenLocalSemaphore(NMAP.sem.main, 0);
    XplOpenLocalSemaphore(NMAP.sem.shutdown, 1);

    NMAP.id.group = XplGetThreadGroupID();

    NMAP.client.pool = MemPrivatePoolAlloc("NMAP Connections", sizeof(NMAPClient), 0, 8192, TRUE, FALSE, NMAPConnectionAllocCB, NMAPConnectionCloseCB, NULL);
    if (NMAP.client.pool != NULL) {
        if (InitSpoolEntryIDLocks() == FALSE) {
            XplBell();
            XplConsolePrintf("NMAPD: Out of memory - cannot allocate spool locks; exiting!\r\n");
            XplBell();

            MemPrivatePoolFree(NMAP.client.pool);

            MemoryManagerClose(MSGSRV_AGENT_NMAP);

            exit(-1);
        }
    } else {
        XplBell();
        XplConsolePrintf("NMAPD: Out of memory - cannot allocate connection pool; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_NMAP);

        exit(-1);
    }

    ConnStartup(NMAP_CONNECTION_TIMEOUT, TRUE);

    MDBInit();
    if ((NMAP.handle.directory = (MDBHandle)MsgInit()) == NULL) {
        XplBell();
        XplConsolePrintf("NMAPD: Invalid directory credentials; exiting!\r\n");
        XplBell();

        MemoryManagerClose(MSGSRV_AGENT_NMAP);

        exit(-1);
    }

    NMAPInitialize(NMAP.handle.directory);

    NLockInit(8000);

    NmapSqlLocksInit(2048);

    SetCurrentNameSpace(NWOS2_NAME_SPACE);

    MsgGetServerDN(NMAP.server.dn);

    XplOpenLocalSemaphore(NMAP.client.semaphore, 1);
    XplOpenLocalSemaphore(NMAP.client.worker.todo, 0);

    XplOpenLocalSemaphore(NMAP.queue.semaphore, 1);
    XplOpenLocalSemaphore(NMAP.queue.worker.todo, 0);

    XplOpenLocalSemaphore(NMAP.guid.semaphore, 0);
    GuidReset();
    XplSignalLocalSemaphore(NMAP.guid.semaphore);

    for (ccode = 0; ccode <= MBOX_LOCK_SEM_ARRAY_SIZE; ccode++) {
        XplOpenLocalSemaphore(NMAP.lock.mailbox.sems[ccode], 1);
    }

    for (ccode = 0; ccode <= CSTORE_LOCK_SEM_ARRAY_SIZE; ccode++) {
        XplOpenLocalSemaphore(NMAP.lock.calendar.sems[ccode], 1);
    }

    memset(NMAP.lock.mailbox.array, 0, MBOX_LOCK_ARRAY_SIZE + 1);
    memset(NMAP.lock.calendar.array, 0, CSTORE_LOCK_ARRAY_SIZE + 1);
    
    XplRWLockInit(&NMAP.lock.config);

    NMAP.handle.logging = LoggerOpen("hulanmap");
    
    if (NMAP.handle.logging != NULL) {
        ;
    } else {
        XplConsolePrintf("NMAPD: Unable to initialize Nsure Audit.  Logging disabled.\r\n");
    }

    InitSearchModule();

    SetCurrentNameSpace(NWOS2_NAME_SPACE);

#if !defined(NMAP_SQL_CALENDARS)
    if (ReadConfiguration(&recover)) {
#else
    if (OpenICal2Parser() && ReadConfiguration(&recover)) {
#endif
        if (argc > 1) {
            for (ccode = 1; ccode < argc; ccode++)  {
                if (XplStrCaseCmp(argv[ccode], "recover")==0) {
                    recover = TRUE;
                } else if (XplStrCaseCmp(argv[ccode], "showload")==0) {
                    display = TRUE;
                }
            }
        }

        /* Create and bind the server connection */
        if (ServerSocketInit() < 0) {
            XplConsolePrintf("hulanmap: Exiting.\n");
            return 1;
        }

        /* Done binding to ports, drop privs permanently */
        if (XplSetRealUser(MsgGetUnprivilegedUser()) < 0) {
            XplConsolePrintf("hulanmap: Could not drop to unprivileged user '%s', exiting.\n", MsgGetUnprivilegedUser());
            return 1;
        }

        /* Start the Queue management */
        ccode = CreateQueueThreads(recover);
        if (!ccode) {
            /* Start the Disk management */
            XplBeginCountedThread(&id, CheckDiskspace, 8192, NULL, ccode, NMAP.server.active);

            /* Start the Load management */
            XplBeginCountedThreadGroup(&id, LoadMonitor, 8192, (void *)display, ccode, NMAP.server.active);

            /* Start the Configuration monitor */
            XplBeginCountedThread(&id, ConfigMonitor, 8192, NULL, ccode, NMAP.server.active);
        } else {
            XplConsolePrintf("hulanmap: Could not initialize NMAP queue.\r\n");
            return(1);
        }

        NMAP.server.ssl.config.method = SSLv23_server_method;
        NMAP.server.ssl.config.options = SSL_OP_DONT_INSERT_EMPTY_FRAGMENTS;
        NMAP.server.ssl.config.mode = SSL_MODE_AUTO_RETRY;
        NMAP.server.ssl.config.cipherList = NULL;
        NMAP.server.ssl.config.certificate.type = SSL_FILETYPE_PEM;
        NMAP.server.ssl.config.certificate.file = MsgGetTLSCertPath(NULL);
        NMAP.server.ssl.config.key.type = SSL_FILETYPE_PEM;
        NMAP.server.ssl.config.key.file = MsgGetTLSKeyPath(NULL);

        NMAP.server.ssl.context = ConnSSLContextAlloc(&(NMAP.server.ssl.config));
        if (NMAP.server.ssl.context) {
            NMAP.server.ssl.enable = TRUE;
        }

        /* Management Client Startup */
        if ((ManagementInit(MSGSRV_AGENT_NMAP, NMAP.handle.directory)) 
                && (ManagementSetVariables(GetNMAPManagementVariables(), GetNMAPManagementVariablesCount())) 
                && (ManagementSetCommands(GetNMAPManagementCommands(), GetNMAPManagementCommandsCount()))) {
            XplBeginThread(&id, ManagementServer, DMC_MANAGEMENT_STACKSIZE, NULL, ccode);
        }

        InternalSetServerState("Running");

        /* Start the NMAP Server */
        XplStartMainThread(PRODUCT_SHORT_NAME, &id, NMAPServer, 8192, NULL, ccode);
    } else {
        MsgShutdown();

        MemPrivatePoolFree(NMAP.client.pool);

        MemoryManagerClose(MSGSRV_AGENT_NMAP);
    }

    XplUnloadApp(XplGetThreadID());

    return(0);
}
