/****************************************************************************
 *
 * Copyright (c) 1997-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <logger.h>
#include <mdb.h>
#include <nmap.h>
#include <nmlib.h>

#include "nmapdp.h"

int
ConnectToNMAPShare(NMAPClient *client, unsigned char *owner, unsigned char *name, unsigned long type, unsigned long *permissions)
{
    int ccode;
    unsigned int count;
    unsigned int length;
    unsigned long used;
    unsigned char *ptr;
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    unsigned char buffer[CONN_BUFSIZE + 1];
    BOOL result;
    struct sockaddr_in nmap;
    MDBValueStruct *vs;
    Connection **conn;

    memset(&nmap, 0, sizeof(struct sockaddr_in));

    nmap.sin_family = AF_INET;
    nmap.sin_port = htons(NMAP_PORT);

    if ((owner[0] != '-') || (owner[1] != '\0')) {
        result = MsgFindObject(owner, dn, NULL, &nmap.sin_addr, NULL);
    } else {
        strcpy(dn, MSGSRV_ROOT);
        result = TRUE;
    }

    if (result) {
        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
            case NMAP_SHARE_MAILBOX: {
                length = sprintf(buffer, "MB%s\r", name);

                count = strlen(client->user);
                memcpy(buffer + length, client->user, count);
                count += length;

                buffer[count++] = '\r';
                buffer[count] = '\0';

                conn = &(client->share.mailbox.conn);
                break;
            }

            case NMAP_SHARE_CALENDAR: {
                length = sprintf(buffer, "CA%s\r", name);

                count = strlen(client->user);
                memcpy(buffer + length, client->user, count);
                count += length;

                buffer[count++] = '\r';
                buffer[count] = '\0';

                conn = &(client->share.calendar.conn);
                break;
            }

            case NMAP_SHARE_ADDRESS_BOOK: {
                return(1000);
            }

            default: {
                return(5004);
            }
        }

        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);

        ccode = 0;
        MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs);
        for (used = 0; used < vs->Used; used++) {
            if ((XplStrNCaseCmp(vs->Value[used], buffer, count) != 0) 
                    && ((XplStrNCaseCmp(vs->Value[used], buffer, length) != 0) 
                            || (vs->Value[used][length] != '-') 
                            || (vs->Value[used][length + 1] != '\r'))) {
                continue;
            }

            if (permissions != NULL) {
                ptr = strrchr(vs->Value[used], '\r');
                if (ptr) {
                    *permissions = atol(++ptr);
                } else {
                    *permissions = 0;
                }
            }

            ccode = 1;
            break;
        }

        MDBDestroyValueStruct(vs);

        if (ccode == 0) {
            return(4240);
        }
    } else {
        return(4224);
    }

    if ((*conn = NMAPConnect(NULL, &nmap)) != NULL) {
        if (NMAPAuthenticate(*conn, buffer, CONN_BUFSIZE)) {
            ccode = NMAPSendCommandF(*conn, "FLAG %lu\r\n", (long unsigned int)(NMAP_SHOWDELETED | NMAP_OOBMESSAGES));
            if (ccode != -1) {
                ccode = NMAPReadAnswer(*conn, buffer, CONN_BUFSIZE, TRUE);
                if (ccode != -1) {
                    if (ccode == NMAP_FLAGS_SET) {
                        return(ccode);
                    }
                }
            }
        }

        NMAPQuit(*conn);
        ConnFree(*conn);
        *conn = NULL;

        ccode = 5004;
    } else {
        return(5004);
    }

    return(ccode);
}

BOOL 
SelectNMAPShare(NMAPClient *client, unsigned char *owner, unsigned char *name, unsigned long type)
{
    unsigned long ccode;
    Connection *conn;

    if (type == NMAP_SHARE_MAILBOX) {
        conn = client->share.mailbox.conn;
    } else if (type == NMAP_SHARE_CALENDAR) {
        conn = client->share.calendar.conn;
    } else {
        return(FALSE);
    }

    if (NMAPSendCommandF(conn, "USER %s\r\n", owner)) {
        ccode = NMAPReadAnswer(conn, client->line, CONN_BUFSIZE, FALSE);
        if (ccode == 1000) {
            if (type == NMAP_SHARE_MAILBOX) {
                if (NMAPSendCommandF(conn, "MBOX %s\r\n", name)) {
                    ccode = NMAPReadAnswer(conn, client->line, CONN_BUFSIZE, FALSE);
                    if (ccode == 1000) {
                        return(SyncSharedMaildrop(client, NULL));
                    }
                }
            } else {
                if (NMAPSendCommandF(conn, "CSOPEN %s\r\n", name)) {
                    ccode = NMAPReadAnswer(conn, client->line, CONN_BUFSIZE, FALSE);
                    if (ccode == 1000) {
                        return(SyncSharedCalendar(client, NULL, FALSE));
                    }
                }
            }
        }
    }

    return(FALSE);
}

BOOL 
IsAvailableShare(unsigned char *user, unsigned char *name, unsigned long type, unsigned char **owner, unsigned char **trueName)
{
    unsigned long used;
    unsigned char *ptr;
    unsigned char *displayName;
    unsigned char pattern[3];
    unsigned char dn[MDB_MAX_UTF8_OBJECT_CHARS + 1];
    BOOL result = FALSE;
    MDBValueStruct *vs;

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (MsgFindObject(user, dn, NULL, NULL, vs)) {
        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
            case NMAP_SHARE_MAILBOX: { strcpy(pattern, "MB"); break; }
            case NMAP_SHARE_CALENDAR: { strcpy(pattern, "CA"); break; }
            case NMAP_SHARE_ADDRESS_BOOK: { strcpy(pattern, "AB"); break; }
            default: { MDBDestroyValueStruct(vs); return(FALSE); }
        }

        MDBFreeValues(vs);
        MDBRead(dn, MSGSRV_A_AVAILABLE_SHARES, vs);
        for (used = 0; used < vs->Used; used++) {
            if (strncmp(vs->Value[used], pattern, 2) != 0) {
                continue;
            }

            displayName = strrchr(vs->Value[used], '\r');
            if (displayName && (XplStrCaseCmp(++displayName, name) == 0)) {
                result = TRUE;

                ptr = strchr(vs->Value[used], '\r');

                if (owner != NULL) {
                    *(--displayName) = '\0';
                    *owner = MemStrdup(ptr + 1);
                }

                if (trueName != NULL) {
                    *ptr = '\0';
                    *trueName = MemStrdup(vs->Value[used] + 2);
                }

                break;
            }
        }
    }

    if (!result) {
        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
            case NMAP_SHARE_MAILBOX: { strcpy(pattern, "MB"); break; }
            case NMAP_SHARE_CALENDAR: { strcpy(pattern, "CA"); break; }
            case NMAP_SHARE_ADDRESS_BOOK: { strcpy(pattern, "AB"); break; }
            default: { MDBDestroyValueStruct(vs); return(FALSE); }
        }

        MDBFreeValues(vs);
        MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);
        for (used = 0; used < vs->Used; used++) {
            if (strncmp(vs->Value[used], pattern, 2) != 0) {
                continue;
            }

            displayName = strrchr(vs->Value[used], '\r');
            if (displayName && (XplStrCaseCmp(++displayName, name) == 0)) {
                result = TRUE;

                ptr = strchr(vs->Value[used], '\r');

                if (owner != NULL) {
                    *(--displayName) = '\0';
                    *owner = MemStrdup(ptr + 1);
                }

                if (trueName != NULL) {
                    *ptr = '\0';
                    *trueName = MemStrdup(vs->Value[used] + 2);
                }

                break;
            }
        }
    }

    MDBDestroyValueStruct(vs);

    return(result);
}

BOOL 
SetShareSubscription(const unsigned char *store, unsigned char *user, unsigned long hash, unsigned char *resource, unsigned long type, BOOL subscribed)
{
    unsigned int len;
    unsigned long flags;
    unsigned char path[XPL_MAX_PATH + 1];
    unsigned char buffer[CONN_BUFSIZE + 1];
    BOOL result;
    FILE *index;
    NLockStruct *lock;

    lock = NULL;

    switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
        case NMAP_SHARE_MAILBOX: {
            if (ReadNLockAquire(&lock, &hash, user, resource, 3000) == NLOCK_AQUIRED) {
                if (WriteNLockAquire(lock, 3000) == NLOCK_AQUIRED) {
                    sprintf(path, "%s/%s/%s.sdx", store, user, resource);
                    break;
                }
                ReadNLockRelease(lock);
                lock = NULL;
            }

            return(FALSE);
        }

        case NMAP_SHARE_CALENDAR: {
            sprintf(path, "%s/%s/%s.sdc", store, user, resource);
            WaitforAndLockCalendar(store, user, resource);
            break;
        }

        case NMAP_SHARE_ADDRESS_BOOK: {
            return(TRUE);
        }

        default: {
            return(FALSE);
        }
    }

    result = FALSE;
    if (access(path, 0) == 0) {
        index = fopen(path, "r+b");
    } else {
        /* The index hasn't been created yet. */
        index = fopen(path, "wb");
        if (index != NULL) {
            fseek(index, 0, SEEK_END);

            switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR)) {
                case NMAP_SHARE_MAILBOX: {  fprintf(index, NMAP_IDX_VERSION"\r\n");     break;  }
                case NMAP_SHARE_CALENDAR: { fprintf(index, NMAP_CAL_IDX_VERSION"\r\n"); break;  }
            }

            fprintf(index, "%010lu\r\n0000000000\r\n0000000000\r\n0000000000\r\n0000000000\r\n", (long unsigned int)RESOURCE_FLAG_UNSUBSCRIBED);
        }
    }

    if (index != NULL) {
        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR)) {
            case NMAP_SHARE_MAILBOX: {  fseek(index, 9, SEEK_SET);  break;  }
            case NMAP_SHARE_CALENDAR: { fseek(index, 10, SEEK_SET); break;  }
        }

        if (fgets(buffer, CONN_BUFSIZE, index) != NULL) {
            flags = atol(buffer);
        } else {
            fseek(index, 0, SEEK_SET);

            switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR)) {
                case NMAP_SHARE_MAILBOX: {  fprintf(index, NMAP_IDX_VERSION"\r\n");     break;  }
                case NMAP_SHARE_CALENDAR: { fprintf(index, NMAP_CAL_IDX_VERSION"\r\n"); break;  }
            }

            fprintf(index, "%010lu\r\n0000000000\r\n0000000000\r\n0000000000\r\n0000000000\r\n", (long unsigned int)RESOURCE_FLAG_UNSUBSCRIBED);

            flags = RESOURCE_FLAG_UNSUBSCRIBED;
        }

        if ((subscribed) && (flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
            flags &= ~RESOURCE_FLAG_UNSUBSCRIBED;
        } else if ((subscribed == FALSE) && !(flags & RESOURCE_FLAG_UNSUBSCRIBED)) {
            flags |= RESOURCE_FLAG_UNSUBSCRIBED;
        }

        len = sprintf(buffer, "%010lu\r\n", flags);

        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR)) {
            case NMAP_SHARE_MAILBOX: {  fseek(index, 9, SEEK_SET);  break;  }
            case NMAP_SHARE_CALENDAR: { fseek(index, 10, SEEK_SET); break;  }
        }

        fwrite(buffer, sizeof(unsigned char), len, index);

        fclose(index);
        result = TRUE;
    } else {
        result = FALSE;
    }

    switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR)) {
        case NMAP_SHARE_MAILBOX: {
            WriteNLockRelease(lock);
            ReadNLockRelease(lock);
            lock = NULL;
            break;
        }

        case NMAP_SHARE_CALENDAR: {
            UnlockCalendar(store, user, resource);
            break;
        }
    }

    return(result);
}

#if 0
/* fixme - enhancment (support renaming shared resources)? */
BOOL 
RenameAvailableShare(NMAPClient *client, unsigned long ResourceType, unsigned char *OldResourceName, unsigned char *NewResourceName)
{
    unsigned char    dn[MDB_MAX_UTF8_OBJECT_CHARS + 1];
    unsigned char    buffer[XPL_MAX_PATH + 1];
    BOOL                result = FALSE;
    MDBValueStruct    *V;

    switch(ResourceType & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
        case NMAP_SHARE_MAILBOX: {
            sprintf(buffer, "%s/%s/%s.box", client->store, client->user, NewResourceName);
            if (access(buffer, 0) == 0) {
                ConnWrite(client->conn, MSG4226MBOXEXISTS, sizeof(MSG4226MBOXEXISTS) - 1);
                return(FALSE);
            }
        }

        case NMAP_SHARE_CALENDAR: {
            sprintf(buffer, "%s/%s/%s.cal", client->store, client->user, NewResourceName);
            if (access(buffer, 0) == 0) {
                ConnWrite(client->conn, MSG4226CALEXISTS, sizeof(MSG4226CALEXISTS) - 1);
                return(FALSE);
            }
        }

        case NMAP_SHARE_ADDRESS_BOOK: {
            break;
        }

        default: {
            return(FALSE);
        }
    }

    V = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (MsgFindObject(client->user, client->dn, NULL, NULL, V)) {
        int    i;
        int    share;
        unsigned char    name[XPL_MAX_PATH + 1];
        unsigned char    owner[MAXEMAILNAMESIZE + 1];
        unsigned char    displayName[XPL_MAX_PATH + 1];
        unsigned char    pattern[3];

        switch(ResourceType & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
            case NMAP_SHARE_MAILBOX: {            strcpy(pattern, "MB");    break;    }
            case NMAP_SHARE_CALENDAR: {        strcpy(pattern, "CA");    break;    }
            case NMAP_SHARE_ADDRESS_BOOK: {    strcpy(pattern, "AB");    break;    }
        }

        MDBFreeValues(V);
        MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, V);

        /*    Check for name collisions with an existing share.    */
        for (i = 0, share = -1; i < V->Used; i++) {
            if (strncmp(V->Value[i], pattern, 2) != 0) {
                continue;
            }

            if (sscanf(V->Value[i], "%s\r%s\r%s", name, owner, displayName) == 3) {
                if (XplStrCaseCmp(displayName, OldResourceName) == 0) {
                    share = i;
                } else if (XplStrCaseCmp(displayName, NewResourceName) == 0) {
                    ConnWrite(client->conn, MSG4226MBOXEXISTS, sizeof(MSG4226MBOXEXISTS) - 1);
                    MDBDestroyValueStruct(V);
                    return(FALSE);
                }
            }
        }

        /*    Change the display name on the original share.    */
        if (share != -1) {
            MDBFreeValue(share, V);

            sprintf(buffer, "%s\r%s\r%s", name, owner, NewResourceName);
            MDBAddValue(buffer, V);
            MDBWrite(client->dn, MSGSRV_A_AVAILABLE_SHARES, V);

            switch(ResourceType & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
                case NMAP_SHARE_MAILBOX: {
                    sprintf(name, "%s/%s/%s.sdx", client->store, client->user, OldResourceName);
                    sprintf(buffer, "%s/%s/%s.sdx", client->store, client->user, NewResourceName);
                    break;
                }

                case NMAP_SHARE_CALENDAR: {
                    sprintf(name, "%s/%s/%s.sdc", client->store, client->user, OldResourceName);
                    sprintf(buffer, "%s/%s/%s.sdc", client->store, client->user, NewResourceName);
                    break;
                }

                case NMAP_SHARE_ADDRESS_BOOK: {
                    ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                    MDBDestroyValueStruct(V);
                    return(TRUE);
                }
            }

            rename(name, buffer);
            SetLongPathName(buffer);

            ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            MDBDestroyValueStruct(V);
            return(TRUE);
        }

        ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
    } else {
        ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1);
    }

    MDBDestroyValueStruct(V);

    return(result);
}
#endif

BOOL 
RemoveAvailableShare(NMAPClient *client, unsigned long type, unsigned char *resource)
{
    unsigned int count;
    unsigned long used;
    unsigned char name[XPL_MAX_PATH + 1];
    unsigned char owner[MAXEMAILNAMESIZE + 1];
    unsigned char displayName[XPL_MAX_PATH + 1];
    unsigned char pattern[3];
    unsigned char dn[MDB_MAX_UTF8_OBJECT_CHARS + 1];
    BOOL result = FALSE;
    MDBValueStruct *vs;

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
        switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
            case NMAP_SHARE_MAILBOX: {      strcpy(pattern, "MB"); break; }
            case NMAP_SHARE_CALENDAR: {     strcpy(pattern, "CA"); break; }
            case NMAP_SHARE_ADDRESS_BOOK: { strcpy(pattern, "AB"); break; }
            default: { MDBDestroyValueStruct(vs); return(FALSE); }
        }

        MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
        for (used = 0; used < vs->Used; used++) {
            if (strncmp(vs->Value[used], pattern, 2) != 0) {
                continue;
            }

            if ((sscanf(vs->Value[used], "%s\r%s\r%s", name, owner, displayName) == 3) 
                    && (XplStrCaseCmp(displayName, resource) == 0)) {
                MDBFreeValue(used, vs);
                MDBWrite(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
                MDBFreeValues(vs);

                switch(type & (NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK)) {
                    case NMAP_SHARE_MAILBOX: {  sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, resource); break; }
                    case NMAP_SHARE_CALENDAR: { sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, resource); break; }
                }

                unlink(client->path);

                if (MsgFindObject(owner, dn, NULL, NULL, NULL)) {
                    count = sprintf(displayName, "%s\r%s\r", name, client->user);

                    MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs);
                    for (used = 0; used < vs->Used; used++) {
                        if (XplStrNCaseCmp(vs->Value[used], displayName, count) != 0) {
                            continue;
                        }

                        MDBFreeValue(used, vs);
                        MDBWrite(dn, MSGSRV_A_OWNED_SHARES, vs);
                        MDBFreeValues(vs);
                        break;
                    }
                }

                result = TRUE;
                break;
            }
        }
    }

    MDBDestroyValueStruct(vs);

    return(result);
}

BOOL 
RemoveAvailableShares(NMAPClient *client)
{
    unsigned int count;
    unsigned long used;
    unsigned long used2;
    unsigned char dn[MDB_MAX_UTF8_OBJECT_CHARS + 1];
    unsigned char owner[MAXEMAILNAMESIZE + 1];
    unsigned char name[XPL_MAX_PATH + 1];
    BOOL result;
    MDBValueStruct *vs;
    MDBValueStruct *oVS;

    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    oVS = MDBCreateValueStruct(NMAP.handle.directory, NULL);

    if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
        MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
        result = TRUE;
        for (used = 0; used < vs->Used; used++) {
            if (sscanf(vs->Value[used], "%s\r%s\r%*s", name, owner) == 2) {
                result = FALSE;
                if (MsgFindObject(owner, dn, NULL, NULL, NULL)) {
                    count = sprintf(client->line, "%s\r%s\r", name, client->user);

                    MDBRead(dn, MSGSRV_A_OWNED_SHARES, oVS);
                    for (used2 = 0; used2 < oVS->Used; used2++) {
                        if (XplStrNCaseCmp(oVS->Value[used2], client->line, count) != 0) {
                            continue;
                        }

                        MDBFreeValue(used2, oVS);
                        MDBWrite(dn, MSGSRV_A_OWNED_SHARES, oVS);

                        result = TRUE;

                        break;
                    }

                    MDBFreeValues(oVS);
                }

                if (result) {
                    continue;
                }
            }

            /* fixme - suppress missing associations and just continue? */
            result = FALSE;
            break;
        }

        MDBFreeValues(vs);

        if (result) {
            MDBWrite(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
        }
    } else {
        result = FALSE;
    }

    MDBDestroyValueStruct(vs);
    MDBDestroyValueStruct(oVS);

    return(result);
}

BOOL 
SyncSharedMaildrop(NMAPClient *client, BOOL *changed)
{
    int count;
    unsigned int i;
    unsigned long localUsed;
    unsigned long msgCount;
    unsigned long id;
    unsigned long idLocal;
    unsigned long indexStartingOffset;
    unsigned char *ptr;
    BOOL result;
    FILE *index;
    struct stat sb;
    MessageInfoStruct *msgInfo;
    MessageInfoStruct *mLocal;
    MessageInfoStruct mIS;

    if ((client->mailbox.name[0] != '\0') 
            && (client->user[0] != '\0') 
            && (client->share.mailbox.conn) 
            && !(client->share.flags & NMAP_SHARE_MAILBOX_DENIED) 
            && (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED)) {
        if (client->mailbox.fh != NULL) {
            fclose(client->mailbox.fh);
            client->mailbox.fh = NULL;
        }

        if (changed) {
            *changed = FALSE;
        }
    } else {
        return(FALSE);
    }

    /* Open the existing index file for the shared mailbox; or, create a new index file for the shared mailbox.
       Read, or initialize, the mailbox flags, size, count, id and next UID.    */
ReSyncSharedMaildrop:
    client->mailbox.newCount = 0;
    client->mailbox.newSize = 0;

    sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
    index = fopen(client->path, "rb");
    if (index != NULL) {
        stat(client->path, &sb);
        if (client->mailbox.indexTime != sb.st_mtime) {
            if (changed) {
                *changed = TRUE;
            }

            if (fgets(client->line, CONN_BUFSIZE, index) != NULL) {
                if ((client->line[0] == 'N') && (client->line[6] == '4')) {
                    fgets(client->line, CONN_BUFSIZE, index);
                    client->mailbox.flags = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->mailbox.size = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    msgCount = client->mailbox.message.count;
                    client->mailbox.message.count = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->mailbox.id = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->mailbox.nextUID = atol(client->line);

                    if (ftell(index) == NMAP_IDX_HEADERSIZE) {
                        indexStartingOffset = NMAP_IDX_HEADERSIZE;
                    } else {
                        fclose(index);
                        index = NULL;

                        unlink(client->path);

                        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, -1, 0, NULL, 0);

                        goto ReSyncSharedMaildrop;
                    }
                } else {
                    /* Unknown index file format.  Abort. */
                    goto FailSyncingSharedMaildrop;
                }
            } else {
                /* fixme - possibly a zero length indexfile.  Close it, destroy it and restart. */
                fclose(index);
                index = NULL;

                unlink(client->path);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, -1, 0, NULL, 0);

                goto ReSyncSharedMaildrop;
            }

            /* Ensure that we have enough elements in the message info array for the entries within the local index. */
            if (client->mailbox.message.info != NULL) {
                localUsed = client->mailbox.message.used;

                if (client->mailbox.message.count > client->mailbox.message.allocated) {
                    if (AddMsgInfo(client, (client->mailbox.message.count - client->mailbox.message.allocated) + 36)) {
                        ;
                    } else {
                        goto FailSyncingSharedMaildrop;    
                    }
                }

                client->mailbox.message.used = client->mailbox.message.count;
            } else {
                localUsed = 0;

                client->mailbox.message.allocated = client->mailbox.message.count + 36;
                client->mailbox.message.info = (MessageInfoStruct *)MemMalloc(client->mailbox.message.allocated * sizeof(MessageInfoStruct));
                if (client->mailbox.message.info != NULL) {
                    client->mailbox.message.used = client->mailbox.message.count;
                } else {
                    goto FailSyncingSharedMaildrop;
                }
            }

            if (changed != NULL) {
                *changed = FALSE;
            }

            /* Re-Sync the local shared index file with memory to detect deltas. */
            if (sb.st_mtime >= NMAP.time.startup) {
                fseek(index, indexStartingOffset + (localUsed * sizeof(MessageInfoStruct)), SEEK_SET);
                for (i = localUsed, msgInfo = client->mailbox.message.info; i < client->mailbox.message.count; i++, msgInfo++) {
                    fread(msgInfo, sizeof(MessageInfoStruct), 1, index);
                }
            } else {
                for (i = 0, msgInfo = client->mailbox.message.info; i < client->mailbox.message.count; i++, msgInfo++) {
                    fread(msgInfo, sizeof(MessageInfoStruct), 1, index);

                    if (msgInfo->TailSig == NMAP_IDX_LINE_TAIL_SIGNATURE) {
                        continue;
                    }

                    /* fixme - this message information block is invalid; toss the index and restart; repair it? */
                    client->mailbox.message.count = 0;
                    client->mailbox.message.used = 0;

                    fclose(index);

                    unlink(client->path);

                    XplConsolePrintf("NMAPD: Unable to use index file %s for user %s; entry %d corrupted.\r\n", client->mailbox.name, client->user, i);
                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->mailbox.name, i, 0, NULL, 0);

                    goto ReSyncSharedMaildrop;
                }
            }

            client->mailbox.indexTime = sb.st_mtime;
        }

        fclose(index);
        index = NULL;
    } else {
        /* No index exists at this time; create a new one. */
        client->mailbox.flags = RESOURCE_FLAG_UNSUBSCRIBED;

        if (client->mailbox.unseenFlags) {
            MemFree(client->mailbox.unseenFlags);
            client->mailbox.unseenFlags = NULL;
        }

        if (client->mailbox.message.info != NULL) {
            MemFree(client->mailbox.message.info);
            client->mailbox.message.info = NULL;
        }

        localUsed = 0;
        client->mailbox.message.count = 0;
        client->mailbox.message.used = 0;
        client->mailbox.message.allocated = 0;
        client->mailbox.size = 0;
        client->mailbox.id = 0;

        client->mailbox.indexTime = 0;
        client->mailbox.unseenPurge = FALSE;
        client->mailbox.unseenFlags = NULL;

        index = fopen(client->path, "wb");
        if (index != NULL) {
            fseek(index, 0, SEEK_SET);
            fprintf(index, NMAP_IDX_VERSION"\r\n");
            fprintf(index, "%010lu\r\n0000000000\r\n0000000000\r\n0000000000\r\n0000000000\r\n", client->mailbox.flags);
            fclose(index);
            index = NULL;
        } else {
            goto FailSyncingSharedMaildrop; 
        }
    }

    /* If we can determine that the hosting NMAP server has no changes, we are done. */
    /* Send the SINFO command to the hosting NMAP server to synchronize everything. */
    if (NMAPSendCommand(client->share.mailbox.conn, "SINFO\r\n", 7) != -1) {
        if (NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE) == 2002) {
            ptr = strchr(client->line, ' ');

            if (ptr != NULL) {
                *ptr = '\0';

                msgCount = atol(client->line);
                id = atol(++ptr);
            } else {
                /* Improper response from the hosting NMAP server. */
                goto FailSyncingSharedMaildrop;
            }
        } else {
            /* Unexpected communication error from the hosting NMAP server. */
            goto FailSyncingSharedMaildrop;
        }
    } else {
        /* Unable to communicate with the hosting NMAP server. */
        goto FailSyncingSharedMaildrop;
    }

    /* Ensure that we have enough elements in the MsgInfo array for everything. */
    client->mailbox.message.count = msgCount;
    if (client->mailbox.message.info != NULL) {
        localUsed = client->mailbox.message.used;

        if (client->mailbox.message.count > client->mailbox.message.allocated) {
            if (AddMsgInfo(client, (client->mailbox.message.count - client->mailbox.message.allocated) + 36)) {
                ;
            } else {
                goto FailSyncingSharedMaildrop;
            }
        }

        client->mailbox.message.used = client->mailbox.message.count;
    } else {
        localUsed = 0;

        client->mailbox.message.allocated = client->mailbox.message.count + 1;
        client->mailbox.message.info = (MessageInfoStruct *)MemMalloc(client->mailbox.message.allocated * sizeof(MessageInfoStruct));
        if (client->mailbox.message.info != NULL) {
            client->mailbox.message.used = client->mailbox.message.count;
        } else {
            /* Flush the responses from the SINFO command. */
            for (i = 0; i < client->mailbox.message.count; i++) {
                NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
            }

            NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);

            goto FailSyncingSharedMaildrop;
        }
    }

    if (client->mailbox.id == id) {
        /* Synchronizing with a trusted index on the hosting NMAP Server. */
        msgInfo = mLocal = client->mailbox.message.info;
        for (i = 0, idLocal = 0; i < client->mailbox.message.count; i++, msgInfo++) {
            NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);

            mIS.TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

            count = sscanf(client->line, "%*u %lu %lu %lu %lu %lu %lu %lu %lu %s %lu %lu %lu %lu %lu %lu %lu\r\n", 
                &(mIS.MSize), 
                &(mIS.SSize), 
                &(mIS.HeadPos), 
                &(mIS.BodyPos), 
                &(mIS.AuthPos), 
                (long unsigned int *)&(mIS.AuthLen), 
                &(mIS.BodyLines), 
                &(mIS.State), 
                mIS.UIDL, 
                &(mIS.UID), 
                &(mIS.DateSent), 
                &(mIS.DateReceived), 
                &(mIS.UseSCMS), 
                &(mIS.SCMSID), 
                &(mIS.RealStart), 
                &(mIS.RealSize));

            if (count == 16) {
                result = FALSE;
                while (idLocal < localUsed) {
                    if (mIS.UID == mLocal->UID) {
                        result = TRUE;

                        if (mIS.State & (MSG_STATE_PURGED | MSG_STATE_PRIVATE)) {
                            mLocal->State = MSG_STATE_PURGED;
                        }

                        /* Check to see if the entry is out of sequence.  If not, then continue; otherwise, move it. */
                        if (i == idLocal) {
                            break;
                        }

                        memcpy(msgInfo, mLocal, sizeof(MessageInfoStruct));
                        break;
                    } else {
                        idLocal++;
                        mLocal++;
                    }
                }

                if (result) {
                    idLocal++;
                    mLocal++;
                    continue;
                }

                /* This is a new entry for this index.  If it isn't purged, reset the flags. */
                if (changed) {
                    *changed = TRUE;
                }

                if (!(mIS.State & (MSG_STATE_PURGED | MSG_STATE_PRIVATE))) {
                    mIS.State &= (MSG_STATE_NEWUIDL | MSG_STATE_ESCAPE | MSG_STATE_PRIOHIGH | MSG_STATE_PRIOLOW | MSG_STATE_ANSWERED | MSG_STATE_COMPLETED | MSG_STATE_RECENT);
                    if (mIS.State & MSG_STATE_RECENT) {
                        client->mailbox.newCount += 1;
                        client->mailbox.newSize += mIS.MSize;
                    }
                } else {
                    mIS.State = MSG_STATE_PURGED;
                }

                /* Copy the entry. */
                memcpy(msgInfo, &mIS, sizeof(MessageInfoStruct));

                continue;
            } else {
                /* Could not parse the response from the hosting NMAP server. */
                for (i++; i < client->mailbox.message.count; i++) {
                    NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                }

                NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE);

                goto FailSyncingSharedMaildrop;
            }
        }
    } else {
        /* Synchronizing with a UNTRUSTED index on the hosting NMAP Server. */
        client->mailbox.indexTime = 0;

        msgInfo = mLocal = client->mailbox.message.info;
        for (i = 0, idLocal = 0; i < client->mailbox.message.count; i++, msgInfo++) {
            NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);

            mIS.TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

            count = sscanf(client->line, "%*u %lu %lu %lu %lu %lu %lu %lu %lu %s %lu %lu %lu %lu %lu %lu %lu\r\n", 
                &(mIS.MSize), 
                &(mIS.SSize), 
                &(mIS.HeadPos), 
                &(mIS.BodyPos), 
                &(mIS.AuthPos), 
                (long unsigned int *)&(mIS.AuthLen), 
                &(mIS.BodyLines), 
                &(mIS.State), 
                mIS.UIDL, 
                &(mIS.UID), 
                &(mIS.DateSent), 
                &(mIS.DateReceived), 
                &(mIS.UseSCMS), 
                &(mIS.SCMSID), 
                &(mIS.RealStart), 
                &(mIS.RealSize));

            if (count == 16) {
                result = FALSE;
                while (idLocal < localUsed) {
                    if (memcmp(mIS.UIDL, mLocal->UIDL, UIDLLEN) == 0) {
                        mLocal->UID = mIS.UID;
                        result = TRUE;

                        if (mIS.State & (MSG_STATE_PURGED | MSG_STATE_PRIVATE)) {
                            mLocal->State = MSG_STATE_PURGED;
                        }

                        /* Check to see if the entry is out of sequence.  If not, then continue; otherwise, move it. */
                        if (i == idLocal) {
                            break;
                        }

                        memcpy(msgInfo, mLocal, sizeof(MessageInfoStruct));
                        break;
                    } else {
                        idLocal++;
                        mLocal++;
                    }
                }

                if (result) {
                    idLocal++;
                    mLocal++;
                    continue;
                }

                /* This is a new entry for this index.  If it isn't purged, reset the flags. */
                if (changed) {
                    *changed = TRUE;
                }

                if (!(mIS.State & (MSG_STATE_PURGED | MSG_STATE_PRIVATE))) {
                    mIS.State &= (MSG_STATE_NEWUIDL | MSG_STATE_ESCAPE | MSG_STATE_PRIOHIGH | MSG_STATE_PRIOLOW | MSG_STATE_ANSWERED | MSG_STATE_COMPLETED | MSG_STATE_RECENT);
                    if (mIS.State & MSG_STATE_RECENT) {
                        client->mailbox.newCount += 1;
                        client->mailbox.newSize += mIS.MSize;
                    }
                } else {
                    mIS.State = MSG_STATE_PURGED;
                }

                /*    Copy the entry.    */
                memcpy(msgInfo, &mIS, sizeof(MessageInfoStruct));

                continue;
            } else {
                /* Could not parse the response from the hosting NMAP server. */
                for (i++; i < client->mailbox.message.count; i++) {
                    NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, FALSE);
                }

                NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE);

                goto FailSyncingSharedMaildrop;
            }
        }
    }

    if (NMAPReadAnswer(client->share.mailbox.conn, client->line, CONN_BUFSIZE, TRUE) == 1000) {
        client->mailbox.id = id;
    } else {
        /* Improper response from the hosting NMAP server. */
        goto FailSyncingSharedMaildrop;
    }

    /* We are now in sync with the hosting NMAP server.  Do we need to rewrite the local index? */
    if (!(client->flags & NMAP_READONLY)) {
        if ((client->mailbox.indexTime != 0) && (client->mailbox.newCount == 0)) {
            WriteNLockRelease(client->mailbox.lock);
            return(TRUE);
        } else if (client->mailbox.indexTime == 0) {
            localUsed = 0;
        }

        index = fopen(client->path, "r+b");
        if (index != NULL) {
            fseek(index, 0, SEEK_SET);
            fprintf(index, NMAP_IDX_VERSION"\r\n");
            fprintf(index, "%010lu\r\n0000000000\r\n%010lu\r\n%010lu\r\n0000000000\r\n", client->mailbox.flags, client->mailbox.message.count, client->mailbox.id);

            for (id = localUsed, msgInfo = &(client->mailbox.message.info[id]); id < client->mailbox.message.count; id++, msgInfo++) {
                if (!(msgInfo->State & MSG_STATE_RECENT)) {
                    fwrite(msgInfo, sizeof(MessageInfoStruct), 1, index);
                } else {
                    msgInfo->State &= ~MSG_STATE_RECENT;
                    fwrite(msgInfo, sizeof(MessageInfoStruct), 1, index);
                    msgInfo->State |= MSG_STATE_RECENT;
                }
            }

            fclose(index);

            stat(client->path, &sb);
            client->mailbox.indexTime = sb.st_mtime;
        } else {
            client->mailbox.indexTime = 0;
        }
    }

    WriteNLockRelease(client->mailbox.lock);
    return(TRUE);

FailSyncingSharedMaildrop:
    XplConsolePrintf("NMAPD: Failed for %s [%lu:%lu]\r\n", client->mailbox.name, client->mailbox.id, client->mailbox.message.count);
    if (index != NULL) {
        fclose(index);
    }

    client->mailbox.message.count = 0;
    client->mailbox.message.used = 0;
    client->mailbox.message.allocated = 0;
    client->mailbox.size = 0;
    client->mailbox.indexTime = 0;

    WriteNLockRelease(client->mailbox.lock);

    return(FALSE);
}

BOOL 
SyncSharedCalendar(NMAPClient *client, BOOL *FlagsChanged, BOOL Locked)
{
#if !defined(NMAP_SQL_CALENDARS)
     int count;
     unsigned int i;
     unsigned long l;
     unsigned long cUsed;
     unsigned long cCount;
     unsigned long id;
     unsigned long idLocal;
     unsigned long indexStartingOffset;
     unsigned long newCount;
     unsigned char *ptr;
     BOOL result;
     FILE *index;
     struct stat sb;
     CalendarInfoStruct *cLocal;
     CalendarInfoStruct *cInfo;
     CalendarInfoStruct cIS;

    if ((client->calendar.name[0] != '\0') 
            && (client->user[0] != '\0') 
            && (client->share.calendar.conn) 
            && !(client->share.flags & NMAP_SHARE_CALENDAR_DENIED) 
            && (Locked || (WaitforAndLockCalendar(client->store, client->user, client->calendar.name) == 1))) {
        if (client->calendar.fh != NULL) {
            fclose(client->calendar.fh);
            client->calendar.fh = NULL;
        }
    } else {
        return(FALSE);
    }

    /* Open the existing index file for the shared mailbox; or, create a new index file for the shared mailbox.
       Read, or initialize, the mailbox flags, size, count, id and next UID.    */
ReSyncSharedCalendar:
    newCount = 0;

    sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
    index = fopen(client->path, "rb");
    if (index != NULL) {
        stat(client->path, &sb);
        if (client->calendar.indexTime != sb.st_mtime) {
            if (fgets(client->line, CONN_BUFSIZE, index) != NULL) {
                if ((client->line[0] == 'C') && (client->line[7] == '5')) {
                    fgets(client->line, CONN_BUFSIZE, index);
                    client->calendar.flags = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->calendar.size = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    cCount = client->calendar.entries.count;
                    client->calendar.entries.count = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->calendar.id = atol(client->line);

                    fgets(client->line, CONN_BUFSIZE, index);
                    client->calendar.nextUID = atol(client->line);

                    if (ftell(index) == NMAP_CAL_IDX_HEADERSIZE) {
                        indexStartingOffset = NMAP_CAL_IDX_HEADERSIZE;
                    } else {
                        fclose(index);
                        index = NULL;

                        unlink(client->path);

                        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, -1, 0, NULL, 0);

                        goto ReSyncSharedCalendar;
                    }
                } else {
                    /* Unknown index file format.  Abort. */
                    goto FailSyncingSharedCalendar;
                }
            } else {
                /* fixme - possibly a zero length indexfile.  Close it, destroy it and restart. */
                fclose(index);
                index = NULL;

                unlink(client->path);

                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, -1, 0, NULL, 0);

                goto ReSyncSharedCalendar;
            }

            /* Ensure that we have enough elements in the calendar.entries.info array for the entries within the local index. */
            if (client->calendar.entries.info != NULL) {
                cUsed = client->calendar.entries.used;

                if (client->calendar.entries.count > client->calendar.entries.allocated) {
                    if (!AddCalendarInfo(client, (client->calendar.entries.count - client->calendar.entries.allocated) + 36)) {
                        ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);

                        goto FailSyncingSharedCalendar;
                    }
                }

                client->calendar.entries.used = client->calendar.entries.count;
            } else {
                cUsed = 0;

                client->calendar.entries.allocated = client->calendar.entries.count + 36;
                client->calendar.entries.info = (CalendarInfoStruct *)MemMalloc(client->calendar.entries.allocated * sizeof(CalendarInfoStruct));
                if (client->calendar.entries.info != NULL) {
                    client->calendar.entries.used = client->calendar.entries.count;
                } else {
                    goto FailSyncingSharedCalendar;
                }
            }

            if (FlagsChanged != NULL) {
                *FlagsChanged = FALSE;
            }

            /* Re-Sync the local shared index file with memory to detect deltas. */
            if (sb.st_mtime >= NMAP.time.startup) {
                fseek(index, indexStartingOffset + (cUsed * sizeof(CalendarInfoStruct)), SEEK_SET);
                for (i = cUsed, cInfo = client->calendar.entries.info; i < client->calendar.entries.count; i++, cInfo++) {
                    fread(cInfo, sizeof(CalendarInfoStruct), 1, index);
                }
            } else {
                for (i = 0, cInfo = client->calendar.entries.info; i < client->calendar.entries.count; i++, cInfo++) {
                    fread(cInfo, sizeof(CalendarInfoStruct), 1, index);

                    if (cInfo->TailSig == NMAP_IDX_LINE_TAIL_SIGNATURE) {
                        continue;
                    }

                    /* This message information block is invalid; toss the index and restart. */
                    MemFree(client->calendar.entries.info);
                    client->calendar.entries.info = NULL;

                    client->calendar.entries.allocated = 0;
                    client->calendar.entries.count = 0;
                    client->calendar.entries.used = 0;

                    fclose(index);

                    unlink(client->path);

                    XplConsolePrintf("NMAPD: Unable to use index file %s for user %s; entry %d corrupted.\r\n", client->calendar.name, client->user, i);
                    LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_INDEX_CORRUPTION, LOG_ERROR, 0, client->user, client->calendar.name, i, 0, NULL, 0);

                    goto ReSyncSharedCalendar;
                }
            }

            client->calendar.indexTime = sb.st_mtime;
        }

        fclose(index);
        index = NULL;
    } else {
        /* No index exists at this time; initialize for new one. */
        client->calendar.flags = RESOURCE_FLAG_UNSUBSCRIBED;

        if (client->calendar.entries.info != NULL) {
            MemFree(client->calendar.entries.info);
            client->calendar.entries.info = NULL;
        }

        cUsed = 0;
        client->calendar.entries.count = 0;
        client->calendar.entries.used = 0;
        client->calendar.entries.allocated = 0;
        client->calendar.size = 0;
        client->calendar.id = 0;

        client->calendar.indexTime = 0;
        client->calendar.indexChanged = FALSE;

        /*    Create an empty index file!    */
        index = fopen(client->path, "wb");
        if (index != NULL) {
            fseek(index, 0, SEEK_SET);
            fprintf(index, NMAP_CAL_IDX_VERSION"\r\n");
            fprintf(index, "%010lu\r\n0000000000\r\n0000000000\r\n0000000000\r\n0000000000\r\n", client->calendar.flags);
            fclose(index);
            index = NULL;
        } else {
            goto FailSyncingSharedCalendar; 
        }
    }

    /* If we can determine that the hosting NMAP server has no changes, we are done. */

    /*    Send the CSSINFO command to the hosting NMAP server to synchronize everything.    */
    if (NMAPSendCommandF(client->share.calendar.conn, "CSSINFO\r\n", 9) != -1) {
        if (NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, TRUE) == 2002) {
            ptr = strchr(client->line, ' ');

            if (ptr != NULL) {
                *ptr = '\0';

                cCount = atol(client->line);
                id = atol(++ptr);
            } else {
                /* Improper response from the hosting NMAP server. */
                goto FailSyncingSharedCalendar;
            }
        } else {
            /* Unexpected communication error from the hosting NMAP server. */
            goto FailSyncingSharedCalendar;
        }
    } else {
        /* Unable to communicate with the hosting NMAP server. */
        goto FailSyncingSharedCalendar;
    }

    /* Ensure that we have enough elements in the MsgInfo array for everything. */
    client->calendar.entries.count = cCount;
    if (client->calendar.entries.info != NULL) {
        cUsed = client->calendar.entries.used;

        if (client->calendar.entries.count > client->calendar.entries.allocated) {
            if (!AddCalendarInfo(client, (client->calendar.entries.count - client->calendar.entries.allocated) + 36)) {
                goto FailSyncingSharedCalendar;
            }
        }

        client->calendar.entries.used = client->calendar.entries.count;
    } else {
        cUsed = 0;

        client->calendar.entries.allocated = client->calendar.entries.count + 36;
        client->calendar.entries.info = (CalendarInfoStruct *)MemMalloc(client->calendar.entries.allocated * sizeof(CalendarInfoStruct));
        if (client->calendar.entries.info != NULL) {
            client->calendar.entries.used = client->calendar.entries.count;
        } else {
            /* Flush the responses from the SINFO command. */
            for (i = 0; i < client->calendar.entries.count; i++) {
                NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
            }

            NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);

            goto FailSyncingSharedCalendar;
        }
    }

    if (client->calendar.id == id) {
        /* Synchronizing with a trusted index on the hosting NMAP Server. */
        cInfo = cLocal = client->calendar.entries.info;
    } else {
        /* Synchronizing with a UNTRUSTED index on the hosting NMAP Server. */
        client->calendar.indexTime = 0;
        cUsed = 0;

        cInfo = cLocal = client->calendar.entries.info;
    }

    for (i = 0, idLocal = 0; i < client->calendar.entries.count; i++, cInfo++) {
        NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);

        cIS.TailSig = NMAP_IDX_LINE_TAIL_SIGNATURE;

        ptr = client->line;
        l = 0;
        while (l++ < 24) {
            do {
                ptr++;
            } while (*ptr != ' ');

            switch (l) {
                case 1: {   cIS.State = atol(++ptr);                continue;    }
                case 2: {   cIS.UTCEnd = atol(++ptr);               continue;    }
                case 3: {   cIS.UTCStart = atol(++ptr);             continue;    }
                case 4: {   cIS.UTCDue = atol(++ptr);               continue;    }
                case 5: {   cIS.Type = (unsigned char)atol(++ptr);  continue;    }
                case 6: {   cIS.Sequence = atol(++ptr);             continue;    }
                case 7: {   cIS.UID = atol(++ptr);                  continue;    }
                case 8: {   cIS.Recurring = atol(++ptr);            continue;    }
                case 9: {   cIS.RecurrenceID = atol(++ptr);         continue;    }
                case 10: {  cIS.UIDPos = atol(++ptr);               continue;    }
                case 11: {  cIS.OrganizerPos = atol(++ptr);         continue;    }
                case 12: {  cIS.AttendeePos = atol(++ptr);          continue;    }
                case 13: {  cIS.AttendeeCount = atol(++ptr);        continue;    }
                case 14: {  cIS.AttendeeSize = atol(++ptr);         continue;    }
                case 15: {  cIS.Pos = atol(++ptr);                  continue;    }
                case 16: {  cIS.Size = atol(++ptr);                 continue;    }
                case 17: {  cIS.CalSize = atol(++ptr);              continue;    }
                case 18: {
                    count = atol(++ptr);

                    /*    Find the space before the iCal UID.    */
                    do {
                        ptr++;
                    } while (*ptr != ' ');

                    memcpy(cIS.ICalUID, ++ptr, count);
                    cIS.ICalUID[count] = '\0';

                    ptr += count;
                    l += 2;

                    /*    Fall Through    */
                }

                case 20: {
                    count = atol(++ptr);
                    
                    do {
                     ptr++;
                    } while (*ptr != ' ');
                    
                    memcpy(cIS.Organizer, ++ptr, count);
                    cIS.Organizer[count] = '\0';

                    ptr += count;
                    l += 2;

                    /*    Fall Through    */
                }

                case 22: {
                    count = atol(++ptr);
                    
                    do {
                     ptr++;
                    } while (*ptr != ' ');
                    
                    memcpy(cIS.Summary, ++ptr, count);
                    cIS.Summary[count] = '\0';

                    ptr += count;
                    l += 2;

                    continue;
                }

                default: {
                    /* Could not parse the response from the hosting NMAP server. */
                    for (i++; i < client->mailbox.message.count; i++) {
                        NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, FALSE);
                    }

                    NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, TRUE);

                    goto FailSyncingSharedCalendar;
                }
            }
        }

        result = FALSE;
        while (idLocal < cUsed) {
            if (cIS.UID == cLocal->UID) {
                result = TRUE;

                if (cIS.State & MSG_STATE_PURGED) {
                    cLocal->State |= MSG_STATE_PURGED;
                }

                /* Check to see if the entry is out of sequence.  If not, then continue; otherwise, move it. */
                if (i == idLocal) {
                    break;
                }

                memcpy(cInfo, cLocal, sizeof(CalendarInfoStruct));
                break;
            } else {
                idLocal++;
                cLocal++;
            }
        }

        if (result) {
            idLocal++;
            cLocal++;
            continue;
        }

        /* This is a new entry for this index.  If it isn't purged, reset the flags. */
        if (!(cIS.State & MSG_STATE_PURGED)) {
            if (!(cIS.State & MSG_STATE_RECENT)) {
                cIS.State = 0;
            } else {
                cIS.State = MSG_STATE_RECENT;
                newCount++;
            }
        } else {
            cIS.State = MSG_STATE_PURGED;
        }

        /* Copy the entry. */
        memcpy(cInfo, &cIS, sizeof(CalendarInfoStruct));

        continue;
    }

    if (NMAPReadAnswer(client->share.calendar.conn, client->line, CONN_BUFSIZE, TRUE) != 1000) {
        /* Improper response from the hosting NMAP server. */
        goto FailSyncingSharedCalendar;
    }

    client->calendar.id = id;

    /* We are now in sync with the hosting NMAP server.  Do we need to rewrite the local index? */
    if (!(client->flags & NMAP_READONLY)) {
        if ((client->calendar.indexTime != 0) && (newCount == 0)) {
            UnlockCalendar(client->store, client->user, client->calendar.name);

            return(TRUE);
        } else if (client->calendar.indexTime == 0) {
            cUsed = 0;
        }

        index = fopen(client->path, "r+b");
        if (index != NULL) {
            fseek(index, 0, SEEK_SET);
            fprintf(index, NMAP_CAL_IDX_VERSION"\r\n");
            fprintf(index, "%010lu\r\n0000000000\r\n%010lu\r\n%010lu\r\n0000000000\r\n", client->calendar.flags, client->calendar.entries.count, client->calendar.id);

            for (id = cUsed, cInfo = &(client->calendar.entries.info[id]); id < client->calendar.entries.count; id++, cInfo++) {
                if (!(cInfo->State & MSG_STATE_RECENT)) {
                    fwrite(cInfo, sizeof(CalendarInfoStruct), 1, index);
                } else {
                    cInfo->State &= ~MSG_STATE_RECENT;
                    fwrite(cInfo, sizeof(CalendarInfoStruct), 1, index);
                    cInfo->State |= MSG_STATE_RECENT;
                }
            }

            fclose(index);

            stat(client->path, &sb);
            client->calendar.indexTime = sb.st_mtime;
        } else {
            client->calendar.indexTime = 0;
        }
    }

    UnlockCalendar(client->store, client->user, client->calendar.name);

    return(TRUE);

FailSyncingSharedCalendar:
    XplConsolePrintf("NMAPD: Failed for %s [%lu:%lu]\r\n", client->calendar.name, client->mailbox.id, client->mailbox.message.count);
    if (index != NULL) {
        fclose(index);
    }

    client->calendar.entries.count = 0;
    client->calendar.entries.used = 0;
    client->calendar.size = 0;
    client->calendar.indexTime = 0;

    UnlockCalendar(client->store, client->user, client->calendar.name);

#else
    /*
        fixme for sql based calendaring
    */
#endif

    return(FALSE);
}

int 
HandleShareStor(NMAPClient *client, unsigned char *mailbox, unsigned long date, unsigned long flags, BOOL noAuth, unsigned long size, unsigned char *owner, unsigned char *name)
{
    int ccode;
    unsigned int count;
    unsigned char timeBuf[80];
    time_t time_of_day;
    NMAPClient *cclient;

    do {
        if ((client->share.mailbox.conn) && (XplStrCaseCmp(client->mailbox.name, mailbox) == 0)) {
            /*    The currently selected share is the target.    */
            if (client->share.mailbox.permissions & NMAP_SHARE_INSERT) {
                cclient = client;
            } else {
                return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
            }
        } else {
            cclient = NMAPClientAlloc();
            if (cclient != NULL) {
                strcpy(cclient->user, client->user);
                cclient->userHash = client->userHash; 
                cclient->store = client->store;
                cclient->mailbox.newline = TRUE;
            } else {
                return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
            }

            if (((ccode = ConnectToNMAPShare(cclient, owner, name, NMAP_SHARE_MAILBOX, &(cclient->share.mailbox.permissions))) == 1000) 
                    && (cclient->share.mailbox.permissions & NMAP_SHARE_INSERT)) {
                if ((ccode = NMAPSendCommandF(cclient->share.mailbox.conn, "USER %s\r\n", owner)) != -1) {
                    ccode = NMAPReadAnswerLine(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE);
                }
            } else {
                if (ccode == 1000) {
                    ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                }

                break;
            }

            if (ccode != 1000) {
                ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
                break;
            }
        }

        time(&time_of_day);
        strftime(timeBuf, 80, "%a %b %d %H:%M %Y", gmtime(&time_of_day));
        GENERATE_FROM_DELIMITER(timeBuf, client->user, client->line, count);

        if (noAuth == FALSE) {
            count += sprintf(client->line + count, "X-Auth-OK: %s@%s\r\n", client->user, NMAP.officialName);
        }

        if ((flags != 0) || (date != 0)) {
            if (date != 0) {
                count += sprintf(client->line + count, "X-NIMS-flags: %lu %lu\r\n", (flags != 0)? flags: MSG_STATE_RECENT, date);
            } else {
                count += sprintf(client->line + count, "X-NIMS-flags: %lu\r\n", flags);
            }
        }

        if (size > 0) {
            ccode = NMAPSendCommandF(cclient->share.mailbox.conn, "STRAW %s %lu\r\n", name, count + size);
        } else {
            ccode = NMAPSendCommandF(cclient->share.mailbox.conn, "STRAW %s 0\r\n", name);
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswer(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE)) == 2002) 
                && ((ccode = ConnWrite(client->conn, "2002 Send message\r\n", 19)) != -1) 
                && ((ccode = ConnFlush(client->conn)) != -1) 
                && ((ccode = ConnWrite(cclient->share.mailbox.conn, client->line, count)) != -1)) {
            if (size > 0) {
                ccode = ConnReadToConn(client->conn, cclient->share.mailbox.conn, size);
            } else {
                ccode = ConnReadToConnUntilEOS(client->conn, cclient->share.mailbox.conn);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE)) != -1)) {
            ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
        }

        break;
    } while (TRUE);

    if (cclient != client) {
        NMAPQuit(cclient->share.mailbox.conn);

        ConnClose(cclient->share.mailbox.conn, 0);
        ConnFree(cclient->share.mailbox.conn);
        cclient->share.mailbox.conn = NULL;

        if (cclient->mailbox.message.info != NULL) {
            MemFree(cclient->mailbox.message.info);
            cclient->mailbox.message.info = NULL;
        }

        NMAPClientFree(cclient);
    }

    return(ccode);
}

int 
HandleShareCsstor(NMAPClient *client, unsigned char *calendar, unsigned char *sender, unsigned char *authenticatedSender, unsigned long size, unsigned char *owner, unsigned char *resource)
{
#if !defined(NMAP_SQL_CALENDARS)
    int ccode;
    NMAPClient *cclient;

    do {
        if ((client->share.calendar.conn) && (XplStrCaseCmp(client->calendar.name, calendar) == 0)) {
            /*    The currently selected share is the target.    */
            if (client->share.calendar.permissions & NMAP_SHARE_INSERT) {
                cclient = client;
            } else {
                return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
            }
        } else {
            cclient = NMAPClientAlloc();
            if (cclient != NULL) {
                strcpy(cclient->user, client->user);
                cclient->userHash = client->userHash; 
                cclient->store = client->store;
                cclient->mailbox.newline = TRUE;
            } else {
                return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
            }

            if (((ccode = ConnectToNMAPShare(cclient, owner, resource, NMAP_SHARE_MAILBOX, &(cclient->share.calendar.permissions))) == 1000) 
                    && (cclient->share.calendar.permissions & NMAP_SHARE_INSERT)) {
                if ((ccode = NMAPSendCommandF(cclient->share.calendar.conn, "USER %s\r\n", owner)) != -1) {
                    ccode = NMAPReadAnswerLine(cclient->share.calendar.conn, cclient->line, CONN_BUFSIZE, FALSE);
                }
            } else {
                if (ccode == 1000) {
                    ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                }

                break;
            }

            if (ccode != 1000) {
                ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
                break;
            }
        }

        if (size > 0) {
            ccode = NMAPSendCommandF(cclient->share.calendar.conn, "CSSTOR %s %lu FROM %s %s\r\n", resource, size, sender, authenticatedSender);
        } else {
            ccode = NMAPSendCommandF(cclient->share.calendar.conn, "CSSTOR %s 0 FROM %s %s\r\n", resource, sender, authenticatedSender);
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswer(cclient->share.calendar.conn, cclient->line, CONN_BUFSIZE, FALSE)) == 2002) 
                && ((ccode = ConnWrite(client->conn, "2002 Send calendar data\r\n", 25)) != -1) 
                && ((ccode = ConnFlush(client->conn)) != -1) 
                && ((ccode = ConnWrite(cclient->share.calendar.conn, client->line, strlen(client->line))) != -1)) {
            if (size > 0) {
                ccode = ConnReadToConn(client->conn, cclient->share.calendar.conn, size);
            } else {
                ccode = ConnReadToConnUntilEOS(client->conn, cclient->share.calendar.conn);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(cclient->share.calendar.conn, cclient->line, CONN_BUFSIZE, FALSE)) != -1)) {
            ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
        }

        break;
    } while (TRUE);

    if (cclient != client) {
        NMAPQuit(cclient->share.calendar.conn);

        ConnClose(cclient->share.calendar.conn, 0);
        ConnFree(cclient->share.calendar.conn);
        cclient->share.calendar.conn = NULL;

        if (cclient->calendar.entries.info != NULL) {
            MemFree(cclient->calendar.entries.info);
            cclient->calendar.entries.info = NULL;
        }

        NMAPClientFree(cclient);
    }

    return(ccode);
#else
    /*
        fixme for sql based calendaring
    */

    return(-1);
#endif
}

BOOL 
HandleShareStraw(NMAPClient *client, unsigned char *mailbox, unsigned long size, unsigned char *owner, unsigned char *resource)
{
    int ccode;
    NMAPClient *cclient;

    do {
        if ((client->share.mailbox.conn) && (XplStrCaseCmp(client->mailbox.name, mailbox) == 0)) {
            /*    The currently selected share is the target.    */
            if (client->share.mailbox.permissions & NMAP_SHARE_INSERT) {
                cclient = client;
            } else {
                return(ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1));
            }
        } else {
            cclient = NMAPClientAlloc();
            if (cclient != NULL) {
                strcpy(cclient->user, client->user);
                cclient->userHash = client->userHash; 
                cclient->store = client->store;
                cclient->mailbox.newline = TRUE;
            } else {
                return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
            }

            if (((ccode = ConnectToNMAPShare(cclient, owner, resource, NMAP_SHARE_MAILBOX, &(cclient->share.mailbox.permissions))) == 1000) 
                    && (cclient->share.mailbox.permissions & NMAP_SHARE_INSERT)) {
                if ((ccode = NMAPSendCommandF(cclient->share.mailbox.conn, "USER %s\r\n", owner)) != -1) {
                    ccode = NMAPReadAnswerLine(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE);
                }
            } else {
                if (ccode == 1000) {
                    ccode = ConnWrite(client->conn, MSG4240NOPERMISSION, sizeof(MSG4240NOPERMISSION) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4224CANTREADMBOX, sizeof(MSG4224CANTREADMBOX) - 1);
                }

                break;
            }

            if (ccode != 1000) {
                ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
                break;
            }
        }

        ccode = NMAPSendCommandF(cclient->share.mailbox.conn, "STRAW %s %lu\r\n", resource, size);

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswer(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE)) == 2002) 
                && ((ccode = ConnWrite(client->conn, "2002 Send message\r\n", 19)) != -1) 
                && ((ccode = ConnFlush(client->conn)) != -1) 
                && ((ccode = ConnWrite(cclient->share.mailbox.conn, client->line, strlen(client->line))) != -1)) {
            if (size > 0) {
                ccode = ConnReadToConn(client->conn, cclient->share.mailbox.conn, size);
            } else {
                ccode = ConnReadToConnUntilEOS(client->conn, cclient->share.mailbox.conn);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);

            break;
        }

        if ((ccode != -1) 
                && ((ccode = NMAPReadAnswerLine(cclient->share.mailbox.conn, cclient->line, CONN_BUFSIZE, FALSE)) != -1)) {
            ccode = ConnWrite(client->conn, cclient->line, strlen(cclient->line));
        } else {
            ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
        }

        break;
    } while (TRUE);

    if (cclient != client) {
        NMAPQuit(cclient->share.mailbox.conn);

        ConnClose(cclient->share.mailbox.conn, 0);
        ConnFree(cclient->share.mailbox.conn);
        cclient->share.mailbox.conn = NULL;

        if (cclient->mailbox.message.info != NULL) {
            MemFree(cclient->mailbox.message.info);
            cclient->mailbox.message.info = NULL;
        }

        NMAPClientFree(cclient);
    }

    return(ccode);
}

#define VSR_CONT_TRUE    2
#define VSR_EXIT_TRUE    1
#define VSR_EXIT_FALSE    0

/* This function requires a write lock */
__inline static BOOL
SetMBoxFlags(NMAPClient *client, unsigned long flags)
{
    FILE *index;
    unsigned long count;

    sprintf(client->path, "%s/%s/%s.idx", client->store, client->user, client->mailbox.name);

    index = fopen(client->path, "r+b");

    if (index) {
        count = sprintf(client->line, "%010lu\r\n", flags);

        if (fseek(index, 9, SEEK_SET) == 0) {/*    Skip Version    */
            if (fwrite(client->line, sizeof(unsigned char), count, index) == count) {
                fclose(index);
                return(TRUE);
            }
        }
        fclose(index);
    }

    return(FALSE);
}

/* This function requires a write lock */
__inline static BOOL
SetCalendarFlags(NMAPClient *client, unsigned long flags)
{
#if !defined(NMAP_SQL_CALENDARS)
    FILE *index;
    unsigned long count;
                 
    sprintf(client->path, "%s/%s/%s.idc", client->store, client->user, client->calendar.name);

    index = fopen(client->path, "r+b");
    if (index) {
        count = sprintf(client->line, "%010lu\r\n", flags);

        if (fseek(index, 10, SEEK_SET) == 0) {             /*    Skip Version    */
            if (fwrite(client->line, sizeof(unsigned char), count, index) == count) {
                fclose(index);
                return(TRUE);
            }
        }
        fclose(index);
    }
#else
    /*
        fixme for sql based calendaring
    */
#endif

    return(FALSE);
}

__inline static BOOL
RemoveSharedResourceFlag(NMAPClient *client, unsigned long request)
{
    if (request & NMAP_SHARE_MAILBOX) {
        client->mailbox.flags &= ~(RESOURCE_FLAG_SHARED | RESOURCE_FLAG_SHARED_PUBLIC);
        if (ReadNLockAquire(&client->mailbox.lock, &client->userHash, client->user, client->mailbox.name, 3000) == NLOCK_AQUIRED) {
            if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
                if (SetMBoxFlags(client, client->mailbox.flags)) {
                    WriteNLockRelease(client->mailbox.lock);
                    ReadNLockRelease(client->mailbox.lock);
                    client->mailbox.lock = NULL;
                    return(TRUE);
                }
                WriteNLockRelease(client->mailbox.lock);
            }
            ReadNLockRelease(client->mailbox.lock);
            client->mailbox.lock = NULL;
        }
        return(FALSE);
    } 

#if !defined(NMAP_SQL_CALENDARS)
    if (request & NMAP_SHARE_CALENDAR) {
        client->calendar.flags &= ~(RESOURCE_FLAG_SHARED | RESOURCE_FLAG_SHARED_PUBLIC);
        if (ReadNLockAquire(&client->calendar.lock, &client->userHash, client->user, client->calendar.name, 3000) == NLOCK_AQUIRED) {
            if (WriteNLockAquire(client->calendar.lock, 3000) == NLOCK_AQUIRED) {
                if (SetCalendarFlags(client, client->calendar.flags)) {
                    WriteNLockRelease(client->calendar.lock);
                    ReadNLockRelease(client->calendar.lock);
                    client->mailbox.lock = NULL;
                    return(TRUE);
                }
                WriteNLockRelease(client->calendar.lock);
            }
            ReadNLockRelease(client->calendar.lock);
            client->mailbox.lock = NULL;
        }
        return(FALSE);
    }
#else
    /*
        fixme for sql based calendaring
    */
#endif

    /* This is an unrecognized resource type */
    return(FALSE);
}

__inline static long
VerifySharedResource(NMAPClient *client, unsigned long request, unsigned long permissions)
{
    unsigned long flags;

    if (!(request & NMAP_SHARE_ADDRESS_BOOK)) {
        if (request & NMAP_SHARE_MAILBOX) {
            /* verify shared folder */
            sprintf(client->path, "%s/%s/%s.box", client->store, client->user, client->mailbox.name);
            if ((strstr(client->path, "../") == NULL) && (strstr(client->path, "..\\") == NULL)) {
                if (access(client->path, 0) == 0) {
                    if (ReadNLockAquire(&client->mailbox.lock, &client->userHash, client->user, client->mailbox.name, 0) == NLOCK_AQUIRED) {
                        if (ParseMaildrop(client)) {
                            flags = client->mailbox.flags;

                            if (flags & RESOURCE_FLAG_SHARED) {
                                ReadNLockRelease(client->mailbox.lock);
                                client->mailbox.lock = 0;
                                return(VSR_CONT_TRUE);
                            }

                            flags |= RESOURCE_FLAG_SHARED;
                            if (request & NMAP_SHARE_PUBLIC) {
                                flags |= RESOURCE_FLAG_SHARED_PUBLIC;
                            }

                            /* The share is not marked as shared yet */
                            if (WriteNLockAquire(client->mailbox.lock, 3000) == NLOCK_AQUIRED) {
                                if (SetMBoxFlags(client, flags)) {
                                    WriteNLockRelease(client->mailbox.lock);
                                    ReadNLockRelease(client->mailbox.lock);
                                    client->mailbox.lock = NULL;
                                    return(VSR_CONT_TRUE);
                                }

                                WriteNLockRelease(client->mailbox.lock);
                            }
                        }

                        ReadNLockRelease(client->mailbox.lock);
                        client->mailbox.lock = NULL;
                        return(VSR_EXIT_FALSE);
                    }
                } else if (permissions == 0) {
                    return(VSR_CONT_TRUE);
                }
            }
            return(VSR_EXIT_TRUE);
#if !defined(NMAP_SQL_CALENDARS)
        } else {
            /* verify shared calendar */
            sprintf(client->path, "%s/%s/%s.cal", client->store, client->user, client->calendar.name);
            if ((strstr(client->path, "../") == NULL) && (strstr(client->path, "..\\") == NULL)) {
                if (access(client->path, 0) == 0) {
                    if (ReadNLockAquire(&client->calendar.lock, &client->userHash, client->user, client->calendar.name, 0) == NLOCK_AQUIRED) {
                        if (ParseCalendar(client)) {
                            flags = client->calendar.flags;

                            if (flags & RESOURCE_FLAG_SHARED) {
                                ReadNLockRelease(client->calendar.lock);
                                client->calendar.lock = 0;
                                return(VSR_CONT_TRUE);
                            }

                            flags |= RESOURCE_FLAG_SHARED;
                            if (request & NMAP_SHARE_PUBLIC) {
                                flags |= RESOURCE_FLAG_SHARED_PUBLIC;
                            }

                            /* The share is not marked as shared yet */
                            if (WriteNLockAquire(client->calendar.lock, 3000) == NLOCK_AQUIRED) {
                                if (SetCalendarFlags(client, flags)) {
                                    WriteNLockRelease(client->calendar.lock);
                                    ReadNLockRelease(client->calendar.lock);
                                    client->calendar.lock = NULL;
                                    return(VSR_CONT_TRUE);
                                }
                                WriteNLockRelease(client->calendar.lock);
                            }
                        }

                        ReadNLockRelease(client->calendar.lock);
                        client->calendar.lock = NULL;
                        return(VSR_EXIT_FALSE);
                    }
                } else if (permissions == 0) {
                    return(VSR_CONT_TRUE);
                }
            }
            return(VSR_EXIT_TRUE);
#else
            /*
                fixme for sql based calendaring
            */
#endif
        }

        return(FALSE);
    }

    /* We don't have shared Address Books yet but when we do, add the code to verify them here */
    return(VSR_CONT_TRUE);
}

int 
HandleShareCreate(NMAPClient *client, unsigned long request, unsigned char *resource, unsigned char *command)
{
    int ccode;
    unsigned int count;
    unsigned int len;
    unsigned long used;
    unsigned long permissions;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *dn;
    unsigned char *grantee;
    unsigned char *granteeDN;
    unsigned char *buffer;
    unsigned char *displayName;
    unsigned char *scratch;
    BOOL result;
    NMAPClient *cclient;
    MDBValueStruct *v = NULL;
    struct in_addr granteeNMAPs;
    ShareAction action;

    cclient = NMAPClientAlloc();
    if (cclient) {
        buffer = (unsigned char *)MemMalloc((MDB_MAX_UTF8_OBJECT_CHARS + 1) + (MAXEMAILNAMESIZE + 1) + (MDB_MAX_UTF8_OBJECT_CHARS + 1) + (CONN_BUFSIZE + 1));
        if (buffer) {
            dn = buffer;
            grantee = buffer + MDB_MAX_UTF8_OBJECT_CHARS + 1;
            granteeDN = grantee + MAXEMAILNAMESIZE + 1;
            scratch = granteeDN + MDB_MAX_UTF8_OBJECT_CHARS + 1;

            /*    Initialize the temporary connection structure only.    */
            strcpy(cclient->user, client->user);
            cclient->userHash = client->userHash; 
            cclient->store = client->store;
            cclient->mailbox.newline = TRUE;
        } else {
            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    do {
        if (request & NMAP_SHARE_MAILBOX) {
            strcpy(cclient->mailbox.name, resource);
#if !defined(NMAP_SQL_CALENDARS)
        } else if (request & NMAP_SHARE_CALENDAR) {
            strcpy(cclient->calendar.name, resource);
#else
            /*
                fixme for sql based calendaring
            */
#endif
        }

        result = TRUE;
        ptr = command;

        /* Parse Grantee for creating new shares. */
        if (ptr != NULL) {
            request |= NMAP_SHARE_SEND_MESSAGE;

            while (isspace(*ptr)) {
                ptr++;
            }

            ptr2 = strchr(ptr, ' ');
            if (ptr2 != NULL) {
                *ptr2 = '\0';

                if ((ptr[0] != '-') || (ptr[1] != '\0')){
                    strcpy(grantee, ptr);
                } else {
                    memcpy(grantee, "-", 3);

                    request |= NMAP_SHARE_PUBLIC;
                }

                /* Permissions */
                ptr = strchr(++ptr2, ' ');
                if (ptr != NULL) {
                    *ptr = '\0';

                    permissions = atol(ptr2);

                    /* NOMESSAGE */
                    if (XplStrCaseCmp(++ptr, "NOMESSAGE") == 0) {
                        request &= ~NMAP_SHARE_SEND_MESSAGE;
                    } else {
                        result = FALSE;
                    }
                } else {
                    permissions = atol(ptr2);
                }
            } else {
                /* Missing required Permissions value. */
                result = FALSE;
                permissions = NMAP_SHARE_VISIBLE;
            }
        } else {
            /* Missing required Grantee value. */
            result = FALSE;
            permissions = NMAP_SHARE_VISIBLE;
        }

        /* Enforce minimal and maximal permissions assignments.
                NMAP_SHARE_VISIBLE is always granted.    
                NMAP_SHARE_ADMINISTER and NMAP_SHARE_POST are never granted    */
        if (permissions) {
            permissions |= NMAP_SHARE_VISIBLE;
            permissions &= (NMAP_SHARE_VISIBLE | NMAP_SHARE_SEEN | NMAP_SHARE_READ | NMAP_SHARE_WRITE | NMAP_SHARE_INSERT | NMAP_SHARE_CREATE | NMAP_SHARE_DELETE);
        }

        /* Check parsing results. */
        if (result) {
            if (request & NMAP_SHARE_PUBLIC) {
                request &= ~NMAP_SHARE_SEND_MESSAGE;
            }
        } else {
            ccode = ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);
            break;
        }

        v = MDBCreateValueStruct(NMAP.handle.directory, NULL);

        /*    Verify the grantee name.    */
        if (!(request & NMAP_SHARE_PUBLIC)) {
            if (MsgFindObject(grantee, granteeDN, NULL, &granteeNMAPs, v)) {
                if (strcmp(cclient->user, v->Value[0]) != 0) {
                    if (strcmp(grantee, v->Value[0]) != 0) {
                        strcpy(grantee, v->Value[0]);
                    }
                } else {
                    ccode = ConnWriteF(client->conn, MSG3014REQNOTALLOWED, " Cannot SHARE with self.");
                    break;
                }
            } else {
                /*    The grantee doesn't exist.    */
                request &= ~NMAP_SHARE_SEND_MESSAGE;
                permissions = 0;
                granteeDN[0] = '\0';
            }

            MDBFreeValues(v);
        } else {
            strcpy(granteeDN, MSGSRV_ROOT);
        }

        /* Verify resource */
        result = VerifySharedResource(cclient, request, permissions);
        if (result == VSR_CONT_TRUE) {
            ;
        } else {
            result = FALSE;
            ccode = ConnWrite(client->conn, MSG4224NOMBOX, sizeof(MSG4224NOMBOX) - 1);
            break;
        }

        /* Load the owned shares */
        if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
            MDBRead(client->dn, MSGSRV_A_OWNED_SHARES, v);
        } else {
            result = FALSE;
            ccode = ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1);
            break;
        }

        /* Create filter */
        if (request & NMAP_SHARE_MAILBOX) {
            sprintf(client->line, "MB%s\r", cclient->mailbox.name);
#if !defined(NMAP_SQL_CALENDARS)
        } else if (request & NMAP_SHARE_CALENDAR) {
            sprintf(client->line, "CA%s\r", cclient->calendar.name);
#else
            /*
                fixme for sql based calendaring
            */
#endif
        } else {
            strcpy(client->line, "ABAddressBook\r");
        }

        strcat(client->line, grantee);
        strcat(client->line, "\r");

        /* Determine if this is a new share; 
           an update to an existing share, removing an existing share or
           a reshare of an existing share with no changes;    */
        if (permissions) {
            action = NMAP_CREATE_SHARE;
        } else {
            action = NMAP_DELETE_SHARE;
        }

        count = strlen(client->line);
        for (used = 0; used < v->Used; used++) {
            if (XplStrNCaseCmp(client->line, v->Value[used], count) != 0) {
                continue;
            }

            if (permissions) {
                len = strlen(v->Value[used]);
                if (permissions != (unsigned long)atol(&(v->Value[used][len - 10]))) {
                    action = NMAP_MODIFY_SHARE;
                    MDBFreeValue(used, v);
                } else {
                    action = NMAP_DUPLICATE_SHARE;
                }
            } else {
                action = NMAP_DELETE_SHARE;
                MDBFreeValue(used, v);
            }

            break;
        }

        switch (action) {
            case NMAP_CREATE_SHARE: {
                sprintf(scratch, "%s%010lu", client->line, permissions);
                MDBAddValue(scratch, v);
                MDBWrite(client->dn, MSGSRV_A_OWNED_SHARES, v);
                MDBFreeValues(v);

                MDBRead(granteeDN, MSGSRV_A_AVAILABLE_SHARES, v);
                if (request & NMAP_SHARE_MAILBOX) {
                    count = sprintf(client->line, "MB%s\r%s\r", cclient->mailbox.name, cclient->user);
#if !defined(NMAP_SQL_CALENDARS)
                } else if (request & NMAP_SHARE_CALENDAR) {
                    count = sprintf(client->line, "CA%s\r%s\r", cclient->calendar.name, cclient->user);
#else
                    /*
                        fixme for sql based calendaring
                    */
#endif
                } else {
                    count = sprintf(client->line, "ABAddressBook\r%s\r", cclient->user);
                }

                displayName = client->line + count;
                count--;
                result = FALSE;
                for (used = 0; used < v->Used; used++) {
                    if ((v->Value[used][count] != '\r') || (XplStrNCaseCmp(client->line, v->Value[used], count) != 0)) {
                        continue;
                    }

                    result = TRUE;
                    break;
                }

                if (result == FALSE) {
                    if (request & NMAP_SHARE_MAILBOX) {
                        sprintf(displayName, "%s/%s/%s", NMAP.nameSpace.userPrefix, client->user, cclient->mailbox.name);
#if !defined(NMAP_SQL_CALENDARS)
                    } else if (request & NMAP_SHARE_CALENDAR) {
                        sprintf(displayName, "%s/%s/%s", NMAP.nameSpace.userPrefix, client->user, cclient->calendar.name);
#else
                        /*
                            fixme for sql based calendaring
                        */
#endif
                    } else {
                        sprintf(displayName, "%s/%s/address book", NMAP.nameSpace.userPrefix, client->user);
                    }

                    MDBAddValue(client->line, v);
                    MDBWrite(granteeDN, MSGSRV_A_AVAILABLE_SHARES, v);
                }

                MDBFreeValues(v);

                result = TRUE;

                /* Notify grantee of new share. */
                if (!(request & NMAP_SHARE_PUBLIC) 
                        && (request & NMAP_SHARE_SEND_MESSAGE)) {
                    if (NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW) {
                        ccode = ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1);
                        break;
                    }

                    cclient->queue.target = 0;

                    XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
                    cclient->queue.id = NMAP.queue.id++ & ((1 << 28) - 1);
                    XplSignalLocalSemaphore(NMAP.queue.semaphore);

                    sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, cclient->queue.id);
                    cclient->queue.control = fopen(client->path, "wb");
                    if (cclient->queue.control == NULL) {
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                        break;
                    }

                    fprintf(cclient->queue.control, QUEUES_DATE"%lu\r\n", time(NULL));
                    fprintf(cclient->queue.control, QUEUES_FROM"-\r\n");
                    fprintf(cclient->queue.control, QUEUES_RECIP_REMOTE"%s\r\n", grantee);
                    fclose(cclient->queue.control);

                    sprintf(client->path, "%s/d%07lx.msg", NMAP.path.spool, cclient->queue.id);
                    cclient->queue.data = fopen(client->path, "wb");
                    fprintf(cclient->queue.data, "From: postmaster@%s\r\n", NMAP.officialName);
                    fprintf(cclient->queue.data, "To: undisclosed-recipient-list\r\n");

                    ptr = displayName;
                    while((ptr = strchr(ptr, 127)) != NULL) {
                        *(ptr++) = ' ';
                    }

                    if (!strchr(cclient->user, '@')) {                
                        if (request & NMAP_SHARE_MAILBOX) {
                            fprintf(cclient->queue.data, "Subject: New shared folder from %s@%s\r\n", cclient->user, NMAP.officialName);
                        } else if (request & NMAP_SHARE_CALENDAR) {
                            fprintf(cclient->queue.data, "Subject: New shared calendar from %s@%s\r\n", cclient->user, NMAP.officialName);
                        } else {
                            fprintf(cclient->queue.data, "Subject: New shared address book from %s@%s\r\n", cclient->user, NMAP.officialName);
                        }
                    } else {
                        if (request & NMAP_SHARE_MAILBOX) {
                            fprintf(cclient->queue.data, "Subject: New shared folder from %s\r\n", cclient->user);
                        } else if (request & NMAP_SHARE_CALENDAR) {
                            fprintf(cclient->queue.data, "Subject: New shared calendar from %s\r\n", cclient->user);
                        } else {
                            fprintf(cclient->queue.data, "Subject: New shared address book from %s\r\n", cclient->user);
                        }
                    }

                    fprintf(cclient->queue.data, "X-Sender: Hula NMAP Agent\r\n");
                    fprintf(cclient->queue.data, "MIME-Version: 1.0\r\n");
                    fprintf(cclient->queue.data, "Content-Type: text/plain; charset=\"UTF-8\"\r\n");
                    fprintf(cclient->queue.data, "Content-Transfer-Encoding: 8bit\r\n\r\n");
                    fprintf(cclient->queue.data, NMAP.newShareMessage);
                    fprintf(cclient->queue.data, "\r\n");

                    fclose(cclient->queue.data);

                    sprintf(scratch,"%s/c%07lx.in", NMAP.path.spool, cclient->queue.id);
                    sprintf(client->path, "%s/c%07lx.000", NMAP.path.spool, cclient->queue.id);
                    rename(scratch, client->path);
                }

                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                break;
            }

            case NMAP_MODIFY_SHARE: {
                /*    Modified share; no update to grantee object; no message generated.    */

                sprintf(scratch, "%s%010lu", client->line, permissions);
                MDBAddValue(scratch, v);
                MDBWrite(client->dn, MSGSRV_A_OWNED_SHARES, v);

                result = TRUE;
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                break;
            }

            case NMAP_DELETE_SHARE: {
                /*    Deleted share; remove attribute from grantee object; generate message.    */
                MDBWrite(client->dn, MSGSRV_A_OWNED_SHARES, v);

                /*    Are there any remaining shares for this resource.    */
                count = 0;
                ptr = strchr(client->line, '\r');
                len = (unsigned int)(ptr - client->line);
                for (used = 0; used < v->Used; used++) {
                    if ((v->Value[used][len] != '\r') || (XplStrNCaseCmp(client->line, v->Value[used], len) != 0)) {
                        continue;
                    }

                    count++;
                }

                if (count == 0) {
                    /*    We have removed the last of the shares for this resource.  Reset the flags on the index.    */
                    RemoveSharedResourceFlag(cclient, request);
                }
                MDBFreeValues(v);

                if (granteeDN[0]) {
                    MDBRead(granteeDN, MSGSRV_A_AVAILABLE_SHARES, v);
                    if (request & NMAP_SHARE_MAILBOX) {
                        count = sprintf(client->line, "MB%s\r%s\r", cclient->mailbox.name, cclient->user);
#if !defined(NMAP_SQL_CALENDARS)
                    } else if (request & NMAP_SHARE_CALENDAR) {
                        count = sprintf(client->line, "CA%s\r%s\r", cclient->calendar.name, cclient->user);
#else
                        /*
                            fixme for sql based calendaring
                        */
#endif
                    } else {
                        count = sprintf(client->line, "ABAddressBook\r%s\r", cclient->user);
                    }
                
                    displayName = client->line + count;
                    count--;
                    for (used = 0; used < v->Used; used++) {
                        if ((v->Value[used][count] != '\r') || (XplStrNCaseCmp(client->line, v->Value[used], count) != 0)) {
                            continue;
                        }
                    
                        strcpy(displayName, &(v->Value[used][count + 1]));
                    
                        MDBFreeValue(used, v);
                        MDBWrite(granteeDN, MSGSRV_A_AVAILABLE_SHARES, v);
                    
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                        break;
                    }
                
                    MDBFreeValues(v);

                    result = TRUE;
                
                    /* Notify grantee of new share. */
                    if (!(request & NMAP_SHARE_PUBLIC) 
                         && (request & NMAP_SHARE_SEND_MESSAGE)) {
                        if (NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW) {
                            ccode = ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1);
                            break;
                        }
                    
                        cclient->queue.target = 0;
                    
                        XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
                        cclient->queue.id = NMAP.queue.id++ & ((1 << 28) - 1);
                        XplSignalLocalSemaphore(NMAP.queue.semaphore);
                    
                        sprintf(scratch, "%s/c%07lx.in", NMAP.path.spool, cclient->queue.id);
                        cclient->queue.control = fopen(scratch, "wb");
                        if (cclient->queue.control == NULL) {
                            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                            break;
                        }
                    
                        fprintf(cclient->queue.control, QUEUES_DATE"%lu\r\n", time(NULL));
                        fprintf(cclient->queue.control, QUEUES_FROM"-\r\n");
                        fprintf(cclient->queue.control, QUEUES_RECIP_REMOTE"%s\r\n", grantee);
                        fclose(cclient->queue.control);
                    
                        sprintf(scratch, "%s/d%07lx.msg", NMAP.path.spool, cclient->queue.id);
                        cclient->queue.data = fopen(scratch, "wb");
                        fprintf(cclient->queue.data, "From: postmaster@%s\r\n", NMAP.officialName);
                        fprintf(cclient->queue.data, "To: undisclosed-recipient-list\r\n");
                    
                        ptr = displayName;
                        while((ptr = strchr(ptr, 127)) != NULL) {
                            *(ptr++) = ' ';
                        }

                        if (!strchr(cclient->user, '@')) {                    
                            if ((request & NMAP_SHARE_MAILBOX) && (request & NMAP_SHARE_CALENDAR)) {
                                fprintf(cclient->queue.data, "Subject: %s@%s has un-shared a folder\r\n", cclient->user, NMAP.officialName);
                            } else if (request & NMAP_SHARE_CALENDAR) {
                                fprintf(cclient->queue.data, "Subject: %s@%s has un-shared a calendar\r\n", cclient->user, NMAP.officialName);
                            } else {
                                fprintf(cclient->queue.data, "Subject: %s@%s has un-shared address book\r\n", cclient->user, NMAP.officialName);
                            }
                        } else {
                            if (request & NMAP_SHARE_MAILBOX) {
                                fprintf(cclient->queue.data, "Subject: %s has un-shared a folder\r\n", cclient->user);
                            } else if (request & NMAP_SHARE_CALENDAR) {
                                fprintf(cclient->queue.data, "Subject: %s has un-shared a calendar\r\n", cclient->user);
                            } else {
                                fprintf(cclient->queue.data, "Subject: %s has un-shared address book\r\n", cclient->user);
                            }
                        }
                    
                        fprintf(cclient->queue.data, "X-Sender: Hula NMAP Agent\r\n");
                        fprintf(cclient->queue.data, "MIME-Version: 1.0\r\n");
                        fprintf(cclient->queue.data, "Content-Type: text/plain\r\n\r\n");
                        fprintf(cclient->queue.data, "\r\n");
                    
                        fclose(cclient->queue.data);
                    
                        sprintf(scratch,"%s/c%07lx.in", NMAP.path.spool, cclient->queue.id);
                        sprintf(client->line, "%s/c%07lx.000", NMAP.path.spool, cclient->queue.id);
                        rename(scratch, client->line);
                    }
                }

                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                break;
            }

            case NMAP_DUPLICATE_SHARE: {
                /* Duplicate share; no update needed; no messages generated. */
                result = TRUE;
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                break;
            }

            default: {
                result = FALSE;
                ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
                break;
            }
        }

        break;
    } while (TRUE);

    if (buffer != NULL) {
        MemFree(buffer);
    }

    if (v != NULL) {
        MDBFreeValues(v);
        MDBDestroyValueStruct(v);
    }

    if (cclient != NULL) {
#if !defined(NMAP_SQL_CALENDARS)
        if (cclient->calendar.entries.info != NULL) {
            MemFree(cclient->calendar.entries.info);
        }
#else
        /*
            fixme for sql based calendaring
        */
#endif

        if (cclient->mailbox.message.info != NULL) {
            MemFree(cclient->mailbox.message.info);
        }

        NMAPClientFree(cclient);
    }

    if (ccode != -1) {
        if (result) {
            ccode = 1;
        } else {
            ccode = 0;
        }
    }

    return(ccode);
}

int 
HandleProxyCreate(NMAPClient *client)
{
    int i;
    int ccode;
    unsigned long count;
    unsigned long used;
    unsigned long permissions;
    unsigned char *grantee;
    unsigned char *granteeDN;
    unsigned char *ptr;
    unsigned char *ptr2;
    BOOL result;
    struct in_addr granteeNMAPs;
    MDBValueStruct *v;

    ptr = client->buffer + 5;
    do {
        ptr++;
    } while (isspace(*ptr));

    /* Parse Grantee Name and permissions. */
    ptr2 = strchr(ptr, ' ');
    if (ptr2 != NULL) {
        *ptr2 = '\0';

        do {
            ptr2++;
        } while(isspace(*ptr2));

        if (*ptr2 != '\0') {
            permissions = atol(ptr2);
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    grantee = (unsigned char *)MemMalloc(MAXEMAILNAMESIZE + 1);
    granteeDN = (unsigned char *)MemMalloc(MDB_MAX_UTF8_OBJECT_CHARS + 1);
    if ((grantee != NULL) && (granteeDN != NULL)) {
        if ((ptr[0] != '-') || (ptr[1] != '\0')) {
            i = strlen(ptr);
            if (i < MAXEMAILNAMESIZE) {
                memcpy(grantee, ptr, ++i);
            } else {
                MemFree(granteeDN);
                MemFree(grantee);

                return(ConnWriteF(client->conn, MSG3014REQNOTALLOWED, "  Bad name length."));
            }
        } else {
            grantee[0] = '\0';
        }
    } else {
        if (grantee != NULL) {
            MemFree(grantee);
        }

        if (granteeDN != NULL) {
            MemFree(granteeDN);
        }

        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    v = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    while (v) {
        /*    Verify the grantee name.    */
        if (grantee[0] != '\0') {
            if (MsgFindObject(grantee, granteeDN, NULL, &granteeNMAPs, v)) {
                if (strcmp(client->user, v->Value[0]) != 0) {
                    if (strcmp(grantee, v->Value[0]) != 0) {
                        strcpy(grantee, v->Value[0]);
                    }
                } else {
                    result = TRUE;
                    ccode = ConnWriteF(client->conn, MSG3014REQNOTALLOWED, " Cannot PROXY with self.");
                    break;
                }
            } else {
                /*    The grantee doesn't exist.    */
                permissions = 0;
            }

            MDBFreeValues(v);
        } else {
            strcpy(granteeDN, MSGSRV_ROOT);
        }

        /*    Load the owned proxies for client->user    */
        if (MsgFindObject(client->user, client->dn, NULL, NULL, NULL)) {
            MDBRead(client->dn, MSGSRV_A_OWNED_PROXIES, v);
        } else {
            result = FALSE;
            ccode = ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1);
            break;
        }

        result = FALSE;
        count = sprintf(client->line, "%s\r", grantee);
        for (used = 0; used < v->Used; used++) {
            if (XplStrNCaseCmp(client->line, v->Value[used], count) != 0) {
                continue;
            }

            result = TRUE;

            /*    If the granted permissions have changed, update, or MemFree, this instance.    */
            count = strlen(v->Value[used]);
            if (permissions != (unsigned long)atol(&(v->Value[used][count - 10]))) {
                if (permissions != 0) {
                    sprintf(&(v->Value[used][count - 10]), "%010lu", permissions);
                } else {
                    MDBFreeValue(used, v);
                }
            }
        }

        if ((result == FALSE) && (permissions != 0)) {
            sprintf(client->line, "%s\r%010lu", grantee, permissions);
            MDBAddValue(client->line, v);
        }

        MDBWrite(client->dn, MSGSRV_A_OWNED_PROXIES, v);
        MDBFreeValues(v);

        /*    Load the available proxies for the grantee    */
        MDBRead(granteeDN, MSGSRV_A_AVAILABLE_PROXIES, v);

        result = FALSE;
        for (used = 0; used < v->Used; used++) {
            if (XplStrCaseCmp(client->user, v->Value[used]) != 0) {
                continue;
            }

            result = TRUE;

            /*    If the granted permissions have been revoked, MemFree this instance.    */
            if (permissions == 0) {
                MDBFreeValue(used, v);
            }
        }

        if ((result == FALSE) && (permissions != 0)) {
            MDBAddValue(client->user, v);
        }

        MDBWrite(granteeDN, MSGSRV_A_AVAILABLE_PROXIES, v);
        MDBFreeValues(v);

        result = TRUE;
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);

        break;
    }

    if (v) {
        MDBDestroyValueStruct(v);
    } else {
        result = FALSE;
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
    }

    MemFree(grantee);
    MemFree(granteeDN);

    if (ccode != -1) {
        if (result) {
            ccode = 1;
        } else {
            ccode = 0;
        }
    }

    return(ccode);
}

BOOL
StoreSharedMaildrop(NMAPClient *client)
{
    unsigned long i;
    unsigned char path[XPL_MAX_PATH + 1];
    struct stat sb;
    FILE *fh;

    if ((client->user[0] != '\0') && (client->mailbox.name[0] != '\0') && !(client->flags & NMAP_READONLY)) {
        if (PurgeNLockAquire(client->mailbox.lock, 0) == NLOCK_AQUIRED) {
            sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
            if (client->flags & NMAP_OOBMESSAGES) {
                GET_CHANGED_FLAGS(client->path, TAKE_PURGES, SILENT);
            }

            /* We have to save the shared fh file */
            fh = fopen(client->path, "wb");
            if (fh != NULL) {
                SetLongName(client->path, path);

                fprintf(fh, NMAP_IDX_VERSION"\r\n");
                fprintf(fh, "%010lu\r\n", client->mailbox.flags);
                fprintf(fh, "%010lu\r\n", (long unsigned int)0);
                fprintf(fh, "%010lu\r\n", client->mailbox.message.count);
                fprintf(fh, "%010lu\r\n", client->mailbox.id);
                fprintf(fh, "%010lu\r\n", client->mailbox.nextUID);

                for (i = 0; i < client->mailbox.message.count; i++) {
                    client->mailbox.message.info[i].State &= ~MSG_STATE_RECENT;
                    fwrite(&(client->mailbox.message.info[i]), sizeof(MessageInfoStruct), 1, fh);
                }

                fclose(fh);

                stat(client->path, &sb);
                client->mailbox.indexTime = sb.st_mtime;

                PurgeNLockRelease(client->mailbox.lock);

                client->mailbox.changed = FALSE;

                return(TRUE);
            }

            PurgeNLockRelease(client->mailbox.lock);
        }
    }

    return(FALSE);
}

BOOL
StoreSharedCalendar(NMAPClient *client)
{
#if !defined(NMAP_SQL_CALENDARS)
    unsigned long i;
    unsigned long state[2];
    unsigned char path[XPL_MAX_PATH + 1];
    FILE *index;
    struct stat sb;
    CalendarInfoStruct *cInfo;

    if ((client->user[0] != '\0') && (client->calendar.name[0] != '\0') && !(client->flags & NMAP_READONLY)) {
        if (PurgeNLockAquire(client->calendar.lock, 0) == NLOCK_AQUIRED) {
            if (client->calendar.indexChanged) {
                /* Detect the delta between disk and memory; the only thing that could change are flags */
                sprintf(client->path, "%s/%s/%s.sdx", client->store, client->user, client->mailbox.name);
                index = fopen(client->path, "rb");
                if (index) {
                    for (i = 0, cInfo = client->calendar.entries.info; i < client->calendar.entries.used; i++, cInfo++) {
                        if (fseek(index, NMAP_CAL_IDX_HEADERSIZE + (cInfo - client->calendar.entries.info) + offsetof(CalendarInfoStruct, State), SEEK_SET) == 0) {
                            /* This is tricky; we read the State and UTCend at once... */
                            fread(&state, sizeof(unsigned long), 2, index);

                            cInfo->UTCEnd = state[1];

                            if (state[0] & MSG_STATE_PURGED) {
                                client->calendar.changed = TRUE;
                            }

                            if (cInfo->State == state[0]) {
                                continue;
                            }

                            cInfo->State = state[0];

                            continue;
                        }
                    }

                    fclose(index);
                }

                client->calendar.indexChanged = FALSE;
            }

            /* We have to save the shared index file */
            sprintf(client->path, "%s/%s/%s.sdc", client->store, client->user, client->calendar.name);
            index = fopen(client->path, "wb");
            if (index != NULL) {
                SetLongName(client->path, path);
                fprintf(index, NMAP_CAL_IDX_VERSION"\r\n");
                fprintf(index, "%010lu\r\n", client->calendar.flags);
                fprintf(index, "%010lu\r\n", (long unsigned int)0);
                fprintf(index, "%010lu\r\n", client->calendar.entries.count);
                fprintf(index, "%010lu\r\n", client->calendar.id);
                fprintf(index, "%010lu\r\n", client->calendar.nextUID);

                for (i = 0, cInfo = client->calendar.entries.info; i < client->calendar.entries.count; i++, cInfo++) {
                    cInfo->State &= ~MSG_STATE_RECENT;
                    fwrite(cInfo, sizeof(CalendarInfoStruct), 1, index);
                }

                fclose(index);

                stat(client->path, &sb);
                client->calendar.indexTime = sb.st_mtime;
                
                PurgeNLockRelease(client->calendar.lock);

                client->mailbox.changed = FALSE;

                return(TRUE);
            }
            
            PurgeNLockRelease(client->calendar.lock);
        }
    }
#else
    /*
        fixme for sql based calendaring
    */
#endif

    return(FALSE);
}

int 
NmapCommandShareMbox(void *param)
{
    int count;
    int ccode;
    unsigned int len;
    unsigned long used;
    unsigned long request = NMAP_SHARE_MAILBOX;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char buffer[CONN_BUFSIZE + 1];
    unsigned char resource[XPL_MAX_PATH + 1];
    BOOL result;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs = NULL;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 10;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* SHARE MBOX[ <resource name>[ <grantee> <permissions>]] */
    if (*ptr == '\0') {
        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    } else if ((*ptr++ == ' ') && (*ptr) && (!isspace(*ptr))) {
        request |= NMAP_SHARE_RESOURCE;

        ptr2 = strchr(ptr, ' ');
        if (ptr2 == NULL) {
            strcpy(resource, ptr);
        } else if ((ptr2[1]) && (!isspace(ptr2[1]))) {
            *ptr2++ = '\0';
            strcpy(resource, ptr);

            if (*ptr2 != '\0') {
                return(HandleShareCreate(client, request, resource, ptr2));
            }
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }

        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (result) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1));
    }

    if (vs) {
        strcpy(buffer, "MB");

        if (request & NMAP_SHARE_RESOURCE) {
            sprintf(buffer + strlen(buffer), "%s\r", resource);
        }

        count = strlen(buffer);

        ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16);

        MDBRead(client->dn, MSGSRV_A_OWNED_SHARES, vs);
        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            if (XplStrNCaseCmp(buffer, vs->Value[used], count) != 0) {
                continue;
            }

            len = strlen(vs->Value[used]);
            vs->Value[used][len - 11] = '\0';

            ptr = strchr(vs->Value[used], '\r');
            if (ptr != NULL) {
                *(ptr++) = '\0';

                if (!(request & NMAP_SHARE_RESOURCE)) {
                    ccode = ConnWriteF(client->conn, "2002-%s %s %lu\r\n", vs->Value[used] + 2, ptr, atol(&(vs->Value[used][len - 10])));
                } else {
                    ccode = ConnWriteF(client->conn, "2002-%s %lu\r\n", ptr, atol(&(vs->Value[used][len - 10])));
                }
            }
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        MDBDestroyValueStruct(vs);
    } else {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
    }

    return(ccode);
}

int 
NmapCommandShowproxy(void *param)
{
    int ccode;
    unsigned long used;
    unsigned char *ptr;
    BOOL result;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 9;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (*ptr == '\0') {
        result = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    if (result) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWrite(client->conn, MSG4224NOUSER, sizeof(MSG4224NOUSER) - 1));
    }

    if (vs) {
        MDBRead(client->dn, MSGSRV_A_AVAILABLE_PROXIES, vs);

        ccode = ConnWriteF(client->conn, "2002 %lu\r\n", vs->Used);
        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            ccode = ConnWriteF(client->conn, "2002-%s\r\n", vs->Value[used]);
        }

        MDBDestroyValueStruct(vs);

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
    }

    return(ccode);
}

int 
NmapCommandShowsh(void *param)
{
    int count;
    int ccode;
    int request;
    unsigned long used;
    unsigned long used2;
    unsigned long permissions;
    unsigned char type;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *ptr3;
    unsigned char dn[MDB_MAX_OBJECT_CHARS + 1];
    BOOL found;
    NMAPClient *client = (NMAPClient *)param;
    MDBValueStruct *vs = NULL;
    MDBValueStruct *vs2 = NULL;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* SHOWSH[ <ADBK | CAL | MBOX> */
    if (*ptr == '\0') {
        request = NMAP_SHARE_MAILBOX | NMAP_SHARE_CALENDAR | NMAP_SHARE_ADDRESS_BOOK;
    } else if (*ptr == ' ') {
        if (XplStrNCaseCmp(++ptr, "MBOX", 4) == 0) {
            ptr += 4;
            request = NMAP_SHARE_MAILBOX;
        } else if (XplStrNCaseCmp(ptr, "CAL", 3) == 0) {
            ptr += 4;
            request = NMAP_SHARE_CALENDAR;
        } else if (XplStrNCaseCmp(ptr, "ADBK", 4) == 0) {
            ptr += 4;
            request = NMAP_SHARE_ADDRESS_BOOK;
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    found = MsgFindObject(client->user, client->dn, NULL, NULL, NULL);
    if (found) {
        vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
        vs2 = MDBCreateValueStruct(NMAP.handle.directory, NULL);
    } else {
        return(ConnWriteF(client->conn, MSG5244USERLOOKUPFAILURE, client->user));
    }

    if (vs && vs2) {
        ccode = ConnWrite(client->conn, "2002-Coming up\r\n", 16);
    } else {
        if (vs) {
            MDBDestroyValueStruct(vs);
        }

        if (vs2) {
            MDBDestroyValueStruct(vs2);
        }

        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    MDBRead(client->dn, MSGSRV_A_AVAILABLE_SHARES, vs);
    while (request != 0) {
        if (request & NMAP_SHARE_MAILBOX) {
            request &= ~NMAP_SHARE_MAILBOX;
            type = 'M';
        } else if (request & NMAP_SHARE_CALENDAR) {
            request &= ~NMAP_SHARE_CALENDAR;
            type = 'C';
        } else if (request & NMAP_SHARE_ADDRESS_BOOK) {
            request &= ~NMAP_SHARE_ADDRESS_BOOK;
            type = 'A';
        } else {
            /* Unsupported share type */
            break;
        }

        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            ptr = vs->Value[used];

            if (*ptr != type) {
                continue;
            }

            ptr = strchr(ptr, '\r');
            if (ptr != NULL) {
                ptr++;
                ptr2 = strrchr(ptr, '\r');
                if (ptr2 != NULL) {
                    *ptr2++ = '\0';
                    permissions = 0;

                    found = MsgFindObject(ptr, dn, NULL, NULL, NULL);
                    if (found) {
                        found = FALSE;

                        MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs2);
                        for (used2 = 0; (ccode != -1) && (used2 < vs2->Used); used2++) {
                            if (vs2->Value[used2][0] != type) {
                                continue;
                            }

                            count = strlen(vs2->Value[used2]);

                            /* Punch out the '\r' following the grantee name. */
                            vs2->Value[used2][count - 11] = '\0';

                            ptr3 = strchr(vs2->Value[used2], '\r');
                            if ((ptr3 == NULL) || (XplStrCaseCmp(++ptr3, client->user) != 0)) {
                                continue;
                            }

                            ccode = ConnWriteF(client->conn, "2002-%s %lu\r\n", ptr2, atol(vs2->Value[used2] + count - 10));

                            found = TRUE;
                            break;
                        }

                        if (found) {
                            continue;
                        }
                    }

                    ccode = ConnWriteF(client->conn, "2002-%s 0\r\n", ptr2);
                }
            }
        }
    }

    MDBFreeValues(vs);
    MDBRead(MSGSRV_ROOT, MSGSRV_A_AVAILABLE_SHARES, vs);
    while (request != 0) {
        if (request & NMAP_SHARE_MAILBOX) {
            request &= ~NMAP_SHARE_MAILBOX;
            type = 'M';
        } else if (request & NMAP_SHARE_CALENDAR) {
            request &= ~NMAP_SHARE_CALENDAR;
            type = 'C';
        } else if (request & NMAP_SHARE_ADDRESS_BOOK) {
            request &= ~NMAP_SHARE_ADDRESS_BOOK;
            type = 'A';
        } else {
            /* Unsupported share type */
            break;
        }

        for (used = 0; (ccode != -1) && (used < vs->Used); used++) {
            ptr = vs->Value[used];

            if (*ptr != type) {
                continue;
            }

            ptr = strchr(ptr, '\r');
            if (ptr != NULL) {
                ptr++;
                ptr2 = strrchr(ptr, '\r');
                if (ptr2 != NULL) {
                    *ptr2++ = '\0';
                    permissions = 0;

                    found = MsgFindObject(ptr, dn, NULL, NULL, NULL);
                    if (found) {
                        found = FALSE;

                        MDBRead(dn, MSGSRV_A_OWNED_SHARES, vs2);
                        for (used2 = 0; (ccode != -1) && (used2 < vs2->Used); used2++) {
                            if (vs2->Value[used2][0] != type) {
                                continue;
                            }

                            count = strlen(vs2->Value[used2]);

                            /* Punch out the '\r' following the grantee name. */
                            vs2->Value[used2][count - 11] = '\0';

                            ptr3 = strchr(vs2->Value[used2], '\r');
                            if ((ptr3 == NULL) || (XplStrCaseCmp(++ptr3, "-") != 0)) {
                                continue;
                            }

                            ccode = ConnWriteF(client->conn, "2002-%s %lu\r\n", ptr2, atol(vs2->Value[used2] + count - 10));

                            found = TRUE;
                            break;
                        }

                        if (found) {
                            continue;
                        }
                    }

                    ccode = ConnWriteF(client->conn, "2002-%s 0\r\n", ptr2);
                }
            }
        }
    }

    if (ccode != -1) {
        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
    }

    MDBDestroyValueStruct(vs);
    MDBDestroyValueStruct(vs2);

    return(ccode);
}
