/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>
#include <msgapi.h>
#include <logger.h>
#include <nmlib.h>

#include "nmapdp.h"
#include "sql-lock.h"

#if defined(NMAP_SQL_CALENDARS)

#include <libical2.h>

#define SQLTRAN_BEGIN "BEGIN IMMEDIATE TRANSACTION;"
#define SQLTRAN_END "END TRANSACTION;"
#define SQLTRAN_ROLLBACK "ROLLBACK TRANSACTION;"

#define SQLSEL_CALID_BY_NAME "SELECT id FROM calendars WHERE name=?;"
#define SQLSEL_CALENDAR_ID_GUID_NAME "SELECT id, guid, name FROM calendars;"
#define SQLSEL_OBJID_BY_GUID "SELECT id FROM objects WHERE guid=?;"
#define SQLSEL_OBJID_BY_ALTNAME "SELECT object FROM names WHERE name=?;"

#define SQLSEL_LAST_COMPID_BY_OBJECT "SELECT id FROM components WHERE object=? ORDER BY id DESC LIMIT 1;"
#define SQLSEL_LAST_PROPID_BY_COMP "SELECT id FROM properties WHERE component=? ORDER BY id DESC LIMIT 1;"

#define SQLSEL_CSINFO_ON_OBJ "SELECT id FROM objects WHERE calendar=?;"
#define SQLSEL_CSINFO_ON_COMP "SELECT id, flags, parent, type, position, length, julianday(start), julianday(end), julianday(modified) FROM components WHERE object=? ORDER BY id;"
#define SQLSEL_CSINFO_ON_PROP "SELECT value FROM properties WHERE component=? AND name=?;"

#define SQLSEL_LIST_ON_OBJ "SELECT flags, length FROM objects WHERE id=?;"
#define SQLSEL_LIST_ON_COMP "SELECT id, flags, parent, name FROM components WHERE object=? ORDER BY position;"
#define SQLSEL_LIST_ON_PROP "SELECT id, flags, position, name, place, value FROM properties WHERE component=? ORDER BY position, place;"
#define SQLSEL_LIST_ON_PARAM "SELECT id, flags, name, place, value FROM parameters WHERE property=? ORDER BY position, place;"

#define SQLINS_CALENDAR "INSERT INTO calendars (flags, guid, name, created, modified) VALUES (?, ?, ?, DATETIME('now'), DATETIME('now'));"
#define SQLINS_OBJECT "INSERT INTO objects (flags, guid, calendar, length, received, sender, authSender, modified) VALUES (?, ?, ?, ?, DATETIME('now'), ?, ?, DATETIME('now'));"
#define SQLINS_ALTNAME "INSERT INTO names (flags, object, name) VALUES (?, ?, ?);"
#define SQLINS_COMPONENT "INSERT INTO components (flags, object, parent, type, position, length, name, start, end, modified) VALUES (?, ?, ?, ?, ?, ?, ?, DATETIME(?), DATETIME(?), DATETIME(?));"
#define SQLINS_PROPERTY "INSERT INTO properties (flags, component, position, syntax, name, place, value, modified) VALUES (?, ?, ?, ?, ?, ?, ?, DATETIME('now'));"
#define SQLINS_PARAMETER "INSERT INTO parameters (flags, property, position, syntax, name, place, value, modified) VALUES (?, ?, ?, ?, ?, ?, ?, DATETIME('now'));"

#define SQLUPD_CALENDAR_NAME "UPDATE calendars SET name=? WHERE name=?;"

typedef struct _CSListNext {
    NmapCalID compID;
    NmapCalID parentID;

    unsigned long flags;

    unsigned char *name;
} CSListNext;

static int 
NmapSqlCalendarCreate(NMAPClient *client)
{
    int ccode;
    char *errmsg = NULL;
    unsigned char *guid;

    guid = GuidAlloc();
    if (guid) {
        ccode = sqlite3_exec(client->cal.handle, SQLTRAN_BEGIN, NULL, NULL, &errmsg);
    } else {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->user, client->path, NMAP_GUID_LENGTH, 0, NULL, 0);
        return(FALSE);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE calendars (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "guid VARCHAR(46) NOT NULL UNIQUE, " \
                                                            "name TEXT NOT NULL UNIQUE, " \
                                                            "created DATETIME NOT NULL, " \
                                                            "objects INTEGER DEFAULT 0, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);

    }

    if ((ccode == SQLITE_OK) 
            && ((ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX calendars_index ON calendars (guid, name);", NULL, NULL, &errmsg)) == SQLITE_OK)) {
        sprintf(client->line, "INSERT INTO calendars (flags, guid, name, created, modified) VALUES (%d, '%s', 'MAIN', DATETIME('now'), DATETIME('now'));", 0, guid);

        ccode = sqlite3_exec(client->cal.handle, client->line, NULL, NULL, &errmsg);
    }

    GuidFree(guid);

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE objects (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "guid VARCHAR(46) NOT NULL UNIQUE, " \
                                                            "calendar INTEGER NOT NULL, " \
                                                            "length INTEGER NOT NULL, " \
                                                            "received DATETIME NOT NULL, " \
                                                            "sender TEXT NOT NULL, " \
                                                            "authSender TEXT NOT NULL, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE names (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "object INTEGER NOT NULL, " \
                                                            "name TEXT NOT NULL UNIQUE" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE components (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "object INTEGER NOT NULL, " \
                                                            "parent INTEGER NOT NULL, " \
                                                            "type INTEGER NOT NULL, " \
                                                            "position INTEGER NOT NULL, " \
                                                            "length INTEGER NOT NULL, " \
                                                            "name TEXT NOT NULL, " \
                                                            "start DATETIME DEFAULT 0, " \
                                                            "end DATETIME DEFAULT 0, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE properties (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "component INTEGER NOT NULL, " \
                                                            "position INTEGER NOT NULL, " \
                                                            "syntax INTEGER NOT NULL, " \
                                                            "name TEXT NOT NULL, " \
                                                            "place INTEGER NOT NULL, " \
                                                            "value TEXT NOT NULL, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE parameters (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "property INTEGER NOT NULL, " \
                                                            "position INTEGER NOT NULL, " \
                                                            "syntax INTEGER NOT NULL, " \
                                                            "name TEXT NOT NULL, " \
                                                            "place INTEGER NOT NULL, " \
                                                            "value TEXT NOT NULL, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

#if 0
    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE access (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "guid VARCHAR(46) NOT NULL UNIQUE, " \
                                                            "name TEXT NOT NULL UNIQUE, " \
                                                            "uri TEXT NOT NULL, " \
                                                            "modified DATETIME NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE rules (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "component INTEGER NOT NULL, " \
                                                            "position INTEGER NOT NULL, " \
                                                            "parent INTEGER NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle,   "CREATE TABLE dates (" \
                                                            "id INTEGER PRIMARY KEY AUTOINCREMENT, " \
                                                            "flags INTEGER DEFAULT 0, " \
                                                            "component INTEGER NOT NULL, " \
                                                            "position INTEGER NOT NULL, " \
                                                            "parent INTEGER NOT NULL" \
                                                        ");", NULL, NULL, &errmsg);
    }
#endif

    if ((ccode == SQLITE_OK) 
            && ((ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX objects_index ON objects (guid, calendar);", NULL, NULL, &errmsg)) == SQLITE_OK) 
            && ((ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX names_index ON names (object, name);", NULL, NULL, &errmsg)) == SQLITE_OK) 
            && ((ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX components_index ON components (object, name, start, end);", NULL, NULL, &errmsg)) == SQLITE_OK) 
            && ((ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX properties_index ON properties (component, name);", NULL, NULL, &errmsg)) == SQLITE_OK)) {
        ccode = sqlite3_exec(client->cal.handle, "CREATE INDEX parameters_index ON parameters (property, name);", NULL, NULL, &errmsg);
    }

    if (ccode == SQLITE_OK) {
        ccode = sqlite3_exec(client->cal.handle, SQLTRAN_END, NULL, NULL, &errmsg);
    } else {
        sqlite3_exec(client->cal.handle, SQLTRAN_ROLLBACK, NULL, NULL, NULL);
        sqlite3_exec(client->cal.handle, SQLTRAN_END, NULL, NULL, NULL);
    }

    if (errmsg) {
#if defined(DEBUG)
        XplConsolePrintf("NMAP [SQL %d]: Create database failed; %s\r\n", __LINE__, errmsg);
#endif

        sqlite3_free(errmsg);
    }

    return(ccode);
}

static int 
NmapSqlPrepare(sqlite3 *handle, unsigned char *statement, sqlite3_stmt **stmt)
{
    if (*stmt) {
        return(SQLITE_OK);
    }

    return (sqlite3_prepare(handle, statement, -1, stmt, NULL));

}

static int
NmapSqlCalendarOpen(NMAPClient *client)
{
    register int ccode;
    register int present;

    if (client->states & NMAP_CLIENT_CAL) {
        return(SQLITE_OK);
    }

    sprintf(client->path, "%s/%s/calendar.db", client->store, client->user);
    present = access(client->path, 0);

    ccode = sqlite3_open(client->path, &client->cal.handle);
    if (ccode == SQLITE_OK) {
        if (present == 0) {
            return(SQLITE_OK);
        }

        ccode = NmapSqlCalendarCreate(client);
    }

#if defined(DEBUG)
    if (ccode != SQLITE_OK) {
        XplConsolePrintf("NMAP [SQL %d]: Open database failed; %s\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
    }
#endif

    return(ccode);
}

static void 
NmapSqlCalendarClose(NMAPClient *client)
{
    if (client->states & NMAP_CLIENT_CAL) {
        return;
    }

    if (client->cal.handle) {
        sqlite3_close(client->cal.handle);
        client->cal.handle = NULL;

        client->cal.id = 0;
    }

    return;
}

static int 
NmapSqlBegin(NMAPClient *client)
{
    register int ccode;
    register int count = 30;

    do {
        if (client->cal.sql.begin) {
            do {
                ccode = sqlite3_step(client->cal.sql.begin);

                sqlite3_reset(client->cal.sql.begin);
                if (ccode == SQLITE_DONE) {
                    return(SQLITE_DONE);
                }

                if (ccode == SQLITE_BUSY) {
                    XplDelay(333);

                    count--;
                    continue;
                }

                break;
            } while (count);

            break;
        }

        if (((ccode = NmapSqlPrepare(client->cal.handle, SQLTRAN_BEGIN, &(client->cal.sql.begin))) == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLTRAN_END, &(client->cal.sql.end))) == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLTRAN_ROLLBACK, &(client->cal.sql.abort))) == SQLITE_OK)) {
            continue;
        }

        break;
    } while (TRUE);

#if defined(DEBUG)
    if (ccode != SQLITE_DONE) {
        XplConsolePrintf("NMAP [SQL %d]: Begin transaction failed; %s\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
    }
#endif

    return(ccode);
}

static int 
NmapSqlEnd(NMAPClient *client)
{
    register int ccode;

    ccode = sqlite3_step(client->cal.sql.end);

    sqlite3_reset(client->cal.sql.end);

#if defined(DEBUG)
    if (ccode != SQLITE_DONE) {
        XplConsolePrintf("NMAP [SQL %d]: End transaction failed; %s\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
    }
#endif

    return(ccode);
}

static void 
NmapSqlAbort(NMAPClient *client)
{
    register int ccode;

    ccode = sqlite3_step(client->cal.sql.abort);

    sqlite3_reset(client->cal.sql.abort);

#if defined(DEBUG)
    if (ccode != SQLITE_DONE) {
        XplConsolePrintf("NMAP [SQL %d]: Abort transaction failed; %s\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
    }
#endif

    ccode = sqlite3_step(client->cal.sql.end);

    sqlite3_reset(client->cal.sql.end);

#if defined(DEBUG)
    if (ccode != SQLITE_DONE) {
        XplConsolePrintf("NMAP [SQL %d]: End transaction failed; %s\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
    }
#endif

    return;
}

static void 
NmapSqlDateTimeFromICal2DateTime(unsigned char *sqlDateTime, ICal2DateTimeValue *dateTime)
{
    ICal2DateTimeValue dtZero;
    ICal2DateTimeValue *dt;

    if (dateTime) {
        dt = dateTime;
    } else {
        memset(&dtZero, 0, sizeof(ICal2DateTimeValue));

        dt = &dtZero;
    }

    /*
        fixme

        Enhance for non-UTC time values
    */
    sprintf(sqlDateTime, "%04d-%02d-%02d %02d:%02d", dt->date.year, dt->date.month, dt->date.day, dt->time.hour, dt->time.minute);

    return;
}

static int 
NmapCsID(NMAPClient *client, unsigned char *line, NmapCalID *id, unsigned char **next)
{
    int ccode;
    unsigned char *ptr = line;
    sqlite3_stmt *stmt;

    *id = 0;
    if (*ptr++ == ' ') {
        if (isdigit(*ptr)) {
            *next = strchr(ptr, ' ');
            if (*next) {
                **next = '\0';
                *next++;
            }

            *id = atol(ptr);
            return(0);
        } else {
            if (toupper(*ptr) == 'G') {
                *next = ptr + NMAP_GUID_LENGTH + 1;
                if ((**next == '\0') || (**next == ' ')) {
                    ptr++;
                    ccode = NmapSqlBegin(client);
                } else {
                    *next = ptr;
                    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
                }

                if (ccode == SQLITE_DONE) {
                    ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_OBJID_BY_GUID, &(client->cal.sql.sel.objIDByGuid));
                    stmt = client->cal.sql.sel.objIDByGuid;
                } else {
                    return(ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1));
                }
            } else if (toupper(*ptr) == 'N') {
                *next = strchr(ptr, ' ');
                if ((*next == NULL) || (**next == '\0') || (**next == ' ')) {
                    ptr++;
                    ccode = NmapSqlBegin(client);
                } else {
                    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
                }

                if (ccode == SQLITE_DONE) {
                    ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_OBJID_BY_ALTNAME, &(client->cal.sql.sel.objIDByAltName));
                    stmt = client->cal.sql.sel.objIDByAltName;
                } else {
                    return(ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1));
                }
            } else {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }

            if ((ccode == SQLITE_OK) 
                    && ((ccode = sqlite3_bind_text(stmt, 1, ptr, -1, NULL)) == SQLITE_OK) 
                    && ((ccode = sqlite3_step(stmt)) == SQLITE_ROW)) {
                *id = sqlite3_column_int64(stmt, 0);

                ccode = 0;
            } else if (ccode == SQLITE_DONE) {
                ccode = ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1);
            } else {
                ccode = ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1);
            }

            sqlite3_reset(stmt);

            NmapSqlEnd(client);

            return(ccode);
        }
    }

    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
}

CalendarInfoStruct *
AddCalendarInfo(NMAPClient *client, unsigned long increment)
{
    return(NULL);
}

int 
WaitforAndLockCalendar(const unsigned char *store, unsigned char *user, unsigned char *calendar)
{
    return(-1);
}

void
UnlockCalendar(const unsigned char *store, unsigned char *user, unsigned char *calendar)
{
    return;
}

int 
ParseCalendar(NMAPClient *client)
{
    return(-1);
}

static int 
StorePropertyParameters(NMAPClient *client, ICal2Parameter *parameters, NmapCalID propID, int *position)
{
    int ccode = SQLITE_OK;
    int place = 0;
    unsigned long count;
    ICal2ParameterDetail *detail;

    detail = &(parameters->items.detail[0]);
    for (count = 0; (ccode == SQLITE_OK) && (count < parameters->items.count); count++, detail++, place++) {
        if (((ccode = sqlite3_bind_int(client->cal.sql.ins.param, 1, 0)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.param, 2, propID)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.param, 3, *position)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.param, 4, parameters->type)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.ins.param, 5, parameters->name, -1, NULL)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.param, 6, place)) == SQLITE_OK)) {
            switch (parameters->type) {
                case ICAL2_PARAMETER_ALTREP: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->altRep.uri, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_CN: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->cn.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_CUTYPE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->cuType.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_DELEGATED_FROM: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->delFrom.calAddress, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_DELEGATED_TO: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->delTo.calAddress, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_DIR: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->dir.uri, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_ENCODING: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->encoding.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_FMTTYPE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->format.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_FBTYPE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->freeBusy.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_LANGUAGE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->language.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_MEMBER: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->member.calAddress, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_PARTSTAT: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->partStat.status, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_RANGE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->range.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_RELATED: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->related.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_RELTYPE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->relType.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_ROLE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->role.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_RSVP: {
                    if (detail->rsvp.yes) {
                        ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, "TRUE", -1, NULL);
                    } else {
                        ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, "FALSE", -1, NULL);
                    }
                    break;
                }

                case ICAL2_PARAMETER_SENT_BY: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->sentBy.calAddress, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_TZID: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->tzID.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_VALUE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->value.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_X: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->x.value, -1, NULL);
                    break;
                }

                case ICAL2_PARAMETER_IANA: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.param, 7, detail->iana.value, -1, NULL);
                    break;
                }

                default: {
                    ccode = SQLITE_CONSTRAINT;
                    break;
                }
            }

            if ((ccode == SQLITE_OK) 
                    && ((ccode = sqlite3_step(client->cal.sql.ins.param)) == SQLITE_DONE)) {
                ccode = SQLITE_OK;
            }
        }

        sqlite3_reset(client->cal.sql.ins.param);

#if defined(DEBUG)
        if (ccode != SQLITE_OK) {
            XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
        }
#endif
    }

    *position += 1;

    return(ccode);
}

static int 
StoreComponentProperty(NMAPClient *client, ICal2Property *property, NmapCalID compID, int *position)
{
    int ccode = SQLITE_OK;
    int place = 0;
    int rsCodeLen;
    int rsDescriptionLen;
    unsigned long count;
    NmapCalID propID = 0;
    ICal2Parameter *parameter;
    ICal2ValueDetail *detail;
    ICal2Value *value;

    value = property->value;
    detail = &(value->items.detail[0]);
    for (count = 0; (ccode == SQLITE_OK) && (count < value->items.count); count++, detail++, place++) {
        if (((ccode = sqlite3_bind_int(client->cal.sql.ins.prop, 1, 0)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.prop, 2, compID)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.prop, 3, *position)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.prop, 4, value->type)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 5, property->name, -1, NULL)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.prop, 6, place)) == SQLITE_OK)) {

            switch (value->type) {
                case ICAL2_VALUE_TYPE_BINARY: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->binary.encoded, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_BOOLEAN: {
                    if (detail->boolean.yes) {
                        ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, "1", -1, NULL);
                    } else {
                        ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, "0", -1, NULL);
                    }

                    break;
                }

                case ICAL2_VALUE_TYPE_CAL_ADDRESS: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->calAddress.uri, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_DATE: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->date.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_DATE_TIME: {
                    detail->dateTime.value[8] = 'T';

                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->dateTime.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_DURATION: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->duration.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_FLOAT: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->f.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_INTEGER: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->i.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_PERIOD: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->period.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_RECUR: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->recur.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_REQUEST_STATUS: {
                    rsCodeLen = strlen(detail->requestStatus.code);
                    rsDescriptionLen = strlen(detail->requestStatus.description);

                    detail->requestStatus.code[rsCodeLen] = ';';
                    detail->requestStatus.description[rsDescriptionLen] = ';';

                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->requestStatus.code, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_STATUS: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->status.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_TEXT: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->text.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_TIME: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->time.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_URI: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->uri.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_UTC_OFFSET: {
                    ccode = sqlite3_bind_text(client->cal.sql.ins.prop, 7, detail->utcOffset.value, -1, NULL);
                    break;
                }

                case ICAL2_VALUE_TYPE_X:
                case ICAL2_VALUE_TYPE_IANA:
                case ICAL2_VALUE_TYPE_NONE:
                default: {
                    ccode = SQLITE_CONSTRAINT;
                    break;
                }
            }

            if ((ccode == SQLITE_OK) 
                    && ((ccode = sqlite3_step(client->cal.sql.ins.prop)) == SQLITE_DONE)) {
                ccode = SQLITE_OK;
            }

            switch (value->type) {
                case ICAL2_VALUE_TYPE_DATE_TIME: {
                    detail->dateTime.value[8] = '\0';
                    break;
                }

                case ICAL2_VALUE_TYPE_REQUEST_STATUS: {
                    detail->requestStatus.code[rsCodeLen] = '\0';
                    detail->requestStatus.description[rsDescriptionLen] = '\0';
                    break;
                }

                case ICAL2_VALUE_TYPE_BINARY:
                case ICAL2_VALUE_TYPE_BOOLEAN:
                case ICAL2_VALUE_TYPE_CAL_ADDRESS:
                case ICAL2_VALUE_TYPE_DATE:
                case ICAL2_VALUE_TYPE_DURATION:
                case ICAL2_VALUE_TYPE_FLOAT:
                case ICAL2_VALUE_TYPE_INTEGER:
                case ICAL2_VALUE_TYPE_PERIOD:
                case ICAL2_VALUE_TYPE_RECUR:
                case ICAL2_VALUE_TYPE_STATUS:
                case ICAL2_VALUE_TYPE_TEXT:
                case ICAL2_VALUE_TYPE_TIME:
                case ICAL2_VALUE_TYPE_URI:
                case ICAL2_VALUE_TYPE_UTC_OFFSET:
                case ICAL2_VALUE_TYPE_X:
                case ICAL2_VALUE_TYPE_IANA:
                case ICAL2_VALUE_TYPE_NONE:
                default: {
                    break;
                }
            }

            if (!propID && (ccode == SQLITE_OK)) {
                if (((ccode = sqlite3_bind_int64(client->cal.sql.sel.lastPropIDByComp, 1, compID)) == SQLITE_OK)
                        && ((ccode = sqlite3_step(client->cal.sql.sel.lastPropIDByComp)) == SQLITE_ROW)) {
                    propID = sqlite3_column_int64(client->cal.sql.sel.lastPropIDByComp, 0);
                }

                sqlite3_reset(client->cal.sql.sel.lastPropIDByComp);

                if (propID) {
                    ccode = SQLITE_OK;
                }
            }
        }

        sqlite3_reset(client->cal.sql.ins.prop);

#if defined(DEBUG)
        if (ccode != SQLITE_OK) {
            XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
        }
#endif
    }

    *position += 1;

    parameter = property->parameters.head;
    while (parameter && (ccode == SQLITE_OK)) {
        ccode = StorePropertyParameters(client, parameter, propID, position);

        parameter = parameter->next;
    }

    return(ccode);
}

static int 
StoreObjectComponent(NMAPClient *client, ICal2Component *component, NmapCalID calID, NmapCalID parentID, NmapCalID objID, int *position)
{
    int ccode;
    unsigned char start[20];
    unsigned char end[20];
    unsigned char due[20];
    unsigned char *organizer;
    unsigned char *summary;
    NmapCalID compID = 0;
    ICal2DateTimeValue dtStart;
    ICal2Component *child;
    ICal2Property *property;

    /*
        fixme

        Calculate end and due dates as well as start date.
    */
    switch (component->type) {
        case ICAL2_COMPONENT_VEVENT: {
            property = component->detail.vEvent.organizer;
            if (property) {
                organizer = property->value->items.detail[0].text.value;
            } else {
                organizer = NULL;
            }

            property = component->detail.vEvent.summary;
            if (property) {
                summary = property->value->items.detail[0].text.value;
            } else {
                summary = NULL;
            }

            if ((property = component->detail.vEvent.dtStart) != NULL) {
                NmapSqlDateTimeFromICal2DateTime(start, &(property->value->items.detail[0].dateTime));

                memcpy(&dtStart, &(property->value->items.detail[0].dateTime), sizeof(ICal2DateTimeValue));
            } else {
                NmapSqlDateTimeFromICal2DateTime(start, NULL);
            }

            if ((property = component->detail.vEvent.dtEnd) != NULL) {
                NmapSqlDateTimeFromICal2DateTime(end, &(property->value->items.detail[0].dateTime));
            } else if ((property = component->detail.vEvent.duration) != NULL) {
                /*
                    fixme

                    Adjust the value of dtStart, using duration, to calculate the end date.

                NmapSqlDateTimeFromICal2DateTime(end, &(property->value->items.detail[0].duration));
                */
            } else {
                return(SQLITE_ERROR);
            }

                NmapSqlDateTimeFromICal2DateTime(due, NULL);
            break;
        }

        case ICAL2_COMPONENT_VALARM:
        case ICAL2_COMPONENT_VTODO:
        case ICAL2_COMPONENT_VJOURNAL:
        case ICAL2_COMPONENT_VFREEBUSY:
        case ICAL2_COMPONENT_VTIMEZONE:
        case ICAL2_COMPONENT_DAYLIGHT:
        case ICAL2_COMPONENT_STANDARD: {
            /* fall-through for now */;
        }

        case ICAL2_COMPONENT_VCALENDAR:
        case ICAL2_COMPONENT_X:
        case ICAL2_COMPONENT_IANA:
        default: {
            organizer = NULL;
            summary = NULL;

            NmapSqlDateTimeFromICal2DateTime(start, NULL);
            NmapSqlDateTimeFromICal2DateTime(end, NULL);
            NmapSqlDateTimeFromICal2DateTime(due, NULL);
            break;
        }
    }

    if (((ccode = sqlite3_bind_int(client->cal.sql.ins.comp, 1, 0)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.comp, 2, objID)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.comp, 3, parentID)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_int(client->cal.sql.ins.comp, 4, component->type)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_int(client->cal.sql.ins.comp, 5, *position)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_int(client->cal.sql.ins.comp, 6, component->counts.length)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_text(client->cal.sql.ins.comp, 7, component->name, -1, NULL)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_text(client->cal.sql.ins.comp, 8, start, -1, NULL)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_text(client->cal.sql.ins.comp, 9, end, -1, NULL)) == SQLITE_OK) 
            && ((ccode = sqlite3_bind_text(client->cal.sql.ins.comp, 10, "now", -1, NULL)) == SQLITE_OK)) {
        ccode = sqlite3_step(client->cal.sql.ins.comp);
    }

    *position += 1;

    sqlite3_reset(client->cal.sql.ins.comp);

    if (ccode == SQLITE_DONE) {
        if (((ccode = sqlite3_bind_int64(client->cal.sql.sel.lastCompIDByObj, 1, objID)) == SQLITE_OK)
                && ((ccode = sqlite3_step(client->cal.sql.sel.lastCompIDByObj)) == SQLITE_ROW)) {
            compID = sqlite3_column_int64(client->cal.sql.sel.lastCompIDByObj, 0);
        }

        sqlite3_reset(client->cal.sql.sel.lastCompIDByObj);

#if defined(DEBUG)
    } else {
        XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif
    }

    if (ccode == SQLITE_ROW) {
        if (compID) {
            ccode = SQLITE_OK;
            property = component->properties.head;
            while (property && (ccode == SQLITE_OK)) {
                ccode = StoreComponentProperty(client, property, compID, position);

                property = property->next;
            }

            child = component->head;
            while (child && (ccode == SQLITE_OK)) {
                ccode = StoreObjectComponent(client, child, calID, compID, objID, position);

                child = child->next;
            }
        } else {
            ccode = SQLITE_NOTFOUND;

#if defined(DEBUG)
            XplConsolePrintf("NMAP [SQL %d]: New component \"%s\" not found.\r\n", __LINE__, component->name);
#endif
        }
#if defined(DEBUG)
    } else {
        XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif
    }

    return(ccode);
}

static int 
StoreCalendarObject(NMAPClient *client, ICal2Object *ical, unsigned char *sender, unsigned char *authSender, unsigned char *calendar, unsigned char *altName, unsigned char **guid)
{
    int ccode;
    int position = 0;
    unsigned char name[16];
    unsigned char *tGuid;
    NmapCalID calID = 0;
    NmapCalID objID = 0;
    ICal2Component *child;

    tGuid = GuidAlloc();
    if (tGuid) {
        if (((ccode = NmapSqlCalendarOpen(client)) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CALID_BY_NAME, &(client->cal.sql.sel.calIDByName))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_OBJECT, &(client->cal.sql.ins.object))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_OBJID_BY_GUID, &(client->cal.sql.sel.objIDByGuid))) == SQLITE_OK)
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_COMPONENT, &(client->cal.sql.ins.comp))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LAST_COMPID_BY_OBJECT, &(client->cal.sql.sel.lastCompIDByObj))) == SQLITE_OK)
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_PROPERTY, &(client->cal.sql.ins.prop))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LAST_PROPID_BY_COMP, &(client->cal.sql.sel.lastPropIDByComp))) == SQLITE_OK)
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_PARAMETER, &(client->cal.sql.ins.param))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_ALTNAME, &(client->cal.sql.ins.altName))) == SQLITE_OK)) {
            ccode = NmapSqlBegin(client);
        }
    } else {
        return(SQLITE_NOMEM);
    }

    do {
        if (ccode == SQLITE_DONE) {
            if (((ccode = sqlite3_bind_text(client->cal.sql.sel.calIDByName, 1, calendar, -1, NULL)) == SQLITE_OK) 
                    && ((ccode = sqlite3_step(client->cal.sql.sel.calIDByName)) == SQLITE_ROW)) {
                calID = sqlite3_column_int64(client->cal.sql.sel.calIDByName, 0);
            }

            sqlite3_reset(client->cal.sql.sel.calIDByName);
        } else {
#if defined(DEBUG)
            XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif
            break;
        }

        do {
            if ((ccode == SQLITE_ROW) && calID) {
                if (((ccode = sqlite3_bind_int(client->cal.sql.ins.object, 1, 0)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.object, 2, tGuid, -1, NULL)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.object, 3, calID)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_int(client->cal.sql.ins.object, 4, ical->length)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.object, 5, sender, -1, NULL)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.object, 6, authSender, -1, NULL)) == SQLITE_OK)) {
                    ccode = sqlite3_step(client->cal.sql.ins.object);
                }

                sqlite3_reset(client->cal.sql.ins.object);
                break;
            }

            if (((ccode = sqlite3_bind_text(client->cal.sql.sel.calIDByName, 1, "MAIN", -1, NULL)) == SQLITE_OK) 
                    && ((ccode = sqlite3_step(client->cal.sql.sel.calIDByName)) == SQLITE_ROW)) {
                calID = sqlite3_column_int64(client->cal.sql.sel.calIDByName, 0);
            }

            sqlite3_reset(client->cal.sql.sel.calIDByName);

            if (calID) {
                continue;
            }

#if defined(DEBUG)
            XplConsolePrintf("NMAP [SQL %d]: Calendar \"%s\" not found.\r\n", __LINE__, calendar);
#endif

            ccode = SQLITE_NOTFOUND;
            break;
        } while (TRUE);

        if (ccode == SQLITE_DONE) {
            if (((ccode = sqlite3_bind_text(client->cal.sql.sel.objIDByGuid, 1, tGuid, -1, NULL)) == SQLITE_OK) 
                    && ((ccode = sqlite3_step(client->cal.sql.sel.objIDByGuid)) == SQLITE_ROW)) {
                objID = sqlite3_column_int64(client->cal.sql.sel.objIDByGuid, 0);
            }

            sqlite3_reset(client->cal.sql.sel.objIDByGuid);
        } else {
#if defined(DEBUG)
            XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif

            break;
        }

        if (objID) {
            if (!altName) {
                sprintf(name, "%lu.ics", objID);
                altName = name;
            }

            if (ccode == SQLITE_ROW) {
                if (((ccode = sqlite3_bind_int(client->cal.sql.ins.altName, 1, 0)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_int64(client->cal.sql.ins.altName, 2, objID)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.altName, 3, altName, -1, NULL)) == SQLITE_OK)) {
                    ccode = sqlite3_step(client->cal.sql.ins.altName);
                }

                sqlite3_reset(client->cal.sql.ins.altName);
            } else {
#if defined(DEBUG)
                XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif

                break;
            }

            if (ccode == SQLITE_DONE) {
                ccode = SQLITE_OK;

                child = ical->components.head;
                while (child && (ccode == SQLITE_OK)) {
                    ccode = StoreObjectComponent(client, child, calID, 0, objID, &position);

                    child = child->next;
                }
            } else {
#if defined(DEBUG)
                XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif
                break;
            }
        } else {
            ccode = SQLITE_ERROR;

#if defined(DEBUG)
            XplConsolePrintf("NMAP [SQL %d]: New object unreadable.\r\n", __LINE__);
#endif

            break;
        }

        if (ccode == SQLITE_OK) {
            ccode = NmapSqlEnd(client);

            ccode = (ccode == SQLITE_DONE)? SQLITE_OK: ccode;
            break;
        }

#if defined(DEBUG)
        XplConsolePrintf("NMAP [SQL %d]: \"%s\"\r\n", __LINE__, sqlite3_errmsg(client->cal.handle));
#endif

        NmapSqlAbort(client);
        ccode = SQLITE_ABORT;
    } while (TRUE);

    if ((ccode == SQLITE_OK) && guid) {
        *guid = tGuid;
    } else if (tGuid) {
        GuidFree(tGuid);
    }

    NmapSqlCalendarClose(client);

    return(ccode);
}

int
StoreCalendarEvent(unsigned char *sender, unsigned char *authSender, unsigned char *recipient, unsigned char *calendar, FILE *data, unsigned long length, BOOL raw)
{
    int ccode;
    unsigned long spaceUsed = 0;
    unsigned long enforcedQuota = 0;
    unsigned long pos;
    unsigned char line[CONN_BUFSIZE + 1];
    MDBValueStruct *vs;
    NMAPClient *client;
    ICal2Object *iCal;

    if ((client = NMAPClientAlloc()) != NULL) {
        client->states |= NMAP_CLIENT_USER;

        strcpy(client->user, recipient);

        client->store = MsgFindUserStore(client->user, NMAP.path.mail);
    } else {
        return(DELIVER_TRY_LATER);
    }

    sprintf(client->path, "%s/%s", client->store, client->user);
    if (NMAP.quota.useUser || NMAP.quota.useSystem) {
        spaceUsed = XplGetDiskspaceUsed(client->path);
        spaceUsed += (length / 256 + 4) / 4;
        if (NMAP.quota.useUser) {
            vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
            if (MsgFindObject(client->user, client->dn, NULL, NULL, vs)) {
                MDBFreeValues(vs);

                if ((MsgGetUserFeature(client->dn, FEATURE_NMAP_QUOTA, MSGSRV_A_USE_QUOTA, vs) > 0) && (vs->Value[0][0] == '1')) {
                    MDBFreeValues(vs);
                    if (MsgGetUserFeature(client->dn, FEATURE_NMAP_QUOTA, MSGSRV_A_QUOTA_VALUE, vs) > 0) {
                        enforcedQuota = atol(vs->Value[0]);            
                    } else if (NMAP.quota.useSystem) {
                        enforcedQuota = NMAP.quota.defaultValue;
                    }
                } else if (NMAP.quota.useSystem) {
                    enforcedQuota = NMAP.quota.defaultValue;
                }
            }

            MDBDestroyValueStruct(vs); 
        } else {
            enforcedQuota = NMAP.quota.defaultValue;    
        } 

        if (enforcedQuota > 0) {
            if ((spaceUsed > enforcedQuota) && ((recipient[0] != NMAP.postMaster[0]) || (XplStrCaseCmp(recipient, NMAP.postMaster) != 0))) {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OVER_QUOTA, LOG_INFO, 0, recipient, sender, spaceUsed, enforcedQuota, NULL, 0);

                NMAPClientFree(client);

                return(DELIVER_QUOTA_EXCEEDED);
            } 
        }
    }

    /* Find the start of the ical object */
    do {
        pos = ftell(data);
        if (fgets(line, CONN_BUFSIZE, data) != NULL) {
            if (QuickCmp(line, "BEGIN:VCALENDAR\r\n")) {
                break;
            }
        }
    } while (!feof(data) && !ferror(data));

    ccode = DELIVER_PROCESSING_ERROR;
    if (feof(data) == 0) {
        fseek(data, pos, SEEK_SET);

        /* Here's where we parse the ical object */
        pos = ftell(data);
        iCal = ICal2ObjectParse(data, ICAL2_OBJECT_FLAG_FILE_HANDLE, length);
        if (iCal) {
            ccode = StoreCalendarObject(client, iCal, sender, authSender, calendar, NULL, NULL);

            ICal2ObjectFree(iCal);

            if (ccode == SQLITE_OK) {
                ccode = DELIVER_SUCCESS;

                XplSafeAdd(NMAP.stats.storedLocal.bytes, length);
                XplSafeIncrement(NMAP.stats.storedLocal.count);
                XplSafeIncrement(NMAP.stats.storedLocal.recipients);
            } else {
                ccode = DELIVER_TRY_LATER;
            }
        }
    }

    NMAPClientFree(client);

    return(ccode);
}

BOOL 
StoreCalendar(NMAPClient *client)
{
    return(FALSE);
}

static BOOL 
PurgeCalendar(NMAPClient *client)
{
    return(FALSE);
}

static int 
NmapSqlCalendarSelect(NMAPClient *client, unsigned char *calendar)
{
    int ccode;
    unsigned int allocated;
    unsigned char *text;
    NmapCalID objID;
    sqlite3_stmt *onObj;
    sqlite3_stmt *onComp;
    sqlite3_stmt *onProp;
    NmapSqlCalCompInfo *info = client->cal.objects;
    NmapSqlCalComp *comp;

    if (client->cal.objects) {
        info = client->cal.objects;
    } else if ((info = (NmapSqlCalCompInfo *)MemMalloc(sizeof(NmapSqlCalCompInfo) + (sizeof(NmapSqlCalComp) * 32))) != NULL) {
        client->cal.objects = info;

        info->allocated = 32;
        info->used = 0;
    } else {
        return(SQLITE_NOMEM);
    }

    comp = &(info->list[0]);
    ccode = NmapSqlBegin(client);
    if (ccode == SQLITE_DONE) {
        if (((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CALID_BY_NAME, &(client->cal.sql.sel.calIDByName))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CSINFO_ON_OBJ, &(client->cal.sql.info.onObj))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CSINFO_ON_COMP, &(client->cal.sql.info.onComp))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CSINFO_ON_PROP, &(client->cal.sql.info.onProp))) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.sel.calIDByName, 1, calendar, -1, NULL)) == SQLITE_OK)) {
            onObj = client->cal.sql.sel.calIDByName;
            if ((ccode = sqlite3_step(onObj)) == SQLITE_ROW) {
                client->cal.id = sqlite3_column_int64(onObj, 0);
                client->states |= NMAP_CLIENT_CALSEL;

                strcpy(client->cal.name, calendar);
            } else if ((XplStrCaseCmp(calendar, "MAIN") == 0) 
                    && ((ccode = sqlite3_reset(onObj)) == SQLITE_OK) 
                    && ((ccode = sqlite3_bind_text(onObj, 1, "MAIN", -1, NULL)) == SQLITE_OK) 
                    && ((ccode = sqlite3_step(onObj)) == SQLITE_ROW)) {
                client->states |= NMAP_CLIENT_CALSEL;
                client->cal.id = sqlite3_column_int64(onObj, 0);

                strcpy(client->cal.name, "MAIN");
            }

            sqlite3_reset(onObj);
        } else {
            client->states &= ~NMAP_CLIENT_CAL;
            return(ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
        }

        if (client->states & NMAP_CLIENT_CALSEL) {
            onObj = client->cal.sql.info.onObj;
            onComp = client->cal.sql.info.onComp;
            onProp = client->cal.sql.info.onProp;

            if ((ccode = sqlite3_bind_int64(onObj, 1, client->cal.id)) == SQLITE_OK) {
                ccode = sqlite3_step(onObj);
            }

            while (ccode == SQLITE_ROW) {
                objID = sqlite3_column_int64(onObj, 0);
                if ((ccode = sqlite3_bind_int64(onComp, 1, objID)) == SQLITE_OK) {
                    ccode = sqlite3_step(onComp);
                    while (ccode == SQLITE_ROW) {
                        if (info->used < info->allocated) {
                            comp->summary = NULL;
                            comp->organizer = NULL;

                            comp->compID = sqlite3_column_int64(onComp, 0);
                            comp->flags = sqlite3_column_int(onComp, 1);
                            comp->parentID = sqlite3_column_int64(onComp, 2);
                            comp->type = sqlite3_column_int(onComp, 3);
                            comp->position = sqlite3_column_int(onComp, 4);
                            comp->length = sqlite3_column_int(onComp, 5);
                            comp->start = sqlite3_column_double(onComp, 6);
                            comp->end = sqlite3_column_double(onComp, 7);
                            comp->modified = sqlite3_column_double(onComp, 8);

                            if (((ccode = sqlite3_bind_int64(onProp, 1, comp->compID)) == SQLITE_OK) 
                                    && ((ccode = sqlite3_bind_text(onProp, 2, ICAL2_KEYWORD_SUMMARY, -1, NULL)) == SQLITE_OK)) {
                                ccode = sqlite3_step(onProp);
                                if (ccode == SQLITE_ROW) {
                                    text = (unsigned char *)sqlite3_column_text(onProp, 0);
                                    if (text) {
                                        comp->summary = MemStrdup(text);
                                        if (comp->summary == NULL) {
                                            ccode = SQLITE_NOMEM;
                                            break;
                                        }
                                    }
                                }

                                sqlite3_reset(onProp);
                            } else {
                                break;
                            }

                            if (((ccode = sqlite3_bind_int64(onProp, 1, comp->compID)) == SQLITE_OK) 
                                    && ((ccode = sqlite3_bind_text(onProp, 2, ICAL2_KEYWORD_ORGANIZER, -1, NULL)) == SQLITE_OK)) {
                                ccode = sqlite3_step(onProp);
                                if (ccode == SQLITE_ROW) {
                                    text = (unsigned char *)sqlite3_column_text(onProp, 0);
                                    if (text) {
                                        comp->organizer = MemStrdup(text);
                                        if (comp->organizer == NULL) {
                                            ccode = SQLITE_NOMEM;
                                            break;
                                        }
                                    }
                                }

                                sqlite3_reset(onProp);
                            } else {
                                break;
                            }

                            comp++;
                            info->used++;

                            ccode = sqlite3_step(onComp);
                            continue;
                        }

                        allocated = info->allocated + 32;
                        info = (NmapSqlCalCompInfo *)MemRealloc(info, sizeof(NmapSqlCalCompInfo) + (sizeof(NmapSqlCalComp) * allocated));
                        if (info) {
                            client->cal.objects = info;

                            info->allocated = allocated;
                            comp = &(info->list[info->used]);
                            continue;
                        }

                        ccode = SQLITE_NOMEM;
                        break;
                    }

                    sqlite3_reset(onComp);
                }

                if (ccode == SQLITE_DONE) {
                    ccode = sqlite3_step(onObj);
                    continue;
                }

                break;
            }

            sqlite3_reset(onObj);

            if (ccode == SQLITE_DONE) {
                ccode = ConnWriteF(client->conn, "1000 %d %s\r\n", client->cal.id, MSG1000MBOX);
            } else {
                client->states &= ~(NMAP_CLIENT_CAL | NMAP_CLIENT_CALSEL);

                if (ccode != SQLITE_NOMEM) {
                    ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1);
                }
            }
        } else {
            client->states &= ~NMAP_CLIENT_CAL;
            ccode = ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1);
        }

        NmapSqlEnd(client);
        return(ccode);
    }

    client->states &= ~NMAP_CLIENT_CAL;
    return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
}

static int 
NmapSqlCalendarList(NMAPClient *client)
{
    int ccode;

    ccode = NmapSqlBegin(client);
    if (ccode == SQLITE_DONE) {
        if ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CALENDAR_ID_GUID_NAME, &(client->cal.sql.sel.calIDGuidName))) == SQLITE_OK) {
            client->cal.id = 0;

            ccode = ConnWriteF(client->conn, "2002-Coming up\r\n", 16);
            while ((ccode != -1) && ((ccode = sqlite3_step(client->cal.sql.sel.calIDGuidName)) == SQLITE_ROW)) {
                ccode = ConnWriteF(client->conn, "2002-%s %s %s\r\n", sqlite3_column_text(client->cal.sql.sel.calIDGuidName, 0), sqlite3_column_text(client->cal.sql.sel.calIDGuidName, 1), sqlite3_column_text(client->cal.sql.sel.calIDGuidName, 2));
            }

            if (ccode == SQLITE_DONE) {
                ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
            } else {
                client->states &= ~NMAP_CLIENT_CAL;
            }

            sqlite3_reset(client->cal.sql.sel.calIDGuidName);
        } else {
            client->states &= ~NMAP_CLIENT_CAL;

            ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
        }

        NmapSqlEnd(client);
    } else {
        client->states &= ~NMAP_CLIENT_CAL;

        ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsopen(void *param)
{
    int ccode;
    unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (!(client->states & NMAP_CLIENT_CAL) || ((*ptr == ' ') && !(client->states & NMAP_CLIENT_CALSEL))) {
        ccode = NmapSqlCalendarOpen(client);
    } else {
        return(ConnWrite(client->conn, MSG4221CALDONE, sizeof(MSG4221CALDONE) - 1));
    }

    if (ccode == SQLITE_OK) {
        client->states |= NMAP_CLIENT_CAL;

        if (*ptr == ' ') {
            ccode = NmapSqlCalendarSelect(client, ++ptr);
        } else {
            ccode = NmapSqlCalendarList(client);
        }
    }

    NmapSqlCalendarClose(client);

    return(ccode);
}

int 
NmapCommandCal(void *param)
{
    int count;
    unsigned char *ptr;
    unsigned char *ptr2;
    NMAPClient *client = (NMAPClient *)param;

    ptr = client->buffer + 3;
    count = strlen(ptr);
    ptr += count;
    ptr2 = ptr + 3;

    do {
        *ptr2-- = *ptr--;
    } while (count-- != 0);

    memcpy(client->buffer, "CSOPEN", 6);

    return(NmapCommandCsopen(param));
}

int 
NmapCommandCsorg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsinfo(void *param)
{
    int used;
    int ccode;
    unsigned char *ptr;
    NmapCalID id;
    NMAPClient *client = (NMAPClient *)param;
    NmapSqlCalCompInfo *info = client->cal.objects;
    NmapSqlCalComp *comp;

    /* CSINFO[ <component ID>] */
    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_CALSEL) {
            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    info = client->cal.objects;
    if (*ptr == '\0') {
        id = 0;

        used = info->used;
        comp = &(info->list[0]);

        ccode = ConnWriteF(client->conn, "2002 %lu\r\n", used);
        while ((ccode != -1) && used) {
            ccode = ConnWriteF(client->conn, "%d %I64d %I64d %I64d %d %lu %lu %lu %f %f %f \"%s\" \"%s\"\r\n", 
                        ++id, 
                        comp->compID, 
                        comp->objID, 
                        comp->parentID, 
                        comp->position, 
                        comp->type, 
                        comp->length, 
                        comp->flags, 
                        comp->start, 
                        comp->end, 
                        comp->modified, 
                        comp->organizer, 
                        comp->summary);

            used--;
            comp++;
        }

        if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
        }

        return(ccode);
    }

    if ((*ptr++ == ' ') && (isdigit(*ptr))) {
        id = atoi(ptr);
        if (id && (id <= info->used)) {
            id--;
            comp = &(info->list[id]);

            ccode = ConnWriteF(client->conn, "%d %I64d %I64d %I64d %d %lu %lu %lu %f %f %f \"%s\" \"%s\"\r\n", 
                        ++id, 
                        comp->compID, 
                        comp->objID, 
                        comp->parentID, 
                        comp->position, 
                        comp->type, 
                        comp->length, 
                        comp->flags, 
                        comp->start, 
                        comp->end, 
                        comp->modified, 
                        comp->organizer, 
                        comp->summary);
        } else {
            ccode = ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);
    }

    return(ccode);
}

int 
NmapCommandCsfilt(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsfind(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsgflg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

typedef struct _CSListNameValue {
    NmapCalID id;

    unsigned int flags;
    unsigned int place;
    unsigned int position;

    const unsigned char *name;
    const unsigned char *value;
} CSListNameValue;

static int 
SendObjectComponent(NMAPClient *client, NmapCalID parentID, NmapCalID compID, unsigned long flags, unsigned char *name, CSListNext *next)
{
    int ccode;
    unsigned int lastPosition;
    unsigned char *child = NULL;
    CSListNameValue prop;
    CSListNameValue param;
    sqlite3_stmt *onComp = client->cal.sql.list.onComp;
    sqlite3_stmt *onProp = client->cal.sql.list.onProp;
    sqlite3_stmt *onParam = client->cal.sql.list.onParam;

    if (((ccode = ConnWriteF(client->conn, "BEGIN:%s\r\n", name)) != -1) 
            && ((ccode = sqlite3_bind_int64(onProp, 1, compID)) == SQLITE_OK)) {
        memset(&prop, 0, sizeof(CSListNameValue));
        memset(&param, 0, sizeof(CSListNameValue));

        ccode = sqlite3_step(onProp);
    } else {
        return(-1);
    }

    while (ccode == SQLITE_ROW) {
        if (!prop.id) {
            prop.id = sqlite3_column_int64(onProp, 0);
            prop.flags = sqlite3_column_int(onProp, 1);
            prop.position = sqlite3_column_int(onProp, 2);
            prop.name = sqlite3_column_text(onProp, 3);
            prop.place = sqlite3_column_int(onProp, 4);
            prop.value = sqlite3_column_text(onProp, 5);
        }

        if (prop.name 
                && prop.value 
                && ((ccode = ConnWrite(client->conn, prop.name, strlen(prop.name))) != -1) 
                && ((ccode = sqlite3_bind_int64(onParam, 1, prop.id)) == SQLITE_OK)) {

            ccode = sqlite3_step(onParam);
            while (ccode == SQLITE_ROW) {
                param.id = sqlite3_column_int64(onParam, 0);
                param.flags = sqlite3_column_int(onParam, 1);
                param.name = sqlite3_column_text(onParam, 2);
                param.place = sqlite3_column_int(onParam, 3);
                param.value = sqlite3_column_text(onParam, 4);
                if (param.name && param.value) {
                    if (!param.place) {
                        ccode = ConnWriteF(client->conn, ";%s=%s", param.name, param.value);
                    } else {
                        ccode = ConnWriteF(client->conn, ",%s", param.value);
                    }

                    if (ccode != -1) {
                        ccode = sqlite3_step(onParam);
                        continue;
                    }
                }

                ccode = -1;
                break;
            }

            sqlite3_reset(onParam);

            if (ccode == -1) {
                break;
            }
        } else {
            ccode = -1;
            break;
        }

        lastPosition = prop.position;
        do {
            if (!prop.place) {
                ccode = ConnWriteF(client->conn, ":%s", prop.value);
            } else {
                ccode = ConnWriteF(client->conn, ",%s", prop.value);
            }

            if (ccode != -1) {
                prop.id = 0;
                if ((ccode = sqlite3_step(onProp)) == SQLITE_ROW) {
                    prop.id = sqlite3_column_int64(onProp, 0);
                    prop.flags = sqlite3_column_int(onProp, 1);
                    prop.position = sqlite3_column_int(onProp, 2);
                    prop.name = sqlite3_column_text(onProp, 3);
                    prop.place = sqlite3_column_int(onProp, 4);
                    prop.value = sqlite3_column_text(onProp, 5);

                    if (prop.position == lastPosition) {
                        continue;
                    }
                }

                ccode = ConnWrite(client->conn, "\r\n", 2);
                break;
            }

            ccode = -1;
            break;
        } while (ccode == SQLITE_ROW);

        if ((ccode != -1) && prop.id) {
            ccode = SQLITE_ROW;
            continue;
        }

        break;
    }

    sqlite3_reset(onProp);

    if (ccode != -1) {
        ccode = sqlite3_step(onComp);
    } else {
        return(-1);
    }

    if (ccode == SQLITE_ROW) {
        next->compID = sqlite3_column_int64(onComp, 0);
        next->flags = sqlite3_column_int(onComp, 1);
        next->parentID = sqlite3_column_int64(onComp, 2);
        next->name = MemStrdup(sqlite3_column_text(onComp, 3));
    } else if (ccode == SQLITE_DONE) {
        next->compID = 0;
        return(ConnWriteF(client->conn, "END:%s\r\n", name));
    } else {
        return(-1);
    }

    if (next->name) {
        while (next->parentID == compID) {
            child = next->name;

            ccode = SendObjectComponent(client, compID, next->compID, next->flags, next->name, next);

            MemFree(child);
            child = NULL;

            if ((ccode != -1) && next->compID) {
                continue;
            }

            break;
        }

        if (ccode != -1) {
            ccode = ConnWriteF(client->conn, "END:%s\r\n", name);
        }

        return(ccode);
    }

    if (child) {
        MemFree(child);
    }

    return(ccode);
}

int 
NmapCommandCslist(void *param)
{
    int ccode;
    unsigned int length;
    unsigned long flags;
    unsigned char *ptr;
    unsigned char *name;
    NmapCalID objID;
    NmapCalID compID;
    NmapCalID parentID;
    sqlite3_stmt *onObj;
    sqlite3_stmt *onComp;
    NMAPClient *client = (NMAPClient *)param;
    CSListNext next;

    memset(&next, 0, sizeof(CSListNext));

    /* CSLIST <objID> || G<guid> || N<altName> */
    if (client->states & NMAP_CLIENT_USER) {
        if (client->states & NMAP_CLIENT_CAL) {
            ptr = client->buffer + 6;
        } else {
            return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (((ccode = NmapCsID(client, ptr, &objID, &ptr)) != -1) 
            && objID) {
        ccode = NmapSqlBegin(client);
    } else {
        return(ccode);
    }

    if (ccode == SQLITE_DONE) {
        if (((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LIST_ON_OBJ, &(client->cal.sql.list.onObj))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LIST_ON_COMP, &(client->cal.sql.list.onComp))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LIST_ON_PROP, &(client->cal.sql.list.onProp))) == SQLITE_OK) 
                && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_LIST_ON_PARAM, &(client->cal.sql.list.onParam))) == SQLITE_OK)) {
            onObj = client->cal.sql.list.onObj;
            onComp = client->cal.sql.list.onComp;

            ccode = sqlite3_bind_int64(onObj, 1, objID);
        } else {
            NmapSqlEnd(client);
            return(ConnWrite(client->conn, MSG5004INTERNALERR, sizeof(MSG5004INTERNALERR) - 1));
        }

        if (ccode == SQLITE_OK) {
            ccode = sqlite3_step(onObj);
            if (ccode == SQLITE_DONE) {
                sqlite3_reset(onObj);

                NmapSqlEnd(client);
                return(ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1));
            }
        } else {
            NmapSqlEnd(client);
            return(ConnWrite(client->conn, MSG4224NOCSTORE, sizeof(MSG4224NOCSTORE) - 1));
        }

        flags = sqlite3_column_int(onObj, 0);
        length = sqlite3_column_int(onObj, 1);

        if (!(flags & NMAP_CAL_FLAG_DELETED) 
                && ((ccode = sqlite3_bind_int64(onComp, 1, objID)) == SQLITE_OK) 
                && ((ccode = ConnWriteF(client->conn, "2023 %lu\r\n", length)) != -1)) {
            if ((ccode = sqlite3_step(onComp)) == SQLITE_ROW) {
                compID = sqlite3_column_int64(onComp, 0);
                flags = sqlite3_column_int(onComp, 1);
                parentID = sqlite3_column_int64(onComp, 2);
                name = MemStrdup(sqlite3_column_text(onComp, 3));
                if (name) {
                    ccode = SendObjectComponent(client, 0, compID, flags, name, &next);

                    MemFree(name);

                    if (ccode != -1) {
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                    }
                } else {
                    ccode = ConnWrite(client->conn, MSG5230NOMEMORYERR, sizeof(MSG5230NOMEMORYERR) - 1);
                }
            } else {
                ccode = ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1);
            }

            sqlite3_reset(onComp);
        } else if (ccode != -1) {
            ccode = ConnWrite(client->conn, MSG4220NOMSG, sizeof(MSG4220NOMSG) - 1);
        }

        sqlite3_reset(onObj);

        NmapSqlEnd(client);
    } else {
        ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
    }

    return(ccode);
}

int 
NmapCommandCscrea(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *guid;
    NmapCalID calID;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
        if ((*ptr == ' ') && !strchr(++ptr, ' ')) {
            guid = GuidAlloc();
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (guid) {
        ccode = NmapSqlCalendarOpen(client);
    } else {
        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->user, ptr, NMAP_GUID_LENGTH, 0, NULL, 0);

        return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
    }

    if ((ccode == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_CALENDAR, &(client->cal.sql.ins.cal))) == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CALID_BY_NAME, &(client->cal.sql.sel.calIDByName))) == SQLITE_OK)) {
        if (((ccode = NmapSqlBegin(client)) == SQLITE_DONE) 
                && ((ccode = sqlite3_bind_int(client->cal.sql.ins.cal, 1, 0)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.ins.cal, 2, guid, -1, NULL)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.ins.cal, 3, ptr, -1, NULL)) == SQLITE_OK)) {
            ccode = sqlite3_step(client->cal.sql.ins.cal);

            sqlite3_reset(client->cal.sql.ins.cal);

            if (ccode == SQLITE_DONE) {
                ccode = NmapSqlEnd(client);
                if (ccode == SQLITE_DONE) {
                    ccode = ConnWrite(client->conn, MSG1000CALENDARMADE, sizeof(MSG1000CALENDARMADE) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                }
            } else {
                calID = 0;
                if ((ccode = sqlite3_bind_text(client->cal.sql.sel.calIDByName, 1, ptr, -1, NULL)) == SQLITE_OK) {
                    ccode = sqlite3_step(client->cal.sql.sel.calIDByName);
                    if (ccode == SQLITE_ROW) {
                        calID = sqlite3_column_int64(client->cal.sql.sel.calIDByName, 0);
                    }

                    sqlite3_reset(client->cal.sql.sel.calIDByName);
                }

                if (calID) {
                    ccode = ConnWrite(client->conn, MSG4226CALEXISTS, sizeof(MSG4226CALEXISTS) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                }

                NmapSqlAbort(client);
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
    }

    NmapSqlCalendarClose(client);

    GuidFree(guid);

    return(ccode);
}

int 
NmapCommandCscheck(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCscomp(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

static int 
CopyCalendarFromLocalToLocal(NMAPClient *client, unsigned int id, unsigned char *target)
{
    return(0);
}

static int 
CopyCalendarFromLocalToShare(NMAPClient *client, unsigned int id, unsigned char *target)
{
    return(0);
}

static int 
CopyCalendarFromShareToLocal(NMAPClient *client, unsigned int id, unsigned char *target)
{
    return(0);
}

static int 
CopyCalendarFromShareToShare(NMAPClient *client, unsigned int id, unsigned char *target)
{
    return(0);
}

int 
NmapCommandCscopy(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsdele(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsdflg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsaflg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsatnd(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsupda(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsunsubs(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsrmov(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsrnam(void *param)
{
    int ccode;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *guid = NULL;
    NmapCalID calID;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
        if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr && ((ptr2 = strchr(ptr, ' ')) != NULL)) {
            *ptr2++ = '\0';
        } else {
            return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    if (XplStrCaseCmp(ptr, "MAIN") == 0) {
        guid = GuidAlloc();
        if (!guid) {
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->user, ptr2, NMAP_GUID_LENGTH, 0, NULL, 0);

            return(ConnWrite(client->conn, MSG5001NOMEMORY, sizeof(MSG5001NOMEMORY) - 1));
        }
    }

    if (XplStrCaseCmp(client->cal.name, ptr) != 0) {
        if (((strlen(client->store) + strlen(client->user) + strlen(ptr) + 4) < XPL_MAX_PATH) 
                && ((strlen(client->store) + strlen(client->user) + strlen(ptr2) + 4) < XPL_MAX_PATH)) {
            ccode = NmapSqlCalendarOpen(client);
        } else {
            if (guid) {
                GuidFree(guid);
            }

            return(ConnWrite(client->conn, MSG4230PATHTOOLONG, sizeof(MSG4230PATHTOOLONG) - 1));
        }
    } else {
        if (guid) {
            GuidFree(guid);
        }

        return(ConnWrite(client->conn, MSG4227TGTCALSELECTED, sizeof(MSG4227TGTCALSELECTED) - 1));
    }

    if ((ccode == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLUPD_CALENDAR_NAME, &(client->cal.sql.upd.renCal))) == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLINS_CALENDAR, &(client->cal.sql.ins.cal))) == SQLITE_OK) 
            && ((ccode = NmapSqlPrepare(client->cal.handle, SQLSEL_CALID_BY_NAME, &(client->cal.sql.sel.calIDByName))) == SQLITE_OK)) {
        if (((ccode = NmapSqlBegin(client)) == SQLITE_DONE) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.upd.renCal, 1, ptr2, -1, NULL)) == SQLITE_OK) 
                && ((ccode = sqlite3_bind_text(client->cal.sql.upd.renCal, 2, ptr, -1, NULL)) == SQLITE_OK)) {
            ccode = sqlite3_step(client->cal.sql.upd.renCal);

            sqlite3_reset(client->cal.sql.upd.renCal);

            if (ccode == SQLITE_DONE) {
                if (guid 
                        && ((ccode = sqlite3_bind_int(client->cal.sql.ins.cal, 1, 0)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.cal, 2, guid, -1, NULL)) == SQLITE_OK) 
                        && ((ccode = sqlite3_bind_text(client->cal.sql.ins.cal, 3, "MAIN", -1, NULL)) == SQLITE_OK)) {
                    ccode = sqlite3_step(client->cal.sql.ins.cal);

                    sqlite3_reset(client->cal.sql.ins.cal);
                }

                if (ccode == SQLITE_DONE) {
                    ccode = NmapSqlEnd(client);
                    if (ccode == SQLITE_DONE) {
                        ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                    } else {
                        ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                    }
                } else {
                    NmapSqlAbort(client);

                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                }
            } else {
                calID = 0;
                if ((ccode = sqlite3_bind_text(client->cal.sql.sel.calIDByName, 1, ptr, -1, NULL)) == SQLITE_OK) {
                    ccode = sqlite3_step(client->cal.sql.sel.calIDByName);
                    if (ccode == SQLITE_ROW) {
                        calID = sqlite3_column_int64(client->cal.sql.sel.calIDByName, 0);
                    }

                    sqlite3_reset(client->cal.sql.sel.calIDByName);
                }

                NmapSqlAbort(client);

                if (calID) {
                    ccode = ConnWrite(client->conn, MSG4226CALEXISTS, sizeof(MSG4226CALEXISTS) - 1);
                } else {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                }
            }
        } else {
            ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
    }

    NmapSqlCalendarClose(client);

    GuidFree(guid);

    return(ccode);
}

int 
NmapCommandCsstraw(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsstor(void *param)
{
    int ccode;
    unsigned long id;
    unsigned long count;
    unsigned long spaceUsed = 0;
    unsigned long enforcedQuota = 0;
    unsigned char *ptr;
    unsigned char *ptr2;
    unsigned char *altName = NULL;
    unsigned char *calendar;
    unsigned char *sender;
    unsigned char *authSender;
    struct stat sb;
    FILE *writeCal;
    MDBValueStruct *vs;
    NMAPClient *client = (NMAPClient *)param;
    ICal2Object *iCal;

    if (client->states & NMAP_CLIENT_USER) {
        ptr = client->buffer + 6;
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    /* CSSTOR <calendar>[[ <count>][ FROM <sender> <authenticated sender>]][ ALTNAME <name>]CRLF */
    if ((*ptr++ == ' ') && (!isspace(*ptr)) && *ptr) {
        sender = client->user;
        authSender = client->user;

        calendar = ptr;

        ptr2 = strchr(ptr, ' ');
        if (ptr2) {
            *ptr2++ = '\0';

            if (isdigit(*ptr2)) {
                count = atol(ptr2);

                ptr2 = strchr(ptr2, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';
                }
            } else {
                count = 0;
            }

            if (ptr2 && (XplStrNCaseCmp(ptr2, "FROM ", 5) == 0)) {
                sender = ptr2 + 5;
                ptr2 = strchr(sender, ' ');
                if (ptr2) {
                    *ptr2++ = '\0';

                    authSender = ptr2;

                    ptr2 = strchr(ptr2, ' ');
                    if (ptr2) {
                        *ptr2++ = '\0';
                    }
                } else {
                    return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
                }
            }

            if (ptr2 && (XplStrNCaseCmp(ptr2, "ALTNAME ", 8) == 0)) {
                altName = ptr2 + 8;
                ptr2 = NULL;
            }

            if (ptr2) {
                return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
            }
        } else {
            count = 0;
        }
    } else {
        return(ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1));
    }

    /*
        fixme

        The new iCal2 parser can be modified to read from a socked in 
        addition to memory and a file handle.
    */
    if (!(NMAP.flags & NMAP_FLAG_DISK_SPACE_LOW)) {
        XplWaitOnLocalSemaphore(NMAP.queue.semaphore);
        id = NMAP.queue.id++ & ((1 << 28) - 1);
        XplSignalLocalSemaphore(NMAP.queue.semaphore);

        sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);
        writeCal = fopen(client->path, "wb");
        if (writeCal) {
            ccode = ConnWrite(client->conn, "2002 Send calendar data\r\n", 25);
            if (ccode != -1) {
                ConnFlush(client->conn);
            }
        } else {
            return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG5221SPACELOW, sizeof(MSG5221SPACELOW) - 1));
    }

    if (count > 0) {
        ccode = ConnReadToFile(client->conn, writeCal, count);
    } else {
        ccode = ConnReadToFileUntilEOS(client->conn, writeCal);
    }

    if (ccode != -1) {
        do {
            if (!(NMAP.flushFlags & FLUSH_CAL_ON_STORE)) {
                fclose(writeCal);
            } else {
                XplFlushWrites(writeCal);
                fclose(writeCal);
            }

            stat(client->path, &sb);

            if (NMAP.quota.useUser || NMAP.quota.useSystem) {
                sprintf(client->path, "%s/%s", client->store, client->user);

                spaceUsed = XplGetDiskspaceUsed(client->path);
                spaceUsed += (sb.st_size / 256 + 4) / 4;
                if (NMAP.quota.useUser) {
                    vs = MDBCreateValueStruct(NMAP.handle.directory, NULL);
                    if (MsgFindObject(client->user, client->dn, NULL, NULL, vs)) {
                        MDBFreeValues(vs);

                        if ((MsgGetUserFeature(client->dn, FEATURE_NMAP_QUOTA, MSGSRV_A_USE_QUOTA, vs) > 0) && (vs->Value[0][0] == '1')) {
                            MDBFreeValues(vs);
                            if (MsgGetUserFeature(client->dn, FEATURE_NMAP_QUOTA, MSGSRV_A_QUOTA_VALUE, vs) > 0) {
                                enforcedQuota = atol(vs->Value[0]);            
                            } else if (NMAP.quota.useSystem) {
                                enforcedQuota = NMAP.quota.defaultValue;
                            }
                        } else if (NMAP.quota.useSystem) {
                            enforcedQuota = NMAP.quota.defaultValue;
                        }
                    }

                    MDBDestroyValueStruct(vs); 
                } else {
                    enforcedQuota = NMAP.quota.defaultValue;    
                } 

                if (enforcedQuota > 0) {
                    if ((spaceUsed > enforcedQuota) && ((client->user[0] != NMAP.postMaster[0]) || (XplStrCaseCmp(client->user, NMAP.postMaster) != 0))) {
                        LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_OVER_QUOTA, LOG_INFO, 0, client->user, sender, spaceUsed, enforcedQuota, NULL, 0);

                        ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
                        break;
                    } 
                }

                sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);
            }

            writeCal = fopen(client->path, "rb");
            if (writeCal) {
                iCal = ICal2ObjectParse(writeCal, ICAL2_OBJECT_FLAG_FILE_HANDLE, sb.st_size);
                if (iCal) {
                    ptr = NULL;

                    ccode = StoreCalendarObject(client, iCal, sender, authSender, calendar, altName, &ptr);
                    if (ccode == SQLITE_OK) {
                        ccode = ConnWriteF(client->conn, MSG1000STORED, ptr);

                        XplSafeAdd(NMAP.stats.storedLocal.bytes, sb.st_size);
                        XplSafeIncrement(NMAP.stats.storedLocal.count);
                        XplSafeIncrement(NMAP.stats.storedLocal.recipients);
                    } else if (ccode == SQLITE_BUSY) {
                        ccode = ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1);
                    } else {
                        ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                    }

                    ICal2ObjectFree(iCal);

                    if (ptr) {
                        GuidFree(ptr);
                    }

                    break;
                }

                ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                break;
            }

            ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
            break;
        } while (TRUE);
    } else {
        ccode = errno;

        if (!(NMAP.flushFlags & FLUSH_CAL_ON_STORE)) {
            fclose(writeCal);
        } else {
            XplFlushWrites(writeCal);
            fclose(writeCal);
        }

        if (ccode == ENOSPC) {
            ccode = ConnWrite(client->conn, MSG5220QUOTAERR, sizeof(MSG5220QUOTAERR) - 1);
        } else {
            ccode = ConnWrite(client->conn, MSG4229CANTWRITEMBOX, sizeof(MSG4229CANTWRITEMBOX) - 1);
        }
    }

    NmapSqlCalendarClose(client);

    sprintf(client->path, "%s/c%07lx.in", NMAP.path.spool, id);
    unlink(client->path);
    
    return(ccode);
}

int 
NmapCommandCsstat(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsshow(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCsshowsub(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCssubs(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCssalv(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCssflg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandCssinfo(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

static int 
CalSqlQueryCB(void *param, int cCount, char **cValues, char **cNames)
{
    int i;
    QueryResults *qData = (QueryResults *)param;

    if (qData->column.count == 0) {
        qData->column.count = cCount;

        for (i = 0; i < cCount; i++) {
            MDBAddValue(cNames[i], qData->column.names);
        }
    }

    qData->row.count++;
    for (i = 0; i < cCount; i++) {
        if (cValues[i]) {
            MDBAddValue(cValues[i], qData->row.values);
        } else {
            MDBAddValue("NULL", qData->row.values);
        }
    }

    return(0);
}

int 
NmapCommandCssql(void *param)
{
    int ccode;
    int llen;
    unsigned long used;
    unsigned long cCount;
    unsigned char *ptr;
    NMAPClient *client = (NMAPClient *)param;
    QueryResults results;

    if (client->states & NMAP_CLIENT_CAL) {
        ptr = client->buffer + 5;

        memset(&results, 0, sizeof(results));
    } else {
        return(ConnWrite(client->conn, MSG4225NOCALENDAR, sizeof(MSG4225NOCALENDAR) - 1));
    }

    /* CSSQL <SQL query> */
    if ((*ptr++ == ' ') && (!isspace(*ptr))) {
        if (((results.column.names = MDBCreateValueStruct(NMAP.handle.directory, NULL)) != NULL) 
                && ((results.row.values = MDBShareContext(results.column.names)) != NULL)) {
            ccode = NmapSqlBegin(client);
            if (ccode == SQLITE_DONE) {
                ccode = sqlite3_exec(client->cal.handle, ptr, CalSqlQueryCB, (void *)&results, NULL);

                NmapSqlEnd(client);
            } else {
                return(ConnWrite(client->conn, MSG4120CALLOCKED, sizeof(MSG4120CALLOCKED) - 1));
            }

            if (ccode != SQLITE_OK) {
                XplConsolePrintf("hulanmap: Failed calendar query for \"%s\" of \"%s\"; %s\r\n", client->user, ptr, sqlite3_errmsg(client->cal.handle));
                ccode = -1;
            }

            if (!ccode) {
                if (results.row.count) {
                    ccode = ConnWriteF(client->conn, "2002-%lu rows to follow\r\n", results.row.count + 2);
                }

                llen = 0;
                for (used = 0; (ccode != -1) && (used < results.column.names->Used); ) {
                    cCount = strlen(results.column.names->Value[used]);
                    llen += cCount;

                    ccode = ConnWrite(client->conn, results.column.names->Value[used], cCount);
                    if ((ccode != -1) && (++used < results.column.names->Used)) {
                        llen++;

                        ccode = ConnWrite(client->conn, "|", 1);
                    }
                }

                if (llen && (ccode != -1)) {
                    ccode = ConnWrite(client->conn, "\r\n", 2);
                }

                cCount = 0;
                while (llen && (ccode != -1)) {
                    ccode = ConnWrite(client->conn, "-", 1);

                    cCount++;
                    llen--;
                }

                if (cCount && (ccode != -1)) {
                    ccode = ConnWrite(client->conn, "\r\n", 2);
                }

                for (used = 0; (ccode != -1) && (used < results.row.values->Used); used += cCount) {
                    for (cCount = 0; (ccode != -1) && (cCount < results.column.names->Used); ) {
                        llen = strlen(results.row.values->Value[used + cCount]);

                        ccode = ConnWrite(client->conn, results.row.values->Value[used + cCount], llen);
                        if ((ccode != -1) && (++cCount < results.column.names->Used)) {
                            ccode = ConnWrite(client->conn, "|", 1);
                        }
                    }

                    if (ccode != -1) {
                        ccode = ConnWrite(client->conn, "\r\n", 2);
                    }
                }

                if (ccode != -1) {
                    ccode = ConnWrite(client->conn, MSG1000OK, sizeof(MSG1000OK) - 1);
                }
            } else {
                LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_DATABASE, LOGGER_EVENT_DATABASE_FIND_ERROR, LOG_ERROR, 0, ptr, NULL, ccode, 0, NULL, 0);
                ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
            }
        } else {
            LoggerEvent(NMAP.handle.logging, LOGGER_SUBSYSTEM_GENERAL, LOGGER_EVENT_NMAP_OUT_OF_MEMORY, LOG_CRITICAL, 0, client->buffer, NULL, sizeof(MDBValueStruct), 0, NULL, 0);
            ccode = ConnWrite(client->conn, MSG5230NOMEMORYERR, sizeof(MSG5230NOMEMORYERR) - 1);
        }

        if (results.row.values) {
            MDBDestroyValueStruct(results.row.values);
        }

        if (results.column.names) {
            MDBDestroyValueStruct(results.column.names);
        }
    } else {
        ccode = ConnWrite(client->conn, MSG3010BADARGC, sizeof(MSG3010BADARGC) - 1);
    }

    return(ccode);
}

int 
NmapCommandCspurg(void *param)
{
    return(ConnWrite(((NMAPClient *)param)->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1));
}

int 
NmapCommandShareCal(void *param)
{
    return(-1);
}

#if defined(DEBUG)
int 
NmapCommandCsfill(void *param)
{
    int i;
    int len;
    int ccode;
    unsigned char *ptr;
    unsigned char *guid = NULL;
    unsigned char *objects[] = {
        "BEGIN:VCALENDAR\r\n" \
        "PRODID:-//RDU Software//NONSGML HandCal//EN\r\n" \
        "VERSION:2.0\r\n" \
        "BEGIN:VTIMEZONE\r\n" \
        "TZID:US-Eastern\r\n" \
        "BEGIN:STANDARD\r\n" \
        "DTSTART:19981025T020000\r\n" \
        "RDATE:19981025T020000\r\n" \
        "TZOFFSETFROM:-0400\r\n" \
        "TZOFFSETTO:-0500\r\n" \
        "TZNAME:EST\r\n" \
        "END:STANDARD\r\n" \
        "BEGIN:DAYLIGHT\r\n" \
        "DTSTART:19990404T020000\r\n" \
        "RDATE:19990404T020000\r\n" \
        "TZOFFSETFROM:-0500\r\n" \
        "TZOFFSETTO:-0400\r\n" \
        "TZNAME:EDT\r\n" \
        "END:DAYLIGHT\r\n" \
        "END:VTIMEZONE\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:19980309T231000Z\r\n" \
        "UID:guid-1.host1.com\r\n" \
        "ORGANIZER;ROLE=CHAIR:MAILTO:mrbig@host.com\r\n" \
        "ATTENDEE;RSVP=TRUE;ROLE=REQ-PARTICIPANT;CUTYPE=GROUP:\r\n" \
        " MAILTO:employee-A@host.com\r\n" \
        "DESCRIPTION:Project XYZ Review Meeting\r\n" \
        "CATEGORIES:MEETING\r\n" \
        "CLASS:PUBLIC\r\n" \
        "CREATED:19980309T130000Z\r\n" \
        "SUMMARY:XYZ Project Review\r\n" \
        "DTSTART;TZID=US-Eastern:19980312T083000\r\n" \
        "DTEND;TZID=US-Eastern:19980312T093000\r\n" \
        "LOCATION:1CP Conference Room 4350\r\n" \
        "END:VEVENT\r\n" \
        "END:VCALENDAR\r\n", 

        "BEGIN:VCALENDAR\r\n" \
        "VERSION:2.0\r\n" \
        "PRODID:-//ORACLE//NONSGML OCDC.DCAF//EN\r\n" \
        "METHOD:PUBLISH\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+6+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RECURRENCE-ID:20050908T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050908T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050908T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+5+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RECURRENCE-ID:20050825T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050825T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050825T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+4+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RECURRENCE-ID:20050811T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050811T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050811T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+3+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RECURRENCE-ID:20050728T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050728T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050728T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+2+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RECURRENCE-ID:20050714T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050714T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050714T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+30169+102+1+455354238\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050606T071718Z-75d9-66-7598-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC-EventPub Conf Call\r\n" \
        "RDATE:20050908T170000Z\r\n" \
        "RDATE:20050825T170000Z\r\n" \
        "RDATE:20050811T170000Z\r\n" \
        "RDATE:20050728T170000Z\r\n" \
        "RDATE:20050714T170000Z\r\n" \
        "ORGANIZER;CN=Jeff McCullough:jeffmc@berkeley.edu\r\n" \
        "LOCATION:1-800-261-3225 (8409391#)\r\n" \
        "DTSTART:20050616T170000Z\r\n" \
        "DTSTAMP:20050606T071720Z\r\n" \
        "DTEND:20050616T180000Z\r\n" \
        "CLASS:PUBLIC\r\n" \
        "END:VEVENT\r\n" \
        "END:VCALENDAR\r\n", 

        "BEGIN:VCALENDAR\r\n" \
        "CALSCALE:GREGORIAN\r\n" \
        "PRODID:-//Cyrusoft International\\, Inc.//Mulberry v4.0//EN\r\n" \
        "VERSION:2.0\r\n" \
        "X-WR-CALNAME:calconnect calls\r\n" \
        "BEGIN:VTIMEZONE\r\n" \
        "LAST-MODIFIED:20040110T032845Z\r\n" \
        "TZID:US/Eastern\r\n" \
        "BEGIN:DAYLIGHT\r\n" \
        "DTSTART:20000404T020000\r\n" \
        "RRULE:FREQ=YEARLY;BYDAY=1SU;BYMONTH=4\r\n" \
        "TZNAME:EDT\r\n" \
        "TZOFFSETFROM:-0500\r\n" \
        "TZOFFSETTO:-0400\r\n" \
        "END:DAYLIGHT\r\n" \
        "BEGIN:STANDARD\r\n" \
        "DTSTART:20001026T020000\r\n" \
        "RRULE:FREQ=YEARLY;BYDAY=-1SU;BYMONTH=10\r\n" \
        "TZNAME:EST\r\n" \
        "TZOFFSETFROM:-0400\r\n" \
        "TZOFFSETTO:-0500\r\n" \
        "END:STANDARD\r\n" \
        "END:VTIMEZONE\r\n" \
        "BEGIN:VTIMEZONE\r\n" \
        "LAST-MODIFIED:20040110T032845Z\r\n" \
        "TZID:US/Pacific\r\n" \
        "BEGIN:DAYLIGHT\r\n" \
        "DTSTART:20000404T020000\r\n" \
        "RRULE:FREQ=YEARLY;BYDAY=1SU;BYMONTH=4\r\n" \
        "TZNAME:PDT\r\n" \
        "TZOFFSETFROM:-0800\r\n" \
        "TZOFFSETTO:-0700\r\n" \
        "END:DAYLIGHT\r\n" \
        "BEGIN:STANDARD\r\n" \
        "DTSTART:20001026T020000\r\n" \
        "RRULE:FREQ=YEARLY;BYDAY=-1SU;BYMONTH=10\r\n" \
        "TZNAME:PST\r\n" \
        "TZOFFSETFROM:-0700\r\n" \
        "TZOFFSETTO:-0800\r\n" \
        "END:STANDARD\r\n" \
        "END:VTIMEZONE\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162433Z\r\n" \
        "DTSTART;TZID=US/Eastern:20050608T110000\r\n" \
        "DURATION:PT1H\r\n" \
        "RRULE:FREQ=WEEKLY;COUNT=4\r\n" \
        "SUMMARY:TC-RECURR\r\n" \
        "UID:174A88F20D9DDC07241EA846@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162459Z\r\n" \
        "DTSTART;TZID=US/Pacific:20050609T100000\r\n" \
        "DURATION:PT1H\r\n" \
        "SUMMARY:TC-USECASE\r\n" \
        "UID:1EC87FBF6FCA8E5105700CB8@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162639Z\r\n" \
        "DTSTART;TZID=US/Eastern:20050610T130000\r\n" \
        "DURATION:PT1H\r\n" \
        "SUMMARY:TC-TIMEZONE\r\n" \
        "UID:33792EACFEE952AC51D04C27@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162558Z\r\n" \
        "DTSTART;TZID=US/Eastern:20050615T130000\r\n" \
        "DURATION:PT1H\r\n" \
        "SUMMARY:TC-REALTIME\r\n" \
        "UID:5A1F0A525FF85356BA439D8E@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162355Z\r\n" \
        "DTSTART;TZID=US/Eastern:20050606T123000\r\n" \
        "DURATION:PT1H\r\n" \
        "SUMMARY:DST Task Force\r\n" \
        "UID:5DCE100D8958AA5F3DFB12D2@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162255Z\r\n" \
        "DTSTART;TZID=US/Eastern:20050606T113000\r\n" \
        "DURATION:PT1H\r\n" \
        "LAST-MODIFIED:20050606T162648Z\r\n" \
        "RRULE:FREQ=WEEKLY;COUNT=10\r\n" \
        "SEQUENCE:4\r\n" \
        "SUMMARY:TC-CHAIRS\r\n" \
        "UID:7B2A562B4296F81771E6339A@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "DTSTAMP:20050606T162523Z\r\n" \
        "DTSTART;TZID=US/Pacific:20050616T100000\r\n" \
        "DURATION:PT1H\r\n" \
        "SUMMARY:TC-EVENTPUB\r\n" \
        "UID:83FDA5D2697823FAB5EB5394@ninevah.local\r\n" \
        "END:VEVENT\r\n" \
        "END:VCALENDAR\r\n", 

        "BEGIN:VCALENDAR\r\n" \
        "VERSION:2.0\r\n" \
        "PRODID:-//ORACLE//NONSGML Windows Desktop Calendar 10.1.1.0.2.1748//EN\r\n" \
        "METHOD:PUBLISH\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+117240+391+2+451757179\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050425T160619Z-1c9f8-187-60db-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC Timezone conf call\r\n" \
        "STATUS:TENTATIVE\r\n" \
        "RECURRENCE-ID:20050519T170000Z\r\n" \
        "PRIORITY:0\r\n" \
        "ORGANIZER;CN=Simon Vaillancourt:mailto:SIMON.VAILLANCOURT@ORACLE.COM\r\n" \
        "DESCRIPTION:Phone numbers:\\n\\n1-888-967-2253 (America\\, toll-free)\\n+44-11\r\n" \
        " 8-924-9000 (Europe)\\n+1-650-607-2253 (other international)\\n\\nMeeting ID \r\n" \
        " #:759620\\nMeeting Passcode:759620\\n\\n- Problem draft update/review\\n- Tim\r\n" \
        " ezone questionnaire answers\r\n" \
        "DTSTART:20050519T170000Z\r\n" \
        "DTSTAMP:20050606T154734Z\r\n" \
        "DTEND:20050519T180000Z\r\n" \
        "CREATED:20050425T160619Z\r\n" \
        "X-ORACLE-CLASS:NORMAL\r\n" \
        "CLASS:PRIVATE\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+117240+391+3+451757179\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050425T160619Z-1c9f8-187-60db-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC Timezone conf call\r\n" \
        "STATUS:TENTATIVE\r\n" \
        "RECURRENCE-ID:20050610T170000Z\r\n" \
        "PRIORITY:0\r\n" \
        "ORGANIZER;CN=Simon Vaillancourt:mailto:SIMON.VAILLANCOURT@ORACLE.COM\r\n" \
        "DESCRIPTION:Phone numbers:\\n\\n1-888-967-2253 (America\\, toll-free)\\n+44-11\r\n" \
        " 8-924-9000 (Europe)\\n+1-650-607-2253 (other international)\\n\\nMeeting ID \r\n" \
        " #:759620\\nMeeting Passcode:759620\\n\\n- Problem draft update/review\\n- Tim\r\n" \
        " ezone questionnaire answers\r\n" \
        "DTSTART:20050610T170000Z\r\n" \
        "DTSTAMP:20050606T154734Z\r\n" \
        "DTEND:20050610T180000Z\r\n" \
        "CREATED:20050425T160619Z\r\n" \
        "X-ORACLE-CLASS:NORMAL\r\n" \
        "CLASS:PRIVATE\r\n" \
        "END:VEVENT\r\n" \
        "BEGIN:VEVENT\r\n" \
        "X-ORACLE-EVENTINSTANCE-GUID:I1+117240+391+1+451757179\r\n" \
        "X-ORACLE-EVENTTYPE:APPOINTMENT\r\n" \
        "UID:20050425T160619Z-1c9f8-187-60db-Oracle\r\n" \
        "TRANSP:OPAQUE\r\n" \
        "SUMMARY:TC Timezone conf call\r\n" \
        "STATUS:TENTATIVE\r\n" \
        "RDATE:20050519T170000Z\r\n" \
        "RDATE:20050610T170000Z\r\n" \
        "ORGANIZER;CN=Simon Vaillancourt:mailto:SIMON.VAILLANCOURT@ORACLE.COM\r\n" \
        "DESCRIPTION:Phone numbers:\\n\\n1-888-967-2253 (America\\, toll-free)\\n+44-11\r\n" \
        " 8-924-9000 (Europe)\\n+1-650-607-2253 (other international)\\n\\nMeeting ID \r\n" \
        " #:759620\\nMeeting Passcode:759620\\n\\n- Problem draft update/review\\n- Tim\r\n" \
        " ezone questionnaire answers\r\n" \
        "DTSTART:20050512T170000Z\r\n" \
        "DTSTAMP:20050606T154734Z\r\n" \
        "DTEND:20050512T180000Z\r\n" \
        "CREATED:20050425T160619Z\r\n" \
        "X-ORACLE-CLASS:NORMAL\r\n" \
        "CLASS:PRIVATE\r\n" \
        "END:VEVENT\r\n" \
        "END:VCALENDAR\r\n", 

        NULL
    };
    ICal2Object *iCal;
    NMAPClient *client = (NMAPClient *)param;

    if (client->states & NMAP_CLIENT_USER) {
        if (!(client->states & NMAP_CLIENT_CAL)) {
            ptr = client->buffer + 6;
            if (*ptr++ == '\0') {
                ptr = "MAIN";
            }
        } else {
            return(ConnWrite(client->conn, MSG4221CALDONE, sizeof(MSG4221CALDONE) - 1));
        }
    } else {
        return(ConnWrite(client->conn, MSG3241NOUSER, sizeof(MSG3241NOUSER) - 1));
    }

    sprintf(client->path, "%s/%s/calendar.db", client->store, client->user);
    unlink(client->path);

    if ((ccode = NmapSqlCalendarOpen(client)) == SQLITE_OK) {
        for (i = 0; objects[i]; i++) {
            len = strlen(objects[i]);

            iCal = ICal2ObjectParse(objects[i], ICAL2_OBJECT_FLAG_MEMORY, len);
            if (iCal) {
                guid = NULL;

                ccode = StoreCalendarObject(client, iCal, client->user, client->user, ptr, NULL, &guid);
                if (ccode == SQLITE_OK) {
                    ccode = ConnWriteF(client->conn, MSG1000STORED, guid);
                    if (ccode != -1) {
                        ccode = SQLITE_OK;
                    }

                    XplSafeAdd(NMAP.stats.storedLocal.bytes, len);
                    XplSafeIncrement(NMAP.stats.storedLocal.count);
                    XplSafeIncrement(NMAP.stats.storedLocal.recipients);
                } else {
                    ccode = ConnWrite(client->conn, MSG4228CANTWRITECAL, sizeof(MSG4228CANTWRITECAL) - 1);
                }

                ICal2ObjectFree(iCal);

                if (guid) {
                    GuidFree(guid);
                    guid = NULL;
                }

                if (ccode == SQLITE_OK) {
                    continue;
                }

                ccode = SQLITE_ERROR;
            } else {
                ccode = SQLITE_NOMEM;
            }

            break;
        }

        NmapSqlCalendarClose(client);
    } else {
        ccode = ConnWrite(client->conn, MSG4224CANTREADCAL, sizeof(MSG4224CANTREADCAL) - 1);
    }

    return(ccode);
}
#endif /* defined(DEBUG) */
#endif /* defined(NMAP_SQL_CALENDARS) */
