/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <mdb.h>

#include "rules.h"

#define PRODUCT_NAME "Hula Rules Agent"
#define PRODUCT_DESCRIPTION "Provide processing of user-created message rules."
#define PRODUCT_VERSION "$Revision: 1.8 $"

static BOOL ReadRulesVariable(unsigned int variable, unsigned char *data, size_t *length);

static BOOL RulesDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL RulesShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);
static BOOL SendRulesStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection);

ManagementCommands RulesManagementCommands[] = {
    { DMCMC_HELP, RulesDMCCommandHelp }, 
    { DMCMC_SHUTDOWN, RulesShutdown },
    { DMCMC_STATS, SendRulesStatistics }, 
    { DMCMC_DUMP_MEMORY_USAGE, ManagementMemoryStats }, 
};

ManagementVariables RulesManagementVariables[] = {
    { DMCMV_SERVER_THREAD_COUNT, DMCMV_SERVER_THREAD_COUNT_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_CONNECTION_COUNT, DMCMV_CONNECTION_COUNT_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_IDLE_CONNECTION_COUNT, DMCMV_IDLE_CONNECTION_COUNT_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_NMAP_ADDRESS, DMCMV_NMAP_ADDRESS_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_OFFICIAL_NAME, DMCMV_OFFICIAL_NAME_HELP, ReadRulesVariable, NULL }, 
    { RULEMV_USE_USER_RULES, RULEMV_USE_USER_RULES_HELP, ReadRulesVariable, NULL }, 
    { RULEMV_USE_SYSTEM_RULES, RULEMV_USE_SYSTEM_RULES_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_REVISIONS, DMCMV_REVISIONS_HELP, ReadRulesVariable, NULL }, 
    { DMCMV_VERSION, DMCMV_VERSION_HELP, ReadRulesVariable, NULL }, 
};

ManagementVariables *
GetRulesManagementVariables(void)
{
    return(RulesManagementVariables);
}

int 
GetRulesManagementVariablesCount(void)
{
    return(sizeof(RulesManagementVariables) / sizeof(ManagementVariables));
}

ManagementCommands *
GetRulesManagementCommands(void)
{
    return(RulesManagementCommands);
}

int 
GetRulesManagementCommandsCount(void)
{
    return(sizeof(RulesManagementCommands) / sizeof(ManagementCommands));
}

static BOOL 
RulesShutdown(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    XplThreadID id;

    if (response) {
        if (!arguments) {
            if (Rules.nmap.conn) {
                *response = MemStrdup("Shutting down.\r\n");
                if (*response) {
                    id = XplSetThreadGroupID(Rules.id.group);

                    Rules.state = RULES_STATE_UNLOADING;

                    if (Rules.nmap.conn) {
                        ConnClose(Rules.nmap.conn, 1);
                        Rules.nmap.conn = NULL;
                    }

                    if (closeConnection) {
                        *closeConnection = TRUE;
                    }

                    XplSetThreadGroupID(id);
                }
            } else if (Rules.state != RULES_STATE_RUNNING) {
                *response = MemStrdup("Shutdown in progress.\r\n");
            }

            if (*response) {
                return(TRUE);
            }

            return(FALSE);
        }

        *response = MemStrdup("arguments not allowed.\r\n");
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
RulesDMCCommandHelp(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    BOOL responded = FALSE;

    if (response) {
        if (arguments) {
            switch(toupper(arguments[0])) {
                case 'M': {
                    if (XplStrCaseCmp(arguments, DMCMC_DUMP_MEMORY_USAGE) == 0) {
                        if ((*response = MemStrdup(DMCMC_DUMP_MEMORY_USAGE_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                case 'S': {
                    if (XplStrCaseCmp(arguments, DMCMC_SHUTDOWN) == 0) {
                        if ((*response = MemStrdup(DMCMC_SHUTDOWN_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    } else if (XplStrCaseCmp(arguments, DMCMC_STATS) == 0) {
                        if ((*response = MemStrdup(DMCMC_STATS_HELP)) != NULL) {
                            responded = TRUE;
                        }

                        break;
                    }
                }

                default: {
                    break;
                }
            }
        } else if ((*response = MemStrdup(DMCMC_HELP_HELP)) != NULL) {
            responded = TRUE;
        }

        if (responded || ((*response = MemStrdup(DMCMC_UNKOWN_COMMAND)) != NULL)) {
            return(TRUE);
        }
    }

    return(FALSE);
}

static BOOL 
SendRulesStatistics(unsigned char *arguments, unsigned char **response, BOOL *closeConnection)
{
    MemStatistics poolStats;

    if (!arguments && response) {
        memset(&poolStats, 0, sizeof(MemStatistics));

        *response = MemMalloc(sizeof(PRODUCT_NAME)
                            + sizeof(PRODUCT_SHORT_NAME)
                            + 124);

        MemPrivatePoolStatistics(Rules.nmap.pool, &poolStats);

        if (*response) {
            sprintf(*response, "%s (%s: v%d.%d.%d)\r\n%lu:%lu:%lu:%lu:%d:%d:%d\r\n", 
                    PRODUCT_NAME, 
                    PRODUCT_SHORT_NAME, 
                    PRODUCT_MAJOR_VERSION, 
                    PRODUCT_MINOR_VERSION, 
                    PRODUCT_LETTER_VERSION, 
                    poolStats.totalAlloc.count, 
                    poolStats.totalAlloc.size, 
                    poolStats.pitches, 
                    poolStats.strikes, 
                    XplSafeRead(Rules.server.active), 
                    XplSafeRead(Rules.nmap.worker.active), 
                    XplSafeRead(Rules.nmap.worker.idle));

            return(TRUE);
        }

        if ((*response = MemStrdup("Out of memory.\r\n")) != NULL) {
            return(TRUE);
        }
    } else if ((arguments) && ((*response = MemStrdup("arguments not allowed.\r\n")) != NULL)) {
        return(TRUE);
    }

    return(FALSE);
}

static BOOL 
ReadRulesVariable(unsigned int variable, unsigned char *data, size_t *length)
{
	size_t count;
	unsigned char *ptr;

	switch (variable) {
		case 0: {
			if (data && (*length > 12)) {
				sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Rules.server.active));
			}

			*length = 12;
			break;
		}

		case 1: {
			if (data && (*length > 12)) {
				sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Rules.nmap.worker.active));
			}

			*length = 12;
			break;
		}

		case 2: {
			if (data && (*length > 12)) {
				sprintf(data, "%010lu\r\n", (long unsigned int)XplSafeRead(Rules.nmap.worker.idle));
			}

			*length = 12;
			break;
		}

		case 3: {
			count = strlen(Rules.nmap.address) + 2;
			if (data && (*length > count)) {
				sprintf(data, "%s\r\n", Rules.nmap.address);
			}

			*length = count;
			break;
		}

		case 4: {
			count = strlen(Rules.officialName) + 2;
			if (data && (*length > count)) {
				sprintf(data, "%s\r\n", Rules.officialName);
			}

			*length = count;
			break;
		}

		case 5: {
			if ((Rules.flags & RULES_FLAG_USER_RULES) == FALSE) {
				ptr = "FALSE\r\n";
				count = 7;
			} else {
				ptr = "TRUE\r\n";
				count = 6;
			}

			if (data && (*length > count)) {
				strcpy(data, ptr);
			}

			*length = count;
			break;
		}

		case 6: {
			if ((Rules.flags & RULES_FLAG_SYSTEM_RULES) == FALSE) {
				ptr = "FALSE\r\n";
				count = 7;
			} else {
				ptr = "TRUE\r\n";
				count = 6;
			}

			if (data && (*length > count)) {
				strcpy(data, ptr);
			}

			*length = count;
			break;
		}

		case 7: {
			unsigned char	version[30];

			PVCSRevisionToVersion(PRODUCT_VERSION, version);
			count = strlen(version) + 13;

			if (data && (*length > count)) {
				ptr = data;

				PVCSRevisionToVersion(PRODUCT_VERSION, version);
				ptr += sprintf(ptr, "Rulesrv.c: %s\r\n", version);

				*length = ptr - data;
			} else {
				*length = count;
			}

			break;
		}

		case 8: {
			DMC_REPORT_PRODUCT_VERSION(data, *length);
			break;
		}
	}

	return(TRUE);
}
