/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>

#define ICAL2_C

#include <libical2.h>

/*
    RFC 2445 Section 4.6.6 Alarm Component

    alarmc
        "BEGIN" ":" "VALARM" CRLF

        (audioprop / dispprop / emailprop / procprop)

        "END" ":" "VALARM" CRLF
*/
BOOL 
ICal2AlarmComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(TRUE);
}

/*
    RFC 2445 Section 4.6.1 Event Component

    eventc
        "BEGIN" ":" "VEVENT" CRLF

        *(
            ; the following are optional,
            ; but MUST NOT occur more than once
            class / created / description / dtstart / geo /
            last-mod / location / organizer / priority /
            dtstamp / seq / status / summary / transp /
            uid / url / recurid /

            ; either 'dtend' or 'duration' may appear in
            ; a 'eventprop', but 'dtend' and 'duration'
            ; MUST NOT occur in the same 'eventprop'
            dtend / duration /

            ; the following are optional,
            ; and MAY occur more than once
            attach / attendee / categories / comment /
            contact / exdate / exrule / rstatus / related /
            resources / rdate / rrule / x-prop
        )

        *alarmc

        "END" ":" "VEVENT" CRLF
*/
BOOL 
ICal2EventComponentParser(ICal2Object *ical)
{
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail.vEvent.next = NULL;
    if ((component->detail.vEvent.prev = ical->components.vEvent.tail) != NULL) {
        component->detail.vEvent.prev->detail.vEvent.next = component;
    } else {
        ical->components.vEvent.head = component;
    }
    ical->components.vEvent.tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_CLASS: {
                property->list.prev = NULL;

                component->detail.vEvent.course = property;
                break;
            }

            case ICAL2_PROPERTY_CREATED: {
                property->list.prev = NULL;

                component->detail.vEvent.created = property;
                break;
            }

            case ICAL2_PROPERTY_DESCRIPTION: {
                property->list.prev = NULL;

                component->detail.vEvent.description = property;
                break;
            }

            case ICAL2_PROPERTY_DTSTART: {
                property->list.prev = NULL;

                component->detail.vEvent.dtStart = property;
                break;
            }

            case ICAL2_PROPERTY_GEO: {
                property->list.prev = NULL;

                component->detail.vEvent.geo = property;
                break;
            }

            case ICAL2_PROPERTY_LAST_MODIFIED: {
                property->list.prev = NULL;

                component->detail.vEvent.lastModified = property;
                break;
            }

            case ICAL2_PROPERTY_LOCATION: {
                property->list.prev = NULL;

                component->detail.vEvent.location = property;
                break;
            }

            case ICAL2_PROPERTY_ORGANIZER: {
                property->list.prev = NULL;

                component->detail.vEvent.organizer = property;
                break;
            }

            case ICAL2_PROPERTY_PRIORITY: {
                property->list.prev = NULL;

                component->detail.vEvent.priority = property;
                break;
            }

            case ICAL2_PROPERTY_DTSTAMP: {
                property->list.prev = NULL;

                component->detail.vEvent.dtStamp = property;
                break;
            }

            case ICAL2_PROPERTY_SEQUENCE: {
                property->list.prev = NULL;

                component->detail.vEvent.seq = property;
                break;
            }

            case ICAL2_PROPERTY_STATUS: {
                property->list.prev = NULL;

                component->detail.vEvent.status = property;
                break;
            }

            case ICAL2_PROPERTY_SUMMARY: {
                property->list.prev = NULL;

                component->detail.vEvent.summary = property;
                break;
            }

            case ICAL2_PROPERTY_TRANSP: {
                property->list.prev = NULL;

                component->detail.vEvent.transp = property;
                break;
            }

            case ICAL2_PROPERTY_UID: {
                property->list.prev = NULL;

                component->detail.vEvent.uid = property;
                break;
            }

            case ICAL2_PROPERTY_URL: {
                property->list.prev = NULL;

                component->detail.vEvent.url = property;
                break;
            }

            case ICAL2_PROPERTY_RECURRENCE_ID: {
                property->list.prev = NULL;

                component->detail.vEvent.recurID = property;
                break;
            }

            case ICAL2_PROPERTY_DTEND: {
                property->list.prev = NULL;

                component->detail.vEvent.dtEnd = property;
                break;
            }

            case ICAL2_PROPERTY_DURATION: {
                property->list.prev = NULL;

                component->detail.vEvent.duration = property;
                break;
            }

            case ICAL2_PROPERTY_ATTACH: {
                if ((property->list.prev = component->detail.vEvent.attach.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.attach.head = property;
                }
                component->detail.vEvent.attach.tail = property;

                break;
            }

            case ICAL2_PROPERTY_ATTENDEE: {
                if ((property->list.prev = component->detail.vEvent.attendee.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.attendee.head = property;
                }
                component->detail.vEvent.attendee.tail = property;

                break;
            }

            case ICAL2_PROPERTY_CATEGORIES: {
                if ((property->list.prev = component->detail.vEvent.categories.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.categories.head = property;
                }
                component->detail.vEvent.categories.tail = property;

                break;
            }

            case ICAL2_PROPERTY_COMMENT: {
                if ((property->list.prev = component->detail.vEvent.comment.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.comment.head = property;
                }
                component->detail.vEvent.comment.tail = property;

                break;
            }

            case ICAL2_PROPERTY_CONTACT: {
                if ((property->list.prev = component->detail.vEvent.contact.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.contact.head = property;
                }
                component->detail.vEvent.contact.tail = property;

                break;
            }

            case ICAL2_PROPERTY_EXDATE: {
                if ((property->list.prev = component->detail.vEvent.exDate.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.exDate.head = property;
                }
                component->detail.vEvent.exDate.tail = property;

                break;
            }

            case ICAL2_PROPERTY_EXRULE: {
                if ((property->list.prev = component->detail.vEvent.exRule.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.exRule.head = property;
                }
                component->detail.vEvent.exRule.tail = property;

                break;
            }

            case ICAL2_PROPERTY_REQUEST_STATUS: {
                if ((property->list.prev = component->detail.vEvent.rstatus.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.rstatus.head = property;
                }
                component->detail.vEvent.rstatus.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RELATED_TO: {
                if ((property->list.prev = component->detail.vEvent.related.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.related.head = property;
                }
                component->detail.vEvent.related.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RESOURCES: {
                if ((property->list.prev = component->detail.vEvent.resources.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.resources.head = property;
                }
                component->detail.vEvent.resources.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RDATE: {
                if ((property->list.prev = component->detail.vEvent.rDate.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.rDate.head = property;
                }
                component->detail.vEvent.rDate.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RRULE: {
                if ((property->list.prev = component->detail.vEvent.rRule.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.rRule.head = property;
                }
                component->detail.vEvent.rRule.tail = property;

                break;
            }

            case ICAL2_PROPERTY_X: {
                if ((property->list.prev = component->detail.vEvent.xProperties.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.xProperties.head = property;
                }
                component->detail.vEvent.xProperties.tail = property;

                break;
            }

            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_IANA:
            default: {
                if ((property->list.prev = component->detail.vEvent.iana.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vEvent.iana.head = property;
                }
                component->detail.vEvent.iana.tail = property;

                break;
            }
        }

        property = property->next;
    }

    return(TRUE);
}

/*
    RFC 2445 Section 4.6.4 Free/Busy Component

    freebusyc
        "BEGIN" ":" "VFREEBUSY" CRLF

        *(
            ; the following are optional,
            ; but MUST NOT occur more than once
            contact / dtstart / dtend / duration / dtstamp /
            organizer / uid / url /

            ; the following are optional,
            ; and MAY occur more than once
            attendee / comment / freebusy / rstatus / x-prop
        )

        "END" ":" "VFREEBUSY" CRLF
*/
BOOL 
ICal2FreeBusyComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(TRUE);
}

/*
    RFC 2445 Section 4.6.3 Journal Component

    journalc
        "BEGIN" ":" "VJOURNAL" CRLF

        *(
            ; the following are optional,
            ; but MUST NOT occur more than once
            class / created / description / dtstart / dtstamp /
            last-mod / organizer / recurid / seq / status /
            summary / uid / url /

            ; the following are optional,
            ; and MAY occur more than once
            attach / attendee / categories / comment /
            contact / exdate / exrule / related / rdate /
            rrule / rstatus / x-prop
        )

        "END" ":" "VJOURNAL" CRLF
*/
BOOL 
ICal2JournalComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(TRUE);
}

/*
    RFC 2445 Section 4.6.5 Time Zone Component

    standardc
        "BEGIN" ":" "STANDARD" CRLF

        tzprop

        "END" ":" "STANDARD" CRLF

    tzprop
        3*(
            ; the following are each REQUIRED,
            ; but MUST NOT occur more than once
            dtstart / tzoffsetto / tzoffsetfrom /

            ; the following are optional,
            ; and MAY occur more than once
            comment / rdate / rrule / tzname / x-prop
        )
*/
BOOL 
ICal2DaylightComponentParser(ICal2Object *ical)
{
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail.daylight.next = NULL;
    if ((component->detail.daylight.prev = ical->components.daylight.tail) != NULL) {
        component->detail.daylight.prev->detail.daylight.next = component;
    } else {
        ical->components.daylight.head = component;
    }
    ical->components.daylight.tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_DTSTART: {
                property->list.prev = NULL;

                component->detail.daylight.dtStart = property;
                break;
            }

            case ICAL2_PROPERTY_TZOFFSETFROM: {
                property->list.prev = NULL;

                component->detail.daylight.tzOffsetFrom = property;
                break;
            }

            case ICAL2_PROPERTY_TZOFFSETTO: {
                property->list.prev = NULL;

                component->detail.daylight.tzOffsetTo = property;
                break;
            }

            case ICAL2_PROPERTY_COMMENT: {
                if ((property->list.prev = component->detail.daylight.comment.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.comment.head = property;
                }
                component->detail.daylight.comment.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RDATE: {
                if ((property->list.prev = component->detail.daylight.rDate.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.rDate.head = property;
                }
                component->detail.daylight.rDate.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RRULE: {
                if ((property->list.prev = component->detail.daylight.rRule.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.rRule.head = property;
                }
                component->detail.daylight.rRule.tail = property;

                break;
            }

            case ICAL2_PROPERTY_TZNAME: {
                if ((property->list.prev = component->detail.daylight.tzName.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.tzName.head = property;
                }
                component->detail.daylight.tzName.tail = property;

                break;
            }

            case ICAL2_PROPERTY_X: {
                if ((property->list.prev = component->detail.daylight.xProperties.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.xProperties.head = property;
                }
                component->detail.daylight.xProperties.tail = property;

                break;
            }

            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_IANA:
            default: {
                if ((property->list.prev = component->detail.daylight.iana.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.daylight.iana.head = property;
                }
                component->detail.daylight.iana.tail = property;

                break;
            }
        }

        property = property->next;
    }

    if (component->detail.daylight.dtStart 
            && component->detail.daylight.tzOffsetFrom 
            && component->detail.daylight.tzOffsetTo) {
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.6.5 Time Zone Component

    standardc
        "BEGIN" ":" "STANDARD" CRLF

        tzprop

        "END" ":" "STANDARD" CRLF

    tzprop
        3*(
            ; the following are each REQUIRED,
            ; but MUST NOT occur more than once
            dtstart / tzoffsetto / tzoffsetfrom /

            ; the following are optional,
            ; and MAY occur more than once
            comment / rdate / rrule / tzname / x-prop
        )
*/
BOOL 
ICal2StandardComponentParser(ICal2Object *ical)
{
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail.standard.next = NULL;
    if ((component->detail.standard.prev = ical->components.standard.tail) != NULL) {
        component->detail.standard.prev->detail.standard.next = component;
    } else {
        ical->components.standard.head = component;
    }
    ical->components.standard.tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_DTSTART: {
                property->list.prev = NULL;

                component->detail.standard.dtStart = property;
                break;
            }

            case ICAL2_PROPERTY_TZOFFSETFROM: {
                property->list.prev = NULL;

                component->detail.standard.tzOffsetFrom = property;
                break;
            }

            case ICAL2_PROPERTY_TZOFFSETTO: {
                property->list.prev = NULL;

                component->detail.standard.tzOffsetTo = property;
                break;
            }

            case ICAL2_PROPERTY_COMMENT: {
                if ((property->list.prev = component->detail.standard.comment.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.comment.head = property;
                }
                component->detail.standard.comment.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RDATE: {
                if ((property->list.prev = component->detail.standard.rDate.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.rDate.head = property;
                }
                component->detail.standard.rDate.tail = property;

                break;
            }

            case ICAL2_PROPERTY_RRULE: {
                if ((property->list.prev = component->detail.standard.rRule.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.rRule.head = property;
                }
                component->detail.standard.rRule.tail = property;

                break;
            }

            case ICAL2_PROPERTY_TZNAME: {
                if ((property->list.prev = component->detail.standard.tzName.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.tzName.head = property;
                }
                component->detail.standard.tzName.tail = property;

                break;
            }

            case ICAL2_PROPERTY_X: {
                if ((property->list.prev = component->detail.standard.xProperties.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.xProperties.head = property;
                }
                component->detail.standard.xProperties.tail = property;

                break;
            }

            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_IANA:
            default: {
                if ((property->list.prev = component->detail.standard.iana.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.standard.iana.head = property;
                }
                component->detail.standard.iana.tail = property;

                break;
            }
        }

        property = property->next;
    }

    if (component->detail.standard.dtStart 
            && component->detail.standard.tzOffsetFrom 
            && component->detail.standard.tzOffsetTo) {
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.6.5 Time Zone Component

    timezonec
        "BEGIN" ":" "VTIMEZONE" CRLF

        2*(
            ; 'tzid' is required, but MUST NOT occur more
            ; than once
            tzid /

            ; 'last-mod' and 'tzurl' are optional,
            ; but MUST NOT occur more than once
            last-mod / tzurl /

            ; one of 'standardc' or 'daylightc' MUST occur
            ; and each MAY occur more than once.
            standardc / daylightc /

            ; the following is optional,
            ; and MAY occur more than once
            x-prop
        )

        "END" ":" "VTIMEZONE" CRLF

*/
BOOL 
ICal2TimeZoneComponentParser(ICal2Object *ical)
{
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail.vTimeZone.next = NULL;
    if ((component->detail.vTimeZone.prev = ical->components.vTimeZone.tail) != NULL) {
        component->detail.vTimeZone.prev->detail.vTimeZone.next = component;
    } else {
        ical->components.vTimeZone.head = component;
    }
    ical->components.vTimeZone.tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_TZID: {
                property->list.prev = NULL;

                component->detail.vTimeZone.tzID = property;
                break;
            }

            case ICAL2_PROPERTY_TZURL: {
                property->list.prev = NULL;

                component->detail.vTimeZone.tzUrl = property;
                break;
            }

            case ICAL2_PROPERTY_LAST_MODIFIED: {
                property->list.prev = NULL;

                component->detail.vTimeZone.lastModified = property;
                break;
            }

            case ICAL2_PROPERTY_X: {
                if ((property->list.prev = component->detail.vTimeZone.xProperties.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vTimeZone.xProperties.head = property;
                }
                component->detail.vTimeZone.xProperties.tail = property;

                break;
            }

            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_IANA:
            default: {
                if ((property->list.prev = component->detail.vTimeZone.iana.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vTimeZone.iana.head = property;
                }
                component->detail.vTimeZone.iana.tail = property;

                break;
            }
        }

        property = property->next;
    }

    if (component->detail.vTimeZone.tzID) {
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.6.2 To-do Component

    todoc
        "BEGIN" ":" "VTODO" CRLF

        *(
            ; the following are optional,
            ; but MUST NOT occur more than once
            class / completed / created / description / dtstamp /
            dtstart / geo / last-mod / location / organizer /
            percent / priority / recurid / seq / status /
            summary / uid / url /

            ; either 'due' or 'duration' may appear in
            ; a 'todoprop', but 'due' and 'duration'
            ; MUST NOT occur in the same 'todoprop'
            due / duration /

            ; the following are optional,
            ; and MAY occur more than once
            attach / attendee / categories / comment / contact /
            exdate / exrule / rstatus / related / resources /
            rdate / rrule / x-prop
        )
        
        *alarmc

        "END" ":" "VTODO" CRLF
*/
BOOL 
ICal2ToDoComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(FALSE);
}

/*
    RFC 2445 Section 4.6 Calendar Components

    x-comp
        "BEGIN" ":" x-name CRLF

        1*contentline

        "END" ":" x-name CRLF
*/
BOOL 
ICal2XComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(FALSE);
}

/*
    RFC 2445 Section 4.6 Calendar Components

    iana-comp
        "BEGIN" ":" iana-token CRLF

        1*contentline

        "END" ":" iana-token CRLF
*/
BOOL 
ICal2IanaComponentParser(ICal2Object *ical)
{
#if 0
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail..next = NULL;
    if ((component->detail..prev = ical->components..tail) != NULL) {
        component->detail..prev->detail..next = component;
    } else {
        ical->components..head = component;
    }
    ical->components..tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_CALSCALE:
            case ICAL2_PROPERTY_METHOD:
            case ICAL2_PROPERTY_PRODID:
            case ICAL2_PROPERTY_VERSION:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_X:
            case ICAL2_PROPERTY_IANA:
            default: {
                break;
            }
        }

        property = property->next;
    }
#endif

    return(FALSE);
}

/*
    RFC 2445 Section 4.4 iCalendar Object

    icalobject
        1*(
            "BEGIN" ":" "VCALENDAR" CRLF

            icalbody

            "END" ":" "VCALENDAR" CRLF
        )

    RFC 2445 Section 4.4 Calendar Components

    icalbody
        calprops component

    calprops
        2*(
            ; 'prodid' and 'version' are both REQUIRED,
            ; but MUST NOT occur more than once
            prodid /version /

            ; 'calscale' and 'method' are optional,
            ; but MUST NOT occur more than once
            calscale / method /

            ; the following are optional,
            ; and MAY occur more than once?
            x-prop
        )

    component
        1*(
            eventc / todoc / journalc / freebusyc /
            / timezonec / iana-comp / x-comp
        )

    iana-comp
        "BEGIN" ":" iana-token CRLF

        1*contentline

        "END" ":" iana-token CRLF

    x-comp
        "BEGIN" ":" x-name CRLF

        1*contentline

        "END" ":" x-name CRLF
*/
BOOL 
ICal2CalendarComponentParser(ICal2Object *ical)
{
    ICal2Property *property;
    ICal2Component *component;

    component = ical->parse.component;

    component->detail.vCalendar.next = NULL;
    if ((component->detail.vCalendar.prev = ical->components.vCalendar.tail) != NULL) {
        component->detail.vCalendar.prev->detail.vCalendar.next = component;
    } else {
        ical->components.vCalendar.head = component;
    }
    ical->components.vCalendar.tail = component;

    property = component->properties.head;
    while (property) {
        property->list.next = NULL;

        switch (property->type) {
            case ICAL2_PROPERTY_CALSCALE: {
                property->list.prev = NULL;

                component->detail.vCalendar.calScale = property;
                break;
            }

            case ICAL2_PROPERTY_METHOD: {
                property->list.prev = NULL;

                component->detail.vCalendar.method = property;
                break;
            }

            case ICAL2_PROPERTY_PRODID: {
                property->list.prev = NULL;

                component->detail.vCalendar.prodID = property;
                break;
            }

            case ICAL2_PROPERTY_VERSION: {
                property->list.prev = NULL;

                component->detail.vCalendar.version = property;
                break;
            }

            case ICAL2_PROPERTY_X: {
                if ((property->list.prev = component->detail.vCalendar.xProperties.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vCalendar.xProperties.head = property;
                }
                component->detail.vCalendar.xProperties.tail = property;

                break;
            }

            case ICAL2_PROPERTY_BEGIN:
            case ICAL2_PROPERTY_END:
            case ICAL2_PROPERTY_ATTACH:
            case ICAL2_PROPERTY_CATEGORIES:
            case ICAL2_PROPERTY_CLASS:
            case ICAL2_PROPERTY_COMMENT:
            case ICAL2_PROPERTY_DESCRIPTION:
            case ICAL2_PROPERTY_GEO:
            case ICAL2_PROPERTY_LOCATION:
            case ICAL2_PROPERTY_PERCENT_COMPLETE:
            case ICAL2_PROPERTY_PRIORITY:
            case ICAL2_PROPERTY_RESOURCES:
            case ICAL2_PROPERTY_STATUS:
            case ICAL2_PROPERTY_SUMMARY:
            case ICAL2_PROPERTY_COMPLETED:
            case ICAL2_PROPERTY_DTEND:
            case ICAL2_PROPERTY_DUE:
            case ICAL2_PROPERTY_DTSTART:
            case ICAL2_PROPERTY_DURATION:
            case ICAL2_PROPERTY_FREEBUSY:
            case ICAL2_PROPERTY_TRANSP:
            case ICAL2_PROPERTY_TZID:
            case ICAL2_PROPERTY_TZNAME:
            case ICAL2_PROPERTY_TZOFFSETFROM:
            case ICAL2_PROPERTY_TZOFFSETTO:
            case ICAL2_PROPERTY_TZURL:
            case ICAL2_PROPERTY_ATTENDEE:
            case ICAL2_PROPERTY_CONTACT:
            case ICAL2_PROPERTY_ORGANIZER:
            case ICAL2_PROPERTY_RECURRENCE_ID:
            case ICAL2_PROPERTY_RELATED_TO:
            case ICAL2_PROPERTY_URL:
            case ICAL2_PROPERTY_UID:
            case ICAL2_PROPERTY_EXDATE:
            case ICAL2_PROPERTY_EXRULE:
            case ICAL2_PROPERTY_RDATE:
            case ICAL2_PROPERTY_RRULE:
            case ICAL2_PROPERTY_ACTION:
            case ICAL2_PROPERTY_REPEAT:
            case ICAL2_PROPERTY_TRIGGER:
            case ICAL2_PROPERTY_CREATED:
            case ICAL2_PROPERTY_DTSTAMP:
            case ICAL2_PROPERTY_LAST_MODIFIED:
            case ICAL2_PROPERTY_SEQUENCE:
            case ICAL2_PROPERTY_REQUEST_STATUS:
            case ICAL2_PROPERTY_IANA:
            default: {
                if ((property->list.prev = component->detail.vCalendar.iana.tail) != NULL) {
                    property->list.prev->list.next = property;
                } else {
                    component->detail.vCalendar.iana.head = property;
                }
                component->detail.vCalendar.iana.tail = property;

                break;
            }
        }

        property = property->next;
    }

    if (component->detail.vCalendar.prodID 
            && component->detail.vCalendar.version) {
        return(TRUE);
    }

    return(FALSE);
}
