/****************************************************************************
 *
 * Copyright (c) 2005 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>
#include <hulautil.h>

#define ICAL2_C

#include <libical2.h>

#define ICAL2_RECURBY_LIST_LENGTH(c) (sizeof(ICal2RecurByList) + ((c) * (sizeof(unsigned int) + sizeof(enum ICal2RecurWeekdays) + sizeof(unsigned char *))))

/*
    RFC 2445 Section 4.3.4 Date

    date
        date-fullyear date-month date-mday

    date-fullyear
        4DIGIT

    date-month
        2DIGIT  ;01-12

    date-mday
        2DIGIT  ;01-28, 01-29, 01-30, 01-31
                ;based on month/year
*/
static unsigned char *
ICal2DateParser(ICal2DateValue *date)
{
    unsigned char *ptr = date->value;

    while (isdigit(*ptr)) {
        ptr++;
    }

    if ((ptr - date->value) == 8) {
        ptr = date->value;

        date->year = (*ptr++ - '0') * 1000;
        date->year += (*ptr++ - '0') * 100;
        date->year += (*ptr++ - '0') * 10;
        date->year += (*ptr++ - '0');

        date->month = (*ptr++ - '0') * 10;
        date->month += (*ptr++ - '0');

        date->day = (*ptr++ - '0') * 10;
        date->day += (*ptr++ - '0');

        return(ptr);
    }

    return(NULL);
}

/*
    RFC 2445 Section 4.3.12 Time

    time
        time-hour time-minute time-second [time-utc]

    time-hour
        2DIGIT ;0-23

    time-minute
        2DIGIT  ;00-59

    time-second
        2DIGIT  ;00-60
                ;The "60" value is used to account for "leap" seconds

    time-utc
        "Z"
*/
static unsigned char *
ICal2TimeParser(ICal2TimeValue *time)
{
    unsigned char *ptr = time->value;

    while (isdigit(*ptr)) {
        ptr++;
    }

    if ((ptr - time->value) == 6) {
        ptr = time->value;

        time->hour = (*ptr++ - '0') * 10;
        time->hour += (*ptr++ - '0');

        time->minute = (*ptr++ - '0') * 10;
        time->minute += (*ptr++ - '0');

        time->second = (*ptr++ - '0') * 10;
        time->second += (*ptr++ - '0');

        if (toupper(*ptr) == 'Z') {
            time->utc = TRUE;
            ptr++;
        } else {
            time->utc = FALSE;
        }

        return(ptr);
    }

    return(NULL);
}

/*
    RFC 2445 Section 4.3.11 Text

    text
        ;Folded according to description above
        *(TSAFE-CHAR / ":" / DQUOTE / ESCAPED-CHAR)

    ESCAPED-CHAR
        ; \\ encodes \, \N or \n encodes newline
        ; \; encodes ;, \, encodes ,
        "\\" / "\;" / "\," / "\N" / "\n")

    TSAFE-CHAR
        ; Any character except CTLs not needed by the current
        ; character set, DQUOTE, ";", ":", "\", ","
        %x20-21 / %x23-2B / %x2D-39 / %x3C-5B /
        %x5D-7E / NON-US-ASCII
*/
unsigned char *
ICal2TextParser(unsigned char *text)
{
    unsigned char *src;
    unsigned char *dest;

    src = dest = text;
    do {
        while (ICal2IsText(*src)) {
            *dest++ = *src++;
        }

        switch (*src) {
            case '\\': {
                switch (src[1]) {
                    case '\\' :
                    case ';': 
                    case ',': {
                        src++;
                        *dest++ = *src++;
                        continue;
                    }

                    case 'N':
                    case 'n': {
                        src += 2;
                        *dest++ = '\n';
                        continue;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case '\0':
            case ',':
            default: {
                if (src != dest) {
                    *dest = '\0';
                }

                break;
            }
        }

        break;
    } while (TRUE);

    return(src);
}

/*
    RFC 2445 Section 4.3.6 Duration

    dur-value
        (["+"] / "-") "P" (dur-date / dur-time / dur-week)

    dur-date
        dur-day [dur-time]

    dur-time
        "T" (dur-hour / dur-minute / dur-second)

    dur-week
        1*DIGIT "W"

    dur-hour
        1*DIGIT "H" [dur-minute]

    dur-minute
        1*DIGIT "M" [dur-second]

    dur-second
        1*DIGIT "S"

    dur-day
        1*DIGIT "D"
*/
static unsigned char *
ICal2DurationParser(ICal2DurationValue *duration)
{
    unsigned char *ptr;
    unsigned char *cur;
    unsigned char delim;

    ptr = duration->value;
    if (*ptr != '-') {
        duration->sign = 1;

        if (*ptr == '+') {
            ptr++;
        }
    } else {
        duration->sign = -1;

        ptr++;
    }

    if (toupper(*ptr++) == 'P') {
        do {
            switch (toupper(*ptr)) {
                case '0':
                case '1':
                case '2':
                case '3':
                case '4':
                case '5':
                case '6':
                case '7':
                case '8':
                case '9': {
                    /* ( dur-date [dur-time] / dur-week ) */
                    cur = ptr;

                    do {
                        ptr++;
                    } while (isdigit(*ptr));

                    delim = *ptr;
                    *ptr = '\0';
                    if (toupper(delim) == 'D') {
                        duration->days = atoi(cur);
                        *ptr++ = delim;

                        duration->total = (60 * 60 * 24) * duration->days;
                        continue;
                    }
                    
                    if (toupper(delim) == 'W') {
                        duration->weeks = atoi(cur);
                        *ptr++ = delim;

                        duration->total = (60 * 60 * 24 * 7) * duration->weeks;
                        duration->total *= duration->sign;

                        return(ptr);
                    }

                    *ptr = delim;
                    break;
                }

                case 'T': {
                    ptr++;

                    do {
                        /* dur-time */
                        cur = ptr;
                        while (isdigit(*ptr)) {
                            ptr++;
                        }

                        delim = *ptr;
                        *ptr = '\0';
                        if (toupper(delim) == 'H') {
                            duration->hours = atoi(cur);
                            *ptr++ = delim;

                            duration->total += (60 * 60) * duration->hours;
                            continue;
                        }
                        
                        if (toupper(delim) == 'M') {
                            duration->minutes = atoi(cur);
                            *ptr++ = delim;

                            duration->total += 60 * duration->minutes;
                            continue;
                        }

                        if (toupper(delim) == 'S') {
                            duration->seconds = atoi(cur);
                            *ptr++ = delim;

                            duration->total += duration->seconds;
                            break;
                        }

                        *ptr = delim;
                        break;
                    } while (isdigit(*ptr));

                    if (*ptr == '\0') {
                        duration->total *= duration->sign;
                        return(ptr);
                    }

                    break;
                }

                default: {
                    break;
                }
            }

            break;
        } while (TRUE);
    }

    return(NULL);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    "FREQ" "=" freq

    freq
        "SECONDLY" / "MINUTELY" / "HOURLY" / "DAILY" /
        "WEEKLY" / "MONTHLY" / "YEARLY"
*/
BOOL 
ICal2RecurFreqParser(ICal2Object *ical)
{
    unsigned char *key;
    unsigned char *ptr;
    unsigned char delim;
    BOOL result = FALSE;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    key = ptr = ical->parse.value + 5;
    while (isalpha(*ptr)) {
        ptr++;
    }

    delim = *ptr;
    *ptr = '\0';

    switch (toupper(*key)) {
        case 'D': {
            if (XplStrCaseCmp(key, "DAILY") == 0) {
                recur->freq = ICAL2_RECUR_DAILY;
                result = TRUE;
            }

            break;
        }

        case 'H': {
            if (XplStrCaseCmp(key, "HOURLY") == 0) {
                recur->freq = ICAL2_RECUR_HOURLY;
                result = TRUE;
            }

            break;
        }

        case 'M': {
            if (XplStrCaseCmp(key, "MONTHLY") == 0) {
                recur->freq = ICAL2_RECUR_MONTHLY;
                result = TRUE;
            } else if (XplStrCaseCmp(key, "MINUTELY") == 0) {
                recur->freq = ICAL2_RECUR_MINUTELY;
                result = TRUE;
            }

            break;
        }

        case 'S': {
            if (XplStrCaseCmp(key, "SECONDLY") == 0) {
                recur->freq = ICAL2_RECUR_SECONDLY;
                result = TRUE;
            }

            break;
        }

        case 'W': {
            if (XplStrCaseCmp(key, "WEEKLY") == 0) {
                recur->freq = ICAL2_RECUR_WEEKLY;
                result = TRUE;
            }

            break;
        }

        case 'Y': {
            if (XplStrCaseCmp(key, "YEARLY") == 0) {
                recur->freq = ICAL2_RECUR_YEARLY;
                result = TRUE;
            }

            break;
        }

        default: {
            break;
        }
    }

    *ptr = delim;

    ical->parse.value = ptr;
    return(result);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; either UNTIL or COUNT may appear in a 'recur',
    ; but UNTIL and COUNT MUST NOT occur in the same 'recur'
    ( ";" "UNTIL" "=" enddate ) /

    enddate
        date

        =/ date-time    ;An UTC value
*/
BOOL 
ICal2RecurUntilParser(ICal2Object *ical)
{
    unsigned long index;
    unsigned char *ptr;
    ICal2ParameterDetail *detail;
    ICal2Parameter *parameter;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    if ((parameter = ical->parse.property->parameters.head) != NULL) {
        do {
            if (parameter->type == ICAL2_PARAMETER_VALUE) {
                break;
            }

            parameter = parameter->next;
        } while (parameter);

        if (parameter) {
            detail = parameter->items.detail;
            for (index = 0; index < parameter->items.count; index++, detail++) {
                if (detail->value.type == ICAL2_VALUE_TYPE_DATE) {
                    recur->untilType = ICAL2_VALUE_TYPE_DATE;
                    recur->until.date.value = ical->parse.value + 6;

                    ptr = ICal2DateParser(&(recur->until.date));
                    if (ptr) {
                        ical->parse.value = ptr;
                        return(TRUE);
                    }

                    return(FALSE);
                }

                if (detail->value.type == ICAL2_VALUE_TYPE_DATE_TIME) {
                    break;
                }
            }
        }
    }

    recur->untilType = ICAL2_VALUE_TYPE_DATE_TIME;
    recur->until.end.date.value = ical->parse.value + 6;

    ptr = ICal2DateParser(&(recur->until.end.date));
    if (ptr && (toupper(*ptr) == 'T')) {
        *ptr++ = '\0';
        recur->until.end.time.value = ptr;
        ptr = ICal2TimeParser(&(recur->until.end.time));
        if (ptr) {
            ical->parse.value = ptr;
            return(TRUE);
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; either UNTIL or COUNT may appear in a 'recur',
    ; but UNTIL and COUNT MUST NOT occur in the same 'recur'
    ( ";" "COUNT" "=" 1*DIGIT ) /
*/
BOOL 
ICal2RecurCountParser(ICal2Object *ical)
{
    unsigned char *ptr;
    unsigned char *count;
    unsigned char delim;
    BOOL result = FALSE;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    count = ptr = ical->parse.value + 6;
    while (isdigit(*ptr)) {
        ptr++;
    }

    delim = *ptr;
    *ptr = '\0';

    recur->untilType = ICAL2_VALUE_TYPE_INTEGER;
    recur->until.count = atoi(count);

    *ptr = delim;

    ical->parse.value = ptr;
    return(TRUE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "INTERVAL" "=" 1*DIGIT ) /
*/
BOOL 
ICal2RecurIntervalParser(ICal2Object *ical)
{
    unsigned char *ptr;
    unsigned char *interval;
    unsigned char delim;
    BOOL result = FALSE;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    interval = ptr = ical->parse.value + 9;
    while (isdigit(*ptr)) {
        ptr++;
    }

    delim = *ptr;
    *ptr = '\0';

    recur->interval = atoi(interval);

    *ptr = delim;

    ical->parse.value = ptr;
    return(TRUE);
}

__inline unsigned char *
ICal2RecurByListNumbers(ICal2RecurValue *recur, unsigned char *line, int minimum, int maximum, ICal2RecurByList **recurByList)
{
    int sign;
    int count;
    unsigned char *ptr;
    unsigned char *number;
    unsigned char delim;
    BOOL result;
    ICal2RecurByListValue *value;
    ICal2RecurByList *recurBy;
    ICal2RecurByList *recurByNew;

    recurBy = (ICal2RecurByList *)MemMalloc(ICAL2_RECURBY_LIST_LENGTH(1));
    if (recurBy) {
        memset(recurBy, 0, ICAL2_RECURBY_LIST_LENGTH(1));
        recurBy->allocated = 1;

        result = TRUE;

        ptr = line;
        *ptr = '\0';

        do {
            number = ++ptr;

            if (*ptr != '-') {
                sign = 1;

                if (*ptr == '+') {
                    ptr++;
                }
            } else {
                sign = -1;
                ptr++;
            }

            while (isdigit(*ptr)) {
                ptr++;
            }

            delim = *ptr;
            *ptr = '\0';

            count = atoi(number);
            if ((count >= minimum) && (count <= maximum)) {
                if (recurBy->count < recurBy->allocated) {
                    value = &(recurBy->list[recurBy->count++]);
                } else {
                    recurBy->allocated += 4;

                    recurByNew = (ICal2RecurByList *)MemRealloc(recurBy, ICAL2_RECURBY_LIST_LENGTH(recurBy->allocated));
                    if (recurByNew) {
                        recurBy = recurByNew;

                        value = &(recurBy->list[recurBy->count++]);
                    } else {
                        result = FALSE;
                    }
                }
            } else {
                result = FALSE;
            }

            if (result) {
                value->weekday = ICAL2_RECUR_WEEKDAY_NONE;
                value->number = count;
                value->string = NULL;
            }

            *ptr = delim;
        } while (result && (*ptr == ','));

        if (result) {
            *recurByList = recurBy;
            return(ptr);
        }

        MemFree(recurBy);
    }

    return(NULL);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYSECOND" "=" byseclist ) /

    byseclist
        seconds / ( seconds *("," seconds) )

    seconds
        1DIGIT / 2DIGIT ;0 to 59
*/
BOOL 
ICal2RecurBySecondParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 8, 0, 59, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->bySeconds) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->bySeconds = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYMINUTE" "=" byminlist ) /

    byminlist
        minutes / ( minutes *("," minutes) )

    minutes
        1DIGIT / 2DIGIT ;0 to 59
*/
BOOL 
ICal2RecurByMinuteParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 8, 0, 59, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byMinutes) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byMinutes = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYHOUR" "=" byhrlist ) /

    byhrlist
        hour / ( hour *("," hour) )

    hour
        1DIGIT / 2DIGIT ;0 to 23
*/
BOOL 
ICal2RecurByHourParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 6, 0, 23, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byHours) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byHours = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYDAY" "=" bywdaylist ) /

    bywdaylist
        weekdaynum / ( weekdaynum *("," weekdaynum) )

    weekdaynum
        [([plus] ordwk / minus ordwk)] weekday

    plus
        "+"

    minus
        "-"

    ordwk
        1DIGIT / 2DIGIT ;1 to 53

    weekday
        "SU" / "MO" / "TU" / "WE" / "TH" / "FR" / "SA"
        ;Corresponding to SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY,
        ;FRIDAY, SATURDAY and SUNDAY days of the week.
*/
BOOL 
ICal2RecurByDayParser(ICal2Object *ical)
{
    int sign;
    int count;
    unsigned char *ptr;
    unsigned char *cur;
    BOOL result = TRUE;
    ICal2RecurByListValue *value;
    ICal2RecurByList *recurBy;
    ICal2RecurByList *recurByNew;
    ICal2RecurValue *recur;

    recurBy = (ICal2RecurByList *)MemMalloc(ICAL2_RECURBY_LIST_LENGTH(1));
    if (recurBy) {
        memset(recurBy, 0, ICAL2_RECURBY_LIST_LENGTH(1));
        recurBy->allocated = 1;
        recurBy->name = ptr = ical->parse.value;

        recur = (ICal2RecurValue *)ical->parse.detail;

        ptr += 5;
        *ptr = '\0';

        do {
            ptr++;

            sign = 1;

            if (recurBy->count < recurBy->allocated) {
                value = &(recurBy->list[recurBy->count++]);
            } else {
                recurBy->allocated += 4;

                recurByNew = (ICal2RecurByList *)MemRealloc(recurBy, ICAL2_RECURBY_LIST_LENGTH(recurBy->allocated));
                if (recurByNew) {
                    recurBy = recurByNew;

                    value = &(recurBy->list[recurBy->count++]);
                } else {
                    result = FALSE;
                }
            }

            if (result) {
                value->string = NULL;

                if (*ptr != '-') {
                    sign = 1;

                    if (*ptr == '+') {
                        ptr++;
                    }
                } else {
                    sign = -1;
                    ptr++;
                }

                cur = ptr;
                while (isdigit(*ptr)) {
                    ptr++;
                }

                if (cur != ptr) {
                    count = atoi(cur);
                    if ((count > 0) && (count < 54)) {
                        value->number = count * sign;
                    } else {
                        result = FALSE;
                    }
                } else {
                    value->number = 0;
                }
            }

            if (result && ((ptr[2] == ',') || (ptr[2] == ';') || (ptr[2] == '\0'))) {
                switch (toupper(*ptr)) {
                    case 'S': {
                        if (toupper(ptr[1]) == 'A') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_SATURDAY;
                            break;
                        }

                        if (toupper(ptr[1]) == 'U') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_SUNDAY;
                            break;
                        }

                        result = FALSE;
                        break;
                    }

                    case 'M': {
                        if (toupper(ptr[1]) == 'O') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_MONDAY;
                            break;
                        }

                        result = FALSE;
                        break;
                    }

                    case 'T': {
                        if (toupper(ptr[1]) == 'H') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_THURSDAY;
                            break;
                        }

                        if (toupper(ptr[1]) == 'U') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_TUESDAY;
                            break;
                        }

                        result = FALSE;
                        break;
                    }

                    case 'W': {
                        if (toupper(ptr[1]) == 'E') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_WEDNESDAY;
                            break;
                        }

                        result = FALSE;
                        break;
                    }

                    case 'F': {
                        if (toupper(ptr[1]) == 'R') {
                            value->weekday = ICAL2_RECUR_WEEKDAY_FRIDAY;
                            break;
                        }

                        result = FALSE;
                        break;
                    }

                    default: {
                        result = FALSE;
                        break;
                    }
                }
            }

            if (result) {
                ptr += 2;
            }
        } while (result && (*ptr == ','));

        if (result) {
            recur->byDays = recurBy;

            ical->parse.value = ptr;
        }
    }

    return(result);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYMONTHDAY" "=" bymodaylist ) /

    bymodaylist
        monthdaynum / ( monthdaynum *("," monthdaynum) )

    monthdaynum
        ([plus] ordmoday) / (minus ordmoday)

    plus
        "+"

    minus
        "-"

    ordmoday
        1DIGIT / 2DIGIT ;1 to 31
*/
BOOL 
ICal2RecurByMonthDayParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 10, 1, 31, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byMonthDays) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byMonthDays = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYYEARDAY" "=" byyrdaylist ) /

    byyrdaylist
        yeardaynum / ( yeardaynum *("," yeardaynum) )

    yeardaynum
        ([plus] ordyrday) / (minus ordyrday)

    ordyrday
        1DIGIT / 2DIGIT / 3DIGIT    ;1 to 366

    plus
        "+"

    minus
        "-"
*/
BOOL 
ICal2RecurByYearDayParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 9, 1, 366, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byYearDays) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byYearDays = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYWEEKNO" "=" bywknolist ) /

    bywknolist
        weeknum / ( weeknum *("," weeknum) )

    weeknum
        ([plus] ordwk) / (minus ordwk)

    plus
        "+"

    minus
        "-"

    ordwk
        1DIGIT / 2DIGIT ;1 to 53
*/
BOOL 
ICal2RecurByWeekNoParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 8, 1, 53, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byWeekNumbers) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byWeekNumbers = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYMONTH" "=" bymolist ) /

    bymolist
        monthnum / ( monthnum *("," monthnum) )

    monthnum
        1DIGIT / 2DIGIT ;1 to 12
*/
BOOL 
ICal2RecurByMonthParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 7, 1, 12, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->byMonths) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->byMonths = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" "BYSETPOS" "=" bysplist ) /

    bysplist
        setposday / ( setposday *("," setposday) )

    setposday
        yeardaynum

    yeardaynum
        ([plus] ordyrday) / (minus ordyrday)

    ordyrday
        1DIGIT / 2DIGIT / 3DIGIT    ;1 to 366

    plus
        "+"

    minus
        "-"
*/
BOOL 
ICal2RecurBySetPosParser(ICal2Object *ical)
{
    unsigned char *ptr = ical->parse.value;
    ICal2RecurByList *recurBy = NULL;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ICal2RecurByListNumbers(recur, ptr + 8, 1, 366, &recurBy);
    if (ptr && recurBy) {
        recurBy->name = ical->parse.value;

        recurBy->prev = NULL;
        if ((recurBy->next = recur->bySets) != NULL) {
            recurBy->next->prev = recurBy;
        }
        recur->bySets = recurBy;

        ical->parse.value = ptr;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ( ";" "WKST" "=" weekday ) /

    weekday
        "SU" / "MO" / "TU" / "WE" / "TH" / "FR" / "SA"
        ;Corresponding to SUNDAY, MONDAY, TUESDAY, WEDNESDAY, THURSDAY,
        ;FRIDAY, SATURDAY and SUNDAY days of the week.
*/
BOOL 
ICal2RecurWkstParser(ICal2Object *ical)
{
    unsigned char *ptr;
    BOOL result = FALSE;
    ICal2RecurValue *recur;

    recur = (ICal2RecurValue *)ical->parse.detail;

    ptr = ical->parse.value + 5;
    if ((ptr[2] == ';') || (ptr[2] == '\0')) {
        switch (toupper(*ptr)) {
            case 'S': {
                if (toupper(ptr[1]) == 'A') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_SATURDAY;
                    result = TRUE;
                } else if (toupper(ptr[1]) == 'U') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_SUNDAY;
                    result = TRUE;
                }

                break;
            }

            case 'M': {
                if (toupper(ptr[1]) == 'O') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_MONDAY;
                    result = TRUE;
                }

                break;
            }

            case 'T': {
                if (toupper(ptr[1]) == 'H') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_THURSDAY;
                    result = TRUE;
                } else if (toupper(ptr[1]) == 'U') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_TUESDAY;
                    result = TRUE;
                }

                break;
            }

            case 'W': {
                if (toupper(ptr[1]) == 'E') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_WEDNESDAY;
                    result = TRUE;
                }

                break;
            }

            case 'F': {
                if (toupper(ptr[1]) == 'R') {
                    recur->weekday = ICAL2_RECUR_WEEKDAY_FRIDAY;
                    result = TRUE;
                }

                break;
            }

            default: {
                break;
            }
        }

        if (result) {
            ptr += 2;
        }
    }

    return(result);
}

/*
    RFC 2445 Section 4.3.10 Recurrence Rule

    ; the rest of these keywords are optional,
    ; but MUST NOT occur more than once
    ( ";" x-name "=" text )
*/
BOOL 
ICal2RecurXNameParser(ICal2Object *ical)
{
    unsigned char *ptr;
    BOOL result = TRUE;
    ICal2RecurByListValue *value;
    ICal2RecurByList *recurBy;
    ICal2RecurByList *recurByNew;
    ICal2RecurValue *recur;

    recurBy = (ICal2RecurByList *)MemMalloc(ICAL2_RECURBY_LIST_LENGTH(1));
    if (recurBy) {
        memset(recurBy, 0, ICAL2_RECURBY_LIST_LENGTH(1));
        recurBy->allocated = 1;
        recurBy->name = ptr = ical->parse.value;

        recur = (ICal2RecurValue *)ical->parse.detail;

        while (ICal2IsXName(*ptr)) {
            *ptr++;
        }

        if (*ptr == '=') {
            *ptr++ = '\0';
        } else {
            result = FALSE;
        }

        while (result) {
            if (recurBy->count < recurBy->allocated) {
                value = &(recurBy->list[recurBy->count++]);
            } else {
                recurBy->allocated += 4;

                recurByNew = (ICal2RecurByList *)MemRealloc(recurBy, ICAL2_RECURBY_LIST_LENGTH(recurBy->allocated));
                if (recurByNew) {
                    recurBy = recurByNew;

                    value = &(recurBy->list[recurBy->count++]);
                } else {
                    result = FALSE;
                }
            }

            if (result) {
                value->number = 0;
                value->weekday = ICAL2_RECUR_WEEKDAY_NONE;
                value->string = ptr;

                ptr = ICal2TextParser(ptr);
                if ((*ptr == ';') || (*ptr == '\0')) {
                    recur->xNames = recurBy;

                    ical->parse.value = ptr;
                    return(TRUE);
                }

                if (*ptr == ',') {
                    *ptr++ = '\0';
                    continue;
                }
            }
        }
    }

    return(FALSE);
}

unsigned char *
ICal2RecurParser(ICal2Object *ical)
{
    unsigned char *ptr;
    BOOL result = TRUE;
    unsigned long index;

    while (result && ical->parse.value) {
        ptr = ical->parse.value;

        index = HulaKeywordFind(ICal2RecurIndex, ptr);
        if (index != -1) {
            ical->parse.token = &ICal2RecurTokens[index];
            if (ptr[ical->parse.token->nameLen] == '=') {
                result = ical->parse.token->parser.value(ical);
            } else {
                result = ICal2RecurXNameParser(ical);
            }
        } else {
            result = ICal2RecurXNameParser(ical);
        }

        if (result) {
            if (ical->parse.value) {
                ptr = ical->parse.value;
                if (*ptr == ';') {
                    *ptr++ = '\0';
                    ical->parse.value = ptr;
                    continue;
                }

                if ((*ptr == '\0') || (*ptr == ',')) {
                    return(ptr);
                }
            }
        }
    }

    return(NULL);
}

/*
    RFC 2445 Section 4.3.1 Binary

    binary
        *(4b-char) [b-end]  ;A "BASE64" encoded character string, as defined by [RFC 2045].

    b-end
        (2b-char "==") / (3b-char "=")

    b-char
        ALPHA / DIGIT / "+" / "/"
*/
BOOL 
ICal2BinaryValueParser(ICal2Object *ical)
{
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_BINARY;
        value->items.count = 1;
        value->items.detail[0].binary.encoded = ical->parse.value;

        ical->parse.value = NULL;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.2 Boolean

    boolean
        "TRUE" / "FALSE"
*/
BOOL 
ICal2BooleanValueParser(ICal2Object *ical)
{
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_BOOLEAN;
        value->items.count = 1;
        value->items.detail[0].boolean.value = ical->parse.value;

        ical->parse.value = NULL;

        if (XplStrCaseCmp(value->items.detail[0].boolean.value, "TRUE") == 0) {
            value->items.detail[0].boolean.yes = TRUE;
            return(TRUE);
        } else if (XplStrCaseCmp(value->items.detail[0].boolean.value, "FALSE") == 0) {
            value->items.detail[0].boolean.yes = FALSE;
            return(TRUE);
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.3 Calendar User Address

    cal-address
        uri
*/
BOOL 
ICal2CalAddressValueParser(ICal2Object *ical)
{
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_CAL_ADDRESS;
        value->items.count = 1;
        value->items.detail[0].calAddress.uri = ical->parse.value;

        ical->parse.value = NULL;

        return(TRUE);
    }

    return(FALSE);
}

BOOL 
ICal2DateValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_DATE;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->date.value = ptr;

            ptr = ICal2DateParser(&(detail->date));
            if (ptr && (*ptr == '\0')) {
                ical->parse.value = NULL;
                return(TRUE);
            }

            if (ptr && (*ptr == ',')) {
                *ptr++ = '\0';
                continue;
            }
        }

        break;
    } while (TRUE);

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.5 Date-Time

    date-time
        date "T" time
*/
BOOL 
ICal2DateTimeValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_DATE_TIME;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->dateTime.value = detail->dateTime.date.value = ptr;
            ptr = ICal2DateParser(&(detail->dateTime.date));
            if (ptr && (toupper(*ptr) == 'T')) {
                *ptr++ = '\0';
                detail->dateTime.time.value = ptr;
            } else {
                break;
            }

            ptr = ICal2TimeParser(&(detail->dateTime.time));
            if (ptr && (*ptr == '\0')) {
                ical->parse.value = NULL;
                return(TRUE);
            }

            if (ptr && (*ptr == ',')) {
                ptr++;
                continue;
            }
        }

        break;
    } while (TRUE);

    return(FALSE);
}

BOOL 
ICal2DurationValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_DURATION;
        value->items.count = 1;

        detail = &(value->items.detail[0]);
    } else {
        return(FALSE);
    }

    memset(&(detail->duration), 0, sizeof(ICal2DurationValue));

    detail->duration.value = ical->parse.value;
    ptr = ICal2DurationParser(&(detail->duration));
    if (ptr && (*ptr == '\0')) {
        ical->parse.value = NULL;
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.7 Float

    float
        (["+"] / "-") 1*DIGIT ["." 1*DIGIT]
*/
BOOL 
ICal2FloatValueParser(ICal2Object *ical)
{
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_FLOAT;
        value->items.count = 1;

        detail = &(value->items.detail[0]);
        detail->f.value = ical->parse.value;
        detail->f.f = atof(ical->parse.value);

        ical->parse.value = NULL;
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.8 Integer

    integer
        (["+"] / "-") 1*DIGIT
*/
BOOL 
ICal2IntegerValueParser(ICal2Object *ical)
{
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_INTEGER;
        value->items.count = 1;

        detail = &(value->items.detail[0]);
        detail->i.value = ical->parse.value;
        detail->i.i = atoi(ical->parse.value);

        ical->parse.value = NULL;
        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.9 Period

    period
        period-explicit / period-start

    period-explicit
        date-time "/" date-time ; [ISO 8601] complete representation basic format for a period of
                                ; time consisting of a start and end. The start MUST be before the
                                ; end.

    period-start
        date-time "/" dur-value ; [ISO 8601] complete representation basic format for a period of
                                ; time consisting of a start and positive duration of time.
*/
BOOL 
ICal2PeriodValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_PERIOD;
        value->items.count = 1;

        detail = &(value->items.detail[0]);
        detail->period.value = ical->parse.value;
    } else {
        return(FALSE);
    }

    detail->period.start.value = detail->period.start.date.value = detail->period.value;
    ptr = ICal2DateParser(&(detail->period.start.date));
    if (ptr && (toupper(*ptr) == 'T')) {
        *ptr++ = '\0';
        detail->period.start.time.value = ptr;
        ptr = ICal2TimeParser(&(detail->period.start.time));
        if (ptr && (*ptr == '/')) {
            *ptr++ = '\0';
            if ((*ptr != '-') && (*ptr != '+') && (toupper(*ptr) != 'P')) {
                detail->period.duration = FALSE;

                detail->period.u.stop.value = detail->period.u.stop.date.value = ptr;
                ptr = ICal2DateParser(&(detail->period.u.stop.date));
                if (ptr && (toupper(*ptr) == 'T')) {
                    *ptr++ = '\0';
                    detail->period.u.stop.time.value = ptr;
                    ptr = ICal2TimeParser(&(detail->period.u.stop.time));
                } else {
                    ptr = NULL;
                }
            } else {
                detail->period.duration = TRUE;
                detail->period.u.duration.value = ptr;
                ptr = ICal2DurationParser(&(detail->period.u.duration));
            }

            if (ptr && (*ptr == '\0')) {
                ical->parse.value = NULL;
                return(TRUE);
            }
        }
    }

    return(FALSE);
}

BOOL 
ICal2RecurValueParser(ICal2Object *ical)
{
    unsigned char *src;
    BOOL result = FALSE;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;
    ICal2ObjectParsing parse;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_RECUR;
    } else {
        return(FALSE);
    }

    memcpy(&parse, &(ical->parse), sizeof(ICal2ObjectParsing));

    src = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->recur.value = src;
            detail->recur.untilType = ICAL2_VALUE_TYPE_NONE;
            detail->recur.weekday = ICAL2_RECUR_WEEKDAY_NONE;

            ical->parse.detail = (void *)&(detail->recur);
            src = ICal2RecurParser(ical);
            if (src) {
                if (*src == '\0') {
                    ical->parse.value = NULL;
                    result = TRUE;
                    break;
                }

                if (*src == ',') {
                    *src++ = '\0';
                    continue;
                }
            }
        }

        break;
    } while (*src);

    memcpy(&(ical->parse), &parse, sizeof(ICal2ObjectParsing));

    return(result);
}

BOOL 
ICal2TextValueParser(ICal2Object *ical)
{
    unsigned char *src;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_TEXT;
    } else {
        return(FALSE);
    }

    src = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->text.value = src;

            src = ICal2TextParser(src);
            if (*src == '\0') {
                ical->parse.value = NULL;
                return(TRUE);
            }

            if (*src == ',') {
                *src++ = '\0';
                continue;
            }
        }

        break;
    } while (*src);

    return(FALSE);
}

BOOL 
ICal2TimeValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_TIME;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->time.value = ptr;

            ptr = ICal2TimeParser(&(detail->time));
            if (ptr && (*ptr == '\0')) {
                ical->parse.value = NULL;
                return(TRUE);
            }

            if (ptr && (*ptr == ',')) {
                *ptr++ = '\0';
                continue;
            }
        }

        break;
    } while (TRUE);

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.13 URI

    uri
        <As defined by any IETF RFC>
*/
BOOL 
ICal2UriValueParser(ICal2Object *ical)
{
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_URI;
        value->items.count = 1;
        value->items.detail[0].uri.value = ical->parse.value;

        ical->parse.value = NULL;

        return(TRUE);
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.3.14 UTC Offset

    utc-offset
        ;As defined above in time data type
        time-numzone

    time-numzone
        ("+" / "-") time-hour time-minute [time-second]
*/
BOOL 
ICal2UtcOffsetValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_UTC_OFFSET;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;
    do {
        ICAL2_PULL_VALUE_DETAIL(property, detail);
        if (detail) {
            detail->utcOffset.value = ptr;

            if ((*ptr == '+') || (*ptr == '-')) {
                ptr++;
                if (isdigit(ptr[0]) && isdigit(ptr[1])) {
                    detail->utcOffset.hours = (*ptr++ - '0') * 10;
                    detail->utcOffset.hours += (*ptr++ - '0');

                    if (isdigit(ptr[0]) && isdigit(ptr[1])) {
                        detail->utcOffset.minutes = (*ptr++ - '0') * 10;
                        detail->utcOffset.minutes += (*ptr++ - '0');
                    }

                    if (isdigit(ptr[0]) && isdigit(ptr[1])) {
                        detail->utcOffset.seconds = (*ptr++ - '0') * 10;
                        detail->utcOffset.seconds += (*ptr++ - '0');
                    }

                    detail->utcOffset.offset = (detail->utcOffset.hours * 60 * 60) 
                                                + (detail->utcOffset.minutes * 60) 
                                                + detail->utcOffset.seconds;

                    if (detail->utcOffset.value[0] == '-') {
                        detail->utcOffset.offset *= -1;
                    }

                    if (*ptr == '\0') {
                        ical->parse.value = NULL;
                        return(TRUE);
                    }

                    if (*ptr == ',') {
                        *ptr++ = '\0';
                        ical->parse.value = ptr;
                        continue;
                    }
                }
            }
        }

        break;
    } while (TRUE);

    return(FALSE);
}

/*
    RFC 2445 Section 4.1 Content Lines

    value
        *VALUE-CHAR

    VALUE-CHAR
        WSP / %x21-7E / NON-US-ASCII
*/
BOOL 
ICal2ValueParser(ICal2Object *ical)
{
    unsigned char *src;
    unsigned char *dest;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    property = ical->parse.property;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_IANA;
    } else {
        return(FALSE);
    }

    src = dest = ical->parse.value;
    ICAL2_PULL_VALUE_DETAIL(property, detail);
    if (detail) {
        detail->text.value = src;

        while (ICal2IsValue(*src)) {
            *dest++ = *src++;
        }

        if (*src == '\0') {
            *dest++ = '\0';
            ical->parse.value = NULL;

            return(TRUE);
        }
    }

    return(FALSE);
}

BOOL 
ICal2AttachValueParser(ICal2Object *ical)
{
    unsigned long index;
    ICal2ParameterDetail *detail;
    ICal2Parameter *parameter;

    if ((parameter = ical->parse.property->parameters.head) != NULL) {
        do {
            if (parameter->type == ICAL2_PARAMETER_VALUE) {
                break;
            }

            parameter = parameter->next;
        } while (parameter);

        if (parameter) {
            detail = parameter->items.detail;
            for (index = 0; index < parameter->items.count; index++, detail++) {
                if (detail->value.type == ICAL2_VALUE_TYPE_BINARY) {
                    return(ICal2BinaryValueParser(ical));
                }

                if (detail->value.type == ICAL2_VALUE_TYPE_URI) {
                    break;
                }
            }
        }
    }

    return(ICal2UriValueParser(ical));
}

BOOL 
ICal2GeoValueParser(ICal2Object *ical)
{
    unsigned char *ptr = strchr(ical->parse.value, ';');
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;

    if (ptr) {
        *ptr++ = '\0';

        property = ical->parse.property;

        value = (ICal2Value *)MemMalloc(sizeof(ICal2Value) + (2 * sizeof(ICal2ValueDetail)));
        if (value) {
            memset(value, 0, sizeof(ICal2Value));

            property->value = value;

            value->type = ICAL2_VALUE_TYPE_FLOAT;
            value->items.allocated = 2;
            value->items.count = 2;

            detail = &(value->items.detail[0]);
            detail->f.value = ical->parse.value;
            detail->f.f = atof(ical->parse.value);

            detail++;
            detail->f.value = ptr;
            detail->f.f = atof(ptr);

            ical->parse.value = NULL;
            return(TRUE);
        }
    }

    return(FALSE);
}

BOOL 
ICal2DateDateTimeValueParser(ICal2Object *ical)
{
    unsigned long index;
    ICal2ParameterDetail *detail;
    ICal2Parameter *parameter;

    if ((parameter = ical->parse.property->parameters.head) != NULL) {
        do {
            if (parameter->type == ICAL2_PARAMETER_VALUE) {
                break;
            }

            parameter = parameter->next;
        } while (parameter);

        if (parameter) {
            detail = parameter->items.detail;
            for (index = 0; index < parameter->items.count; index++, detail++) {
                if (detail->value.type == ICAL2_VALUE_TYPE_DATE) {
                    return(ICal2DateValueParser(ical));
                }

                if (detail->value.type == ICAL2_VALUE_TYPE_DATE_TIME) {
                    break;
                }
            }
        }
    }

    return(ICal2DateTimeValueParser(ical));
}

/*
    RFC 2445 Section 4.8.6.3 Trigger

    trigger
        "TRIGGER" (trigrel / trigabs)

    trigrel
        *(
            ; the following are optional,
            ; but MUST NOT occur more than once
            (";" "VALUE" "=" "DURATION") /
            (";" trigrelparam) /

            ; the following is optional,
            ; and MAY occur more than once

            (";" xparam)
        ) ":"  dur-value

    trigabs
        1*(
            ; the following is REQUIRED,
            ; but MUST NOT occur more than once
            (";" "VALUE" "=" "DATE-TIME") /

            ; the following is optional,
            ; and MAY occur more than once

            (";" xparam)
        ) ":" date-time


*/
BOOL 
ICal2TriggerValueParser(ICal2Object *ical)
{
    unsigned long index;
    ICal2ParameterDetail *detail;
    ICal2Parameter *parameter;

    if ((parameter = ical->parse.property->parameters.head) != NULL) {
        do {
            if (parameter->type == ICAL2_PARAMETER_VALUE) {
                break;
            }

            parameter = parameter->next;
        } while (parameter);

        if (parameter) {
            detail = parameter->items.detail;
            for (index = 0; index < parameter->items.count; index++, detail++) {
                if (detail->value.type == ICAL2_VALUE_TYPE_DATE_TIME) {
                    return(ICal2DateTimeValueParser(ical));
                }

                if (detail->value.type == ICAL2_VALUE_TYPE_DURATION) {
                    break;
                }
            }
        }
    }

    return(ICal2DurationValueParser(ical));
}

/*
    RFC 2445 Section 4.8.1.11 Status

    statvalue
        ;Status values for a "VEVENT"
        "TENTATIVE" /       ;Indicates event is tentative.
        "CONFIRMED" /       ;Indicates event is definite.
        "CANCELLED"         ;Indicates event was cancelled.

        ;Status values for "VTODO".
        "NEEDS-ACTION" /    ;Indicates to-do needs action.
        "COMPLETED" /       ;Indicates to-do completed.
        "IN-PROCESS" /      ;Indicates to-do in process of
        "CANCELLED"         ;Indicates to-do was cancelled.

        ;Status values for "VJOURNAL".
        "DRAFT" /           ;Indicates journal is draft.
        "FINAL" /           ;Indicates journal is final.
        "CANCELLED"         ;Indicates journal is removed.
*/
BOOL 
ICal2StatusValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;
    ICal2Component *component;

    property = ical->parse.property;
    component = ical->parse.component;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_STATUS;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;

    ICAL2_PULL_VALUE_DETAIL(property, detail);
    if (detail && isalpha(*ptr)) {
        detail->status.value = ptr;

        switch (component->type) {
            case ICAL2_COMPONENT_VEVENT: {
                switch (toupper(ptr[1])) {
                    case 'A': {
                        if (XplStrCaseCmp(ptr, "CANCELLED") == 0) {
                            detail->status.type = ICAL2_STATUS_CANCELLED;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'E': {
                        if (XplStrCaseCmp(ptr, "TENTATIVE") == 0) {
                            detail->status.type = ICAL2_STATUS_TENTATIVE;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'O': {
                        if (XplStrCaseCmp(ptr, "CONFIRMED") == 0) {
                            detail->status.type = ICAL2_STATUS_CONFIRMED;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case ICAL2_COMPONENT_VTODO: {
                switch (toupper(ptr[1])) {
                    case 'A': {
                        if (XplStrCaseCmp(ptr, "CANCELLED") == 0) {
                            detail->status.type = ICAL2_STATUS_CANCELLED;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'E': {
                        if (XplStrCaseCmp(ptr, "NEEDS-ACTION") == 0) {
                            detail->status.type = ICAL2_STATUS_NEEDS_ACTION;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'N': {
                        if (XplStrCaseCmp(ptr, "IN-PROCESS") == 0) {
                            detail->status.type = ICAL2_STATUS_IN_PROCESS;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'O': {
                        if (XplStrCaseCmp(ptr, "COMPLETED") == 0) {
                            detail->status.type = ICAL2_STATUS_COMPLETED;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case ICAL2_COMPONENT_VJOURNAL: {
                switch (toupper(ptr[1])) {
                    case 'A': {
                        if (XplStrCaseCmp(ptr, "CANCELLED") == 0) {
                            detail->status.type = ICAL2_STATUS_CANCELLED;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'I': {
                        if (XplStrCaseCmp(ptr, "FINAL") == 0) {
                            detail->status.type = ICAL2_STATUS_FINAL;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    case 'R': {
                        if (XplStrCaseCmp(ptr, "DRAFT") == 0) {
                            detail->status.type = ICAL2_STATUS_DRAFT;
                            ical->parse.value = NULL;
                            return(TRUE);
                        }

                        break;
                    }

                    default: {
                        break;
                    }
                }

                break;
            }

            case ICAL2_COMPONENT_VCALENDAR:
            case ICAL2_COMPONENT_VALARM:
            case ICAL2_COMPONENT_VFREEBUSY:
            case ICAL2_COMPONENT_VTIMEZONE:
            case ICAL2_COMPONENT_X:
            case ICAL2_COMPONENT_IANA:
            case ICAL2_COMPONENT_DAYLIGHT:
            case ICAL2_COMPONENT_STANDARD:
            default: {
                break;
            }
        }
    }

    return(FALSE);
}

/*
    RFC 2445 Section 4.8.8.2 Request Status

    statcode ";" statdesc [";" extdata]

    statcode
        ;Hierarchical, numeric return status code
        1*DIGIT *("." 1*DIGIT)

    statdesc
        ;Textual status description
        text

    extdatax
        ;Textual exception data. For example, the offending property
        ;name and value or complete property line.
        text
*/
BOOL 
ICal2RequestStatusValueParser(ICal2Object *ical)
{
    unsigned char *ptr;
    ICal2ValueDetail *detail;
    ICal2Value *value;
    ICal2Property *property;
    ICal2Component *component;

    property = ical->parse.property;
    component = ical->parse.component;

    value = (ICal2Value *)MemMalloc(sizeof(ICal2Value));
    if (value) {
        memset(value, 0, sizeof(ICal2Value));

        property->value = value;

        value->type = ICAL2_VALUE_TYPE_REQUEST_STATUS;
    } else {
        return(FALSE);
    }

    ptr = ical->parse.value;

    ICAL2_PULL_VALUE_DETAIL(property, detail);
    if (detail && isdigit(*ptr)) {
        detail->requestStatus.code = ptr;
        while (isdigit(*ptr) || (*ptr == '.')) {
            ptr++;
        }

        if (*ptr == ';') {
            *ptr++ = '\0';

            detail->requestStatus.description = ptr;

            ptr = ICal2TextParser(ptr);
            if (*ptr == ';') {
                *ptr++ = '\0';

                detail->requestStatus.exception = ptr;

                ptr = ICal2TextParser(ptr);
            }

            ical->parse.value = NULL;
            return(TRUE);
        }
    }

    return(FALSE);
}
