/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <xpl.h>

#include <mdb.h>
#include <msgapi.h>
#include <nmap.h>
#include <hulautil.h>

#include <modweb.h>		/* APIs */
#include <mwtempl.h>		/* Token definition */

#include "mwcal.h"

long
NMAPtoCalendarID(SessionStruct *Session, CalendarSessionStruct *CalendarSession, unsigned long NMAP)
{
	register unsigned long	Start;
	register unsigned long	End;
	register unsigned long	Middle;

	if (Session->NumOfEvents==0) {
		return(-1);
	}

	Start=0;
	End=Session->NumOfEvents;
	Middle=End/2;

	do {
		if (CalendarSession->IDList[Middle]>NMAP) {
			End=Middle;
		} else {
			Start=Middle;
		}
		Middle=Start+(End-Start)/2;
	} while ((End-Start)>1);

	if (NMAP==CalendarSession->IDList[Start]) {
		return(Start+1);
	}

	return(-1);
}

static BOOL
LoadCalendarIDList(SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned long	NMAPCount;
	unsigned long	State;
	unsigned long	ID;
	int				ReplyInt;
	unsigned char	Reply[1024];

	MWSendNMAPServer(Session, "CSINFO\r\n", 8);
	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=2002) {
		return(FALSE);
	}

	NMAPCount=atol(Reply);

	if (CalendarSession->IDList!=NULL) {
		MemFree(CalendarSession->IDList);
	}

	Session->NumOfEvents=0;
	CalendarSession->IDList=MemMalloc(sizeof(unsigned long)*(NMAPCount+1));
	if (!CalendarSession->IDList) {
		return(FALSE);
	}

	for (ID=1; ID<=NMAPCount; ID++) {
		MWGetNMAPAnswer(Session, Reply, sizeof(Reply), FALSE);
		sscanf(Reply, "%*u %*u %*u %*u %lu", &State);
		if (!(State & MSG_STATE_PURGED)) {
			CalendarSession->IDList[Session->NumOfEvents]=ID;
			Session->NumOfEvents++;
		}
	}

	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=1000) {
		Session->NumOfEvents=0;
		if (CalendarSession->IDList!=NULL) {
			MemFree(CalendarSession->IDList);
			CalendarSession->IDList=NULL;
		}
		return(FALSE);
	}
	return(TRUE);
}

BOOL
SelectCalendar(unsigned long CalendarID, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;
	unsigned char	*folder;

	if (CalendarID == 0) {
		/*
			Special case

			We know that 0 will always be INBOX, so we are going to select MAIN instead.
		*/

		do {
			CalendarID++;
		} while (CalendarID < Session->FolderList->Used && !MWQuickCmp(Session->FolderList->Value[CalendarID] + 3, "MAIN"));
	}

	if (CalendarID < Session->FolderList->Used && (!CalendarSession->CalendarSelected || CalendarSession->CurrentCalendar != CalendarID)) {
		folder = Session->FolderList->Value[CalendarID];

		if (*folder == 'C') {
			/* Invalidate all cache data & folder data */
			MWSendNMAPServer(Session, "RSET CAL\r\n", 10);
			MWGetNMAPAnswer(Session, Answer, BUFSIZE, TRUE);

			/* Need to invalidate any cache entry */
			CalendarSession->CalendarSelected = FALSE;
			FreeCalendarView(Session, CalendarSession);
			FreeCalendarEntryDetails(Session, CalendarSession);

			/* We only set the times to 0, keep the memory around for re-use */
			CalendarSession->EntryID = 0;
			CalendarSession->DetailID = 0;

			/* Invalidate any cached data */
			CalendarSession->ViewUTCStart = 0;
			CalendarSession->ViewUTCEnd = 0;

			ReplyInt = snprintf(Answer, sizeof(Answer), "CSOPEN %s\r\n", folder + 3);
			MWSendNMAPServer(Session, Answer, ReplyInt);
			ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
			if ((ReplyInt!=1000) && (ReplyInt!=1020)) {
				CalendarSession->Error = 1 /* FIX-ME ERROR_SELECTING_FOLDER__MBOX */;
				ModDebug("SelectCalendar(): NMAP error %s\n", Answer);
				Session->NumOfEvents=0;
				CalendarSession->CalendarSelected=FALSE;
				return(FALSE);
			} else {
				/* We've got the calendar selected, get the stats */
				MWSendNMAPServer(Session, "CSSTAT\r\n", 8);
				ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
				if (ReplyInt!=1000) {
					/* Get back into a known state */
					ModDebug("SelectFolder(): NMAP error %s\n", Answer);
					MWSendNMAPServer(Session, "RSET CAL\r\n", 10);
					ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
					Session->NumOfEvents=0;
					CalendarSession->CalendarSelected=FALSE;
					return(FALSE);
				} else {
					LoadCalendarIDList(Session, CalendarSession);
					CalendarSession->CalendarSelected=TRUE;
					CalendarSession->CurrentCalendar=CalendarID;
				}
			}
		}
	}

	return(TRUE);
}


BOOL
LoadCalendarEntry(unsigned long CalendarID, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;
	unsigned char	*ptr;

	if (CalendarID==CalendarSession->EntryID) {
		return(TRUE);
	}
	CalendarSession->EntryRole = '\0';
	CalendarSession->IAmAttendee = FALSE;
	CalendarSession->EntryID=CalendarID;
	CalendarID--;

	ReplyInt=snprintf(Answer, sizeof(Answer), "CSINFO %lu\r\n", CalendarSession->IDList[CalendarID]);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	if ((ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE))==2001) {
		ptr=Answer+strlen(Answer)-1;

		/* This is the end of the summary string quote */
		if (ptr[0]!='"') {
			CalendarSession->EntryID=0;
			return(FALSE);
		}
		ptr[0]='\0';
		sscanf(Answer, "%*u %*u %*u %*u %lu", &CalendarSession->EntryState);

		/* Find the beginning of the summary string */
		while (*ptr!='"') {
			ptr--;
		}
		HulaStrNCpy(CalendarSession->EntrySummary, ptr+1, sizeof(CalendarSession->EntrySummary));

		/* Find the end of the organizer string quote */
		ptr--;
		while (*ptr!='"') {
			ptr--;
		}
		ptr[0]='\0';

		/* Find the beginning of the summary string */
		while (*ptr!='"') {
			ptr--;
		}
		HulaStrNCpy(CalendarSession->EntryOrganizer, ptr+1, sizeof(CalendarSession->EntryOrganizer));

		/* Update MY role and attendee information */
		ReplyInt=snprintf(Answer, sizeof(Answer), "CSATND %lu\r\n", CalendarSession->IDList[CalendarID]);
		MWSendNMAPServer(Session,Answer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);	/* List comming up */
		if (ReplyInt==2002) {
			while ((ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE)) == 2002) {
				unsigned char		*Address;

				/* Find the e-mail address */
				Address = strchr(Answer + 8, ' ');
				if (Address) {
					*Address = '\0';	/* We don't need anything after the address */
				}
				Address = Answer + 8;


				if (MWQuickCmp(Session->EMailAddress, Address)){
					CalendarSession->EntryRole = Answer[2];
					CalendarSession->EntryStatus = Answer[0];
					/* we can't bail because we need to drain connection */
				}
			}
		}
		return(TRUE);
	}
	CalendarSession->EntryID=0;
	return(FALSE);
}

BOOL
LoadCalendarEntryDetails(unsigned long CalendarID, SessionStruct *Session, CalendarSessionStruct *CalendarSession, BOOL ReLoad)
{
	unsigned char	*Cache = NULL;
	unsigned long	CacheSize;
	unsigned char	Answer[BUFSIZE+1];
	int				ReplyInt;
	unsigned long			Count;
	int				len;
	unsigned char	*ptr;

	/* Is it already cached? */
	if (!ReLoad && (CalendarID == CalendarSession->DetailID) && CalendarSession->CalendarSelected && 
		 (CalendarSession->CurrentCalendar == CalendarSession->DetailCalendar) && CalendarSession->ICal) {
		ModDebug("LoadCalendarEntry(): Event %d already cached\n", (int)CalendarID);
		return(TRUE);
	}

	if (CalendarSession->ICal) {
		ICalFreeObjects(CalendarSession->ICal);
		CalendarSession->ICal=NULL;
	}

	/* Ensure that CalenderSession->Entry is up to date  */
	if (!LoadCalendarEntry(CalendarID, Session, CalendarSession)){
		/* Initialize CalendarSession back to zero so it will retry */
		CalendarSession->DetailID = 0;
		return(FALSE);
	}
	CalendarSession->DetailID = CalendarID;
	CalendarSession->DetailCalendar = CalendarSession->CurrentCalendar;
	CalendarID--;		/* Our index is 0-based */
	if (CalendarID >= Session->NumOfEvents) {
		return(FALSE);
	}

	/* Request the object */
	ReplyInt=snprintf(Answer, sizeof(Answer), "CSLIST %lu\r\n", CalendarSession->IDList[CalendarID]);
	MWSendNMAPServer(Session, Answer, ReplyInt);
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);

	if (ReplyInt!=2023) {
		CalendarSession->ICal=NULL;
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);		
	}

	CacheSize=atol(Answer);
	Count = CacheSize;
	Cache=MemMalloc(sizeof(unsigned char)*(CacheSize+1));
	if (!Cache) {
		/* Still need to read all... :-( */
		if (Session->NBufferPtr > 0) {
			if (strlen(Session->NBuffer) > Count) {		/* Got the 1000 code in the buffer, too */
				Session->NBufferPtr=strlen(Session->NBuffer + Count);
				memmove(Session->NBuffer, Session->NBuffer + Count, Session->NBufferPtr+1);
				Count = 0;
			} else if (Count >= (unsigned long)Session->NBufferPtr) {
				Count -= Session->NBufferPtr;
				Session->NBufferPtr = 0;
				Session->NBuffer[0] = '\0';
			} else {				
				CalendarSession->ICal = NULL;
				ModDebug("NMAP connection out of sync\n");
				return(FALSE);
			}
		}
		while (Count > 0) {
			if (Count < sizeof(Answer)) {
				len=MWDirectNMAPRead(Session, Answer, Count);
			} else {
				len=MWDirectNMAPRead(Session, Answer, sizeof(Answer));
			}

			if ((len > 0) && (Count >= (unsigned long)len)) {
    			  Count -= len;
			} else {
				CalendarSession->ICal=NULL;
				ModDebug("Lost NMAP connection\n");
				return(FALSE);
			}
		}
		CalendarSession->ICal=NULL;
		return(FALSE);
	}

	ptr=Cache;
	Count = CacheSize;
	if (Session->NBufferPtr>0) {
		if (strlen(Session->NBuffer) > Count) {		/* Got the 1000 code in the buffer, too */
			memcpy(ptr, Session->NBuffer, Count);
			Session->NBufferPtr=strlen(Session->NBuffer + Count);
			memmove(Session->NBuffer, Session->NBuffer + Count, Session->NBufferPtr+1);
			Count = 0;
		} else if (Count >= (unsigned long)Session->NBufferPtr) {
			memcpy(ptr, Session->NBuffer, Session->NBufferPtr);
			ptr += Session->NBufferPtr;
			Count -= Session->NBufferPtr;
			Session->NBufferPtr = 0;
			Session->NBuffer[0] = '\0';
		} else {
			MemFree(Cache);
			CalendarSession->ICal = NULL;
			ModDebug("NMAP connection out of sync\n");
			return(FALSE);		    
		}
	}
	while (Count > 0) {
		if (Count < sizeof(Answer)) {
			len=MWDirectNMAPRead(Session, Answer, Count);
		} else {
			len=MWDirectNMAPRead(Session, Answer, sizeof(Answer));
		}
		if ((len > 0) && (Count <= (unsigned long)len)) {
			memcpy(ptr, Answer, len);
			ptr+=len;
			Count -= len;
		} else {
			MemFree(Cache);
			CalendarSession->ICal=NULL;
			ModDebug("Lost NMAP connection\n");
			return(FALSE);
		}
	}
	Cache[CacheSize]='\0';

	/* Grab confirmation of CSLIST */
	ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);
	if (ReplyInt!=1000) {
		if (Cache) {
			MemFree(Cache);
		}
		ModDebug("NMAP error: %s\n", Answer);
		return(FALSE);
	}
	CalendarSession->ICal = ICalParseObject(NULL, Cache, CacheSize);
	MemFree(Cache);

	/* Update the cached event as well */
	/* We need to update the status for all attendees in the ICal object with the status from NMAP */
	if (CalendarSession->ICal) {
		ICalObject *ICal = CalendarSession->ICal;
		ICalVAttendee *Attendee;
		for (Attendee = ICal->Attendee; Attendee; Attendee = Attendee->Next) {
			if (MWQuickCmp(Session->EMailAddress,Attendee->Address)) {
				CalendarSession->IAmAttendee = TRUE;
			}
		}

		ReplyInt=snprintf(Answer, sizeof(Answer), "CSATND %lu\r\n", CalendarSession->IDList[CalendarID]);
		MWSendNMAPServer(Session, Answer, ReplyInt);
		ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE);	/* List comming up */
		if (ReplyInt==2002) {
			while ((ReplyInt=MWGetNMAPAnswer(Session, Answer, sizeof(Answer), TRUE)) == 2002) {
				ICalVAttendee			*CurrentAttendee;
				unsigned char		*Address;

				/* Find the e-mail address */
				Address = strchr(Answer + 8, ' ');
				if (Address) {
					*Address = '\0';	/* We don't need anything after the address */
				}
				Address = Answer + 8;
				CurrentAttendee = CalendarSession->ICal->Attendee;
				while (CurrentAttendee) {
					if ( MWQuickCmp(Address, CurrentAttendee->Address)) {
						switch (toupper(Answer[0])) {
							case NMAP_CAL_STATE_NEED_ACT: {	CurrentAttendee->State = ICAL_PARTSTAT_NEEDS_ACTION;	break;	}
							case NMAP_CAL_STATE_ACCEPTED: {	CurrentAttendee->State = ICAL_PARTSTAT_ACCEPTED;	break;	}
							case NMAP_CAL_STATE_DECLINED: {	CurrentAttendee->State = ICAL_PARTSTAT_DECLINED;	break;	}
							case NMAP_CAL_STATE_TENTATIVE: {	CurrentAttendee->State = ICAL_PARTSTAT_TENTATIVE;	break;	}
							case NMAP_CAL_STATE_DELEGATED: {	CurrentAttendee->State = ICAL_PARTSTAT_DELEGATED;	break;	}
							case NMAP_CAL_STATE_COMPLETED: {	CurrentAttendee->State = ICAL_PARTSTAT_COMPLETED;	break;	}
							case NMAP_CAL_STATE_INPROCESS: {	CurrentAttendee->State = ICAL_PARTSTAT_IN_PROCESS;	break;	}
						}
					}
					CurrentAttendee = CurrentAttendee->Next;
				}
			}
		}
	}
	return(CalendarSession->ICal ? TRUE : FALSE);
}

BOOL
FreeCalendarEntryDetails(SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	if (CalendarSession->ICal!=NULL) {
		ICalFreeObjects(CalendarSession->ICal);
		CalendarSession->ICal=NULL;
	}
	return(TRUE);
}

BOOL
FindCalendarEntry(unsigned long Type, unsigned long UTCStart, unsigned long UTCEnd, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	if (LoadCalendarView(UTCStart, UTCEnd, Session, CalendarSession)==0) {
		return(FALSE);
	}

	while (CalendarSession->ViewPointer<CalendarSession->ViewUsed) {
		if (CalendarSession->ViewCache[CalendarSession->ViewPointer].Type==Type) {
			switch(Type) {
				case NMAP_CAL_EVENT:
					if ((CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCStart>=UTCStart && CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCStart<=UTCEnd) ||
						 (CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCEnd>=UTCStart && CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCEnd<=UTCEnd) ||
						 (CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCStart<=UTCStart && CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCEnd>=UTCEnd)) {
						CalendarSession->ViewPointer++;
						return(TRUE);
					}
					break;
				case NMAP_CAL_JOURNAL:
					if ((CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCStart>=UTCStart && CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCStart<=UTCEnd)) {
						CalendarSession->ViewPointer++;
						return(TRUE);
					}
					break;
				case NMAP_CAL_TODO:
					/* This one must include floating appointments */
					if (!(CalendarSession->ViewCache[CalendarSession->ViewPointer].State & MSG_STATE_COMPLETED)) {
						/* This is a floating appointment which only shows if today's date is being displayed */
						unsigned long CurrentTime = (time(NULL) + Session->TimezoneOffset);

						/* today's time  must fall within range */
						if ((CurrentTime >= UTCStart) && (CurrentTime <= UTCEnd)) {
							/* we can skip this one */
							CalendarSession->ViewPointer++;
							return(TRUE);
						}
					} else {
						if (CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCEnd>=UTCStart && CalendarSession->ViewCache[CalendarSession->ViewPointer].UTCEnd<=UTCEnd) {
							CalendarSession->ViewPointer++;
							return(TRUE);
						}
					}
					break;
				default:
					break;
			}
		}
		CalendarSession->ViewPointer++;
	}
	CalendarSession->ViewPointer=0;
	return(FALSE);
}

static int
SortViewCacheResults(const void *Entry1, const void *Entry2)
{
	ViewCacheStruct	*Cal1=(ViewCacheStruct *)Entry1;
	ViewCacheStruct	*Cal2=(ViewCacheStruct *)Entry2;

	if (Cal1->UTCStart<Cal2->UTCStart) {
		return(-1);
	}
	if (Cal1->UTCStart>Cal2->UTCStart) {
		return(1);
	}
	return(0);
}

#define	VIEWCACHE_ALLOC_STEPS	80

unsigned long
LoadCalendarView(unsigned long UTCStart, unsigned long UTCEnd, SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	int				ReplyInt;
	unsigned char	Reply[1024];
	unsigned int	NMAPID;
	long unsigned int  thisUTCStart,thisUTCEnd,State;
	unsigned long NMAPCount;
	unsigned long ID;
	unsigned int Type;

	if (CalendarSession->ViewUTCStart <= UTCStart && UTCEnd <= CalendarSession->ViewUTCEnd && CalendarSession->ViewCache) {
		return(CalendarSession->ViewUsed);
	}

	if (!CalendarSession->ViewCache) {
		CalendarSession->ViewCache = MemMalloc((CalendarSession->ViewAllocated + VIEWCACHE_ALLOC_STEPS)*sizeof(ViewCacheStruct));
	}

   /* No longer use CSFILT because we want all tasks that have occurred previously to show up only on todays date, or completed date. */
	MWSendNMAPServer(Session, "CSINFO\r\n", 8);
	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=2002) {
		return(FALSE);
	}
	NMAPCount = atol(Reply);
	CalendarSession->ViewUsed = 0;
	for (ID = 1; ID <= NMAPCount; ID++){
		MWGetNMAPAnswer(Session, Reply, sizeof(Reply), FALSE);

		if ((CalendarSession->ViewUsed+1) >= CalendarSession->ViewAllocated) {
			CalendarSession->ViewCache = MemRealloc(CalendarSession->ViewCache, (CalendarSession->ViewAllocated + VIEWCACHE_ALLOC_STEPS)*sizeof(ViewCacheStruct));
			if (!CalendarSession->ViewCache) {
				/* Bad news - clean up  */
				for( ;ID <= NMAPCount; ID++){
					MWGetNMAPAnswer(Session, Reply, sizeof(Reply), FALSE);
				}

				/* get 1000 ok */
				ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
				CalendarSession->ViewUsed = 0;
				return(FALSE);
			}
			CalendarSession->ViewAllocated += VIEWCACHE_ALLOC_STEPS;
		}
		sscanf(Reply, "%u %u %*u %*u %lu %lu %lu", &NMAPID, &Type, &State, &thisUTCStart, &thisUTCEnd);

		/* see if we can filter this entry */
		if (State & (MSG_STATE_PURGED|MSG_STATE_DELETED)) {
			continue;
		}
		if ( Type == NMAP_CAL_EVENT) {
			/* appointment - begin or end time must fall within range, or begin end time must encompass entire range */
			if ( !(((thisUTCStart >= UTCStart) && (thisUTCStart <= UTCEnd)) || ((thisUTCEnd >= UTCStart) && (thisUTCEnd <= UTCEnd))) && 
				  !((thisUTCStart <= UTCStart) && (thisUTCEnd >=UTCEnd))) {
				/* we can skip this one */
				continue;
			}
			CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCDue = 0;
		} else if (Type == NMAP_CAL_JOURNAL) {
			/* notes - begin time must fall within range */
			if ( !((thisUTCStart >= UTCStart) && (thisUTCStart <= UTCEnd))) {
				/* we can skip this one */
				continue;
			}

			/* UTCDue is either 0 or the due date of the task. */
			/* UTCEnd is initiatized to 0x7fffffff and is set to completion time when task is completed. */
			/* I don't like this because it ambiguous between CSFILT, and CSINFO calls */
			/* CSINFO returns only one parameter which can be either UTCEnd or UTCDue depending on state. */
			/* If UTCEnd == 0x7fffffff then it returns the UTCDue value, otherwise it returns the UTCEnd value */
			/* On the other hand CSFilter returns 0x7fffffff and UTCDue, or if complete only UTCEnd. */
			/* But can't change now. */
			/* To keep this the same as before we set UTCDue = 0 if not completed. */
         /* Otherwise we set UTCEnd and UTCDue equal to same value.*/
			CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCDue = 0;
		} else if (Type == NMAP_CAL_TODO) {
			/* task - if not completed then begin todays time must fall within range, otherwise end time must fall within range */
			if (!(State & MSG_STATE_COMPLETED)) {
				unsigned long CurrentTime = (time(NULL) + Session->TimezoneOffset);

				/* today's time  must fall within range */
				if ( !((CurrentTime >= UTCStart) && (CurrentTime <= UTCEnd))) {
					/* we can skip this one */
					continue;
				}
				CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCDue = 0;
			} else {
				/* end time must fall within range */
				if ( !((thisUTCEnd >= UTCStart) && (thisUTCEnd <= UTCEnd))) {
					/* we can skip this one */
					continue;
				}
				CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCDue = UTCEnd;
			}
		} else {
			/* UnRecognized */
			continue;
		}
		CalendarSession->ViewCache[CalendarSession->ViewUsed].Type = Type;
		CalendarSession->ViewCache[CalendarSession->ViewUsed].State = State;
		CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCStart = thisUTCStart;
		CalendarSession->ViewCache[CalendarSession->ViewUsed].UTCEnd = thisUTCEnd;
		ReplyInt = NMAPtoCalendarID(Session, CalendarSession, NMAPID);
		if (ReplyInt != -1) {
		    CalendarSession->ViewCache[CalendarSession->ViewUsed].ID = ReplyInt;
		    CalendarSession->ViewUsed++;
		}
	}

	/* Get the last reply very important */
	ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	if (ReplyInt!=1000) {
		if (CalendarSession->ViewUsed > 0) {
			MemFree(CalendarSession->ViewCache);
			CalendarSession->ViewCache = NULL;
		}
		CalendarSession->ViewUsed = 0;
		return(FALSE);
	}

	/* Sort the calendar */
	qsort(CalendarSession->ViewCache, CalendarSession->ViewUsed, sizeof(ViewCacheStruct), SortViewCacheResults);
	CalendarSession->ViewUTCStart = UTCStart;
	CalendarSession->ViewUTCEnd = UTCEnd;
	CalendarSession->ViewPointer = 0;

#if 0 //DEBUG
{
	int i;
	for (i=0; i<CalendarSession->ViewUsed; i++) {
		XplConsolePrintf("ID:%d, NMAPID:%d, Type:%d, Start:%x End:%x State:%d\n", CalendarSession->ViewCache[i].ID, CalendarSession->IDList[CalendarSession->ViewCache[i].ID-1], CalendarSession->ViewCache[i].Type, CalendarSession->ViewCache[i].UTCStart, CalendarSession->ViewCache[i].UTCEnd, CalendarSession->ViewCache[i].State);

	}
}
#endif
	return(CalendarSession->ViewUsed);
}

BOOL
FreeCalendarView(SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	if (CalendarSession->ViewCache != NULL) {
		MemFree(CalendarSession->ViewCache);
		CalendarSession->ViewCache = NULL;
	}
	CalendarSession->ViewUsed = 0;
	return(TRUE);
}


BOOL
UpdateCalendar(SessionStruct *Session, CalendarSessionStruct *CalendarSession)
{
	int				ReplyInt;
	unsigned char	Reply[1024];
	unsigned long	Total;
	unsigned long	Purged;

	if (!CalendarSession->CalendarSelected) {
		return(FALSE);
	}

	MWSendNMAPServer(Session, "CSUPDA\r\n", 8);

	do {
		ReplyInt=MWGetNMAPAnswer(Session, Reply, sizeof(Reply), TRUE);
	}while(ReplyInt == 6001);
	

	if (sscanf(Reply, "%*s %lu %*u %*u %*u %lu", &Total, &Purged) == 2) {

		FreeCalendarEntryDetails(Session, CalendarSession);

		/* We only set the times to 0, keep the memory around for re-use */
		CalendarSession->EntryID = 0;
		CalendarSession->DetailID = 0;

		LoadCalendarIDList(Session, CalendarSession);

		/* Invalidate any cached data */
		CalendarSession->ViewUTCStart = 0;
		CalendarSession->ViewUTCEnd = 0;

		return(TRUE);
	}

	CalendarSession->ViewUTCStart = 0;
	CalendarSession->ViewUTCEnd = 0;

	return(FALSE);
}
