/****************************************************************************
 *
 * Copyright (c) 2001-2002 Novell, Inc.
 * All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of version 2.1 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, contact Novell, Inc.
 *
 * To contact Novell about this file by physical or electronic mail,
 * you may find current contact information at www.novell.com
 *
 ****************************************************************************/

#include <config.h>
#include <stdlib.h>
#include <xpl.h>

#include <mwtempl.h>
#include <webadmin.h>

#include <rulesrv.h>
#include <hulautil.h>
#include <mdb.h>
#include <msgapi.h>

#include <wanmail.tok>
#include <wanmail.ary>

WAAPIDefine;

/* Features stuff */
#define FEATURE_USER_FIRST                  'O'        /* Object first        */
#define FEATURE_PARENT_FIRST                'P'        /* Parent first        */
#define FEATURE_USE_USER                    'U'        /* use User                */
#define FEATURE_USE_PARENT                  'I'        /* Inherit                */
#define FEATURE_AVAILABLE                   '1'        /* 1 = On                */
#define FEATURE_NOT_AVAILABLE               '0'        /* 0 = Off                */

#define FEATURE_IMAP                        'A', 1
#define FEATURE_POP                         'A', 2
#define FEATURE_ADDRESSBOOK                 'A', 3
#define FEATURE_PROXY                       'A', 4
#define FEATURE_FORWARD                     'A', 5
#define FEATURE_AUTOREPLY                   'A', 6
#define FEATURE_RULES                       'A', 7
#define FEATURE_FINGER                      'A', 8
#define FEATURE_SMTP_SEND_COUNT_LIMIT       'A', 9
#define FEATURE_SMTP_SEND_SIZE_LIMIT        'A', 10
#define FEATURE_NMAP_QUOTA                  'A', 11
#define FEATURE_NMAP_STORE                  'A', 12
#define FEATURE_WEBMAIL                     'A', 13
#define FEATURE_MODWEB                      'A', 14
#define FEATURE_MWMAIL_ADDRBOOK_P           'A', 15
#define FEATURE_MWMAIL_ADDRBOOK_S           'A', 16
#define FEATURE_MWMAIL_ADDRBOOK_G           'A', 17
#define FEATURE_MODWEB_WAP                  'A', 18
#define FEATURE_CALAGENT                    'A', 19
#define FEATURE_CALENDAR                    'A', 20
#define FEATURE_ANTIVIRUS                   'A', 21
#define FEATURE_SHARED_FOLDERS              'A', 22

#define DEFAULT_FEATURE_ROW_A               "AUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"
#define DEFAULT_FEATURE_ROW_B               "BUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"
#define DEFAULT_FEATURE_ROW_C               "CUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"
#define DEFAULT_FEATURE_ROW_D               "DUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"
#define DEFAULT_FEATURE_ROW_E               "EUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUUU"

#define SYSLOG_A_CONFIG "Novonyx:Syslog Config"
#define LOG_DEFAULT                         0
#define LOG_MAIL                            1
#define LOG_NEWS                            2
#define LOG_NDS                             3
#define LOG_NLM                             4
#define LOG_AUTH                            5
#define LOG_AUTHPRIV                        6
#define LOG_CRON                            7
#define LOG_DAEMON                          8
#define LOG_KERN                            9
#define LOG_LPR                             10
#define LOG_SYSLOG                          11
#define LOG_USER                            12
#define LOG_UUCP                            13
#define LOG_LOCAL0                          14
#define LOG_LOCAL1                          15
#define LOG_LOCAL2                          16
#define LOG_LOCAL3                          17
#define LOG_LOCAL4                          18
#define LOG_LOCAL5                          19
#define LOG_LOCAL6                          20
#define LOG_LOCAL7                          21

/* Requests */
#define REQUEST_NETMAIL_USER                15520
#define REQUEST_RULE_DELETE                 15521
#define REQUEST_RULE_EDIT                   15522
#define REQUEST_RULE_TOGGLE_STATE           15523
#define REQUEST_RULE_UP                     15524
#define REQUEST_RULE_DOWN                   15525

/* Bitfield for antivirus */
#define AV_USE_CA                           (1 << 0)
#define AV_USE_MCAFEE                       (1 << 1)
#define AV_USE_SYMANTEC                     (1 << 2)
#define AV_NOTIFY_USER                      (1 << 3)
#define AV_NOTIFY_SENDER                    (1 << 4)
#define AV_SCAN_INCOMING                    (1 << 5)
#define AV_USE_COMMANDAV                    (1 << 6)
#define AV_USE_CLAMAV                       (1 << 7)

/* Bitfield for lists */
#define LIST_CONFIG_MODERATED               (1 << 0)
#define LIST_CONFIG_SUBSCRIBE_CLOSED        (1 << 1)
#define LIST_CONFIG_ON_HOLD                 (1 << 2)
#define LIST_CONFIG_OPEN_LIST               (1 << 3)
#define LIST_CONFIG_SMTP_AUTH_REQUIRED      (1 << 4)        // used for NDS List only
#define LIST_CONFIG_ARCHIVE                 (1 << 5)
#define LIST_CONFIG_DIGEST                  (1 << 6)
#define LIST_CONFIG_NO_ATTACHMENTS          (1 << 7)
#define LIST_CONFIG_SIGNATURE               (1 << 8)
#define LIST_CONFIG_SIGNATURE_HTML          (1 << 9)
#define LIST_CONFIG_LIST_REPLY_TO           (1 << 10)
#define LIST_CONFIG_ANNOUNCEMENTS           (1 << 11)
#define LIST_CONFIG_SUBSCRIBE_OWNER         (1 << 12)
#define LIST_CONFIG_REVIEW_PRIVATE          (1 << 13)
#define LIST_CONFIG_REVIEW_OWNER            (1 << 14)

/* Rules helper functions */

/* Increment the rule pointer by x, and check for errors */
#define RuleInc(x)    {                                     \
    if (Rule + (x) <= RuleEnd) {                            \
        Rule += (x);                                        \
    } else {                                                \
        WASendClient(Client, "Broken Rule", 11);            \
        return(FALSE);                                      \
    }                                                       \
}

/* Check to make sure the rule has x more chars */
#define    RuleCheck(x) {                                   \
    if (Rule + (x) > (RuleEnd + 1)) {                       \
        WASendClient(Client, "Broken Rule", 11);            \
        return(FALSE);                                      \
    }                                                       \
}

/* Send a rule argument */
#define    RuleSendArgument(Send) {                         \
    unsigned long        Length;                            \
                                                            \
    RuleCheck(4);                                           \
    Length = ((Rule[0] - '0') * 100)                        \
        + ((Rule[1] - '0') * 10)                            \
        + (Rule[2] - '0');                                  \
    Rule += 3;                                              \
                                                            \
    if (Length > 0) {                                       \
        RuleCheck(Length);                                  \
        if (Rule[Length] != 'Z') {                          \
            WASendClient(Client, "Broken argument", 15);    \
            return(FALSE);                                  \
        }                                                   \
                                                            \
        if (Send) {                                         \
            WASendClient(Client, Rule, Length);             \
        }                                                   \
        Rule += Length;                                     \
                                                            \
        if (Send) {                                         \
            WASendClient(Client, " ", 1);                   \
        }                                                   \
    }                                                       \
    Rule ++; /* The Z */                                    \
}

struct {
    BOOL unload;

    struct {
        XplThreadID main;
        XplThreadID group;
    } id;

    struct {
        XplSemaphore shutdown;
    } sem;
} WANMail;

typedef struct {
    MDBEnumStruct *Enum;
    MDBValueStruct *CacheStruct;
    unsigned char *CurrentObject;
    unsigned char Object[MDB_MAX_OBJECT_CHARS + 1];
    unsigned long Counter;
    unsigned long RuleID;

    unsigned char *CurrentRuleSegment;
    unsigned char *NextRuleSegment;

    unsigned char RuleSegmentType;
    unsigned char RuleSegmentBool;

    unsigned long RuleSegmentArg1Len;
    unsigned char *RuleSegmentArg1;

    unsigned long RuleSegmentArg2Len;
    unsigned char *RuleSegmentArg2;

    MDBValueStruct *Domains;
    MDBValueStruct *Contexts;
} NMSessionStruct;

/* Temp data for syslog settings */
typedef struct {
    unsigned long Level;
    unsigned long Type;
    unsigned char FileName[64];
} NMSyslogSessionStruct;

/* Temp data for alias settings */
typedef struct {
    unsigned long Alias;
    BOOL WriteAlias;
} NMAliasSessionStruct;

/* Temp data for storing features */
typedef struct {
    MDBValueStruct *CacheStruct;
} NMFeaturesSessionStruct;

/* Temp data for saving rules */
typedef struct {
    MDBValueStruct *RuleNames;
    MDBValueStruct *RuleValues;
    unsigned long ID;
} RulesSessionStruct;

/*
    prototypes for registration functions
*/
BOOL WANMAILInitSession(SessionStruct *Session, void **ModuleData);
BOOL WANMAILDestroySession(SessionStruct *Session, void *ModuleData);
BOOL WANMAILHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData);
BOOL WANMAILHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData);

BOOL WANMAILSaveInetServicesFuncInit(SessionStruct *Session, void **ModuleData);
BOOL WANMAILSaveInetServicesFuncDestroy(SessionStruct *Session, void *ModuleData);

unsigned long WANMAILSaveInetServicesFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);
unsigned long WANMAILSaveComplexBitfieldsFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);
unsigned long WANMAILSaveNMAPFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);

BOOL WANMAILSaveAliasFuncInit(SessionStruct *Session, void **ModuleData);
BOOL WANMAILSaveAliasFuncDestroy(SessionStruct *Session, void *ModuleData);

unsigned long WANMAILSaveAliasFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);

BOOL WANMAILSaveFeaturesFuncInit(SessionStruct *Session, void **ModuleData);
BOOL WANMAILSaveFeaturesFuncDestroy(SessionStruct *Session, void *ModuleData);

unsigned long WANMAILSaveFeaturesFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);

unsigned long WANMAILNMAPCreateFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *ObjectName, const unsigned char *ParentDN, MDBValueStruct *Name, MDBValueStruct *Value, void *ModuleData);

BOOL WANMAILSaveRulesInit(SessionStruct *Session, void **ModuleData);
BOOL WANMAILSaveRulesDestroy(SessionStruct *Session, void *ModuleData);
unsigned long WANMAILSaveRules(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData);

unsigned long WANMAILPreventRename(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *NewName, void *ModuleData);
unsigned long WANMAILNMAPDelete(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, void *ModuleData);

/* 
    This function requires a value struct that is already filled out
    It simply returns a pointer to the right character
*/
static unsigned char *
FindFeature(MDBValueStruct *Values, unsigned char FeatureRow, unsigned long FeatureCol)
{
    unsigned long        c;

    for (c = 0; c < Values->Used; c++) {
        if (Values->Value[c][0] == FeatureRow) {
            return(Values->Value[c] + FeatureCol);
        }
    }

    return(NULL);
}

/*
    Replace MsgFindObject so that we don't have to rely on MSGApi.  We simply find the object,
    and return the DN, and the Class.

    We expect an email address to be passed in, so the first step is to search through our list
    of domains and try to match it.  If we do match it then we strip off the domain before looking
    for the user.  If we don't match it we assume its a hosted user and search with the full
    email address.
*/
static BOOL
WAFindUserByEmail(SessionStruct *Session, NMSessionStruct *NMSession, const unsigned char *Name, unsigned char *DN, unsigned char *Type)
{
    unsigned char        *atPtr = NULL;
    unsigned long        i;
    MDBValueStruct        *V = MDBCreateValueStruct(Session->AuthHandle, NULL);;

    atPtr = strchr(Name, '@');
    if (atPtr) {
        for (i = 0; i < NMSession->Domains->Used && *atPtr; i++) {
            if (WAQuickCmp(atPtr + 1, NMSession->Domains->Value[i])) {
                *atPtr = '\0';
            }
        }
    }

    for (i = 0; i < NMSession->Contexts->Used; i++) {
        MDBSetValueStructContext(NMSession->Contexts->Value[i], V);
        if (MDBIsObject(Name, V)) {
            MDBGetObjectDetails(Name, Type, NULL, DN, V);

            if (atPtr) {
                *atPtr = '@';
            }
            MDBFreeValues(V);
            return(TRUE);
        }
    }

    MDBDestroyValueStruct(V);
    return(FALSE);
}

static BOOL
CheckRuleString(ConnectionStruct *Client, unsigned char *CurrentRule, SessionStruct *Session)
{
    unsigned char        *Rule = CurrentRule + 9;
    unsigned char        *RuleEnd = CurrentRule + strlen(CurrentRule);
    BOOL                    ConditionsDone = FALSE;

    while (!ConditionsDone) {
        switch (*Rule) {
            case RULE_COND_ANY:
            case RULE_COND_FROM:
            case RULE_COND_TO:
            case RULE_COND_SUBJECT:
            case RULE_COND_PRIORITY:
            case RULE_COND_HEADER:
            case RULE_COND_BODY:
            case RULE_COND_FROM_NOT:
            case RULE_COND_TO_NOT:
            case RULE_COND_SUBJECT_NOT:
            case RULE_COND_PRIORITY_NOT:
            case RULE_COND_HEADER_NOT:
            case RULE_COND_BODY_NOT:
            case RULE_COND_SIZE_MORE:
            case RULE_COND_SIZE_LESS:
            case RULE_COND_FREE:
            case RULE_COND_FREENOT:
            case RULE_COND_HASMIMETYPE:
            case RULE_COND_HASMIMETYPENOT: {
                /* We have a valid condition */
                break;
            }
            default: {
                ConditionsDone = TRUE;
            }
        }
        RuleInc(1);

        if (!ConditionsDone) {
            /* Argument 1 */
            RuleSendArgument(FALSE);

            /* Argument 2 */
            RuleSendArgument(FALSE);

            switch (*Rule) {
                case RULE_ACT_BOOL_AND:
                case RULE_ACT_BOOL_OR:
                case RULE_ACT_BOOL_NOT: {
                    /* We're still good */
                    RuleInc(1);
                    break;
                }
                default: {
                    ConditionsDone = TRUE;
                }
            }
        }
    }

    while (Rule < RuleEnd) {
        switch (*Rule) {
            case RULE_ACT_REPLY:
            case RULE_ACT_DELETE:
            case RULE_ACT_FORWARD:
            case RULE_ACT_COPY:
            case RULE_ACT_MOVE:
            case RULE_ACT_ACCEPT:
            case RULE_ACT_DECLINE:
            case RULE_ACT_STOP: {
                break;
            }
        }
        RuleInc(1);

        /* Argument 1 */
        RuleSendArgument(FALSE);

        /* Argument 2 */
        RuleSendArgument(FALSE);
    }
    return(TRUE);
}

static BOOL
SendRuleString(ConnectionStruct *Client, unsigned char *CurrentRule, SessionStruct *Session)
{
    unsigned char        *Rule = CurrentRule + 9;
    unsigned char        *RuleEnd = CurrentRule + strlen(CurrentRule);
    unsigned char        *ptr = NULL;
    BOOL                    ConditionsDone = FALSE;
    BOOL                    First = TRUE;
    unsigned char        ConditionChar    = '\0';

    while (!ConditionsDone) {
        ConditionChar = *Rule;

        switch (*Rule) {
            case RULE_COND_ANY:                    ptr = "for all messages";                            break;
            case RULE_COND_FROM:                    ptr = "if from contains";                            break;
            case RULE_COND_TO:                    ptr = "if to contains";                                break;
            case RULE_COND_SUBJECT:                ptr = "if subject contains";                        break;
            case RULE_COND_PRIORITY:            ptr = "if priority contains";                        break;
            case RULE_COND_HEADER:                ptr = "if header field";                            break;
            case RULE_COND_BODY:                    ptr = "if body contains";                            break;
            case RULE_COND_FROM_NOT:            ptr = "if from doens't contain";                    break;
            case RULE_COND_TO_NOT:                ptr = "if to doens't contain";                    break;
            case RULE_COND_SUBJECT_NOT:        ptr = "if subject doens't contain";                break;
            case RULE_COND_PRIORITY_NOT:        ptr = "if priority doens't contain";            break;
            case RULE_COND_HEADER_NOT:            ptr = "if header field";                            break;
            case RULE_COND_BODY_NOT:            ptr = "if body doens't contain";                    break;
            case RULE_COND_SIZE_MORE:            ptr = "if size is more than";                        break;
            case RULE_COND_SIZE_LESS:            ptr = "if size is less than";                        break;
            case RULE_COND_FREE:                    ptr = "if appointment does not conflict";        break;
            case RULE_COND_FREENOT:                ptr = "if appointment does conflict";            break;
            case RULE_COND_HASMIMETYPE:        ptr = "if has attachment of type";                break;
            case RULE_COND_HASMIMETYPENOT:    ptr = "if doesn't have attachment of type";    break;
            default: {
                ptr = NULL;
                ConditionsDone = TRUE;
            }
        }

        if (ptr) {
            if (First) {
                unsigned char        FirstLetter = toupper(*ptr);
                /*
                    We want the first letter to be uppercase, and its a pain
                */

                WASendClient(Client, &FirstLetter, 1);
                WASendClient(Client, ptr + 1, strlen(ptr + 1));
                First = FALSE;
            } else {
                WASendClient(Client, ptr, strlen(ptr));
            }
        }

        RuleInc(1);

        WASendClient(Client, " ", 1);
        if (!ConditionsDone) {
            /* Argument 1 */
            RuleSendArgument(TRUE);

            if (ConditionChar == RULE_COND_HEADER) {
                WASendClient(Client, " contains ", strlen(" contains "));
            } else if (ConditionChar == RULE_COND_HEADER_NOT) {
                WASendClient(Client, " doesn't contains ", strlen(" doesn't contains "));
            }

            /* Argument 2 */
            RuleSendArgument(TRUE);

            switch (*Rule) {
                case RULE_ACT_BOOL_AND:    ptr = "and";    break;
                case RULE_ACT_BOOL_OR:    ptr = "or";        break;
                case RULE_ACT_BOOL_NOT:    ptr = "not";    break;
                default: {
                    ptr = NULL;
                    ConditionsDone = TRUE;
                }
            }
            if (ptr) {
                WASendClient(Client, ptr, strlen(ptr));
                RuleInc(1);
                WASendClient(Client, " ", 1);
            }
        }
    }

    while (Rule < RuleEnd) {
        switch (*Rule) {
            case RULE_ACT_REPLY:        ptr = "then reply with";        break;
            case RULE_ACT_DELETE:    ptr = "then delete message";    break;
            case RULE_ACT_FORWARD:    ptr = "then forward to";        break;
            case RULE_ACT_COPY:        ptr = "then carbon copy to";    break;
            case RULE_ACT_MOVE:        ptr = "then move to";            break;
            case RULE_ACT_STOP:        ptr = "stop processing";        break;
            case RULE_ACT_ACCEPT:    ptr = "then accept";                break;
            case RULE_ACT_DECLINE:    ptr = "then decline";            break;
            default: {
                ptr = NULL;
            }
        }

        if (ptr) {
            WASendClient(Client, ptr, strlen(ptr));
        }

        RuleInc(1);
        WASendClient(Client, " ", 1);

        /* Argument 1 */
        RuleSendArgument(TRUE);

        /* Argument 2 */
        RuleSendArgument(TRUE);

        if (Rule < RuleEnd) {
            WASendClient(Client, "and ", 4);
        }
    }
    return(TRUE);
}

/*
    InitSession is called once per user login; it can store any private 
    data behind ModuleData; it will get it passed back in all other calls
*/

BOOL
WANMAILInitSession(SessionStruct *Session, void **ModuleData)
{
    MDBValueStruct        *config = NULL;
    NMSessionStruct    *NMSession;

    NMSession = malloc(sizeof(NMSessionStruct));
    if (!NMSession ) {
        XplConsolePrintf("WANMail Module out of memory!\n");
        return(FALSE);
    }

    memset(NMSession, 0, sizeof(NMSessionStruct));

    NMSession->CacheStruct = MDBShareContext(Session->V);

    config = MDBCreateValueStruct(Session->AuthHandle, NULL);
    if (!config) {
        return(FALSE);
    }

    /*
        Read a list of all NetMail system domains, so that we can tell if
        a user should be looked up as just the user name, or with the
        domain included (ie hosted user).
    */
    NMSession->Domains = MDBCreateValueStruct(Session->AuthHandle, NULL);
    NMSession->Contexts = MDBCreateValueStruct(Session->AuthHandle, NULL);

    MDBEnumerateObjects(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, config);
    while (config->Used) {
        MDBSetValueStructContext(config->Value[0], NMSession->Domains);
        MDBRead(MSGSRV_AGENT_SMTP, MSGSRV_A_DOMAIN, NMSession->Domains);

        MDBFreeValue(0, config);
    }

    /*
        Read a list of all configured NetMail contexts, to use for looking up users.
    */

    MDBEnumerateObjects(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, config);
    while (config->Used) {
        MDBRead(config->Value[0], MSGSRV_A_CONTEXT, NMSession->Contexts);

        MDBFreeValue(0, config);
    }

    MDBDestroyValueStruct(config);

    *ModuleData = (void *)NMSession;
    return(TRUE);
}


/*
    DestroySession is called once per user logout or timeout; it is supposed
    to destroy any private data stored behind ModuleData.
*/
BOOL
WANMAILDestroySession(SessionStruct *Session, void *ModuleData)
{
    NMSessionStruct    *NMSession = (NMSessionStruct *)ModuleData;

    if (NMSession->CacheStruct) {
        MDBDestroyValueStruct(NMSession->CacheStruct);
    }

    if (NMSession->Contexts) {
        MDBDestroyValueStruct(NMSession->Contexts);
    }

    if (NMSession->Domains) {
        MDBDestroyValueStruct(NMSession->Domains);
    }


    if (NMSession) {
        free(NMSession);
    }

    return(TRUE);
}

static int 
CompareRuleID(const void *First, const void *Second)
{
    const unsigned char    *One    = *(unsigned char**)First;
    const unsigned char    *Two    = *(unsigned char**)Second;
    unsigned long            IDOne;
    unsigned long            IDTwo;
    unsigned char            Temp;

    if (!One || !Two || strlen((unsigned char *)One) < 9 || strlen((unsigned char *)Two) < 9) {
        return(0);
    }

    Temp = ((unsigned char *)One)[8];
    ((unsigned char *)One)[8] = '\0';
    IDOne = strtol((unsigned char *)One, NULL, 16);
    ((unsigned char *)One)[8] = Temp;

    Temp = ((unsigned char *)Two)[8];
    ((unsigned char *)Two)[8] = '\0';
    IDTwo = strtol((unsigned char *)Two, NULL, 16);
    ((unsigned char *)Two)[8] = Temp;

    if (IDOne < IDTwo) {
        return(-1);
    } else if (IDOne > IDTwo) {
        return(1);
    } else {
        return(0);
    }
}

BOOL
WANMAILHandleTemplate(ConnectionStruct *Client, SessionStruct *Session, unsigned long Page, TokenOverlayStruct *Token, unsigned long *GotoToken, void *ModuleData)
{
    NMSessionStruct    *NMSession = (NMSessionStruct *)ModuleData;
    unsigned char        Buffer[BUFSIZE+1];
    unsigned char        URL[256];
    int                    Len;

    switch(Token->TokenID) {
        case T_QUEUE_LIST_BEGIN: {
            const unsigned char        *CurrentObject;
            BOOL                            Found = FALSE;

            if (!NMSession->Enum) {
                /* We are starting */

                NMSession->Enum = MDBCreateEnumStruct(Session->V);
                if (Token->ArgumentOffsetOrID[0]) {
                    MDBRead(Session->CurrentObject, Session->Strings[Token->ArgumentOffsetOrID[0]], Session->V);
                }
            }

            while (!Found) {
                CurrentObject = MDBEnumerateObjectsEx(MSGSRV_ROOT, MSGSRV_C_SERVER, NULL, 0, NMSession->Enum, Session->V);
                if (CurrentObject) {
                    snprintf(NMSession->Object, sizeof(NMSession->Object), "%s\\%s", CurrentObject, MSGSRV_AGENT_NMAP);
                    if (MDBIsObject(NMSession->Object, Session->V)) {
                        Found = TRUE;
                    }
                } else {
                    /* We are done with the enumeration, so we skip to T_QUEUE_LIST_END */
                    MDBDestroyEnumStruct(NMSession->Enum, Session->V);
                    MDBFreeValues(Session->V);
                    NMSession->Enum = NULL;

                    *GotoToken = T_QUEUE_LIST_END;
                    return(TOKEN_MOVE_FORWARD);
                }
            }

            break;
        }

        case T_QUEUE_LIST_END: {
            if (NMSession->Enum) {
                *GotoToken = T_QUEUE_LIST_BEGIN;
                return(TOKEN_MOVE_BACKWARD);
            }

            break;
        }

        case T_QUEUE_LIST_VALUE: {
            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_QNAME: {
                    WAMDBtoX500(NMSession->Object, Buffer, Session->V);
                    WASendClient(Client, Buffer, strlen(Buffer));
                    break;
                }

                case AA_SELECTED: {
                    BOOL                    Match = FALSE;
                    unsigned long        i;

                    for (i = 0; i < Session->V->Used; i++) {
                        if (WAQuickCmp(Session->V->Value[i], NMSession->Object)) {
                            Match = TRUE;
                        }
                    }
                    if (Match) {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                    } else {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                    }

                    break;
                }
            }

            break;
        }

        case T_SYSLOG: {
            unsigned long            i;
            unsigned long            Level = 0;
            unsigned long            Type = 3;

            Buffer[0] = '\0';

            if (MDBRead(Session->CurrentObject, SYSLOG_A_CONFIG, Session->V)) {
                for (i = 0; i < Session->V->Used; i++) {
                    if (atol(Session->V->Value[i]) == 1) {                /* Mail facility */
                        sscanf(Session->V->Value[i], "%*d %lu %lu %*s %s", &Level, &Type, Buffer);
                    }
                }
                MDBFreeValues(Session->V);
            }

            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_LEVEL: {
                    if (Level >= Token->ArgumentOffsetOrID[3]) {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                    } else {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                    }
                    break;
                }

                case AA_FILE: {
                    WASendClient(Client, Buffer, strlen(Buffer));
                    break;
                }

                case AA_ENABLED: {
                    if (Type == 2) {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                    } else {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                    }
                    break;
                }
            }
            break;
        }

        case T_GLOBAL_ALIASES: {
            unsigned char        *ptr;

            HulaStrNCpy(Buffer, Session->CurrentObject, sizeof(Buffer));
            ptr = strchr(Buffer + 1, '\\');
            if (ptr) {
                ptr = strchr(ptr + 1, '\\');
                if (ptr) {
                    *ptr = '\0';
                    MDBRead(Buffer, MSGSRV_A_ALIAS, Session->V);

                    while (Session->V->Used) {
                        WASendClient(Client, Session->V->Value[0], strlen(Session->V->Value[0]));
                        WASendClient(Client, "\r\n", 2);
                        MDBFreeValue(0, Session->V);
                    }
                }
            }

            break;
        }

        case T_BOUNCE: {
            unsigned long        Len;
            unsigned long        BounceEnabled;
            unsigned long        BounceInterval;
            unsigned long        BounceEntries;

            if (MDBRead(Session->CurrentObject, MSGSRV_A_RTS_ANTISPAM_CONFIG, Session->V)) {
                sscanf(Session->V->Value[0], "Enabled:%lu Delay:%lu Threshhold:%lu", &BounceEnabled, &BounceInterval, &BounceEntries);
                MDBFreeValues(Session->V);
            } else {
                BounceEnabled    = 0;
                BounceInterval    = 0;
                BounceEntries    = 0;
            }

            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_ENABLED: {
                    if (BounceEnabled) {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                    } else {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                    }
                    break;
                }

                case AA_INTERVAL: {
                    Len = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", BounceInterval);
                    WASendClient(Client, Client->Temp, Len);
                    break;
                }

                case AA_ENTRIES: {
                    Len = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", BounceEntries);
                    WASendClient(Client, Client->Temp, Len);
                    break;
                }
            }

            break;
        }

        case T_CONTEXT: {
            unsigned char        *ptr;

            /* List all the contexts on the Messaging Server object that contains this object */

            HulaStrNCpy(Buffer, Session->CurrentObject, sizeof(Buffer));
            ptr = strrchr(Buffer, '\\');
            if (ptr) {
                *ptr = '\0';
                MDBRead(Buffer, MSGSRV_A_CONTEXT, Session->V);

                while (Session->V->Used) {
                    WAMDBtoX500(Session->V->Value[0], Buffer, Session->V);
                    WASendClient(Client, Buffer, strlen(Buffer));

                    MDBFreeValue(0, Session->V);

                    if (Session->V->Used >= 1) {
                        WASendClient(Client, "\r\n", 2);
                    }
                }
            }

            break;
        }

        case T_DEFAULT_LANG_LIST_BEGIN: {
            BOOL            Match = FALSE;

            if (NMSession->Counter == 0) {
                MDBRead(Session->CurrentObject, MSGSRV_A_LANGUAGE, Session->V);
            }

            do {
                if (XplReturnLanguageName(NMSession->Counter, NMSession->Object) == 0) {
                    if (!WAQuickNCmp("Unknown (ID:", NMSession->Object, 12)) {
                        Match = TRUE;
                    }
                }
                NMSession->Counter++;
            } while (NMSession->Counter < 100 && !Match);

            if (!Match) {
                *GotoToken = T_DEFAULT_LANG_LIST_END;
                return(TOKEN_MOVE_FORWARD);
            }

            break;
        }

        case T_DEFAULT_LANG_LIST_END: {
            if (NMSession->Counter < 100) {

                *GotoToken = T_DEFAULT_LANG_LIST_BEGIN;
                return(TOKEN_MOVE_BACKWARD);
            }
            NMSession->Counter = 0;
            MDBFreeValues(Session->V);

            break;
        }

        case T_DEFAULT_LANG_LIST_VALUE: {
            signed long        len;

            if (Session->V->Used) {
                len = atol(Session->V->Value[0]);
            } else {
                len = -1;
            }

            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_SELECTEDLANGUAGE: {
                    if (len == (NMSession->Counter - 1)) {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                    } else {
                        WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                    }

                    break;
                }

                case AA_LANGUAGENAME: {
                    WASendClient(Client, NMSession->Object, strlen(NMSession->Object));

                    break;
                }

                case AA_LANGUAGENUMBER: {
                    Len = snprintf(Client->Temp, sizeof(Client->Temp), "%lu", NMSession->Counter - 1);
                    WASendClient(Client, Client->Temp, Len);
                }
            }

            break;
        }

        case T_FEATURE_END: {
            MDBFreeValues(NMSession->CacheStruct);
            break;
        }

        case T_FEATURE_VALUE: {
            unsigned char        *ptr = NULL;

            if (!NMSession->CacheStruct->Used) {
                MDBRead(Session->CurrentObject, MSGSRV_A_FEATURE_SET, NMSession->CacheStruct);
            }

            switch (Token->ArgumentOffsetOrID[0]) {
                case AA_FEATURE_IMAP:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_IMAP);                            break;    }
                case AA_FEATURE_POP:                            {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_POP);                            break;    }
                case AA_FEATURE_ADDRESSBOOK:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_ADDRESSBOOK);                break;    }
                case AA_FEATURE_PROXY:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_PROXY);                        break;    }
                case AA_FEATURE_FORWARD:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_FORWARD);                        break;    }
                case AA_FEATURE_AUTOREPLY:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_AUTOREPLY);                    break;    }
                case AA_FEATURE_RULES:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_RULES);                        break;    }
                case AA_FEATURE_FINGER:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_FINGER);                        break;    }
                case AA_FEATURE_SMTP_SEND_COUNT_LIMIT:    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_SMTP_SEND_COUNT_LIMIT);    break;    }
                case AA_FEATURE_SMTP_SEND_SIZE_LIMIT:    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_SMTP_SEND_SIZE_LIMIT);    break;    }
                case AA_FEATURE_NMAP_QUOTA:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_NMAP_QUOTA);                    break;    }
                case AA_FEATURE_NMAP_STORE:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_NMAP_STORE);                    break;    }
                case AA_FEATURE_WEBMAIL:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_WEBMAIL);                        break;    }
                case AA_FEATURE_MODWEB:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MODWEB);                        break;    }
                case AA_FEATURE_MWMAIL_ADDRBOOK_P:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_P);        break;    }
                case AA_FEATURE_MWMAIL_ADDRBOOK_S:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_S);        break;    }
                case AA_FEATURE_MWMAIL_ADDRBOOK_G:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_G);        break;    }
                case AA_FEATURE_MODWEB_WAP:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MODWEB_WAP);                    break;    }
                case AA_FEATURE_CALAGENT:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_CALAGENT);                    break;    }
                case AA_FEATURE_CALENDAR:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_CALENDAR);                    break;    }
                case AA_FEATURE_ANTIVIRUS:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_ANTIVIRUS);                    break;    }
            }

            if (ptr && (toupper(*ptr) == toupper((Token->Data + Token->ArgumentOffsetOrID[1])[0]))) {
                WASendClient(Client, Token->Data + Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
            } else {
                WASendClient(Client, Token->Data + Token->ArgumentOffsetOrID[3], Token->ArgumentSize[3]);
            }
            break;
        }

        case T_FEATURE_SET: {
            Len = snprintf(Buffer, sizeof(Buffer), "OF_%lu", Token->ArgumentOffsetOrID[0]);
            WASendClient(Client, Buffer, Len);
            
            break;
        }

        case T_GOTO_USER: {
            WAEncodeURL(Session, URL, URL_TYPE_LINK, REQUEST_NETMAIL_USER, Token->ArgumentOffsetOrID[0], Token->ArgumentOffsetOrID[1], 0, 0);
            WASendClient(Client, URL, strlen(URL));

            break;
        }

        case T_USER_RULE_LIST_BEGIN : {
            if (Session->V->Used == 0) {
                MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V);

                if (Session->V->Used == 0) {
                    *GotoToken=T_USER_RULE_LIST_END;
                    return(TOKEN_MOVE_FORWARD);
                }

                /*
                    Sort by unique rule ID, because we are using the id to define order
                */
                qsort(Session->V->Value, Session->V->Used, sizeof(unsigned char*), CompareRuleID);
            }
            return(TRUE);
        }

        case T_USER_RULE_LIST_END : {
            if (Session->V->Used) {
                MDBFreeValue(0, Session->V);
            }

            if (Session->V->Used) {
                /* We have more to display */

                *GotoToken = T_USER_RULE_LIST_BEGIN;
                return(TOKEN_MOVE_BACKWARD);
            }
            NMSession->CurrentRuleSegment = NULL;
            NMSession->RuleID = 0;

            return(TRUE);
        }

        case T_USER_RULE_LIST_VALUE : {
            if (Session->V->Used) {
                if (Session->V->Used && strlen(Session->V->Value[0]) > 8) {
                    switch (Token->ArgumentOffsetOrID[0]) {
                        case AA_RULESTRING: {
                            if (CheckRuleString(Client, Session->V->Value[0], Session)) {
                                SendRuleString(Client, Session->V->Value[0], Session);
                            }
                            break;
                        }
                        case AA_RULEACTIVE: {
                            if (Session->V->Value[0][8] == RULE_ACTIVE) {
                                WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[1], Token->ArgumentSize[1]);
                            } else {
                                WASendClient(Client, Token->Data+Token->ArgumentOffsetOrID[2], Token->ArgumentSize[2]);
                            }
                            break;
                        }
                    }
                }
            }
            return(TRUE);
        }

        case T_USER_RULE_LIST_ACTION : {
            unsigned long        ID = 0;
            unsigned long        Action = REQUEST_OBJECT;

            if (Session->V->Used && strlen(Session->V->Value[0]) > 8) {
                unsigned char        Temp = Session->V->Value[0][8];

                Session->V->Value[0][8] = '\0';
                ID = strtol(Session->V->Value[0], NULL, 16);
                Session->V->Value[0][8] = Temp;

                switch (Token->ArgumentOffsetOrID[0]) {
                    case AA_RULEEDIT: {
                        Action = REQUEST_RULE_EDIT;
                        break;
                    }

                    case AA_RULEDELETE: {
                        Action = REQUEST_RULE_DELETE;
                        break;
                    }

                    case AA_RULETOGGLE: {
                        Action = REQUEST_RULE_TOGGLE_STATE;
                        break;
                    }

                    case AA_RULEUP: {
                        Action = REQUEST_RULE_UP;
                        break;
                    }

                    case AA_RULEDOWN: {
                        Action = REQUEST_RULE_DOWN;
                        break;
                    }
                }
            }
            WAEncodeURL(Session, URL, URL_TYPE_LINK, Action, Token->ArgumentOffsetOrID[1], ID, TRUE, TRUE);
            WASendClient(Client, URL, strlen(URL));

            WASendURLExtra(Client, Session->CurrentObject);
            return(TRUE);
        }

        case T_USER_RULE_ID: {
            Len = snprintf(Buffer, sizeof(Buffer), "%lu", NMSession->RuleID);
            WASendClient(Client, Buffer, Len);
            break;
        }

        case T_USER_RULE_PART_BEGIN: {
            /*
                If the rule type (argument 0) is AA_RULECONDITION we just need to find the
                selected rule, and setup to display the first condition.  If the type
                is AA_RULEACTION then we need to make sure we skip past the conditions of
                the rule.  Since about 95% of the time the actions will directly follow
                the conditions in the template it seems like a waste to read and parse
                the rule twice, but we can't rely on that.  We have to make sure we
                clean up for each.
            */

            if (!Session->V->Used) {
                BOOL            Found = FALSE;

                /* We are comming in for the first time */

                if (MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V)) {

                    /*
                        A single rule matching NMSession->RuleID should
                        be left after this for loop
                    */

                    while (!Found && Session->V->Used > 0) {
                        unsigned long        ID;

                        if (strlen(Session->V->Value[0]) > 8) {
                            unsigned char        Temp = Session->V->Value[0][8];

                            Session->V->Value[0][8] = '\0';
                            ID = strtol(Session->V->Value[0], NULL, 16);
                            Session->V->Value[0][8] = Temp;

                            if (ID != NMSession->RuleID) {
                                /*
                                    We found a non-matching ID, so lets rip it out
                                */
                                MDBFreeValue(0, Session->V);
                            } else {
                                /* Lets free all values except the first one */
                                while (Session->V->Used > 1) {
                                    MDBFreeValue(1, Session->V);
                                }
                                Found = TRUE;
                            }
                        } else {
                            MDBFreeValue(0, Session->V);
                        }
                    }
                }

                if (!Session->V->Used) {
                    /*
                        We didn't read a value, so there is no rule
                        on this object.  We will continue as normal
                        and RuleCondEnd will break the loop
                    */
                    break;
                }
                if (Token->ArgumentOffsetOrID[0] == AA_RULECONDITION) {
                    NMSession->CurrentRuleSegment = Session->V->Value[0] + 9;
                } else if (Token->ArgumentOffsetOrID[0] == AA_RULEACTION) {
                    unsigned char        *ptr;

                    /*
                        We now need to get past all the conditions, so we know
                        we are at the start of the actions.
                    */

                    ptr = Session->V->Value[0] + 8;        /* The ID, Active/Inactive flag, and the first condition type */
                    do {
                        unsigned long        Len;

                        ptr += 2;    /* Condition and bool (active/inactive flag on the first instead of bool */

                        /* We know this is a condition, so skip past its args */
                        Len = ((ptr[0] - '0') * 100) + ((ptr[1] - '0') * 10) + (ptr[2] - '0');
                        if ((Len + 3) < strlen(ptr)) {
                            ptr += Len + 4;    /* The length plus the terminator character */
                        } else {
                            /* Incomplete rule.  Just bail */
                            ptr = NULL;
                        }

                        if (ptr) {
                            Len = ((ptr[0] - '0') * 100) + ((ptr[1] - '0') * 10) + (ptr[2] - '0');
                            if ((Len + 3) < strlen(ptr)) {
                                ptr += Len + 4;    /* The length plus the terminator character */
                            } else {
                                /* Incomplete rule.  Just bail */
                                ptr = NULL;
                            }
                        }
                    } while (ptr && *ptr >= RULE_ACT_BOOL_START);

                    NMSession->CurrentRuleSegment = ptr;
                }
            } else {
                if (Token->ArgumentOffsetOrID[0] == AA_RULECONDITION) {
                    if (NMSession->CurrentRuleSegment && (*NMSession->CurrentRuleSegment >= RULE_ACT_BOOL_START && *NMSession->CurrentRuleSegment <= RULE_ACT_BOOL_NOT)) {
                        NMSession->RuleSegmentBool = *NMSession->CurrentRuleSegment;
                        NMSession->CurrentRuleSegment += 1;
                    } else {
                        /* We are done with conditions. */
                        NMSession->CurrentRuleSegment = NULL;

                        *GotoToken = T_USER_RULE_PART_END;
                        return(TOKEN_MOVE_FORWARD);
                    }
                } else if (Token->ArgumentOffsetOrID[0] == AA_RULEACTION) {
                    NMSession->RuleSegmentBool = '\0';
                }
            }

            if (NMSession->CurrentRuleSegment) {
                NMSession->RuleSegmentType = NMSession->CurrentRuleSegment[0];
                NMSession->CurrentRuleSegment += 1;
            }

            /* Grab arg1 */
            if (NMSession->CurrentRuleSegment && strlen(NMSession->CurrentRuleSegment) > 3) {
                NMSession->RuleSegmentArg1Len = ((NMSession->CurrentRuleSegment[0] - '0') * 100) + ((NMSession->CurrentRuleSegment[1] - '0') * 10) + (NMSession->CurrentRuleSegment[2] - '0');
                if (NMSession->RuleSegmentArg1Len + 3 <= strlen(NMSession->CurrentRuleSegment)) {
                    NMSession->RuleSegmentArg1 = NMSession->CurrentRuleSegment + 3;

                    /* The three digit length, the string itself and the terminator */
                    NMSession->CurrentRuleSegment += (4 + NMSession->RuleSegmentArg1Len);
                } else {
                    NMSession->CurrentRuleSegment = NULL;
                }
            }

            /* Grab arg2 */
            if (NMSession->CurrentRuleSegment && strlen(NMSession->CurrentRuleSegment) > 3) {
                NMSession->RuleSegmentArg2Len = ((NMSession->CurrentRuleSegment[0] - '0') * 100) + ((NMSession->CurrentRuleSegment[1] - '0') * 10) + (NMSession->CurrentRuleSegment[2] - '0');
                if (NMSession->RuleSegmentArg2Len + 3 <= strlen(NMSession->CurrentRuleSegment)) {
                    NMSession->RuleSegmentArg2 = NMSession->CurrentRuleSegment + 3;

                    /* The three digit length, the string itself and the terminator */
                    NMSession->CurrentRuleSegment += (4 + NMSession->RuleSegmentArg2Len);
                } else {
                    NMSession->CurrentRuleSegment = NULL;
                }
            }

            break;
        }

        case T_USER_RULE_PART_END: {
            if (NMSession->CurrentRuleSegment && *(NMSession->CurrentRuleSegment)) {
                /* We still have stuff to send, so lets jump back to RuleCondBegin */

                /* Clean up for the next time through */
                NMSession->RuleSegmentType = '\0';
                NMSession->RuleSegmentBool = '\0';

                NMSession->RuleSegmentArg1Len = 0;
                NMSession->RuleSegmentArg1 = NULL;

                NMSession->RuleSegmentArg2Len = 0;
                NMSession->RuleSegmentArg2 = NULL;

                *GotoToken = T_USER_RULE_PART_BEGIN;
                return(TOKEN_MOVE_BACKWARD);
            }
            NMSession->CurrentRuleSegment = NULL;
            MDBFreeValues(Session->V);

            break;
        }

        case T_USER_RULE_COND_VALUE:
        case T_USER_RULE_ACTION_VALUE: {
            if (NMSession->CurrentRuleSegment) {
                switch (Token->ArgumentOffsetOrID[0]) {
                    case AA_RULEBOOL: {
                        if (NMSession->RuleSegmentBool) {
                            WASendClient(Client, &(NMSession->RuleSegmentBool), 1);
                        } else {
                            WASendClient(Client, "0", 1);
                        }
                        break;
                    }

                    case AA_RULECONDITION:
                    case AA_RULEACTION: {
                        if (NMSession->RuleSegmentType) {
                            WASendClient(Client, &(NMSession->RuleSegmentType), 1);
                        } else {
                            WASendClient(Client, "0", 1);
                        }
                        break;
                    }

                    case AA_RULEARG1: {
                        if (NMSession->RuleSegmentArg1) {
                            WASendClient(Client, NMSession->RuleSegmentArg1, NMSession->RuleSegmentArg1Len);
                        }
                        break;
                    }

                    case AA_RULEARG2: {
                        if (NMSession->RuleSegmentArg2) {
                            WASendClient(Client, NMSession->RuleSegmentArg2, NMSession->RuleSegmentArg2Len);
                        }
                        break;
                    }
                }
            }
            break;
        }

        default:
            break;
    }

    return(FALSE);
}

/*
    WANMAILHandleURL gets called for each all URLs directed to this module
*/
BOOL
WANMAILHandleURL(ConnectionStruct *Client, SessionStruct *Session, URLStruct *URL, void *ModuleData)
{
    NMSessionStruct    *NMSession = (NMSessionStruct *)ModuleData;

    switch(URL->Request) {
        case REQUEST_NETMAIL_USER: {
            unsigned char            *Data;
            unsigned char            *ptr;
            unsigned long            Allocated=0;
            unsigned long            ValueSize;
            BOOL                        Found = FALSE;

            Allocated=BUFSIZE;
            Data=malloc(Allocated*sizeof(unsigned char));
            ptr=Data;

            WAProtectNMAP(Client, TRUE);

            while (WAGetFormNameEx(Client, Client->Temp, NULL, NULL, BUFSIZE)) {
                ptr=Data;
                ValueSize = Allocated-(ptr-Data);

                while (WAGetFormValue(Client, ptr, &ValueSize) != FORMFIELD_NEXT) {
                    ptr+=ValueSize;

                    if ((ptr - Data + 512) > Allocated) {
                        unsigned long offset = ptr - Data;

                        Allocated += BUFSIZE;
                        Data = realloc(Data, Allocated*sizeof(unsigned char));
                        ptr = Data + offset;
                    }
                    ValueSize = Allocated - (ptr-Data);
                }

                /*
                    Right now we have both the form name (in Client->Temp) and the form
                    data (in Data) so lets do sumfin.
                */

                if (WAQuickCmp(Client->Temp, "Name")) {
                    if (WAFindUserByEmail(Session, NMSession, Data, Session->CurrentObject, Session->CurrentClass)) {
                        Found = TRUE;
                    }

                    MDBFreeValues(Session->V);

                    if (Found && !WAQuickCmp(Session->CurrentClass, C_USER)) {
                        Found = FALSE;
                    }

                    if (Found) {
                        MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);
                        Session->LastError = WAERR_SUCCESS;
                    } else {
                        Session->CurrentObject[0] = '\0';
                        Session->LastError = WAERR_COULD_NOT_FIND;
                    }
                }

                WAProtectNMAP(Client, FALSE);

                if (Data) {
                    free(Data);
                }
                if (Found) {
                    WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);
                } else {
                    WAHandleTemplate(Client, Session, URL->Argument[1], TRUE);
                }

                return(TRUE);
            }

            break;
        }

        case REQUEST_RULE_EDIT: {
            unsigned long        ID;
            unsigned long        i;

            WAURLExtraDecode(Client, Session->CurrentObject, sizeof(Session->CurrentObject));
            MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);

            Session->LastError = WAERR_UNKNOWN;

            MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (strlen(Session->V->Value[i]) > 8) {
                    unsigned char        Temp = Session->V->Value[i][8];

                    Session->V->Value[i][8] = '\0';
                    ID = strtol(Session->V->Value[i], NULL, 16);
                    Session->V->Value[i][8] = Temp;

                    if (ID == URL->Argument[1]) {
                        /*
                            We found the matching ID, so lets store
                            it so when the next page comes up we can
                            find it again.
                        */
                        NMSession->RuleID = ID;
                        MDBFreeValues(Session->V);
                        Session->LastError = WAERR_SUCCESS;
                    }
                }
            }
            MDBFreeValues(Session->V);

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
            break;
        }

        case REQUEST_RULE_DELETE: {
            unsigned long        ID;
            unsigned long        i;

            WAURLExtraDecode(Client, Session->CurrentObject, sizeof(Session->CurrentObject));
            MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);

            Session->LastError = WAERR_OBJECT_NOT_DELETED;

            MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (strlen(Session->V->Value[i]) > 8) {
                    unsigned char        Temp = Session->V->Value[i][8];

                    Session->V->Value[i][8] = '\0';
                    ID = strtol(Session->V->Value[i], NULL, 16);
                    Session->V->Value[i][8] = Temp;

                    if (ID == URL->Argument[1]) {
                        /*
                            We found the matching ID, so lets pull it
                            out of the value struct and write it back.
                        */
                        MDBFreeValue(i, Session->V);
                        MDBWrite(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
                        MDBFreeValues(Session->V);
                        Session->LastError = WAERR_SUCCESS;
                    }
                }
            }
            MDBFreeValues(Session->V);

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
            break;
        }

        case REQUEST_RULE_TOGGLE_STATE: {
            unsigned long        ID;
            unsigned long        i;

            WAURLExtraDecode(Client, Session->CurrentObject, sizeof(Session->CurrentObject));
            MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);

            Session->LastError = WAERR_OBJECT_NOT_DELETED;

            MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (strlen(Session->V->Value[i]) > 8) {
                    unsigned char        Temp = Session->V->Value[i][8];

                    Session->V->Value[i][8] = '\0';
                    ID = strtol(Session->V->Value[i], NULL, 16);
                    Session->V->Value[i][8] = Temp;

                    if (ID == URL->Argument[1]) {
                        /*
                            We found the matching ID, so lets toggle
                            the active/non active state and write it
                            back.
                        */
                        if (Session->V->Value[i][8] == RULE_ACTIVE) {
                            Session->V->Value[i][8] = RULE_INACTIVE;
                        } else {
                            Session->V->Value[i][8] = RULE_ACTIVE;
                        }

                        MDBWrite(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
                        MDBFreeValues(Session->V);
                        Session->LastError = WAERR_SUCCESS;
                    }
                }
            }
            MDBFreeValues(Session->V);

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
            break;
        }

        case REQUEST_RULE_UP: {
            unsigned long        ID;
            unsigned long        i;
            unsigned char        TempID[8];

            /* Swap with the rule above it, if there is one */

            WAURLExtraDecode(Client, Session->CurrentObject, sizeof(Session->CurrentObject));
            MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);

            MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
            qsort(Session->V->Value, Session->V->Used, sizeof(unsigned char*), CompareRuleID);

            /* Start at 1 because the first can't be moved up */
            for (i = 1; i < Session->V->Used; i++) {
                if (strlen(Session->V->Value[i]) > 8) {
                    unsigned char        Temp = Session->V->Value[i][8];

                    Session->V->Value[i][8] = '\0';
                    ID = strtol(Session->V->Value[i], NULL, 16);
                    Session->V->Value[i][8] = Temp;

                    if (ID == URL->Argument[1]) {
                        /*
                            We found the matching ID
                        */
                        if (i >= 1) {
                            /* Get the ID of the rule before it, then swap them */
                            memcpy(TempID, Session->V->Value[i - 1], 8);
                            memcpy(Session->V->Value[i - 1], Session->V->Value[i], 8);
                            memcpy(Session->V->Value[i], TempID, 8);
                        }

                        MDBWrite(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
                        MDBFreeValues(Session->V);
                        Session->LastError = WAERR_SUCCESS;
                    }
                }
            }
            MDBFreeValues(Session->V);

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
            break;
        }

        case REQUEST_RULE_DOWN: {
            unsigned long        ID;
            unsigned long        i;
            unsigned char        TempID[8];

            /* Swap with the rule above it, if there is one */

            WAURLExtraDecode(Client, Session->CurrentObject, sizeof(Session->CurrentObject));
            MDBGetObjectDetails(Session->CurrentObject, Session->CurrentClass, NULL, NULL, Session->V);

            if (MDBRead(Session->CurrentObject, MSGSRV_A_RULE, Session->V)) {
                qsort(Session->V->Value, Session->V->Used, sizeof(unsigned char*), CompareRuleID);

                for (i = 0; i < Session->V->Used; i++) {
                    if (strlen(Session->V->Value[i]) > 8) {
                        unsigned char        Temp = Session->V->Value[i][8];

                        Session->V->Value[i][8] = '\0';
                        ID = strtol(Session->V->Value[i], NULL, 16);
                        Session->V->Value[i][8] = Temp;

                        if (ID == URL->Argument[1]) {
                            /*
                                We found the matching ID
                            */
                            if (i < Session->V->Used - 1) {
                                /* Get the ID of the rule before it, then swap them */
                                memcpy(TempID, Session->V->Value[i + 1], 8);
                                memcpy(Session->V->Value[i + 1], Session->V->Value[i], 8);
                                memcpy(Session->V->Value[i], TempID, 8);
                            }

                            MDBWrite(Session->CurrentObject, MSGSRV_A_RULE, Session->V);
                            MDBFreeValues(Session->V);
                            Session->LastError = WAERR_SUCCESS;
                        }
                    }
                }
                MDBFreeValues(Session->V);
            }

            WAHandleTemplate(Client, Session, URL->Argument[0], TRUE);

            return(TRUE);
            break;
        }

        default:
            break;
    }

    return(FALSE);
}

BOOL
WANMAILSaveInetServicesFuncInit(SessionStruct *Session, void **ModuleData)
{
    NMSyslogSessionStruct    *NMSession;

    NMSession = malloc(sizeof(NMSyslogSessionStruct));
    if (!NMSession) {
        XplConsolePrintf("WANMail Module out of memory!\n");
        return(FALSE);
    }

    memset(NMSession, 0, sizeof(NMSyslogSessionStruct));
    *ModuleData = (void *)NMSession;

    return(TRUE);
}

BOOL
WANMAILSaveInetServicesFuncDestroy(SessionStruct *Session, void *ModuleData)
{
    NMSyslogSessionStruct    *NMSession = (NMSyslogSessionStruct *)ModuleData;

    if (NMSession) {
        free(NMSession);
    }

    return(TRUE);
}

#ifndef LOG_LAST
#define    LOG_LAST            (LOG_LOCAL7+1)
#endif

/* Save handling for antivirus agent */
unsigned long
WANMAILSaveInetServicesFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    NMSyslogSessionStruct    *NMSession = (NMSyslogSessionStruct *)ModuleData;

    if (Name && Value) {
        switch (toupper(Name[0])) {
            case 'L':    {                            /* Level */
                unsigned long            NewLevel = atol(Value);

                if (NewLevel > NMSession->Level) {
                    NMSession->Level = NewLevel;
                }
                break;
            }

            case 'F':    {                            /* File */
                if (Value[0] != '\001') {
                    HulaStrNCpy(NMSession->FileName, Value, sizeof(NMSession->FileName));
                }
                break;
            }

            case 'E':    {                            /* Enabled */
                if (Value[0] == '1') {
                    NMSession->Type = 2;
                } else {
                    NMSession->Type = 3;
                }
                break;
            }
        }
    } else {
        unsigned long        i;
        unsigned char        Buffer[64];

        /* We have all values, so its time to write */

        for (i = 0; i < LOG_LAST; i++) {
            if (i != 1) {
                snprintf(Buffer, sizeof(Buffer), "%lu 0 0 -", i);
            } else {
                snprintf(Buffer, sizeof(Buffer), "1 %lu %lu - %s", NMSession->Level, NMSession->Type, NMSession->FileName);
            }
            MDBAddValue(Buffer, Session->V);
        }

        MDBWrite(DN, SYSLOG_A_CONFIG, Session->V);
        MDBFreeValues(Session->V);

        /* Reset the values in NMSession, in case we write this again */
        NMSession->Level = 0;
        NMSession->Type = 0;
        NMSession->FileName[0] = '\0';
    }
    return(0);
}

/* Save handling for antivirus agent */
unsigned long
WANMAILSaveComplexBitfieldsFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    if (!Name || !Value) {
        return(0);
    }

    switch (toupper(Name[0])) {
        case 'E':    {                            /* Anti Virus Engine */
            unsigned char            Answer[BUFSIZE+1];
            unsigned long            Config = 0;
            unsigned long            i;

            MDBRead(DN, MSGSRV_A_CONFIGURATION, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (WAQuickNCmp(Session->V->Value[i], "Settings: ", 10)) {
                    Config = atol(Session->V->Value[i] + 10);
                    MDBFreeValue(i, Session->V);

                    continue;
                }
            }
            /* Turn off all three, then we just turn on the one we want */
            Config &= ~AV_USE_CA;
            Config &= ~AV_USE_MCAFEE;
            Config &= ~AV_USE_SYMANTEC;
            Config &= ~AV_USE_COMMANDAV;
            Config &= ~AV_USE_CLAMAV;

            switch (toupper(Value[0])) {
                default:                                                        /* Make sure at least one is set */
                case 'C': {
                    if (toupper(Value[1]) == 'A') {
                        Config |= AV_USE_CA;                                /* CA */
                    } else if (toupper(Value[1]) == 'L') { /* ClamAV */
			Config |= AV_USE_CLAMAV;
		    } else {
                        Config |= AV_USE_COMMANDAV;                    /* Command*/
                    }
                    break;
                }
                case 'M':    Config |= AV_USE_MCAFEE;    break;    /* Mcafee */
                case 'S':    Config |= AV_USE_SYMANTEC;    break;    /* Symantic */
            }

            snprintf(Answer, sizeof(Answer), "Settings: %lu", Config);
            MDBAddValue(Answer, Session->V);
            MDBWrite(DN, MSGSRV_A_CONFIGURATION, Session->V);

            break;
        }

        case 'R':                                /* list review */
        case 'P':                                /* list postings */
        case 'S':    {                            /* list subscriptions */
            unsigned char            Answer[BUFSIZE+1];
            unsigned long            Config = 0;
            unsigned long            i;

            if (MDBRead(DN, MSGSRV_A_LIST_CONFIGURATION, Session->V)) {
                Config = atol(Session->V->Value[0]);
                MDBFreeValues(Session->V);
            }

            /* Turn off all three, then we just turn on the one we want */
            if (toupper(Name[0] == 'S')) {            /* Subscriptions */
                Config &= ~LIST_CONFIG_SUBSCRIBE_CLOSED;
                Config &= ~LIST_CONFIG_SUBSCRIBE_OWNER;
            } else if (toupper(Name[0] == 'R')) {    /* Review */
                Config &= ~LIST_CONFIG_REVIEW_PRIVATE;
                Config &= ~LIST_CONFIG_REVIEW_OWNER;
            } else if (toupper(Name[0] == 'P')) {    /* Postings */
                Config &= ~LIST_CONFIG_MODERATED;
                Config &= ~LIST_CONFIG_OPEN_LIST;
                Config &= ~LIST_CONFIG_ANNOUNCEMENTS;
            }

            if (Value[0] >= '0' && Value[0] <= '9') {
                i = atol(Value);
                Config |= (1 << i);
            }

            snprintf(Answer, sizeof(Answer), "%lu", Config);
            MDBAddValue(Answer, Session->V);
            MDBWrite(DN, MSGSRV_A_LIST_CONFIGURATION, Session->V);

            break;
        }

        case 'M':    {                            /* ModWeb Attachment Create Encoding */
            unsigned char            Answer[BUFSIZE+1];
            unsigned long            Config = 0;
            unsigned long            i;

            MDBRead(DN, MSGSRV_A_CONFIGURATION, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (WAQuickNCmp(Session->V->Value[i], "ANESend=", 8)) {
                    Config = atol(Session->V->Value[i] + 8);
                    MDBFreeValue(i, Session->V);

                    continue;
                }
            }
        
            switch (toupper(Value[0])) {
                default:                                                        /* Make sure at least one is set */
                case '2': {
                    if (toupper(Value[1]) == '2') {
                        Config = 2;                                        /* 2231 */
                    } else {
                        Config = 8;                                        /* 2047*/
                    }
                    break;
                }
                case '7':    Config = 1;    break;                    /* US-ASCII */
                case 'R':    Config = 4;    break;                    /* RAW */
            }

            snprintf(Answer, sizeof(Answer), "ANESend=%lu", Config);
            MDBAddValue(Answer, Session->V);
            MDBWrite(DN, MSGSRV_A_CONFIGURATION, Session->V);

            break;
        }

        case 'N':    {                            /* ModWeb Attachment Download Encoding */
            unsigned char            Answer[BUFSIZE+1];
            unsigned long            Config = 0;
            unsigned long            i;

            MDBRead(DN, MSGSRV_A_CONFIGURATION, Session->V);
            for (i = 0; i < Session->V->Used; i++) {
                if (WAQuickNCmp(Session->V->Value[i], "ANERecv=", 8)) {
                    Config = atol(Session->V->Value[i] + 8);
                    MDBFreeValue(i, Session->V);

                    continue;
                }
            }
        
            switch (toupper(Value[0])) {
                default:                                                        /* Make sure at least one is set */
                case '2': {
                    if (toupper(Value[1]) == '2') {
                        Config = 2;                                        /* 2231 */
                    } else {
                        Config = 8;                                        /* 2047*/
                    }
                    break;
                }
                case '7':    Config = 1;    break;                    /* US-ASCII */
                case 'R':    Config = 4;    break;                    /* RAW */
            }

            snprintf(Answer, sizeof(Answer), "ANERecv=%lu", Config);
            MDBAddValue(Answer, Session->V);
            MDBWrite(DN, MSGSRV_A_CONFIGURATION, Session->V);

            break;
        }
    }

    MDBFreeValues(Session->V);
    return(0);
}

/* Save handling for NMAP agent */
unsigned long
WANMAILSaveNMAPFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    if (!Name || !Value) {
        return(0);
    }

    switch (toupper(Name[0])) {
        case 'C':    {                            /* Contexts */
            unsigned char        *Begin;
            unsigned char        *End;
            unsigned char        Buffer[BUFSIZE+1];
            unsigned char        Buffer2[BUFSIZE+1];
            unsigned long        Changed = 0;

            Begin = Value;
            do {
                End = strchr(Begin, '\n');
                if (End) {
                    End[0] = '\0';
                    if (End[-1] == '\r') {
                        End[-1] = '\0';
                    }
                }

                WAX500toMDB(Begin, Buffer, Session->V);
                MDBAddValue(Buffer, Session->V);

                if (End) {
                    Begin = End + 1;
                } else {
                    Begin = NULL;
                }
            } while (Begin);

            HulaStrNCpy(Buffer, Session->CurrentObject, sizeof(Buffer));
            Begin = strrchr(Buffer, '\\');
            if (Begin) {
                *Begin = '\0';
                MDBWriteDN(Buffer, MSGSRV_A_CONTEXT, Session->V);
            }
            MDBFreeValues(Session->V);

            /* Set Config Changes */
            MDBRead(Buffer, MSGSRV_A_CONFIG_CHANGED, Session->V);
            if (Session->V->Used > 0) {
                Changed = atol(Session->V->Value[0]);
            }
            Changed++;

            snprintf(Buffer2, sizeof(Buffer2), "%lu", Changed);
            MDBAdd(Buffer, MSGSRV_A_CONFIG_CHANGED, Buffer2, Session->V);
            MDBFreeValues(Session->V);

            /* Set Config Changes on Internet Server Object */
            Changed =0;

            /* Set Config Changes */
            MDBRead(MSGSRV_ROOT, MSGSRV_A_CONFIG_CHANGED, Session->V);
            if (Session->V->Used > 0) {
                Changed = atol(Session->V->Value[0]);
            }
            Changed++;
            MDBFreeValues(Session->V);
            snprintf(Buffer2, sizeof(Buffer2), "%lu", Changed);
            MDBAddValue(Buffer2, Session->V);
            MDBWrite(MSGSRV_ROOT, MSGSRV_A_CONFIG_CHANGED, Session->V);
            MDBFreeValues(Session->V);
            break;
        }

        case 'B':    {                            /* Bounce */
            if (strlen(Name) > 6) {
                unsigned char        Buffer[BUFSIZE+1];
                unsigned long        BounceEnabled;
                unsigned long        BounceInterval;
                unsigned long        BounceEntries;

                if (MDBRead(Session->CurrentObject, MSGSRV_A_RTS_ANTISPAM_CONFIG, Session->V)) {
                    sscanf(Session->V->Value[0], "Enabled:%lu Delay:%lu Threshhold:%lu", &BounceEnabled, &BounceInterval, &BounceEntries);
                    MDBFreeValues(Session->V);
                } else {
                    BounceEnabled    = 0;
                    BounceInterval    = 0;
                    BounceEntries    = 0;
                }

                switch(toupper(Name[6])) {
                    case 'C': {                    /* BounceControl */
                        BounceEnabled = atol(Value);

                        break;
                    }

                    case 'I': {                    /* BounceInterval */
                        BounceInterval = atol(Value);

                        break;
                    }

                    case 'E': {                    /* BounceEntries */
                        BounceEntries = atol(Value);

                        break;
                    }
                }

                snprintf(Buffer, sizeof(Buffer), "Enabled:%lu Delay:%lu Threshhold:%lu", BounceEnabled, BounceInterval, BounceEntries);
                MDBAddValue(Buffer, Session->V);
                MDBWrite(Session->CurrentObject, MSGSRV_A_RTS_ANTISPAM_CONFIG, Session->V);
            }
            break;
        }
    }
    return(0);
}

BOOL
WANMAILSaveAliasFuncInit(SessionStruct *Session, void **ModuleData)
{
    NMAliasSessionStruct    *NMSession;

    NMSession = malloc(sizeof(NMAliasSessionStruct));
    if (!NMSession ) {
        XplConsolePrintf("WANMail Module out of memory!\n");
        return(FALSE);
    }

    memset(NMSession, 0, sizeof(NMAliasSessionStruct));
    *ModuleData = (void *)NMSession;

    return(TRUE);
}

BOOL
WANMAILSaveAliasFuncDestroy(SessionStruct *Session, void *ModuleData)
{
    NMAliasSessionStruct    *NMSession = (NMAliasSessionStruct *)ModuleData;

    if (NMSession) {
        free(NMSession);
    }

    return(TRUE);
}

#define ALIAS_INBOUND           (1 << 0)
#define ALIAS_AUTOMATIC         (1 << 2)
#define ALIAS_FIRST_LAST        (1 << 3)
#define ALIAS_FLASTNAME         (1 << 4)
#define ALIAS_FIRST_D_LAST      (1 << 5)
#define ALIAS_FULL_DOT          (1 << 6)
#define ALIAS_FULL_UNDER        (1 << 7)
#define ALIAS_AUTO_POPULATE     (1 << 8)
#define POPULATE_DEFAULT        (1 << 9)
#define POPULATE_FIRST_LAST     (1 << 10)
#define POPULATE_FLASTNAME      (1 << 11)
#define POPULATE_FIRST_D_LAST   (1 << 12)
#define POPULATE_FULL_DOT       (1 << 13)
#define POPULATE_FULL_UNDER     (1 << 14)

/* Save handling for alias agent */
unsigned long
WANMAILSaveAliasFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    NMAliasSessionStruct        *NMSession = (NMAliasSessionStruct *)ModuleData;
    unsigned char                Buffer[BUFSIZE+1];

    if (Name && Value) {
        unsigned long        Bit;

        if (WAQuickCmp(Name, "Global")) {
            unsigned char        *Begin;
            unsigned char        *End;
            unsigned char        RootDN[MDB_MAX_OBJECT_CHARS+1];
            unsigned char        *ptr;

            /* Global Aliases */

            if (Value[0] != '\001') {
                Begin = Value;
                do {
                    End = strchr(Begin, '\n');
                    if (End) {
                        End[0] = '\0';
                        if (End[-1] == '\r') {
                            End[-1] = '\0';
                        }
                    }

                    MDBAddValue(Begin, Session->V);

                    if (End) {
                        Begin = End + 1;
                    } else {
                        Begin = NULL;
                    }
                } while (Begin);
            }

            HulaStrNCpy(RootDN, Session->CurrentObject, sizeof(RootDN));
            ptr = strchr(RootDN + 1, '\\');
            if (ptr) {
                ptr = strchr(ptr + 1, '\\');
                if (ptr) {
                    *ptr = '\0';
                    MDBWrite(RootDN, MSGSRV_A_ALIAS, Session->V);
                    MDBFreeValues(Session->V);
                }
            }

        } else {
            Bit = atol(Value);
            NMSession->Alias |= (1 << Bit);
            NMSession->WriteAlias = TRUE;
        }
    } else if (NMSession->WriteAlias) {
        unsigned long        Value = 0;

        /* Alright we have all the data, time to save it */
        if (MDBRead(DN, MSGSRV_A_ALIAS_OPTIONS, Session->V)) {
            Value = atol(Session->V->Value[0]);
            MDBFreeValues(Session->V);
        }

        /* We need to figure out if we are saving the automatic attribute or the regular aliasing options */
        if (NMSession->Alias & POPULATE_DEFAULT || NMSession->Alias & POPULATE_FIRST_LAST || NMSession->Alias & POPULATE_FLASTNAME || NMSession->Alias & POPULATE_FIRST_D_LAST || NMSession->Alias & POPULATE_FULL_DOT || NMSession->Alias & POPULATE_FULL_UNDER) {
            Value &= ~(POPULATE_DEFAULT | POPULATE_FIRST_LAST | POPULATE_FLASTNAME | POPULATE_FIRST_D_LAST | POPULATE_FULL_DOT | POPULATE_FULL_UNDER);
            Value |= NMSession->Alias;
        } else {
            Value &= ~(ALIAS_FIRST_LAST | ALIAS_FLASTNAME | ALIAS_FIRST_D_LAST | ALIAS_FULL_DOT | ALIAS_FULL_UNDER);
            Value |= NMSession->Alias;
        }

        snprintf(Buffer, sizeof(Buffer), "%lu", Value);
        MDBAddValue(Buffer, Session->V);
        MDBWrite(DN, MSGSRV_A_ALIAS_OPTIONS, Session->V);
        MDBFreeValues(Session->V);
        NMSession->Alias = 0;
        NMSession->WriteAlias = 0;
    }
    return(0);
}

BOOL
WANMAILSaveFeaturesFuncInit(SessionStruct *Session, void **ModuleData)
{
    NMFeaturesSessionStruct    *NMSession;

    NMSession = malloc(sizeof(NMFeaturesSessionStruct));
    if (!NMSession ) {
        XplConsolePrintf("WANMail Module out of memory!\n");
        return(FALSE);
    }

    memset(NMSession, 0, sizeof(NMFeaturesSessionStruct));
    *ModuleData = (void *)NMSession;

    return(TRUE);
}

BOOL
WANMAILSaveFeaturesFuncDestroy(SessionStruct *Session, void *ModuleData)
{
    NMFeaturesSessionStruct    *NMSession = (NMFeaturesSessionStruct *)ModuleData;

    if (NMSession) {
        free(NMSession);
    }

    return(TRUE);
}

unsigned long
WANMAILSaveFeaturesFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    NMFeaturesSessionStruct        *NMSession = (NMFeaturesSessionStruct *)ModuleData;
    unsigned char                    *ptr = NULL;

    if (Name && Value) {
        if ((strlen(Name) < 3) || (toupper(Name[0]) != 'F') || (Name[1] != '_')) {
            /* We don't handle this one */

            return(0);
        }

        if (!NMSession->CacheStruct) {
            NMSession->CacheStruct = MDBShareContext(Session->V);

            if (MDBRead(DN, MSGSRV_A_FEATURE_SET, NMSession->CacheStruct) == 0) {
                /* Fill out the default values */
                MDBAddValue(DEFAULT_FEATURE_ROW_A, NMSession->CacheStruct);
                MDBAddValue(DEFAULT_FEATURE_ROW_B, NMSession->CacheStruct);
                MDBAddValue(DEFAULT_FEATURE_ROW_C, NMSession->CacheStruct);
                MDBAddValue(DEFAULT_FEATURE_ROW_D, NMSession->CacheStruct);
                MDBAddValue(DEFAULT_FEATURE_ROW_E, NMSession->CacheStruct);
            }
        }

        switch (atol(Name + 2)) {
            case AA_FEATURE_IMAP:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_IMAP);                            break;    }
            case AA_FEATURE_POP:                            {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_POP);                            break;    }
            case AA_FEATURE_ADDRESSBOOK:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_ADDRESSBOOK);                break;    }
            case AA_FEATURE_PROXY:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_PROXY);                        break;    }
            case AA_FEATURE_FORWARD:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_FORWARD);                        break;    }
            case AA_FEATURE_AUTOREPLY:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_AUTOREPLY);                    break;    }
            case AA_FEATURE_RULES:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_RULES);                        break;    }
            case AA_FEATURE_FINGER:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_FINGER);                        break;    }
            case AA_FEATURE_SMTP_SEND_COUNT_LIMIT:    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_SMTP_SEND_COUNT_LIMIT);    break;    }
            case AA_FEATURE_SMTP_SEND_SIZE_LIMIT:    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_SMTP_SEND_SIZE_LIMIT);    break;    }
            case AA_FEATURE_NMAP_QUOTA:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_NMAP_QUOTA);                    break;    }
            case AA_FEATURE_NMAP_STORE:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_NMAP_STORE);                    break;    }
            case AA_FEATURE_WEBMAIL:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_WEBMAIL);                        break;    }
            case AA_FEATURE_MODWEB:                        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MODWEB);                        break;    }
            case AA_FEATURE_MWMAIL_ADDRBOOK_P:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_P);        break;    }
            case AA_FEATURE_MWMAIL_ADDRBOOK_S:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_S);        break;    }
            case AA_FEATURE_MWMAIL_ADDRBOOK_G:        {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MWMAIL_ADDRBOOK_G);        break;    }
            case AA_FEATURE_MODWEB_WAP:                {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_MODWEB_WAP);                    break;    }
            case AA_FEATURE_CALAGENT:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_CALAGENT);                    break;    }
            case AA_FEATURE_CALENDAR:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_CALENDAR);                    break;    }
            case AA_FEATURE_ANTIVIRUS:                    {            ptr = FindFeature(NMSession->CacheStruct, FEATURE_ANTIVIRUS);                    break;    }
        }

        /* Alright we should have our pointer now */
        if (ptr) {
            *ptr = toupper(Value[0]);
        }
    } else {
        /* Alright its time to write */
        if (NMSession->CacheStruct) {
            MDBWrite(DN, MSGSRV_A_FEATURE_SET, NMSession->CacheStruct);

            MDBDestroyValueStruct(NMSession->CacheStruct);
            NMSession->CacheStruct = NULL;
        }
    }
    return(0);
}

/* Create a NMAP object */
unsigned long    
WANMAILNMAPCreateFunc(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *ObjectName, const unsigned char *ParentDN, MDBValueStruct *Name, MDBValueStruct *Value, void *ModuleData)
{
    unsigned char        ObjectBuffer[MDB_MAX_OBJECT_CHARS + 1];
    MDBValueStruct        *AttrName;
    MDBValueStruct        *AttrValue;
    MDBValueStruct        *Contexts;
    unsigned char        *Start;
    unsigned char        *End;
    unsigned long        RetVal = WAERR_SUCCESS;
    unsigned long        i;

    AttrName=MDBCreateValueStruct(Session->AuthHandle, ParentDN);
    AttrValue=MDBShareContext(AttrName);
    Contexts=MDBShareContext(AttrName);

    for (i=0; i<Name->Used; i++) {
        if (WAQuickCmp(Name->Value[i], "context")) {
            /* Contexts to write to the Messaging Server object */

            Start = Value->Value[i];
            do {
                End = strchr(Start, '\n');
                if (End) {
                    End[0] = '\0';
                    if (End[-1] == '\r') {
                        End[-1] = '\0';
                    }
                }

                WAX500toMDB(Start, ObjectBuffer, Session->V);
                MDBAddValue(ObjectBuffer, Contexts);

                if (End) {
                    Start = End + 1;
                } else {
                    Start = NULL;
                }
            } while (Start);

        } else if (WAQuickCmp(Name->Value[i], "store")) {
            /* Message store */

            Start = Value->Value[i];
            do {
                End = strchr(Start, '\n');
                if (End) {
                    End[0] = '\0';
                    if (End[-1] == '\r') {
                        End[-1] = '\0';
                    }
                }

                MDBAddStringAttribute(MSGSRV_A_MESSAGE_STORE, Start, AttrName, AttrValue);

                if (End) {
                    Start = End + 1;
                } else {
                    Start = NULL;
                }
            } while (Start);

        } else if (WAQuickCmp(Name->Value[i], "modulename")) {
            /* modulename */

            MDBAddStringAttribute(MSGSRV_A_MODULE_NAME, Value->Value[i], AttrName, AttrValue);
        } else if (WAQuickCmp(Name->Value[i], "moduleversion")) {
            /* moduleversion */

            MDBAddStringAttribute(MSGSRV_A_MODULE_VERSION, Value->Value[i], AttrName, AttrValue);
        }
    }

    if (!MDBCreateObject(ObjectName, ObjectClass, AttrName, AttrValue, AttrName)) {
        RetVal=WAERR_OBJECT_NOT_CREATED;
        Session->LastError = WAERR_OBJECT_NOT_CREATED;
    } else {
        /* Write the context list to the messaging server object */
        MDBWriteDN(ParentDN, MSGSRV_A_CONTEXT, Contexts);
    }

    MDBDestroyValueStruct(AttrValue);
    MDBDestroyValueStruct(AttrName);
    MDBDestroyValueStruct(Contexts);

    if (Session->LastError == WAERR_SUCCESS) {
        Session->LastError = WAERR_SHOW_SUCCESS;
    }    
    return(RetVal);
}

BOOL
WANMAILSaveRulesInit(SessionStruct *Session, void **ModuleData)
{
    RulesSessionStruct    *RulesSession;

    RulesSession = malloc(sizeof(RulesSessionStruct));
    if (!RulesSession ) {
        XplConsolePrintf("WANAudit Module out of memory!\n");
        return(FALSE);
    }
    memset(RulesSession, 0, sizeof(RulesSessionStruct));

    RulesSession->RuleNames = MDBShareContext(Session->V);
    RulesSession->RuleValues = MDBShareContext(Session->V);

    *ModuleData = (void *)RulesSession;

    return(TRUE);
}

BOOL
WANMAILSaveRulesDestroy(SessionStruct *Session, void *ModuleData)
{
    RulesSessionStruct    *RulesSession = (RulesSessionStruct *)ModuleData;

    if (RulesSession) {
        if (RulesSession->RuleNames) {
            MDBDestroyValueStruct(RulesSession->RuleNames);
        }

        if (RulesSession->RuleValues) {
            MDBDestroyValueStruct(RulesSession->RuleValues);
        }

        free(RulesSession);
    }

    return(TRUE);
}

unsigned long
WANMAILSaveRules(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *Name, unsigned char *Value, void *ModuleData)
{
    RulesSessionStruct        *RulesSession = (RulesSessionStruct *)ModuleData;

    if (Name && Value) {
        MDBValueStruct            *TempNames;
        MDBValueStruct            *TempValues;
        unsigned long            i;
        unsigned long            Order;

        if (WAQuickCmp(Name, "Ignore") || WAQuickCmp(Name, "IgnoreMe")) {
            /* The form has "control" elements that we don't want to act on */

            return(FALSE);
        }

        if (strlen(Name) > 2 && WAQuickCmp(Name + 2, "ID")) {
            RulesSession->ID = atol(Value);
            return(TRUE);
        }

        TempNames = MDBShareContext(Session->V);
        TempValues = MDBShareContext(Session->V);
        /*
            We need to keep track of each name value pair until we have them
            all, and we are sorting them here instead of later.  Chances are
            they will already be in order, so it won't make much of a
            difference, but we need to be sure.
        */
        Order = atol(Name);
        for (i = 0; i < RulesSession->RuleNames->Used; i++) {
            if (atol(RulesSession->RuleNames->Value[i]) > Order) {
                /* It goes here */

                while (i < RulesSession->RuleNames->Used) {
                    MDBAddValue(RulesSession->RuleNames->Value[i], TempNames);
                    MDBAddValue(RulesSession->RuleValues->Value[i], TempValues);

                    MDBFreeValue(i, RulesSession->RuleNames);
                    MDBFreeValue(i, RulesSession->RuleValues);
                }
            }
        }

        MDBAddValue(Name, RulesSession->RuleNames);
        MDBAddValue(Value, RulesSession->RuleValues);

        while (TempNames->Used > 0) {
            MDBAddValue(TempNames->Value[0], RulesSession->RuleNames);
            MDBAddValue(TempValues->Value[0], RulesSession->RuleValues);

            MDBFreeValue(0, TempNames);
            MDBFreeValue(0, TempValues);
        }

        MDBDestroyValueStruct(TempNames);
        MDBDestroyValueStruct(TempValues);
    } else {
        unsigned long        i;
        unsigned long        Len = 10;        /* 8 digit ID, active flag, terminator */
        unsigned long        Id;
        unsigned char        *Buffer;
        unsigned char        *ptr;

        /* All the elements needed to build a single condition */
        unsigned char        Condition = '\0';
        unsigned char        Action = '\0';
        unsigned char        Bool = '\0';
        unsigned char        *Arg1 = NULL;
        unsigned char        *Arg2 = NULL;
        BOOL                    FirstCondition = TRUE;
	int bufLen;
	int count;

        /*
            Ok, now we have all the data in RuleNames, and RuleValues.  We should have
            them sorted it by the ID number.  Each value in RuleNames should start
            with a number followed by a '-' which specifies the order.  We have no
            idea what order the browser sent them in, so we need to sort them.

            Next we need to piece together each portion (all parts of a portion
            begin with the same id) and then putting the rule together should be easy.

            Before doing any of that though we need a buffer to store our new rule in.
            Since the length is 3 digits the max length is 1000 chars.  Since there are
            two args per portion that can get REALLY big, so lets figure out just how
            big first
        */

        if (RulesSession->RuleNames->Used == 0) {
            return(FALSE);
        }

        for (i = 0; i < RulesSession->RuleNames->Used; i++) {
            unsigned char        *ptr;

            /* Figure out what part it is and add the proper length */
            ptr = strchr(RulesSession->RuleNames->Value[i], '-');
            if (ptr) {
                ptr++;

                if (WAQuickCmp(ptr, "FirstArgument") || WAQuickCmp(ptr, "SecondArgument")) {
                    Len += strlen(RulesSession->RuleValues->Value[i]);
                } else if (WAQuickCmp(ptr, "Condition") || WAQuickCmp(ptr, "Action")) {
                    Len += 10;    /* 1 for the condition or action, 1 for a possible bool, and 8 for the arg lenghts and terminators */
                } else {
                    /* Anything else will add a max of two chars, and its faster not to check */
                    Len += 2;
                }
            }
        }

        Buffer = malloc(Len);
        ptr = Buffer;
	bufLen = Len;

        if (!Buffer) {
            XplConsolePrintf("WANMail Module out of memory!\n");
            return(FALSE);
        }

        if (RulesSession->ID == 0) {
            RulesSession->ID = time(NULL);
        }
        count = snprintf(ptr, bufLen, "%08XA", RulesSession->ID);
	ptr += count;
	bufLen -= count;

        /* Alright, lets make a rule */
        Id = atol(RulesSession->RuleNames->Value[0]);
        for (i = 0; i < RulesSession->RuleNames->Used; i++) {
            unsigned long        NextId;
            unsigned char        *Name = strchr(RulesSession->RuleNames->Value[i], '-');
            unsigned char        *Value = RulesSession->RuleValues->Value[i];

            if (!Name) {
                return(FALSE);
            }
            Name++;    /* Find the start of the actual name */

            if (i + 1 < RulesSession->RuleNames->Used) {
                NextId = atol(RulesSession->RuleNames->Value[i + 1]);
            } else {
                /* Just make sure they aren't the same */
                NextId = Id + 1;
            }

            /* What are we looking at? */

            if (WAQuickCmp(Name, "Condition")) {

                if ((*Value >= 'A' && *Value <= 'Z') || (*Value >= 'a' && *Value <= 'z')) {
                    Condition = Value[0];
                }
            } else if (WAQuickCmp(Name, "Action")) {

                if ((*Value >= 'A' && *Value <= 'Z') || (*Value >= 'a' && *Value <= 'z')) {
                    Action = Value[0];
                }
            } else if (WAQuickCmp(Name, "Bool")) {
                if (*Value == RULE_ACT_BOOL_AND || *Value == RULE_ACT_BOOL_OR || *Value == RULE_ACT_BOOL_NOT) {
                    Bool = *Value;
                }
            } else if (WAQuickCmp(Name, "FirstArgument")) {

                Arg1 = Value;
            } else if (WAQuickCmp(Name, "SecondArgument")) {

                Arg2 = Value;
            }

            if (Id != NextId) {
                /* We are done with this condition, make sure its ok and print it */
                Id = NextId;

                if (Condition || Action) {
                    unsigned long        ArgNumber;

                    if (Condition) {
                        if (FirstCondition) {
                            FirstCondition = FALSE;
                        } else if (Bool) {
                            *ptr = Bool;
                            ptr++;
                        }

                        *ptr = Condition;
                        ptr++;
                    } else if (Action) {
                        *ptr = Action;
                        ptr++;
                    }

                    for (ArgNumber = 1; ArgNumber <=2; ArgNumber++) {
                        unsigned char        *Arg = (ArgNumber == 1) ? Arg1 : Arg2;
                        unsigned long        Length = 0;

                        /*
                            Text:

                            The string must be proceeded with a 3 digit length, and no more
                            than 255 chars may be stored.  Termination is not required.
                        */
                        if (Arg && *Arg != '\001') {
                            Length = strlen(Arg);
                            if (Length > 999) {
                                Length = 999;
                                Arg[999] = '\0';
                            }
                            count = snprintf(ptr, bufLen, "%03lu%sZ", Length, Arg);
			    ptr += count;
			    bufLen -= count;
                        } else {
                            count = snprintf(ptr, bufLen, "000Z");
			    ptr += count;
			    bufLen -= count;
                        }
                    }
                }

                /* Clean up for the next time through */
                Condition = '\0';
                Action = '\0';
                Bool = '\0';
                Arg1 = NULL;
                Arg2 = NULL;
            }
        }

        /* Clean up and write */

        MDBFreeValues(RulesSession->RuleNames);
        MDBFreeValues(RulesSession->RuleValues);
        MDBFreeValues(Session->V);

        MDBRead(DN, MSGSRV_A_RULE, Session->V);
        /*
            If there is already a rule with that ID we need to remove it.
        */
        for (i = 0; i < Session->V->Used; i++) {
            unsigned long        ID;
            unsigned char        Temp = Session->V->Value[i][8];

            Session->V->Value[i][8] = '\0';
            ID = strtol(Session->V->Value[i], NULL, 16);
            Session->V->Value[i][8] = Temp;

            if (ID == RulesSession->ID) {
                MDBFreeValue(i, Session->V);
                i = Session->V->Used;
            }
        }

        MDBAddValue(Buffer, Session->V);
        MDBWrite(DN, MSGSRV_A_RULE, Session->V);
        MDBFreeValues(Session->V);

        RulesSession->ID = 0;

        free(Buffer);
    }
    return(FALSE);
}

unsigned long
WANMAILPreventRename(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, unsigned char *NewName, void *ModuleData)
{
    return(WAERR_NOT_ALLOWED);
}

unsigned long
WANMAILNMAPDelete(SessionStruct *Session, const unsigned char *ObjectClass, const unsigned char *DN, void *ModuleData)
{
    unsigned char        buffer[BUFSIZE+1];
    unsigned char        *end;

    /*
        We just need to remove the context list from the Messaging Server object.  It is
        a mandatory attribute, so we default it to "Internet Services".
    */

    HulaStrNCpy(buffer, DN, sizeof(buffer));
    end = strrchr(buffer, '\\');
    if (end) {
        *end = '\0';

        end = strrchr(buffer, '\\');
        if (end) {
            *end = '\0';
            MDBAddValue(buffer, Session->V);
            *end = '\\';
        }

        MDBWriteDN(buffer, MSGSRV_A_CONTEXT, Session->V);
        MDBFreeValues(Session->V);
    }

    return(WAERR_SHOW_SUCCESS);
}

/************************************************************************
    Now comes the "stock" stuff any modweb module needs
*************************************************************************/

EXPORT BOOL
LIBWANMAILInit(WAAPIArg)
{
    ModuleRegisterStruct    Register;

    WAAPISet;

    /* Register HandleURL and HandleTemplate */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                                = MODULE_TEMPLATE;
    Register.Module.Template.InitSession        = WANMAILInitSession;
    Register.Module.Template.DestroySession    = WANMAILDestroySession;
    Register.Module.Template.HandleURL            = WANMAILHandleURL;
    Register.Module.Template.HandleTemplate    = WANMAILHandleTemplate;
    Register.Module.Template.TokenRangeStart    = WANMAIL_TOKEN_START;
    Register.Module.Template.TokenRangeEnd        = WANMAIL_TOKEN_END;

    WARegisterModule(&Register);


    /* Create NMAP object */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_OBJECT;
    Register.Module.Object.ObjectClass        = MSGSRV_C_NMAP;
    Register.Module.Object.Flags                = WAOBJ_PRIMARY_HANDLER;
    Register.Module.Object.CreateFunction    = WANMAILNMAPCreateFunc;

    WARegisterModule(&Register);

    /* Handle custom save elements for AntiVirus Agent */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_ANTIVIRUS;
    Register.Module.Save.SaveFunction        = WANMAILSaveComplexBitfieldsFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_LIST;
    Register.Module.Save.SaveFunction        = WANMAILSaveComplexBitfieldsFunc;

    WARegisterModule(&Register);

    /* Handle custom save elements for ModWeb Agent */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_MODWEB;
    Register.Module.Save.SaveFunction        = WANMAILSaveComplexBitfieldsFunc;

    WARegisterModule(&Register);

    /* Handle custom save elements for ModWeb Mail Agent */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_MWMAIL_MODULE;
    Register.Module.Save.SaveFunction        = WANMAILSaveComplexBitfieldsFunc;

    WARegisterModule(&Register);

    /* Handle custom save elements for Internet Services Agent */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_ROOT;
    Register.Module.Save.InitSession            = WANMAILSaveInetServicesFuncInit;
    Register.Module.Save.DestroySession        = WANMAILSaveInetServicesFuncDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveInetServicesFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_ALIAS;
    Register.Module.Save.InitSession            = WANMAILSaveAliasFuncInit;
    Register.Module.Save.DestroySession        = WANMAILSaveAliasFuncDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveAliasFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_NMAP;
    Register.Module.Save.SaveFunction        = WANMAILSaveNMAPFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = C_USER;
    Register.Module.Save.InitSession            = WANMAILSaveFeaturesFuncInit;
    Register.Module.Save.DestroySession        = WANMAILSaveFeaturesFuncDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveFeaturesFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_PARENTOBJECT;
    Register.Module.Save.InitSession            = WANMAILSaveFeaturesFuncInit;
    Register.Module.Save.DestroySession        = WANMAILSaveFeaturesFuncDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveFeaturesFunc;

    WARegisterModule(&Register);

    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = MSGSRV_C_MWMAIL_MODULE;
    Register.Module.Save.InitSession            = WANMAILSaveFeaturesFuncInit;
    Register.Module.Save.DestroySession        = WANMAILSaveFeaturesFuncDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveFeaturesFunc;

    WARegisterModule(&Register);

    /* Handle custom save rules */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_SAVE;
    Register.Module.Save.ObjectClass            = C_USER;
    Register.Module.Save.InitSession            = WANMAILSaveRulesInit;
    Register.Module.Save.DestroySession        = WANMAILSaveRulesDestroy;
    Register.Module.Save.SaveFunction        = WANMAILSaveRules;

    WARegisterModule(&Register);

    Register.Module.Save.ObjectClass            = MSGSRV_C_RESOURCE;
    WARegisterModule(&Register);

    /* Prevent Rename and Deletion of NetMail agents */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_RENAME;
    Register.Module.Rename.RenameFunction    = WANMAILPreventRename;

    Register.Module.Rename.ObjectClass        = MSGSRV_C_ROOT;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_ADDRESSBOOK;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_ALIAS;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_ANTISPAM;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_AUTOREPLY;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_FINGER;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_IMAP;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_NMAP;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_POP;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_PROXY;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_SMTP;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_WEBMAIL;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_SIGNUP;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_CONNMGR;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_RULESRV;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_LISTAGENT;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MODWEB;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MWMAIL_MODULE;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MWCAL_MODULE;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MWPREF_MODULE;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MWTOM_MODULE;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_CALAGENT;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_ANTIVIRUS;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = MSGSRV_C_MWSIGNUP_MODULE;
    WARegisterModule(&Register);
    Register.Module.Rename.ObjectClass        = PLUSPACK_C;
    WARegisterModule(&Register);

    /* Cleanup Messaging server when NMAP is deleted */
    memset(&Register, 0, sizeof(ModuleRegisterStruct));
    Register.ModuleType                            = MODULE_DELETE;
    Register.Module.Delete.DeleteFunction    = WANMAILNMAPDelete;

    Register.Module.Delete.ObjectClass        = MSGSRV_C_NMAP;
    WARegisterModule(&Register);


    WANMail.unload = FALSE;

    return(TRUE);
}

EXPORT BOOL
LIBWANMAILShutdown(void)
{
    WANMail.unload = TRUE;

    return(TRUE);
}

/************************************************************************
    Below are all the things that make loading and unloading possible :-)
*************************************************************************/

#if defined(NETWARE) || defined(LIBC)
int
RequestUnload(void)
{
    if (!WANMail.unload) {
        int    OldTGid;

        OldTGid=XplSetThreadGroupID(WANMail.id.group);

        XplConsolePrintf("\rThis NLM will automatically be unloaded by the thread that loaded it.\n");
        XplConsolePrintf("\rIt does not allow manual unloading.\n");
        SetCurrentScreen(CreateScreen("System Console", 0));
        XplUngetCh('n');

        XplSetThreadGroupID(OldTGid);

        return(1);
    }

    return(0);
}


void
WANMailSigHandler(int Signal)
{
    int    OldTGid;

    OldTGid=XplSetThreadGroupID(WANMail.id.group);

    XplSignalLocalSemaphore(WANMail.sem.shutdown);    /* The signal will release main() */
    XplWaitOnLocalSemaphore(WANMail.sem.shutdown);    /* The wait will wait until main() is gone */

    /* Do any required cleanup */
    XplCloseLocalSemaphore(WANMail.sem.shutdown);
    XplSetThreadGroupID(OldTGid);
}


int
main(int argc, char *argv[])
{
    WANMail.id.group=XplGetThreadGroupID();

    signal(SIGTERM, WANMailSigHandler);

    MDBInit();

    XplOpenLocalSemaphore(WANMail.sem.shutdown, 0);
    XplWaitOnLocalSemaphore(WANMail.sem.shutdown);
    XplSignalLocalSemaphore(WANMail.sem.shutdown);
    return(0);
}
#endif
