/* This file is part of MemMXtest
 * Copyright (C) 1999, 2000  J.A. Bezemer
 * This program is licensed under the GNU General Public License,
 * see the top-level README file for details.
 */

/* Prepare the address updates */

#include "defines.h"
#include "mtest.h"
#include "tests.h"


unsigned long bitpos32[32] =
{
  0x00000001, 0x00000002, 0x00000004, 0x00000008,
  0x00000010, 0x00000020, 0x00000040, 0x00000080,
  0x00000100, 0x00000200, 0x00000400, 0x00000800,
  0x00001000, 0x00002000, 0x00004000, 0x00008000,
  0x00010000, 0x00020000, 0x00040000, 0x00080000,
  0x00100000, 0x00200000, 0x00400000, 0x00800000,
  0x01000000, 0x02000000, 0x04000000, 0x08000000,
  0x10000000, 0x20000000, 0x40000000, 0x80000000
};


/* From the _ml: */
extern unsigned char update_adr_start, update_adr;
extern unsigned char update_adr_up_startB, update_adr_up_startE;
extern unsigned char update_adr_upB1, update_adr_upE1;
extern unsigned char update_adr_upB2, update_adr_upE2;
extern unsigned char update_adr_upB3, update_adr_upE3;
extern unsigned char update_adr_dn_startB, update_adr_dn_startE;
extern unsigned char update_adr_dnB1, update_adr_dnE1;
extern unsigned char update_adr_dnB2, update_adr_dnE2;
extern unsigned char update_adr_dnB3, update_adr_dnE3;


unsigned long
bitrange32 (int start, int end)
/* Returns a bit mask with 1's from position `start' to `end', both
   inclusive */
{
  unsigned long result = 0;
  int i;

  for (i = start; i <= end; i++)
    result |= bitpos32[i];

  return result;
}


unsigned long
proc2mem (unsigned long procadr, adrbitmap_t adrbitmap)
{
  unsigned long memadr = 0;
  char passeddefined = 0;
  int currentmax = -1;
  int i;

  for (i = 0; i < 32; i++)
    {
      if (adrbitmap[i] >= 32)
	passeddefined = 1;
      if (!passeddefined)
	{
	  if ((procadr & bitpos32[adrbitmap[i]]) != 0)
	    memadr |= bitpos32[i];
	  if (adrbitmap[i] > currentmax)
	    currentmax = adrbitmap[i];
	}
      else
	{
	  currentmax++;
	  if (currentmax < 32 && (procadr & bitpos32[currentmax]) != 0)
	    memadr |= bitpos32[i];
	}
    }

  return memadr;
}


unsigned long
mem2proc (unsigned long memadr, adrbitmap_t adrbitmap)
{
  unsigned long procadr = 0;
  char passeddefined = 0;
  int currentmax = -1;
  int i;

  for (i = 0; i < 32; i++)
    {
      if (adrbitmap[i] >= 32)
	passeddefined = 1;
      if (!passeddefined)
	{
	  if ((memadr & bitpos32[i]) != 0)
	    procadr |= bitpos32[adrbitmap[i]];
	  if (adrbitmap[i] > currentmax)
	    currentmax = adrbitmap[i];
	}
      else
	{
	  currentmax++;
	  if (currentmax < 32 && (memadr & bitpos32[i]) != 0)
	    procadr |= bitpos32[currentmax];
	}
    }

  return procadr;
}


/* Global vars to prevent stack overflow */
unsigned long autonominc, memstartinc, procautonomrange, memstartadr, memendadr,
  selectmask;

#define autonomdec    autonominc	/* Are used mutually exclusive */
#define memstartdec   memstartinc


int
prepare_update_adr_up (adrbitmap_t adrbitmap, unsigned long procstartadr,
		       unsigned long procendadr, char allowranges,
		       adrcode_t adrcode)
/* adrcode is destination; return value is resulting length (to pass to
   _ml routines) */
/* allowranges=1: map addresses much faster by shifting ranges
   instead of individual bits */
/* adrbitmap example (fast-x as in _ml.S):

   .     index:  1F           06 05 04 03 02 01 00      <- mem-adr   REVERSE
   .     value:  FF FF ... FF FF 02 01 00 04 03 05      <- proc-adr    INDEX!

   The first FF from the right (or anything >= 0x20(=32)) is the end of the
   sequence; if possible they will be replaced by numbers starting from
   the present max. + 1 and increasing. This may be a range shift
   depending on allowranges.
   Do not specify the max. value in the right-most position, as in
   FF ... FF 03 01 02, as this will be split in two shifts. Make it
   FF ... FF FF 01 02 instead.
 */
{
  int i, acp, b1, b2;
  signed char shiftcount, currentmax;
  signed char passeddefined = 0;
  unsigned char meminseqend;

  /* Detect autonomous range, which is defined as the largest series
     of in-sequence bits from the right.
     Test for 30 instead of 31 because we need the 32th bit... */
  meminseqend = 0;
  i = adrbitmap[0];
  while (meminseqend < 30 && i < 30 && adrbitmap[meminseqend + 1] == (i + 1))
    {
      meminseqend++;
      i++;
    }

  /* Now meminseqend is the number of the last mem-adr bit that is in
     the autonomous range. */

  /* Set the various variables */
  autonominc = bitpos32[adrbitmap[0]];
  memstartinc = bitpos32[meminseqend + 1];
  procautonomrange = bitrange32 (adrbitmap[0], adrbitmap[meminseqend]);

  memstartadr = proc2mem (procstartadr, adrbitmap)
    & bitrange32 (meminseqend + 1, 31);
  memendadr = proc2mem (procendadr, adrbitmap);

  /* "Start code": */

  /* memcpy(adrcode, &update_adr_up_startB, 32); 
     Well, yes, it _doesn't_ use the C library ("macro-like") but
     I don't use it because there's no guarantee... */

  for (i = 0; i < 32; i++)
    *(adrcode + i) = *(&update_adr_up_startB + i);	/* Copy... */

  *((unsigned long *) (adrcode + 0x1)) = memstartadr;	/* & Mutate! */
  *((unsigned long *) (adrcode + 0x6)) = autonominc;
  *((unsigned long *) (adrcode + 0x10)) = procstartadr;

  acp = 32;			/* Adrcode pointer points to next
				   address-to-be-filled. */

  /* "Header": */

  for (i = 0; i < &update_adr_upE1 - &update_adr_upB1; i++)
    *(adrcode + acp + i) = *(&update_adr_upB1 + i);	/* Copy... */

  *((unsigned long *) (adrcode + acp + 0x2)) = memstartinc;	/* & Mutate! */

  acp += i;

  /* "Repeating part": */

  currentmax = -1;

  for (b1 = 0; b1 < 32; b1++)	/* b1 is start of range */
    {				/* b2 is end of range (poss. =b1) */
      if (adrbitmap[b1] >= 32)
	passeddefined = 1;

      if (!passeddefined)
	{
	  if (adrbitmap[b1] > currentmax)
	    currentmax = adrbitmap[b1];

	  b2 = b1;
	  if (allowranges)
	    while (b2 < 31 && adrbitmap[b2 + 1] == adrbitmap[b2] + 1
		   && adrbitmap[b2 + 1] < 32)
	      {
		b2++;
		if (adrbitmap[b2] > currentmax)
		  currentmax = adrbitmap[b2];
	      }

	  selectmask = bitrange32 (b1, b2);
	  shiftcount = adrbitmap[b1] - b1;	/* positive = shift left */

	  b1 = b2;
	}
      else
	/* passeddefined */
      if (allowranges)
	{
	  currentmax++;
	  selectmask = bitrange32 (b1, 31);
	  shiftcount = currentmax - b1;		/* positive = shift left */

	  b1 = 32;
	}
      else
	{
	  currentmax++;
	  selectmask = bitpos32[b1];
	  shiftcount = currentmax - b1;		/* positive = shift left */
	}

      for (i = 0; i < &update_adr_upE2 - &update_adr_upB2; i++)
	*(adrcode + acp + i) = *(&update_adr_upB2 + i);		/* Copy... */

      *((unsigned long *) (adrcode + acp + 0x1)) = selectmask;	/* & Mutate! */
      if (shiftcount < 0)
	{
	  *((unsigned char *) (adrcode + acp + 0x8)) = (unsigned char) 0xEA;
	  shiftcount = -shiftcount;
	}
      *((unsigned char *) (adrcode + acp + 0x9)) = (unsigned char) shiftcount;

      acp += i;

    }

  /* "Footer": */

  for (i = 0; i < &update_adr_upE3 - &update_adr_upB3; i++)
    *(adrcode + acp + i) = *(&update_adr_upB3 + i);	/* Copy... */

  *((unsigned long *) (adrcode + acp + 0x4)) = procautonomrange;
  *((unsigned long *) (adrcode + acp + 0xA)) = procendadr;	/* & Mutate! */
  *((unsigned long *) (adrcode + acp + 0x12)) = memendadr;
  *((unsigned long *) (adrcode + acp + 0x19)) = procstartadr;
  *((unsigned long *) (adrcode + acp + 0x20)) = procendadr;
  *((unsigned long *) (adrcode + acp + 0x29)) = procendadr;

  acp += i;

  return acp;
}

int
prepare_update_adr_dn (adrbitmap_t adrbitmap, unsigned long procstartadr,
		       unsigned long procendadr, char allowranges,
		       adrcode_t adrcode)
{
  int i, acp, b1, b2;
  signed char shiftcount, currentmax;
  signed char passeddefined = 0;
  unsigned char meminseqend;

  /* Detect autonomous range, which is defined as the largest series
     of in-sequence bits from the right.
     Test for 30 instead of 31 because we need the 32th bit... */
  meminseqend = 0;
  i = adrbitmap[0];
  while (meminseqend < 30 && i < 30 && adrbitmap[meminseqend + 1] == (i + 1))
    {
      meminseqend++;
      i++;
    }

  /* Now meminseqend is the number of the last mem-adr bit that is in
     the autonomous range. */

  /* Set the various variables */
  autonomdec = ~bitpos32[adrbitmap[0]] + 1;
  memstartdec = bitpos32[meminseqend + 1];
  procautonomrange = bitrange32 (adrbitmap[0], adrbitmap[meminseqend]);

  memstartadr = proc2mem (procstartadr, adrbitmap)
    | bitrange32 (0, meminseqend);
  memendadr = proc2mem (procendadr, adrbitmap);

  /* "Start code": */

  /* memcpy(adrcode, &update_adr_dn_startB, 32); 
     Well, yes, it _doesn't_ use the C library ("macro-like") but
     I don't use it because there's no guarantee... */

  for (i = 0; i < 32; i++)
    *(adrcode + i) = *(&update_adr_dn_startB + i);	/* Copy... */

  *((unsigned long *) (adrcode + 0x1)) = memstartadr;	/* & Mutate! */
  *((unsigned long *) (adrcode + 0x6)) = autonomdec;
  *((unsigned long *) (adrcode + 0x10)) = procstartadr;

  acp = 32;			/* Adrcode pointer points to next
				   address-to-be-filled. */

  /* "Header": */

  for (i = 0; i < &update_adr_dnE1 - &update_adr_dnB1; i++)
    *(adrcode + acp + i) = *(&update_adr_dnB1 + i);	/* Copy... */

  *((unsigned long *) (adrcode + acp + 0x2)) = memstartdec;	/* & Mutate! */

  acp += i;

  /* "Repeating part": */

  currentmax = -1;

  for (b1 = 0; b1 < 32; b1++)	/* b1 is start of range */
    {				/* b2 is end of range (poss. =b1) */
      if (adrbitmap[b1] >= 32)
	passeddefined = 1;

      if (!passeddefined)
	{
	  if (adrbitmap[b1] > currentmax)
	    currentmax = adrbitmap[b1];

	  b2 = b1;
	  if (allowranges)
	    while (b2 < 31 && adrbitmap[b2 + 1] == adrbitmap[b2] + 1
		   && adrbitmap[b2 + 1] < 32)
	      {
		b2++;
		if (adrbitmap[b2] > currentmax)
		  currentmax = adrbitmap[b2];
	      }

	  selectmask = bitrange32 (b1, b2);
	  shiftcount = adrbitmap[b1] - b1;	/* positive = shift left */

	  b1 = b2;
	}
      else
	/* passeddefined */
      if (allowranges)
	{
	  currentmax++;
	  selectmask = bitrange32 (b1, 31);
	  shiftcount = currentmax - b1;		/* positive = shift left */

	  b1 = 32;
	}
      else
	{
	  currentmax++;
	  selectmask = bitpos32[b1];
	  shiftcount = currentmax - b1;		/* positive = shift left */
	}

      for (i = 0; i < &update_adr_dnE2 - &update_adr_dnB2; i++)
	*(adrcode + acp + i) = *(&update_adr_dnB2 + i);		/* Copy... */

      *((unsigned long *) (adrcode + acp + 0x1)) = selectmask;	/* & Mutate! */
      if (shiftcount < 0)
	{
	  *((unsigned char *) (adrcode + acp + 0x8)) = (unsigned char) 0xEA;
	  shiftcount = -shiftcount;
	}
      *((unsigned char *) (adrcode + acp + 0x9)) = (unsigned char) shiftcount;

      acp += i;

    }

  /* "Footer": */

  for (i = 0; i < &update_adr_dnE3 - &update_adr_dnB3; i++)
    *(adrcode + acp + i) = *(&update_adr_dnB3 + i);	/* Copy... */

  *((unsigned long *) (adrcode + acp + 0x4)) = procautonomrange;
  *((unsigned long *) (adrcode + acp + 0xA)) = procendadr;	/* & Mutate! */
  *((unsigned long *) (adrcode + acp + 0x12)) = memendadr;
  *((unsigned long *) (adrcode + acp + 0x19)) = procstartadr;
  *((unsigned long *) (adrcode + acp + 0x20)) = procendadr;
  *((unsigned long *) (adrcode + acp + 0x29)) = procendadr;

  acp += i;

  return acp;
}
