/** 
* @file read_input.c
* @brief file parsing part
* @author serge guelton
* @date 2008-02-13
*/
/*
 * This file is part of hyantes.
 *
 * hyantes is free software; you can redistribute it and/or modify
 * it under the terms of the CeCILL-C License
 *
 * You should have received a copy of the CeCILL-C License
 * along with this program.  If not, see <http://www.cecill.info/licences>.
 */

#include "hs_compat.h"
#include "read_input.h"

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <float.h>
#include <assert.h>

/* internals */
static int stockcmp(const void *s1, const void *s2);
static void check_uniq(Stock * stocks,
   size_t *
   stocks_count) /*@modifies stocks@ *//*@modifies stocks_count@ */ ;
static void convert_to_radian(Stock * stocks, size_t stocks_count,
   hs_coord_t * pCInit) /*@modifies stocks@ *//*@modifies pCInit@ */ ;


/* exported */
/** 
* @brief read stocks from a tabulated file, fill the maximum window, set the number of different stocks and of cumulated stock
* 
* @param pFile pointer to an opened file containg lines at the followinf format
*   latitude longitude potential
*   other lines are ignored and a message is printed out
*   in case of duplicated lines (same latitude, same longitude) one of the line is slightly modified by adding
*   a small value to its latitude and longitude
* @param pCInit pointer to the maximum window, filled during processing
* @param stocks_count pointer to the number of different element in the file
* @param pStockTotal  pointer to the value of cumulated stocks (the sum of all stocks in the file)
* 
* @return allocated array containing latitude-sorted stocks, or NULL if an error occured
*   it also sets the value of pCInit, stocks_count and pStockTotal
*/
Stock *read_input(FILE * pFile, hs_coord_t * pCInit, size_t * stocks_count,
   double *pStockTotal)
{
    int size;
    int nb_line = 0;
    size_t allocated_stocks_count = 1;
    Stock *pWrite = NULL;
    Stock *stocks = NULL;
    /* init */
    *pStockTotal = 0;
    stocks = (Stock *) malloc(sizeof(*stocks) * allocated_stocks_count);
    if(stocks == NULL)
    {
        perror("[read_input::malloc] ");
        exit(EXIT_FAILURE);
    }
    stocks->lat = stocks->lon = stocks->pot = 0.f;

    /* read data file *//*@-branchstate@ */
    for(*stocks_count = 0; feof(pFile) == 0;)
    {

        /* reallocate output ptr */
        if(allocated_stocks_count < *stocks_count + 1)
        {
            allocated_stocks_count *= 2;
            stocks =
               (Stock *) realloc(stocks,
               sizeof(*stocks) * allocated_stocks_count);
            if(stocks == NULL)
            {
                perror("[read_input::realloc] ");
                exit(EXIT_FAILURE);
            }
        }

        /* init read iterator */
        pWrite = stocks + (*stocks_count);

        /* lecture d'une ligne */
        size =
           fscanf(pFile, "%lf %lf %lf ", &pWrite->lat, &pWrite->lon,
           &pWrite->pot);
        ++nb_line;

        /* analyze result */
        if(size > 2)
        {
            if(pCInit->mLat - pWrite->lat > FLT_EPSILON)
                pCInit->mLat = pWrite->lat;
            if(pWrite->lat - pCInit->MLat > FLT_EPSILON)
                pCInit->MLat = pWrite->lat;
            if(pCInit->mLon - pWrite->lon > FLT_EPSILON)
                pCInit->mLon = pWrite->lon;
            if(pWrite->lon - pCInit->MLon > FLT_EPSILON)
                pCInit->MLon = pWrite->lon;

            *pStockTotal += pWrite->pot;
            (*stocks_count)++;
        }
        else if(size == 0)
        {
            char line[256];
            fprintf(stderr, "failed to parse line %d", nb_line);
            if(fgets(line, (int) (sizeof(line) / sizeof(*line)),
                  pFile) != NULL)
            {
                fprintf(stderr, ":  %s", line);
            }
            else
            {
                fprintf(stderr, ", cannot even read this line !!\n");
                exit(EXIT_FAILURE);
            }
        }
        else if(size == EOF)
        {
            fprintf(stderr, "[read_input::fscanf] input error\n");
        }
    }
    /* resize final array if necessary */
    if(allocated_stocks_count > *stocks_count)
    {
        allocated_stocks_count = (*stocks_count);
        stocks =
           (Stock *) realloc(stocks,
           sizeof(*stocks) * allocated_stocks_count);
        if(stocks == NULL)
        {
            perror("[read_input::realloc] ");
            exit(EXIT_FAILURE);
        }
    }

    /* fermeture du fichier d'entre */
    if(fclose(pFile) != 0)
    {
        perror("[read_input::fclose] ");
        exit(EXIT_FAILURE);
    }

    /*@-formatcode@ */
    fprintf(stdout, "# sorting %zd values ...\n", *stocks_count);

    assert(stocks != NULL);
    qsort(stocks, *stocks_count, sizeof(*stocks), &stockcmp);
    check_uniq(stocks, stocks_count);
    convert_to_radian(stocks, *stocks_count, pCInit);

    return stocks;
}

/******************************************************************************\
*
*      routines
*
\******************************************************************************/

/**
 * compare two stocks
 *  s1 < s2  <> (s1.lat < s2.lat) || ( s1.lat == s2.lat && (s1.lon < s2.lon)
 *  \param s1 first geostock adress
 *  \param s2 second geostock adress
 *  \return 0 if equals, negative if s1<s2, postive otherwise
 */
static int stockcmp(const void *s1, const void *s2)
{
    double lat_diff =
       ((const Stock *) s1)->lat - ((const Stock *) s2)->lat;
    double lon_diff =
       ((const Stock *) s1)->lon - ((const Stock *) s2)->lon;
    if(fabs(lat_diff) < FLT_EPSILON)
    {
        if(fabs(lon_diff) < FLT_EPSILON)
            return 0;
        else
            return (lon_diff > FLT_EPSILON) ? 1 : -1;;
    }
    else
    {
        return (lat_diff > FLT_EPSILON) ? 1 : -1;
    }
}
/**
 * look for twice the same hs_coordinate in an array, and repair data fault
 *  \param stocks sorted stocks to check
 *  \param stocks_count number of stocks
 */
static void check_uniq(Stock * stocks, size_t * stocks_count)
{
    size_t i;
    for(i = 1; i < *stocks_count; i++)
    {
        if(fabs(stocks[i].lat - stocks[i - 1].lat) < FLT_EPSILON
           && fabs(stocks[i].lon - stocks[i - 1].lon) < FLT_EPSILON)
        {
            fprintf(stderr, "duplicate data: %f %f %f \n", stocks[i].lat,
               stocks[i].lon, stocks[i].pot);
            stocks[i].lat += rand() * FLT_EPSILON;
            stocks[i].lon += rand() * FLT_EPSILON;
            stocks[i].pot = stocks[i].pot;
        }
    }
}

/** 
* @brief converts degree angle to radian angle for stocks and pCInit
* 
* @param stocks array containg potential with degree cordinate
* @param stocks_count size of this array
* @param pCInit structure containg the maximum window in radian
*
*/
static void convert_to_radian(Stock * stocks, size_t stocks_count,
   hs_coord_t * pCInit)
{
    size_t i;
    for(i = 0; i < stocks_count; i++)
    {
        stocks[i].lat *= M_PI / 180;
        stocks[i].lon *= M_PI / 180;
    }

    pCInit->mLat *= M_PI / 180;
    pCInit->mLon *= M_PI / 180;
    pCInit->MLat *= M_PI / 180;
    pCInit->MLon *= M_PI / 180;
}
