#ifdef USE_PRAGMA_IDENT_HDR
#pragma ident "@(#)abstractInterpreter.hpp	1.1 07/08/29 13:42:23 JVM"
#endif
/*
 * Copyright 1997-2007 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 *  
 */

// This file contains the platform-independant parts
// of the abstract interpreter and the abstract interpreter generator.

// Organization of the interpreter(s). There exists two different interpreters in hotpot
// an assembly language version (aka template interpreter) and a high level language version
// (aka c++ interpreter). Th division of labor is as follows:

// Template Interpreter          C++ Interpreter        Functionality
//
// templateTable*                bytecodeInterpreter*   actual interpretation of bytecodes
//
// templateInterpreter*          cppInterpreter*        generation of assembly code that creates
//                                                      and manages interpreter runtime frames. 
//                                                      Also code for populating interpreter 
//                                                      frames created during deoptimization.
//
// For both template and c++ interpreter. There are common files for aspects of the interpreter
// that are generic to both interpreters. This is the layout:
//
// abstractInterpreter.hpp: generic description of the interpreter.
// interpreter*:            generic frame creation and handling.
//

//------------------------------------------------------------------------------------------------------------------------
// The C++ interface to the bytecode interpreter(s).

class AbstractInterpreter: AllStatic {
  friend class VMStructs;
  friend class Interpreter;
  friend class CppInterpreterGenerator;
 public:
  enum MethodKind {        
    zerolocals,                                                 // method needs locals initialization
    zerolocals_synchronized,                                    // method needs locals initialization & is synchronized
    native,                                                     // native method
    native_synchronized,                                        // native method & is synchronized
    empty,                                                      // empty method (code: _return)
    accessor,                                                   // accessor method (code: _aload_0, _getfield, _(a|i)return)
    abstract,                                                   // abstract method (throws an AbstractMethodException)
    java_lang_math_sin,                                         // implementation of java.lang.Math.sin   (x)
    java_lang_math_cos,                                         // implementation of java.lang.Math.cos   (x)
    java_lang_math_tan,                                         // implementation of java.lang.Math.tan   (x)
    java_lang_math_abs,                                         // implementation of java.lang.Math.abs   (x)
    java_lang_math_sqrt,                                        // implementation of java.lang.Math.sqrt  (x)
    java_lang_math_log,                                         // implementation of java.lang.Math.log   (x)
    java_lang_math_log10,                                       // implementation of java.lang.Math.log10 (x)
    number_of_method_entries,
    invalid = -1
  };

  enum SomeConstants {
    number_of_result_handlers = 10                              // number of result handlers for native calls
  };    

 protected:
  static StubQueue* _code;                                      // the interpreter code (codelets)
  
  static bool       _notice_safepoints;                         // true if safepoints are activated

  static address    _native_entry_begin;                        // Region for native entry code
  static address    _native_entry_end;

  // method entry points
  static address    _entry_table[number_of_method_entries];     // entry points for a given method
  static address    _native_abi_to_tosca[number_of_result_handlers];  // for native method result handlers
  static address    _slow_signature_handler;                              // the native method generic (slow) signature handler

  static address    _rethrow_exception_entry;                   // rethrows an activation in previous frame


  
  friend class      AbstractInterpreterGenerator;
  friend class              InterpreterGenerator;
  friend class      InterpreterMacroAssembler;

 public:
  // Initialization/debugging
  static void       initialize();
  static StubQueue* code()                                      { return _code; }


  // Method activation
  static MethodKind method_kind(methodHandle m);
  static address    entry_for_kind(MethodKind k)                { assert(0 <= k && k < number_of_method_entries, "illegal kind"); return _entry_table[k]; }
  static address    entry_for_method(methodHandle m)            { return _entry_table[method_kind(m)]; }

  static void       print_method_kind(MethodKind kind)          PRODUCT_RETURN;

  // Runtime support

  // length = invoke bytecode length (to advance to next bytecode)
  static address    deopt_entry   (TosState state, int length) { ShouldNotReachHere(); return NULL; }
  static address    return_entry  (TosState state, int length) { ShouldNotReachHere(); return NULL; }

  static address    rethrow_exception_entry()                   { return _rethrow_exception_entry; }

  // Activation size in words for a method that is just being called.
  // Parameters haven't been pushed so count them too.
  static int        size_top_interpreter_activation(methodOop method);

  // Deoptimization support
  static address    continuation_for(methodOop method,
				     address bcp,
				     int callee_parameters,
				     bool is_top_frame,
				     bool& use_next_mdp);

  // share implementation of size_activation and layout_activation:
  static int        size_activation(methodOop method,
				    int temps,
                                    int popframe_args,
				    int monitors,
				    int callee_params,
				    int callee_locals,
				    bool is_top_frame);

  static int       layout_activation(methodOop method,
				      int temps,
                                      int popframe_args,
				      int monitors,
				      int callee_params,
				      int callee_locals,
				      frame* caller,
				      frame* interpreter_frame,
				      bool is_top_frame);

  // Runtime support
  static bool       is_not_reached(                       methodHandle method, int bci);
  // Safepoint support
  static void       notice_safepoints()                         { ShouldNotReachHere(); } // stops the thread when reaching a safepoint
  static void       ignore_safepoints()                         { ShouldNotReachHere(); } // ignores safepoints

  // Support for native calls
  static address    slow_signature_handler()                    { return _slow_signature_handler; }
  static address    result_handler(BasicType type)              { return _native_abi_to_tosca[BasicType_as_index(type)]; }
  static int        BasicType_as_index(BasicType type);         // computes index into result_handler_by_index table
  static bool       in_native_entry(address pc)                 { return _native_entry_begin <= pc && pc < _native_entry_end; }
  // Debugging/printing
  static void       print();                                    // prints the interpreter code

  // Support for Tagged Stacks
  //
  // Tags are stored on the Java Expression stack above the value:
  // 
  //  tag   
  //  value
  //
  // For double values:
  //
  //  tag2  
  //  high word
  //  tag1
  //  low word

 public:
  static int stackElementWords()   { return TaggedStackInterpreter ? 2 : 1; }
  static int stackElementSize()    { return stackElementWords()*wordSize; }
  static int logStackElementSize() { return
                 TaggedStackInterpreter? LogBytesPerWord+1 : LogBytesPerWord; }

  // Tag is at pointer, value is one below for a stack growing down
  // (or above for stack growing up)
  static int  value_offset_in_bytes()  {
    return TaggedStackInterpreter ?
      frame::interpreter_frame_expression_stack_direction() * wordSize : 0;
  }
  static int  tag_offset_in_bytes()    { 
    assert(TaggedStackInterpreter, "should not call this");
    return 0;
  }

  // Tagged Locals
  // Locals are stored relative to Llocals:
  //
  // tag    <- Llocals[n]
  // value
  //
  // Category 2 types are indexed as:
  //
  // tag    <- Llocals[-n]
  // high word
  // tag    <- Llocals[-n+1]
  // low word
  //

  // Local values relative to locals[n]
  static int  local_offset_in_bytes(int n) {
    return ((frame::interpreter_frame_expression_stack_direction() * n) *
            stackElementSize()) + value_offset_in_bytes();
  }
  static int  local_tag_offset_in_bytes(int n) {
    assert(TaggedStackInterpreter, "should not call this");
    return ((frame::interpreter_frame_expression_stack_direction() * n) * 
            stackElementSize()) + tag_offset_in_bytes();
  }

};

//------------------------------------------------------------------------------------------------------------------------
// The interpreter generator.

class Template;
class AbstractInterpreterGenerator: public StackObj {
 protected:
  InterpreterMacroAssembler* _masm;

  // shared code sequences
  // Converter for native abi result to tosca result
  address generate_result_handler_for(BasicType type);
  address generate_slow_signature_handler();

  // entry point generator
  address generate_method_entry(AbstractInterpreter::MethodKind kind);
  
  void bang_stack_shadow_pages(bool native_call);

  void generate_all();

 public:
  AbstractInterpreterGenerator(StubQueue* _code);
};

