/*
 * Copyright 2000-2004 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */
package javax.print.attribute.standard;

import javax.print.attribute.Attribute;
import javax.print.attribute.IntegerSyntax;
import javax.print.attribute.PrintServiceAttribute;

/**
 * Class PagesPerMinuteColor is an integer valued printing attribute that 
 * indicates the nominal number of pages per minute to the nearest whole number 
 * which may be generated by this printer when printing color (e.g., simplex, 
 * color). For purposes of this attribute, "color" means the same as for the 
 * {@link ColorSupported ColorSupported} attribute, namely, the device is 
 * capable of any type of color printing at all, including highlight color as 
 * well as full process color. This attribute is informative, not a service 
 * guarantee. Generally, it is the value used in the marketing literature to 
 * describe the color capabilities of this device. A value of 0 indicates a 
 * device that takes more than two minutes to process a page. If a color device 
 * has several color modes, it may use the pages-per- minute value for this 
 * attribute that corresponds to the mode that produces the highest number. 
 * <P>
 * A black and white only printer must not include the PagesPerMinuteColor 
 * attribute in its attribute set or service registration. If this attribute is 
 * present, then the {@link ColorSupported ColorSupported} printer description 
 * attribute must also be present and have a value of SUPPORTED. 
 * <P>
 * <B>IPP Compatibility:</B> The integer value gives the IPP integer value. The 
 * category name returned by <CODE>getName()</CODE> gives the IPP attribute 
 * name. 
 * <P>
 *
 * @author  Alan Kaminsky
 */
public final class PagesPerMinuteColor extends IntegerSyntax
	implements PrintServiceAttribute {

    static final long serialVersionUID = 1684993151687470944L;

    /**
     * Construct a new pages per minute color attribute with the given integer 
     * value. 
     *
     * @param  value  Integer value.
     *
     * @exception  IllegalArgumentException
     *    (Unchecked exception) Thrown if <CODE>value</CODE> is less than 0.
     */
    public PagesPerMinuteColor(int value) {
	super(value, 0, Integer.MAX_VALUE);
    }

    /**
     * Returns whether this pages per minute color attribute is equivalent to 
     * the passed in object. To be equivalent, all of the following conditions 
     * must be true: 
     * <OL TYPE=1>
     * <LI>
     * <CODE>object</CODE> is not null.
     * <LI>
     * <CODE>object</CODE> is an instance of class PagesPerMinuteColor.
     * <LI>
     * This pages per minute attribute's value and <CODE>object</CODE>'s 
     * value are equal. 
     * </OL>
     *
     * @param  object  Object to compare to.
     *
     * @return  True if <CODE>object</CODE> is equivalent to this pages per
     *          minute color attribute, false otherwise. 
     */
    public boolean equals(Object object) {
	return (super.equals(object) &&
		object instanceof PagesPerMinuteColor);
    }

    /**
     * Get the printing attribute class which is to be used as the "category" 
     * for this printing attribute value.
     * <P>
     * For class PagesPerMinuteColor, the 
     * category is class PagesPerMinuteColor itself. 
     *
     * @return  Printing attribute class (category), an instance of class
     *          {@link java.lang.Class java.lang.Class}.
     */
    public final Class<? extends Attribute> getCategory() {
	return PagesPerMinuteColor.class;
    }

    /**
     * Get the name of the category of which this attribute value is an 
     * instance. 
     * <P>
     * For class PagesPerMinuteColor, the 
     * category name is <CODE>"pages-per-minute-color"</CODE>. 
     *
     * @return  Attribute category name.
     */
    public final String getName() {
	return "pages-per-minute-color";
    }

}
