/*
 * Copyright 2001 Sun Microsystems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Sun Microsystems, Inc., 4150 Network Circle, Santa Clara,
 * CA 95054 USA or visit www.sun.com if you need additional information or
 * have any questions.
 */

/*
 * @test
 * @bug 4469866 
 * @summary Connecting to a link-local IPv6 address should not 
 *          causes a SocketException to be thrown.
 */
import java.net.*;
import java.util.Enumeration;

public class LinkLocal {

    static int testCount = 0;
    static int failed = 0;

    static void TcpTest(InetAddress ia) throws Exception {
	System.out.println("**************************************");
	testCount++;
	System.out.println("Test " + testCount + ": TCP connect to " + ia);

	/*
         * Create ServerSocket on wildcard address and then
         * try to connect Socket to link-local address.
         */
        ServerSocket ss = new ServerSocket(0);

        Socket s = new Socket();
        try {
            s.connect(new InetSocketAddress(ia, ss.getLocalPort()));

	    System.out.println("Test passed - connection established.");

            // connection was established so accept it
            Socket s2 = ss.accept();
            s2.close();
        } catch (SocketException e) {
	    failed++;
	    System.out.println("Test failed: " + e);
        }

        // clean up
        s.close();
        ss.close();
    }

    static void UdpTest(InetAddress ia, boolean connected) throws Exception {

	System.out.println("**************************************");
	testCount++;

	if (connected) {
	    System.out.println("Test " + testCount + ": UDP connect to " + ia);
	} else {
	    System.out.println("Test " + testCount + ": UDP send to " + ia);
	}

	DatagramSocket ds1 = new DatagramSocket();
	DatagramSocket ds2 = new DatagramSocket();

	try {
	    byte b[] = "Hello".getBytes();
	    DatagramPacket p = new DatagramPacket(b, b.length);

	    if (connected) {
		ds1.connect( new InetSocketAddress(ia, ds2.getLocalPort()) );
		System.out.println("DatagramSocket connected.");
	    } else {
		p.setAddress(ia);	
		p.setPort(ds2.getLocalPort());
	    }
	    ds1.send(p);
	    System.out.println("Packet has been sent.");

	    ds2.setSoTimeout(1000);
	    ds2.receive(p);
	    System.out.println("Test passed - packet received.");
	} catch (SocketException e) {
	    failed++;
	    System.out.println("Test failed: " + e);
	}

	ds1.close();
	ds2.close();
    }

    static void TestAddress(InetAddress ia) throws Exception {
	TcpTest(ia);
	UdpTest(ia, true);	/* unconnected */
	UdpTest(ia, false);	/* connected */
    }

    public static void main(String args[]) throws Exception {

	/*
	 * If an argument is provided ensure that it's
	 * a link-local IPv6 address.
	 */
	if (args.length > 0) {
	    InetAddress ia = InetAddress.getByName(args[0]);

	    if ( !(ia instanceof Inet6Address) ||
	        !ia.isLinkLocalAddress()) {
		throw new Exception(ia + 
			" is not a link-local IPv6 address");
	    }

	    TestAddress(ia);
	}

	/*
	 * If no argument is provided then enumerate the
	 * local addresses and run the test on each link-local
	 * IPv6 address.
	 */
	if (args.length == 0) {
	    Enumeration nifs = NetworkInterface.getNetworkInterfaces();
	    while (nifs.hasMoreElements()) {
	        NetworkInterface ni = (NetworkInterface)nifs.nextElement();
	        Enumeration addrs = ni.getInetAddresses();
	        while (addrs.hasMoreElements()) {
		    InetAddress addr = (InetAddress)addrs.nextElement();

		    if (addr instanceof Inet6Address &&
		        addr.isLinkLocalAddress()) {
	
			TestAddress(addr);
		    }
		}
	    }
	}

	/*
	 * Print results
	 */
	if (testCount == 0) {
	    System.out.println("No link-local IPv6 addresses - test skipped!");
	} else {
	    System.out.println("**************************************");
	    System.out.println(testCount + " test(s) executed, " + 
		failed + " failed.");
	    if (failed > 0) {
		throw new Exception( failed + " test(s) failed.");
	    }
	}
    }
}
