# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module contains the following classes:
- ValueDelegate
- StandardItemModelBoundaries
- StandardItemModelCoal
- StandardItemModelCoalMass
- BoundaryConditionsView
"""

#-------------------------------------------------------------------------------
# Standard modules
#-------------------------------------------------------------------------------

import string, logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from BoundaryConditionsForm import Ui_BoundaryConditionsForm

from Base.Toolbox import GuiParam
from Base.QtPage import DoubleValidator, ComboModel
from Pages.LocalizationModel import LocalizationModel, Zone
#from Pages.FunctionInterpretorDialog import FunctionInterpretorDialog
from Pages.Boundary import Boundary

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("BoundaryConditionsView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# Line edit delegate with a Double validator (positive value)
#-------------------------------------------------------------------------------

class ValueDelegate(QItemDelegate):
    def __init__(self, parent=None):
        super(ValueDelegate, self).__init__(parent)
        self.parent = parent

    def createEditor(self, parent, option, index):
        editor = QLineEdit(parent)
        validator = DoubleValidator(editor, min=0.)
        editor.setValidator(validator)
        #editor.installEventFilter(self)
        return editor

    def setEditorData(self, editor, index):
        value = index.model().data(index, Qt.DisplayRole).toString()
        editor.setText(value)

    def setModelData(self, editor, model, index):
        value, ok = editor.text().toDouble()
        if editor.validator().state == QValidator.Acceptable:
            model.setData(index, QVariant(value), Qt.DisplayRole)

#-------------------------------------------------------------------------------
# StandarItemModel class to display boundaries in a QTreeView
#-------------------------------------------------------------------------------

class StandardItemModelBoundaries(QStandardItemModel):
    def __init__(self):
        QStandardItemModel.__init__(self)
        self.headers = [self.tr("Label"), self.tr("Zone"),
                        self.tr("Nature"), self.tr("Localization")]
        self.setColumnCount(len(self.headers))
        self.dataBoundary = []


    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if role == Qt.DisplayRole:
            return QVariant(self.dataBoundary[index.row()][index.column()])
        return QVariant()


    def flags(self, index):
        if not index.isValid():
            return Qt.ItemIsEnabled
        else:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable


    def headerData(self, section, orientation, role):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            return QVariant(self.headers[section])
        return QVariant()


    def setData(self, index, value, role):
        self.emit(SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), index, index)
        return True


    def insertItem(self, label, codeNumber, var_nature, local):
        line = [label, codeNumber, var_nature, local]
        self.dataBoundary.append(line)
        row = self.rowCount()
        self.setRowCount(row+1)


    def getItem(self, row):
        return self.dataBoundary[row]

#-------------------------------------------------------------------------------
# StandarItemModel class to display Coals in a QTableView
#-------------------------------------------------------------------------------

class StandardItemModelCoal(QStandardItemModel):

    def __init__(self, case):
        QStandardItemModel.__init__(self)
        self.headers = [
            self.tr("Coal name"), self.tr("Coal value"), self.tr("Coal unit"),
            self.tr("Coal Temp. \nname"), self.tr("Coal Temp. \nvalue"), self.tr("Coal Temp. \nunit")]
        self.setColumnCount(len(self.headers))
        self.dataCoal = []
        self.case = case

    def setBoundaryFromLabel(self, label):
        self.modelBoundary = Boundary('coal_inlet', label, self.case)

    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if role == Qt.DisplayRole:
            return QVariant(self.dataCoal[index.row()][index.column()])
        return QVariant()

    def flags(self, index):
        if not index.isValid():
            return Qt.ItemIsEnabled
        elif index.column() in [1,4]:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable | Qt.ItemIsEditable
        else:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable

    def headerData(self, section, orientation, role):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            return QVariant(self.headers[section])
        return QVariant()

    def setData(self, index, value, role):
        row = index.row()
        col = index.column()
        if not hasattr(self, "modelBoundary"):
            log.debug("ERROR in setData (StandardItemModelCoal) : no Boundary model defined")
            return
        v, ok = value.toDouble()
        self.dataCoal[row][col] = v
        if col == 1:
            self.modelBoundary.setCoalFlow(v, row)
        elif col == 4:
            self.modelBoundary.setCoalTemperature(v, row)
        self.emit(SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), index, index)
        return True

    def insertItem(self, nameCoal, valCoal, unitCoal, nameCoalTemp, valCoalTemp, unitCoalTemp):
        line = [nameCoal, valCoal, unitCoal, nameCoalTemp, valCoalTemp, unitCoalTemp]
        self.dataCoal.append(line)
        row = self.rowCount()
        self.setRowCount(row+1)

    def deleteAll(self):
        self.dataCoal = []
        self.setRowCount(0)

#-------------------------------------------------------------------------------
# StandarItemModel class to display Coal masses in a QTableView
#-------------------------------------------------------------------------------

class StandardItemModelCoalMass(QStandardItemModel):

    def __init__(self, case, coalNumber, coalClassesNumber):
        QStandardItemModel.__init__(self)
        self.case = case
        self.coalNumber = coalNumber
        self.coalClassesNumber = coalClassesNumber

    def setRatio(self, ratio):
        cols = len(ratio)
        if type(ratio[0]) == type([]):
            rows = max([len(c) for c in ratio])
        else:
            rows = 1
        self.setColumnCount(cols)
        self.setRowCount(rows)
        self.ratio = ratio

    def setBoundaryFromLabel(self, label):
        log.debug("setBoundaryFromLabel")
        self.modelBoundary = Boundary('coal_inlet', label, self.case)

    def data(self, index, role):
        if not index.isValid():
            return QVariant()
        if role == Qt.DisplayRole:
            classe = index.row()
            coal   = index.column()
            if classe < self.coalClassesNumber[coal]:
                try:
                    return QVariant(self.ratio[coal][classe])
                except:
                    log.debug("ERROR no data for self.ratio[%i][%i] "%(coal, classe))
        return QVariant()

    def flags(self, index):
        if not index.isValid():
            return Qt.ItemIsEnabled
        elif index.row() >= self.coalClassesNumber[index.column()]:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable
        else:
            return Qt.ItemIsEnabled | Qt.ItemIsSelectable | Qt.ItemIsEditable

    def headerData(self, section, orientation, role):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            return QVariant("Coal" + " " + str(section+1))
        if orientation == Qt.Vertical and role == Qt.DisplayRole:
            return QVariant("Class" + " " + str(section+1))
        return QVariant()

    def setData(self, index, value, role):
        if not hasattr(self, "modelBoundary"):
            log.debug("ERROR in setData (StandardItemModelCoalMass) : no Boundary model defined")
            return
        classe = index.row()
        coal   = index.column()
        v, ok = value.toDouble()
        self.ratio[coal][classe] = v
        log.debug("setData v = %f "%v)

        liste = self.modelBoundary.getCoalRatios(coal)

        lastValue = 0
        for iclasse in range(0, self.coalClassesNumber[coal]-1):
            lastValue += self.ratio[coal][iclasse]

        if lastValue < 100.+ 1e-6 :
            liste[classe] = self.ratio[coal][classe]
            lastValue = 100 - lastValue
            self.ratio[coal][self.coalClassesNumber[coal]-1] = lastValue
            liste[self.coalClassesNumber[coal]-1] = lastValue
            self.modelBoundary.setCoalRatios(coal, liste)
##             self.getRatioLastClass(coal)
        else :
##             self.ratio[coal][classe].set(model.getClassCoalRatio(coal, classe))
            self.ratio[coal][classe] = liste[classe]

        self.emit(SIGNAL("dataChanged(const QModelIndex &, const QModelIndex &)"), index, index)
        return True

    def deleteAll(self):
        self.ratio = []
        self.setRowCount(0)

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class BoundaryConditionsView(QWidget, Ui_BoundaryConditionsForm):
    def __init__(self, parent, case):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_BoundaryConditionsForm.__init__(self)
        self.setupUi(self)

        self.case = case

        import Pages.CoalCombustionModel as CoalCombustion
        coalModel =  CoalCombustion.CoalCombustionModel(self.case)
        if coalModel.getCoalCombustionModel() != "off" :
            import Pages.CoalThermoChemistry as CoalThermoChemistry
            coalModel = CoalThermoChemistry.CoalThermoChemistryModel("dp_FCP", self.case)
            self.coalNumber = coalModel.getCoals().getNumber()
            self.coalClassesNumber = []
            for coal in range(0, self.coalNumber):
                self.coalClassesNumber.append(coalModel.getCoals().getCoal(coal+1).getClassesNumber())
        else :
            self.coalNumber = 0
            self.coalClassesNumber = [0]

        self.ratio = self.coalNumber*[0]
        for i in range(0, self.coalNumber) :
            self.ratio[i] = self.coalClassesNumber[i]*[0]

        # Create the Page layout.

        # Model and QTreeView for Boundaries
        self.modelBoundaries = StandardItemModelBoundaries()
        self.treeViewBoundaries.setModel(self.modelBoundaries)

        # Coal table
        self.modelCoal = StandardItemModelCoal(self.case)
        self.tableViewCoal.setModel(self.modelCoal)

        delegateValue = ValueDelegate(self.tableViewCoal)
        self.tableViewCoal.setItemDelegateForColumn(1, delegateValue)
        self.tableViewCoal.setItemDelegateForColumn(4, delegateValue)

        # Coal mass table
        self.modelCoalMass = StandardItemModelCoalMass(self.case, self.coalNumber, self.coalClassesNumber)
        self.tableViewCoalMass.setModel(self.modelCoalMass)

        delegateValueMass = ValueDelegate(self.tableViewCoalMass)
        for c in range(self.modelCoalMass.columnCount()):
            self.tableViewCoalMass.setItemDelegateForColumn(c, delegateValueMass)

        # Connections
        self.connect(self.treeViewBoundaries, SIGNAL("clicked(const QModelIndex &)"), self.slotSelectBoundary)
        self.connect(self.groupBoxSliding, SIGNAL("clicked(bool)"), self.slotSlidingWall)

        self.connect(self.lineEditSlideU, SIGNAL("textChanged(const QString &)"), self.slotVelocityU)
        self.connect(self.lineEditSlideV, SIGNAL("textChanged(const QString &)"), self.slotVelocityV)
        self.connect(self.lineEditSlideW, SIGNAL("textChanged(const QString &)"), self.slotVelocityW)

        self.connect(self.radioButtonSmooth, SIGNAL("clicked()"), self.slotRoughness)
        self.connect(self.radioButtonRough,  SIGNAL("clicked()"), self.slotRoughness)

        self.connect(self.lineEditRoughCoef, SIGNAL("textChanged(const QString &)"), self.slotRoughnessHeight)

        self.connect(self.comboBoxVelocity, SIGNAL("activated(const QString&)"), self.slotChoiceVelocity)
        self.connect(self.lineEditVelocity, SIGNAL("textChanged(const QString &)"), self.slotVelocityValue)

        self.connect(self.lineEditXVelocity, SIGNAL("textChanged(const QString &)"), self.slotDirX)
        self.connect(self.lineEditYVelocity, SIGNAL("textChanged(const QString &)"), self.slotDirY)
        self.connect(self.lineEditZVelocity, SIGNAL("textChanged(const QString &)"), self.slotDirZ)

        # Coals
        self.connect(self.comboBoxTypeInlet,   SIGNAL("activated(const QString&)"), self.slotCoalFlowType)
        self.connect(self.comboBoxAirVelocity, SIGNAL("activated(const QString&)"), self.slotCoalChoiceVelocity)
        self.connect(self.lineEditAirVelocity, SIGNAL("textChanged(const QString &)"), self.slotCoalAirFlow)
        self.connect(self.lineEditTemperature, SIGNAL("textChanged(const QString &)"), self.slotTemperature)

#        self.connect(self.comboBoxDirectionVelocityCoal, SIGNAL("activated(const QString&)"), self.slotCoalDirFlow)

        self.connect(self.lineEditXVelocityCoal, SIGNAL("textChanged(const QString &)"), self.slotCoalDirX)
        self.connect(self.lineEditYVelocityCoal, SIGNAL("textChanged(const QString &)"), self.slotCoalDirY)
        self.connect(self.lineEditZVelocityCoal, SIGNAL("textChanged(const QString &)"), self.slotCoalDirZ)

        # Turbulence
        self.connect(self.comboBoxTurbulence, SIGNAL("activated(const QString&)"), self.slotChoiceTurbu)
        self.connect(self.lineEditDiameter, SIGNAL("textChanged(const QString &)"), self.slotDiam)
        self.connect(self.lineEditIntensity, SIGNAL("textChanged(const QString &)"), self.slotIntensity)
        self.connect(self.lineEditDiameterIntens, SIGNAL("textChanged(const QString &)"), self.slotDiam)


        # Combo models

        self.modelTypeInlet = ComboModel(self.comboBoxTypeInlet, 2, 1)
        self.modelTypeInlet.addItem(self.tr("Air"), 'airflow')
        self.modelTypeInlet.addItem(self.tr("Air & Coal"), 'coalflow')

        self.modelVelocity = ComboModel(self.comboBoxVelocity, 6, 1)
        self.modelVelocity.addItem(self.tr("Velocity"), 'norm')
        self.modelVelocity.addItem(self.tr("Mass flow rate"), 'flow1')
        self.modelVelocity.addItem(self.tr("Volumic flow rate"), 'flow2')
        self.modelVelocity.addItem(self.tr("Velocity and direction"), 'norm+direction')
        self.modelVelocity.addItem(self.tr("Mass flow rate and direction"), 'flow1+direction')
        self.modelVelocity.addItem(self.tr("Volumic flow rate and direction"), 'flow2+direction')

        self.modelAirVelocity = ComboModel(self.comboBoxAirVelocity, 6, 1)
        self.modelAirVelocity.addItem(self.tr("Velocity"), 'norm')
        self.modelAirVelocity.addItem(self.tr("Mass flow rate"), 'flow1')
        self.modelAirVelocity.addItem(self.tr("Volumic flow rate"), 'flow2')
        self.modelAirVelocity.addItem(self.tr("Velocity and direction"), 'norm+direction')
        self.modelAirVelocity.addItem(self.tr("Mass flow rate and direction"), 'flow1+direction')
        self.modelAirVelocity.addItem(self.tr("Volumic flow rate and direction"), 'flow2+direction')

#        self.modelDirVelocityCoal = ComboModel(self.comboBoxDirectionVelocityCoal,3,1)
#        self.modelDirVelocityCoal.addItem(self.tr("Direction of the flow normal to the inlet"), "normal")
#        self.modelDirVelocityCoal.addItem(self.tr("Specified direction of the flow by vector"), "vector")
#        self.modelDirVelocityCoal.addItem(self.tr("Fixed profile"), "formula")

        self.modelTurbulence = ComboModel(self.comboBoxTurbulence, 2, 1)
        self.modelTurbulence.addItem(self.tr("Calculation by hydraulic diameter"), 'hydraulic_diameter')
        self.modelTurbulence.addItem(self.tr("Calculation by turbulent intensity"), 'turbulent_intensity')

        # Validators

        validatorSlideU = DoubleValidator(self.lineEditSlideU)
        validatorSlideV = DoubleValidator(self.lineEditSlideV)
        validatorSlideW = DoubleValidator(self.lineEditSlideW)
        validatorRoughCoef = DoubleValidator(self.lineEditRoughCoef)
        validatorVelocity = DoubleValidator(self.lineEditVelocity)
        validatorX = DoubleValidator(self.lineEditXVelocity)
        validatorY = DoubleValidator(self.lineEditYVelocity)
        validatorZ = DoubleValidator(self.lineEditZVelocity)
        validatorAirFlow = DoubleValidator(self.lineEditAirVelocity)
        validatorTemp = DoubleValidator(self.lineEditTemperature, min=0.)
        validatorXCoal = DoubleValidator(self.lineEditXVelocity)
        validatorYCoal = DoubleValidator(self.lineEditYVelocity)
        validatorZCoal = DoubleValidator(self.lineEditZVelocity)
        validatorDiam = DoubleValidator(self.lineEditDiameter, min=0.)
        validatorDiam.setExclusiveMin(True)
        validatorIntensity = DoubleValidator(self.lineEditIntensity, min=0.)

        # Apply validators

        self.lineEditSlideU.setValidator(validatorSlideU)
        self.lineEditSlideV.setValidator(validatorSlideV)
        self.lineEditSlideW.setValidator(validatorSlideW)
        self.lineEditRoughCoef.setValidator(validatorRoughCoef)
        self.lineEditVelocity.setValidator(validatorVelocity)
        self.lineEditXVelocity.setValidator(validatorX)
        self.lineEditYVelocity.setValidator(validatorY)
        self.lineEditZVelocity.setValidator(validatorZ)
        self.lineEditAirVelocity.setValidator(validatorAirFlow)
        self.lineEditTemperature.setValidator(validatorTemp)
        self.lineEditXVelocityCoal.setValidator(validatorXCoal)
        self.lineEditYVelocityCoal.setValidator(validatorYCoal)
        self.lineEditZVelocityCoal.setValidator(validatorZCoal)
        self.lineEditDiameter.setValidator(validatorDiam)
        self.lineEditDiameterIntens.setValidator(validatorDiam)
        self.lineEditIntensity.setValidator(validatorIntensity)

        # Initialize Widgets

        self.entriesNumber = 0
        d = LocalizationModel('BoundaryZone', self.case)
        for zone in d.getZones():
            label = zone.getLabel()
            nature = zone.getNature()
            codeNumber = zone.getCodeNumber()
            local = zone.getLocalization()
            self.modelBoundaries.insertItem(label, codeNumber, nature, local)
        self.forgetStandardWindows()
        self.forgetCoalWindows()


    def setWindowsForCoalVelocityChoice(self, choice):
        """
        Put windows beyond choice of velocity for inlet nature
        """
        if choice  == 'norm':
            self.labelUnitVelocityAir.setText(QString(str('m/s')))
        elif choice == 'flow1':
            self.labelUnitVelocityAir.setText(QString(str('kg/s')))
        elif choice == 'flow2':
            self.labelUnitVelocityAir.setText(QString(str('m<sup>3</sup>/s')))


    def updateSpecDirWindowForCoal(self, label):
        """
        Put special window for direction of inlet for coal case
        """
        # ??? Same function as above ???
        #self.updateSpecDirWindow(label)
        model = Boundary('inlet', label, self.case)
        log.debug("updateSpecDirWindowForCoal model.getVelocityChoice() = %s"%model.getVelocityChoice())
        if model.getVelocityChoice()[-9:] == "direction":
            x = model.getDirection('direction_x')
            y = model.getDirection('direction_y')
            z = model.getDirection('direction_z')
            self.lineEditXVelocityCoal.setText(QString(str(x)))
            self.lineEditYVelocityCoal.setText(QString(str(y)))
            self.lineEditZVelocityCoal.setText(QString(str(z)))
            self.frameVelocity.show()
        else:
            self.frameVelocity.hide()


    @pyqtSignature("const QModelIndex&")
    def slotSelectBoundary(self, index):
        label, codeNumber, nature, local = self.modelBoundaries.getItem(index.row())
        if hasattr(self, "modelScalars"): del self.modelScalars
        log.debug("slotSelectBoundary label %s (%s)" % (label, nature))

        self.forgetStandardWindows()
        self.forgetCoalWindows()

        self.boundary = Boundary(nature, label, self.case)

        if nature == 'wall':
            self.groupBoxSliding.show()
            self.groupBoxRough.show()

            if self.boundary.getVelocityChoice() == "on":
                self.groupBoxSliding.setChecked(True)
                checked = True
            else:
                self.groupBoxSliding.setChecked(False)
                checked = False

            self.slotSlidingWall(checked)

            if self.boundary.getRoughnessChoice() == "on":
                self.radioButtonSmooth.setChecked(False)
                self.radioButtonRough.setChecked(True)
            else:
                self.radioButtonSmooth.setChecked(True)
                self.radioButtonRough.setChecked(False)

            self.slotRoughness()

        elif nature == 'inlet':
            choice = self.boundary.getVelocityChoice()

            if (self.coalNumber == 0):
                self.modelVelocity.setItem(str_model=choice)
                text = self.modelVelocity.dicoM2V[choice]
                self.slotChoiceVelocity(QString(text))

                self.groupBoxVelocity.show()
                self.groupBoxFlowTemp.hide()
                self.groupBoxCoal.hide()
                self.groupBoxCoalMass.hide()

            else:
                self.boundary = Boundary('coal_inlet', label, self.case)
                type = self.boundary.getCoalType()

                self.groupBoxVelocity.hide()
                self.groupBoxFlowTemp.show()
                self.groupBoxCoal.show()
                self.groupBoxCoalMass.show()

                if type in ("coalflow", "airflow"):
                    self.modelTypeInlet.setItem(str_model=type)
                else:
                    msg = "Error :invalid velocity_pressure choice for coal combustion"
                    raise ValueError, msg

                choice = string.split(choice, '+')[0]
                log.debug("slotSelectBoundary COAL INLET choice = %s "%choice)
                self.slotCoalFlowType(self.comboBoxTypeInlet.currentText())

                self.setWindowsForCoalVelocityChoice(choice)
                self.updateSpecDirWindowForCoal(label)

            from TurbulenceModel import TurbulenceModel
            if TurbulenceModel(self.case).getTurbulenceVariable():
                self.groupBoxTurbulence.show()
                turb_choice = self.boundary.getTurbulenceChoice()
                self.modelTurbulence.setItem(str_model=turb_choice)
                if turb_choice == "hydraulic_diameter":
                    self.frameTurbDiameter.show()
                    self.frameTurbIntensity.hide()
                    d = self.boundary.getHydraulicDiameter()
                    self.lineEditDiameter.setText(QString(str(d)))
                if turb_choice == "turbulent_intensity":
                    self.frameTurbIntensity.show()
                    self.frameTurbDiameter.hide()
                    i = self.boundary.getTurbulentIntensity()
                    d = self.boundary.getHydraulicDiameter()
                    self.lineEditIntensity.setText(QString(str(i)))
                    self.lineEditDiameterIntens.setText(QString(str(d)))


    def forgetStandardWindows(self):
        """
        For forget standard windows
        """
        self.groupBoxSliding.hide()
        self.groupBoxRough.hide()
        self.groupBoxVelocity.hide()
        self.groupBoxTurbulence.hide()


    def forgetCoalWindows(self):
        """
        For forget coal windows
        """
        self.groupBoxFlowTemp.hide()


    @pyqtSignature("bool")
    def slotSlidingWall(self, checked):
        """
        Private slot.

        Activates sliding wall boundary condition.

        @type checked: C{True} or C{False}
        @param checked: if C{True}, shows the QGroupBox sliding wall parameters.
        """
        self.groupBoxSliding.setFlat(not checked)

        if checked:
            self.boundary.setVelocityChoice("on")
            self.frameSlideVelocity.show()
            u, v, w = self.boundary.getVelocities()
        else:
            self.boundary.setVelocityChoice("off")
            self.frameSlideVelocity.hide()
            u, v, w = 0.0, 0.0, 0.0
        self.lineEditSlideU.setText(QString(str(u)))
        self.lineEditSlideV.setText(QString(str(v)))
        self.lineEditSlideW.setText(QString(str(w)))


    @pyqtSignature("const QString&")
    def slotVelocityU(self, text):
        """
        Private slot.

        If sliding wall activated, input U velocity component.

        @type text: C{QString}
        @param text: sliding wall U velocity component.
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setVelocityComponent(value, 'velocity_U')


    @pyqtSignature("const QString&")
    def slotVelocityV(self, text):
        """
        Private slot.

        If sliding wall activated, input V velocity component.

        @type text: C{QString}
        @param text: sliding wall V velocity component.
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setVelocityComponent(value, 'velocity_V')


    @pyqtSignature("const QString&")
    def slotVelocityW(self, text):
        """
        Private slot.

        If sliding wall activated, input W velocity component.

        @type text: C{QString}
        @param text: sliding wall W velocity component.
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setVelocityComponent(value, 'velocity_W')


    @pyqtSignature("")
    def slotRoughness(self):
        """
        Private slot.

        Selects if the wall is rought or smooth.
        """
        if self.radioButtonSmooth.isChecked():
            self.frameRoughness.hide()
            self.boundary.setRoughnessChoice('off')

        elif self.radioButtonRough.isChecked():
            self.frameRoughness.show()
            self.boundary.setRoughnessChoice('on')
            r = self.boundary.getRoughness()
            self.lineEditRoughCoef.setText(QString(str(r)))


    @pyqtSignature("const QString&")
    def slotRoughnessHeight(self, text):
        """
        Private slot.

        Input the roughness height for the selected wall.

        @type text: C{QString}
        @param text: roughness height.
        """
        r, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setRoughness(r)


    @pyqtSignature("const QString&")
    def slotChoiceVelocity(self, text):
        """
        Private slot.

        Input the velocity boundary type choice (norm, ).

        @type text: C{QString}
        @param text: velocity boundary type choice.
        """
        c = self.modelVelocity.dicoV2M[str(text)]
        log.debug("slotChoiceVelocity: %s " % c)
        self.boundary.setVelocityChoice(c)

        # update the value associated to the velocity boundary type choice
        v = self.boundary.getVelocity()
        self.lineEditVelocity.setText(QString(str(v)))

        cc = string.split(c, '+')[0]
        if cc  == 'norm':
            self.labelUnitVelocity.setText(QString(str('m/s')))
        elif cc == 'flow1':
            self.labelUnitVelocity.setText(QString(str('kg/s')))
        elif cc == 'flow2':
            self.labelUnitVelocity.setText(QString(str('m<sup>3</sup>/s')))

        if c[-9:] == 'direction':
            x = self.boundary.getDirection('direction_x')
            y = self.boundary.getDirection('direction_y')
            z = self.boundary.getDirection('direction_z')
            self.lineEditXVelocity.setText(QString(str(x)))
            self.lineEditYVelocity.setText(QString(str(y)))
            self.lineEditZVelocity.setText(QString(str(z)))
            self.frameVelocity.show()
        else:
            self.boundary.deleteDirectionNodes()
            self.frameVelocity.hide()


    @pyqtSignature("const QString&")
    def slotVelocityValue(self, text):
        """
        Private slot.

        New value associated to the velocity boundaru type.

        @type text: C{QString}
        @param text: value
        """
        v, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setVelocity(v)


    @pyqtSignature("const QString&")
    def slotDirX(self, text):
        """
        INPUT value into direction of inlet flow
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setDirection('direction_x', value)


    @pyqtSignature("const QString&")
    def slotDirY(self, text):
        """
        INPUT value into direction of inlet flow
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setDirection('direction_y', value)


    @pyqtSignature("const QString&")
    def slotDirZ(self, text):
        """
        INPUT value into direction of inlet flow
        """
        value, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.boundary.setDirection('direction_z', value)

    #======
    # COAL
    #======

    @pyqtSignature("const QString&")
    def slotCoalFlowType(self, text):
        """
        INPUT inlet type : 'air' or 'air + coal'
        """
        value = self.modelTypeInlet.dicoV2M[str(text)]
        log.debug("slotCoalFlowType value = %s " % value)
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            coal_model = Boundary('coal_inlet', label, self.case)
            velocity = model.getVelocity()
            self.lineEditAirVelocity.setText(QString(str(velocity)))
            temperature = coal_model.getAirTemperature()
            self.lineEditTemperature.setText(QString(str(temperature)))

            if value == 'airflow':
                self.groupBoxCoal.hide()
                self.groupBoxCoalMass.hide()
                #self.slotCoalChoiceVelocity()
##                 self.groupBoxDirection.show()
##                 self.groupBoxCoal.show()
            else :
                self.groupBoxCoal.show()
                self.groupBoxCoalMass.show()
                self.groupBoxDirection.show()
                self.groupBoxCoal.show()

                self.modelCoal.deleteAll()
                self.modelCoal.setBoundaryFromLabel(label)

                self.modelCoalMass.deleteAll()
                self.modelCoalMass.setBoundaryFromLabel(label)

                for coal in range(0, self.coalNumber):
                    # Flow and temperature
                    self.modelCoal.insertItem(self.tr("Coal Flow") + " " + str(coal+1),
                                              coal_model.getCoalFlow(coal), "kg/s",
                                              self.tr("Coal temperature") + " " + str(coal+1),
                                              coal_model.getCoalTemperature(coal), "K")

                for coal in range(0, self.coalNumber) :
                    lastValue = 0.
                    for coalClass in range(0, self.coalClassesNumber[coal]-1):
##                    lastValue += float(coal_model._getClassCoalRatio(coal, coalClass))
                        list = coal_model.getCoalRatios(coal)
                        lastValue += list[coalClass]

                        self.ratio[coal][coalClass] = list[coalClass]

##                             if (coalClass == self.coalClassesNumber[coal]-1) :
##                                 self.coale5[coal][coalClass].bind("<<Event>>",TkPage.Callback(self.getRatioLastClass, coal))
##                             else:
##                                 self.coale5[coal][coalClass].bind("<<Event>>",TkPage.Callback(self.getCoale5, coal, coalClass))
##                                 self.coale5[coal][coalClass].config(fg='black')

                    # last class is computed
                    coalClass = self.coalClassesNumber[coal]-1
                    lastValue = 100 - lastValue
                    self.ratio[coal][coalClass] = lastValue
                    #
                    self.getRatioLastClass(coal)

            self.modelCoalMass.setRatio(self.ratio)
            coal_model.setCoalType(value)


    def getRatioLastClass(self, coal):
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary('coal_inlet', label, self.case)
##             model.setClassCoalRatio(self.ratio[coal][self.coalClassesNumber[coal]-1].get(), coal, self.coalClassesNumber[coal]-1)


    @pyqtSignature("const QString&")
    def slotCoalChoiceVelocity(self, text):
        """
        INPUT choice of method of calculation of the velocity for air (coal)
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            choice = model.getVelocityChoice()
            coal_model = Boundary('coal_inlet', label, self.case)
            type = coal_model.getCoalType()
            new_type = type
            self.type_coal_flow = str(self.comboBoxTypeInlet.currentText())
            new_type = self.type_coal_flow

            coalchoiceflow = self.modelAirVelocity.dicoV2M[str(self.comboBoxAirVelocity.currentText())]
            log.debug("slotCoalChoiceVelocity coalchoiceflow = %s "%coalchoiceflow)
            if coalchoiceflow != choice:
                new_choice = coalchoiceflow
            else:
                new_choice = choice
            model.setVelocityChoice(new_choice)

            self.forgetCoalWindows()
            self.groupBoxCoal.show()

            #
            self.groupBoxTurbulence.show()

            self.setWindowsForCoalVelocityChoice(new_choice)
            val = model.getVelocity()
            self.lineEditAirVelocity.setText(QString(str(val)))


    @pyqtSignature("const QString&")
    def slotCoalAirFlow(self, text):
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            self.flow, ok = text.toDouble()
            model.setVelocity(self.flow)


    @pyqtSignature("const QString&")
    def slotTemperature(self, text):
        """
        INPUT air temperature
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary('coal_inlet', label, self.case)
            temperature, ok = text.toDouble()
            if self.sender().validator().state == QValidator.Acceptable:
                model.setAirTemperature(temperature)


    @pyqtSignature("const QString&")
    def slotCoalDirFlow(self, text):
        """
        INPUT Flow for the velocity
        """
        dico = {"normal" : 0, "vector" : 1, "formula" : 2}
        self.frameVelocity.hide()
        direction = self.modelDirVelocityCoal.dicoV2M[str(text)]
        log.debug("slotCoalDirFlow direction = %s "%direction)
        dir = dico[direction]

        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            if dir == 1:
                model.updateVelocityChoiceForDirection(dir)
                self.frameVelocity.show()
                x = model.getDirection('direction_x')
                y = model.getDirection('direction_y')
                z = model.getDirection('direction_z')
                self.lineEditXVelocityCoal.setText(QString(str(x)))
                self.lineEditYVelocityCoal.setText(QString(str(y)))
                self.lineEditZVelocityCoal.setText(QString(str(z)))
            else:
                model.updateVelocityChoiceForDirection(dir)
                model.deleteDirectionNodes()


    @pyqtSignature("const QString&")
    def slotCoalDirX(self, text):
        """
        INPUT value into direction of inlet flow
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            if model.getVelocityChoice()[-9:] == 'direction':
                value, ok = text.toDouble()
                if self.sender().validator().state == QValidator.Acceptable:
                    model.setDirection('direction_x', value)
##         else:
##             msg = self.tr("You must select one wall or inlet in the list.")
##             self.stbar.showMessage(msg, 2000)


    @pyqtSignature("const QString&")
    def slotCoalDirY(self, text):
        """
        INPUT value into direction of inlet flow
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            if model.getVelocityChoice()[-9:] == 'direction':
                value, ok = text.toDouble()
                if self.sender().validator().state == QValidator.Acceptable:
                    model.setDirection('direction_y', value)
##         else:
##             msg = self.tr("You must select one wall or inlet in the list.")
##             self.stbar.showMessage(msg, 2000)


    @pyqtSignature("const QString&")
    def slotCoalDirZ(self, text):
        """
        INPUT value into direction of inlet flow
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            if model.getVelocityChoice()[-9:] == 'direction':
                value, ok = text.toDouble()
                if self.sender().validator().state == QValidator.Acceptable:
                    model.setDirection('direction_z', value)
##         else:
##             msg = self.tr("You must select one wall or inlet in the list.")
##             self.stbar.showMessage(msg, 2000)


    #============
    # TURBULENCE
    #============

    @pyqtSignature("const QString&")
    def slotChoiceTurbu(self, text):
        """
        INPUT choice of method of calculation of the turbulence
        """
        turb_choice = self.modelTurbulence.dicoV2M[str(text)]
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            model.setTurbulenceChoice(turb_choice)

            self.frameTurbDiameter.hide()
            self.frameTurbIntensity.hide()

            if turb_choice  == 'hydraulic_diameter':
                self.frameTurbDiameter.show()
                d = model.getHydraulicDiameter()
                self.lineEditDiameter.setText(QString(str(d)))
            elif turb_choice == 'turbulent_intensity':
                self.frameTurbIntensity.show()
                i = model.getTurbulentIntensity()
                self.lineEditIntensity.setText(QString(str(i)))
                d = model.getHydraulicDiameter()
                self.lineEditDiameterIntens.setText(QString(str(d)))


    @pyqtSignature("const QString&")
    def slotDiam(self, text):
        """
        INPUT hydraulic diameter
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            diam, ok = text.toDouble()
            if self.sender().validator().state == QValidator.Acceptable:
                model.setHydraulicDiameter(diam)


    @pyqtSignature("const QString&")
    def slotIntensity(self, text):
        """
        INPUT turbulent intensity
        """
        cindex = self.treeViewBoundaries.currentIndex()
        if cindex != (-1,-1):
            row = cindex.row()
            label, zone, nature, local = self.modelBoundaries.getItem(row)
            model = Boundary(nature, label, self.case)
            intens, ok = text.toDouble()
            if self.sender().validator().state == QValidator.Acceptable:
                model.setTurbulentIntensity(intens)


    def tr(self, text):
        """
        Translation
        """
        return text


#-------------------------------------------------------------------------------
# Testing part
#-------------------------------------------------------------------------------


if __name__ == "__main__":
    pass


#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
