# -*- coding: iso-8859-1 -*-
#
#-------------------------------------------------------------------------------
#
#     This file is part of the Code_Saturne User Interface, element of the
#     Code_Saturne CFD tool.
#
#     Copyright (C) 1998-2008 EDF S.A., France
#
#     contact: saturne-support@edf.fr
#
#     The Code_Saturne User Interface is free software; you can redistribute it
#     and/or modify it under the terms of the GNU General Public License
#     as published by the Free Software Foundation; either version 2 of
#     the License, or (at your option) any later version.
#
#     The Code_Saturne User Interface is distributed in the hope that it will be
#     useful, but WITHOUT ANY WARRANTY; without even the implied warranty
#     of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with the Code_Saturne Kernel; if not, write to the
#     Free Software Foundation, Inc.,
#     51 Franklin St, Fifth Floor,
#     Boston, MA  02110-1301  USA
#
#-------------------------------------------------------------------------------

"""
This module defines the values of reference.

This module contains the following classes and function:
- MobilMeshView
"""

#-------------------------------------------------------------------------------
# Library modules import
#-------------------------------------------------------------------------------

import logging

#-------------------------------------------------------------------------------
# Third-party modules
#-------------------------------------------------------------------------------

from PyQt4.QtCore import *
from PyQt4.QtGui  import *

#-------------------------------------------------------------------------------
# Application modules import
#-------------------------------------------------------------------------------

from Base.Toolbox import GuiParam
from MobilMeshForm import Ui_MobilMeshForm
import Base.QtPage as QtPage
from Pages.MobilMeshModel import MobilMeshModel

#-------------------------------------------------------------------------------
# log config
#-------------------------------------------------------------------------------

logging.basicConfig()
log = logging.getLogger("MobilMeshView")
log.setLevel(GuiParam.DEBUG)

#-------------------------------------------------------------------------------
# Main class
#-------------------------------------------------------------------------------

class MobilMeshView(QWidget, Ui_MobilMeshForm):
    """
    Class to open Page.
    """
    def __init__(self, parent, case):
        """
        Constructor
        """
        QWidget.__init__(self, parent)

        Ui_MobilMeshForm.__init__(self)
        self.setupUi(self)

        self.case = case
        self.mdl = MobilMeshModel(self.case)

        # Combo model

        self.modelVISCOSITY = QtPage.ComboModel(self.comboBoxVISCOSITY,2,1)

        self.modelVISCOSITY.addItem(self.tr("isotropic"), 'isotrop')
        self.modelVISCOSITY.addItem(self.tr("orthotropic"), 'orthotrop')

        # Connections
        self.connect(self.groupBoxALE, SIGNAL("clicked(bool)"), self.slotMethod)
        self.connect(self.lineEditNALINF, SIGNAL("textChanged(const QString &)"), self.slotNalinf)
        self.connect(self.lineEditNALIMX, SIGNAL("textChanged(const QString &)"), self.slotNalimx)
        self.connect(self.lineEditEPALIM, SIGNAL("textChanged(const QString &)"), self.slotEpalim)
        self.connect(self.comboBoxVISCOSITY, SIGNAL("activated(const QString&)"), self.slotViscosityType)

        # Validators

        validatorNALINF = QtPage.IntValidator(self.lineEditNALINF, min=0)
        self.lineEditNALINF.setValidator(validatorNALINF)

        validatorNALIMX = QtPage.IntValidator(self.lineEditNALIMX, min=1)
        self.lineEditNALIMX.setValidator(validatorNALIMX)

        validatorEPALIM = QtPage.DoubleValidator(self.lineEditEPALIM, min=0.0)
        validatorEPALIM.setExclusiveMin(True)
        self.lineEditEPALIM.setValidator(validatorEPALIM)

        if self.mdl.getMethod() == 'on':
            self.groupBoxALE.setChecked(True)
            checked = True
        else:
            self.groupBoxALE.setChecked(False)
            checked = False

        self.slotMethod(checked)


    @pyqtSignature("bool")
    def slotMethod(self, checked):
        """
        Private slot.

        Activates ALE method.

        @type checked: C{True} or C{False}
        @param checked: if C{True}, shows the QGroupBox ALE parameters
        """
        self.groupBoxALE.setFlat(not checked)
        if checked:
            self.frame.show()
            self.mdl.setMethod ("on")
            nalinf = self.mdl.getSubIterations()
            self.lineEditNALINF.setText(QString(str(nalinf)))
            nalimx = self.mdl.getMaxIterations()
            self.lineEditNALIMX.setText(QString(str(nalimx)))
            epalim = self.mdl.getPrecision()
            self.lineEditEPALIM.setText(QString(str(epalim)))
            value = self.mdl.getViscosity()
            self.modelVISCOSITY.setItem(str_model=value)
        else:
            self.frame.hide()
            self.mdl.setMethod("off")


    @pyqtSignature("const QString&")
    def slotNalinf(self, text):
        """
        Input viscosity type of mesh : isotrop or orthotrop.
        """        
        nalinf, ok = text.toInt()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setSubIterations(nalinf)


    @pyqtSignature("const QString&")
    def slotNalimx(self, text):
        """
        Input viscosity type of mesh : isotrop or orthotrop.
        """
        nalimx, ok = text.toInt()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setMaxIterations(nalimx)


    @pyqtSignature("const QString&")
    def slotEpalim(self, text):
        """
        Input viscosity type of mesh : isotrop or orthotrop.
        """
        epalim, ok = text.toDouble()
        if self.sender().validator().state == QValidator.Acceptable:
            self.mdl.setPrecision(epalim)


    @pyqtSignature("const QString&")
    def slotViscosityType(self, text):
        """
        Input viscosity type of mesh : isotrop or orthotrop.
        """
        self.viscosity_type = self.modelVISCOSITY.dicoV2M[str(text)]
        visco = self.viscosity_type
        self.mdl.setViscosity(visco)
        return visco


    def tr(self, text):
        """
        Translation
        """
        return text 


#-------------------------------------------------------------------------------
# Testing part
#-------------------------------------------------------------------------------


if __name__ == "__main__":
    pass


#-------------------------------------------------------------------------------
# End
#-------------------------------------------------------------------------------
