/*
#   vorbisdec.c: ogg vorbis file decoder for xlplayer
#   Copyright (C) 2007 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <vorbis/vorbisfile.h>
#include "xlplayer.h"
#include "vorbisdec.h"

#define TRUE 1
#define FALSE 0
#define ACCEPTED 1
#define REJECTED 0

static void vorbisdecode_init(struct xlplayer *xlplayer)
   {
   struct vorbisdecode_vars *self = xlplayer->dec_data;
   int src_error;
   
   self->ibuf = NULL;
   self->flbuf = NULL;
   if (ov_test_open(&(self->vf)))
      {
      fprintf(stderr, "vorbisdecode_init: ov_test_open reports a problem with the file '%s'\n", xlplayer->pathname);
      ov_clear(&(self->vf));
      xlplayer->playmode = PM_STOPPED;
      xlplayer->command = CMD_COMPLETE;
      free(self);
      return;
      }
   switch (ov_time_seek(&(self->vf), (double)xlplayer->seek_s))
      {
      case 0:
         break;
      case OV_ENOSEEK:
         fprintf(stderr, "vorbisdecode_init: file '%s' is not seekable\n", xlplayer->pathname);	/* non fatal error */
         break;
      default:
         fprintf(stderr, "vorbisdecode_init: error occured during seek of file '%s'\n", xlplayer->pathname);
         ov_clear(&(self->vf));
         xlplayer->playmode = PM_STOPPED;
         xlplayer->command = CMD_COMPLETE;
         free(self);
         return;
      }
   self->vi = ov_info(&(self->vf), -1);
   if (self->vi->rate != xlplayer->samplerate)
      {
      fprintf(stderr, "vorbisdecode_init: configuring resampler\n");
      xlplayer->src_state = src_new(xlplayer->rsqual, self->vi->channels, &src_error);
      if (src_error)
         {
         fprintf(stderr, "vorbisdecode_init: src_new reports %s\n", src_strerror(src_error));
         ov_clear(&(self->vf));
         xlplayer->playmode = PM_STOPPED;
         xlplayer->command = CMD_COMPLETE;
         free(self);
         return;
         }
      xlplayer->src_data.output_frames = 0;
      xlplayer->src_data.data_in = xlplayer->src_data.data_out = NULL;
      xlplayer->src_data.src_ratio = (double)xlplayer->samplerate / (double)self->vi->rate;
      xlplayer->src_data.end_of_input = 0;
      self->resample = TRUE;
      }
   else
      self->resample = FALSE;
   }
   
static void vorbisdecode_play(struct xlplayer *xlplayer)
   {
   struct vorbisdecode_vars *self = xlplayer->dec_data;
   long ov_retval, pcmsamples;
   int bitstream = 0, src_error, i, j, channels;
   float **pcm, *lo, *ro, *li, *ri, *di, gain;
   
   if (self->resample)
      {
      ov_retval = ov_read_float(&(self->vf), &pcm, 4096, &bitstream);
      switch (ov_retval)
         {
         case OV_HOLE:
            fprintf(stderr, "xlplayer: %s warning: hole in stream %s\n", xlplayer->playername, xlplayer->pathname);
            return;
         case OV_EBADLINK:
            fprintf(stderr, "xlplayer: %s warning: invalid stream section or corrupt link %s\n", xlplayer->playername, xlplayer->pathname);
            return;
         default:
            if (ov_retval < 0L)
               {
               fprintf(stderr, "xlplayer: %s warning: ov_read reports error code %ld\n", xlplayer->playername, pcmsamples);
               xlplayer->playmode = PM_EJECTING;
               return;
               }
            self->vi = ov_info(&(self->vf), -1);
            channels = self->vi->channels;
            pcmsamples = ov_retval;
            xlplayer->src_data.input_frames = pcmsamples;
            xlplayer->src_data.end_of_input = pcmsamples == 0;
            xlplayer->src_data.data_in = di = realloc(xlplayer->src_data.data_in, xlplayer->src_data.input_frames * channels * sizeof (float));
            xlplayer->src_data.output_frames = (int)(xlplayer->src_data.input_frames * xlplayer->src_data.src_ratio) + 2 + (512 * xlplayer->src_data.end_of_input);
            xlplayer->src_data.data_out = realloc(xlplayer->src_data.data_out, xlplayer->src_data.output_frames * channels * sizeof (float));
            for (i = 0; i < pcmsamples; i++)
               for (j=0; j < channels; j++)
                  *di++ = pcm[j][i];
            if ((src_error = src_process(xlplayer->src_state, &(xlplayer->src_data))))
               {
               fprintf(stderr, "xlplayer: %s src_process reports - %s\n", xlplayer->playername, src_strerror(src_error));
               xlplayer->playmode = PM_EJECTING;
               return;
               }
            xlplayer_demux_channel_data(xlplayer, xlplayer->src_data.data_out, xlplayer->src_data.output_frames_gen, self->vi->channels);
            xlplayer_write_channel_data(xlplayer);
            if (xlplayer->src_data.end_of_input)
               {
               xlplayer->playmode = PM_EJECTING;
               return;
               }
         }
      }
   else
      {
      ov_retval = ov_read_float(&(self->vf), &pcm, 4096, &bitstream);
      switch (ov_retval)
         {
         case OV_HOLE:
            fprintf(stderr, "xlplayer: %s warning: hole in stream; probably harmless %s\n", xlplayer->playername, xlplayer->pathname);
            return;
         case OV_EBADLINK:
            fprintf(stderr, "xlplayer: %s warning: invalid stream section or corrupt link %s\n", xlplayer->playername, xlplayer->pathname);
            return;
         case 0:
            xlplayer->playmode = PM_EJECTING;
            return;
         default:
            if (ov_retval < 0L)
               {
               fprintf(stderr, "xlplayer: %s warning: ov_read reports error code %ld\n", xlplayer->playername, pcmsamples);
               xlplayer->playmode = PM_EJECTING;
               return;
               }
            self->vi = ov_info(&(self->vf), -1);
            xlplayer->op_buffersize = (pcmsamples = ov_retval) * sizeof (float);
            xlplayer->leftbuffer = lo = realloc(xlplayer->leftbuffer, xlplayer->op_buffersize);
            xlplayer->rightbuffer = ro = realloc(xlplayer->rightbuffer, xlplayer->op_buffersize);
            xlplayer->op_buffersize = pcmsamples * sizeof (float);
            if (self->vi->channels == 1)
               li = ri = &pcm[0][0];
            else
               {
               li = &pcm[0][0];
               ri = &pcm[1][0];
               }
            while (pcmsamples--)
               {
               gain = xlplayer_get_next_gain(xlplayer);
               *lo++ = *li++ * gain;
               *ro++ = *ri++ * gain;
               }
            xlplayer_write_channel_data(xlplayer);
         }
      }
   }
   
static void vorbisdecode_eject(struct xlplayer *xlplayer)
   {
   struct vorbisdecode_vars *self = xlplayer->dec_data;
   
   ov_clear(&(self->vf));
   if (self->flbuf)
      free(self->flbuf);
   if (self->ibuf)
      free(self->ibuf);
   if (self->resample)
      {
      if (xlplayer->src_data.data_in)
         free(xlplayer->src_data.data_in);
      if (xlplayer->src_data.data_out)
         free(xlplayer->src_data.data_out);
      xlplayer->src_state = src_delete(xlplayer->src_state);
      }
   }

int vorbisdecode_reg(struct xlplayer *xlplayer)
   {
   struct vorbisdecode_vars *self;
   
   if (!(self = xlplayer->dec_data = malloc(sizeof (struct vorbisdecode_vars))))
      {
      fprintf(stderr, "vorbisdecode_reg: malloc failure\n");
      return REJECTED;
      }
   if (!(self->fp = fopen(xlplayer->pathname, "r")))
      {
      fprintf(stderr, "vorbisdecode_test_vorbisness: unable to open media file %s\n", xlplayer->pathname);
      free(self);
      return REJECTED;
      }
   if (!ov_test(self->fp, &(self->vf), NULL, 0))
      {
      xlplayer->dec_init = vorbisdecode_init;
      xlplayer->dec_play = vorbisdecode_play;
      xlplayer->dec_eject = vorbisdecode_eject;
      return ACCEPTED;
      }
   ov_clear(&(self->vf));
   free(self);
   return REJECTED;
   }
