/*
#   xlplayer.c: a xine-lib based player module for idjc
#   n.b. ogg and flac files are decoded directly due to xine-lib limitations (can't seek on them)
#   Copyright (C) 2006 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/
#define XINE_ENABLE_EXPERIMENTAL_FEATURES

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <jack/jack.h>
#include <jack/ringbuffer.h>
#include <xine.h>
#include <xine/xineutils.h>
#include <vorbis/vorbisfile.h>
#include <FLAC/all.h>
#include <samplerate.h>
#include "xlplayer.h"

#define TRUE 1
#define FALSE 0

typedef jack_default_audio_sample_t sample_t;

static sample_t *fadeinlookuptable;
static size_t fadeinlookuptable_numsamples;

/* get_playertype: return one of the playertype_t constants based on the file type */
static int get_playertype(struct xlplayer *self, OggVorbis_File *vf, FLAC__StreamMetadata *metainfo)
   {
   FILE *fp;
   
   if (!(fp = fopen(self->pathname, "r")))
      {
      fprintf(stderr, "xlplayer: %s unable to open '%s'\n", self->playername, self->pathname);
      return XL_BADFILE;
      }
   if (!ov_test(fp, vf, NULL, 0))
      return XL_VORBIS;
   else
      ov_clear(vf);
   if (FLAC__metadata_get_streaminfo(self->pathname, metainfo))
      return XL_FLAC;
   return XL_XINE;
   }

/* make_audio_to_float: convert the audio to the format used by jack and libsamplerate */
static float *make_audio_to_float(struct xlplayer *self, float *buffer, uint8_t *data, int num_samples, int bits_per_sample, int num_channels)
   {
   int num_bytes;
   int i;
   uint32_t msb_mask;
   uint32_t neg_mask;
   uint32_t holder;
   uint32_t mult;
   float *fptr = buffer;
   float fscale;
   const float half_randmax = (float)(RAND_MAX >> 1);
   float dscale;
 
   msb_mask = 1UL << (bits_per_sample - 1);		/* negative number detector */
   neg_mask = (uint32_t)((~0UL) << (bits_per_sample));	/* negative number maker */
   fscale = 1.0F/(float)msb_mask;			/* multiplier to make the floating point range -1 to +1 */
   dscale = 0.25F / half_randmax * fscale;

   if (bits_per_sample > 32)
      {
      memset(buffer, 0, sizeof (sample_t) * num_samples * num_channels);
      }
   else
      {
      while (num_samples--)
         {
         for (i = 0; i < num_channels; i++)
            {
            for (num_bytes = (bits_per_sample + 7) >> 3, mult = 1, holder = 0; num_bytes--; mult <<=8)
               {
               holder |= ((uint32_t)*data++) * mult;
               }
            if (holder & msb_mask)
               holder |= neg_mask;
            if (self->dither && bits_per_sample < 20)
               /* performs triangular dither no noise shaping to speak of */
               *fptr++ = (((float)(int32_t)holder) * fscale) + 
               (((((float)rand_r(&self->seed)) - half_randmax) +
               (((float)rand_r(&self->seed)) - half_randmax)) * dscale); 
            else
               *fptr++ = ((float)(int32_t)holder) * fscale;
            }
         }
      }
   return buffer;
   }

/* get_next_gain: compute the gain of the next sample */
/* used to fade in the audio when not starting from the beginning */
inline static sample_t get_next_gain(struct xlplayer *self)
   {
   if (self->fadein_index >= 0)
      {
      if (self->fadein_index == fadeinlookuptable_numsamples)
         {
         self->fadein_index = -1;
         return 1.0F;
         }
      return fadeinlookuptable[self->fadein_index++];
      }
   else
      return 1.0F;
   }

/* demux_channel_data: this is where down/upmixing is performed - audio split to 2 channels */
static void demux_channel_data(struct xlplayer *self, sample_t *buffer, int num_samples, int num_channels)
   {
   self->op_buffersize = num_samples * sizeof (sample_t);
   int i;
   sample_t *lc, *rc, *src, gain;
   
   if (!(self->leftbuffer = realloc(self->leftbuffer, self->op_buffersize)))
      {
      fprintf(stderr, "xlplayer: malloc failure");
      exit(5);
      }
   if (!(self->rightbuffer = realloc(self->rightbuffer, self->op_buffersize)))
      {
      fprintf(stderr, "xlplayer: malloc failure");
      exit(5);
      }
   switch (num_channels)
      {
      case 0:
         break;			/* this is a wtf case */
      case 1:
         for (lc = self->leftbuffer, src = buffer, i = 0; i < num_samples; i++)
            {
            gain = get_next_gain(self);	/* used for fade-in */
            *lc++ = *src++ * gain;
            }
         memcpy(self->rightbuffer, self->leftbuffer, self->op_buffersize);
         break;
      case 2:
         for (lc = self->leftbuffer, rc = self->rightbuffer, src = buffer, i = 0; i < num_samples; i++)
            {
            gain = get_next_gain(self);
            *lc++ = *src++ * gain;	/* stereo mix is a simple demultiplex job */
            *rc++ = *src++ * gain;
            }
         break;
      case 3:
         for (lc = self->leftbuffer, rc = self->rightbuffer, src = buffer, i = 0; i < num_samples; i++)
            {
            gain = get_next_gain(self) * 0.5F;
            *lc = (*src++) * gain;	/* do the left and right channels */
            *rc = (*src++) * gain;
            *(lc++) += (*src) *gain;	/* downmix the middle channel to the left and right one */
            *(rc++) += (*src++) *gain;
            }
         break;
      case 4:
         for (lc = self->leftbuffer, rc = self->rightbuffer, src = buffer, i = 0; i < num_samples; i++, src += 4)
            {
            gain = get_next_gain(self) * 0.5F;
            *lc++ = (src[0] + src[3]) * gain;
            *rc++ = (src[2] + src[4]) * gain;
            }
         break;
      case 5:
         for (lc = self->leftbuffer, rc = self->rightbuffer, src = buffer, i = 0; i < num_samples; i++, src += 5)
            {
            gain = get_next_gain(self) * 0.5F;
            *lc++ = (src[0] + src[3]) * gain;	/* this is for 4.1 channels with sub discarded */
            *rc++ = (src[2] + src[4]) * gain;
            }
         break;
      case 6:
         for (lc = self->leftbuffer, rc = self->rightbuffer, src = buffer, i = 0; i < num_samples; i++, src += 6)
            {
            gain = get_next_gain(self) * 0.33333333F;
            *lc++ = (src[0] + src[3] + src[4]) * gain;	/* this is for 5.1 channels */
            *rc++ = (src[2] + src[4] + src[5]) * gain;   /* sub discarded */
            }
         break;
      }
   }

static void write_channel_data(struct xlplayer *self)
   {
   u_int32_t samplecount;
   
   if (self->op_buffersize > jack_ringbuffer_write_space(self->right_ch))
      {
      self->write_deferred = TRUE;	/* prevent further accumulation of data that would clobber */
      usleep(20000);
      }
   else
      {
      jack_ringbuffer_write(self->left_ch, (char *)self->leftbuffer, self->op_buffersize);
      jack_ringbuffer_write(self->right_ch, (char *)self->rightbuffer, self->op_buffersize);
      samplecount = self->op_buffersize / sizeof (sample_t);
      self->samples_written += samplecount;
      self->sleep_samples += samplecount; 
      self->write_deferred = FALSE;
      if (self->sleep_samples > 6000)
         {
         self->sleep_samples = 0;
         usleep(10000);
         }
      }
   }

/* xlplayer_update_progress_time_ms: a rather ugly calculator of where the play progress is up to */
static u_int32_t xlplayer_update_progress_time_ms(struct xlplayer *self)
   {
   int32_t rb_time_ms;  /* the amount of time it would take to play all the samples in the buffer */
   int32_t progress;
   
   rb_time_ms = jack_ringbuffer_read_space(self->right_ch) / sizeof (sample_t) * 1000 / self->samplerate;
   progress = self->samples_written * 1000 / self->samplerate - rb_time_ms + self->seek_s * 1000;
   if (progress >= 0)
      return self->play_progress_ms = progress;
   else
      return self->play_progress_ms = 0;
   }

static void make_flac_audio_to_float(struct xlplayer *self, float *flbuf, const FLAC__int32 * const inputbuffer[], unsigned int numsamples, unsigned int bits_per_sample, unsigned int numchannels)
   {
   int sample, channel, shiftvalue = 32 - bits_per_sample;
   const float half_randmax = (float)(RAND_MAX >> 1);
   float dither;
   float dscale;
   
   if (!self->dither || bits_per_sample >= 20)
      {
      for (sample = 0; sample < numsamples; sample++)
         for (channel = 0; channel < numchannels; channel++)
            *flbuf++ = ((float)(inputbuffer[channel][sample] << shiftvalue)) / 2147483648.0F;
      }
   else
      {
      dscale = 0.25F / (half_randmax * powf(2.0F, (float)bits_per_sample));
      for (sample = 0; sample < numsamples; sample++)
         for (channel = 0; channel < numchannels; channel++)
            {
            dither = ((((float)rand_r(&self->seed)) - half_randmax) +
                     (((float)rand_r(&self->seed)) - half_randmax)) * dscale;
            *flbuf++ = ((float)(inputbuffer[channel][sample] << shiftvalue)) / 2147483648.0F + dither;
            }
      }
   }

static FLAC__StreamDecoderWriteStatus flac_writer_callback(const FLAC__FileDecoder *decoder, const FLAC__Frame *frame, const FLAC__int32 * const inputbuffer[], void *client_data)
   {
   struct xlplayer *self = client_data;
   SRC_DATA *src_data = self->src_data;
   int src_error;

   if (self->suppress_flac_output == FALSE)
      {
      if (self->src_state)
         {
         if (frame->header.number_type == FLAC__FRAME_NUMBER_TYPE_FRAME_NUMBER && frame->header.number.frame_number == 0)
            {
            fprintf(stderr, "xlplayer: performance warning -- can't determine if a block is the last one or not for this file\n");
            }
         else
            {
            if (frame->header.number.sample_number + frame->header.blocksize == self-> totalsamples)
               src_data->end_of_input = TRUE;
            }
         src_data->input_frames = frame->header.blocksize;
         src_data->data_in = realloc(src_data->data_in, src_data->input_frames * frame->header.channels * sizeof (float));
         src_data->output_frames = (int)(src_data->input_frames * src_data->src_ratio) + 2 + (512 * src_data->end_of_input);
         src_data->data_out = realloc(src_data->data_out, src_data->output_frames * frame->header.channels * sizeof (float));
         make_flac_audio_to_float(self, src_data->data_in, inputbuffer, frame->header.blocksize, frame->header.bits_per_sample, frame->header.channels);
         if ((src_error = src_process(self->src_state, src_data)))
            {
            fprintf(stderr, "xlplayer: %s src_process reports - %s\n", self->playername, src_strerror(src_error));
            self->playmode = XL_EJECTING;
            return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
            }
         demux_channel_data(self, src_data->data_out, src_data->output_frames_gen, frame->header.channels);
         }
      else
         {
         if ((self->flbuf = realloc(self->flbuf, sizeof (float) * frame->header.blocksize * frame->header.channels)) == NULL)
            {
            fprintf(stderr, "xlplayer: malloc failure\n");
            exit(5);
            }
         make_flac_audio_to_float(self, self->flbuf, inputbuffer, frame->header.blocksize, frame->header.bits_per_sample, frame->header.channels);
         demux_channel_data(self, self->flbuf, frame->header.blocksize, frame->header.channels);
         }
      write_channel_data(self);
      }
   return FLAC__STREAM_DECODER_WRITE_STATUS_CONTINUE;
   }

static void flac_metadata_callback(const FLAC__FileDecoder *decoder, const FLAC__StreamMetadata *md, void *client_data)
   {
   /* do nothing with the metadata */
   }

static void flac_error_callback(const FLAC__FileDecoder *decoder,FLAC__StreamDecoderErrorStatus se, void *client_data)
   {
   struct xlplayer *self = client_data;
         
   fprintf(stderr, "xlplayer: flac decoder detected an error in file %s\n", self->pathname);
   }

static void *xlplayer_main(struct xlplayer *self)
   {
   xine_t              *xine;		/* xine-lib stuff */
   xine_stream_t       *stream = NULL;
   xine_audio_port_t   *ao_port = NULL;
   xine_audio_frame_t   frame;
   char		       *configfile;
   const char 	       *xinehomedir;
            
   OggVorbis_File vf;
   vorbis_info *vi = NULL;
   long pcmsamples;
   uint8_t *ibuf = NULL;
   int bitstream = 0;
   
   FLAC__FileDecoder *decoder;		/* flac stuff */
   FLAC__StreamMetadata metainfo;
   int decoderstate;
      
   SRC_STATE  *src_state = NULL;	/* resampler stuff */
   SRC_DATA    src_data;
   int	       src_error;
   int         resampling = FALSE;

   float      *flbuf = NULL;		/* generally used */
   int	       playertype = XL_BADFILE;
   
   if ((xine = xine_new()))
      {
      if (!(configfile = malloc(strlen(xinehomedir = xine_get_homedir()) + 14)))
         {
         fprintf(stderr, "malloc failure\n");
         exit(5);
         }
      sprintf(configfile, "%s%s", xinehomedir, "/.xine/config");
      xine_config_load(xine, configfile);
      xine_init(xine);
      }
   else
      fprintf(stderr, "xlplayer: %s could not initialise xine xine_new failed\n", self->playername);

   if (!(decoder = FLAC__file_decoder_new()))
      {
      fprintf(stderr, "xlplayer: %s could not initialise flac decoder\n", self->playername);
      }

   for(;;)
      {
      switch (self->command)
         {
         case XL_COMPLETE:
            break;
         case XL_PLAY:
            self->playmode = XL_INITIATE;
            break;
         case XL_PLAYMANY:
            self->pathname = self->playlist[self->playlistindex = 0];
            self->playmode = XL_INITIATE;
            break;
         case XL_EJECT:
            if (self->playmode != XL_STOPPED)
               self->playmode = XL_EJECTING;
            else
               {
               self->jack_flush = TRUE;
               while (self->jack_is_flushed == 0 && *(self->jack_shutdown_f) == FALSE)
                  usleep(10000);
               self->jack_is_flushed = 0;
               self->command = XL_COMPLETE;
               }
            break;
         case XL_EXIT:
            if (self->playlist)
               free(self->playlist);
            if (xine)
               xine_exit(xine);
            if (decoder)
               FLAC__file_decoder_delete(decoder);
            if (self->flbuf)
               free(self->flbuf);
            free(configfile);
            self->command = XL_COMPLETE;
            return 0;				/* exit point */
         }
      switch (self->playmode)
         {
         case XL_STOPPED:
            usleep(10000);
            continue;
         case XL_INITIATE:
            self->initial_audio_context = -1;	/* pre-select failure return code */
            playertype = get_playertype(self, &vf, &metainfo);
            switch (playertype)
               {
               case XL_BADFILE:
                  self->playmode = XL_STOPPED;
                  self->command = XL_COMPLETE;
                  continue;
               case XL_VORBIS:
                  if (ov_test_open(&vf))
                     {
                     fprintf(stderr, "xlplayer: %s ov_test_open reports a problem with the file '%s'\n", self->playername, self->pathname);
                     ov_clear(&vf);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  switch (ov_time_seek(&vf, (double)self->seek_s))
                     {
                     case 0:
                        break;
                     case OV_ENOSEEK:
                        fprintf(stderr, "xlplayer: %s file '%s' is not seekable\n", self->playername, self->pathname);	/* non fatal error */
                        break;
   	             default:
      			fprintf(stderr, "xlplayer: %s error occured during seek of file '%s'\n", self->playername, self->pathname);
      			ov_clear(&vf);
                        self->playmode = XL_STOPPED;
                        self->command = XL_COMPLETE;
                        continue;
                     }
		  vi = ov_info(&vf, -1);
                  if (vi->rate != self->samplerate)
                     {
                     fprintf(stderr, "xlplayer: %s configuring resampler\n", self->playername);
                     src_state = src_new(SRC_LINEAR, vi->channels, &src_error);
                     if (src_error)
                        {
                        fprintf(stderr, "xlplayer: %s src_new reports - %s\n", self->playername, src_strerror(src_error));
                        ov_clear(&vf);
                        self->playmode = XL_STOPPED;
                        self->playmode = XL_COMPLETE;
                        continue;
                        }
                     src_data.output_frames = 0;
                     src_data.data_in = src_data.data_out = NULL;
                     src_data.src_ratio = (double)self->samplerate / (double)vi->rate;
                     src_data.end_of_input = 0;
                     resampling = TRUE;
                     }
                  else
                     resampling = FALSE;
		  break;
               case XL_FLAC:
                  if (metainfo.data.stream_info.sample_rate != self->samplerate)
                     {
                     fprintf(stderr, "xlplayer: %s configuring resampler\n", self->playername);
                     src_state = src_new(SRC_LINEAR, metainfo.data.stream_info.channels, &src_error);
                     if (src_error)
                        {
                        fprintf(stderr, "xlplayer: %s src_new reports - %s\n", self->playername, src_strerror(src_error));
                        ov_clear(&vf);
                        self->playmode = XL_STOPPED;
                        self->playmode = XL_COMPLETE;
                        continue;
                        }
                     src_data.output_frames = 0;
                     src_data.data_in = src_data.data_out = NULL;
                     src_data.src_ratio = (double)self->samplerate / (double)metainfo.data.stream_info.sample_rate;
                     src_data.end_of_input = 0;
                     self->src_state = src_state;
                     self->src_data = &src_data;
                     self->totalsamples = metainfo.data.stream_info.total_samples; 
                     resampling = TRUE;
                     }
                  else
                     {
                     self->src_state = NULL;		/* indicate no resampling */
                     resampling = FALSE;
                     }
                  FLAC__file_decoder_set_client_data(decoder, self);
                  FLAC__file_decoder_set_write_callback(decoder, flac_writer_callback);
                  FLAC__file_decoder_set_error_callback(decoder, flac_error_callback);
                  FLAC__file_decoder_set_metadata_callback(decoder, flac_metadata_callback);
                  FLAC__file_decoder_set_filename(decoder, self->pathname);
                  if ((decoderstate = FLAC__file_decoder_init(decoder)) != FLAC__FILE_DECODER_OK)
                     {
                     fprintf(stderr, "xlplayer: %s error during flac player initialisation\n", self->playername);
                     FLAC__file_decoder_finish(decoder);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  if (self->seek_s)
                     {
                     self->suppress_flac_output = TRUE;		/* prevent seek noise */
                     FLAC__file_decoder_seek_absolute(decoder, ((FLAC__uint64)self->seek_s) * ((FLAC__uint64)metainfo.data.stream_info.sample_rate));
                     self->suppress_flac_output = FALSE;
                     }
                  break;
               case XL_XINE:
                  if (!xine)
                     {
                     fprintf(stderr, "xlplayer: %s xine disabled\n", self->playername);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  ao_port = xine_new_framegrab_audio_port(xine); 
                  stream = xine_stream_new(xine, ao_port, NULL);
                  if (!xine_open(stream, self->pathname)) 
                     {
                     fprintf(stderr, "xlplayer: %s unable to open '%s'\n", self->playername, self->pathname);
                     xine_dispose(stream);
                     xine_close_audio_driver(xine, ao_port);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  if (!xine_get_stream_info(stream, XINE_STREAM_INFO_AUDIO_HANDLED))
                     {
                     fprintf(stderr, "xlplayer: %s no audio codec for '%s'\n", self->playername, self->pathname);
                     xine_dispose(stream);
                     xine_close_audio_driver(xine, ao_port);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  if (!xine_get_stream_info(stream, XINE_STREAM_INFO_SEEKABLE))
                     {
                     fprintf(stderr, "xlplayer: %s %s is not seekable\n", self->playername, self->pathname); 
                     }   
                  if (!xine_play(stream, 0, self->seek_s * 1000))
                     {
                     fprintf(stderr, "xlplayer: %s failed to play '%s'\n", self->playername, self->pathname);
                     xine_dispose(stream);
                     xine_close_audio_driver(xine, ao_port);

                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  if (!xine_get_next_audio_frame(ao_port, &frame))
                     {
                     fprintf(stderr, "xlplayer: %s failed to get an audio frame for '%s'\n", self->playername, self->pathname);
                     xine_close(stream);
                     xine_dispose(stream);
                     xine_close_audio_driver(xine, ao_port);
                     self->playmode = XL_STOPPED;
                     self->command = XL_COMPLETE;
                     continue;
                     }
                  if (frame.sample_rate != self->samplerate)
                     {
                     fprintf(stderr, "xlplayer: %s configuring resampler\n", self->playername);
                     src_state = src_new(SRC_LINEAR, frame.num_channels, &src_error);
                     if (src_error)
                        {
                        fprintf(stderr, "xlplayer: %s src_new reports %s\n", self->playername, src_strerror(src_error));
                        xine_close(stream);
                        xine_dispose(stream);
                        xine_close_audio_driver(xine, ao_port);
                        self->playmode = XL_STOPPED;
                        self->playmode = XL_COMPLETE;
                        continue;
                        }
                     src_data.output_frames = 0;
                     src_data.data_in = src_data.data_out = NULL;
                     src_data.src_ratio = (double)self->samplerate / (double)frame.sample_rate;
                     src_data.end_of_input = 0;
                     resampling = TRUE;
                     }
                  else
                     resampling = FALSE;
               }
            self->write_deferred = 0;
            self->pause = 0;
            self->samples_written = 0;
            self->sleep_samples = 0;
            self->fadein_index = self->seek_s ? 0 : -1;
            if (self->command != XL_COMPLETE)
               ++self->current_audio_context;
            self->initial_audio_context = self->current_audio_context;
            self->play_progress_ms = 0;
            self->playmode = XL_PLAYING;
            self->command = XL_COMPLETE;
         case XL_PLAYING:
            if (self->write_deferred)
               write_channel_data(self);
            else
               {
               switch (playertype)
                  {
                  case XL_VORBIS:
                     if (resampling)
                        {
                        ibuf = realloc(ibuf, sizeof (int16_t) * 6144 * vi->channels);
                        pcmsamples = ov_read(&vf, ibuf, sizeof (int16_t) * 6144 * vi->channels, 0, 2, 1, &bitstream) / sizeof (int16_t) / vi->channels;
                        if (pcmsamples < 0)
                           {
                           fprintf(stderr, "xlplayer: %s ov_read reports an error on file '%s'\n", self->playername, self->pathname);
                           continue;
                           }
                        src_data.input_frames = pcmsamples;
                        src_data.end_of_input = pcmsamples == 0;
                        src_data.data_in = realloc(src_data.data_in, src_data.input_frames * vi->channels * sizeof (float));
                        src_data.output_frames = (int)(src_data.input_frames * src_data.src_ratio) + 2 + (512 * src_data.end_of_input);
                        src_data.data_out = realloc(src_data.data_out, src_data.output_frames * vi->channels * sizeof (float));
                        make_audio_to_float(self, src_data.data_in, ibuf, pcmsamples, 16, vi->channels);
                        if ((src_error = src_process(src_state, &src_data)))
                           {
                           fprintf(stderr, "xlplayer: %s src_process reports - %s\n", self->playername, src_strerror(src_error));
                           self->playmode = XL_EJECTING;
                           continue;
                           }
                        demux_channel_data(self, src_data.data_out, src_data.output_frames_gen, vi->channels);
                        write_channel_data(self);
                        if (src_data.end_of_input)
                           {
                           self->playmode = XL_EJECTING;
                           continue;
                           }
                        }
                     else
                        {
                        ibuf = realloc(ibuf, sizeof (int16_t) * 6144 * vi->channels);
                        pcmsamples = ov_read(&vf, ibuf, sizeof (int16_t) * 6144 * vi->channels, 0, 2, 1, &bitstream) / sizeof (int16_t) / vi->channels;
                        if (pcmsamples < 0)
                           {
                           fprintf(stderr, "xlplayer: %s ov_read reports an error on file '%s'\n", self->playername, self->pathname);
                           continue;		/* skip this 'bad' frame */
                           }
                        if (pcmsamples > 0)
                           {
                           flbuf = realloc(flbuf, sizeof (float) * pcmsamples * vi->channels);
                           flbuf = make_audio_to_float(self, flbuf, ibuf, pcmsamples, 16, vi->channels);
                           demux_channel_data(self, flbuf, pcmsamples, vi->channels);
                           write_channel_data(self);
                           }
                        else
                           self->playmode = XL_EJECTING;
                        }
                     break;
                  case XL_FLAC:
                     FLAC__file_decoder_process_single(decoder);
                     if (FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_OK) 
                        self->playmode = XL_EJECTING;
                     break;
                  case XL_XINE:
                     if (resampling)
                        {
                        src_data.input_frames = frame.num_samples;
                        src_data.data_in = realloc(src_data.data_in, src_data.input_frames * frame.num_channels * sizeof (float));
                        src_data.output_frames = (int)(src_data.input_frames * src_data.src_ratio) + 2 + (512 * src_data.end_of_input);
                        src_data.data_out = realloc(src_data.data_out, src_data.output_frames * frame.num_channels * sizeof (float));
                        make_audio_to_float(self, src_data.data_in, frame.data, frame.num_samples, frame.bits_per_sample, frame.num_channels);
                        if ((src_error = src_process(src_state, &src_data)))
                           {
                           fprintf(stderr, "xlplayer: %s src_process reports - %s\n", self->playername, src_strerror(src_error));
                           self->playmode = XL_EJECTING;
                           continue;
                           }
                        demux_channel_data(self, src_data.data_out, src_data.output_frames_gen, frame.num_channels);
                        write_channel_data(self);
                        if (src_data.end_of_input)
                           {
                           self->playmode = XL_EJECTING;
                           continue;
                           }
                        xine_free_audio_frame (ao_port, &frame);
                        src_data.end_of_input = !xine_get_next_audio_frame(ao_port, &frame);
                        }
                     else
                        {
                        flbuf = realloc(flbuf, sizeof (float) * frame.num_samples * frame.num_channels);
                        flbuf = make_audio_to_float(self, flbuf, frame.data, frame.num_samples, frame.bits_per_sample, frame.num_channels);
                        demux_channel_data(self, flbuf, frame.num_samples, frame.num_channels);
                        write_channel_data(self);
                        xine_free_audio_frame(ao_port, &frame);               
                        if (!xine_get_next_audio_frame(ao_port, &frame))
                           {
                           self->playmode = XL_EJECTING;
                           }
                        }
                  }
               }
            break;
         case XL_EJECTING:
            self->play_progress_ms = 0;
            if (resampling)
               {
               if (src_data.data_in)
                  free(src_data.data_in);
               if (src_data.data_out)
                  free(src_data.data_out);
               src_state = src_delete(src_state);
               }
            switch (playertype)
               {
               case XL_VORBIS:
                  ov_clear(&vf);
                  break;
               case XL_FLAC:
                  FLAC__file_decoder_finish(decoder);
                  break;
               case XL_XINE:
                  xine_close(stream);
                  xine_dispose(stream);
                  xine_close_audio_driver(xine, ao_port);
               }
            if (self->playlistmode)
               {
               if (self->command != XL_EJECT)
                  {
                  /* implements the internal playlist here */
                  if (++self->playlistindex == self->playlistsize && self->loop)
                     self->playlistindex = 0;			/* perform looparound if relevant */
                  if (self->playlistindex < self->playlistsize)	/* check for non end of playlist */
                     {
                     self->pathname = self->playlist[self->playlistindex];
                     self->playmode = XL_INITIATE;
                     continue;
                     }
                  }
               else
                  while (self->playlistsize--)
                     free(self->playlist[self->playlistsize]);
               }
            ++self->current_audio_context;
            self->playmode = XL_STOPPED;
            break;
         } 
      }
   }

struct xlplayer *xlplayer_create(int samplerate, size_t rbsize, char *playername, int *shutdown_f)
   {
   struct xlplayer *self;
   
   if (!(self = malloc(sizeof (struct xlplayer))))
      {
      fprintf(stderr, "xlplayer: malloc failure");
      exit(5);
      }
   self->rbsize = rbsize;

   if (!(self->left_ch = jack_ringbuffer_create(rbsize)))
      {
      fprintf(stderr, "xlplayer: ringbuffer creation failure");
      exit(5);
      }
   if (!(self->right_ch = jack_ringbuffer_create(rbsize)))
      {
      fprintf(stderr, "xlplayer: ringbuffer creation failure");
      exit(5);
      }
   if (!(self->left_fade = jack_ringbuffer_create(rbsize)))
      {
      fprintf(stderr, "xlplayer: ringbuffer creation failure");
      exit(5);
      }
   if (!(self->right_fade = jack_ringbuffer_create(rbsize)))
      {
      fprintf(stderr, "xlplayer: ringbuffer creation failure");
      exit(5);
      }
   self->playername = playername;
   self->leftbuffer = self->rightbuffer = NULL;
   self->have_data_f = self->have_swapped_buffers_f = 0;
   self->pause = 0;
   self->jack_flush = self->jack_is_flushed = 0;
   self->fadeindex = -1;
   self->fadeout_f = 0;
   self->fadein_index = -1;
   self->dither = 0;
   self->seed = 17234;
   self->samplerate = samplerate;
   self->current_audio_context = 0;
   self->playlist = NULL;
   self->flbuf = NULL;
   self->noflush = FALSE;
   self->suppress_flac_output = FALSE;
   self->jack_shutdown_f = shutdown_f;
   self->command = XL_COMPLETE;
   self->playmode = XL_STOPPED;
   pthread_create(&self->thread, NULL, (void *(*)(void *)) xlplayer_main, self);
   return self;
   }
 
void xlplayer_destroy(struct xlplayer *self)
   {
   if (self)
      {
      self->command = XL_EXIT;
      pthread_join(self->thread, NULL);
      jack_ringbuffer_free(self->left_ch);
      jack_ringbuffer_free(self->right_ch);
      jack_ringbuffer_free(self->left_fade);
      jack_ringbuffer_free(self->right_fade);
      free(self);
      }
   }
   
int xlplayer_play(struct xlplayer *self, char *pathname, int seek_s)
   {
   if (self->playmode == XL_PLAYING)
      xlplayer_eject(self);
   self->pathname = pathname;
   self->seek_s = seek_s;
   self->loop = FALSE;
   self->playlistmode = FALSE;
   self->command = XL_PLAY;
   while (self->command)
      usleep(10000);
   return self->initial_audio_context;
   }
   
int xlplayer_playmany(struct xlplayer *self, char *playlist, int loop_f)
   {
   char *start = playlist, *end;
   int payloadlen, i;
   
   if (self->playmode == XL_PLAYING)
      xlplayer_eject(self);
   /* this is where we parse the playlist starting with getting the number of entries */
   while (*start++ != '#');
   start[-1] = '\0';
   self->playlistsize = atoi(playlist);
   /* generate an array of pointers to point to the playlist entries which must be a copy */
   if (!(self->playlist = realloc(self->playlist, self->playlistsize * sizeof (char *))))
      {
      fprintf(stderr, "xlplayer: malloc failure\n");
      exit(5);
      }
   /* now we parse the playlist entries */
   for (i = 0; *start++ == 'd'; i++)
      {
      for (end = start; *end != ':'; end++);
      *end = '\0';
      payloadlen = atoi(start);
      start = end + 1;
      end = start + payloadlen;
      if ((self->playlist[i] = malloc(payloadlen + 1)))
         {
         memcpy(self->playlist[i], start, payloadlen);
         self->playlist[i][payloadlen] = '\0';
         }
      else
         {
         fprintf(stderr, "xlplayer: malloc failure\n");
         exit(5);
         }
      start = end;
      }
   self->seek_s = 0;
   self->loop = loop_f;
   self->playlistmode = TRUE;
   self->command = XL_PLAYMANY;
   while (self->command)
      usleep(10000);
   return self->initial_audio_context;
   }   
   
int xlplayer_play_noflush(struct xlplayer *self, char *pathname, int seek_s)
   {
   self->noflush = TRUE;
   if (self->playmode == XL_PLAYING)
      xlplayer_eject(self);
   self->pathname = pathname;
   self->seek_s = seek_s;
   self->loop = FALSE;
   self->playlistmode = FALSE;
   self->command = XL_PLAY;
   while (self->command)
      usleep(10000);
   self->noflush = FALSE;
   return self->initial_audio_context;
   }
   
void xlplayer_pause(struct xlplayer *self)
   {
   self->pause = TRUE;
   }
   
void xlplayer_unpause(struct xlplayer *self)
   {
   self->pause = FALSE;
   }

void xlplayer_dither(struct xlplayer *self, int dither_f)
   {
   self->dither = dither_f;
   }
 
void xlplayer_eject(struct xlplayer *self)
   {
   if (self->playmode == XL_PLAYING)
      {
      if (!self->fadeout_f)
         xlplayer_pause(self);
      self->command = XL_EJECT;
      while (self->command)
         usleep(10000);
      }
   }

size_t read_from_player(struct xlplayer *self, sample_t *left_buf, sample_t *right_buf, sample_t *left_fbuf, sample_t *right_fbuf, jack_nframes_t nframes)
   {
   jack_ringbuffer_t *swap;
   size_t todo, favail, ftodo;
   
   self->have_swapped_buffers_f = FALSE;
   if (self->jack_flush)
      {
      if (self->noflush == FALSE)
         {
         if (self->pause == 0)
            {
            swap = self->left_ch;
            self->left_ch = self->left_fade;
            self->left_fade = swap;
            swap = self->right_ch;
            self->right_ch = self->right_fade;
            self->right_fade = swap;
            self->fadeindex = 0;
            self->have_swapped_buffers_f = TRUE;
            }
         jack_ringbuffer_reset(self->left_ch);
         jack_ringbuffer_reset(self->right_ch);
         }
      self->jack_is_flushed = 1;
      self->jack_flush = 0;
      self->pause = 0;
      }
   
   self->avail = jack_ringbuffer_read_space(self->right_ch) / sizeof (sample_t);
   todo = (self->avail > nframes ? nframes : self->avail);
   favail = jack_ringbuffer_read_space(self->right_fade) / sizeof (sample_t);
   ftodo = (favail > nframes ? nframes : favail);
   
   if (self->pause == 0)
      {
      /* fill the frame with whatever data is available, then pad as needed with zeroes */
      jack_ringbuffer_read(self->left_ch, (char *)left_buf, todo * sizeof (sample_t));
      memset(left_buf + todo, 0, (nframes - todo) * sizeof (sample_t)); 
      jack_ringbuffer_read(self->right_ch, (char *)right_buf, todo * sizeof (sample_t));
      memset(right_buf + todo, 0, (nframes - todo) * sizeof (sample_t));
      if (left_fbuf && right_fbuf)
         {
         jack_ringbuffer_read(self->left_fade, (char *)left_fbuf, ftodo * sizeof (sample_t));
         memset(left_fbuf + ftodo, 0, (nframes - ftodo) * sizeof (sample_t)); 
         jack_ringbuffer_read(self->right_fade, (char *)right_fbuf, ftodo * sizeof (sample_t));
         memset(right_fbuf + ftodo, 0, (nframes - ftodo) * sizeof (sample_t));
         }
      self->have_data_f = todo > 0;
      }
   else
      {
      memset(left_buf, 0, nframes * sizeof (sample_t));
      memset(right_buf, 0, nframes * sizeof (sample_t));
      if (left_fbuf && right_fbuf)
         {
         memset(left_fbuf, 0, nframes * sizeof (sample_t));
         memset(right_fbuf, 0, nframes * sizeof (sample_t));
         }
      }
   xlplayer_update_progress_time_ms(self);
   return todo;
   }
   
int xlplayer_create_fadein_lookup(float size_seconds, int samplerate)
   {
   uint32_t i;
   sample_t ratio;
   
   fadeinlookuptable_numsamples = (size_t)(size_seconds * (float)samplerate);
   
   if (!(fadeinlookuptable = malloc(fadeinlookuptable_numsamples * sizeof (sample_t))))
      return FALSE;

   for (i = 0; i < fadeinlookuptable_numsamples; i++)
      {
      ratio = ((sample_t)i) / ((sample_t)fadeinlookuptable_numsamples); /* ranges 0 -> 1 */
      fadeinlookuptable[i] = powf(10.0F, ((ratio - 1.0F) * 99.0F) / 20.0F);	/* ranges -99 to 0db */
      }
   return TRUE;
   }
   
/* xlplayer_destroy_fadein_lookup: frees the lookup table used for fading */
void xlplayer_destroy_fadein_lookup()
   {
   if (fadeinlookuptable)
      free(fadeinlookuptable);
   }
