/*
#   compressor.c: Audio dynamic range compression code from IDJC.
#   Copyright (C) 2005-2006 Stephen Fairchild
#
#   This program is free software; you can redistribute it and/or modify
#   it under the terms of the GNU General Public License as published by
#   the Free Software Foundation; either version 2 of the License, or
#   (at your option) any later version.
#
#   This program is distributed in the hope that it will be useful,
#   but WITHOUT ANY WARRANTY; without even the implied warranty of
#   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#   GNU General Public License for more details.
#
#   You should have received a copy of the GNU General Public License
#   along with this program; if not, write to the Free Software
#   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#define _GNU_SOURCE
#include <math.h>
#include <assert.h>
#include "dbconvert.h"
#include "compressor.h"

#include <stdio.h>

/*#define GNUPLOT*/

/* compressor: a fully featured dynamic range compressor */
compaudio_t compressor(struct compressor *self, compaudio_t signal, int skip_rms)
   {
   compaudio_t ess, gots, target_db, gain_target_db, ndb, cf, diff;

#ifndef GNUPLOT
   if (skip_rms)
#endif
      self->signal_db = gots = level2db(fabs(signal));
#ifndef GNUPLOT
   else
      self->signal_db = gots = level2db(fabs(rms_calc(self->rms_filter, signal)));
#endif
   if (!isfinite(gots))
      self->signal_db = gots = -100.0F;
   else
      if (self->filter)
         {
#ifndef GNUPLOT
         ess = digital_filter(self->filter, signal);
         ess = level2db(fabsf(ess)) + self->de_ess_db;
	 if (ess > gots && isfinite(ess))
	    gots += (ess - gots) * 1.5F;
#endif
	 }
   if (gots <= self->k2)
      target_db = gots;
   else
      target_db = self->k2;
   
   if (gots > self->k2 && self->k2 < self->k1)
      {
      /* number of db that are in the knee */
      ndb = ((gots >= self->k1) ? self->k1 : gots) - self->k2;
      /* a value between 0 and 1 representing where we are between k1 and k2 */
      cf = (ndb/ (self->k1 - self->k2));
      /* finally compute the ratio to use */
      cf = sqrt(((self->ratio-1) * cf * cf * 0.285) + 1);
      target_db += ndb / cf;
      }
   if (gots > self->k1)
      {
      ndb = gots - self->k1;
      target_db += ndb / self->ratio;
      }
#ifdef GNUPLOT 
   self->curve = target_db;
#endif
   gain_target_db = target_db - gots;

   if (fabs(diff = gain_target_db - self->gain_db) > 0.0000004)
      {
      if (self->gain_db > gain_target_db)
         self->gain_db += diff * self->attack;
      else
         self->gain_db += diff * self->release;
      }

#ifndef GNUPLOT
   /* code to calculate the amount of ducking needed */
   if (self->gain_db * self->ducking <= self->ducking_db)
      {
      self->ducking_db = self->gain_db * self->ducking;
      self->ducking_hold_count = 0;
      }
   else
      if(++self->ducking_hold_count > self->ducking_hold)
	 self->ducking_db += -self->ducking_db * 0.001F;
#endif
   /* Return the gain of the compressor in dB */
   return self->gain_db + self->opgain;
   }   

/* limiter: a basic hard knee compressor - called limiter because that is the mode */
/* in which IDJC uses it */
compaudio_t limiter(struct compressor *self, compaudio_t left, compaudio_t right) 
   {
   compaudio_t gots, target_db, gain_target_db, ndb, diff;   

   gots = level2db(fabs((fabs(left) > fabs(right)) ? left : right));
   if (!isfinite(gots))
      gots = -100.0;

   if (gots <= self->k2)
      target_db = gots;
   else
      target_db = self->k2;
   if (gots > self->k1)
      {
      ndb = gots - self->k1;
      target_db += ndb / self->ratio;
      }            
   gain_target_db = target_db - gots; 
   
   if (fabs(diff = gain_target_db - self->gain_db) > 0.0000004)
      {  
      if (self->gain_db > gain_target_db)
         self->gain_db += diff * self->attack;
      else
         self->gain_db += diff * self->release;
      }
   return self->gain_db;
   }

/* the variable maxlevel dictates the amount by which the volume can be turned up */
/* when the ceiling level is breached the volume level is reduced */
compaudio_t normalizer(struct normalizer *self, compaudio_t left, compaudio_t right)
   {
   compaudio_t gots;

   gots = level2db(fabs((fabs(left) > fabs(right)) ? left : right));
   if (!isfinite(gots))
      gots = -90.3089987F;

   if (gots + self->level > self->ceiling && self->active != 0)
      {
      self->level -= (self->level - self->ceiling) * self->fall;
      }
   else
      {
      if (self->active)
         self->level += (self->maxlevel - self->level) * self->rise;
      else
         self->level += (0.0F - self->level) * self->rise;
      if (self->level > self->maxlevel)
         self->level = self->maxlevel;
      }
   return self->level;
   }

#ifdef GNUPLOT

/* compile with: gcc compressor.c dbconvert.c -lm */
/* generates a table of the compression curve for gnuplot */
/* in gnuplot enter: plot "output.dat" */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

int main()
   {
   struct compressor comp;
   int i;
   float x, y;
   FILE *fp;
   
   comp.gain_db =  1.0F;
   comp.k1 = -20.0;	/* -10db */
   comp.k2 = -60.0;	/* -20db */
   comp.ratio = 4.0F;	/* fairly typical value */
   comp.attack = 1.0F;	/* instant movement so we can get a plot of the acutual compression curve */
   comp.release = 1.0F;
   comp.opgain = 0.0F;
   comp.ducking = 0.0F;
   comp.ducking_hold = 1;
   comp.ducking_hold_count = 0;
   comp.ducking_db = 0.0F;
   comp.de_ess_db = 0.0F;
   comp.signal_db = 0.0F;
   comp.filter = NULL;
   comp.rms_filter = NULL;
   
   init_dblookup_table();
   init_signallookup_table();
   
   if (!(fp = fopen("output.dat", "w")))
      {
      printf("could not open output file for writing\n");
      exit(5);
      }
   
   for (i = -1000; i <= 0; i++)
      {
      x = db2level(i/10.0F);
      y = compressor(&comp, x, 1);

      fprintf(fp, "%f %f %f\n", i/10.0F, y, comp.curve);
      if (ferror(fp))
         {
         printf("error writing output file\n");
         exit(5);
         }
      }
   fclose(fp);
   printf("output file successfully writen\n");
   
   free_dblookup_table();
   free_signallookup_table();
   }
#endif
