/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/****************************************************************************
** ui.h extension file, included from the uic-generated form implementation.
**
** If you wish to add, delete or rename slots use Qt Designer which will
** update this file, preserving your code. Create an init() slot in place of
** a constructor, and a destroy() slot in place of a destructor.
*****************************************************************************/


#include "idialogpaletteeditor.h"

#include "ivtk.h"
#include "iqt.h"
#include "iqtwindow.h"
#include "iglobals.h"
#include "ienvironment.h"
#include "ipalette.h"
#include "ipiecewisefunction.h"

#include "iqtdefs.h"

#include <qapplication.h>
#include <qcombobox.h>
#include <qfile.h>
#include <qfiledialog.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qmessagebox.h>
#include <qpushbutton.h>
#include <qtextstream.h>


void reportNullPointer(int ec);


void iDialogPaletteEditor::init()
{
    
    int i;
	PaletteList->blockSignals(true);
    for(i=0; i<iPaletteList::getInstance()->getNumPalettes(); i++){
		PaletteList->insertItem(*(iPaletteList::getInstance()->getPaletteName(i)),-1);
    }
	PaletteList->blockSignals(false);
    
    pixmapPaint = QPixmap(256,32);
    
    cpFun[0] = cpFun[1] = cpFun[2] = 0;
    
    currentFileName = iEnvironment::getInstance()->get_IFRIT_PALETTE_DIR();
    changed = false;
    
}


void iDialogPaletteEditor::destroy()
{

}


void iDialogPaletteEditor::closeEvent( QCloseEvent * e )
{
    
    if(changed){
	switch( QMessageBox::warning(this,"IFRIT Palette Editor",
				      "The current palette is not saved.\n\n",
				      "Save", "Ignore", 0, 0, 1 ) ) {
	    case 0: // The user clicked the Retry again button or pressed Enter
	    { this->fileSave(); break; }
	    case 1: // The user clicked the Quit or pressed Escape
	    { break; }
	}
    }
    qApp->exit_loop();
    e->accept();
}


void iDialogPaletteEditor::exec()
{
    
    setWFlags(WShowModal);
    show();
    
    int n = PaletteList->currentItem();
    this->updatePalettePixmap(n); 
    
    qApp->enter_loop();

}


void iDialogPaletteEditor::fileLoad()
{

    QString s = QFileDialog::getOpenFileName(currentFileName,
		"IFRIT palette file (*.ipf)",
		this,
		"open IFRIT palette file",
		"Choose a file" );
	
    if(!s.isEmpty()) {
	
	QFile f(s);
	
	if(!f.exists() || !f.open(IO_ReadOnly)) {
	    switch( QMessageBox::warning(this,"IFRIT Palette Editor",
					  "Could not open the palette file.\n\n",
					  "Ignore", "Close Palette Editor", 0, 0, 1 ) ) {
		case 0: // The user clicked the Retry again button or pressed Enter
			break;
		case 1: // The user clicked the Quit or pressed Escape
			{ this->close(); break; }
		}
		return;
	}

	int i, j, n;
	float x, y;
	bool ok, err = false;
	QString s1, name;
	iPiecewiseFunction *p = new iPiecewiseFunction[3];
	if(p == 0) reportNullPointer(9201);
	
	QTextStream ts(&f);
	//
	//  Read the palette name
	//
	s1 = ts.readLine(); if(!s1.isEmpty()) name = s1; else err = true;
	//
	//  Read 3 piecewise functions
	//
	for(j=0; j<3 && !err; j++){
	    //
	    //  Read number of points
	    //
	    s1 = ts.readLine(); 
	    n = s1.toInt(&ok);
	    if(!ok) err = true; else {
		for(i=0; i<n && !err; i++){
		    s1 = ts.readLine(); 
		    x = s1.section(",",0,0).toFloat(&ok); if(!ok) err = true;
		    y = s1.section(",",1).toFloat(&ok); if(!ok) err = true;
		    if(i==0 || i==n-1) p[j].movFunctionPoint(i,x,y); else p[j].addFunctionPoint(x,y); 
		}
	    }
	}
	f.close();
	
	if(err){
	    //
	    //  Error
	    //
	    switch( QMessageBox::warning(this,"IFRIT Palette Editor",
					  "Palette file is corrupted.\n\n",
					  "Ignore", "Close Palette Editor", 0, 0, 1 ) ) {
		case 0: // The user clicked the Retry again button or pressed Enter
			break;
		case 1: // The user clicked the Quit or pressed Escape
			{ this->close(); break; }
	    }
	} else {	    
	    //
	    //  Successfull read
	    //
	    this->paletteCreate();
	    int c = PaletteList->currentItem();
	    this->updatePaletteName(name);
	    iPaletteList::getInstance()->getPalette(c)->setPiecewiseFunctions(p+0,p+1,p+2);
	    this->updatePalettePixmap(c); 
	    currentFileName = s;
	    changed = false;
	}
	
	delete [] p;
	
    }

}

void iDialogPaletteEditor::fileSave()
{
    
    QString s = QFileDialog::getSaveFileName(currentFileName,
		"IFRIT palette file (*.ipf)",
		this,
		"save IFRIT palette file",
		"Choose a file" );
	
    if(!s.isEmpty()) {
	
	QFile f(s);
	
	if(!f.open(IO_WriteOnly)) {
	    switch( QMessageBox::warning(this,"IFRIT Palette Editor",
					  "Could not open the file for writing.\n\n",
					  "Ignore", "Close Palette Editor", 0, 0, 1 ) ) {
		case 0: // The user clicked the Retry again button or pressed Enter
			break;
		case 1: // The user clicked the Quit or pressed Escape
			{ this->close(); break; }
	    }
	    changed = false;
	    return;
	}

	int i, j, n;
	float *x, *y;
	bool err = false;
	iPiecewiseFunction *pf[3];
	iPaletteList::getInstance()->getPalette(PaletteList->currentItem())->getPiecewiseFunctions(pf[0],pf[1],pf[2]);
	
	QTextStream ts(&f);
	//
	//  Write the palette name
	//
	ts << *(iPaletteList::getInstance()->getPaletteName(PaletteList->currentItem())) << "\n";
	//
	//  Write 3 piecewise functions
	//
	for(j=0; j<3 && !err; j++){
	    //
	    //  Read number of points
	    //
	    n = pf[j]->getFunctionN();
	    x = pf[j]->getFunctionX();
	    y = pf[j]->getFunctionY();
	    
	    ts << n << "\n";
	    
	    for(i=0; i<n && !err; i++){
		ts << x[i] << "," << y[i] << "\n";
	    }
	}
	f.close();
	
	if(err){
	    //
	    //  Error
	    //
	    switch( QMessageBox::warning(this,"IFRIT Palette Editor",
					  "Palette file could not be saved.\n\n",
					  "Retry", "Ignore", 0, 0, 1 ) ) {
		case 0: // The user clicked the Retry again button or pressed Enter
		        { this->fileSave(); break; }
		case 1: // The user clicked the Quit or pressed Escape
			{ break; }
	    }
	} else {	    
	    //
	    //  Successfull write
	    //
	    currentFileName = s;
	}
	
    }

}


void iDialogPaletteEditor::paletteCreate()
{
    
    if(!iPaletteList::getInstance()->addEmptyPalette()){
		QMessageBox::critical(this,"IFRIT",
			QString("Allocated palette memory is exhausted.\n") +
			"Please, remove some of the other palettes before creating a new one.\n" );
		ButtonCreate->setEnabled(false);
    } else {
		PaletteList->blockSignals(true);
		PaletteList->insertItem(*(iPaletteList::getInstance()->getPaletteName(PaletteList->count())),-1);
		PaletteList->setCurrentItem(PaletteList->count()-1);
		PaletteList->blockSignals(false);
		this->updatePalettePixmap(PaletteList->currentItem());
		if(!ButtonRemove->isEnabled()) ButtonRemove->setEnabled(true);
		changed = true;
    }
	
}


void iDialogPaletteEditor::paletteRemove()
{
    
    if(PaletteList->count() > 1) {
		PaletteList->blockSignals(true);
		int n = PaletteList->currentItem();
		iPaletteList::getInstance()->removePalette(n);
		PaletteList->removeItem(n);
		if(n >= PaletteList->count()) n--;
		PaletteList->setCurrentItem(n);
		this->updatePalettePixmap(PaletteList->currentItem());
		if(!ButtonCreate->isEnabled()) ButtonCreate->setEnabled(true);
		PaletteList->blockSignals(false);
    } else {
		ButtonRemove->setEnabled(false);
    }
	
}	


void iDialogPaletteEditor::paletteReverse()
{
    
    iPaletteList::getInstance()->getPalette(PaletteList->currentItem())->reverse();
    this->updatePalettePixmap(PaletteList->currentItem());
    
}


void iDialogPaletteEditor::updatePalettePixmap( int n )
{
    
    iQTWindow::getCurrentWindow()->updatePalettePixmap(iPaletteList::getInstance()->getPalette(n),&pixmapPaint);
    PalettePixmap->setPixmap(pixmapPaint);
    iPaletteList::getInstance()->getPalette(n)->getPiecewiseFunctions(fun[0],fun[1],fun[2]);
    this->updateFunctionPixmapR((QMouseEvent *)0);
    this->updateFunctionPixmapG((QMouseEvent *)0);
    this->updateFunctionPixmapB((QMouseEvent *)0);
    CALL_FUNCTION1(PaletteNameEdit,setText,*(iPaletteList::getInstance()->getPaletteName(n)));
	    
}


void iDialogPaletteEditor::updateFunctionPixmapR( QMouseEvent *e )
{
    this->updateFunctionPixmap(e,fun[0],ILabelR,cpFun[0]);
}


void iDialogPaletteEditor::updateFunctionPixmapG( QMouseEvent *e )
{
    this->updateFunctionPixmap(e,fun[1],ILabelG,cpFun[1]);
}


void iDialogPaletteEditor::updateFunctionPixmapB( QMouseEvent *e )
{
    this->updateFunctionPixmap(e,fun[2],ILabelB,cpFun[2]);
}


void iDialogPaletteEditor::updateFunctionPixmap( QMouseEvent * e, iPiecewiseFunction *vfun, iLabel *lab, int &vcpFun)
{
    
    iQTWindow::getCurrentWindow()->updateFunctionPixmap(e,vfun,lab,vcpFun,0);
    if(e != 0){
		iPaletteList::getInstance()->getPalette(PaletteList->currentItem())->update();
		iQTWindow::getCurrentWindow()->updatePalettePixmap(iPaletteList::getInstance()->getPalette(PaletteList->currentItem()),&pixmapPaint);
		PalettePixmap->setPixmap(pixmapPaint);
    }
    
}


void iDialogPaletteEditor::updatePaletteName(const QString &s)
{
    
    iPaletteList::getInstance()->setPaletteName(PaletteList->currentItem(),s);
    PaletteList->changeItem(s,PaletteList->currentItem());

}


void iDialogPaletteEditor::updateRadd()
{
    this->updateAdd(0);
}


void iDialogPaletteEditor::updateGadd()
{
    this->updateAdd(1);
}


void iDialogPaletteEditor::updateBadd()
{
    this->updateAdd(2);
}


void iDialogPaletteEditor::updateAdd(int n)
{
    
    fun[n]->addFunctionPoint(cpFun[n]);
    if(cpFun[n] < fun[n]->getFunctionN()-1) cpFun[n]++;
    switch (n) {
	case 0: { this->updateFunctionPixmapR(0); break; }
	case 1: { this->updateFunctionPixmapG(0); break; }
	case 2: { this->updateFunctionPixmapB(0); break; }
	}
    
}


void iDialogPaletteEditor::updateRdel()
{
    this->updateDel(0);
}


void iDialogPaletteEditor::updateGdel()
{
    this->updateDel(1);
}


void iDialogPaletteEditor::updateBdel()
{
    this->updateDel(2);
}


void iDialogPaletteEditor::updateDel(int n)
{
    
    fun[n]->delFunctionPoint(cpFun[n]);
    if(cpFun[n] > 0) cpFun[n]--;
    switch (n) {
	case 0: { this->updateFunctionPixmapR(0); break; }
	case 1: { this->updateFunctionPixmapG(0); break; }
	case 2: { this->updateFunctionPixmapB(0); break; }
	}
    iPaletteList::getInstance()->getPalette(PaletteList->currentItem())->update();
    iQTWindow::getCurrentWindow()->updatePalettePixmap(iPaletteList::getInstance()->getPalette(PaletteList->currentItem()),&pixmapPaint);
    PalettePixmap->setPixmap(pixmapPaint);
    
}


void iDialogPaletteEditor::updateRreset()
{
    this->updateReset(0);
}


void iDialogPaletteEditor::updateGreset()
{
    this->updateReset(1);
}


void iDialogPaletteEditor::updateBreset()
{
    this->updateReset(2);
}


void iDialogPaletteEditor::updateReset(int n)
{
    int i, m;
    
    m = fun[n]->getFunctionN();
    for(i=0; i<m-2; i++) fun[n]->delFunctionPoint(1);
    fun[n]->movFunctionPoint(0,0.0,0.0);
    fun[n]->movFunctionPoint(1,1.0,1.0);
    cpFun[n]=0;
    switch (n) {
	case 0: { this->updateFunctionPixmapR(0); break; }
	case 1: { this->updateFunctionPixmapG(0); break; }
	case 2: { this->updateFunctionPixmapB(0); break; }
	}
    iPaletteList::getInstance()->getPalette(PaletteList->currentItem())->update();
    iQTWindow::getCurrentWindow()->updatePalettePixmap(iPaletteList::getInstance()->getPalette(PaletteList->currentItem()),&pixmapPaint);
    PalettePixmap->setPixmap(pixmapPaint);
    
}


void iDialogPaletteEditor::setCurrentPalette(int n)
{
	
	if(n>=0 && n<iPaletteList::getInstance()->getNumPalettes()){
		CALL_FUNCTION1(PaletteList,setCurrentItem,n);
		this->updatePalettePixmap(n);
	}
	
}
