/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "imarker.h"

#include "iobjectfactory.h"

#include "ivtk.h"
#include "ivisualobjectobserver.h"
#include "icaption.h"
#include "iscaledlodactor.h"
#include "idatareader.h"

#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProperty.h>
#include <vtkProperty2D.h>
#include <vtkCoordinate.h>
#include <vtkMath.h>

#include <vtkPoints.h>
#include <vtkCellArray.h>

#include <vtkSphereSource.h>
#include <vtkCubeSource.h>
#include <vtkCylinderSource.h>
#include <vtkConeSource.h>
#include <vtkArrowSource.h>

#ifndef I_VTK_VERSION_40
#include <vtkTextProperty.h>
#endif


void reportNullPointer(int);


char * iMarker::modeName[] = { "None", "Point", "Sphere", "Cone", "Cube", "Cylinder", "Arrow" };



iMarker* iMarker::New(iVTK *mVTK)
{
	return new iMarker(mVTK); // non-inheritable, so no need to use Object Factory
}


iMarker::iMarker(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	mode = 2;
	size = 0.1;
	colorAuto = true;
	showCaption = true;

	color = iColor(100,150,200);
	textColor = iColor(0,0,0);
	caption = "";

	//
	//  Do VTK stuff
	//	
	markerActor = iScaledLODActor::New();
	if(markerActor == 0) reportNullPointer(4801);
	markerMapper = vtkPolyDataMapper::New();
	if(markerMapper == 0) reportNullPointer(4802);
	markerObject = vtkPolyData::New();
	if(markerObject == 0) reportNullPointer(4803);
	markerCaption = iCaption::New();
	if(markerCaption == 0) reportNullPointer(4804);

	markerMapper->SetInput(markerObject);
	
	markerActor->SetMapper(markerMapper);
	markerActor->GetProperty()->SetPointSize(4.0);

	markerActor->RotateX(180.0*vtkMath::Random());
	markerActor->RotateY(180.0*vtkMath::Random());
	markerActor->RotateZ(180.0*vtkMath::Random());
	
	//
	//  Add observer to keep information about this object
	//
	markerActor->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_MARK,IDATATYPE_NONE);
	objectObserver->setUserValue(0.0);

	double x[3] = { 0.0, 0.0, 0.0 };
	this->setPosition(x);
	this->setMode(mode);
	this->setColor(color);
	this->setTextColor(textColor);
	this->setSize(size);
	this->setFontSize(myVTK->getFontSize());

}


iMarker::~iMarker()
{
	
	if(!init) 
	{
		myVTK->removeProp(markerActor);
		myVTK->removeProp(markerCaption);
	}
	markerActor->Delete();	
	markerMapper->Delete();
	markerObject->Delete();
	markerCaption->Delete();

}



void iMarker::setMode(int m)
{
	if(m>=0 && m<=6)
	{
		mode = m;
		
		markerActor->VisibilityOn();
		switch (mode)
		{
		case 0:
			{
				markerActor->VisibilityOff();
				break;
			}
		case 1:
			{
				markerObject->Initialize();
				vtkPoints *pts = vtkPoints::New(VTK_FLOAT);
				pts->SetNumberOfPoints(1);
				pts->SetPoint(0,0.0,0.0,0.0);
				markerObject->SetPoints(pts);
				pts->Delete();
				vtkCellArray *cd = vtkCellArray::New();
				cd->InsertNextCell(1);
				cd->InsertCellPoint(0);
				markerObject->SetVerts(cd);
				cd->Delete();
				break;
			}
		case 2:
			{
				vtkSphereSource *s = vtkSphereSource::New();
				s->SetThetaResolution(20);
				s->SetPhiResolution(20);
				s->Update();
				markerObject->DeepCopy(s->GetOutput());
				s->Delete();
				break;
			}
		case 3:
			{
				vtkConeSource *s = vtkConeSource::New();
				s->Update();
				markerObject->DeepCopy(s->GetOutput());
				s->Delete();
				break;
			}
		case 4:
			{
				vtkCubeSource *s = vtkCubeSource::New();
				s->Update();
				markerObject->DeepCopy(s->GetOutput());
				s->Delete();
				break;
			}
		case 5:
			{
				vtkCylinderSource *s = vtkCylinderSource::New();
				s->Update();
				markerObject->DeepCopy(s->GetOutput());
				s->Delete();
				break;
			}
		case 6:
			{
				vtkArrowSource *s = vtkArrowSource::New();
				s->SetTipRadius(0.3);
				s->SetShaftRadius(0.15);
				s->Update();
				markerObject->DeepCopy(s->GetOutput());
				s->Delete();
				break;
			}

		}
		this->setSize(size);
	}

}


void iMarker::setPosition(double *x)
{
	int j;
	for(j=0; j<3; j++) pos[j] = x[j];
	markerActor->SetPosition(x[0],x[1],x[2]);
	markerCaption->SetAttachmentPoint(x[0],x[1],x[2]);
}


double *iMarker::getPosition()
{ 
	return pos;
}


void iMarker::setSize(float s)
{
	if(s>0.0 && s<100.0)
	{
		size = s;
		if(mode > 0) markerActor->setBasicScale(size); else markerActor->setBasicScale(1.0);
	}
}


void iMarker::setOpacity(float s)
{
	if(s>0.0 && s<=1.0)
	{
		opa = s;
		markerActor->GetProperty()->SetOpacity(opa);
	}
}


void iMarker::setFontSize(int s)
{
	markerCaption->setFontSize(s);
}


void iMarker::setColor(iColor c)
{
	color = c;
	markerActor->GetProperty()->SetColor(c.red()/255.0,c.green()/255.0,c.blue()/255.0);
}


void iMarker::setTextColor(iColor c, bool withAuto)
{
	if(withAuto && c== myVTK->getBackgroundColor()) colorAuto = true; else colorAuto = false;
	if(colorAuto) 
	{
		c = QColor(255-myVTK->getBackgroundColor().red(),255-myVTK->getBackgroundColor().green(),255-myVTK->getBackgroundColor().blue());
	}
	textColor = c;
	markerCaption->GetProperty()->SetColor(c.red()/255.0,c.green()/255.0,c.blue()/255.0);
#ifndef I_VTK_VERSION_40
	markerCaption->GetCaptionTextProperty()->SetColor(c.red()/255.0,c.green()/255.0,c.blue()/255.0);
#endif
}


void iMarker::setCaption(const iString &s)
{
	caption = s;
	markerCaption->SetCaption(caption.latin1());
	if(showCaption && caption.length()>0) markerCaption->VisibilityOn();
	if(caption.length() == 0) markerCaption->VisibilityOff();
}


void iMarker::setShowCaption(bool s)
{
	showCaption = s;
	if(showCaption && caption.length()>0) markerCaption->SetVisibility(markerActor->GetVisibility()); else markerCaption->VisibilityOff();
}


void iMarker::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	markerActor->GetProperty()->SetAmbient(v);
}


void iMarker::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	markerActor->GetProperty()->SetDiffuse(v);
}


void iMarker::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	markerActor->GetProperty()->SetSpecular(v);
}


void iMarker::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	markerActor->GetProperty()->SetSpecularPower(v);
}


void iMarker::show(bool show)
{
	  
	if(show)
	{
		if(init)
		{
			myVTK->addProp(markerActor);
			myVTK->addProp(markerCaption);
			if(colorAuto) this->setTextColor(myVTK->getBackgroundColor());
			init = false;
		}
		if(!init)
		{
			markerActor->VisibilityOn();
			if(showCaption && caption.length()>0) markerCaption->VisibilityOn(); else markerCaption->VisibilityOff(); 
		}
	} 
	else 
	{
		if(!init) 
		{
			markerActor->VisibilityOff();
			markerCaption->VisibilityOff();
		}
	}
	
}


bool iMarker::isVisible()
{ 
	return (markerActor->GetVisibility()==1 && !init); 
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iMarker::packState(iString &s)
{

	iVisualObject::packState(s);

	this->packValue(s,"iMarker::showCaption",showCaption);
	this->packValue(s,"iMarker::colorAuto",colorAuto);

	this->packValue(s,"iMarker::mode",mode);
	this->packValue(s,"iMarker::markerColor",color);
	this->packValue(s,"iMarker::textColor",textColor);

	this->packValue(s,"iMarker::caption",caption);

	this->packValue(s,"iMarker::size",size);
	this->packValue(s,"iMarker::markerActor->GetPosition()",markerActor->GetPosition(),3);
	this->packValue(s,"iMarker::markerCaption->GetPosition()",markerCaption->GetPosition(),2);

}


void iMarker::unpackState(iString s)
{
	int i; bool b; float f;
	vtkFloat v[3]; iColor c; iString z;

	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iMarker::showCaption",b)) this->setShowCaption(b);
	if(this->unpackValue(s,"iMarker::colorAuto",b)) colorAuto = b;

	if(this->unpackValue(s,"iMarker::mode",i)) this->setMode(i);

	if(this->unpackValue(s,"iMarker::markerColor",c)) this->setColor(c);
	if(this->unpackValue(s,"iMarker::textColor",c)) this->setTextColor(c,false);

	if(this->unpackValue(s,"iMarker::caption",z)) this->setCaption(z);

	if(this->unpackValue(s,"iMarker::size",f)) this->setSize(f);
	if(this->unpackValue(s,"iMarker::markerActor->GetPosition()",v,3))
	{
		markerActor->SetPosition(v);
		markerCaption->SetAttachmentPoint(v);
	}
	if(this->unpackValue(s,"iMarker::markerCaption->GetPosition()",v,2)) markerCaption->SetPosition(v);

}


