/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of iparticlessplitter.h
  
=========================================================================*/


#include "ilimits.h"
#include "ivtk.h"
#include "iglobals.h"

#include "iparticlessplitter.h"
#include "imath.h"
#include "ihistogrammaker.h"
#include "ipolygonaldata.h"
#include "idatareader.h"

#include <vtkObjectFactory.h>
#include <vtkFloatArray.h>
#include <vtkDoubleArray.h>
#include <vtkIdTypeArray.h>
#include <vtkPointData.h>
#include <vtkCellArray.h>

#include <vtkTimerLog.h>


void reportNullPointer(int ec);

namespace iParticlesSplitter_Private
{
//
//  QuickFind
//
#define arr(i)	Atts[ncom*(i)+attSort]

bool find2(vtkIdType n, int ncom, int attSort, float *Atts, vtkIdType iStart, float aval, vtkIdType &iLo, vtkIdType &iHi)
{

	vtkIdType i1 = iStart;
	vtkIdType i2 = n - 1;
	vtkIdType ic;

	if(aval < arr(i1))
	{
		iLo = i1;
		iHi = i1;
		return true;
	}
	
	if(aval > arr(i2))
	{
		iLo = i2;
		iHi = i2;
		return true;
	}

	while(i2-i1 > 1)
	{
		ic = (i1+i2)/2;
		if(aval >= arr(ic)) i1 = ic;
		if(aval <= arr(ic)) i2 = ic;
	}

	iLo = i1;
	iHi = i2;

	return true;

}
//
//  QuickSort
//
#define SAVE(CELL,i1)    { for(ii=0; ii<3; ii++) poi##CELL[ii] = Points[3*(i1)+ii]; for(ii=0; ii<ncom; ii++) att##CELL[ii] = Atts[ncom*(i1)+ii]; }

#define MOVE(i1,i2)      { for(ii=0; ii<3; ii++) Points[3*(i1)+ii] = Points[3*(i2)+ii]; for(ii=0; ii<ncom; ii++) Atts[ncom*(i1)+ii] = Atts[ncom*(i2)+ii]; }

#define RESTORE(i2,CELL) { for(ii=0; ii<3; ii++) Points[3*(i2)+ii] = poi##CELL[ii]; for(ii=0; ii<ncom; ii++) Atts[ncom*(i2)+ii] = att##CELL[ii]; }

#define SWAP(i1,i2) { SAVE(1,i1); MOVE(i1,i2); RESTORE(i2,1); }

//
// Do our own quick sort for efficiency reason (based on a Java code by Denis Ahrens)
//

template<class T>
void sort2(vtkIdType n, int ncom, int attSort, T *Points, float *Atts)
{
	T poi1[3];
	float *att1; att1 = new float[ncom];

	doSort(0,n-1,ncom,attSort,Points,Atts,poi1,att1);
	
	delete [] att1;

}

//
//  Recursive worker
//
template<class T>
void doSort(vtkIdType l, vtkIdType r, int ncom, int attSort, T *Points, float *Atts, T *poi1, float *att1)
{
	const int M = 8;
	vtkIdType i, j;
	float v;
	int ii;

	if ((r-l)>M)
	{
		//
		// Use quicksort
		//
		i = (r+l)/2;
		if (arr(l)>arr(i)) SWAP(l,i);     // Tri-Median Methode!
		if (arr(l)>arr(r)) SWAP(l,r);
		if (arr(i)>arr(r)) SWAP(i,r);
		
		j = r-1;
		SWAP(i,j);
		i = l;
		v = arr(j);
		for(;;)
		{
			while(arr(++i) < v);
			while(arr(--j) > v);
			if (j<i) break;
			SWAP(i,j);
		}
		SWAP(i,r-1);
		doSort(l,j,ncom,attSort,Points,Atts,poi1,att1);
		doSort(i+1,r,ncom,attSort,Points,Atts,poi1,att1);
	}
	else 
	{
		//
		// Array is small, use insertion sort. 
		//
		for (i=l+1;i<=r;i++)
		{
			SAVE(1,i);
			v = arr(i);
			j=i;
			while (j>l && arr(j-1)>v)
			{
				MOVE(j,j-1);
				j--;
			}
			RESTORE(j,1);
		}
    }
	
}

};


using namespace iParticlesSplitter_Private;

//------------------------------------------------------------------------------
iParticlesSplitter* iParticlesSplitter::New(iVTK *m)
{
	return new iParticlesSplitter(m); // non-inheritable, so no need to use Object Factory
}


iParticlesSplitter::iParticlesSplitter(iVTK *m) : iObject(m)
{

	limitsSet = false;
	att = 0;
	attMin = -1.0e30; attMax = 1.0e30; 
	nPieces = 1;
	this->SetNumberOfOutputs(1);
	for(int i=0; i<MAXINSTANCE; i++)
	{
		attPieceLo[i] = -1.0e30;
		attPieceHi[i] = 1.0e30;
	}

	sorted = false;

	logScale = true;

	histogramMaker = iHistogramMaker::New();
	if(histogramMaker == 0) reportNullPointer(5901);

	sortingMTime = 0;
	
}


iParticlesSplitter::~iParticlesSplitter()
{
	histogramMaker->Delete();
}

	
void iParticlesSplitter::getAttributeLimits(int n, float &amin, float &amax)
{
	if(n>=0 && n<nPieces && att>0)
	{
		amin = attPieceLo[n]; 
		amax = attPieceHi[n];
	} 
	else 
	{
		amin = -1.0e30;
		amax = 1.0e30;
	}
}


void iParticlesSplitter::setAttributeMinMax(float amin, float amax)
{
	if(amin < amax)
	{
		attMin = amin;
		attMax = amax;
	}
}


void iParticlesSplitter::setAttributeLimits(int n, float amin, float amax)
{

	if(n>=0 && n<nPieces && amin<amax && att>0)
	{
		//
		//  Make sure that there is at least 1 data point between the limits
		//
/*		vtkPolyData *input = this->GetInput();
		vtkFloatArray *inatt = (vtkFloatArray *)input->GetPointData()->GetScalars();
		float *d = inatt->GetPointer(0);
		int ncom = inatt->GetNumberOfComponents();
		vtkIdType ntup = inatt->GetNumberOfTuples();
		
		vtkIdType l;
		bool found = false;
		for(l=0; l<ntup && !found; l++)	if(*(d+att1+l*ncom)>=amin && *(d+att1+l*ncom)<=amax) found = true; 

		if(found)
		{
			attPieceLo[n] = (amin < attMin) ? attMin : amin;
			attPieceHi[n] = (amax > attMax) ? attMax : amax;
		}
*/
		//
		//  Set the limits even if there is no data in this piece
		//
		attPieceLo[n] = (amin < attMin) ? attMin : amin;
		attPieceHi[n] = (amax > attMax) ? attMax : amax;

		this->Modified();

	}

}


void iParticlesSplitter::setAttribute(int a)
{

	if(a == 0)
	{
		att = 0;
		attMin = -1.0e30; attMax = 1.0e30; 
		nPieces = 1;
		this->SetNumberOfOutputs(1);
		for(int i=0; i<MAXINSTANCE; i++)
		{
			attPieceLo[i] = -1.0e30;
			attPieceHi[i] = 1.0e30;
		}
		histogramMaker->SetInput(0,0);
		this->Modified();
		sorted = false;
		limitsSet = true;
		return;
	}
	//
	//  Reset to a single piece if the attribute changes
	//
	if(a!=att && a>0 && a<=myVTK->getReader()->getLimits(dataChannel)->getNatt())
	{

		limitsSet = false;
		att = a;
		nPieces = 1;
		this->SetNumberOfOutputs(1);

		sorted = false;
		histogramMaker->SetInput(this->GetInput(),att);
		this->Modified();

		for(int i=0; i<MAXINSTANCE; i++)
		{
			attPieceLo[i] = -1.0e30;
			attPieceHi[i] = 1.0e30;
		}
		this->updateLimits();

	}

}


void iParticlesSplitter::updateLimits()
{

	if(limitsSet) return;

	limitsSet = false;
	//
	//  Single piece mode
	//
	if(att == 0)
	{
		limitsSet = true;
		return;
	}

	vtkPolyData *input = this->GetInput();
	if(input == 0) return;

	vtkFloatArray *inatt = (vtkFloatArray *)input->GetPointData()->GetScalars();
	if(inatt == 0) return;
	float *data = inatt->GetPointer(0);
	if(data == 0) return;

	int ncom = inatt->GetNumberOfComponents();
	vtkIdType ntup = inatt->GetNumberOfTuples();

	if(att > ncom) return;

	vtkIdType l;
	float d;
	//
	//  Find attribute limits
	//
	attMin = attMax = *(data+att);
	for(l=1; l<ntup; l++)
	{
		d = *(data+att+l*ncom);
		if(d < attMin) attMin = d;
		if(d > attMax) attMax = d; 
	}
	//
	//  Spread limits a little bit so that even the limiting values are inside the interval
	//
	if(attMin < attMax)
	{
		//
		//  Find second lower and upper values
		//
		float attMin2, attMax2;
		attMin2 = attMax;
		attMax2 = attMin;
		for(l=0; l<ntup; l++)
		{
			d = *(data+att+l*ncom);
			if(d>attMin && d<attMin2) attMin2 = d;
			if(d<attMax && d>attMax2) attMax2 = d;
		}
		attMin -= 0.01*(attMin2-attMin);
		attMax += 0.01*(attMax-attMax2);
	}
	else
	{
		if(fabs(attMin) > 1.0) d = 0.0001*fabs(attMin); else d = 0.0001;
		attMin -= d;
		attMax += d;
	}

	int i;
	for(i=0; i<nPieces; i++)
	{
		if(attPieceLo[i] < attMin) attPieceLo[i] = attMin;
		if(attPieceHi[i] > attMax) attPieceHi[i] = attMax;
	}

	this->Modified();

	limitsSet = true;

}


int iParticlesSplitter::createNewPiece(float amin, float amax)
{

	if(nPieces < MAXINSTANCE)
	{
		attPieceLo[nPieces] = -1.0e30;
		attPieceHi[nPieces] = 1.0e30;

		vtkPolyData *newOutput = vtkPolyData::New();
		this->AddOutput(newOutput);
		newOutput->Delete();

		nPieces++;
		this->setAttributeLimits(nPieces-1,amin,amax);
		this->Modified();
		return nPieces-1;
	}
	return 0;

}
	

void iParticlesSplitter::deletePiece(int n)
{

	if(nPieces>1 && n>=0 && n<nPieces)
	{
		this->RemoveOutput(Outputs[n]);
		int i;
		for(i=n; i<nPieces-1; i++)
		{
			this->SetNthOutput(i,this->GetOutput(i+1));
		}
		nPieces--;
		if(n < nPieces) this->SetNumberOfOutputs(nPieces);
	}

}

	
void iParticlesSplitter::ExecuteData(vtkDataObject *)
{
	vtkPolyData *input = this->GetInput();
	if(input == 0) return;
	//
	//  Do it again in case the attribute was set before the data were loaded, like in
	//  restoring from a state file. 
	//
	if(!limitsSet) this->updateLimits();
	//
	//  If updateLimits did not set the limits, then the data are still not available, so exit without
	//  doing anything
	//
	if(!limitsSet) return;

	vtkPolyData *output = (vtkPolyData *)this->GetOutput();
	vtkPoints *newPoints;
	vtkCellArray *newVerts;
	vtkFloatArray *newNorms;
	vtkFloatArray *newAtts;

	if(att == 0){
		output->Initialize();
		output->ShallowCopy(input);
		return;
	}

	vtkPoints *inPoints = input->GetPoints();
	vtkCellArray *inVerts = input->GetVerts();
	vtkFloatArray *inNorms = (vtkFloatArray *)input->GetPointData()->GetNormals();
	vtkFloatArray *inAtts = (vtkFloatArray *)input->GetPointData()->GetScalars();

	float *pPointsF = 0;
	double *pPointsD = 0;

	bool pointsAreFloat;
	switch(inPoints->GetDataType())
	{
	case VTK_FLOAT:
		{
			pointsAreFloat = true;
			pPointsF = (float *)inPoints->GetVoidPointer(0);
			break;
		}
	case VTK_DOUBLE:
		{
			pointsAreFloat = false;
			pPointsD = (double *)inPoints->GetVoidPointer(0);
			break;
		}
	default: 
		{
			output->Initialize();
			output->ShallowCopy(input);
			vtkErrorMacro("Incorrect Points type");
			return;
		}
	}

	vtkIdType  *pVerts = (vtkIdType *)inVerts->GetPointer();
	float *pNorms = (float *)inNorms->GetPointer(0);
	float *pAtts = (float *)inAtts->GetPointer(0);

	int  ncom = inAtts->GetNumberOfComponents();
	vtkIdType ntup = inAtts->GetNumberOfTuples();

	if(!sorted || sortingMTime<input->GetMTime())
	{
		sorted = true;
		sortingMTime = input->GetMTime();
		if(pointsAreFloat) sort2(ntup,ncom,att,pPointsF,pAtts); else sort2(ntup,ncom,att,pPointsD,pAtts);
	}

	int n;
	vtkIdType iLo, iHi;
	vtkIdType iStart, iTotal;

	vtkFloatArray  *fArray;
	vtkDoubleArray *dArray;
	vtkIdTypeArray *iArray;

	for(n=0; n<nPieces; n++)
	{

		if(!find2(ntup,ncom,att,pAtts,0,attPieceLo[n],iLo,iHi))
		{
			vtkErrorMacro("Error #1 in iPieceSplitter");
			iLo = iHi = 0;
		}
		iStart = iHi;

		if(!find2(ntup,ncom,att,pAtts,iLo,attPieceHi[n],iLo,iHi))
		{
			vtkErrorMacro("Error #2 in iPieceSplitter");
			iLo = iHi = ntup - 1;
		}
		iTotal = iLo - iStart + 1;

		if(iTotal<=0 || iStart<0 || iLo>=ntup)
		{
			vtkErrorMacro("Error #3 in iPieceSplitter");
		}
		else
		{

			output = this->GetOutput(n);
			output->Initialize();
			
			newPoints = vtkPoints::New(inPoints->GetDataType());

			if(pointsAreFloat)
			{
				fArray = vtkFloatArray::New();
				fArray->SetNumberOfComponents(3);
				fArray->SetArray(pPointsF+3*iStart,3*iTotal,1);
				newPoints->SetData(fArray);
				fArray->Delete();
			}
			else
			{
				dArray = vtkDoubleArray::New();
				dArray->SetNumberOfComponents(3);
				dArray->SetArray(pPointsD+3*iStart,3*iTotal,1);
				newPoints->SetData(dArray);
				dArray->Delete();
			}
			output->SetPoints(newPoints);
			newPoints->Delete();

			newAtts = vtkFloatArray::New();
			newAtts->SetNumberOfComponents(ncom);
			newAtts->SetArray(pAtts+ncom*iStart,ncom*iTotal,1);
			output->GetPointData()->SetScalars(newAtts);
			newAtts->Delete();

			newNorms = vtkFloatArray::New();
			newNorms->SetNumberOfComponents(3);
			newNorms->SetArray(pNorms+3*iStart,3*iTotal,1);
			output->GetPointData()->SetNormals(newNorms);
			newNorms->Delete();

			newVerts = vtkCellArray::New();
			iArray = vtkIdTypeArray::New();
			iArray->SetNumberOfComponents(1);
			iArray->SetArray(pVerts,2*iTotal,1);
			newVerts->SetCells(iTotal,iArray);
			iArray->Delete();
			output->SetVerts(newVerts);
			newVerts->Delete();

		}

	}

}


void iParticlesSplitter::packState(iString &s)
{

	s = "";

	this->packValue(s,"iParticlesSplitter::logScale",logScale);
	this->packValue(s,"iParticlesSplitter::att",att);
	this->packValue(s,"iParticlesSplitter::nPieces",nPieces);
	this->packValue(s,"iParticlesSplitter::attMin",attMin);
	this->packValue(s,"iParticlesSplitter::attMax",attMax);

	this->packValue(s,"iParticlesSplitter::attPieceLo",attPieceLo,nPieces);
	this->packValue(s,"iParticlesSplitter::attPieceHi",attPieceHi,nPieces);

}


void iParticlesSplitter::unpackState(iString s)
{
	int i, n = nPieces; bool b; float f, f2;
	float v1[MAXINSTANCE], v2[MAXINSTANCE];

	if(this->unpackValue(s,"iParticlesSplitter::logScale",b)) this->setLogScale(b);
	if(this->unpackValue(s,"iParticlesSplitter::att",i)) this->setAttribute(i);
	this->unpackValue(s,"iParticlesSplitter::nPieces",n);
	if(this->unpackValue(s,"iParticlesSplitter::attMin",f) && this->unpackValue(s,"iParticlesSplitter::attMax",f2)) this->setAttributeMinMax(f,f2);

	if(this->unpackValue(s,"iParticlesSplitter::attPieceLo",v1,n) && this->unpackValue(s,"iParticlesSplitter::attPieceHi",v2,n))
	{
		for(i=0; i<nPieces && i<n; i++)	this->setAttributeLimits(i,v1[i],v2[i]);
		for(i=nPieces; i<n; i++) this->createNewPiece(v1[i],v2[i]);
		for(i=n; i<nPieces; i++) this->deletePiece(n);
	}

}

