/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of iguivolume.h
*/

#include "imath.h"
#include "ipiecewisefunction.h"

#include <vtkPiecewiseFunction.h>

void reportNullPointer(int);


//
// iPiecewiseFunction class
//

iPiecewiseFunction::iPiecewiseFunction(float yl, float yr)
{
	
	opax = new float[2];
	if(opax == 0) reportNullPointer(7001);
	opay = new float[2];
	if(opay == 0) reportNullPointer(7002);
	opaymin = 0.0;
	opaymax = 1.0;
	opax[0] = 0.0;
	opax[1] = 1.0;
	yl = (yl < opaymin) ? opaymin : yl;
	yl = (yl > opaymax) ? opaymax : yl;
	yr = (yr < opaymin) ? opaymin : yr;
	yr = (yr > opaymax) ? opaymax : yr;
	opay[0] = yl;
	opay[1] = yr;
	opan = 2;
	
	vtkof = vtkPiecewiseFunction::New();
	if(vtkof == 0) reportNullPointer(7003);
	setVTKFunction();
	
}

iPiecewiseFunction::~iPiecewiseFunction()
{
	
	delete[] opax;
	delete[] opay;
	vtkof->Delete();
	
}

void iPiecewiseFunction::copy(iPiecewiseFunction *f)
{

	if(this==f || f==0) return;

	int n = f->getFunctionN();
	float *x = f->getFunctionX();
	float *y = f->getFunctionY();

	delete[] opax;
	delete[] opay;
	opax = new float[n];
	if(opax == 0) reportNullPointer(7004);
	opay = new float[n];
	if(opay == 0) reportNullPointer(7005);
	if(n < 30)
	{
		for(int i=0; i<n; i++) 
		{
			opax[i] = x[i];
			opay[i] = y[i];
		}
	}
	else
	{
		memcpy(opax,x,n*sizeof(float));
		memcpy(opay,y,n*sizeof(float));
	}
	opan = n;
	setVTKFunction();

}


void iPiecewiseFunction::setFunctionLimits(float ymin, float ymax)
{
	
	opaymin = ymin;
	opaymax = ymax;
	for(int i=0; i<opan; i++) 
	{
		opay[i] = (opay[i] > opaymax) ? opaymax : opay[i];
		opay[i] = (opay[i] < opaymin) ? opaymin : opay[i];
	}
	setVTKFunction();
	
}

void iPiecewiseFunction::addFunctionPoint(int n) 
{
	
	int i;
	if(n>=0 && n<opan) 
	{
		if(n == opan-1) n = opan-2;
		float *xn = new float[opan+1];	
		if(xn == 0) reportNullPointer(7006);
		float *yn = new float[opan+1];	
		if(yn == 0) reportNullPointer(7007);
		for(i=0; i<=n; i++) 
		{
			xn[i] = opax[i];
			yn[i] = opay[i];
		}
		n++;
		xn[n] = (float)0.5*(opax[n-1]+opax[n]); 
		yn[n] = (float)0.5*(opay[n-1]+opay[n]); 
		for(i=n+1; i<=opan; i++) 
		{
			xn[i] = opax[i-1];
			yn[i] = opay[i-1];
		}
		vtkof->AddPoint(255*xn[n],yn[n]);
		delete[] opax;
		delete[] opay;
		opax = xn;
		opay = yn;
		opan++;
	}
	
}

void iPiecewiseFunction::addFunctionPoint(float x, float y) 
{
	
	int i,n;
	if(x>opax[0] && x<opax[opan-1]) 
	{

		n = 0;
		while(opax[n] < x) n++;
		
		float *xn = new float[opan+1];	
		if(xn == 0) reportNullPointer(7008);
		float *yn = new float[opan+1];	
		if(yn == 0) reportNullPointer(7009);
		for(i=0; i<n; i++) 
		{
			xn[i] = opax[i];
			yn[i] = opay[i];
		}
		xn[n] = x; 
		yn[n] = y; 
		for(i=n+1; i<=opan; i++) 
		{
			xn[i] = opax[i-1];
			yn[i] = opay[i-1];
		}
		vtkof->AddPoint(255*x,y);
		delete[] opax;
		delete[] opay;
		opax = xn;
		opay = yn;
		opan++;
	}
	
}

void iPiecewiseFunction::delFunctionPoint(int n) 
{
	
	int i;
	if(n>0 && n<opan-1 && opan > 2) 
	{
		float *xn = new float[opan-1];	
		if(xn == 0) reportNullPointer(7010);
		float *yn = new float[opan-1];	
		if(yn == 0) reportNullPointer(7011);
		for(i=0; i<n; i++) 
		{
			xn[i] = opax[i];
			yn[i] = opay[i];
		}
		for(i=n; i<opan-1; i++) 
		{
			xn[i] = opax[i+1];
			yn[i] = opay[i+1];
		}
		vtkof->RemovePoint(255*opax[n]);
		delete[] opax;
		delete[] opay;
		opax = xn;
		opay = yn;
		opan--;
	}
	
}

void iPiecewiseFunction::movFunctionPoint(int n, float x, float y) 
{

	if(n>0 && n<opan-1) 
	{
		float xl = opax[n-1];
		float xr = opax[n+1];
		x = (x < xl) ? xl : x;
		x = (x > xr) ? xr : x;
		y = (y < opaymin) ? opaymin : y;
		y = (y > opaymax) ? opaymax : y;
		vtkof->RemovePoint(255*opax[n]);
		opax[n] = x;
		opay[n] = y;
		vtkof->AddPoint(255*opax[n],opay[n]);
	} 
	else if(n==0 || n==opan-1) 
	{
		y = (y < opaymin) ? opaymin : y;
		y = (y > opaymax) ? opaymax : y;
		vtkof->RemovePoint(255*opax[n]);
		opay[n] = y;
		vtkof->AddPoint(255*opax[n],opay[n]);
	}
	
}

void iPiecewiseFunction::getFunctionPoint(int n, float &x, float &y)
{
	
	if(n>0 && n<opan-1) 
	{
		x = opax[n];
		y = opay[n];
	} 
	else 
	{
		x = 0.0;
		y = 0.0;
	}
	
}


int iPiecewiseFunction::findFunctionPoint(float x, float y, float &dmin)
{
	
	int i, imin;
	float d, dx, dy;
	
	dmin = 1.0e30;
	imin = -1;
	for(i=0; i<opan; i++) 
	{
		dx = x - opax[i];
		dy = y - opay[i];
		d = dx*dx + dy*dy;
		if(d < dmin) 
		{
			dmin = d;
			imin = i;
		}
	}
	
	dmin = sqrt(dmin);
	return imin;
	
}


float iPiecewiseFunction::getValue(float x)
{
	
	int i;
	
	if(x < opax[0]) return opay[0];
	if(x > opax[opan-1]) return opay[opan-1];

	i = 0;
	while(i<opan && opax[i]<x) i++;
	if(i == 0) i = 1;
	if(i > opan-1) i = opan - 1;
	return opay[i] + (opay[i]-opay[i-1])/(opax[i]-opax[i-1]+1.0e-30)*(x-opax[i]);
	
}


void iPiecewiseFunction::setVTKFunction()
{
	vtkof->RemoveAllPoints();
	for(int i=0; i<opan; i++) vtkof->AddPoint(255*opax[i],opay[i]);
}
