/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ipostscriptwriter.h"

#include <vtkImageData.h>
#include <vtkPointData.h>

#define VTK_MARGIN 0.95


Paper iPostScriptWriter::paper[NFORMATS];


iPostScriptWriter::iPostScriptWriter()
{
	int i = 0;
	if(i < NFORMATS) { paper[i].width = 33.0556; paper[i].height = 46.7778; paper[i].name = "A0"; i++; }
	if(i < NFORMATS) { paper[i].width = 23.3889; paper[i].height = 33.0556; paper[i].name = "A1"; i++; }
	if(i < NFORMATS) { paper[i].width = 16.5278; paper[i].height = 23.3889; paper[i].name = "A2"; i++; }
	if(i < NFORMATS) { paper[i].width = 11.6944; paper[i].height = 16.5278; paper[i].name = "A3"; i++; }
	if(i < NFORMATS) { paper[i].width = 8.26389; paper[i].height = 11.6944; paper[i].name = "A4"; i++; }
	if(i < NFORMATS) { paper[i].width = 5.84772; paper[i].height = 8.26389; paper[i].name = "A5"; i++; }
	if(i < NFORMATS) { paper[i].width = 4.12500; paper[i].height = 5.84772; paper[i].name = "A6"; i++; }
	if(i < NFORMATS) { paper[i].width = 2.91667; paper[i].height = 4.12500; paper[i].name = "A7"; i++; }
	if(i < NFORMATS) { paper[i].width = 2.05556; paper[i].height = 2.91667; paper[i].name = "A8"; i++; }
	if(i < NFORMATS) { paper[i].width = 8.5    ; paper[i].height = 11.0   ; paper[i].name = "Letter"; i++; }
	if(i < NFORMATS) { paper[i].width = 11.0   ; paper[i].height = 17.0   ; paper[i].name = "11x17"; i++; }
	
	format = 9;
	orient = 0;

}


void iPostScriptWriter::setPaperFormat(int f)
{
	if(f>=0 && f<NFORMATS) format = f;
}


void iPostScriptWriter::setOrientation(int o)
{
	if(o>=0 && o<=1) orient = o;
}


void iPostScriptWriter::WriteFileHeader(ofstream *file, vtkImageData *cache)
{
	//
	//  Copied from vtkPostScriptWriter.cxx with addition of options to specify paper size and orientation.
	//
	int min1, max1, min2, max2, min3, max3;
	int bpp;
	int cols, rows, scols, srows;
	float scale = 1;
	int pagewid = (int)(paper[format].width*72);
	int pagehgt = (int)(paper[format].height*72);
	
	// Find the length of the rows to write.
	cache->GetWholeExtent(min1, max1, min2, max2, min3, max3);
	bpp = cache->GetNumberOfScalarComponents();
	
	cols = max1 - min1 + 1;
	rows = max2 - min2 + 1;

	if(orient == 1)
	{
		//
		//  Landscape orientation
		//
		int q = cols;
		cols = rows;
		rows = q;
	}
	//
	//  Fit to width
	//
	scale = (float)pagewid/cols;
	scols = (int)(scale*cols*VTK_MARGIN);
	srows = (int)(scale*rows*VTK_MARGIN);
	//
	//  If too high, fit to height
	//
	if(srows > pagehgt*VTK_MARGIN)
    {
		scale = scale*(pagehgt*VTK_MARGIN/srows);
		scols = (int)(scale*cols*VTK_MARGIN);
		srows = (int)(scale*rows*VTK_MARGIN);
    }
	float llx = (pagewid-scols) / 2;
	float lly = (pagehgt-srows) / 2;
	
	// spit out the PostScript header
	*file << "%!PS-Adobe-2.0 EPSF-2.0\n";
	*file << "%%Creator: IFRIT\n";
	*file << "%%Title: " << this->InternalFileName << endl;
	*file << "%%Pages: 1\n";
	*file << "%%BoundingBox: " << (int) llx << " "  << (int) lly
		<< " " << (int) ( llx + scols + 0.5 ) << " " << 
		(int) ( lly + srows + 0.5 ) << endl;
	*file << "%%EndComments\n";
	*file << "/readstring {\n";
	*file << "  currentfile exch readhexstring pop\n";
	*file << "} bind def\n";
	
	if ( bpp == 3)
    {
		*file << "/rpicstr " << cols << " string def\n";
		*file << "/gpicstr " << cols << " string def\n";
		*file << "/bpicstr " << cols << " string def\n";
    }
	else if (bpp == 1)
    {
		*file << "/picstr " << cols << " string def\n";
    }
	else 
    {
		vtkWarningMacro( " vtkPostScriptWriter only supports 1 and 3 component images");
    }
	
	*file << "%%EndProlog\n";
	*file << "%%Page: 1 1\n";
	*file << "gsave\n";
	*file << llx << " " << lly << " translate\n";
	*file << scols << " " << srows << " scale\n";
	*file << cols << " " << rows << " 8\n";
	*file << "[ " << cols << " 0 0 " << -rows << " 0 " << rows << " ]\n";
	if (bpp == 3)
    {
		*file << "{ rpicstr readstring }\n";
		*file << "{ gpicstr readstring }\n";
		*file << "{ bpicstr readstring }\n";
		*file << "true 3\n";
		*file << "colorimage\n";
    }
	else
    {
		*file << "{ picstr readstring }\n";
		*file << "image\n";
    }
}


void iPostScriptWriter::WriteFile(ofstream *file, vtkImageData *data, int extent[6])
{
	//
	//  Copied from vtkPostScriptWriter.cxx with addition of options to specify paper size and orientation.
	//
	int idxC, idx0, idx1, idx2;
	unsigned char *ptr;
	unsigned long count = 0;
	unsigned long target;
	float progress = this->Progress;
	float area;
	int *wExtent;
	static int itemsperline = 0;
	char* hexits = (char *) "0123456789abcdef";
	
	// Make sure we actually have data.
	if ( !data->GetPointData()->GetScalars())
    {
		vtkErrorMacro(<< "Could not get data from input.");
		return;
    }
	
	// take into consideration the scalar type
	switch (data->GetScalarType())
    {
    case VTK_UNSIGNED_CHAR:
		break;
    default:
		vtkErrorMacro("PostScriptWriter only accepts unsigned char scalars!");
		return; 
    }
	
	int i0 = 0, i1 = 1, i2 = 2, i3 = 3;
	if(orient == 1)
	{
		i0 = 2; i1 = 3; i2 = 0; i3 = 1;
	}

	wExtent = this->GetInput()->GetWholeExtent();
	area = ((extent[5] - extent[4] + 1)*(extent[3] - extent[2] + 1)*
		(extent[1] - extent[0] + 1)) / 
		((wExtent[5] -wExtent[4] + 1)*(wExtent[3] -wExtent[2] + 1)*
		(wExtent[1] -wExtent[0] + 1));
    
	
	int numComponents = data->GetNumberOfScalarComponents();
	// ignore alpha
	int maxComponent = numComponents;
	if (numComponents == 2) 
    {
		maxComponent = 1;
    }
	if (numComponents == 4) 
    {
		maxComponent = 3;
    }
	target = (unsigned long)((extent[5]-extent[4]+1)*
		(extent[i3]-extent[i2]+1)/(50.0*area));
	target++;
	
	int ptrOffset = numComponents;
	int idx1run;
	if(orient == 1)
	{
		ptrOffset = numComponents*(extent[1]-extent[0]+1);
	}

	for (idx2 = extent[4]; idx2 <= extent[5]; ++idx2)
    {
		for (idx1run = extent[i3]; idx1run >= extent[i2]; idx1run--)
		{
			//
			//  Orientation
			//
			idx1 = idx1run;
			if(orient == 1)
			{
				idx1 = extent[i3] + extent[i2] - idx1run;
			}

			if (!(count%target))
			{
				this->UpdateProgress(progress + count/(50.0*target));
			}
			count++;
			// write out components one at a time because
			for (idxC = 0; idxC < maxComponent; idxC++)
			{
				//
				//  Orientation
				//
				if(orient == 1)
				{
					ptr = (unsigned char*)data->GetScalarPointer(idx1,extent[2],idx2);
				}
				else
				{
					ptr = (unsigned char*)data->GetScalarPointer(extent[0],idx1,idx2);
				}

				ptr += idxC;
				for ( idx0 = extent[i0]; idx0 <= extent[i1]; idx0++ )
				{
					if ( itemsperline == 30 )
					{
						*file << endl;
						itemsperline = 0;
					}
					*file << hexits[*ptr >> 4] << hexits[*ptr & 15];
					ptr += ptrOffset;
				}
			}
		}
    }
	
}

