/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
  
//
//  This class is based on original vtkQGLRenderWindow class written by Manish P. Pagey [pagey@drcsdca.com ]
//

#include "iglobals.h"
#include "iopengl.h"
#include "irenderwindow.h"


#ifdef IOPENGL_X11
#include <X11/Xlib.h>
#endif

#include <vtkRenderWindowInteractor.h>
//#include <vtkUnsignedCharArray.h>
//#include <vtkFloatArray.h>

#include <qapplication.h>
#include <qdesktopwidget.h>


iRenderWindow::iRenderWindow(QWidget *parent, const char *name) :  QGLWidget(parent, name)
{
	this->mInitialized = false;
	this->setFocusPolicy(QWidget::StrongFocus);
	GenericContext = NULL;
	GenericDisplayId = NULL;
	inRender = false;
}


iRenderWindow::~iRenderWindow() 
{
	this->ReferenceCount-- ;
}


void iRenderWindow::Start(void) 
{
	// 
	// Initialize the QGLWidget part of the widget if it has not
	// been initialized so far. 
	//
	if(!this->mInitialized ) this->WindowInitialize();
	this->MakeCurrent();
}

// End the rendering process and display the image. 
void iRenderWindow::Frame(void) 
{
	glFlush() ;  //QT flushes if not double buffer
	if( (! this->AbortRender) && // the render is not being aborted
		(! autoBufferSwap() ) && // buffers are not switched automatically
		doubleBuffer() &&        // double buffering is enabled on QGLWidget side
		this->vtkRenderWindow::DoubleBuffer &&    // double buffering is enabled on VTK side
		this->SwapBuffers ) {    // VTK wants us to swap buffers
		QGLWidget::swapBuffers();
	}
}

// Initialize the window for rendering.
void iRenderWindow::WindowInitialize(void) 
{
	if(!this->mInitialized ) 
	{
		this->mInitialized = true;
		this->MakeCurrent();
		
		vtkDebugMacro(<< " glMatrixMode ModelView\n");
		glMatrixMode( GL_MODELVIEW );
		
		vtkDebugMacro(<< " zbuffer enabled\n");
		glDepthFunc( GL_LEQUAL );
		glEnable( GL_DEPTH_TEST );
		
		vtkDebugMacro(" texture stuff\n");
		glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
		
		// initialize blending for transparency
		vtkDebugMacro(<< " blend func stuff\n");
		glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
		glEnable(GL_BLEND);
		
		if (this->PointSmoothing)
		{
			glEnable(GL_POINT_SMOOTH);
		}
		else
		{
			glDisable(GL_POINT_SMOOTH);
		}
		
		if (this->LineSmoothing)
		{
			glEnable(GL_LINE_SMOOTH);
		}
		else
		{
			glDisable(GL_LINE_SMOOTH);
		}
		
		if (this->PolygonSmoothing)
		{
			glEnable(GL_POLYGON_SMOOTH);
		}
		else
		{
			glDisable(GL_POLYGON_SMOOTH);
		}
		
		glEnable( GL_NORMALIZE );
		glAlphaFunc(GL_GREATER,0);
		//    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_true);
		
		this->Mapped = 1;
		//
		// Enable mouse tracking. It is required by trackball interactor style. 
		//
		this->setMouseTracking(true);
		
	}
}

void iRenderWindow::SetFullScreen(int arg) 
{
	//
	// We do not need to do anything if the FullScreen mode
	// is already set to the specified value. 
	//
	if(this->FullScreen==arg || !this->mInitialized) return;
	
	// set the mode 
	this->FullScreen = arg;
	if(arg) this->showFullScreen(); else this->showNormal();
	this->Modified();
	
}

void iRenderWindow::WindowRemap(void) 
{
	//
	// :TODO: Fri Apr 21 16:44:35 2000 Pagey
	// I am not sure why we would ever need to do this under Qt. 
	// Hence, I have not done anything here yet. 
	//
	vtkDebugMacro(<< " QGLWidget::WindowRemap() not supported by QGLWidget yet.\n");
}

void iRenderWindow::PrefFullScreen(void) 
{
	//
	// :TODO: Fri Apr 21 16:46:30 2000 Pagey
	// Since, SetFullScreen() is not supported yet, this is useless.
	//
	vtkDebugMacro(<< " QGLWidget::PrefFullScreen() not supported by QGLWidget yet.\n");
}

void iRenderWindow::SetSize(int w, int h)
{
	if(this->Size[0]!=w || this->Size[1]!=h) 
	{
		this->Modified();
		this->Size[0] = w;
		this->Size[1] = h;
	}
	
	if(this->Interactor) this->Interactor->SetSize(w, h);
	
	// if we arent mapped then just set the ivars 
	if (!this->Mapped) return;
	
	glViewport(0,0,(GLint)w,(GLint)h);
}


void iRenderWindow::MakeCurrent() 
{
	if(this->mInitialized) QGLWidget::makeCurrent();
}


void iRenderWindow::initializeGL() 
{
	//
	//  I don't think we need this - it crashes X-server over the remote connection
	//
	//this->WindowInitialize();
}

void iRenderWindow::paintGL() 
{
	if(inRender)
	{
		vtkErrorMacro("Nested paintGL call");
		return;
	}
	inRender = true;
	//
	//  Must do it this way so that child Render operations are not called from here
	//
	iRenderWindow::Render();
	inRender = false;
}

void iRenderWindow::resizeGL(int w, int h) 
{
	SetSize(w, h);
}

void iRenderWindow::setContext(QGLContext *context, const QGLContext *shareContext, bool deleteOldContex) 
{
	mInitialized = false;
	QGLWidget::setContext(context, shareContext, deleteOldContex);
}

void iRenderWindow::mouseMoveEvent(QMouseEvent *event) 
{
	//
	//  block second rendering - interactor already renders once
	//
	emit MouseMove(event);
}

void iRenderWindow::mousePressEvent(QMouseEvent *event) 
{
	//
	// Emit a ButtonPressed signal for all mouse presses.
	//
	emit ButtonPressed(event);
	//
	// Emit a signal for the button which was pressed.
	//
	switch(event->button()) 
	{
	case LeftButton:
		emit LeftButtonPressed(event);
		break;
	case MidButton:
		emit MiddleButtonPressed(event);
		break;
	case RightButton:
		emit RightButtonPressed(event);
		break;
	default:
		break;
	}
}


void iRenderWindow::mouseReleaseEvent(QMouseEvent *event) 
{
	//
	// Emit a ButtonPressed signal for all mouse releases.
	//
	emit ButtonReleased(event);
	//
	// Emit a signal for the the mouse button which was
	// released.
	//
	switch(event->button()) 
	{
	case LeftButton:
		emit LeftButtonReleased(event);
		break;
	case MidButton:
		emit MiddleButtonReleased(event);
		break;
	case RightButton:
		emit RightButtonReleased(event);
		break;
	default:
		break;
	}
}

void iRenderWindow::keyPressEvent(QKeyEvent * event)
{
	//
	// Let the interactor handle this.
	//
	emit KeyPressed(event);
}


void *iRenderWindow::GetGenericContext()
{

	if(GenericContext == NULL)
	{
#ifdef IOPENGL_WIN
		GenericContext = (void *)GetDC(this->winId());
#else
		GenericContext = XCreateGC((Display *)this->GetGenericDisplayId(),this->winId(),0,0);
#endif
    }
	return GenericContext;

}


void *iRenderWindow::GetGenericDisplayId()
{

	if(GenericDisplayId == NULL)
	{
#ifdef IOPENGL_WIN
		GenericDisplayId = (void *)wglGetCurrentContext();
#else
		GenericDisplayId = XOpenDisplay((char *)NULL);
#endif
	}

	return GenericDisplayId;

}

