/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iglobals.h"
#include "iruler.h"
#include "ivtk.h"

#include <vtkMath.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>
#include <vtkProperty2D.h>
#include <vtkTextMapper.h>
#ifndef I_VTK_VERSION_40
#include <vtkTextProperty.h>
#endif

void reportNullPointer(int);


iRuler* iRuler::New()
{

	return new iRuler;

}


iRuler::iRuler()
{

	fontSize = 12;
	started = false;
	scale = 1.0;
	lineWidth = this->AxisActor->GetProperty()->GetLineWidth();
	tickLength = TickLength;

#ifndef I_VTK_VERSION_40
	this->GetLabelTextProperty()->BoldOn();
	this->GetLabelTextProperty()->ItalicOff();
	this->GetLabelTextProperty()->ShadowOn();
	this->GetTitleTextProperty()->BoldOn();
	this->GetTitleTextProperty()->ItalicOff();
	this->GetTitleTextProperty()->ShadowOn();
	this->LabelMappers[0]->SetTextProperty(this->LabelTextProperty);
	this->LabelMappers[1]->SetTextProperty(this->LabelTextProperty);
	this->LabelTextProperty->SetVerticalJustificationToTop();
#else
	this->BoldOn();
	this->ItalicOff();
	this->ShadowOn();
#endif
	this->SetPoint1(0.25,0.9);
	this->SetPoint2(0.75,0.9);
	this->SetRange(0.0,1.0);
	this->AdjustLabelsOff();
	this->SetNumberOfLabels(2);

	this->PickableOff();

	this->setColor(0.0,0.0,0.0);

}


iRuler::~iRuler()
{

}


void iRuler::ReleaseGraphicsResources(vtkWindow *win)
{
	this->vtkAxisActor2D::ReleaseGraphicsResources(win);
}


int iRuler::RenderOverlay(vtkViewport* viewport)
{

	iVTK *vtk = iVTK::convertFromViewport(viewport);
	int mag = vtk->getRenderingMagnification();

	if(mag == 1)
	{
#ifndef I_VTK_VERSION_40
		this->LabelTextProperty->SetFontSize(fontSize);
#else
		this->SetLabelFactor((float)fontSize/16.0);
#endif
	}
	else
	{
#ifndef I_VTK_VERSION_40
		this->LabelTextProperty->SetFontSize(mag*fontSize);
#else
		this->SetLabelFactor(mag*(float)fontSize/16.0);
#endif
	}
	
	return this->vtkAxisActor2D::RenderOverlay(viewport);

}


int iRuler::RenderOpaqueGeometry(vtkViewport* viewport)
{

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;
	
	iVTK *vtk = iVTK::convertFromViewport(viewport);
	int mag = vtk->getRenderingMagnification();

	vtkFloat x0[3], x1[3];
	vtkFloat dummy = 0.0;
	
	int k;
	vtkFloat vx[3], vy[3], vz[3];
	
	if(mag == 1)
	{
		
		cam->GetViewUp(vy);
		cam->GetDirectionOfProjection(vz);
		for(k=0; k<3; k++) vz[k] = -vz[k];
		
		vtkMath::Cross(vy,vz,vx);
		
		viewport->SetWorldPoint(vx[0],vx[1],vx[2],1.0);
		viewport->WorldToDisplay();
		viewport->GetDisplayPoint(x0);
		viewport->SetWorldPoint(-vx[0],-vx[1],-vx[2],1.0);
		viewport->WorldToDisplay();
		viewport->GetDisplayPoint(x1);
		
		vtkFloat d = sqrt(vtkMath::Distance2BetweenPoints(x1,x0));
		viewport->DisplayToNormalizedDisplay(d,dummy);
		
		if(!started)
		{
			started = true;
			scale = cam->GetParallelScale();
		}
		
		float s = cam->GetParallelScale()/scale;
		dx = 0.5*d*s;
		dx = (dx > 0.3) ? 0.3 : dx;

		if(!vtk->isUsingOpenGLCoordinates()) s *= vtk->getBoxSize();
		this->SetRange(0.0,s);
		
		this->SetPoint1(0.5-dx,0.9);
		this->SetPoint2(0.5+dx,0.9);

		this->AxisActor->GetProperty()->SetLineWidth(lineWidth);
		this->SetTickLength(tickLength);
		
	}
	else
	{

		int winij[2];
		vtk->computePositionShiftsUnderMagnification(winij,viewport);
		
		this->SetPoint1(mag*(0.5-dx)-winij[0],mag*0.9-winij[1]);
		this->SetPoint2(mag*(0.5+dx)-winij[0],mag*0.9-winij[1]);
		
		this->AxisActor->GetProperty()->SetLineWidth(lineWidth*mag);
		this->SetTickLength(mag*tickLength);

	}
	
	this->vtkAxisActor2D::RenderOpaqueGeometry(viewport);
	
	return 1;
	
}


void iRuler::setFontSize(int s)
{ 
	if(s>0 && s<99) fontSize = s; 
}


void iRuler::setColor(float r, float g, float b)
{ 
	this->GetProperty()->SetColor(r,g,b); 
}


void iRuler::setInitialScale(float s)
{
	started = true;
	scale = s;
}


