/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include <stdlib.h>

#include "istring.h"
#include "iscript.h"
#include "imath.h"

#include "iexpressionparser.h"


#ifdef _DEBUG
int iValue::counterCreate = 0;
int iValue::counterDestroy = 0;
int iValue::cval[999];
#include <qmessagebox.h>
#endif


void reportNullPointer(int);
void reportUnallowedPath()
{
#ifdef _DEBUG
	QMessageBox::information(0,"FATAL","Bug in iScrip: unallowed path. Please report.");
#endif
	exit(-1);
}


#define ASSIGN_VALUE(v,rel,rhs) \
{ \
	if(rel & 1) { if(rel & 4) v = (rhs) - v; else v += (rhs); } else { if(rel & 2) v *= (rhs); else v = (rhs); } \
}


using namespace iScriptType;
#define NotUsed(x)

//
//***************************************************************
//
//  iValue functions
//
//***************************************************************
//


#ifdef _DEBUG

iValue::iValue(iString n, iScriptType::ValueType t, iScriptType::ValueMethod m)
{ 
	ReferenceCount = 1; 
	Name = n, Type = t; Method = m; 
	counterAtCreation = counterCreate;
	if(counterCreate < 999) cval[counterCreate] = 1;
	counterCreate++;

	if(counterAtCreation == 122)
	{
		int i = 0;
	}

}

iValue::~iValue()
{ 
	if(counterAtCreation == 11)
	{
		int i = 0;
	}
	if(counterAtCreation < 999) cval[counterAtCreation] = 0;
	counterDestroy++;
}

#endif


//
//  Copy constructor
//
iValue* iValue::New(iValue *v0, bool ref)
{
	
	if(v0 == NULL) return NULL;

	switch(v0->Type)
	{
	case IVALUE_BL: 
		{ 
			iValue_BL *v = (iValue_BL *)v0; 
			if(!ref || v->Method!=ByValue) 
			{
				return iValue::New(v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Name,ByPointer,v->Value,&v->Value,NULL); 
			}
		}
	case IVALUE_IS: 
		{ 
			iValue_IS *v = (iValue_IS *)v0; 
			if(!ref || v->Method!=ByValue) 
			{
				return iValue::New(v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Name,ByPointer,v->Value,&v->Value,NULL); 
			}
		}
	case IVALUE_FS: 
		{ 
			iValue_FS *v = (iValue_FS *)v0; 
			if(!ref || v->Method!=ByValue) 
			{
				return iValue::New(v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Name,ByPointer,v->Value,&v->Value,NULL); 
			}
		}
	case IVALUE_FP: 
		{ 
			iValue_FP *v = (iValue_FP *)v0; 
			if(!ref || v->Method!=ByValue) 
			{
				return iValue::New(v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Name,ByPointer,v->Value,&v->Value,NULL); 
			}
		}
	case IVALUE_ST: 
		{ 
			iValue_ST *v = (iValue_ST *)v0; 
			if(!ref || v->Method!=ByValue) 
			{
				return iValue::New(v->Name,v->Method,v->Value,v->Pointer,v->Function); 
			}
			else
			{
				return iValue::New(v->Name,ByPointer,v->Value,&v->Value,NULL); 
			}
		}
	case IVALUE_VA: 
		{ 
			iValue_VA *v = (iValue_VA *)v0;
			iValue** arr = NULL;
			if(v->Size>0 && v->Array!=NULL)
			{
				arr = new iValue*[v->Size];
				int i;
				for(i=0; i<v->Size; i++) 
				{
					if(v->Array[i] != NULL)
					{
						arr[i] = iValue::New(v->Array[i],ref);  // these are not registered - they are owned by iValue_VA
					}
					else
					{
						arr[i] = NULL;
					}
				}
			}
			return iValue::New(v->Name,v->Method,v->Size,iValue::New(v->Index),arr,v->Function);
		}
	default: { reportUnallowedPath(); return NULL; }
	}
}

//
//  Check that the types are compatible: the order does matter
//
bool iValue::areTypesCompatible(iValue *v1, iValue *v2)
{
	if(v1 == NULL) return false; else return iValue::areTypesCompatible(v1->Type,v2);
}


bool iValue::areTypesCompatible(ValueType t, iValue *v2)
{
	if(v2 == NULL) return false;
	if(t == v2->Type) return true; else
	{
		switch(v2->Type)
		{
			case IVALUE_BL: return false;
			case IVALUE_IS: return (t == IVALUE_FS);
			case IVALUE_FS: return (t == IVALUE_IS);
			case IVALUE_FP: return false;
			case IVALUE_VA: return false;
			default: { reportUnallowedPath(); return false; }
		}
	}
}


iString iValue::getValueAsText()
{
	iString s;

	switch(Type)
	{
	case IVALUE_BL: 
		{
			bool v;
			if(this->getValue(v)) { if(v) s = "true"; else s = "false"; } else s = "undefined";
			return s;
		}
	case IVALUE_IS: 
		{
			int v;
			if(this->getValue(v)) s.setNum(v); else s = "undefined";
			return s;
		}
	case IVALUE_FS: 
		{
			float v;
			if(this->getValue(v)) s.setNum(v); else s = "undefined";
			return s;
		}
	case IVALUE_FP: 
		{
			float *v;
			if(this->getValue(&v)) s.setNum((long)v,16); else s = "undefined";
			return s;
		}
	case IVALUE_VA: 
		{
			int i;
			iString s1;
			for(i=1; i<=((iValue_VA*)this)->getSize() && i<=10; i++)
			{
				if(((iValue_VA*)this)->getComponent(i) != NULL)
				{
					s += ((iValue_VA*)this)->getComponent(i)->getValueAsText();
				}
				else
				{
					s += "undefined";
				}
				s += " ";
			}
			if(((iValue_VA*)this)->getSize() > 10) s += "..."; 
			return s;
		}
	default: { reportUnallowedPath(); return ""; }
	}
}


iString iValue::getTypeAsText()
{
	switch(Type)
	{
	case IVALUE_BL: 
		{
			return "bool scalar";
		}
	case IVALUE_IS: 
		{
			return "int scalar";
		}
	case IVALUE_FS: 
		{
			return "float scalar";
		}
	case IVALUE_FP: 
		{
			return "float pointer";
		}
	case IVALUE_VA: 
		{
			iString s;
			s.setNum(((iValue_VA*)this)->getSize());
			return "array[" + s + "]";
		}
	default: { reportUnallowedPath(); return ""; }
	}
}


void iValue::Delete()
{
	if(--ReferenceCount <= 0) delete this;
}


//
//  BL value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, bool v, bool *p, bool (*f)(void))
{
	return (iValue*)new iValue_BL(n,m,v,p,f);
}


bool iValue_BL::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { if(Pointer != NULL) { Method = m; return true; } else return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_BL::assignValue(Assignment r, iValue *v)
{
	bool b;
	if(v->getValue(b)) return this->assignValue(r,b); else return false;
}


bool iValue_BL::assignValue(Assignment NotUsed(r), bool b)
{
	Value = b; 
	return true; 
}


bool iValue_BL::getValue(void *p)
{
	if(p == NULL) return false;
	switch (Method)
	{
	case ByValue:    { *(bool *)p = Value; return true; }
	case ByPointer:  { if(Pointer != NULL) { *(bool *)p = *Pointer; return true; } else return false; }
	case ByFunction: { *(bool *)p = Function(); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_BL::getValue(bool &v)
{
	return this->getValue((void *)&v);
}


//
//  IS value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, int v, int *p, int (*f)(void))
{
	return (iValue*)new iValue_IS(n,m,v,p,f);
}


bool iValue_IS::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { if(Pointer != NULL) { Method = m; return true; } else return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_IS::assignValue(Assignment r, iValue *v)
{
	int i;
	if(v->getValue(i)) return this->assignValue(r,i); else return false;
}


bool iValue_IS::assignValue(Assignment r, int i)
{
	ASSIGN_VALUE(Value,r,i); 
	return true; 
}


bool iValue_IS::assignValue(Assignment r, float f)
{
	ASSIGN_VALUE(Value,r,(int)f); 
	return true; 
}


bool iValue_IS::getValue(void *p)
{
	if(p == NULL) return false;
	switch (Method)
	{
	case ByValue:    { *(int *)p = Value; return true; }
	case ByPointer:  { if(Pointer != NULL) { *(int *)p = *Pointer; return true; } else return false; }
	case ByFunction: { *(int *)p = Function(); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_IS::getValue(bool &v)
{
	int i;
	if(!this->getValue(i)) return false; else 
	{ 
		if(i==0 || i==1) 
		{
			v = (i==1);
			return true; 
		}
		else return false;
	}
}


bool iValue_IS::getValue(int &v)
{
	return this->getValue((void *)&v);
}


bool iValue_IS::getValue(float &v)
{
	int i;
	if(!this->getValue(i)) return false; else { v = i; return true; }
}


//
//  FS value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, float v, float *p, float (*f)(void))
{
	return (iValue *)new iValue_FS(n,m,v,p,f);
}


bool iValue_FS::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { if(Pointer != NULL) { Method = m; return true; } else return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_FS::assignValue(Assignment r, iValue *v)
{
	float f;
	if(v->getValue(f)) return this->assignValue(r,f); else return false;
}


bool iValue_FS::assignValue(Assignment r, int i)
{
	ASSIGN_VALUE(Value,r,i); 
	return true; 
}


bool iValue_FS::assignValue(Assignment r, float f)
{
	ASSIGN_VALUE(Value,r,f); 
	return true; 
}


bool iValue_FS::getValue(void *p)
{
	if(p == NULL) return false;
	switch (Method)
	{
	case ByValue:    { *(float *)p = Value; return true; }
	case ByPointer:  { if(Pointer != NULL) { *(float *)p = *Pointer; return true; } else return false; }
	case ByFunction: { *(float *)p = Function(); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_FS::getValue(bool &v)
{
	int i;
	float f;
	if(!this->getValue(f)) return false; else 
	{ 
		i = round(f); 
		if(i==0 || i==1) 
		{
			v = (i==1);
			return true; 
		}
		else return false;
	}
}


bool iValue_FS::getValue(int &v)
{
	float f;
	if(!this->getValue(f)) return false; else { v = round(f); return true; }
}


bool iValue_FS::getValue(float &v)
{
	return this->getValue((void *)&v);
}


bool iValue_FS::getValue(double &v)
{
	float v1;
	if(this->getValue((void *)&v1))
	{
		v = v1;
		return true;
	}
	else return false;
}


//
//  FP value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, float* v, float* *p, float* (*f)(void))
{
	return (iValue *)new iValue_FP(n,m,v,p,f);
}


bool iValue_FP::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { if(Pointer != NULL) { Method = m; return true; } else return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_FP::assignValue(Assignment r, iValue *v)
{
	if(v->getType()==IVALUE_FP && r==Absolute) 
	{ 
		this->Value = ((iValue_FP *)v)->Value; 
		return true; 
	} 
	else return false;
}


bool iValue_FP::getValue(void *p)
{
	if(p == NULL) return false;
	switch (Method)
	{
	case ByValue:    { *(float* *)p = Value; return true; }
	case ByPointer:  { if(Pointer != NULL) { *(float* *)p = *Pointer; return true; } else return false; }
	case ByFunction: { *(float* *)p = Function(); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_FP::getValue(float* &v)
{
	return this->getValue((void *)&v);
}


//
//  ST value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, iString v, iString *p, iString (*f)(void))
{
	return (iValue *)new iValue_ST(n,m,v,p,f);
}


bool iValue_ST::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { if(Pointer != NULL) { Method = m; return true; } else return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_ST::assignValue(Assignment r, iValue *v)
{
	if(v->getType()==IVALUE_ST && r==Absolute) 
	{ 
		this->Value = ((iValue_ST *)v)->Value; 
		return true; 
	} 
	else return false;
}


bool iValue_ST::getValue(void *p)
{
	if(p == NULL) return false;
	switch (Method)
	{
	case ByValue:    { *(iString *)p = Value; return true; }
	case ByPointer:  { if(Pointer != NULL) { *(iString *)p = *Pointer; return true; } else return false; }
	case ByFunction: { *(iString *)p = Function(); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_ST::getValue(iString &v)
{
	return this->getValue((void *)&v);
}

//
//  VA value
//
iValue* iValue::New(iString n, iScriptType::ValueMethod m, int s, iValue* i, iValue **a, iValue* (*f)(int))
{
	return (iValue *)new iValue_VA(n,m,s,i,a,f);
}


bool iValue_VA::setMethod(ValueMethod m)
{
	switch (m)
	{
	case ByValue:    { Method = m; return true; }
	case ByPointer:  { return false; }
	case ByFunction: { if(Function != NULL) { Method = m; return true; } else return false; }
	default: { reportUnallowedPath(); return false; }
	}
}


void iValue_VA::setIndex(iValue *i)
{
	if(Index != NULL) Index->Delete();
	Index = i;
	Index->Register();
}


iValue* iValue_VA::getComponent(iValue *v)
{
	int i;
	if(v->getType()!=IVALUE_IS || !v->getValue(i)) return NULL; else return this->getComponent(i);
}


iValue* iValue_VA::getComponent(int i)
{
	if(Size>0 && (i<1 || i>Size)) return NULL;

	switch (Method)
	{
	case ByValue:    { if(Array != NULL) return Array[i-1]; else return NULL; }
	case ByPointer:  { return NULL; }
	case ByFunction: { return Function(i-1); }
	default: { reportUnallowedPath(); return NULL; }
	}
}


bool iValue_VA::assignValue(Assignment r, iValue *v)
{
	int j = 0;
	switch(v->getType())
	{
	case IVALUE_VA:
		{
			if(r==Absolute && this->Array!=NULL && ((iValue_VA *)v)->Array!=NULL && this->Size==((iValue_VA *)v)->Size)
			{
				for(j=0; j<this->Size; j++)
				{
					if(this->Array[j]!=NULL && ((iValue_VA *)v)->Array[j]!=NULL) this->Array[j]->assignValue(r,((iValue_VA *)v)->Array[j]); 
				}
				return true;
			}
			else return false;
		}
	case IVALUE_BL:
	case IVALUE_IS:
	case IVALUE_FS:
	case IVALUE_FP:
		{
			iValue *val;
			if((val=this->getComponent(Index)) == NULL) return false;
			return val->assignValue(r,v);
		}
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_VA::getValue(void *p)
{
	int i;
	//
	//  Get array index
	//
	if(p==NULL || Index==NULL || Index->getType()!=IVALUE_IS || !Index->getValue((void *)&i)) return false;
	if(Size>0 && (i<=0 || i>Size)) return false;  //check the Size if it is positive

	switch (Method)
	{
	case ByValue:    { if(Array!=NULL && Array[i-1]!=NULL) { *(iValue **)p = Array[i-1]; return true; } else return false; }
	case ByPointer:  { return false; }
	case ByFunction: { *(iValue **)p = Function(i-1); return true; }
	default: { reportUnallowedPath(); return false; }
	}
}


bool iValue_VA::getValue(bool &v)
{
	iValue *p = this->getComponent(Index);
	if(p != NULL) return p->getValue(v); else return false;
}


bool iValue_VA::getValue(int &v)
{
	iValue *p = this->getComponent(Index);
	if(p != NULL) return p->getValue(v); else return false;
}


bool iValue_VA::getValue(float &v)
{
	iValue *p = this->getComponent(Index);
	if(p != NULL) return p->getValue(v); else return false;
}


bool iValue_VA::getValue(int n, int *p)
{
	int i;
	bool ok;
	int *w;
	iValue *v = NULL;
	//
	//  Get array index
	//
	if(p==NULL || Index==NULL || Index->getType()!=IVALUE_IS) return false;
	if(Index->getValue(i))
	{
		if(Size>0 && (i<=0 || i>Size || n>Size)) return false;  //check the Size if it is positive
	}
	//
	//  Check what we got: an array of values, or a pointer array value
	//
	if((v=this->getComponent(Index)) == NULL) return false;

	if(iValue::areTypesCompatible(IVALUE_IS,v))
	{
		w = new int[n];
		ok = true;
		for(i=0; i<n && ok; i++) 
		{
			v = this->getComponent(i+1);
			if(v != NULL)
			{
				ok = ok & v->getValue(w[i]);
			}
			else ok = false;
		}
		if(ok) memcpy(p,w,n*sizeof(int));
		delete [] w;
	}
	else ok = false;

	return ok;

}


bool iValue_VA::getValue(int n, float *p)
{
	int i;
	bool ok;
	float *w;
	iValue *v = NULL;
	//
	//  Get array index
	//
	if(p==NULL || Index==NULL || Index->getType()!=IVALUE_IS) return false;
	if(Index->getValue(i))
	{
		if(Size>0 && (i<=0 || i>Size || n>Size)) return false;  //check the Size if it is positive
	}
	//
	//  Check what we got: an array of values, or a pointer array value
	//
	if((v=this->getComponent(Index)) == NULL) return false;

	if(iValue::areTypesCompatible(IVALUE_FS,v))
	{
		w = new float[n];
		ok = true;
		for(i=0; i<n && ok; i++) 
		{
			v = this->getComponent(i+1);
			if(v != NULL)
			{
				ok = ok & v->getValue(w[i]);
			}
			else ok = false;
		}
		if(ok) memcpy(p,w,n*sizeof(float));
		delete [] w;
	}
	else
	{
		if(v->getType() == IVALUE_FP)  // special case
		{
			ok = v->getValue((void *)&w);
			if(ok) memcpy(p,w,n*sizeof(float));  
		}
		else ok = false;
	}

	return ok;

}


void iValue_VA::Delete()
{
	int i;
	if(ReferenceCount <= 1)
	{
		if(Array != NULL)
		{
			for(i=0; i<Size; i++) if(Array[i] != NULL) Array[i]->Delete();
		}
		delete [] Array;
		if(Index != NULL) Index->Delete();
	}
	iValue::Delete();
}

//
//***************************************************************
//
//  iOperation functions
//
//***************************************************************
//

iOperation* iOperation::New(iOperation *o0)
{
	switch(o0->Type)
	{
	case IOPERATION_DC_SV: 
		{
			iOperation_DC_SV *o = (iOperation_DC_SV*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->StatementPrefix,o->Relative,iValue::New(o->Value),o->UserType);
		}
	case IOPERATION_FC_LB: 
		{
			iOperation_FC_LB *o = (iOperation_FC_LB*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Point,iValue::New(o->Index),iValue::New(o->Count),o->Class,o->UserType);
		}
	case IOPERATION_SF_V0: 
		{
			iOperation_SF_V0 *o = (iOperation_SF_V0*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Function,o->UserType);
		}
	case IOPERATION_SF_V1: 
		{
			iOperation_SF_V1 *o = (iOperation_SF_V1*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,o->Relative,iValue::New(o->Value),o->Function,o->UserType);
		}
	case IOPERATION_AF_V1: 
		{
			iOperation_AF_V1 *o = (iOperation_AF_V1*)o0;
			return iOperation::New(o->Script,o->Prefix,o->Command,o->OutputCode,iValue::New(o->Index),o->Relative,iValue::New(o->Value),o->Function,o->UserType);
		}
	default: { reportUnallowedPath(); return NULL; }
	}
}


iOperation::iOperation(iScript *w, iString p, iString s, int o, OperationType t, unsigned char u)
{ 
	static iString tmp = "";
	
	Script = w; 
	Prefix = p; 
	Command = s; 
	OutputCode = o; 
	Type = t; 
	UserType = u; 
	LineInScript = -1;
	if(Script != NULL) pErrorMessage = &(Script->errorMessage); else { pErrorMessage = &tmp; }
}


bool iOperation::assignVariableValue(iString n, Assignment r, iValue *v, iValue *i)
{
	if(Script != NULL) return Script->assignVariableValue(n,r,v,i); else return false;
}


void iOperation::setErrorMessage(iString s)
{ 
	if(Script != NULL) 
	{
		Script->setErrorMessage(s); 
	}
}


//
// Type DC_SV operation creator & executor
//
iOperation* iOperation::New(iScript *w, iString p, iString c, int o, iString sp, Assignment r, iValue *v, unsigned char u)
{
	return (iOperation *)new iOperation_DC_SV(w,p,c,o,sp,r,v,u);
}


bool iOperation_DC_SV::exec()
{
	return true;
}


//
// Type FC_LB operation creator & executor
//
iOperation* iOperation::New(iScript *w, iString p, iString c, int o, EntryPoint q, iValue* i, iValue *n, int l, unsigned char u)
{
	return (iOperation *)new iOperation_FC_LB(w,p,c,o,q,i,n,l,u);
}


bool iOperation_FC_LB::exec()
{
	bool ok = false;
	
	intCount = 0;
	
	if(Point & Exit)
	{
		Gate = Open;
		Done = true;
	}

	if(Point & Entry)
	{
		if(Count != NULL)
		{
			switch(Count->getType())
			{
			case IVALUE_BL:
				{
					//
					//  This is a branch
					//
					if(!Count->getValue(ok)) this->setErrorMessage("Invalid value type in statement: "+this->Prefix+" "+this->Command);
					if(ok) Gate = Open; else Gate = Close;
					Done = true;
					break;
				}
			case IVALUE_IS:
				{
					//
					//  This is a loop
					//
					if(Done) 
					{
						intIndex = 0;
						Done = false;
					}
					
					ok = Count->getValue(intCount);
					if(!ok) this->setErrorMessage("Invalid interation count in statement: "+this->Prefix+" "+this->Command);
					
					intIndex++;
					
					if(Index!=NULL && !Index->getName().isEmpty()) 
					{
						if(Index->assignValue(Absolute,intIndex)) { if(Script != NULL) Script->setVariableChanged(Index); } else this->setErrorMessage("Invalid interation index type in statement: "+this->Prefix+" "+this->Command);
					}
					
					if(intIndex >= intCount) Done = true;
					if(intIndex > intCount) 
					{
						//
						// Never execute this loop
						//
						Gate = Close;
					}
					else
					{
						//
						//
						// Execute this loop
						Gate = Open;
					}
					break;
				}
			default: this->setErrorMessage("Invalid value type in statement: "+this->Prefix+" "+this->Command);
			}
		}
		else
		{
			Gate = Flip;
			Done = true;
		}
	}

	return true;
}


void iOperation_FC_LB::setValue(iValue* v)
{ 
	if(iValue::areTypesCompatible(Index,v)) 
	{ 
		if(Index != NULL) Index->Delete();
		Index = v; 
		Index->Register(); 
	}
}


//
// Type SF_V0 operation creator & executor
//
iOperation* iOperation::New(iScript *w, iString p, iString c, int o, void (*f)(iString*), unsigned char u)
{
	return (iOperation *)new iOperation_SF_V0(w,p,c,o,f,u);
}


bool iOperation_SF_V0::exec()
{
	Function(pErrorMessage);
	return true;
}


//
// Type SF_V1 operation creator & executor
//
iOperation* iOperation::New(iScript *w, iString p, iString c, int o, Assignment r, iValue *v, void (*f)(iValue*,Assignment,iString*), unsigned char u)
{
	return (iOperation *)new iOperation_SF_V1(w,p,c,o,r,v,f,u);
}


bool iOperation_SF_V1::exec()
{
	if(Function != NULL) Function(Value,Relative,pErrorMessage); else if(!this->assignVariableValue(this->Command,Relative,Value)) this->setErrorMessage("Incompatible value type in statement: "+this->Prefix+" "+this->Command);
	return true;
}


void iOperation_SF_V1::setValue(iValue* v)
{ 
	if(iValue::areTypesCompatible(Value,v)) 
	{ 
		if(Value != NULL) Value->Delete();
		Value = v; 
		Value->Register(); 
	}
}


//
// Type AF_V1 operation creator & executor
//
iOperation* iOperation::New(iScript *w, iString p, iString c, int o, iValue *i, Assignment r, iValue *v, void (*f)(iValue*,iValue*,Assignment,iString*), unsigned char u)
{
	return (iOperation *)new iOperation_AF_V1(w,p,c,o,i,r,v,f,u);
}


bool iOperation_AF_V1::exec()
{
	if(Function != NULL) Function(Index,Value,Relative,pErrorMessage); else if(!this->assignVariableValue(this->Command,Relative,Value,Index)) this->setErrorMessage("Incompatible type in statement: "+this->Prefix+" "+this->Command);
	return true;
}


void iOperation_AF_V1::setValue(iValue* v)
{ 
	if(iValue::areTypesCompatible(Value,v)) 
	{ 
		if(Value != NULL) Value->Delete();
		Value = v; 
		Value->Register(); 
	}
}


void iOperation::Delete()
{
	delete this;
}


void iOperation_DC_SV::Delete()
{
	if(Value != NULL) Value->Delete();
	iOperation::Delete();
}


void iOperation_FC_LB::Delete()
{
	if(Index != NULL) Index->Delete();
	if(Count != NULL) Count->Delete();
	iOperation::Delete();
}


void iOperation_SF_V1::Delete()
{
	if(Value != NULL) Value->Delete();
	iOperation::Delete();
}


void iOperation_AF_V1::Delete()
{
	if(Index != NULL) Index->Delete();
	if(Value != NULL) Value->Delete();
	iOperation::Delete();
}


//
//***************************************************************
//
//  iScript functions
//
//***************************************************************
//

iScript::iScript()
{

	active = true;

	caseSensitive = false;
	arraysAreFloat = false;
	errorMessage = "";

	nDummyWords = nPrefixWords = nCommandWords = nParameterWords = nCommand = nVariable = 0;
	nAliasWords = 0;
	nCode = lCode = nBuffer = 0;
	pCode = 0;

	compilerStackSize = 100;
	compilerStackEntryPoint = -1;
	compilerStack = new StackEntry[compilerStackSize];

	lastChangedVariable = -2;  // -2 means no update

	text = "";
	line = numLines = 0;
	compileReturnCode = executeReturnCode = -1;
	entryPointIteration = -1;
	entryPointIterationCount = 0;

	checkAbort = NULL;
	showLine = NULL;

	parser = new iExpressionParser;
	if(parser == NULL) reportNullPointer(7303);

	//
	//  Standard parameters
	//
	this->createParameterWord(iValue::New("false",ByValue,false));
	this->createParameterWord(iValue::New("true",ByValue,true));

	//
	//  Special characters - can be overwritten in a child class
	//
	commentChar = "#";
	leChar = "{";
	geChar = "}";
	eqChar = "?";
	neChar = "~";
	andChar = "&";
	orChar = "|";

}


iScript::~iScript()
{
	int i;

	this->reset();
	delete [] compilerStack;

	for(i=0; i<nParameterWords; i++) pParameterWords[i]->Delete();
	for(i=0; i<nCommandWords; i++) pCommandWords[i]->Delete();

	delete parser;

#ifdef _DEBUG
	if(iValue::counterCreate != iValue::counterDestroy)
	{
		int i, j;
		for(i=0; i<999; i++) if (iValue::cval[i] != 0)
		{
			j = 0;
		}
	}
#endif
}


//
//  Script Creation
//
void iScript::createAliasWord(const char *alias, const char *word)
{
	if(nAliasWords < maxSize) 
	{
		pAliasWords[0][nAliasWords] = iString(alias);
		pAliasWords[1][nAliasWords++] = iString(word);
	}
}


void iScript::createDummyWord(const char *v)
{
	if(nDummyWords < maxSize) pDummyWords[nDummyWords++] = iString(v);
}


void iScript::createPrefixWord(const char *v)
{
	if(nPrefixWords < maxSize) pPrefixWords[nPrefixWords++] = iString(v);
}


void iScript::createCommandWord(iOperation*v)
{
	if(nCommandWords < maxSize) pCommandWords[nCommandWords++] = v;
}


void iScript::createParameterWord(iValue *v)
{
	if(nParameterWords < maxSize) pParameterWords[nParameterWords++] = v;
}


//
//  Public interface
//
void iScript::setText(iString s)
{
	if(!caseSensitive) text = s.lowerScriptText(); else text = s;
	if(text[(int)text.length()-1] != '\n') text += '\n';
	numLines = text.contains('\n');
	this->reset();

}
	
	
int iScript::run(iString s, bool compileOnly, int firstLine, int lastLine)
{
	this->setText(s);
	return this->run(compileOnly,firstLine,lastLine);
}


bool iScript::isCommandWord(const iString s)
{
	bool ok = false;
	for(int i=0; i<nCommandWords; i++) if(pCommandWords[i]->Command == s) ok = true;
	return ok;
}


bool iScript::isParameterWord(const iString s)
{
	bool ok = false;
	for(int i=0; i<nParameterWords; i++) if(pParameterWords[i]->Name == s) ok = true;
	return ok;
}


bool iScript::isDummyWord(const iString s)
{
	bool ok = false;
	for(int i=0; i<nDummyWords; i++) if(pDummyWords[i] == s) ok = true;
	return ok;
}


bool iScript::isPrefixWord(const iString s)
{
	bool ok = false;
	for(int i=0; i<nPrefixWords; i++) if(pPrefixWords[i] == s) ok = true;
	return ok;
}


bool iScript::isReservedWord(const iString s)
{
	return this->isCommandWord(s) || this->isParameterWord(s) || this->isPrefixWord(s) || this->isDummyWord(s);
}


iString iScript::getReservedWord(int i)
{
	if(i>=0 && i<nPrefixWords) return pPrefixWords[i];
	if(i>=nPrefixWords && i<nPrefixWords+nCommandWords) return pCommandWords[i-nPrefixWords]->Command;
	if(i>=nPrefixWords+nCommandWords && i<nPrefixWords+nCommandWords+nParameterWords) return pParameterWords[i-nPrefixWords-nCommandWords]->Name;
	if(i>=nPrefixWords+nCommandWords+nParameterWords && i<nPrefixWords+nCommandWords+nParameterWords+nDummyWords) return pDummyWords[i-nPrefixWords-nCommandWords-nParameterWords];
	return iString();
}


int iScript::getUserTypeForReservedWord(int i)
{
	if(i>=nPrefixWords && i<nPrefixWords+nCommandWords) return pCommandWords[i-nPrefixWords]->UserType; else return 0;
}

//
//  getting information about variables
//
bool iScript::isVariableWord(const iString s)
{
	int i;
	bool ok = false;
	for(i=0; i<nVariable; i++) if(pVariable[i]->Name == s) ok = true;
	return ok;
}


//
//  Assign error message
//
void iScript::setErrorMessage(iString s)
{ 
	errorMessage = s;
}


//
//  get the line of code
//
iString iScript::getText(int i)
{
	if(i>=0 && i<numLines) return text.section('\n',i,i); else return "";
}

//
//  Remove the pseudocode;
//
void iScript::reset()
{
	int i;
	//
	//  Delete the code
	//
	if(nCode > 0)
	{
		for(i=0; i<nCode; i++) if(pCode[i] != NULL) pCode[i]->Delete();
	}
	if(pCode != 0) delete [] pCode;
	pCode = 0;
	nCode = lCode = nBuffer = 0;
	//
	//  Delete the stack
	//
	compilerStackEntryPoint = -1;
	//
	//  Delete the variables and their respective command too
	//
//	for(i=0; i<nValue; i++) pValue[i]->Delete();
//	nValue = nVariable = 0;
	for(i=0; i<nVariable; i++) pVariable[i]->Delete();
	nVariable = 0;

	for(i=nCommandWords; i<nCommand; i++) pCommandWords[i]->Delete();
	nCommand = nCommandWords;

	line = 0;
	this->skipEmptyLines();

	active = true;
	lActivator = 0;
	block = false;

}


void iScript::skipEmptyLines()
{
	iString s;
	//
	//  Skip lines if they are empty
	//
	while(line<numLines)
	{
		s = this->getText(line);
		s = s.simplifyWhiteSpace();
		if(s.isEmpty() || (s.length()==1 && s[0].isSpace()) || s[0]=='#') line++; else break;
	}
}


bool iScript::assignVariableValue(iString n, iScriptType::Assignment r, iValue *v, iValue *i)
{
	int j;	
	for(j=0; j<nVariable; j++) if(n == pVariable[j]->Name)
	{
		//
		//  Is it an array?
		//
		if(i!=NULL && pVariable[j]->Type==IVALUE_VA) 
		{
			((iValue_VA*)pVariable[j])->setIndex(i);
		}
		if(pVariable[j]->assignValue(r,v)) { lastChangedVariable = j; return true; } 
		break;
	}
	return false;
}


void iScript::setVariableChanged(iValue *v)
{
	int j;	
	for(j=0; j<nVariable; j++) if(v->Name == pVariable[j]->Name)
	{
		lastChangedVariable = j; 
		break;
	}
}


int iScript::run(bool compileOnly, int NotUsed(firstLine), int NotUsed(lastLine))
{
	int ret = 0;

	this->reset();
	while(line<numLines && (ret=this->runOneLine(compileOnly))==0);	

	return ret;

}


//
// Compile and execute one line of script
// Return codes:
//  1  compilation failed
//  2  execution failed
//  3  script was terminated
//  -1 script ended
//  0  success
//
int iScript::runOneLine(bool compileOnly)
{
	int thisLine;
	bool thisActive;

	if(block) return 0; //  do not multi-thread me!!!
	block = true;  

	errorMessage = "";

	do
	{
		
		thisLine = line;
		thisActive = active;
		if(showLine!=NULL && thisActive) showLine(thisLine);
		
		parser->SetIgnoreDivisionByZero(compileOnly);
		compileReturnCode = this->compileOneLine();  		
		if(compileReturnCode == -2) { block = false; return 0; } // empty line and comments
		if(compileReturnCode != -1) { block = false; return 1; }
		this->skipEmptyLines();

		if(compileOnly) 
		{
			lCode++; 
			block = false;
			if(line == numLines) return -1; else return 0;
		}
		
		lastChangedVariable = -2;  // -2 means no update
		
		executeReturnCode = executeOneLine();
		if(executeReturnCode == -2) { block = false; return 2; }
		if(executeReturnCode == -1) { block = false; return -1; }
		if(!errorMessage.isEmpty()) { block = false; return 2; }
		
		if(checkAbort!=NULL && (active || thisActive)) if(checkAbort(thisLine,entryPointIteration,entryPointIterationCount,0)) { block = false; return 3; }
		
		if(line == numLines) { block = false; return -1; } //  script ended
		
	}
	while(!active);

	block = false;
	return 0;
	
}


//
//  execute one command in the script
//  return: OutputCode of the command if Ok
//          -1 if the script ended
//			-2 fatal error
//
int iScript::executeOneLine()
{
	int l;
	bool checkGate = false;

	errorMessage = "";
	executeReturnCode = 0;

	if(nCode==0 || pCode==0) 
	{
		errorMessage = "The script is not compiled.";
		return -2;
	}

	if(lCode == nCode)
	{
		return -1;
	}

	entryPointIteration = -1;
	entryPointIterationCount = 0;

	if(pCode[lCode] == NULL) // empty statement
	{
		lCode++;
		return 0;
	}

 	int ret = pCode[lCode]->OutputCode;
	//
	//  Because the branch can activate or dis-activate the script, it must be executed first
	//
	if(pCode[lCode]->Type == IOPERATION_FC_LB)
	{
		iOperation_FC_LB *op = (iOperation_FC_LB *)pCode[lCode];
		op->exec();

		//
		//  Update the compiler stack if needed. 
		//
		if(compilerStackEntryPoint >= compilerStackSize-1)
		{
			int i, newSize = compilerStackSize + 100;
			StackEntry *newStack = new StackEntry[newSize];
			for(i=0; i<compilerStackSize; i++) newStack[i] = compilerStack[i];
			delete [] compilerStack;
			compilerStack = newStack;
			compilerStackSize = newSize;
		}
		
		if(op->Point & Exit)
		{
			//
			// Give an error if incorrect nesting detected.
			//
			if(compilerStackEntryPoint<0 || compilerStackEntryPoint>=compilerStackSize || compilerStack[compilerStackEntryPoint].Class != op->Class || compilerStack[compilerStackEntryPoint].Type != op->Type)
			{
				errorMessage = "Incorrectly nested operator " + iString(op->Command) + ".";
				return -2;
			}
			//
			//  Operation type FC_LB(Point=Exit) passes control to the last operation type FC_LB(Point=Entry) 
			//  of the same class if that operation is not done. Use the stack to find a match.
			//
			l = compilerStack[compilerStackEntryPoint].Line;
			if(pCode[l] == NULL) return -2;
			compilerStackEntryPoint--;
			
			if(active || l==lActivator) // I am a match for the activator, so execute
			{
				checkGate = true;
				if(!((iOperation_FC_LB *)pCode[l])->Done) 
				{ 
					lCode = l; 
					line = pCode[lCode]->LineInScript;
					lCode--; // it will be ++ed at the end of this operator
				} 
				else 
				{
					entryPointIteration = -1;
					entryPointIterationCount = ((iOperation_FC_LB *)pCode[l])->intCount;
					if(checkAbort != NULL) if(checkAbort(pCode[l]->LineInScript,entryPointIteration,entryPointIterationCount,0)) return 3;
				}
			}
		}

		if(op->Point & Entry)
		{
			compilerStackEntryPoint++;
			compilerStack[compilerStackEntryPoint].Type = op->Type;
			compilerStack[compilerStackEntryPoint].Class = op->Class;
			compilerStack[compilerStackEntryPoint].Line = lCode;
			entryPointIterationCount = op->intCount;
			entryPointIteration = op->intIndex;
			if(active) checkGate = true;
		}
		
		if(checkGate)
		{
			switch(op->Gate)
			{
			case Open: { active  = true; break; }
			case Close: { active  = false; lActivator = lCode; break; }
			case Flip: { active  = !active; if(!active) lActivator = lCode; break; }
			case Skip: { break; }
			default: reportUnallowedPath();
			}
		}
		lCode++;
		return ret;
	}

	if(!active) 
	{
		lCode++;
		return ret;
	}

	switch(pCode[lCode]->Type)
	{
	case IOPERATION_DC_SV:
		{
			lastChangedVariable = -1;  // -1 means update all - new are declared
		}
	case IOPERATION_SF_V0:
	case IOPERATION_SF_V1:
	case IOPERATION_AF_V1:
		{
			if(pCode[lCode]->exec()) lCode++;
			break;
		}
	default: 
		{
			reportUnallowedPath();
			errorMessage = "Invalid operation.";
			ret = -2;
		}
	}
	
	return ret;

}


//
//  Compile the line of the script and execute it.
//  If there is an error, return the starting position of the offending text, otherwiese return -1 
//  or -2 if the line is empty 
//
int iScript::compileOneLine()
{
	iString commandText, commandTextOriginal;
	iString pw, cw, ac, pv;
	int i, n, ioff, comind;
	Assignment valueRelative = Absolute;
	bool ok, isArray = false;
	iValue *val = NULL, *arrComponent = NULL;

	errorMessage = "";
	compileReturnCode = -1;

	commandText = commandTextOriginal = this->getText(line++);

	//
	//  Remove comments if they are present
	//
	i = commandText.find(commentChar);
	if(i >= 0) commandText = commandText.left(i);

	//
	//  Check if special characters are present - they should not be.
	//
	i = commandText.find('$');
	if(i >= 0)
	{
		errorMessage = "A symbol is not allowed in the script text.";
		return i;
	}

	i = commandText.find(leChar);
	if(i >= 0)
	{
		errorMessage = "A symbol is not allowed in the script text.";
		return i;
	}

	i = commandText.find(geChar);
	if(i >= 0)
	{
		errorMessage = "A symbol is not allowed in the script text.";
		return i;
	}

	i = commandText.find(eqChar);
	if(i >= 0)
	{
		errorMessage = "A symbol is not allowed in the script text.";
		return i;
	}

	i = commandText.find(neChar);
	if(i >= 0)
	{
		errorMessage = "A symbol is not allowed in the script text.";
		return i;
	}

	//
	//  Replace comparison signs with single char equivalents
	//
	commandText.replace(iString("<="),leChar);
	commandText.replace(iString(">="),geChar);
	commandText.replace(iString("=="),eqChar);
	commandText.replace(iString("!="),neChar);
	commandText.replace(iString("&&"),andChar);
	commandText.replace(iString("||"),orChar);

	//
	//  Remove dummy words - but not when they are parts of other words.
	//
	for(i=0; i<nDummyWords; i++) 
	{
		while((n=commandText.findScriptWord(pDummyWords[i]))>-1) commandText.replace(n,pDummyWords[i].length()," ");
	}

	//
	//  Replace alias words - but not when they are parts of other words.
	//
	for(i=0; i<nAliasWords; i++) 
	{
		while((n=commandText.findScriptWord(pAliasWords[0][i]))>-1) commandText.replace(n,pAliasWords[0][i].length(),pAliasWords[1][i]);
	}

	//
	//  Remove '=' signs - they are just for aestetics. Remove extra white spaces if appeared.
	//
	commandText.replace(iString("="),iString(" "));
	commandText = commandText.simplifyWhiteSpace();

	//
	//  No command in this line
	//
	if(commandText.isEmpty() || (commandText.length()==1 && commandText[0].isSpace()) || commandText[0]=='#')
	{
		return -2;
	}

	//
	//  Read the first word - it must be a command or a prefix word
	//
	ioff = 0;
	cw = commandText.section(' ',ioff,ioff);

	//
	//  Find whether it is a prefix word - if yes, skip it
	//
	for(i=0; i<nPrefixWords; i++)
	{
		if(iString(pPrefixWords[i]) == cw) break;
	}
	if(i < nPrefixWords)
	{
		pw = cw;
		ioff++;
	}
	else pw = "";

	//
	//  Find whether it is a command word: read the word first
	//
	cw = commandText.section(' ',ioff,ioff);

	//
	//  Look for the array component at the end in the form [N] where N is a iValue;
	//
	i = cw.find('[');
	if(i >= 0)    // a square bracket found
	{
		ac = cw.mid(i);
		cw = cw.left(i);
		if(ac.length()<3 || ac.left(1)!="[" || ac.right(1)!="]")
		{
			errorMessage = "Syntax error in array index.";
			i += commandTextOriginal.find(cw,0);
			if(i >= 0) return i; else return 0;
		}
		ac = ac.mid(1,ac.length()-2);
		//
		//  Postpone turning ac into the iValue.
		//
	}

	//
	//  Search the list of command words - enforce the prefix match too.
	//
	for(i=0; i<nCommand; i++)
	{
		if(iString(pCommandWords[i]->Command)==cw && iString(pCommandWords[i]->Prefix)==pw) break;
	}
	if(i == nCommand) //  The word is not a valid command
	{
		errorMessage = "Syntax error - invalid command or undefined variable.";
		i = commandTextOriginal.find(cw,0);  
		if(i >= 0) return i; else return 0;
	}
	comind = i;
	ioff++;

	//
	//  Read the value.
	//
	pv = commandText.section(' ',ioff,ioff);

	//
	//  If the 'value' turned out to be + or - sign, the command was in the form of 
	//	"set param += 3" - thus the value should be treated as relative. For '-' change the
	//  value sign as well. 
	//
	if(pv==iString("+") || pv==iString("-"))
	{
		//
		//  First check if this is allowed
		//
		switch(pCommandWords[comind]->Type)
		{
		case IOPERATION_DC_SV: 
		case IOPERATION_FC_LB: 
		case IOPERATION_SF_V0: 
			{ 
				ok = false; break; 
			}
		case IOPERATION_SF_V1: 
			{ 
				ok = ((iOperation_SF_V1 *)pCommandWords[comind])->Relative & 1; break; 
			}
		case IOPERATION_AF_V1: 
			{ 
				ok = ((iOperation_AF_V1 *)pCommandWords[comind])->Relative & 1; break; 
			}
		default:
			{
				ok = false;	break;
			}
		}
		if(!ok)
		{
			errorMessage = "Incrementing operation not allowed.";
			i = commandTextOriginal.find(pv,0);
			if(i >= 0) return i; else return 0;
		}
		//
		//  It is ok, proceed
		//
		valueRelative = RelativeAdd;
		if(pv==iString("-")) valueRelative = RelativeInvertAdd;
		ioff++;
		pv = commandText.section(' ',ioff); // read the rest of the line
	}
	//
	//  If the 'value' turned out to be * sign, the command was in the form of 
	//	"set param *= 3" - thus the value should be treated as relative.
	//
	else if(pv == iString("*"))
	{
		//
		//  First check if this is allowed
		//
		switch(pCommandWords[comind]->Type)
		{
		case IOPERATION_DC_SV: 
		case IOPERATION_FC_LB: 
		case IOPERATION_SF_V0: 
			{ 
				ok = false; break; 
			}
		case IOPERATION_SF_V1: 
			{ 
				ok = ((iOperation_SF_V1 *)pCommandWords[comind])->Relative & 2; break; 
			}
		case IOPERATION_AF_V1: 
			{ 
				ok = ((iOperation_AF_V1 *)pCommandWords[comind])->Relative & 2; break; 
			}
		default:
			{
				ok = false;	break;
			}
		}
		if(!ok)
		{
			errorMessage = "Incrementing operation not allowed.";
			i = commandTextOriginal.find(pv,0);
			if(i >= 0) return i; else return 0;
		}
		//
		//  It is ok, proceed
		//
		valueRelative = RelativeMult;
		ioff++;
		pv = commandText.section(' ',ioff); // read the rest of the line
	}
	else pv = commandText.section(' ',ioff); // read the rest of the line

	//
	//  Create the arrComponent iValue
	//
	if(!ac.isEmpty())
	{
		arrComponent = this->transformToValue(ac);
		isArray = (arrComponent != NULL);

		if(!isArray)
		{
			errorMessage = "Invalid array index.";
			i = commandTextOriginal.find(cw,0) + cw.find('[');
			if(i >= 0) return i; else return 0;
		}
	}

	//
	//  Check that the array index corresponds to the array
	//
	switch(pCommandWords[comind]->Type)
	{
	case IOPERATION_FC_LB: 
	case IOPERATION_SF_V0: 
	case IOPERATION_SF_V1: 
		{ 
			ok = !isArray; 
			if(!ok) errorMessage = "Variable is not an array.";
			break; 
		}
	case IOPERATION_AF_V1: 
		{ 
			//
			//  Check that the whole array assignment is allowed
			//
			ok = isArray || ((iOperation_AF_V1*)pCommandWords[comind])->Value->Type == IVALUE_VA; 
			if(!ok) errorMessage = "Whole array assignment is not allowed."; 
			break; 
		}
	case IOPERATION_DC_SV: 
		{
			ok = true;	break;
		}
	default:
		{
			ok = false;	break;
		}
	}

	if(!ok)
	{
		if(arrComponent != NULL) arrComponent->Delete();
		i = commandTextOriginal.find(cw,0);
		if(i >= 0) return i; else return 0;
	}

	//
	//  Get the value
	//
	switch(pCommandWords[comind]->Type)
	{
	case IOPERATION_DC_SV:  // a declaration is executed at the compile time
		{ 

			//
			//  Remove commas if they are present - they are just for aestetics
			//
			commandText.replace(iString(","),iString(" "));
			commandText = commandText.simplifyWhiteSpace();

			pv = commandText.section(' ',ioff,ioff++);
			if(pv.isEmpty())
			{
				errorMessage = "Variable name must be specified.";
				ok = false; break;
			}

			ok = true;
			while(!pv.isEmpty())
			{
				//
				// Check that the name is alpha-numeric
				//
				if(!pv[0].isLetter()) ok = false;
				for(i=1; i<(int)pv.length() && ok; i++) if(!pv[i].isLetterOrNumber()) ok = false;
				if(!ok)
				{
					errorMessage = "Variable name must be alpha-numeric.";
					break;
				}

				//
				// Check that the name is unique
				//
				for(i=0; i<nCommand && ok; i++) if(pv == pCommandWords[i]->Command) ok = false;
				if(!ok)
				{
					errorMessage = "Variable name is not unique.";
					break;
				}
				
				//
				//  Is the value configured properly?
				//
				if(((iOperation_DC_SV *)pCommandWords[comind])->Value->Method != ByValue)
				{
					errorMessage = "Script is incorrectly configured. Declared variables must have Method=ByValue.";
					ok = false; break;
				}
				
				//
				//  Do we have room for another variable?
				//
				if(nCommand == maxSize)
				{
					errorMessage = "Compiler capabilities are exhausted.";
					ok = false; break;
				}

				iValue *valtmp;
				//
				//   Is it array?
				//
				if(!isArray)
				{
					//
					//  Simple variable 
					//
					valtmp = iValue::New(((iOperation_DC_SV *)pCommandWords[comind])->Value);
				}
				else
				{
					if(!arrComponent->getValue(n) || n<1 )
					{
						errorMessage = "Incorrect value for the array dimension.";
						ok = false; break;
					}

					//
					//  Create an array of iValues
					//
					iValue **arr = new iValue*[n];
					if(arr == NULL)
					{
						errorMessage = "Not enough memory to create this array.";
						ok = false; break;
					}
					//
					//  Fill it in 
					//
					for(i=0; i<n && ok; i++)
					{
						arr[i] = iValue::New(((iOperation_DC_SV *)pCommandWords[comind])->Value);
						if(arr[i] == NULL) ok = false;
					}
					//
					//  If unsuccessfull, destroy it.
					//
					if(!ok)
					{
						for(i=0; i<n; i++) if(arr[i] != NULL) arr[i]->Delete();
						delete [] arr;  
						errorMessage = "Not enough memory to create this array.";
						break;
					}
					
					//
					//  Create an array-valued variable; iValue_VA takes ownership of everything
					//
					valtmp = iValue::New("",ByValue,n,iValue::New("",ByValue,1),arr);

				}

				if(valtmp != NULL) 
				{
					//
					//  Assign the name to the variable
					//
					valtmp->Name = pv;
					//
					//  Add it to the variable list 
					//
					if(nVariable < maxSize) pVariable[nVariable++] = valtmp;
					//
					//  Find all command words with Command == $variable and the same value type and turn then into new statements
					//
					for(i=0; i<nCommandWords; i++)
					{
						if(pCommandWords[i]->Command=="$variable" && pCommandWords[i]->getValue()!=NULL && pCommandWords[i]->getValue()->Type==valtmp->Type)
						{
							pCommandWords[nCommand] = iOperation::New(pCommandWords[i]);
							pCommandWords[nCommand]->Command = pv;
							pCommandWords[nCommand]->setValue(valtmp);
							nCommand++;
						}
					}
				}
				else ok = false;

				pv = commandText.section(' ',ioff,ioff++);
			}

			if(arrComponent != NULL)
			{
				arrComponent->Delete(); // not needed any more
				arrComponent = NULL;
			}

			break; 
		}
	case IOPERATION_SF_V0: 
		{ 
			ok = true; break; 
		}
	case IOPERATION_FC_LB: 
		{
			//
			//  If there is no count, there should be no value
			//
			if(((iOperation_FC_LB *)pCommandWords[comind])->Count == NULL)
			{
				ok = pv.isEmpty();
			}
			else
			{
				val = this->transformToValue(pv);
				ok = (val != NULL);
			}
			break;
		}
	case IOPERATION_SF_V1: 
	case IOPERATION_AF_V1: 
		{ 
			//
			//  Convert to the iValue type
			//
			val = this->transformToValue(pv);
			ok = (val != NULL);
			break; 
		}
	default:
		{
			errorMessage = "Invalid value.";
			ok = false;	break;
		}
	}

	if(!ok)
	{
		if(arrComponent != NULL) arrComponent->Delete();
		if(errorMessage.isEmpty()) errorMessage = "unknown error";
		i = commandTextOriginal.find(pv,0);
		if(i >= 0) return i; else return 0;
	}

	//
	//  Check that the value is of correct type
	//
	switch(pCommandWords[comind]->Type)
	{
	case IOPERATION_FC_LB: 
		{ 
			if(((iOperation_FC_LB*)pCommandWords[comind])->Count != NULL)
			{
				ok = iValue::areTypesCompatible(((iOperation_FC_LB*)pCommandWords[comind])->Count,val);
			}
			break; 
		}
	case IOPERATION_SF_V1: 
		{ 
			if(((iOperation_SF_V1*)pCommandWords[comind])->Value != NULL)
			{
				ok = iValue::areTypesCompatible(((iOperation_SF_V1*)pCommandWords[comind])->Value,val);
			}
			break; 
		}
	case IOPERATION_AF_V1: 
		{ 
			if(((iOperation_AF_V1*)pCommandWords[comind])->Value != NULL)
			{
				//
				//  Decide whether this is a full array assignment, or a single component.
				//  The possibilities:
				//  arr = arr			rhs = array
				//  arr[i] = arr[i]		rhs = scalar, but pCommandWords[comind] is declared with array iValue
				//  arr[i] = sca		rhs = scalar, but pCommandWords[comind] is declared with scalar iValue
				//
				if(arrComponent == NULL)
				{
					ok = iValue::areTypesCompatible(((iOperation_AF_V1*)pCommandWords[comind])->Value,val);
					arrComponent = iValue::New(((iOperation_AF_V1*)pCommandWords[comind])->Index);  
					arrComponent->assignValue(Absolute,1);  // cannot have zero component in the script command
				}
				else
				{
					if(((iOperation_AF_V1*)pCommandWords[comind])->Value->Type == IVALUE_VA)
					{
						//
						//  Full array assignment
						//
						iValue *valtmp = ((iValue_VA*)((iOperation_AF_V1*)pCommandWords[comind])->Value)->getComponent(arrComponent);
						if(valtmp != NULL) ok = iValue::areTypesCompatible(valtmp,val); else ok = false;
					}
					else 
					{
						//
						//  Single component assignment to an array that cannot be assigned as a whole
						//  (was declared with a single-valued iValue)
						//
						ok = iValue::areTypesCompatible(((iOperation_AF_V1*)pCommandWords[comind])->Value,val);
					}
				}
			}
			break; 
		}
	default:
		{
			ok = true; break;
		}
	}
	
	if(!ok)
	{
		if(arrComponent != NULL) arrComponent->Delete();
		if(val != NULL) val->Delete();
		errorMessage = "Invalid value type.";
		i = commandTextOriginal.find(pv,0);
		if(i >= 0) return i; else return 0;
	}

	//
	//  If the value is an array, check that it has the correct number of components
	//
	if(val!=NULL && val->Type==IVALUE_VA)
	{
		int n = ((iValue_VA *)val)->Size;
		int s;

		switch(pCommandWords[comind]->Type)
		{
		case IOPERATION_SF_V1: 
			{ 
				s = ((iValue_VA *) ((iOperation_SF_V1*)pCommandWords[comind])->Value)->Size;
				ok = (n==0 || s==0 || n==s);
				break; 
			}
		case IOPERATION_AF_V1: 
			{ 
				s = ((iValue_VA *) ((iOperation_AF_V1*)pCommandWords[comind])->Value)->Size;
				ok = (n==0 || s==0 || n==s);
				break; 
			}
		default:
			{
				ok = true; break;
			}
		}
		
		if(!ok)
		{
			if(arrComponent != NULL) arrComponent->Delete();
			if(val != NULL) val->Delete();
			iString s;
			errorMessage = "Array must have " + s.setNum(n) + " components.";
			i = commandTextOriginal.find(pv,0);
			if(i >= 0) return i; else return 0;
		}
	}

	//
	//  extend the buffer if there is no room
	//
	if(lCode==nCode && nCode==nBuffer)
	{
		nBuffer += 100;
		iOperation **buf = new iOperation*[nBuffer];
		if(buf == 0) reportNullPointer(7301);
		for(i=0; i<nCode; i++) buf[i] = pCode[i];
		delete [] pCode;
		pCode = buf;
		//
		//  fill with NULLs to know whether we were here or not
		//
		for(i=nCode; i<nBuffer; i++) pCode[i] = NULL;
	}
	//
	//  save the command in pseudo code if this is the first time we reached this line
	//
	if(pCode[lCode] == NULL)
	{
		switch(pCommandWords[comind]->Type)
		{
		case IOPERATION_DC_SV: 
			{ 
				pCode[lCode] = iOperation::New(this,pCommandWords[comind]->Prefix,pCommandWords[comind]->Command,pCommandWords[comind]->OutputCode,((iOperation_DC_SV *)pCommandWords[comind])->StatementPrefix,((iOperation_DC_SV *)pCommandWords[comind])->Relative,iValue::New(((iOperation_DC_SV *)pCommandWords[comind])->Value)); 
				break; 
			}
		case IOPERATION_FC_LB: 
			{ 
				if(((iOperation_FC_LB *)pCommandWords[comind])->Index != NULL) ((iOperation_FC_LB *)pCommandWords[comind])->Index->Register();
				pCode[lCode] = iOperation::New(this,pCommandWords[comind]->Prefix,pCommandWords[comind]->Command,pCommandWords[comind]->OutputCode,((iOperation_FC_LB *)pCommandWords[comind])->Point,((iOperation_FC_LB *)pCommandWords[comind])->Index,val,((iOperation_FC_LB *)pCommandWords[comind])->Class); 
				break; 
			}
		case IOPERATION_SF_V0: 
			{ 
				pCode[lCode] = iOperation::New(this,pCommandWords[comind]->Prefix,pCommandWords[comind]->Command,pCommandWords[comind]->OutputCode,((iOperation_SF_V0 *)pCommandWords[comind])->Function); 
				break; 
			}
		case IOPERATION_SF_V1: 
			{ 
				pCode[lCode] = iOperation::New(this,pCommandWords[comind]->Prefix,pCommandWords[comind]->Command,pCommandWords[comind]->OutputCode,valueRelative,val,((iOperation_SF_V1 *)pCommandWords[comind])->Function); 
				break; 
			}
		case IOPERATION_AF_V1: 
			{ 
				pCode[lCode] = iOperation::New(this,pCommandWords[comind]->Prefix,pCommandWords[comind]->Command,pCommandWords[comind]->OutputCode,arrComponent,valueRelative,val,((iOperation_AF_V1 *)pCommandWords[comind])->Function); 
				break; 
			}
		default: 
			{
				errorMessage = "Syntax error - statement not understood.";
				return 0;
			}
		}
		if(pCode[lCode] == 0) reportNullPointer(7302);
		pCode[lCode]->LineInScript = line - 1;
		nCode++;
	}
	else
	{
		//
		//  We have been here before - if this is an assignment, then assign the value
		//
		
		switch(pCommandWords[comind]->Type)
		{
		case IOPERATION_DC_SV: 
		case IOPERATION_SF_V0: 
			{ 
				break; // nothing to do here, just execute
			}
		case IOPERATION_FC_LB: 
			{ 
				if(((iOperation_FC_LB *)pCode[lCode])->Count != NULL) ((iOperation_FC_LB *)pCode[lCode])->Count->assignValue(Absolute,val); // assign a value if it exists
				break; 
			}
		case IOPERATION_SF_V1: 
			{ 
				((iOperation_SF_V1 *)pCode[lCode])->Relative = valueRelative;
				((iOperation_SF_V1 *)pCode[lCode])->Value->assignValue(Absolute,val); // this is an assignment to the rhs, not to the value itself
				break; 
			}
		case IOPERATION_AF_V1: 
			{ 
				((iOperation_AF_V1 *)pCode[lCode])->Relative = valueRelative;
				((iOperation_AF_V1 *)pCode[lCode])->Index->assignValue(Absolute,arrComponent); 
				((iOperation_AF_V1 *)pCode[lCode])->Value->assignValue(Absolute,val); // this is an assignment to the rhs, not to the value itself
				break; 
			}
		default: 
			{
				errorMessage = "Syntax error - statement not understood.";
				return 0;
			}
		}
		//
		//  Delete temporary values
		//
		if(arrComponent != NULL) arrComponent->Delete();
		if(val != NULL) val->Delete();
	}
	return -1;

}


iValue* iScript::transformToValue(iString pv0)
{
	iValue *val = this->transformSingleTokenToValue(pv0);

	if(val != NULL) 
	{
		//
		//  This is a single token
		//
		return val; 
	}
	else
	{
		//
		//  Parse the expression to check that it is valid.
		//
		return this->transformExpressionToValue(pv0);
	}
}


iValue* iScript::transformSingleTokenToValue(iString pv0, bool acceptArrays, bool forceFloat)
{
	bool ok;
	int vi;
	float vf;
	iValue *val;

	iString pv = pv0.stripWhiteSpace();

	//
	// A string
	//
	if(pv[0]=='"' && pv[(int)pv.length()-1]=='"')
	{

		return iValue::New("",ByValue,(iString)pv.mid(1,pv.length()-2));

	}

	//
	// A numeric constant
	//
	if(pv[0].isDigit() || pv[0]=='-' || pv[0]=='+')
	{

		//
		//  Is it an integer number?
		//
		if(!forceFloat)
		{
			vi = pv.toInt(&ok);
			if(ok) return iValue::New("",ByValue,vi);
		}

		//
		//  Is it a float number?
		//
		vf = pv.toFloat(&ok);
		if(ok) return iValue::New("",ByValue,vf);

		//
		//  No? Too bad
		//
		errorMessage = "Invalid numeric value.";
		return NULL;

	}

	//
	// An array
	//
	if(pv[0]=='(' && acceptArrays)
	{

		if(pv.right(1) != ")") // no closing parenthisis
		{
			errorMessage = "Incorrect array expression.";
			return NULL;  
		}

		pv = pv.mid(1,pv.length()-2);

		//
		//  Measure the number of components
		//
		int i, n = pv.contains(',') + 1;
		iString s;

		//
		//  Create an array of iValues
		//
		iValue **arr = new iValue*[n];

		//
		//  Fill it in
		//
		ok = true;
		for(i=0; i<n; i++)
		{
			s = pv.section(',',i,i);
			arr[i] = this->transformSingleTokenToValue(s,false,forceFloat | arraysAreFloat);
			if(arr[i]==NULL || (i>0 && !iValue::areTypesCompatible(arr[i],arr[0]))) ok = false;
		}

		//
		//  If unsuccessfull, destroy it.
		//
		if(!ok)
		{
			for(i=0; i<n; i++) if(arr[i] != NULL) arr[i]->Delete();
			delete [] arr; // delete the array, not the values - they will be deleted by iScript
			errorMessage = "Incorrect array expression.";
			return NULL;
		}

		return iValue::New("",ByValue,n,iValue::New("",ByValue,1),arr);

	}

	//
	//  A variable
	//
	if(pv[0].isLetter())
	{
		int i, j;
		iString ac;
		iValue *ind = NULL;
		//
		//  Look for the array component at the end in the form [N] where N is a iValue;
		//
		i = pv.find('[');
		if(i >= 0)    // a square bracket found
		{
			ac = pv.mid(i);
			pv = pv.left(i);
			if(ac.length()<3 || ac.left(1)!="[" || ac.right(1)!="]")
			{
				errorMessage = "Syntax error in the array index.";
				return NULL;
			}
			ac = ac.mid(1,ac.length()-2);
			
			ind = this->transformSingleTokenToValue(ac,false,false);
			if(ind==NULL || ind->getType()!=IVALUE_IS)
			{
				if(ind != NULL) ind->Delete();
				errorMessage = "Incorrect value of the array index.";
				return NULL;
			}
		}

		//
		//  Search for a parameter or a variable with this name: search parameters first, then variables
		//
		iValue **p;
		int n, iv;

		for(j=0; j<2; j++)
		{
			if(j == 0) 
			{
				p = pParameterWords;
				n = nParameterWords;
			}
			else
			{
				p = pVariable;
				n = nVariable;
			}

			for(i=0; i<n; i++) if(p[i] != NULL)
			{
				if(pv == p[i]->Name)
				{
					if(ind != NULL)  // this is an array
					{
						if(!p[i]->isArray())
						{
							ind->Delete();
							errorMessage = "Variable " + pv + " is not an array.";
							return NULL;
						}
						if(!ind->getValue(iv))
						{
							ind->Delete();
							errorMessage = "Variable " + ac + " has no value.";
							return NULL;
						}
						ind->Delete();  // index is not needed any more

						int size = ((iValue_VA*)p[i])->Size;
						if(size>0 && (iv<=0 || iv>size))
						{
							errorMessage = "Array index value is outside the array range.";
							return NULL;
						}
						val = ((iValue_VA*)p[i])->getComponent(iv);
					}
					else val = p[i];

					if(val != NULL) val->Register();  // register this variable to whatever wants it.
					return val;
					
				}
			}

		}

		//
		//  The word is not found - not a valid variable
		//
		if(ind != NULL) ind->Delete();
		errorMessage = "Undefined variable " + pv + ".";
		return NULL;
	}

	//
	//  Not a number, array, or a variable
	//
	errorMessage = "Incorrect syntax.";
	return NULL;

}


iValue* iScript::transformExpressionToValue(iString pv0, bool NotUsed(acceptArrays), bool NotUsed(forceFloat))
{
	int i, j, j1, j2, n, nVar0, iVar0 = 0;
	float f, fa[3];
	double d, da[3];
	iValue **p, *v;
	iString pv, s;

	//
	//  pre-parsing: identify all vector constants (1,1,1) and  vector components v[2], and turn them into
	//  tmp variables
	//
	pv = pv0;
	nVar0 = nVariable;

	while((i=pv.find('[')) >= 0)
	{
		for(j1=i-1; j1>0 && pv[j1].isLetterOrNumber(); j1--);
		if(!pv[j1].isLetterOrNumber()) j1++;

		j2 = pv.find(']',i);
		if(j2 == -1)
		{
			errorMessage = "No closing ] found.";
			return NULL;
		}

		v = this->transformSingleTokenToValue(pv.mid(j1,j2-j1+1));
		if(v == NULL)
		{
			errorMessage = pv.mid(j1,j2) + " is not a valid array component.";
			return NULL;
		}
		v->Name = "$$" + iString::number(iVar0++) + "$$";
		pVariable[nVariable++] = v;
		pv.replace(j1,j2-j1+1,v->Name);
	}

	while((i=pv.find(iRegExp("\\(\\w,\\w,\\w\\)"))) >= 0)
	{
		j1 = i;
		j2 = pv.find(')',i);
		if(j2 == -1)
		{
			errorMessage = "No closing ) found.";
			return NULL;
		}

		v = this->transformSingleTokenToValue(pv.mid(j1,j2-j1+1));
		if(v == NULL)
		{
			errorMessage = pv.mid(j1,j2) + " is not a array expression.";
			return NULL;
		}
		v->Name = "$$" + iString::number(iVar0++) + "$$";
		pVariable[nVariable++] = v;
		pv.replace(j1,j2-j1+1,v->Name);
	}

	//
	//  Set the function for the VTK parser
	//
	parser->RemoveAllVariables();
	parser->SetFunction(pv.latin1());

	//
	//  Assign variables
	//
	for(j=0; j<2; j++)
	{
		if(j == 0) 
		{
			p = pParameterWords;
			n = nParameterWords;
		}
		else
		{
			p = pVariable;
			n = nVariable;
		}
		
		for(i=0; i<n; i++) if(p[i]!=NULL && (j1=pv.findWord(p[i]->Name))>=0)
		{
			switch(p[i]->Type)
			{
			case IVALUE_IS:
			case IVALUE_FS:
				{
					if(!p[i]->getValue(f))
					{
						errorMessage = "The variable " + p[i]->Name + " has no value.";
						return NULL;
					}
					parser->SetScalarVariableValue(p[i]->Name.latin1(),(double)f);
					break;
				}
			case IVALUE_FP:
			case IVALUE_VA:
				{
					if(((iValue_VA*)p[i])->Size!=3 && ((iValue_VA*)p[i])->Size!=0)
					{
						errorMessage = "Only 3-component arrays (vectors) can be used in expressions.";
						return NULL;
					}
					if(!p[i]->getValue(3,fa))
					{
						errorMessage = "The array variable " + p[i]->Name + " has no value.";
						return NULL;
					}
					da[0] = fa[0];
					da[1] = fa[1];
					da[2] = fa[2];
					parser->SetVectorVariableValue(p[i]->Name.latin1(),da);
					break;
				}
			default: ;
			}
		}
		
	}
	

	if(parser->GetScalarResult(d))
	{
		//
		//  Is this a boolean expression?
		//
		if(pv.find('!')>=0 || pv.find('>')>=0 || pv.find('<')>=0 || pv.find(leChar)>=0 || pv.find(geChar)>=0 || pv.find(eqChar)>=0 || pv.find(neChar)>=0 || pv.find(andChar)>=0 || pv.find(orChar)>=0) 
		{
			//
			//  Convert to boolean if possible
			//
			if(fabs(d)<1.0e-12 || fabs(d-1.0)<1.0e-12) 
			{
				v = iValue::New("",ByValue,(round(d)==1));
			}
			else
			{
				errorMessage = "Invalid boolean expression.";
				v = NULL;
			}
		}
		else v = iValue::New("",ByValue,(float)d);
	}
	else if(parser->GetVectorResult(da))
	{
		//
		//  Create an array of iValues
		//
		n = 3;
		iValue **arr = new iValue*[n];
		
		//
		//  Fill it in
		//
		for(i=0; i<n; i++)
		{
			arr[i] = iValue::New("",ByValue,(float)da[i]);
		}
		
		v = iValue::New("",ByValue,n,iValue::New("",ByValue,1),arr);
	}
	else
	{
		errorMessage = parser->GetErrorMessage();
		v = NULL;
	}

	for(i=nVar0; i<nVariable; i++) pVariable[i]->Delete();  // delete temporary variables
	nVariable = nVar0;

	return v;

}


