/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of itensor.h
*/

#include "imath.h"
#include "itensor.h"
#include "iobjectfactory.h"

#include "ipalette.h"
#include "ivtk.h"
#include "ilimits.h"

#include "itensordataconverter.h"
#include "ipolygonaldataextender.h"
#include "ivtkcallback.h"
#include "idatareader.h"
#include "iuniformmeshdata.h"
#include "ivisualobjectobserver.h"
#include "icolorbars.h"

#include <vtkCommand.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkProperty.h>
#include <vtkExtractVOI.h>
#include <vtkSphereSource.h>
#include <vtkTensorGlyph.h>
#include <vtkLookupTable.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>


void reportNullPointer(int ec);


//
// iTensor class
//

iTensor* iTensor::New(iVTK *mVTK)
{
	return (iTensor *)iObjectFactory::createObject("Tensor",mVTK);
}


iTensor::iTensor(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	mode = 0;
	pal = 0;
	sampleRate = 16;
	glyphSize = 1.0;
	colorBy = 0;
	scalingOn = true;

	color = NICE_DEFAULT_COLOR;
	//
	//  Do VTK stuff
	//	
	tensorActor = vtkLODActor::New();
	if(tensorActor == 0) reportNullPointer(6701);
	tensorMapper = vtkPolyDataMapper::New();
	if(tensorMapper == 0) reportNullPointer(6702);
	tensorGlyph = vtkTensorGlyph::New();
	if(tensorGlyph == 0) reportNullPointer(6703);
	tensorExtender = iPolygonalDataExtender::New(this);
	if(tensorExtender == 0) reportNullPointer(6704);
	tensorFilter = vtkExtractVOI::New();
	if(tensorFilter == 0) reportNullPointer(6705);
	tensorDataConverter = iTensorDataConverter::New(this);
	if(tensorDataConverter == 0) reportNullPointer(6706);
	glyphSphere = vtkSphereSource::New(); 
	if(glyphSphere == 0) reportNullPointer(6707);

	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
	tensorMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	tensorGlyph->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	tensorExtender->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	tensorFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	tensorDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

	tensorDataConverter->SetInput(myVTK->getReader()->getTensorOutput());
	tensorDataConverter->setScalars(myVTK->getReader()->getMeshOutput());

	tensorFilter->SetInput(tensorDataConverter->GetOutput());

	tensorGlyph->SetInput(tensorFilter->GetOutput());
	tensorGlyph->SetSource(glyphSphere->GetOutput());
	tensorGlyph->ScalingOn();

	tensorExtender->SetInput(tensorGlyph->GetOutput());

	tensorMapper->SetInput(tensorExtender->GetOutput());
	tensorMapper->ColorByArrayComponent(0,0); 
    tensorMapper->ScalarVisibilityOff();
	tensorMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
	tensorMapper->SetScalarModeToDefault();
	tensorMapper->UseLookupTableScalarRangeOff();
	
	tensorActor->SetMapper(tensorMapper);
	tensorActor->SetPosition(0.0,0.0,0.0);

	tensorProperty = tensorActor->GetProperty();
	tensorProperty->SetInterpolationToGouraud();
	tensorProperty->SetAmbient(ambient);
	tensorProperty->SetDiffuse(diffuse);
	tensorProperty->SetSpecular(specular);
	tensorProperty->SetSpecularPower(specularPower);
	//
	//  Add observer to keep information about this object
	//
	tensorActor->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_TENS,IDATATYPE_TENS);

	tensorActor->VisibilityOff();

}


iTensor::~iTensor()
{
	
	if(!init) myVTK->removeProp(tensorActor);
	tensorActor->Delete();	
	tensorMapper->Delete();
	tensorExtender->Delete();
	tensorFilter->Delete();
	tensorGlyph->Delete();
	tensorDataConverter->Delete();
	glyphSphere->Delete(); 

}


void iTensor::reset()
{
	tensorDataConverter->SetInput(myVTK->getReader()->getTensorOutput());
	tensorDataConverter->setScalars(myVTK->getReader()->getMeshOutput());
	this->updatePipeline();
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(tensorActor);
		init = true;
	}
}


void iTensor::computeSpacing()
{
	curSpacing = myVTK->getReader()->getTensorOutput()->GetSpacing()[0];
}


void iTensor::updatePipeline()
{

	this->computeSpacing();
	tensorGlyph->SetScaleFactor(glyphSize*sampleRate*curSpacing/this->getLimits()->getTenHigh());

	if(colorBy >0 && colorBy <= this->getLimits()->getNvar()) 
	{
		tensorDataConverter->setCurrentVar(colorBy);
		tensorFilter->GetInput()->GetPointData()->SetScalars(tensorDataConverter->GetOutput()->GetPointData()->GetScalars());
		tensorMapper->SetScalarRange(this->getLimits()->getVarLow(colorBy),this->getLimits()->getVarHigh(colorBy));
		tensorMapper->ScalarVisibilityOn();
	}
	else 
	{
		tensorMapper->ScalarVisibilityOff();
	} 

	tensorFilter->SetSampleRate(sampleRate,sampleRate,sampleRate);
	if(sampleRate == 1)
	{
		tensorGlyph->SetInput(myVTK->getReader()->getTensorOutput());
	} 
	else 
	{
		tensorGlyph->SetInput(tensorFilter->GetOutput());
	}

	tensorExtender->SetInput(tensorGlyph->GetOutput());
	
}


void iTensor::setScalingOn(bool s)
{
	if(s != scalingOn)
	{
		scalingOn = s;
		if(s) tensorGlyph->ScalingOn(); else tensorGlyph->ScalingOff();
	}
}


void iTensor::setColor(iColor c)
{ 
	color = c;
	tensorProperty->SetColor(color.red()/255.0,color.green()/255.0,color.blue()/255.0);
}


void iTensor::setGlyphSize(float v)
{ 
	if(v>0.0 && v<1.0e5 && v!=glyphSize)
	{
		glyphSize = v;
		this->updatePipeline();
	}
}


void iTensor::setColorBy(int v)
{ 
	if(v>=0 && v!=colorBy) 
	{
		if(v > 0) this->showColorBars(false);
		colorBy = v;
		if(v > 0) this->showColorBars(true);
		this->updatePipeline();
	}
}


void iTensor::setSampleRate(int v)
{ 
	if(v>0 && v<1600 && v!=sampleRate) 
	{
		sampleRate = v;
		tensorFilter->SetSampleRate(sampleRate,sampleRate,sampleRate);
	}
}


void iTensor::setPal(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	this->showColorBars(false);
	pal = p;
	this->showColorBars(true);
	tensorMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
}


void iTensor::showClipPlane(bool s)
{

	if(s && !clipPlaneOn)
	{
		tensorMapper->AddClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = true;
	} 
	if(!s && clipPlaneOn)
	{
		tensorMapper->RemoveClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = false;
	}

}


void iTensor::showColorBars(bool show)
{
	if(tensorActor->GetVisibility() == 0) return;
	if(colorBy > 0) myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_TENSOR,colorBy,dataChannel,pal,show);
}


void iTensor::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	tensorProperty->SetAmbient(v);
}


void iTensor::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	tensorProperty->SetDiffuse(v);
}


void iTensor::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	tensorProperty->SetSpecular(v);
}


void iTensor::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	tensorProperty->SetSpecularPower(v);
}


void iTensor::show(bool show)
{
	  
	if(show)
	{
		if(init)
		{
			this->updateDataChannel();
			this->updatePipeline();
			this->setColor(color);
			myVTK->addProp(tensorActor);
			init = false;
		}
		if(!init)
		{
			this->updatePipeline();
			tensorActor->VisibilityOn();
			this->showColorBars(true);
		}
	} 
	else 
	{
		if(!init) 
		{
			this->showColorBars(false);
			tensorActor->VisibilityOff();
		}
	}
	
}


float iTensor::getMemorySize()
{
	float s = 0.0;

	s += tensorGlyph->GetOutput()->GetActualMemorySize();
	s += tensorExtender->GetOutput()->GetActualMemorySize();
	s += tensorFilter->GetOutput()->GetActualMemorySize();
	s += tensorDataConverter->GetOutput()->GetActualMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iTensor::packState(iString &s)
{

	iVisualObject::packState(s);

	this->packValue(s,"iTensor::mode",mode);
	this->packValue(s,"iTensor::color",color);
	this->packValue(s,"iTensor::pal",pal);
	this->packValue(s,"iTensor::colorBy",colorBy);
	this->packValue(s,"iTensor::sampleRate",sampleRate);

	this->packValue(s,"iTensor::glyphSize",glyphSize);

}


void iTensor::unpackState(iString s)
{
	int i; float f; iColor c;

	iVisualObject::unpackState(s);

//	if(this->unpackValue(s,"iTensor::mode",i)) this->setMode(i);
	if(this->unpackValue(s,"iTensor::color",c)) this->setColor(c);
	if(this->unpackValue(s,"iTensor::pal",i)) this->setPal(i);
	if(this->unpackValue(s,"iTensor::colorBy",i)) this->setColorBy(i);
	if(this->unpackValue(s,"iTensor::sampleRate",i)) this->setSampleRate(i);
	
	if(this->unpackValue(s,"iTensor::glyphSize",f)) this->setGlyphSize(f);

}

