/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iuniformmeshdataextender.h"

#include "ilimits.h"
#include "ivisualobject.h"
#include "iuniformmeshdata.h"

#include <vtkObjectFactory.h>
#include <vtkImageData.h>
#include <vtkPointData.h>
#include <vtkUnsignedCharArray.h>



//------------------------------------------------------------------------------
iUniformMeshDataExtender* iUniformMeshDataExtender::New(iVisualObject *m)
{
	return new iUniformMeshDataExtender(m);
}


iUniformMeshDataExtender::iUniformMeshDataExtender(iVisualObject *m) : iVisualObjectSource(m)
{
	active = false;
}


void iUniformMeshDataExtender::ExecuteInformation()
{
	int i, ext[6];
	vtkFloat org[3];
	vtkImageData *input = this->GetInput();
	vtkStructuredPoints *output = this->GetOutput();
	
	if (output == NULL || input == NULL)
    {
		return;
    }
	
	input->GetWholeExtent(ext);
	input->GetOrigin(org);
	
	for(i=0; i<3; i++)
	{
		ext[2*i+1] = 1 + (ext[2*i+1]-1)*(myObject->getLimits()->getExtendUp(i)-myObject->getLimits()->getExtendDown(i)+1);
		org[i] += 2.0*myObject->getLimits()->getExtendDown(i);
	}
	
	output->SetWholeExtent(ext);
	output->SetSpacing(input->GetSpacing());
	output->SetOrigin(org);
}


void iUniformMeshDataExtender::ExecuteData(vtkDataObject *)
{
	iUniformMeshData *input = (iUniformMeshData *)this->GetInput();
	iUniformMeshData *output = (iUniformMeshData *)this->GetOutput();
	int dims[3], dimsout[3], ext[3], extDown[3], extUp[3];
	int i,j,i1,j1,k1;
	vtkFloat pos[3];
	
	if(input == 0) return;
	int n = input->GetNumberOfScalarComponents();
	if(n == 0) return;

	output->Initialize();
	
	bool work = false;
	for(i=0; i<3; i++)
	{
		extDown[i] = myObject->getLimits()->getExtendDown(i);
		extUp[i] = myObject->getLimits()->getExtendUp(i);
		if(extDown[i] != 0) work = true;
		if(extUp[i] != 0) work = true;
	}

	if(!work || !active) 
	{
		output->ShallowCopy(input);
		return;
	}

	input->GetSpacing(pos);
	output->SetSpacing(pos);
	
	input->GetDimensions(dims);
	input->GetOrigin(pos);

	int next = 1;
	for(i=0; i<3; i++)
	{
		ext[i] = extUp[i] - extDown[i] + 1;
		dimsout[i] = 1 + (dims[i]-1)*ext[i];
		next *= ext[i];
	}
	vtkIdType size = input->GetScalarSize()*input->GetNumberOfScalarComponents();
	vtkIdType size0 = (dims[0]-1)*size;
	vtkIdType size1 = size0 + size;

	output->SetScalarType(input->GetScalarType());
	output->SetNumberOfScalarComponents(input->GetNumberOfScalarComponents());
		
	for(i=0; i<3; i++) pos[i] += 2.0*extDown[i];
	output->SetOrigin(pos);
	
	output->SetDimensions(dimsout);

	vtkUnsignedCharArray *data = vtkUnsignedCharArray::New();
	data->SetNumberOfComponents(size);
	data->SetNumberOfTuples(dimsout[0]*dimsout[1]*dimsout[2]);
	output->GetPointData()->SetScalars(data);

	unsigned char *ptrin = (unsigned char *)input->GetScalarPointer();
	unsigned char *ptrout = (unsigned char *)data->GetVoidPointer(0);
	
	vtkIdType loff, loff1, loff2;
	for(i1=0; i1<ext[2]; i1++) for(i=0; i<dims[2]; i++) 
	{

		this->UpdateProgress((float)(i+dims[2]*i1)/(dims[2]*ext[2]));
		if(this->GetAbortExecute()) break;

		for(j1=0; j1<ext[1]; j1++) for(j=0; j<dims[1]; j++)
		{
			loff1 = size*dims[0]*(j+dims[1]*i);
			loff2 = size*dimsout[0]*(j+(dims[1]-1)*j1+dimsout[1]*(i+(dims[2]-1)*i1));
			for(k1=0; k1<ext[0]; k1++)
			{
				loff = size0*k1 + loff2;
				memcpy(ptrout+loff,ptrin+loff1,size1); 
			}
		}
	}

	data->Delete();

}


float iUniformMeshDataExtender::getMemorySize()
{
	int i;
	bool work = false;
	//
	// Could be zero if the pipeline has not executed yet.
	//
	if(myObject->getLimits() == 0)
	{
		return 0.0;
	}

	for(i=0; i<3; i++)
	{
		if(myObject->getLimits()->getExtendDown(i) != 0) work = true;
		if(myObject->getLimits()->getExtendUp(i) != 0) work = true;
	}

	if(!work || !active) return 0.0;
	return this->GetOutput()->GetActualMemorySize();

}
