/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include <stdio.h>

#include <qapplication.h>
#include <qrect.h>
#include <qsizepolicy.h>
#include <qfileinfo.h>
#include <qfont.h>
#include <qpaintdevice.h>

#include "ifrit.h"

#include "ivtk.h"
#include "iqt.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "ienvironment.h"
#include "iparallelizator.h"
#include "idialogmessage.h"


void reportNullPointer(int ec);
extern int errorCode;


int main( int argc, char ** argv )
{
	
	int i, ret = 0;
	iDialogMessage *flashWindow = 0;
	//
	//  Various switches setable via command-line options
	//
	bool showFlashWindow = true;
	bool smallWindow = false;
	bool remoteWindow = false;
	bool accept8bit = false;
	bool startDocked = false;
	bool idiosyncratic = true;
	int numProcs = 1;
	iString ignoredOptions;
	//
	//  Let's start!
	//
    QApplication a( argc, argv );
	//
	//  Check that sizes of basic types are correct
	//
	if(sizeof(int)!=4 || sizeof(float)!=4 || sizeof(double)!=8)
	{
		cerr << "IFRIT has not been ported to this machine.\n";
		exit(0);
	}
	
	//
	//  Command-line options
	//
	char *initFile = 0;
	char *scriptFile = 0;
	char *dataDir = 0;
	for(i=1; i<argc; i++)
	{
		if(strcmp(argv[i],"-h")==0 || strcmp(argv[i],"-help")==0)
		{
			cout << "Format: ifrit [option] [dir_name]\n";
			cout << "Options:\n";
			cout << "-8                 accept an 8-bit display\n";
			cout << "-d                 start IFRIT with windows docked\n";
			cout << "-nf                do not show the splash screen at start-up\n";
			cout << "-sw                use small font to reduce the QT window size\n";
			cout << "-rw                IFRIT is used remotely\n";
			cout << "-ni                IFRIT is not idiosyncratic\n";
			cout << "-np <number>       set the number of processors to use\n";
			cout << "-i <file_name>     load the state from file_name\n";
			cout << "-b <file_name>     execute animation script in the screenless mode\n";
			cout << "-h                 show this help" << endl;
			exit(0);
		} 
		else if(strcmp(argv[i],"-nf") == 0)
		{
			showFlashWindow = false;
			ignoredOptions += "-nf";
		} 
		else if(strcmp(argv[i],"-sw") == 0)
		{
			smallWindow = true;
			ignoredOptions += "-sw";
		} 
		else if(strcmp(argv[i],"-rw") == 0)
		{
			remoteWindow = true;
			ignoredOptions += "-rw";
		} 
		else if(strcmp(argv[i],"-ni") == 0)
		{
			idiosyncratic = false;
			ignoredOptions += "-ni";
		} 
		else if(strcmp(argv[i],"-8") == 0)
		{
			accept8bit = true;
			ignoredOptions += "-8";
		} 
		else if(strcmp(argv[i],"-d") == 0)
		{
			startDocked = true;
			ignoredOptions += "-d";
		} 
		else if(strcmp(argv[i],"-b") == 0)
		{
			i++;
			if(i<argc && QFileInfo(QString(argv[i])).isFile())
			{
				scriptFile = argv[i];
			} 
			else 
			{
				cerr << "-b option requires the name of a script file to follow.\n";
				cerr << "IFRIT will now exit.\n";
				exit(1);
			}
		} 
		else if(strcmp(argv[i],"-i") == 0)
		{
			i++;
			if(i<argc && QFileInfo(QString(argv[i])).isFile())
			{
				initFile = argv[i];
			} 
			else 
			{
				cerr << "-i option requires the name of a state file to follow.\n";
				cerr << "IFRIT will now exit.\n";
				exit(1);
			}
		} 
		else if(strcmp(argv[i],"-np") == 0)
		{
			i++;
			bool ok = false;
			if(i < argc) numProcs = QString(argv[i]).toInt(&ok);
			if(!ok)
			{
				cerr << "-np option requires a number to follow.\n";
				cerr << "IFRIT will now exit.\n";
				exit(1);
			}
		} 
		else 
		{
			if(QFileInfo(QString(argv[i])).isDir())
			{
				dataDir = argv[i];
			} 
			else 
			{
				cerr << "Undefined command line option: " << argv[i] << "\n";
				cerr << "IFRIT will now exit.\n";
				exit(1);
			}
		}
	}

	if(scriptFile != 0)
	{
		cout << "IFRIT is being run in the batch mode.";
		if(!ignoredOptions.isEmpty()) cout << " Options: " << ignoredOptions << " are ignored.";
		cout << endl;
	}
	//
	//  Check for the 8-bit display
	//
#ifdef Q_WS_X11
	if(scriptFile==0 && QPaintDevice::x11AppDepth()<16)
	{
		if(!accept8bit)
		{
			cerr << "You are not using a True Color (24-bit) monitor.\n"
				<< "Visualizations will be crappy without the True Color support.\n"
				<< "If you wish to continue anyway, specify -8 as an option.\n";
			exit(0);
		}
	}
#endif

	if(scriptFile == 0)
	{
		
		if(remoteWindow) showFlashWindow = false;
		
		//
		//  Create and show message window
		//
		flashWindow = new iDialogMessage(qApp->desktop(),0,false,Qt::WStyle_Customize|Qt::WStyle_StaysOnTop|Qt::WStyle_NoBorder);
		if(flashWindow == 0) reportNullPointer(2006);
		//
		//  calculate the position of the flash window
		//
		flashWindow->move((qApp->desktop()->width()-flashWindow->width())/2,(qApp->desktop()->height()-flashWindow->height())/2);
		if(showFlashWindow) flashWindow->show(); else cout << "IFRIT is being initialized..." << endl;
		qApp->syncX();
		qApp->flush();
		
	}
	else cout << "IFRIT is being initialized..." << endl;
	//
	//  Construct IFRIT
	//
	bool optionsLoaded = ifrit::construct(flashWindow,initFile);
	//
	//  Apply setting from command line options
	//
	if(scriptFile == 0)
	{
		
		if(dataDir != 0) iEnvironment::getInstance()->set_IFRIT_DATA_DIR(dataDir);
		iQTWindow::getCurrentWindow()->setSmallWindow(smallWindow);
		if(remoteWindow) iQTWindow::getCurrentWindow()->setRemoteControlled();
		if(!idiosyncratic) iQTWindow::getCurrentWindow()->setIdiosyncratic(false);
		iParallelizator::setNumberOfProcessors(numProcs);
		//
		//  Remove the flash window
		//
		flashWindow->hide();
		delete flashWindow;
		flashWindow = 0;
		//
		//  Show ifrit windows
		//
		ifrit::show(!optionsLoaded,startDocked);
		
		a.connect( &a, SIGNAL( lastWindowClosed() ), &a, SLOT( quit() ) );
		ret = a.exec();
		
	}
	else
	{
		ifrit::executeScript(scriptFile);
	}
	//
	//  Destroy ifrit
	//
	ifrit::destruct();
	
	
    return ret;
    
}

