/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of ianimator.h
*/

#include "iglobals.h"
#include "ianimator.h"

#include "ivtk.h"
#include "ivtkwindow.h"

#include "imath.h"
#include "ianimatorscript.h"
#include "idatareader.h"
#include "ixsection.h"

#include <vtkRenderer.h>
#include <vtkMath.h>
#include <vtkCamera.h>
#include <vtkRenderWindow.h>
#include <vtkWindowToImageFilter.h>
#include <vtkImageBlend.h>
#include <vtkImageData.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>
#include <vtkCommand.h>

#ifndef I_VTK_VERSION_40
#include <icamerapath.h>
#include <vtkActor.h>
#include <vtkPolyDataMapper.h>
#include <vtkConeSource.h>
#include <vtkCellArray.h>
#endif

#define REN myVTK->getRenderer()
#define DATAREADER myVTK->getReader()
#define CAM myVTK->getRenderer()->GetActiveCamera()
#define XSECTION myVTK->getXsectionFamily()->getCurrentMember()

#define RAN_U (2.0*vtkMath::Random()-1.0)
#define RAN_N (1.4142*tan(2.0*RAN_U/3.1415926))
#define LEN(x) (sqrt(x[0]*x[0]+x[1]*x[1]+x[2]*x[2]))


#define CAM_FUNCTION0(fun) \
{ \
	if(myVTK->isBroadcastMouseEventsOn()) \
	{ \
		for(int i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->getRenderer()->GetActiveCamera()->fun(); \
	} \
	else \
	{ \
		myVTK->getRenderer()->GetActiveCamera()->fun(); \
	} \
}

#define CAM_FUNCTION1(fun,arg) \
{ \
	if(myVTK->isBroadcastMouseEventsOn()) \
	{ \
		for(int i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->getRenderer()->GetActiveCamera()->fun(arg); \
	} \
	else \
	{ \
		myVTK->getRenderer()->GetActiveCamera()->fun(arg); \
	} \
}



void reportNullPointer(int ec);
void iSleep(int);

#ifndef I_VTK_VERSION_40
//
//  helper class
//
class iAnimatorEventObserver : public vtkCommand, private iVTKSource
{
	
public:
	
	static iAnimatorEventObserver *New(iVTK *m) { return new iAnimatorEventObserver(m); }

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);
	virtual void setCameraPath(iCameraPath *w){ cp = w; }
	virtual void setFocalPointPath(iCameraPath *w){ fp = w; }
	
	bool focalPathToPoint;

protected:

	iAnimatorEventObserver(iVTK *m);
	virtual ~iAnimatorEventObserver();
	
private:

	iCameraPath *cp, *fp;
	vtkActor *cameraPathHeadActor, *focalPathBandActor;
	vtkPolyDataMapper *cameraPathHeadMapper, *focalPathBandMapper;
	vtkConeSource *cameraPathHeadSource;
	vtkPolyData *pathData, *pathData2, *focalPathBand;

	vtkFloat dxPrev[3];

};


iAnimatorEventObserver::iAnimatorEventObserver(iVTK *m) : iVTKSource(m)
{

	cp = fp = 0;

	focalPathToPoint = false;
		
	cameraPathHeadActor = vtkActor::New();
	cameraPathHeadMapper = vtkPolyDataMapper::New();
	cameraPathHeadSource = vtkConeSource::New();
	cameraPathHeadSource->SetResolution(12);
	cameraPathHeadSource->SetRadius(0.1);
	cameraPathHeadSource->SetHeight(0.15);
	cameraPathHeadMapper->SetInput(cameraPathHeadSource->GetOutput());
	cameraPathHeadActor->SetMapper(cameraPathHeadMapper);
	cameraPathHeadActor->GetProperty()->SetColor(0.0,0.68,1.0);
	cameraPathHeadActor->VisibilityOff();
	cameraPathHeadActor->PickableOff();

	focalPathBandActor = vtkActor::New();
	focalPathBandMapper = vtkPolyDataMapper::New();
	focalPathBand = vtkPolyData::New();
	focalPathBandMapper->SetInput(focalPathBand);
	focalPathBandActor->SetMapper(focalPathBandMapper);
	focalPathBandActor->GetProperty()->SetColor(0.4,0.5,0.6);
	focalPathBandActor->VisibilityOff();
	focalPathBandActor->PickableOff();

	dxPrev[0] = 1.0; dxPrev[1] = dxPrev[2] = 0.0;

	REN->AddProp(cameraPathHeadActor);
	REN->AddProp(focalPathBandActor);

	pathData = vtkPolyData::New();
	pathData2 = vtkPolyData::New();

}


iAnimatorEventObserver::~iAnimatorEventObserver()
{

	pathData->Delete();
	pathData2->Delete();
	REN->RemoveProp(cameraPathHeadActor);
	cameraPathHeadActor->Delete();
	cameraPathHeadMapper->Delete();
	cameraPathHeadSource->Delete();
	REN->RemoveProp(focalPathBandActor);
	focalPathBandActor->Delete();
	focalPathBandMapper->Delete();
	focalPathBand->Delete();

}


void iAnimatorEventObserver::Execute(vtkObject *caller, unsigned long eventId, void *vtkNotUsed(callData))
{
	switch (eventId)
	{
	case vtkCommand::InteractionEvent:
		{
			if(cp!=0 && caller==cp)
			{
				cp->GetPolyData(pathData);
				
				vtkFloat q1, x1[3], x2[3], dx[3], ax[3];
				int i, i2;
				
				pathData->GetPoints()->GetPoint(0,x1);
				i2 = 1;
				while(i2 < pathData->GetNumberOfPoints())
				{
					pathData->GetPoints()->GetPoint(i2,x2);
					for(i=0; i<3; i++) dx[i] = x2[i] - x1[i];
					if(vtkMath::Norm(dx) > 0) break;
					i2++;
				}
				vtkMath::Normalize(dx);

				//
				//  Rotate
				//
				cameraPathHeadActor->SetOrientation(0.0,0.0,0.0);
				ax[0] = 0.5*(1+dx[0]); ax[1] = 0.5*dx[1]; ax[2] = 0.5*dx[2];
				if(vtkMath::Norm(ax) > 0.0)
				{
					cameraPathHeadActor->RotateWXYZ(180.0,ax[0],ax[1],ax[2]);
				}
				else
				{
					if(vtkMath::Dot(dx,dxPrev) < 0.0)
					{
						//
						//  Flip
						//
						cameraPathHeadActor->RotateWXYZ(180.0,0.0,1.0,0.0);
					}
				}
				
				//
				//  Translate
				//
				q1 = cameraPathHeadSource->GetHeight();
				for(i=0; i<3; i++) dx[i] = x1[i] - q1*dx[i];
				cameraPathHeadActor->SetPosition(dx);
			}

			if(cp!=0 && fp!=0)
			{
				int i;

				if(caller==fp && focalPathToPoint)
				{
					vtkFloat x[3];
					i = fp->getCurrentHandleIndex();
					if(i<0 || i>=fp->GetNumberOfHandles()) i = 0;
					fp->GetHandlePosition(i,x);
					for(i=0; i<fp->GetNumberOfHandles(); i++) fp->SetHandlePosition(i,x);
				}

				if(caller != cp) cp->GetPolyData(pathData);
				fp->GetPolyData(pathData2);

				int n = pathData->GetNumberOfPoints();
				int nstep = 1;
				if(n > 32) 
				{
					n = 32;
					nstep = pathData->GetNumberOfPoints()/n;
				}

				vtkPoints *newPoints = vtkPoints::New(VTK_FLOAT);
				vtkCellArray *newLines = vtkCellArray::New();
				newPoints->SetNumberOfPoints(2*n);
				newLines->Allocate(3*n);

				vtkIdType cell[2];
				for(i=0; i<n; i++)
				{
					newPoints->SetPoint(2*i+0,pathData->GetPoints()->GetPoint(i*nstep));
					newPoints->SetPoint(2*i+1,pathData2->GetPoints()->GetPoint(i*nstep));
					cell[0] = 2*i; cell[1] = 2*i + 1;
					newLines->InsertNextCell(2,cell);
				}

				focalPathBand->SetPoints(newPoints);
				newPoints->Delete();

				focalPathBand->SetLines(newLines);
				newLines->Delete();

			}

			break;
		}
	case vtkCommand::EnableEvent:
		{
			cameraPathHeadActor->VisibilityOn();
			focalPathBandActor->VisibilityOn();
			break;
		}
	case vtkCommand::DisableEvent:
		{
			cameraPathHeadActor->VisibilityOff();
			focalPathBandActor->VisibilityOff();
			break;
		}
	}
}

#endif

//
//  iAnimator class
//
iAnimator* iAnimator::New(iVTK *m)
{
	return new iAnimator(m); // non-inheritable, so no need to use Object Factory
}


iAnimator::iAnimator(iVTK *m) : iObject(m)
{

	started = debugMode = useScript = inheritSettings = false;
	restoreCamera = true;

	scriptFileName = "";

	newrec = false;
	currec = prevrec = -1;
	curframe = totframe = 0;

	state.mode = 1;
	state.nframes = 1;
	state.dphi = 1.0;
	state.dtheta = 0.0;
	state.dscale = 1.0;
	state.droll = 0.0;
	state.flybySpeed = 0.01;
	state.slideSpeed = 0.01;
	state.nBlendedFrames = 0;
	state.nTransitionFrames = 0;
	state.titlePageFile = iString("");
	state.titlePageNumFrames = 0;
	state.titlePageNumBlendedFrames = 0;
	state.logoFile = iString("");
	state.logoOpacity = 0.5;
	state.logoPosition = 0;

	randstep = 0.03;
	seed = 12345;
	srand(seed);

	numBlenderBase = 0;
	blenderBase = 0;

	titlePageImage = vtkImageData::New();
	if(titlePageImage == 0) reportNullPointer(9521);
	logoImage = vtkImageData::New();
	if(logoImage == 0) reportNullPointer(9522);

	useFocalPath = doingTitlePage = false;

	animatorScript = iAnimatorScript::New(this);
	if(animatorScript == 0) reportNullPointer(9506);

#ifndef I_VTK_VERSION_40

	animatorEventObserver = iAnimatorEventObserver::New(myVTK);
	if(animatorEventObserver == 0) reportNullPointer(9507);

	cameraPath = iCameraPath::New();
	if(cameraPath == 0) reportNullPointer(9508);
	cameraPath->SetNumberOfHandles(3);
	cameraPath->SetResolution(200);
	cameraPath->SetInteractor(myVTK->getInteractor());
	cameraPath->GetLineProperty()->SetLineWidth(4.0);
	cameraPath->GetLineProperty()->SetColor(0.0,0.0,0.0);
	cameraPath->SetHandlePosition(2,-1.0,0.0,1.0);
	cameraPath->SetHandlePosition(1,0.0,0.0,1.0);
	cameraPath->SetHandlePosition(0,1.0,0.0,1.0);

	focalPath = iCameraPath::New();
	if(focalPath == 0) reportNullPointer(9509);
	focalPath->SetNumberOfHandles(3);
	focalPath->SetResolution(200);
	focalPath->SetInteractor(myVTK->getInteractor());
	focalPath->GetLineProperty()->SetLineWidth(4.0);
	focalPath->GetLineProperty()->SetColor(0.4,0.5,0.6);
	focalPath->SetHandlePosition(2,-0.3,0.0,0.0);
	focalPath->SetHandlePosition(1,0.0,0.0,0.0);
	focalPath->SetHandlePosition(0,0.3,0.0,0.0);

	cameraPath->AddObserver(vtkCommand::InteractionEvent,animatorEventObserver);
	cameraPath->AddObserver(vtkCommand::EnableEvent,animatorEventObserver);
	cameraPath->AddObserver(vtkCommand::DisableEvent,animatorEventObserver);
	focalPath->AddObserver(vtkCommand::InteractionEvent,animatorEventObserver);
	focalPath->AddObserver(vtkCommand::EnableEvent,animatorEventObserver);
	focalPath->AddObserver(vtkCommand::DisableEvent,animatorEventObserver);

	animatorEventObserver->setCameraPath(cameraPath);
	cameraPath->InvokeEvent(vtkCommand::InteractionEvent);

	pathData = vtkPolyData::New();
	if(pathData == 0) reportNullPointer(9510);
	pathData2 = vtkPolyData::New();
	if(pathData2 == 0) reportNullPointer(9511);

#endif

}


iAnimator::~iAnimator()
{
	int i;

	titlePageImage->Delete();
	logoImage->Delete();

	delete animatorScript;
	if(blenderBase != 0) 
	{
		for(i=0; i<numBlenderBase; i++) if(blenderBase[i] != 0) blenderBase[i]->Delete();
		delete [] blenderBase;
	}
#ifndef I_VTK_VERSION_40
	cameraPath->RemoveObserver(vtkCommand::InteractionEvent);
	focalPath->RemoveObserver(vtkCommand::InteractionEvent);
	animatorEventObserver->Delete();
	cameraPath->Off();
	focalPath->Off();
	cameraPath->SetInteractor(NULL);
	focalPath->SetInteractor(NULL);
	cameraPath->Delete();
	focalPath->Delete();
	pathData->Delete();
	pathData2->Delete();
#endif

}


//
//  This is the main driver
//
int iAnimator::animate()
{
	static iString nullScript = "render 999999";
	//
	//  Are we using a script?
	//
	if(useScript)
	{
		return animatorScript->run();
	}
	else
	{
		return animatorScript->run(nullScript);
	}
}


void iAnimator::reset()
{
	started = false;
	doingTitlePage = false;
	srand(seed);
	myVTK->justifyLabelLeft(false);
}


void iAnimator::setDebugMode(bool s) 
{ 
	debugMode = s; 
	animatorScript->setDebugMode(s); 
}


void iAnimator::resetCurrentFile()
{
	newrec = true;
	prevrec = currec;
	currec = DATAREADER->getRecordNumber();
	curframe = totframe = 0;
}


void iAnimator::saveState()
{

	if(XSECTION != 0) state.xsecPos = XSECTION->getPos(); else state.xsecPos = -1;
	state.ifBoundingBox = myVTK->isBoundingBoxVisible();
	state.ifColorBars = myVTK->isColorBarsVisible();
	state.ifTimeLabel = myVTK->isLabelVisible();

	state.cameraProjection = CAM->GetParallelProjection();
	CAM->GetPosition(state.cameraPosition);
	CAM->GetFocalPoint(state.cameraFocalPoint);
	CAM->GetViewUp(state.cameraViewUp);
	state.cameraParallelScale = CAM->GetParallelScale();
	CAM->GetClippingRange(state.cameraClippingRange);

	if(DATAREADER != 0) state.currec = myVTK->getReader()->getRecordNumber();
	state2 = state;

}


void iAnimator::restoreState()
{

	state = state2;

	if(DATAREADER!=0 && debugMode)
	{
		DATAREADER->loadRecord(state.currec,true);
	}

	if(XSECTION != 0) XSECTION->setPos(state.xsecPos);
	myVTK->showBox(state.ifBoundingBox );
	myVTK->showColorBars(state.ifColorBars);
	myVTK->showLabel(state.ifTimeLabel);

	if(restoreCamera)
	{
		CAM_FUNCTION1(SetParallelProjection,state.cameraProjection);
		CAM_FUNCTION1(SetPosition,state.cameraPosition);
		CAM_FUNCTION1(SetFocalPoint,state.cameraFocalPoint);
		CAM_FUNCTION1(SetViewUp,state.cameraViewUp);
		CAM_FUNCTION1(SetParallelScale,state.cameraParallelScale);
//		CAM_FUNCTION1(SetClippingRange,state.cameraClippingRange);
	}
}


void iAnimator::resetState()
{
	state.mode = 0;
	state.nframes = 1;
	state.dphi = 0.0;
	state.dtheta = 0.0;
	state.dscale = 1.0;
	state.droll = 0.0;
	state.flybySpeed = 0.01;
	state.slideSpeed = 0.0;
	state.nBlendedFrames = 0;
	state.nTransitionFrames = 0;
	state.titlePageFile = iString("");
	state.titlePageNumFrames = 0;
	state.titlePageNumBlendedFrames = 0;
	state.logoFile = iString("");
	state.logoOpacity = 0.5;
	state.logoPosition = 0;
}


void iAnimator::copyState(iAnimator *anim)
{
	state2 = anim->state;
	restoreCamera = anim->restoreCamera;
	this->restoreState();
}


void iAnimator::setMode(int ma)
{ 
#ifndef I_VTK_VERSION_40
	if(ma>=0 && ma<=4) 
	{
		state.mode = ma; 
		if(ma == 4) 
		{
			cameraPath->On(); 
			if(useFocalPath) focalPath->On();
		}
		else 
		{
			cameraPath->Off();
			if(useFocalPath) focalPath->Off();
		}
	}
#else
	if(ma>=0 && ma<=3) state.mode = ma; 
#endif
}


bool iAnimator::setLogoFile(iString s)
{
	bool ok = iVTK::loadImageFromFile(s,logoImage);
	if(ok) state.logoFile = s; else 
	{ 
		state.logoFile = "";
		logoImage->Initialize();
	}
	return ok;
}


bool iAnimator::setTitlePageFile(iString s)
{
	bool ok = iVTK::loadImageFromFile(s,titlePageImage);
	if(ok) state.titlePageFile = s; else 
	{
		state.titlePageFile = "";
		titlePageImage->Initialize();
	}
	return ok;
}


int iAnimator::step(bool dumpImage)
{
	//
	//  Return codes
	//
	//		 0:		all is ok
	//		-1:		cannot perform off-screen rendering
	//		-2:		no renderer
	//		-3:		no data reader
	//		-10:	file is not animatable
	//		-9:		last file read
	//
	static float Pi = 3.1415927;
	vtkFloat xc[3], x0[3];
	float v0[3], r0[3], r1[3], vA[3], vB[3];
	int i, ret;

	newrec = false;
		
	if(!started)
	{
		
		if(REN == NULL) return (returnCode = -2);
		if(DATAREADER == NULL) return (returnCode = -3);
		if(!DATAREADER->isFileAnimatable()) return (returnCode = -10);
		myVTK->justifyLabelLeft(true);
		
		if(state.mode == 2)
		{
			dphl0 = state.dphi;
			dthl0 = state.dtheta;
			r = 0.0;
			ramp = RAN_N;
		}
		
		if(state.mode == 3)
		{
			for(i=0; i<3; i++)
			{
				xc1[i] = 0.5*RAN_U;
				xc2[i] = 0.5*RAN_U;
			}
			CAM_FUNCTION1(SetParallelProjection,0);
			CAM_FUNCTION1(GetPosition,x);
			if(state.cameraProjection == 1)
			{
				for(i=0; i<3; i++) x[i] = 0.5*x[i];
				CAM_FUNCTION1(SetPosition,x);
			}
			else
			{
				CAM_FUNCTION1(GetFocalPoint,xc1);
			}
			CAM_FUNCTION1(SetFocalPoint,xc1);
			float d = LEN(x);
			v[0] = d*0.5*RAN_U;
			v[1] = d*0.5*RAN_U;
			v[2] = 0.0;
			t = 0.0;
			dt0 = 0.1*Pi;
			dt = dt0;
		}
		
#ifndef I_VTK_VERSION_40
		if(state.mode == 4)
		{
			cameraPath->GetPolyData(pathData);
			cameraPath->Off();
			if(useFocalPath) 
			{
				focalPath->GetPolyData(pathData2);
				focalPath->Off();
			}
			pathStep = 0;
		}	
#endif
		
		if(state.titlePageFile.isEmpty() || debugMode)
		{
			curframe = totframe = 0;
			doingTitlePage = false;
		}
		else
		{
			curframe = totframe = -state.titlePageNumFrames;
			doingTitlePage = true;
		}
		prevrec = -1;
		currec = DATAREADER->getRecordNumber();
		started = true;
		ret = 0;
		
	} 

	if(curframe == state.nframes)
	{
		ret = DATAREADER->loadNextRecord(0,debugMode);
		myVTK->updateLabel();
		newrec = true;
		prevrec = currec;
		currec = DATAREADER->getRecordNumber();
		curframe = 0;
	} 
	else ret = 0;
		
	if(ret == -9) 
	{
		this->reset();
		if(useScript) returnCode = ret; else returnCode = 0;
		return ret;
	}

	if(ret != 0) 
	{
#ifndef I_VTK_VERSION_40
		if(state.mode == 4)	
		{
			cameraPath->On();
			if(useFocalPath) focalPath->On();
		}
#endif
		return (returnCode = ret);
	}

	curframe++;
	totframe++;

	if(totframe > 0)
	{
		
		if(fabs(state.slideSpeed)>1.0e-30 && XSECTION!=0)
		{
			double p = XSECTION->getPos();
			p = p + state.slideSpeed;
			bool ret = XSECTION->setPos(p);
			if(p<-1.0 || ret) state.slideSpeed = -state.slideSpeed;
		}
		//
		//	Add transformations for rotate & tumble
		//	
		if(state.mode==1 || state.mode==2)
		{
			
			CAM_FUNCTION1(Azimuth,-state.dphi);
			CAM_FUNCTION1(Elevation,-state.dtheta);
			CAM_FUNCTION1(Zoom,state.dscale);
			CAM_FUNCTION1(Roll,state.droll);
			CAM_FUNCTION0(OrthogonalizeViewUp);
			
			if(state.mode == 2)
			{
				r = r + randstep;
				float cr = cos(r*ramp);
				float sr = sin(r*ramp);
				state.dphi =  dphl0*cr + dthl0*sr;
				state.dtheta =  dthl0*cr - dphl0*sr;
				if(r > 1.0)
				{
					r = 0.0;
					ramp = RAN_N;
					dphl0 =  state.dphi;
					dthl0 =  state.dtheta;
				}
			}
		}
		//
		//  Add transformations for flyby
		//
		if(state.mode == 3)
		{
			
			for(i=0; i<3; i++)
			{
				xc[i] = xc1[i] + (xc2[i]-xc1[i])*t;
				x0[i] = x[i];
				v0[i] = v[i];
			}
			
			CAM_FUNCTION1(SetFocalPoint,xc);
			
REDO:
			
			for(i=0; i<3; i++)
			{
				r0[i] = x0[i] - xc[i];
				vA[i] = r0[i];
				vB[i] = v0[i];
			}
			
			float cot = cos(dt);
			float sot = sin(dt);
			for(i=0; i<3; i++) r1[i] = vA[i]*cot + vB[i]*sot - r0[i];
			
			float d0 = 0.0, d1 = 0.0;
			for(i=0; i<3; i++)
			{
				d0 = d0 + r0[i]*r0[i];
				d1 = d1 + r1[i]*r1[i];
			}
			d1 = sqrt(d1/d0);
			if(d1>state.flybySpeed && dt>0.001*dt0)
			{
				dt = 0.5*dt;
				goto REDO;
			}
			
			if(d1 < 0.2*state.flybySpeed) dt = 1.5*dt;
			
			for(i=0; i<3; i++)
			{
				v[i] = vB[i]*cot-vA[i]*sot;
				x[i] = r0[i] + r1[i] + xc[i];
			}
			
			CAM_FUNCTION1(SetPosition,x);
			CAM_FUNCTION0(OrthogonalizeViewUp);
			t = t + dt;
			
			if(t > 1.0)
			{
				t = 0.0;
				for(i=0; i<3; i++)
				{
					xc1[i] = xc2[i];
					xc2[i] = 0.5*RAN_U;
				}
			}
			
		}
		
		//
		//  Camera path
		//
#ifndef I_VTK_VERSION_40
		if(state.mode == 4)
		{
			//
			//  Are we done?
			//
			if(pathStep == pathData->GetNumberOfPoints())
			{
				if(this->isPathLoop())
				{
					pathStep = 1;
				}
				else
				{
					this->reset();
					cameraPath->On();
					if(useFocalPath) focalPath->On();
					return (returnCode = -9);
				}
			}
			//
			//  Position the camera
			//
			this->positionCameraOnThePath(pathStep);
			//
			//  Update step counter
			//
			pathStep++;
		}	
#endif
		
		if(myVTK->isBroadcastMouseEventsOn() && iVTKWindow::getMaxWindowIndex()>0)
		{ 
			for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->render(true);
		} 
		else
		{
			myVTK->render(true); 
		}

	}
	//
	//  Image data holder
	//
	vtkImageData *image = 0;
	//
	//  Transition effects
	//
	bool doTransitionFrames = dumpImage && totframe>0 && prevrec>0 && curframe<=state.nTransitionFrames && state.nTransitionFrames>0 && !debugMode;
	if(doTransitionFrames)
	{
		//
		//  Objects for transition effects (image blending)
		//
		vtkImageBlend* iblend = vtkImageBlend::New();
		if(iblend == 0) reportNullPointer(9503);
		vtkImageData* image1 = vtkImageData::New();
		if(image1 == 0) reportNullPointer(9504);
		vtkImageData* image2 = vtkImageData::New();
		if(image2 == 0) reportNullPointer(9505);
		
		if(myVTK->createImageData(image) == 0)
		{
			image1->DeepCopy(image);
			ret = DATAREADER->loadRecord(prevrec,false);
			if(ret == 0)
			{
				myVTK->updateLabel();
				if(myVTK->createImageData(image) == 0)
				{
					image2->DeepCopy(image);
					
					iblend->SetInput(0,image2);
					iblend->SetInput(1,image1);
					iblend->SetOpacity(1,(double)curframe/state.nBlendedFrames);	
					
					iblend->Update();
					image->DeepCopy(iblend->GetOutput());

				}
				else ret = 1;
			}
			else ret = 1;
		}

		iblend->Delete();
		image1->Delete();
		image2->Delete();

		ret = DATAREADER->loadFileSet(currec,false);	

	}

	//
	//  Blending of images
	//
	bool doBlendedFrames = dumpImage && totframe >0 && state.nBlendedFrames>0 && !debugMode;
	if(doBlendedFrames)
	{
		vtkImageBlend* iblend = vtkImageBlend::New();
		if(iblend == 0) reportNullPointer(9501);

		if(myVTK->createImageData(image) == 0)
		{
			//
			//  Update the image list
			//
			if(numBlenderBase < state.nBlendedFrames)
			{
				vtkImageData **tmp = new vtkImageData*[state.nBlendedFrames];
				for(i=0; i<numBlenderBase; i++) tmp[i] = blenderBase[i];
				for(i=numBlenderBase; i<state.nBlendedFrames; i++)
				{
					tmp[i] = vtkImageData::New();
					tmp[i]->DeepCopy(image);
				}
				delete [] blenderBase;
				blenderBase = tmp;
				numBlenderBase = state.nBlendedFrames;
			}
			else if(numBlenderBase > state.nBlendedFrames)
			{
				vtkImageData **tmp = new vtkImageData*[state.nBlendedFrames];
				for(i=0; i<numBlenderBase-state.nBlendedFrames; i++) blenderBase[i]->Delete();
				for(i=0; i<state.nBlendedFrames; i++) tmp[i] = blenderBase[i+numBlenderBase-state.nBlendedFrames];
				delete [] blenderBase;
				blenderBase = tmp;
				numBlenderBase = state.nBlendedFrames;
			}

			blenderBase[0]->Delete();
			for(i=0; i<numBlenderBase-1; i++) blenderBase[i] = blenderBase[i+1];
			blenderBase[numBlenderBase-1] = vtkImageData::New();
			blenderBase[numBlenderBase-1]->DeepCopy(image);

			float f = 2.0/(numBlenderBase*(numBlenderBase+1));
			for(i=0; i<numBlenderBase; i++) 
			{
				iblend->SetInput(i,blenderBase[i]);
				iblend->SetOpacity(i,f*(i+1));
			}
			iblend->SetBlendModeToCompound();
			iblend->Update();
			image->DeepCopy(iblend->GetOutput());

		}
		else ret = 1;

		iblend->Delete();

	}
	
	//
	//  Title page
	//
	vtkImageData *tmp1 = 0, *tmp2 = 0;
	if(dumpImage && doingTitlePage)
	{
		tmp1 = vtkImageData::New();
		if(tmp1 == 0) reportNullPointer(9503);
		tmp1->DeepCopy(titlePageImage);
		iVTK::scaleImage(tmp1,myVTK->getFullImageWidth(),myVTK->getFullImageHeight());

		if(totframe <= 0)
		{
			//
			//  Plain title page
			//
			image = tmp1;
		}
		else if(state.titlePageNumBlendedFrames > 0)
		{
			//
			//  Dissolve it; image already contains the correct image
			//
			if(image == 0) // create the image to blend the title page into
			{
				if(myVTK->createImageData(image) != 0) ret = 1;
			}
			if(ret == 0)
			{
				vtkImageBlend* iblend = vtkImageBlend::New();
				if(iblend == 0) reportNullPointer(9503);
				iblend->SetInput(0,tmp1);
				iblend->SetInput(1,image);
				iblend->SetOpacity(1,(double)totframe/state.titlePageNumBlendedFrames);	
				iblend->Update();
				image->DeepCopy(iblend->GetOutput());
				iblend->Delete();
			}
		}
		if(totframe > state.titlePageNumBlendedFrames) doingTitlePage = false;
				
	}

	//
	//  Logo
	//
	if(dumpImage && !debugMode && !state.logoFile.isEmpty())
	{
		if(image == 0) // create the image to place a logo on
		{
			if(myVTK->createImageData(image) != 0) ret = 1;
		}

		if(ret == 0)  // we do have the image
		{
			int dimsImage[3], dimsLogo[3];
			image->GetDimensions(dimsImage);
			logoImage->GetDimensions(dimsLogo);
			//
			//  If the logo is more than 20% of the image, scale it down.
			//  Use tmp2 as a temp storage, since tmp1 may be image (while displaying the Title Page)
			//
			tmp2 = vtkImageData::New();
			if(tmp2 == 0) reportNullPointer(9503);
			tmp2->DeepCopy(logoImage);
			if(dimsLogo[0]>dimsImage[0]/5 || dimsLogo[1]>dimsImage[1]/5)
			{
				iVTK::scaleImage(tmp2,dimsImage[0]/5,dimsImage[1]/5);
				tmp2->GetDimensions(dimsLogo);
			}

			if(dimsLogo[0]>=2 && dimsLogo[1]>=2)
			{
				//
				//  tmp2 is now the proper logo image
				//
				int k, j, i, ioff, joff;
				long lImage, lLogo;
				unsigned char *iPtr = (unsigned char *)image->GetScalarPointer();
				unsigned char *lPtr = (unsigned char *)tmp2->GetScalarPointer();
				//
				//  totally transparent colors
				//
				unsigned char r0 = lPtr[0];
				unsigned char g0 = lPtr[1];
				unsigned char b0 = lPtr[2];
				//
				//  Where do we place the logo?
				//
				ioff = dimsLogo[0]/5;
				joff = dimsLogo[1]/5;
				switch(state.logoPosition)
				{
				case 1:
					{
						//  upper right corner 
						ioff = dimsImage[0] - dimsLogo[0] - ioff;
						joff = dimsImage[1] - dimsLogo[1] - joff;
						break;
					}
				case 2:
					{
						//  lower left right corner 
						break;
					}
				case 3:
					{
						//  lower right corner 
						ioff = dimsImage[0] - dimsLogo[0] - ioff;
						break;
					}
				default:
					{
						//  upper left corner - the default choice
						joff = dimsImage[1] - dimsLogo[1] - joff;
						break;
					}
				}
				
				float wI = 1.0 - state.logoOpacity;
				float wL = state.logoOpacity;
				for(j=0; j<dimsLogo[1]; j++)
				{
					for(i=0; i<dimsLogo[0]; i++)
					{
						lImage = 3*(i+ioff+dimsImage[0]*(j+joff));
						lLogo = 3*(i+dimsLogo[0]*j);
						if(lPtr[lLogo+0]!=r0 || lPtr[lLogo+1]!=g0 || lPtr[lLogo+2]!=b0)
						{
							for(k=0; k<3; k++) // this is not because I am lazy, it is the fastest way
							{
								iPtr[lImage+k] = (unsigned char)round(wI*iPtr[lImage+k]+wL*lPtr[lLogo+k]);
							}
						}
					}
				}
			}

		}
		else ret = 1;
	}

	if(debugMode) ret = 0; else if(dumpImage) ret = -(myVTK->dumpImage(1,image));
	
	if(tmp1 != 0) tmp1->Delete();
	if(tmp2 != 0) tmp2->Delete();

	return (returnCode = ret);
	
}


void iAnimator::positionCameraOnThePath(int pathStep)
{
#ifndef I_VTK_VERSION_40
	vtkFloat q, x1[3], x2[3], dx[3], vu[3];
	int i, i1, i2;
	double d;

	if(pathStep<0 || pathStep>=pathData->GetNumberOfPoints()) return;

	if(pathStep == 0)
	{
		//
		//  First frame
		//
		i1 = 0;
	}
	else
	{
		i1 = pathStep - 1;
	}
	
	pathData->GetPoints()->GetPoint(i1,x1);
	pathData->GetPoints()->GetPoint(i1,x2);
	i2 = i1;
	while(i2 < pathData->GetNumberOfPoints()-1)
	{
		i2++;
		pathData->GetPoints()->GetPoint(i2,x2);
		for(i=0; i<3; i++) dx[i] = x2[i] - x1[i];
		if(vtkMath::Norm(dx) > 0) break;
	}
	vtkMath::Normalize(dx);
	
	if(pathStep == 0) 
	{
		for(i=0; i<3; i++) x2[i] = x1[i]; 
		i2 = 0;
	}

	d = CAM->GetDistance();
	CAM->GetViewUp(vu);
	q = vtkMath::Dot(dx,vu);
	for(i=0; i<3; i++) vu[i] -= q*dx[i];
	if(vtkMath::Norm(vu) > 0.0)
	{
		vtkMath::Normalize(vu);
		CAM->SetViewUp(vu);
	}

	CAM->SetPosition(x2);

	if(useFocalPath)
	{
		CAM->SetFocalPoint(pathData2->GetPoints()->GetPoint(i2));
	}
	else
	{
		for(i=0; i<3; i++) dx[i] = x2[i] + dx[i]*d;
		CAM->SetFocalPoint(dx);
	}
#endif
}
//
//  Camera path functions
//
int iAnimator::getNumberOfPathHandles()
{ 
#ifndef I_VTK_VERSION_40
	return cameraPath->GetNumberOfHandles(); 
#else
	return 2;
#endif
}


int iAnimator::getNumberOfPathSteps()
{ 
#ifndef I_VTK_VERSION_40
	return cameraPath->GetResolution(); 
#else
	return 1;
#endif
}


bool iAnimator::isPathLoop()
{ 
#ifndef I_VTK_VERSION_40
	return cameraPath->GetClosed()==1; 
#else
	return false;
#endif
}


bool iAnimator::isFocalPointPathUsed()
{ 
	return useFocalPath; 
}


bool iAnimator::isFocalPointPathToPoint()
{ 
#ifndef I_VTK_VERSION_40
	return animatorEventObserver->focalPathToPoint; 
#else
	return false;
#endif
}


void iAnimator::setNumberOfPathHandles(int v)
{
#ifndef I_VTK_VERSION_40
	int r = cameraPath->GetResolution();
	if(v < cameraPath->GetNumberOfHandles())
	{
		cameraPath->SetNumberOfHandles(v);
		cameraPath->SetResolution(r);
		focalPath->SetNumberOfHandles(v);
		focalPath->SetResolution(r);
	}
	else
	{
		cameraPath->SetNumberOfHandles(v);
		focalPath->SetNumberOfHandles(v);
	}
#endif
}


void iAnimator::setNumberOfPathSteps(int n)
{
#ifndef I_VTK_VERSION_40
	cameraPath->SetResolution(n);
	focalPath->SetResolution(n);
#endif
}


void iAnimator::setPathLoop(bool v)
{
#ifndef I_VTK_VERSION_40
	if(v) 
	{
		cameraPath->ClosedOn(); 
		focalPath->ClosedOn(); 
	}
	else 
	{
		cameraPath->ClosedOff();
		focalPath->ClosedOff();
	}
	if(useFocalPath) focalPath->InvokeEvent(vtkCommand::InteractionEvent);
#endif
}


void iAnimator::useFocalPointPath(bool s)
{
#ifndef I_VTK_VERSION_40
	if(s != useFocalPath)
	{
		useFocalPath = s;
		focalPath->SetEnabled((int)s & cameraPath->GetEnabled());
		if(s)
		{
			animatorEventObserver->setFocalPointPath(focalPath);
			focalPath->InvokeEvent(vtkCommand::InteractionEvent);
		}
		else
		{
			animatorEventObserver->setFocalPointPath(0);
		}
	}
#endif
}


void iAnimator::setFocalPointPathToPoint(bool s)
{ 
#ifndef I_VTK_VERSION_40
	if(useFocalPath)
	{
		animatorEventObserver->focalPathToPoint = s;
		if(!s)
		{
			int i;
			vtkFloat x[3], sf;
			focalPath->GetHandlePosition(0,x);
			sf = 0.6/(focalPath->GetNumberOfHandles()-1);
			for(i=1; i<focalPath->GetNumberOfHandles(); i++) 
			{
				x[0] -= sf;
				focalPath->SetHandlePosition(i,x);
			}
		}
		focalPath->InvokeEvent(vtkCommand::InteractionEvent);
	}
#endif
}


void iAnimator::setCameraPathColor(float r, float g, float b)
{
#ifndef I_VTK_VERSION_40
	cameraPath->GetLineProperty()->SetColor(r,g,b);
#endif
}


bool iAnimator::isCameraPathValid()
{
#ifndef I_VTK_VERSION_40
	if(!useFocalPath) return true;

	int i;

	cameraPath->GetPolyData(pathData);
	focalPath->GetPolyData(pathData2);

	for(i=0; i<pathData->GetNumberOfPoints(); i++)
	{
		if(vtkMath::Distance2BetweenPoints(pathData->GetPoints()->GetPoint(i),pathData2->GetPoints()->GetPoint(i)) < 1.0e-8) return false;
	}
#endif
	return true;
}


void iAnimator::playCameraPathDemo()
{
#ifndef I_VTK_VERSION_40
	if(state.mode == 4)
	{
		int i;
		float t;
		vtkFloat camPosition[3], camFocalPoint[3], camViewUp[3];

		cameraPath->GetPolyData(pathData);
		if(useFocalPath) focalPath->GetPolyData(pathData2);
		cameraPath->GetHandleProperty()->SetOpacity(0.5);
		//
		//  Save the camera parameters
		//
		CAM->GetPosition(camPosition);
		CAM->GetFocalPoint(camFocalPoint);
		CAM->GetViewUp(camViewUp);
		//
		//  Measure the render time
		//
		myVTK->render(true); 
		t = REN->GetLastRenderTimeInSeconds();

		if(t < 0.05) t = 0.05 - t; else t = 0.0;

		//
		//  Fly along the main path
		//
		for(i=0; i<pathData->GetNumberOfPoints(); i++)
		{
			positionCameraOnThePath(i);
			if(t > 0.0) iSleep(round(1000.0*t));
			myVTK->render(true); 
		}

		//
		//  Restore the camera parameters
		//
		CAM->SetPosition(camPosition);
		CAM->SetFocalPoint(camFocalPoint);
		CAM->SetViewUp(camViewUp);

		cameraPath->GetHandleProperty()->SetOpacity(1.0);

		myVTK->render(true); 

	}
#endif
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iAnimator::packState(iString &s)
{

	s = "";

	this->packValue(s,"iAnimator::debugMode",debugMode);
	this->packValue(s,"iAnimator::useScript",useScript);
#ifndef I_VTK_VERSION_40
	this->packValue(s,"iAnimator::useFocalPath",useFocalPath);
	this->packValue(s,"iAnimator::cameraPath->GetClosed()",(cameraPath->GetClosed()!=0));
#endif
	this->packValue(s,"iAnimator::restoreCamera",restoreCamera);
	this->packValue(s,"iAnimator::inheritSettings",inheritSettings);

	this->packValue(s,"iAnimator::state.mode",state.mode);
	this->packValue(s,"iAnimator::state.nframes",state.nframes);
	this->packValue(s,"iAnimator::state.nBlendedFrames",state.nBlendedFrames);
	this->packValue(s,"iAnimator::state.nTransitionFrames",state.nTransitionFrames);

	this->packValue(s,"iAnimator::state.dphi",state.dphi);
	this->packValue(s,"iAnimator::state.dtheta",state.dtheta);
	this->packValue(s,"iAnimator::state.droll",state.droll);
	this->packValue(s,"iAnimator::state.dscale",state.dscale);
	this->packValue(s,"iAnimator::state.slideSpeed",state.slideSpeed);
	this->packValue(s,"iAnimator::state.flybySpeed",state.flybySpeed);

	this->packValue(s,"iAnimator::state.titlePageFile",state.titlePageFile);
	this->packValue(s,"iAnimator::state.titlePageNumFrames",state.titlePageNumFrames);
	this->packValue(s,"iAnimator::state.titlePageNumBlendedFrames",state.titlePageNumBlendedFrames);
	this->packValue(s,"iAnimator::state.logoFile",state.logoFile);
	this->packValue(s,"iAnimator::state.logoOpacity",state.logoOpacity);
	this->packValue(s,"iAnimator::state.logoPosition",state.logoPosition);

#ifndef I_VTK_VERSION_40
	this->packValue(s,"iAnimator::getNumberOfPathHandles()",this->getNumberOfPathHandles());
	this->packValue(s,"iAnimator::getNumberOfPathSteps()",this->getNumberOfPathSteps());

	int n = cameraPath->GetNumberOfHandles();
	float *x1 = new float[n];
	float *x2 = new float[n];
	float *x3 = new float[n];

	int i;
	vtkFloat *x;
	for(i=0; i<cameraPath->GetNumberOfHandles(); i++)
	{
		x = cameraPath->GetHandlePosition(i);
		x1[i] = x[0];
		x2[i] = x[1];
		x3[i] = x[2];
	}

	this->packValue(s,"iAnimator::cameraPath1",x1,n);
	this->packValue(s,"iAnimator::cameraPath2",x2,n);
	this->packValue(s,"iAnimator::cameraPath3",x3,n);
#endif

}


void iAnimator::unpackState(iString s)
{
	int i; bool b; float f; iString ss;

	if(this->unpackValue(s,"iAnimator::debugMode",b)) this->setDebugMode(b);
	if(this->unpackValue(s,"iAnimator::useScript",b)) this->setUseScript(b);
#ifndef I_VTK_VERSION_40
	if(this->unpackValue(s,"iAnimator::useFocalPath",b)) this->useFocalPointPath(b);
	if(this->unpackValue(s,"iAnimator::cameraPath->GetClosed()",b)) this->setPathLoop(b);
#endif
	if(this->unpackValue(s,"iAnimator::restoreCamera",b)) this->setRestoreCamera(b);
	if(this->unpackValue(s,"iAnimator::inheritSettings",b)) this->setInheritSettings(b);

	if(this->unpackValue(s,"iAnimator::state.nframes",i)) this->setNframes(i);
	if(this->unpackValue(s,"iAnimator::state.nBlendedFrames",i)) this->setNumBlendedFrames(i);
	if(this->unpackValue(s,"iAnimator::state.nTransitionFrames",i)) this->setNumTransitionFrames(i);

	if(this->unpackValue(s,"iAnimator::state.dphi",f)) this->setDPhi(f);
	if(this->unpackValue(s,"iAnimator::state.dtheta",f)) this->setDTheta(f);
	if(this->unpackValue(s,"iAnimator::state.droll",f)) this->setDRoll(f);
	if(this->unpackValue(s,"iAnimator::state.dscale",f)) this->setDScale(f);
	if(this->unpackValue(s,"iAnimator::state.slideSpeed",f)) this->setSlideSpeed(f);
	if(this->unpackValue(s,"iAnimator::state.flybySpeed",f)) this->setFlybySpeed(f);

	if(this->unpackValue(s,"iAnimator::state.titlePageFile",ss)) this->setTitlePageFile(ss);
	if(this->unpackValue(s,"iAnimator::state.titlePageNumFrames",i)) this->setNumTitlePageFrames(i);
	if(this->unpackValue(s,"iAnimator::state.titlePageNumBlendedFrames",i)) this->setNumTitlePageBlendedFrames(i);
	if(this->unpackValue(s,"iAnimator::state.logoFile",ss)) this->setLogoFile(ss);
	if(this->unpackValue(s,"iAnimator::state.logoOpacity",f)) this->setLogoOpacity(f);
	if(this->unpackValue(s,"iAnimator::state.logoPosition",i)) this->setLogoPosition(i);

#ifndef I_VTK_VERSION_40
	if(this->unpackValue(s,"iAnimator::getNumberOfPathHandles()",i)) this->setNumberOfPathHandles(i);
	if(this->unpackValue(s,"iAnimator::getNumberOfPathSteps()",i)) this->setNumberOfPathSteps(i);

	int n = cameraPath->GetNumberOfHandles();
	float *x1 = new float[n];
	float *x2 = new float[n];
	float *x3 = new float[n];

	if(this->unpackValue(s,"iAnimator::cameraPath1",x1,n) && this->unpackValue(s,"iAnimator::cameraPath2",x2,n) && this->unpackValue(s,"iAnimator::cameraPath3",x3,n))
	{
		vtkFloat x[3];
		for(i=0; i<cameraPath->GetNumberOfHandles(); i++)
		{
			x[0] = x1[i];
			x[1] = x2[i];
			x[2] = x3[i];
			cameraPath->SetHandlePosition(i,x);
		}
	}
#endif

	if(this->unpackValue(s,"iAnimator::state.mode",i)) this->setMode(i);

}


