/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "ianimatorscript.h"

#include "ianimator.h"
#include "ivtk.h"
#include "iqt.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iobjectfactory.h"
#include "ianimator.h"
#include "idatareader.h"
#include "ilimits.h"
#include "iparticlessplitter.h"
#include "ienvironment.h"

#include "imarker.h"
#include "iparticles.h"
#include "isurface.h"
#include "itensor.h"
#include "ivector.h"
#include "ivolume.h"
#include "ixsection.h"

#include <vtkMath.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>


using namespace iScriptType;


iValue* iAnimatorScript::valIS = NULL;
iValue* iAnimatorScript::valFS = NULL;
iValue* iAnimatorScript::valFP = NULL;
iValue* iAnimatorScript::valVA3 = NULL;
int iAnimatorScript::counter = 0;


#define curVTK        iVTKWindow::getCurrentWindow()
#define curReader     iVTKWindow::getCurrentWindow()->getReader()
#define curAnimator   iVTKWindow::getCurrentWindow()->getAnimator()
#define curRenderer   iVTKWindow::getCurrentWindow()->getRenderer()
#define curCamera     iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera()
#define curLimits     iVTKWindow::getCurrentWindow()->getReader()->getCurrentLimits()


#define UPDATE_VALUE_REL(v0,rel,v,rhs) \
{ \
	if(!v0->getValue(v)) \
	{ \
		*pErrorMessage = "Invalid value."; \
		return; \
	} \
	if(rel & 1) { if(rel & 4) v = (rhs) - v; else v += (rhs); } else if(rel & 2) v *= (rhs); \
}


#define UPDATE_VALUE_ABS(v0,v) \
{ \
	if(!v0->getValue(v)) \
	{ \
		*pErrorMessage = "Invalid value."; \
		return; \
	} \
}


#define UPDATE_POSITIONVALUE(v0,rel,n,v,rhs) \
{ \
	float v1[n]; \
	if(!v0->getValue(n,v1)) \
	{ \
		*pErrorMessage = "Invalid array value."; \
		return; \
	} \
	{ \
		int i; \
		for(i=0; i<n; i++) v[i] = v1[i]; \
		if(rel & 1) { if(rel & 4) { for(i=0; i<n; i++) v[i] = rhs[i] - v[i]; } else { for(i=0; i<n; i++) v[i] += rhs[i]; } } else if(rel & 2) { for(i=0; i<n; i++) v[i] *= rhs[i]; } \
	} \
}


void reportNullPointer(int ec);


iAnimatorScript* iAnimatorScript::New(iAnimator *m)
{
	return (iAnimatorScript *)iObjectFactory::createAnimatorScript(m);
}


iAnimatorScript::iAnimatorScript(iAnimator *m) : iScript()
{

	myAnimator = m;

	//
	// Init TMP values
	//
	counter++;
	if(counter == 1)
	{
		valIS = iValue::New("tmpIS",ByValue,0);
		valFS = iValue::New("tmpFS",ByValue,0.0f);
		valFP = iValue::New("tmpFP",ByValue,(float *)NULL);
		valVA3 = this->transformSingleTokenToValue("(0.0,0.0,0.0)");
	}
	//
	//  Use only float arrays
	//
	arraysAreFloat = true;

	checkAbort = iAnimatorScript::myCheckAbort;
	showLine = iAnimatorScript::myShowLine;
	
	//
	//  Dummy words
	//
	this->createDummyWord("to"); 
	this->createDummyWord("then"); 
	this->createDummyWord("set"); 

	//
	//  Alias words (mostly for backward compatibility)
	//
	this->createAliasWord("x-section-speed","xsection-speed"); 
	this->createAliasWord("fly-by-speed","flyby-speed"); 

	//
	//  Prefix words
	//
	this->createPrefixWord("var"); 
	this->createPrefixWord("for"); 
	this->createPrefixWord("load"); 
	//
	//  Parameter words
	//
	this->createParameterWord(iValue::New("static",ByValue,0));
	this->createParameterWord(iValue::New("rotate-scale",ByValue,1));
	this->createParameterWord(iValue::New("tumble",ByValue,2));
	this->createParameterWord(iValue::New("fly-by",ByValue,3));
	this->createParameterWord(iValue::New("camera-path",ByValue,4));
	this->createParameterWord(iValue::New("next",ByValue,-1));
	this->createParameterWord(iValue::New("parallel",ByValue,0));
	this->createParameterWord(iValue::New("perspective",ByValue,1));
	this->createParameterWord(iValue::New("hidden",ByValue,false));
	this->createParameterWord(iValue::New("visible",ByValue,true));
	this->createParameterWord(iValue::New("current",ByValue,-1));
	this->createParameterWord(iValue::New("marker",ByFunction,0,iValue::New("",ByValue,0),NULL,iAnimatorScript::GetMarkerPosition));
	this->createParameterWord(iValue::New("upper-left-corner",ByValue,0));
	this->createParameterWord(iValue::New("upper-right-corner",ByValue,1));
	this->createParameterWord(iValue::New("lower-left-corner",ByValue,2));
	this->createParameterWord(iValue::New("lower-right-corner",ByValue,3));

	//
	//  Command words
	//

	//
	//  Declarations
	//
	this->createCommandWord(iOperation::New(this,"var","int",1,"",RelativeAddMult,iValue::New("",ByValue,0),(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"var","float",1,"",RelativeAddMult,iValue::New("",ByValue,0.0f),(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"var","bool",1,"",Absolute,iValue::New("",ByValue,false),(unsigned char)1));
	//
	//  Flow control operations
	//
	this->createCommandWord(iOperation::New(this,"","loop",1,Entry,NULL,iValue::New("",ByValue,0),1,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"for","$variable",1,Entry,iValue::New("",ByValue,0),iValue::New("",ByValue,0),1,(unsigned char)2));
	this->createCommandWord(iOperation::New(this,"","end",1,Exit,NULL,NULL,1,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","if",1,Entry,NULL,iValue::New("",ByValue,false),2,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","else",1,EntryExit,NULL,NULL,2,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","endif",1,Exit,NULL,NULL,2,(unsigned char)1));

	//
	//  Assignment operations hidden as statements
	//
	this->createCommandWord(iOperation::New(this,"","render",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::Render,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","render-all",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::RenderAll,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","render-set",1,Absolute,iValue::New("",ByValue,0,(iValue*)NULL,(iValue**)NULL),iAnimatorScript::MultiRender,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","load",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::Load,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"","reset",1,iAnimatorScript::Reset,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"load","mesh-file",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::LoadMeshFile,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"load","particle-file",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::LoadPartFile,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"load","vector-file",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::LoadVectFile,(unsigned char)1));
	this->createCommandWord(iOperation::New(this,"load","tensor-file",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::LoadTensFile,(unsigned char)1));

	//
	//  Genuine assignment operations to local variables
	//
	this->createCommandWord(iOperation::New(this,"","$variable",1,RelativeAddMult,iValue::New("",ByValue,false),NULL,(unsigned char)0));
	this->createCommandWord(iOperation::New(this,"","$variable",1,RelativeAddMult,iValue::New("",ByValue,0),NULL,(unsigned char)0));
	this->createCommandWord(iOperation::New(this,"","$variable",1,RelativeAddMult,iValue::New("",ByValue,0.0f),NULL,(unsigned char)0));
	this->createCommandWord(iOperation::New(this,"","$variable",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,3,(iValue*)NULL,(iValue**)NULL),NULL,(unsigned char)0));

	//
	//  Genuine assignment operations to pre-defined variables
	//
	this->createCommandWord(iOperation::New(this,"","style",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::Style,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","frames-per-file",1,RelativeAddMult,iValue::New("",ByValue,0),iAnimatorScript::FramesPerFile,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","blended-frames",1,RelativeAddMult,iValue::New("",ByValue,0),iAnimatorScript::NumBlendedFrames,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","transition-frames",1,RelativeAddMult,iValue::New("",ByValue,0),iAnimatorScript::NumTransitionFrames,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","flyby-speed",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::FlybySpeed,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","rotation-phi",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::RotationPhi,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","rotation-theta",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::RotationTheta,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","rotation-roll",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::RotationRoll,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","scale",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::Scale,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","xsection-speed",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::XsectionSpeed,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","mesh-variable-lower-limit",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::VarLo,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","mesh-variable-upper-limit",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::VarHi,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","particle-attribute-lower-limit",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::AttLo,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","particle-attribute-upper-limit",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::AttHi,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","vector-upper-limit",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::VecHi,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","tensor-upper-limit",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::TenHi,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","camera-view-up",1,RelativeAdd,iValue::New("",ByValue,3,(iValue*)NULL,(iValue**)NULL),iAnimatorScript::CameraViewUp,(unsigned char)5));
	this->createCommandWord(iOperation::New(this,"","camera-scale",1,RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::CameraScale,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","camera-focal-point",1,RelativeAdd,iValue::New("",ByValue,3,(iValue*)NULL,(iValue**)NULL),iAnimatorScript::CameraFocus,(unsigned char)5));
	this->createCommandWord(iOperation::New(this,"","camera-position",1,RelativeAdd,iValue::New("",ByValue,3,(iValue*)NULL,(iValue**)NULL),iAnimatorScript::CameraPosition,(unsigned char)5));
	this->createCommandWord(iOperation::New(this,"","projection",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::Projection,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","time-label",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::TimeLabel,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","time-label-value",1,Absolute,iValue::New("",ByValue,0.0f),iAnimatorScript::TimeLabelValue,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","color-bars",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ColorBars,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","bounding-box",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::BoundingBox,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","title-page-file",1,Absolute,iValue::New("",ByValue,""),iAnimatorScript::TitlePageFile,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","title-page-number-of-frames",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::TitlePageNumFrames,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","title-page-number-of-blended-frames",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::TitlePageNumBlendedFrames,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","logo-file",1,Absolute,iValue::New("",ByValue,""),iAnimatorScript::LogoFile,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","logo-opacity",1,Absolute,iValue::New("",ByValue,0.0f),iAnimatorScript::LogoOpacity,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","logo-position",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::LogoPosition,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","current-window",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::SetCurrentVTK,(unsigned char)4));

	this->createCommandWord(iOperation::New(this,"","surface-var",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,0),iAnimatorScript::SurfaceVar,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","surface-level",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::SurfaceLevel,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","surface-opacity",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::SurfaceOpacity,(unsigned char)4));

	this->createCommandWord(iOperation::New(this,"","xsection-var",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,0),iAnimatorScript::XsectionVar,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","xsection-dir",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,0),iAnimatorScript::XsectionDir,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","xsection-position",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::XsectionPos,(unsigned char)4));

	this->createCommandWord(iOperation::New(this,"","volume-var",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,0),iAnimatorScript::VolumeVar,(unsigned char)4));

	this->createCommandWord(iOperation::New(this,"","particles-opacity",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::ParticlesOpacity,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","particles-downsample-factor",1,Absolute,iValue::New("",ByValue,0),iAnimatorScript::ParticlesDownsampleFactor,(unsigned char)3));

	this->createCommandWord(iOperation::New(this,"","marker-position",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,3,(iValue*)NULL,(iValue**)NULL),iAnimatorScript::MarkerPosition,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","marker-size",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::MarkerSize,(unsigned char)4));
	this->createCommandWord(iOperation::New(this,"","marker-opacity",1,iValue::New("",ByValue,1),RelativeAddMult,iValue::New("",ByValue,0.0f),iAnimatorScript::MarkerOpacity,(unsigned char)4));

	this->createCommandWord(iOperation::New(this,"","surface",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowSurface,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","xsection",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowXsection,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","volume",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowVolume,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","particles",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowParticles,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","vector",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowVector,(unsigned char)3));
	this->createCommandWord(iOperation::New(this,"","tensor",1,Absolute,iValue::New("",ByValue,false),iAnimatorScript::ShowTensor,(unsigned char)3));

	debugMode = false;

	//
	//  Undocumented state assignment
	//
	this->createCommandWord(iOperation::New(this,"","marker-state",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::MarkerState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","particles-state",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::ParticlesState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","surface-state",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::SurfaceState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","tensor-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::TensorState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","vector-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::VectorState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","volume-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::VolumeState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","xsection-state",1,iValue::New("",ByValue,1),Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::XsectionState,(unsigned char)255));

	this->createCommandWord(iOperation::New(this,"","animator-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::AnimatorState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","reader-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::DataReaderState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","limits-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::LimitsState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","splitter-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::ParticlesSplitterState,(unsigned char)255));
	this->createCommandWord(iOperation::New(this,"","vtk-state",1,Absolute,iValue::New("",ByValue,iString("")),iAnimatorScript::VTKState,(unsigned char)255));
}


iAnimatorScript::~iAnimatorScript()
{
	//
	// Delete TMP values
	//
	if(counter == 1)
	{
		valIS->Delete();
		valFS->Delete();
		valFP->Delete();
		valVA3->Delete();
	}
	counter--;

}

//
//  Interrupt animator or do other things
//
bool iAnimatorScript::myCheckAbort(int lineInScript, int entryPointInteration, int entryPointIterationCount, int userCode)
{
	return iQTWindow::getCurrentWindow()->animateBreak(lineInScript,entryPointInteration,entryPointIterationCount,userCode);
}

void iAnimatorScript::myShowLine(int lineInScript)
{
	iQTWindow::getCurrentWindow()->animateShowLine(lineInScript);
}

//
//  Render scene
//
void iAnimatorScript::Render(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int i, v, ret = 0;
	UPDATE_VALUE_ABS(v0,v);

	for(i=0; i<v && ret==0; i++) 
	{
		ret = iVTKWindow::getCurrentWindow()->getAnimator()->step();
		if(iAnimatorScript::myCheckAbort(-1,-1,0,1)) ret = 1; // allow interrupts but do not show lines
	}
	if(ret == -9) *pErrorMessage = "Script has not completed because of lack of data files."; else if(ret != 0) 
	{
		*pErrorMessage = "Error in render operator.";
	}
}

void iAnimatorScript::RenderAll(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int i, k, v, ret = 0, ret1;
	UPDATE_VALUE_ABS(v0,v);

	for(i=0; i<v && ret==0; i++) 
	{
		ret1 = 0;
		for(k=0; ret==0 && k<=iVTKWindow::getMaxWindowIndex(); k++) if(k != iVTKWindow::getCurrentWindowIndex())
		{
			ret1 = iVTKWindow::getWindow(k)->getAnimator()->step(false);
			if(iAnimatorScript::myCheckAbort(-1,-1,0,1)) ret1 = 1; // allow interrupts but do not show lines
			if(ret==0 && ret1!=0) ret = ret1;
		}
		if(ret == 0)
		{
			ret = iVTKWindow::getCurrentWindow()->getAnimator()->step();
			if(iAnimatorScript::myCheckAbort(-1,-1,0,1)) ret = 1; // allow interrupts but do not show lines
		}
	}
	if(ret == -9) *pErrorMessage = "Script has not completed because of lack of data files."; else if(ret != 0) 
	{
		*pErrorMessage = "Error in render operator.";
	}
}

void iAnimatorScript::MultiRender(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	//
	//  Did we get an array?
	//
	if(v0->getType() != IVALUE_VA)
	{
		*pErrorMessage = "Value must be an array."; 
		return;
	}

	int n = ((iValue_VA *)v0)->getSize();
	int *v = new int[n];
	if(v == 0)
	{
		*pErrorMessage = "Unable to allocate memory."; 
		return;
	}

	if(!v0->getValue(n,v)) 
	{
		*pErrorMessage = "Invalid array value."; 
		delete [] v;
		return; 
	} 

	int i, ret = 0;
	for(i=0; i<n; i++) v[i]--;
	for(i=0; ret==0 && i<n; i++) if(v[i]>=0 && v[i]<=iVTKWindow::getMaxWindowIndex() && v[i]!=iVTKWindow::getCurrentWindowIndex())
	{
		ret = iVTKWindow::getWindow(v[i])->getAnimator()->step(false);
		if(iAnimatorScript::myCheckAbort(-1,-1,0,1)) ret = 1; // allow interrupts but do not show lines
	}
	for(i=0; ret==0 && i<n; i++) if(v[i] == iVTKWindow::getCurrentWindowIndex())
	{
		ret = iVTKWindow::getCurrentWindow()->getAnimator()->step();
		if(iAnimatorScript::myCheckAbort(-1,-1,0,1)) ret = 1; // allow interrupts but do not show lines
		break;
	}
	if(ret == -9) *pErrorMessage = "Script has not completed because of lack of data files."; else if(ret != 0) 
	{
		*pErrorMessage = "Error in render operator.";
	}
	delete [] v;
}
//
//  Load sets
//
void iAnimatorScript::Load(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v);

	if(v <= 0)
	{
		if(curReader->loadNextRecord(0,(curAnimator->getScript()->getDebugMode())==0)) curAnimator->resetCurrentFile();
	}
	else
	{
		if(curReader->loadRecord(v,(curAnimator->getScript()->getDebugMode())==0)) curAnimator->resetCurrentFile();
	}
}

//
//  reset Animator settings
//
void iAnimatorScript::Reset(iString *vtkNotUsed(pErrorMessage))
{
	curAnimator->resetState();
	curAnimator->reset();
	curVTK->resetCamera();
}

//
//  Animator settings
//
void iAnimatorScript::Style(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v);
	curAnimator->setMode(v);
	curAnimator->reset();
}

void iAnimatorScript::FramesPerFile(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getNframes());
	curAnimator->setNframes(v);
}

void iAnimatorScript::NumBlendedFrames(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getNumBlendedFrames());
	curAnimator->setNumBlendedFrames(v);
}

void iAnimatorScript::NumTransitionFrames(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getNumTransitionFrames());
	curAnimator->setNumTransitionFrames(v);
}

void iAnimatorScript::FlybySpeed(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getFlybySpeed());
	curAnimator->setFlybySpeed(v);
}

void iAnimatorScript::RotationPhi(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getDPhi());
	curAnimator->setDPhi(v);
}

void iAnimatorScript::RotationTheta(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getDTheta());
	curAnimator->setDTheta(v);
}

void iAnimatorScript::RotationRoll(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getDRoll());
	curAnimator->setDRoll(v);
}

void iAnimatorScript::Scale(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getDScale());
	curAnimator->setDScale(v);
}

void iAnimatorScript::XsectionSpeed(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curAnimator->getSlideSpeed());
	curAnimator->setSlideSpeed(v);
}

//
//  variable limits
//
void iAnimatorScript::VarLo(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	UPDATE_VALUE_ABS(n0,n);
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getVarLow(n));
	curLimits->setVarLow(n,v);
}

void iAnimatorScript::VarHi(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	UPDATE_VALUE_ABS(n0,n);
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getVarHigh(n));
	curLimits->setVarHigh(n,v);
}

void iAnimatorScript::AttLo(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	UPDATE_VALUE_ABS(n0,n);
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getAttLow(n));
	curLimits->setAttLow(n,v);
}

void iAnimatorScript::AttHi(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	UPDATE_VALUE_ABS(n0,n);
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getAttHigh(n));
	curLimits->setAttHigh(n,v);
}

void iAnimatorScript::VecHi(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getVarHigh(1));
	curLimits->setVarHigh(1,v);
}

void iAnimatorScript::TenHi(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_REL(v0,rel,v,curLimits->getTenHigh(1));
	curLimits->setTenHigh(1,v);
}

//
//  focal point & camera location
//
void iAnimatorScript::CameraFocus(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	vtkFloat v[3];
 	double *p = curCamera->GetFocalPoint();

	UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	curCamera->SetFocalPoint(v);
	curCamera->OrthogonalizeViewUp();
}

void iAnimatorScript::CameraPosition(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	vtkFloat v[3];
	double *p = curCamera->GetPosition();

	UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	curCamera->SetPosition(v);
	curCamera->OrthogonalizeViewUp();
}

void iAnimatorScript::CameraViewUp(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	vtkFloat v[3];
	double *p = curCamera->GetViewUp();

	UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	double *dop = curCamera->GetDirectionOfProjection();
	double c[3];
	double vup[3];
	vup[0] = v[0]; vup[1] = v[1]; vup[2] = v[2];
	vtkMath::Cross(dop,vup,c);

	if((v[0]*v[0]+v[1]*v[1]+v[2]*v[2])>1.0e-8 && (c[0]*c[0]+c[1]*c[1]+c[2]*c[2])>1.0e-8)
	{
		curCamera->SetViewUp(v);
		curCamera->OrthogonalizeViewUp();
	}
}

void iAnimatorScript::CameraScale(iValue *v0, Assignment rel, iString *pErrorMessage)
{
	float v;
    UPDATE_VALUE_REL(v0,rel,v,curCamera->GetParallelScale());
	curCamera->SetParallelScale(v);
}

void iAnimatorScript::Projection(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int v;
    UPDATE_VALUE_ABS(v0,v);
	if(v == 0) curVTK->setParallelProjection(true); else curVTK->setParallelProjection(false); 
}

//
//  Features
//
void iAnimatorScript::TimeLabel(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->showLabel(v); 
}

void iAnimatorScript::TimeLabelValue(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	float v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->updateLabel(&v); 
}

void iAnimatorScript::ColorBars(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->showColorBars(v); 
}

void iAnimatorScript::BoundingBox(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->showBox(v); 
}

void iAnimatorScript::TitlePageFile(iValue* v0, Assignment, iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(!curAnimator->setTitlePageFile(v)) *pErrorMessage = "Unable to load title page image.";
}


void iAnimatorScript::TitlePageNumFrames(iValue* v0, Assignment, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v);
	curAnimator->setNumTitlePageFrames(v);
}


void iAnimatorScript::TitlePageNumBlendedFrames(iValue* v0, Assignment, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v);
	curAnimator->setNumTitlePageBlendedFrames(v);
}

void iAnimatorScript::LogoFile(iValue* v0, Assignment, iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(!curAnimator->setLogoFile(v)) *pErrorMessage = "Unable to load logo image.";
}

void iAnimatorScript::LogoOpacity(iValue* v0, Assignment, iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_ABS(v0,v);
	curAnimator->setLogoOpacity(v);
}

void iAnimatorScript::LogoPosition(iValue* v0, Assignment, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v);
	curAnimator->setLogoPosition(v);
}

//
//  Manipulation of VTK windows
//
void iAnimatorScript::SetCurrentVTK(iValue* v0, iScriptType::Assignment rel, iString *pErrorMessage)
{
	int v;
	UPDATE_VALUE_ABS(v0,v); v--;
	if(v>=0 && v<=iVTKWindow::getMaxWindowIndex()) iQTWindow::getCurrentWindow()->setCurrentWindow(v); else *pErrorMessage = "Invalid window number (must be between 1 and "+iString::number(iVTKWindow::getMaxWindowIndex())+").";
}

//
// Manupulation of iVisualObjects
//
void iAnimatorScript::SurfaceVar(iValue *n0, iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	int v;
	iSurface *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getSurfaceFamily()->getCurrentMember(); else s = curVTK->getSurfaceFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		s->setVar(v);
	}
}

void iAnimatorScript::SurfaceLevel(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	iSurface *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getSurfaceFamily()->getCurrentMember(); else s = curVTK->getSurfaceFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getLevel());
		s->setLevel(v);
	}
}

void iAnimatorScript::SurfaceOpacity(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	iSurface *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getSurfaceFamily()->getCurrentMember(); else s = curVTK->getSurfaceFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getOpacity());
		s->setOpacity(v);
	}
}

void iAnimatorScript::XsectionVar(iValue *n0, iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	int v;
	iXsection *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getXsectionFamily()->getCurrentMember(); else s = curVTK->getXsectionFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		s->setVar(v);
	}
}

void iAnimatorScript::XsectionDir(iValue *n0, iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	int v;
	iXsection *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getXsectionFamily()->getCurrentMember(); else s = curVTK->getXsectionFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		s->setDir(v);
	}
}

void iAnimatorScript::XsectionPos(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	double v;
	iXsection *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getXsectionFamily()->getCurrentMember(); else s = curVTK->getXsectionFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getPos());
		s->setPos(v);
	}
}

void iAnimatorScript::VolumeVar(iValue *n0, iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	int v;
	iVolume *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getVolumeFamily()->getCurrentMember(); else s = curVTK->getVolumeFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		s->setVar(v);
	}
}

void iAnimatorScript::ParticlesOpacity(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	iParticles *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getParticlesFamily()->getCurrentMember(); else s = curVTK->getParticlesFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getOpacity());
		s->setOpacity(v);
	}
}

void iAnimatorScript::ParticlesDownsampleFactor(iValue *v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	float v;
	UPDATE_VALUE_ABS(v0,v);
	if(v < 1) return;
	iVTKWindow::getCurrentWindow()->getReader()->setParticlesDownsampleFactor(v);
	if(iVTKWindow::getCurrentWindow()->getReader()->reloadCurrentSet(IDATAREADER_PART) != 0)
	{
		*pErrorMessage = "Could not reload the current particle file.";
	}
	iQTWindow::getCurrentWindow()->resetProgressBar();
}

void iAnimatorScript::MarkerPosition(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	iMarker *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getMarkerFamily()->getCurrentMember(); else s = curVTK->getMarkerFamily()->getMember(n-1);
	if(s != 0)
	{
		double v[3];
		double *p = s->getPosition();
		UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		s->setPosition(v);
	}
}

void iAnimatorScript::MarkerSize(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	iMarker *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getMarkerFamily()->getCurrentMember(); else s = curVTK->getMarkerFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getSize());
		s->setSize(v);
	}
}

void iAnimatorScript::MarkerOpacity(iValue *n0, iValue *v0, Assignment rel, iString *pErrorMessage)
{
	int n;
	float v;
	iMarker *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getMarkerFamily()->getCurrentMember(); else s = curVTK->getMarkerFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_REL(v0,rel,v,s->getOpacity());
		s->setOpacity(v);
	}
}


//
//  Global variable function
//
iValue* iAnimatorScript::GetMarkerPosition(int i)
{
	int j;

	if(i>=0 && i<=curVTK->getMarkerFamily()->getMaxMemberIndex())
	{
		double *p = curVTK->getMarkerFamily()->getMember(i)->getPosition();
		for(j=0; j<3; j++) ((iValue_FS*)((iValue_VA*)valVA3)->getComponent(j+1))->setNumericValue(p[j]);
		return valVA3;
	}
	else
	{
		return NULL;
	}
}


//
//  File loading functions - for batch operations
//
void iAnimatorScript::LoadMeshFile(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(v[0] == '+') v = iEnvironment::getInstance()->getFileName("MESH_DATA_DIR",v.mid(1).latin1());
	int ret = curReader->loadMeshFile(v);
	if(ret != 0) *pErrorMessage = "Error in reading MESH file, error code: "+iString::number(ret);
}


void iAnimatorScript::LoadPartFile(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(v[0] == '+') v = iEnvironment::getInstance()->getFileName("PART_DATA_DIR",v.mid(1).latin1());
	int ret = curReader->loadPartFile(v);
	if(ret != 0) *pErrorMessage = "Error in reading PARTICLES file, error code: "+iString::number(ret);
}


void iAnimatorScript::LoadVectFile(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(v[0] == '+') v = iEnvironment::getInstance()->getFileName("VECT_DATA_DIR",v.mid(1).latin1());
	int ret = curReader->loadVectFile(v);
	if(ret != 0) *pErrorMessage = "Error in reading VECTOR file, error code: "+iString::number(ret);
}


void iAnimatorScript::LoadTensFile(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	UPDATE_VALUE_ABS(v0,v);
	if(v[0] == '+') v = iEnvironment::getInstance()->getFileName("TENS_DATA_DIR",v.mid(1).latin1());
	int ret = curReader->loadTensFile(v);
	if(ret != 0) *pErrorMessage = "Error in reading TENSOR file, error code: "+iString::number(ret);
}


//
//  Object showing/hiding functions - for batch operations
//
void iAnimatorScript::ShowSurface(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getSurfaceFamily()->show(v); 
}


void iAnimatorScript::ShowXsection(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getXsectionFamily()->show(v); 
}


void iAnimatorScript::ShowVolume(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getVolumeFamily()->show(v); 
}


void iAnimatorScript::ShowParticles(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getParticlesFamily()->show(v); 
}


void iAnimatorScript::ShowVector(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getVectorFamily()->show(v); 
}


void iAnimatorScript::ShowTensor(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	bool v;
    UPDATE_VALUE_ABS(v0,v);
	curVTK->getTensorFamily()->show(v); 
}


//
//  iObject state assignment
//
void iAnimatorScript::MarkerState(iValue* n0, iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	iString v;
	iMarker *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getMarkerFamily()->getCurrentMember(); else s = curVTK->getMarkerFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iMarker::");
		s->unpackState(v);
	}
}


void iAnimatorScript::ParticlesState(iValue* n0, iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	iString v;
	iParticles *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getParticlesFamily()->getCurrentMember(); else s = curVTK->getParticlesFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iParticles::");
		s->unpackState(v);
	}
}


void iAnimatorScript::SurfaceState(iValue* n0, iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	iString v;
	iSurface *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getSurfaceFamily()->getCurrentMember(); else s = curVTK->getSurfaceFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iSurface::");
		s->unpackState(v);
	}
}


void iAnimatorScript::TensorState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iTensor *s = curVTK->getTensorFamily()->getCurrentMember();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iTensor::");
		s->unpackState(v);
	}
}


void iAnimatorScript::VectorState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iVector *s = curVTK->getVectorFamily()->getCurrentMember();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iVector::");
		s->unpackState(v);
	}
}


void iAnimatorScript::VolumeState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iVolume *s = curVTK->getVolumeFamily()->getCurrentMember();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iVolume::");
		s->unpackState(v);
	}
}


void iAnimatorScript::XsectionState(iValue* n0, iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	int n;
	iString v;
	iXsection *s;
	UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVTK->getXsectionFamily()->getCurrentMember(); else s = curVTK->getXsectionFamily()->getMember(n-1);
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iXsection::");
		s->unpackState(v);
	}
}


void iAnimatorScript::AnimatorState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iAnimator *s = curVTK->getAnimator();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iAnimator::");
		s->unpackState(v);
	}
}


void iAnimatorScript::DataReaderState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iDataReader *s = curVTK->getReader();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iDataReader::");
		s->unpackState(v);
	}
}


void iAnimatorScript::LimitsState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iLimits *s = curVTK->getReader()->getCurrentLimits();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iLimits::");
		s->unpackState(v);
	}
}


void iAnimatorScript::ParticlesSplitterState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iParticlesSplitter *s = curVTK->getCurrentParticlesSplitter();
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iParticlesSplitter::");
		s->unpackState(v);
	}
}


void iAnimatorScript::VTKState(iValue* v0, Assignment vtkNotUsed(rel), iString *pErrorMessage)
{
	iString v;
	iVTK *s = curVTK;
	if(s != 0)
	{
		UPDATE_VALUE_ABS(v0,v);
		v.replace("+","iVTK::");
		s->unpackState(v);
	}
}




