/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Ifrit data reader: vtkSource class that reads in IFRIT .bin and .txt files
  and keeps the original data. Also controls shifting, extending, and limiting
  the data.

=========================================================================*/

#ifndef IDATAREADER_H
#define IDATAREADER_H


#include "iglobals.h"
#include "iobject.h"

class vtkFloatArray;

class iFile;
class iMeshFromParticlesFilter;
class iLimits;
class iUniformMeshData;
class iPolygonalData;
class iProgressBar;
struct iDataInfo;

#include <vtkSource.h>
#include "istring.h"
#include "ivtksource.h"


#define IDATAREADER_FUNCTIONS_PUBLIC_DECLARATIONS(label,Label) \
    virtual int load##Label##File(const iString &fname, bool inSet = false, bool justCheckIfExists = false, bool saveInTheList = false); \
	inline bool is##Label##FileAnimatable(){ return label##Animatable; } \
    inline bool is##Label##FileBelongToSet(){ return label##Set; } \
	virtual bool isThere##Label##Data(int dc, int ds); \
	inline iString getLast##Label##FileName(){ return last##Label##FileName; } \
	iString getCurrent##Label##FileDir();

#define IDATAREADER_FUNCTIONS_PROTECTED_DECLARATIONS(label,Label) \
    virtual int load##Label##BinFile(const iString &fname); \
    virtual int load##Label##TxtFile(const iString &fname); \
    virtual void operateOn##Label##Data();


#define IDATAREADER_MESH	1
#define IDATAREADER_PART	2
#define IDATAREADER_VECT	4
#define IDATAREADER_TENS	8
#define IDATAREADER_ALL		UINT_MAX


class iDataReader : public iObject, public vtkSource
{

	friend class iObjectFactory;

public:
	
	static iDataReader *New(iVTK *v);
	static bool getEndinessOfMachine(){ return endinessOfMachine; }
	
	//
	//  Have to do this as VTK does not offer enough flexibility in swapping bytes
	//
	static void Swap2Bytes(char* data);
	static void Swap4Bytes(char* data);
	static void Swap8Bytes(char* data);

	static void Swap4BytesRange(char* data, long count); 
	static void Swap8BytesRange(char* data, long count); 

	static void SwapIdType(vtkIdType *p);
	//
	//  Overloads for flexibility
	//
	static void Swap4Bytes(float *p)           { Swap4Bytes((char *)p); }
	static void Swap4Bytes(int *i)             { Swap4Bytes((char *)i); }
	static void Swap4Bytes(unsigned int *i)    { Swap4Bytes((char *)i); }
	static void Swap4Bytes(long *i)            { Swap4Bytes((char *)i); }
	static void Swap4Bytes(unsigned long *i)   { Swap4Bytes((char *)i); }

	static void Swap8Bytes(double *p)          { Swap8Bytes((char *)p); }

	static void Swap4BytesRange(float* data, long count)          { Swap4BytesRange((char *)data,count); }
	static void Swap4BytesRange(int* data, long count)            { Swap4BytesRange((char *)data,count); }
	static void Swap4BytesRange(unsigned int* data, long count)   { Swap4BytesRange((char *)data,count); }
	static void Swap4BytesRange(long* data, long count)           { Swap4BytesRange((char *)data,count); }
	static void Swap4BytesRange(unsigned long* data, long count)  { Swap4BytesRange((char *)data,count); }

	static void Swap8BytesRange(double* data, long count)         { Swap8BytesRange((char *)data,count); }

	//
	//  Non-static members
	//
	virtual ~iDataReader();

	void abort(){ aborted = true; }

	IDATAREADER_FUNCTIONS_PUBLIC_DECLARATIONS(mesh,Mesh)
	IDATAREADER_FUNCTIONS_PUBLIC_DECLARATIONS(part,Part)
	IDATAREADER_FUNCTIONS_PUBLIC_DECLARATIONS(vect,Vect)
	IDATAREADER_FUNCTIONS_PUBLIC_DECLARATIONS(tens,Tens)
 
	inline void setEndinessOfData(bool s){ endinessOfData = s; }
	inline bool getEndinessOfData(){ return endinessOfData; }

	virtual int loadFileSet(const iString &fname, bool saveInTheList = false);
	virtual int loadFileSet(int record, bool justCheckIfExists = false, bool saveInTheList = false);
    virtual int loadRecord(int rec, bool justCheckIfExists = false, bool saveInTheList = false);
    virtual int loadNextRecord(int skip = 0, bool justCheckIfExists = false, bool saveInTheList = false);
	virtual int reloadCurrentSet(int m = IDATAREADER_ALL);

	virtual void eraseData(int dc, int m);
	virtual void detachData(int dc, int m);
		
	inline int getWarningInfo(){ return warno; }

	virtual iUniformMeshData* getMeshOutput(){ return passedMeshOutput; }
	virtual iUniformMeshData* getVectorOutput(){ return passedVectOutput; }
	virtual iUniformMeshData* getTensorOutput(){ return passedTensOutput; }
	virtual iPolygonalData* getParticlesOutput(){ return passedPartOutput; }

	inline char getSetLeader(){ return setLeader; }
	inline bool isFileAnimatable(){ return (currec >= 0); }
	inline int getRecordNumber(){ return currec; }
	inline bool isSet(){ return (currec>=0 && (isThereMeshData(0,0)||isTherePartData(0,0)||isThereVectData(0,0)||isThereTensData(0,0))); }

	virtual void getDataInfo(iDataInfo *i);
	virtual void setDataShift(int d, float f); 
	virtual bool doDataShift(float *f = 0, bool resetAll = true);
	inline float getShift(int d){ if(d>=0 && d<3) return newShift[d]; else return 0.0; }

	virtual void setMeshArrayFunction(const char *s){ meshArrayFunction = iString(s); }
	virtual void setMeshArrayOutput(int n);
	inline iString getMeshArrayFunction(){ return meshArrayFunction; }
	inline int getMeshArrayOutput(){ return meshArrayOutput; }

	virtual void setMeshFromParticles(bool s);
	virtual bool getMeshFromParticles(){ return part2meshOn; }
	virtual iMeshFromParticlesFilter* getMeshFromParticlesFilter(){ return part2meshFilter; }
	virtual void createMeshFromParticles();

	virtual void setAdjustableBounds(bool s){ adjustableBounds = s; }
	inline bool getAdjustableBounds(){ return adjustableBounds; }
		
	virtual void setScaledDimension(int v);
	inline int getScaledDimension(){ return scaledDim; }

	virtual void setCellToPointMode(int m);
	inline int getCellToPointMode(){ return modeCellToPoint; }

	virtual void setPointsAreFloat(bool s){ pointsAreFloat = s; }
	inline bool getPointsAreFloat(){ return pointsAreFloat; }

	virtual void getLevelLimits(int v, float &lmin, float &lmax);

	inline double getParticlesBasicScale(){ return partBasicScale; }

	virtual float getMemorySize();

	inline iLimits* getLimits(int dc, int ds = 0){ if(dc>=0 && dc<=MAXDATACHANNEL && ds>=0 && ds<=maxDataStream[dc]) return limits[dc][ds]; else return 0; }
	inline iLimits* getCurrentLimits(){ if(limits[dataChannel] != NULL) return limits[dataChannel][dataStream[dataChannel]]; else return NULL; }
	inline int getCurrentDataChannel(){ return dataChannel; }
	inline int getMaxDataChannel(){ return maxDataChannel; }
	inline bool isDataChannelActive(int dc){ if(dc>=0 && dc<=MAXDATACHANNEL) return dataChannelActive[dc]; else return false; }
	inline int getCurrentDataStream(int dc){ if(dc>=0 && dc<=MAXDATACHANNEL) return dataStream[dc]; else return -1; }
	inline int getMaxDataStream(int dc){ if(dc>=0 && dc<=MAXDATACHANNEL) return maxDataStream[dc]; else return -1; }
	//
	//  Functions needed to perform iQT functionality removed in v2.2
	//
	virtual iString getLastFileSetName();

	inline iString getLastFileName(){ return lastFileName; }
	inline iString getLastAttemptedFileName(){ return lastAttemptedFileName; }

	inline const iStringList& getVisitedFilesList(){ return visitedFilesList; }

	inline int getParticlesDownsampleMode(){ return partDownsampleMode; }
	virtual void setParticlesDownsampleMode(int v);

	inline float getParticlesDownsampleFactor(){ return partDownsampleFactor; }
	virtual void setParticlesDownsampleFactor(float v);

	virtual void setVarStretch(int v, int s);

	virtual void packState(iString &s);
	virtual void unpackState(iString s);

	virtual void setProgressBar(int mode, iProgressBar *pb);
	virtual bool isProgressBarActive(int mode);

protected:
	
	iDataReader(iVTK *v);

	virtual void ExecuteData(vtkDataObject *out);
	virtual void updateProgress(int mode, float p);

	IDATAREADER_FUNCTIONS_PROTECTED_DECLARATIONS(mesh,Mesh)
	IDATAREADER_FUNCTIONS_PROTECTED_DECLARATIONS(part,Part)
	IDATAREADER_FUNCTIONS_PROTECTED_DECLARATIONS(vect,Vect)
	IDATAREADER_FUNCTIONS_PROTECTED_DECLARATIONS(tens,Tens)

	virtual bool shiftData(int d, float f, bool resetAll);
	virtual void getCellDims(int dim[3], int dimCell[3]);
	virtual void computeSpacing(int dim[3], iUniformMeshData *output);
	virtual void transformToPointData(int mode, bool arrayIsVertical, int numComponents, float *&data, int dims[3], long &size);

	virtual long buildParticlesMask(long ntot0, bool*mask, float pdf);

	bool adjustableBounds;
	static bool endinessOfMachine;
	bool endinessOfData;

	int modeCellToPoint;
	bool aborted;

	bool pointsAreFloat;

	double partBasicScale;

	//
	//  Data channel & stream manipulation
	//
	int dataChannel, maxDataChannel;
	bool dataChannelActive[MAXDATACHANNEL+1];
	iLimits **limits[MAXDATACHANNEL+1];
	int maxDataStream[MAXDATACHANNEL+1];
	int dataStream[MAXDATACHANNEL+1];

	int warno;
	float *meshData, *vectData, *tensData;  
	int meshDims[3], vectDims[3], tensDims[3];
	float newShift[3], oldShift[3];
	long meshSize, vectSize, tensSize;
	bool meshSet, partSet, vectSet, tensSet;
	iString mode;
	int scaledDim;

	float levMin[99];
	float levMax[99];

	iString meshArrayFunction;
	int meshArrayOutput;

	char periodicityLeader, setLeader;

	iString meshFroot, meshFsuf;
	iString partFroot, partFsuf;
	iString vectFroot, vectFsuf;
	iString tensFroot, tensFsuf;
	int currec, endrec;

    iString currentTensFileDir;
    iString currentVectFileDir;
    iString currentPartFileDir;
    iString currentMeshFileDir;
    iString lastTensFileName;
    iString lastVectFileName;
    iString lastPartFileName;
    iString lastMeshFileName;
	iString lastFileName, lastAttemptedFileName;

	iStringList visitedFilesList;

	iUniformMeshData *meshOutput;
	vtkFloatArray *meshArray;

	long partSize;
	iPolygonalData *partOutput;

    iUniformMeshData *vectOutput;
	vtkFloatArray *vectArray;

    iUniformMeshData *tensOutput;
	vtkFloatArray *tensArray;

	iUniformMeshData *passedMeshOutput;
	iPolygonalData *passedPartOutput;
    iUniformMeshData *passedVectOutput;
    iUniformMeshData *passedTensOutput;

	iMeshFromParticlesFilter *part2meshFilter;
	bool part2meshOn, meshDataAssigned;

	bool meshShifted, partShifted, vectShifted, tensShifted;
	bool meshUpdated, partUpdated, vectUpdated, tensUpdated;
	bool meshAnimatable, partAnimatable, vectAnimatable, tensAnimatable;

	int partDownsampleMode;
	float partDownsampleFactor;

	iProgressBar *meshProgressBar, *partProgressBar, *vectProgressBar, *tensProgressBar;

};


inline void iDataReader::Swap2Bytes(char* data)
{ 
  char one_byte;  
  one_byte = data[0]; data[0] = data[1]; data[1] = one_byte;
}

inline void iDataReader::Swap4Bytes(char* data)
{ 
  char one_byte; 
  one_byte = data[0]; data[0] = data[3]; data[3] = one_byte;
  one_byte = data[1]; data[1] = data[2]; data[2] = one_byte; 
}

inline void iDataReader::Swap8Bytes(char* data)
{ 
  char one_byte;
  one_byte = data[0]; data[0] = data[7]; data[7] = one_byte;
  one_byte = data[1]; data[1] = data[6]; data[6] = one_byte;
  one_byte = data[2]; data[2] = data[5]; data[5] = one_byte;
  one_byte = data[3]; data[3] = data[4]; data[4] = one_byte; 
}


inline void iDataReader::SwapIdType(vtkIdType *p)
{ 
#ifdef VTK_USE_64BIT_IDS
	Swap8Bytes((char *)p);
#else
	Swap4Bytes((char *)p);
#endif
}

#endif


