/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  IFRIT entry points - for running the whole thing from other programs
//

#include "ifrit.h"

#include "ivtk.h"
#include "iqt.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iqtextension.h"
#include "ipalette.h"
#include "ienvironment.h"
#include "idialogmessage.h"
#include "iparallelizator.h"
#include "ifile.h"
#include "ianimator.h"
#include "ianimatorscript.h"

#include <qapplication.h>
#include <qmessagebox.h>
#include <qrect.h>
#include <qtextstream.h>

#include <vtkOutputWindow.h>


//
//  Take over VTK display function to prevent it from displaying extra windows
//

class iRedirectedOutputWindow : public vtkOutputWindow
{
	
public:
	
	virtual void DisplayText(const char*);
	virtual void DisplayErrorText(const char*);
	virtual void DisplayWarningText(const char*);
	virtual void DisplayGenericWarningText(const char*);
	virtual void DisplayDebugText(const char*);
	
};


void iRedirectedOutputWindow::DisplayText(const char* message)
{
	iQTWindow::getCurrentWindow()->logLine("VTK",message);
}


void iRedirectedOutputWindow::DisplayErrorText(const char* message)
{
	//
	//  Block some of the VTK error messages - not all are useful
	//
	const int nBlockError = 3;
	const char *classBlockError[] = { 
		"vtkDecimate",
		"vtkPolyDataNormals",
		"vtkWindowedSincPolyDataFilter"
	};
	const char *messgBlockError[] = {
		"No data to decimate!",
		"No data to generate normals for!",
		"No data to smooth!"
	};

	int i;
	for(i=0; i<nBlockError; i++)
	{
		if(strstr(message,classBlockError[i])!=0 && strstr(message,messgBlockError[i])!=0) return;
	}
	//
	//  Cut extra parts of VTK error messages
	//
	char *errorMessage, *fullMessage;
	fullMessage = (char *)message;
	if(iEnvironment::getInstance()->getErrorMessageFirstString()==0)
	{
		errorMessage = fullMessage;
	} 
	else 
	{
		errorMessage = strstr(fullMessage,iEnvironment::getInstance()->getErrorMessageFirstString());
	}
	errorMessage += iEnvironment::getInstance()->getErrorMessageFirstSymbol();
	
	switch(iEnvironment::getInstance()->whereErrorMessagesGo()) 
	{
	case 1:	{ iQTWindow::getCurrentWindow()->logLine("VTK",errorMessage,QColor(255,0,0)); break; }
	case 2: {
		QMessageBox::critical(0,"IFRIT Array Calculator",QString(errorMessage));
		break; }
	default: ;
	}
	
}


void iRedirectedOutputWindow::DisplayWarningText(const char* fullMessage)
{
	//
	//  Cut extra parts of VTK error messages
	//

	char *message = (char *)strstr(fullMessage,"IFRIT:START-HERE");
	if(message == NULL)
	{
		message = (char *)fullMessage;
	}
	else
	{
		message += strlen("IFRIT:START-HERE");
	}
	iQTWindow::getCurrentWindow()->logLine("VTK",message);
}


void iRedirectedOutputWindow::DisplayGenericWarningText(const char* message)
{
	this->DisplayText(message);
}


void iRedirectedOutputWindow::DisplayDebugText(const char* message)
{
#ifdef _DEBUG
	this->DisplayText(message);
#endif
}


bool ifrit::construct(iDialogMessage *flashWindow, const char *initFile)
{

	if(flashWindow != NULL) flashWindow->setMessage("Creating display window...");
	//
	//  Create palettes and environment
	//
	iPaletteList::construct();
	iEnvironment::construct();
	iParallelizator::construct();
	//
	//  Take over VTK output window
	//
	vtkOutputWindow::SetInstance(new iRedirectedOutputWindow);
	//    
	//  Create VTK window
	//    
	iVTKWindow::construct();
	//
	//  Create QT window
	//    
	if(flashWindow != NULL) flashWindow->setMessage("Creating widgets...");
	iQTWindow::construct();
	
	if(flashWindow != NULL) flashWindow->setMessage("Loading settings...");
	return iQTWindow::getCurrentWindow()->optionsLoad(initFile);
	
}


bool ifrit::show(bool positionWindows, bool startDocked)
{
	const int topOff = 60;
	const int frameXoffDefault = 13;
	const int frameYoffDefault = 25;
	int i, frameXoff = 1, frameYoff = 1;
	int leftOffset = 0;
	QRect gvf, gvn, gqf;

	//
	// Calculate the left offset of the VTK window 
	//
	int desktopWidth = qApp->desktop()->width();
	leftOffset = (desktopWidth-iVTKWindow::getCurrentWindow()->width()-iQTWindow::getCurrentWindow()->width()-20)/2;
	if(leftOffset < 0) leftOffset = 0;

	if(positionWindows)
	{
		//
		//  Move VTK window
		//			     
		iVTKWindow::getCurrentWindow()->move(leftOffset,topOff);
	}
	
	//
	//  Show the VTK window and give a chance to the Window Manager to decorate the window
	//			     
	if(startDocked)
	{
		iQTWindow::getCurrentWindow()->placeWindowsInDockedPositions(QPoint(leftOffset,topOff),iVTKWindow::getCurrentWindow()->size());
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->show();
		qApp->flush();
		qApp->processEvents();
		if(iVTKWindow::getMaxWindowIndex() == 0) iVTKWindow::getWindow(0)->showMaximized();
	}
	else 
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->show();
	}
	qApp->flush();
	qApp->processEvents();

	if(positionWindows && !startDocked)
	{
		gvf = iVTKWindow::getCurrentWindow()->frameGeometry();
		gvn = iVTKWindow::getCurrentWindow()->geometry();
		if(gvf.width() > gvn.width()) 
		{
			frameXoff = 1;
		}
		else
		{
			frameXoff = frameXoffDefault; // X11 failed to assign a proper frame
		}
		if(gvf.height() > gvn.height())
		{
			frameYoff = 1;
		}
		else
		{
			frameYoff = frameYoffDefault;
		}

		iQTWindow::getCurrentWindow()->move(gvf.right()+frameXoff,topOff);

	}

	//
	//  Show the QT window and give a chance to the Window Manager to decorate the window
	//			     
	iQTWindow::getCurrentWindow()->show();
	qApp->flush();
	qApp->processEvents();

	if(!startDocked && positionWindows && iQTWindow::getCurrentWindow()->getExtension()!=NULL)
	{
		gqf = iQTWindow::getCurrentWindow()->frameGeometry();
		iQTWindow::getCurrentWindow()->getExtension()->resize(gvn.width(),gqf.height()-gvf.height()-(gvf.height()-gvn.height()));
		iQTWindow::getCurrentWindow()->getExtension()->move(leftOffset,gvf.bottom()+frameYoff);
	}

	if(iQTWindow::getCurrentWindow()->getExtension() != NULL) iQTWindow::getCurrentWindow()->getExtension()->show();

	return true;

}


bool ifrit::executeScript(const char *scriptFile)
{
	int i;
	bool ok = true;
	iString fname;

	if(scriptFile[0] == '+')
	{
		fname = iEnvironment::getInstance()->getFileName("SCRIPT_DIR",scriptFile+1);
	}
	else
	{
		fname = iString(scriptFile);
	}
	iFile f(fname);

	if(!f.exists()) 
	{
		cerr << "File " << scriptFile << " does not exist. Abort.\n"; 
	}
		
		
	if(!f.open(IO_ReadOnly)) 
	{
		cerr << "File " << scriptFile << " cannot be open for reading. Abort.\n"; 
	}
		
	cout << "Reading the file..." << endl;
	QTextStream ts( &f );
	iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->setText(ts.read());

	cout << "Compiling the script..." << endl;
    for(i=0; i<iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getNumberOfLines(); i++)
	{
		
		if(iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->runOneLine(true) > 0)
		{
			ok = false;
			cout << "Syntax error in script, line " << iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getCurrentLine() << " : " << iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getErrorMessage() << endl;
		}
		
    }

	if(ok) 
	{
		//
		//  Set image method to pixmap
		//
		int *im = new int[iVTKWindow::getMaxWindowIndex()+1];
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) im[i] = iVTKWindow::getWindow(i)->getImageMethod();
		bool us = iVTKWindow::getCurrentWindow()->getAnimator()->getUseScript();

		iString s = "iVTK::imageMethod/1/";
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->unpackState(s);
		iVTKWindow::getCurrentWindow()->getAnimator()->setUseScript(true);
		iQTWindow::getCurrentWindow()->blockProgressDialog(true);

		iVTKWindow::getCurrentWindow()->getAnimator()->resetState();
		if(iVTKWindow::getCurrentWindow()->getAnimator()->animate()==2 && !iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getErrorMessage().isEmpty())
		{
			//
			//  Error
			//
			cout << "Error in script: " << iVTKWindow::getCurrentWindow()->getAnimator()->getScript()->getErrorMessage() << endl;
		}

		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) 
		{
			s = "iVTK::imageMethod/" + iString::number(im[i]) + "/";
			iVTKWindow::getWindow(i)->unpackState(s);
		}
		delete [] im;
		iVTKWindow::getCurrentWindow()->getAnimator()->setUseScript(us);
		iQTWindow::getCurrentWindow()->blockProgressDialog(false);
	}

	return ok;
}


bool ifrit::destruct()
{
	
	bool qtDocked = iQTWindow::getCurrentWindow()->isDocked();
	iQTWindow::destruct();
	if(!qtDocked) iVTKWindow::destruct();
	iPaletteList::destruct();
	iEnvironment::destruct();
	iParallelizator::destruct();

	return true;
	
}

