/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iglobals.h"

#ifndef I_VTK_VERSION_40

#include "iinteractoreventrecorder.h"


#include <vtkCallbackCommand.h>
#include <vtkRenderWindowInteractor.h>
#include <vtkObjectFactory.h>


iInteractorEventRecorder::iInteractorEventRecorder()
{
	stopped = false;
}


void iInteractorEventRecorder::SetFileName(const char *s)
{
	if(this->InputStream != 0)
    {
		delete this->InputStream;
		this->InputStream = NULL;
    }
	
	if(this->OutputStream != 0)
    {
		delete this->OutputStream;
		this->OutputStream = NULL;
    }

	vtkInteractorEventRecorder::SetFileName(s);

	this->OutputStream = new fstream(this->FileName, ios::in|ios::out);
	if (this->OutputStream->fail())
	{
		vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
		delete this->OutputStream;
		return;
	}

}


void iInteractorEventRecorder::Record()
{
	if ( this->State == vtkInteractorEventRecorder::Start )
    {
		if(!this->OutputStream ) //need to open file
		{
			this->OutputStream = new fstream(this->FileName, ios::in|ios::out);
			if (this->OutputStream->fail())
			{
				vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
				delete this->OutputStream;
				return;
			}
		}

		this->OutputStream->clear();

		if((int)this->OutputStream->tellp() == 0)
		{
			*this->OutputStream << "# StreamVersion " << vtkInteractorEventRecorder::StreamVersion << "\n";
		}
		
		vtkDebugMacro(<<"Recording");
		this->State = vtkInteractorEventRecorder::Recording;
    }
}


void iInteractorEventRecorder::Play()
{
	//
	//  Copied from vtkInteractorEventRecorder::Play() with minor modifications
	//
	if ( this->State == vtkInteractorEventRecorder::Start )
    {
		if ( this->ReadFromInputString )
		{
			vtkDebugMacro(<< "Reading from InputString");
			int len;
			if ( this->InputString == NULL || (len = strlen(this->InputString) <= 0) )
			{
				vtkErrorMacro(<< "No input string specified");
				return;
			}
			this->InputStream = new istrstream(this->InputString, len);
			if (this->InputStream->fail())
			{
				vtkErrorMacro(<< "Unable to read from string");
				delete this->InputStream;
				return;
			}
		}
		else
		{
			if(!this->OutputStream) //need to open file
			{
				this->OutputStream = new fstream(this->FileName, ios::in|ios::out);
				if (this->OutputStream->fail())
				{
					vtkErrorMacro(<< "Unable to open file: "<< this->FileName);
					delete this->OutputStream;
					return;
				}
//				if(this->OutputStream && this->OutputStream->tellp()>0)
//				{
//					this->OutputStream->seekg(this->OutputStream->tellp());
//				}
			}
		}
		
		vtkDebugMacro(<<"Playing");
		this->State = vtkInteractorEventRecorder::Playing;
		
		// Read events and invoke them on the object in question
		char event[128], keySym[64], buffer[512];
		int pos[2], ctrlKey, shiftKey, keyCode, repeatCount;
		float stream_version = 0.0, tempf;
		
		while(!stopped && !this->OutputStream->eof())
		{
			this->OutputStream->width(256);
			*((fstream *)this->OutputStream) >> event;
			
			// Quick skip comment
			if (*event == '#')
			{
				((fstream *)this->OutputStream)->getline(buffer, 512);
				
				// Parse the StreamVersion (not using >> since comment could be empty)
				// Expecting: # StreamVersion x.y
				
				if (strlen(buffer) > 15 && !strncmp(buffer, " StreamVersion ", 15))
				{
					int res = sscanf(buffer + 15, "%f", &tempf);
					if (res && res != EOF)
					{
						stream_version = tempf;
					}
				}
			}
			else
			{
				unsigned long ievent = vtkCommand::GetEventIdFromString(event);
				if (ievent == vtkCommand::NoEvent)
				{
					((fstream *)this->OutputStream)->ignore(512, '\n');
				}
				else
				{
					if (stream_version >= 1.1)
					{
						// We could grab the time info here
					}
					*((fstream *)this->OutputStream) >> pos[0];
					*((fstream *)this->OutputStream) >> pos[1];
					*((fstream *)this->OutputStream) >> ctrlKey;
					*((fstream *)this->OutputStream) >> shiftKey;
					*((fstream *)this->OutputStream) >> keyCode;
					*((fstream *)this->OutputStream) >> repeatCount;
					*((fstream *)this->OutputStream) >> keySym;
					
					this->Interactor->SetEventPosition(pos);
					this->Interactor->SetControlKey(ctrlKey);
					this->Interactor->SetShiftKey(shiftKey);
					this->Interactor->SetKeyCode(keyCode);
					this->Interactor->SetRepeatCount(repeatCount);
					this->Interactor->SetKeySym(keySym);
					
					this->Interactor->InvokeEvent(ievent, NULL);
				}
			}
		}
    }
	
	stopped = false;
	this->State = vtkInteractorEventRecorder::Start;
	//
	//  Clear current button press events
	//
	this->Interactor->SetControlKey(0);
	this->Interactor->SetShiftKey(0);
	this->Interactor->SetKeyCode(0);
	this->Interactor->SetRepeatCount(0);
	this->Interactor->SetKeySym(0);

	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("LeftButtonReleaseEvent"),NULL);
	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("MiddleButtonReleaseEvent"),NULL);
	this->Interactor->InvokeEvent(vtkCommand::GetEventIdFromString("RightButtonReleaseEvent"),NULL);

}


void iInteractorEventRecorder::Stop()
{
	if(this->State == vtkInteractorEventRecorder::Playing) 
	{
		stopped = true;
//		if(this->OutputStream != 0) this->OutputStream->seekp(this->OutputStream->tellg());
	}
	if(this->State == vtkInteractorEventRecorder::Recording) 
	{
//		if(this->OutputStream != 0) this->OutputStream->seekg(this->OutputStream->tellp());
	}
	this->State = vtkInteractorEventRecorder::Start;
	this->Modified();
}


void iInteractorEventRecorder::Rewind()
{
//	if(this->InputStream != 0) //need to already have an open file
//	{
//		this->InputStream->clear();
//		this->InputStream->seekg(0);
//	}
	if(this->OutputStream != 0) //need to already have an open file
	{
		this->OutputStream->clear();
		((fstream *)this->OutputStream)->seekp(0);
		((fstream *)this->OutputStream)->seekg(0);
	}
}

#endif

