/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/****************************************************************************

  Class containing data limits used in GUI and VTK.

****************************************************************************/

#ifndef ILIMITS_H
#define ILIMITS_H


#include "iglobals.h"
#include "istring.h"
#include "imath.h"


//
//  Function declaration macro
//
#define ILIMITS_FUNCTIONS_DECLARATIONS(label,Label,LABEL) \
inline int get##Label##Stretch(int n=1){ if(n>=0 && n<=n##label) return label##Stretch[n]; else return 0; } \
inline float get##Label##Max(int n=1){ if(n>=0 && n<=n##label) return label##Max[n]; else return  1.0e9; } \
inline float get##Label##Min(int n=1){ if(n>=0 && n<=n##label) return label##Min[n]; else return -1.0e9; } \
inline float get##Label##High(int n=1){ if(n>0 && n<=n##label) return label##High[n]; else return (float)1.0; } \
inline float get##Label##Low(int n=1){ if(n>0 && n<=n##label) return label##Low[n]; else return (float)0.0; } \
inline iString get##Label##Name(int n=1){ if(n>=0 && n<=N##LABEL##MAX) return label##Name[n]; else return iString(""); } \
inline int getN##label(){ return n##label; } \
inline void set##Label##Stretch(int n, int v){ if(n>0 && n<=n##label) { label##Stretch[n] = v; this->restore##Label##Limits(); } } \
inline void set##Label##Max(int n, float v){ if(n>0 && n<=n##label) { label##Max[n] = v; if(label##High[n] > v) label##High[n] = v; if(label##Low[n] > v) label##Low[n] = v; } } \
inline void set##Label##Min(int n, float v){ if(n>0 && n<=n##label) { label##Min[n] = v; if(label##High[n] < v) label##High[n] = v; if(label##Low[n] < v) label##Low[n] = v; } } \
inline void set##Label##MaxSaved(int n, float v){ if(n>0 && n<=N##LABEL##MAX) label##MaxSaved[n] = v; } \
inline void set##Label##MinSaved(int n, float v){ if(n>0 && n<=N##LABEL##MAX) label##MinSaved[n] = v; } \
inline void set##Label##Name(int n, iString v){ if(n>0 && n<=N##LABEL##MAX) label##Name[n] = v; } \
inline void set##Label##NameSaved(int n, iString v){ if(n>0 && n<=N##LABEL##MAX) label##NameSaved[n] = v; } \
inline void setN##label(int n){ if(n>=0 && n<=N##LABEL##MAX) n##label = n; } \
virtual void set##Label##High(int n, float v) \
{ \
	if(n>0 && n<=n##label) \
    { \
		v = ( v <= label##Max[n] ) ? v : label##Max[n]; \
		v = ( v > label##Low[n]) ? v : label##Low[n]; \
		label##High[n] = v; \
	} \
} \
virtual void set##Label##Low(int n, float v) \
{ \
	if(n>0 && n<=n##label) \
    { \
		v = ( v >= label##Min[n] ) ? v : label##Min[n]; \
		v = ( v < label##High[n]) ? v : label##High[n]; \
		label##Low[n] = v; \
	} \
} \
virtual void restore##Label##Names() \
{ \
	for(int i=0; i<=N##LABEL##MAX; i++) label##Name[i] = label##NameSaved[i]; \
} \
virtual void restore##Label##Limits() \
{ \
    for(int i=0; i<=N##LABEL##MAX; i++) \
    { \
		label##Max[i] = label##MaxSaved[i]; \
		label##Min[i] = label##MinSaved[i]; \
	    label##Min[i] = this->stretch##Label(label##Min[i],i); label##Max[i] = this->stretch##Label(label##Max[i],i); \
		this->set##Label##High(i,label##High[i]); this->set##Label##Low(i,label##Low[i]); \
		if(fabs(label##High[i]-label##Low[i])/(1.0e-30+fabs(label##High[i])+fabs(label##Low[i])) < 1.0e-5) \
		{ \
			label##Low[i] = label##Min[i]; label##High[i] = label##Max[i]; \
		} \
	} \
} \
inline float stretch##Label(float f, int i, bool big = false) \
{ \
	if(i>0 && i<=N##LABEL##MAX) return iLimits::stretch(f,label##Stretch[i],big); else return f; \
} \
inline float unstretch##Label(float f, int i) \
{ \
	if(i>0 && i<=N##LABEL##MAX) return iLimits::unstretch(f,label##Stretch[i]); else	return f; \
}




#define ILIMITS_VARIABLES_DECLARATIONS(label,Label,LABEL) \
int n##label; \
int label##Stretch[N##LABEL##MAX+1]; \
float label##Max[N##LABEL##MAX+1]; \
float label##Min[N##LABEL##MAX+1]; \
float label##MaxSaved[N##LABEL##MAX+1]; \
float label##MinSaved[N##LABEL##MAX+1]; \
float label##Low[N##LABEL##MAX+1]; \
float label##High[N##LABEL##MAX+1]; \
iString label##Name[N##LABEL##MAX+1]; \
iString label##NameSaved[N##LABEL##MAX+1];
	
#define ILIMITS_EXTEND_DECLARATIONS(Label)\
public: \
inline bool isExtend##Label() { return do##Label; } \
inline void setExtend##Label(bool s) { do##Label = s; } \
protected: \
bool do##Label;


#include "iobject.h"
#include "idatachannelsource.h"

class iLimits : public iObject, protected iDataChannelSource
{

public:
	
	iLimits(iVTK *m);
	virtual ~iLimits();

	static float stretch(float f, int mode, bool big = false);
	static float unstretch(float f, int mode);

	//
	//  Mesh variables
	//
	ILIMITS_FUNCTIONS_DECLARATIONS(var,Var,VAR)
	//
	//  Particle attributes
	//
	ILIMITS_FUNCTIONS_DECLARATIONS(att,Att,ATT)
	//
	//  Vector fields
	//
	ILIMITS_FUNCTIONS_DECLARATIONS(vec,Vec,VEC)
	//
	//  Tensor fields
	//
	ILIMITS_FUNCTIONS_DECLARATIONS(ten,Ten,TEN)

	inline int getExtendDown(int d){ if(d>=0 && d<3 && periodic[d]) return exDown[d]; else return 0; }
	inline int getExtendUp(int d){ if(d>=0 && d<3 && periodic[d]) return exUp[d]; else return 0; }

	inline void setExtendDown(int d, int v){ if(d>=0 && d<3 && v<=0) exDown[d] = v; }
	inline void setExtendUp(int d, int v){ if(d>=0 && d<3 && v>=0) exUp[d] = v; }

	inline iString getLabelName(){ return labelName; }
	inline iString getLabelUnit(){ return labelUnit; }
	inline  float getLabelScale(){ return labelScale; }

	inline void setLabelName(iString s){ labelName = s; }
	inline void setLabelUnit(iString s){ labelUnit = s; }
	inline void setLabelScale(float v){ labelScale = v; }

	virtual void setVoxelLocation(int v){ if(v>=0 && v<2) voxelLocation = v; }
	inline int getVoxelLocation(){ return voxelLocation; }

	bool periodicBC;
	bool periodic[3];

	virtual void packState(iString &s);
	virtual void unpackState(iString s);

ILIMITS_EXTEND_DECLARATIONS(Surf)
ILIMITS_EXTEND_DECLARATIONS(Volv)
ILIMITS_EXTEND_DECLARATIONS(Part)
ILIMITS_EXTEND_DECLARATIONS(Vect)
ILIMITS_EXTEND_DECLARATIONS(Tens)

protected:
	
	virtual bool edition();

	//
	//  Mesh variables
	//
	ILIMITS_VARIABLES_DECLARATIONS(var,Var,VAR)
	//
	//  Particle attributes
	//
	ILIMITS_VARIABLES_DECLARATIONS(att,Att,ATT)
	//
	//  Vector fields
	//
	ILIMITS_VARIABLES_DECLARATIONS(vec,Vec,VEC)
	//
	//  Tensor fields
	//
	ILIMITS_VARIABLES_DECLARATIONS(ten,Ten,TEN)

	int exDown[3], exUp[3];

	int voxelLocation;

	iString labelName, labelUnit;
	float labelScale;

};

//
//  stretch and un-stretch the data
//
inline float iLimits::stretch(float f, int mode, bool big)
{
	switch(mode)
	{
	case 1:
		{
			if(f > 0.0) 
			{
				return log10(1.0e-30+f); 
			}
			else 
			{
				return (big ? 30.0 : -30.0);
			}
			break;
		}
	default: return f;
	}
}


inline float iLimits::unstretch(float f, int mode)
{
	switch(mode)
	{
	case 1: 
		{
			return pow10(f);
			break;
		}
	default: return f;
	}
}


#endif // ILIMITS_H
