/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include <stdio.h>

#include "iglobals.h"
#include "imeasuringbox.h"
#include "itextbitmap.h"
#include "itextvector.h"
#include "itextactor.h"
#include "ivtk.h"

#include <vtkProperty.h>
#include <vtkPolyDataMapper.h>
#include <vtkCubeSource.h>
#include <vtkRenderer.h>
#include <vtkCamera.h>


void reportNullPointer(int);


iMeasuringBox* iMeasuringBox::New()
{

	return new iMeasuringBox;

}

iMeasuringBox::iMeasuringBox()
{

	started = false;
	size = 0.25;
	scale = 1.0;

	cFactor1r = 0.78; cFactor1g = 0.89; cFactor1b = 0.82; cFactor2 = 0.0;

	cubeFrameMapper = vtkPolyDataMapper::New();
	if(cubeFrameMapper == 0) reportNullPointer(8301);
	cubeMapper = vtkPolyDataMapper::New();
	if(cubeMapper == 0) reportNullPointer(8302);
	cubeSource = vtkCubeSource::New();
	if(cubeSource == 0) reportNullPointer(8303);
	cubeFrame = vtkActor::New();
	if(cubeFrame == 0) reportNullPointer(8304);

	cubeSource->SetCenter(0.0,0.0,0.0);
	cubeSource->SetXLength(2.0);
	cubeSource->SetYLength(2.0);
	cubeSource->SetZLength(2.0);

	cubeMapper->SetInput(cubeSource->GetOutput());
	cubeFrameMapper->SetInput(cubeSource->GetOutput());

	cubeFrame->SetMapper(cubeFrameMapper);
	cubeFrame->GetProperty()->SetRepresentationToWireframe();
	cubeFrame->GetProperty()->SetLineWidth(2);
	cubeFrame->PickableOff();

	this->SetMapper(cubeMapper);
	this->GetProperty()->SetOpacity(0.2);
	this->GetProperty()->SetAmbient(0.5);
	this->GetProperty()->SetDiffuse(0.5);
	this->GetProperty()->SetSpecular(0.7);
	this->GetProperty()->SetSpecularPower(50);

	itext = iTextActor::New();
	if(itext == 0) reportNullPointer(8305);
	itext->setBold(true);
	itext->setPosition(0.5,0.03);
	itext->setHJustification(ITEXT_CENTER);

	this->setColor(0.0,0.0,0.0);

}


iMeasuringBox::~iMeasuringBox()
{

	cubeFrameMapper->Delete();
	cubeFrame->Delete();
	cubeMapper->Delete();
	cubeSource->Delete();
	itext->Delete();

}


void iMeasuringBox::setColor(float r, float g, float b)
{
	itext->setColor(r,g,b); 
	this->GetProperty()->SetColor(cFactor1r+cFactor2*r,cFactor1g+cFactor2*g,cFactor1b+cFactor2*b);
	cubeFrame->GetProperty()->SetColor(r,g,b);
}

	
void iMeasuringBox::setOpacity(float o)
{
	if(o > 0.0)
	{
		if(o > 1.0) o = 1.0;
		this->GetProperty()->SetOpacity(o);
	}
}

	
void iMeasuringBox::ReleaseGraphicsResources(vtkWindow *win)
{
	cubeFrame->ReleaseGraphicsResources(win);
	itext->ReleaseGraphicsResources(win);
	this->vtkOpenGLActor::ReleaseGraphicsResources(win);
}


int iMeasuringBox::RenderOverlay(vtkViewport* viewport)
{

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam==0 || cam->GetParallelProjection()==0) return 0;

	itext->RenderOverlay(viewport);

	return 1;

}


int iMeasuringBox::RenderOpaqueGeometry(vtkViewport* viewport)
{
	
	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam == 0) return 0;
	
	iVTK *vtk = iVTK::convertFromViewport(viewport);

	if(!started)
	{
		started = true;
		scale = cam->GetParallelScale();
	}
	
	if(vtk->getRenderingMagnification() == 1)
	{
		float s = size*cam->GetParallelScale()/scale;
		char t[256];
		sprintf(t,"Box size: %6.2g",s);
		
		itext->setText(t);
		
		this->SetScale(s);
		cubeFrame->SetUserMatrix(this->GetMatrix());
	}
	
	if(this->GetProperty()->GetOpacity() < 1.0) 
	{ 
	} 
	else 
	{
		vtkActor::RenderOpaqueGeometry(viewport);
	}
	itext->RenderOpaqueGeometry(viewport);
	cubeFrame->RenderOpaqueGeometry(viewport);
	
	return 1;
				
}


int iMeasuringBox::RenderTranslucentGeometry(vtkViewport* viewport)
{

	vtkCamera *cam = ((vtkRenderer *)viewport)->GetActiveCamera();
	if(cam == 0) return 0;
	
	if(!started)
	{
		started = true;
		scale = cam->GetParallelScale();
	}

	if(this->GetProperty()->GetOpacity() < 1.0) 
	{
//		this->SetScale(iVTK::convertFromViewport(viewport)->getRenderingMagnification()*size*cam->GetParallelScale()/scale);
		vtkActor::RenderTranslucentGeometry(viewport);
	}

	return 1;
	
}


void iMeasuringBox::setInitialScale(float s)
{
	started = true;
	scale = s;
}


// Get the length of the diagonal of the bounding box.
float iMeasuringBox::GetLength()
{
	return cubeFrame->GetLength();
}


