/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/****************************************************************************

 Classes containing palettes
 
****************************************************************************/

#include "imath.h"
#include "ipalette.h"
#include "istring.h"
#include "ipiecewisefunction.h"

#include <vtkColorTransferFunction.h> 
#include <vtkLookupTable.h>


void reportNullPointer(int);
//
//  Singleton
//
iPaletteList* iPaletteList::me = 0;


void iPaletteList::construct()
{
	if(me == 0)
	{
		me = new iPaletteList;
		if(me == 0) reportNullPointer(2603);
		me->createDefault();
	}
}


void iPaletteList::destruct()
{
	if(me != 0) delete me;
}
//
// Implementation of iPalette class
//
iPalette::iPalette()
{
	
	cf = vtkColorTransferFunction::New();
	if(cf == 0) reportNullPointer(8701);
	lt = vtkLookupTable::New();
	if(lt == 0) reportNullPointer(8702);
	
	funR = new iPiecewiseFunction;
	if(funR == 0) reportNullPointer(8703);
	funG = new iPiecewiseFunction;
	if(funG == 0) reportNullPointer(8704);
	funB = new iPiecewiseFunction;
	if(funB == 0) reportNullPointer(8705);

	this->update();
	
}


iPalette::~iPalette()
{
	
	cf->Delete(); 
	lt->Delete(); 

	delete funR;
	delete funG;
	delete funB;

}


void iPalette::setPiecewiseFunctions(iPiecewiseFunction *r, iPiecewiseFunction *g, iPiecewiseFunction *b)
{
	funR->copy(r);
	funG->copy(g);
	funB->copy(b);
	this->update();
}

	
iColor iPalette::getColor(int n)
{
	
	n = ( n < 0 ) ? 0 : n;
	n = ( n > 255 ) ? 255 : n;
	return iColor(round(255.0*r0[n]),round(255.0*g0[n]),round(255.0*b0[n]));
	
}


void iPalette::update()
{
	
	int i;
	float x;
	for(i=0; i<256; i++)
	{
		x = i/255.0;
		r0[i] = funR->getValue(x);
		g0[i] = funG->getValue(x);
		b0[i] = funB->getValue(x);
		cf->AddRGBPoint(i,r0[i],g0[i],b0[i]);
		lt->SetTableValue(i,r0[i],g0[i],b0[i]);
	}
	
}


void iPalette::fillLookupTable(vtkLookupTable *t)
{
	int i;
	for(i=0; i<256; i++)
	{
		t->SetTableValue(i,r0[i],g0[i],b0[i],1.0);
	}
}


void iPalette::copy(iPalette *p)
{

	iPiecewiseFunction *r, *g, *b;
	p->getPiecewiseFunctions(r,g,b);
	this->setPiecewiseFunctions(r,g,b);

}


void iPalette::reverse()
{
	
	int i, n; 
	float *x, *y, *x1, *y1;
	
	n = funR->getFunctionN();
	x1 = new float[n];
	if(x1 == 0) reportNullPointer(8710);
	y1 = new float[n];
	if(y1 == 0) reportNullPointer(8711);
	x = funR->getFunctionX();
	y = funR->getFunctionY();
	for(i=0; i<n; i++)
	{
		x1[i]  = 1.0 - x[i];
		y1[i] = y[i];
	}
	for(i=0; i<n; i++)
	{
		x[i] = x1[n-1-i];
		y[i] = y1[n-1-i];
	}
	delete [] x1;
	delete [] y1;
	
	n = funG->getFunctionN();
	x1 = new float[n];
	if(x1 == 0) reportNullPointer(8712);
	y1 = new float[n];
	if(y1 == 0) reportNullPointer(8713);
	x = funG->getFunctionX();
	y = funG->getFunctionY();
	for(i=0; i<n; i++)
	{
		x1[i]  = 1.0 - x[i];
		y1[i] = y[i];
	}
	for(i=0; i<n; i++)
	{
		x[i] = x1[n-1-i];
		y[i] = y1[n-1-i];
	}
	delete [] x1;
	delete [] y1;
	
	n = funB->getFunctionN();
	x1 = new float[n];
	if(x1 == 0) reportNullPointer(8714);
	y1 = new float[n];
	if(y1 == 0) reportNullPointer(8715);
	x = funB->getFunctionX();
	y = funB->getFunctionY();
	for(i=0; i<n; i++)
	{
		x1[i]  = 1.0 - x[i];
		y1[i] = y[i];
	}
	for(i=0; i<n; i++)
	{
		x[i] = x1[n-1-i];
		y[i] = y1[n-1-i];
	}
	delete [] x1;
	delete [] y1;

	this->update();
	
}
//
// Implementation of iPaletteList class
//
iPaletteList::iPaletteList()
{
	
	list = new iPalette[MAXPALETTE];
	if(list == 0) reportNullPointer(8720);
	names = new iString[MAXPALETTE];
	if(names == 0) reportNullPointer(8721);
	
	numpal = 0;

}


iPaletteList::~iPaletteList()
{
	
	delete[] list;
	delete[] names;
	
}


void iPaletteList::createDefault()
{

	if(numpal > 0) return;

	setPalette_Rainbow(numpal++);
	setPalette_Temperature(numpal++);
	setPalette_Bluewhite(numpal++);
	setPalette_Prizm(numpal++);
	setPalette_Starlight(numpal++);
	setPalette_Greenwhite(numpal++);
	setPalette_Bluered(numpal++);
	setPalette_Haze(numpal++);
	setPalette_Stern(numpal++);
	setPalette_Greyscale(numpal++);
	setPalette_3color(numpal++);
	setPalette_4color(numpal++);
	
	setPalette_Rainbow(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Temperature(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Bluewhite(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Prizm(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Starlight(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Greenwhite(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Bluered(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Haze(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Stern(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;
	setPalette_Greyscale(numpal); list[numpal].reverse(); names[numpal] += iString(" reversed"); numpal++;

}


iPalette* iPaletteList::getPalette(int n)
{
	
	n = ( n < 0 ) ? 0 : n;
	n = ( n >= numpal ) ? numpal-1 : n;
	return list+n;
	
}


iString* iPaletteList::getPaletteName(int n)
{
	
	n = ( n < 0 ) ? 0 : n;
	n = ( n >= numpal ) ? numpal-1 : n;
	return names+n;
	
}


int iPaletteList::getNumPalettes()
{ return numpal; }


void iPaletteList::setPaletteName(int n, iString s)
{
	if(n>=0 && n<numpal) names[n] = s;
}


bool iPaletteList::addEmptyPalette()
{
	if(numpal < MAXPALETTE)
	{
		names[numpal] = "Unnamed";
		iPalette *p = new iPalette;
		if(p == 0) reportNullPointer(8722);
		list[numpal].copy(p);
		delete p;
		numpal++;
		return true;
	} 
	else return false;
}


void iPaletteList::removePalette(int n)
{

	int i;
	for(i=n; i<numpal-1; i++)
	{
		list[i].copy(&list[i+1]);
		names[i] = names[i+1];
	}
	numpal--;

}


void iPaletteList::copy(iPaletteList *pl)
{

	numpal = pl->getNumPalettes();

	int i;
	for(i=0; i<numpal; i++)
	{
		list[i].copy(pl->getPalette(i));
		names[i] = *(pl->getPaletteName(i));
	}

}
//
//  Private functions for setting the default palettes
//
void iPaletteList::setPalette_Rainbow(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Rainbow";

	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.1);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,0.9);
	b->movFunctionPoint(1,1.0,0.9);

	r->addFunctionPoint(1.0/6.0,0.0);
	g->addFunctionPoint(1.0/6.0,0.0);
	b->addFunctionPoint(1.0/6.0,1.0);

	r->addFunctionPoint(2.0/6.0,0.0);
	g->addFunctionPoint(2.0/6.0,1.0);
	b->addFunctionPoint(2.0/6.0,1.0);

	r->addFunctionPoint(3.0/6.0,0.0);
	g->addFunctionPoint(3.0/6.0,1.0);
	b->addFunctionPoint(3.0/6.0,0.0);

	r->addFunctionPoint(4.0/6.0,1.0);
	g->addFunctionPoint(4.0/6.0,1.0);
	b->addFunctionPoint(4.0/6.0,0.0);

	r->addFunctionPoint(5.0/6.0,1.0);
	g->addFunctionPoint(5.0/6.0,0.0);
	b->addFunctionPoint(5.0/6.0,0.0);
	
	list[n].update();
	
}


void iPaletteList::setPalette_Temperature(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Temperature";
	
	r->movFunctionPoint(0,0.0,0.3);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,0.8);
	b->movFunctionPoint(1,1.0,0.8);

	r->addFunctionPoint(0.7,1.0);
	g->addFunctionPoint(0.6,0.0);
	b->addFunctionPoint(0.8,0.0);
	
	list[n].update();
	
}


void iPaletteList::setPalette_Greyscale(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Greyscale";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,1.0);
	b->movFunctionPoint(1,1.0,1.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Bluewhite(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Blue-white";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.1);

	r->movFunctionPoint(1,1.0,0.9);
	g->movFunctionPoint(1,1.0,0.9);
	b->movFunctionPoint(1,1.0,1.0);

	r->addFunctionPoint(0.75,0.0);
	g->addFunctionPoint(0.38,0.0);
	b->addFunctionPoint(0.78,1.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Prizm(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Prizm";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,0.0);
	g->movFunctionPoint(1,1.0,0.0);
	b->movFunctionPoint(1,1.0,0.0);

	r->addFunctionPoint(0.25,1.0);
	g->addFunctionPoint(0.25,0.0);
	b->addFunctionPoint(0.25,0.0);

	r->addFunctionPoint(0.50,0.0);
	g->addFunctionPoint(0.50,1.0);
	b->addFunctionPoint(0.50,0.0);

	r->addFunctionPoint(0.75,0.0);
	g->addFunctionPoint(0.75,0.0);
	b->addFunctionPoint(0.75,1.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Greenwhite(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Green-white";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.1);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,1.0);
	b->movFunctionPoint(1,1.0,1.0);

	r->addFunctionPoint(0.375,0.0);
	b->addFunctionPoint(0.750,0.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Bluered(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Blue-red";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,0.0);
	b->movFunctionPoint(1,1.0,0.0);

	r->addFunctionPoint(0.25,0.0);
	g->addFunctionPoint(0.25,1.0);
	b->addFunctionPoint(0.25,1.0);

	r->addFunctionPoint(0.5,0.0);
	g->addFunctionPoint(0.5,0.0);
	b->addFunctionPoint(0.5,1.0);

	r->addFunctionPoint(0.75,1.0);
	g->addFunctionPoint(0.75,0.0);
	b->addFunctionPoint(0.75,1.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Stern(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Stern";
	
	r->movFunctionPoint(0,0.0,0.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,1.0);
	b->movFunctionPoint(1,1.0,1.0);

	r->addFunctionPoint(0.05,1.0);
	r->addFunctionPoint(0.25,0.0);
	b->addFunctionPoint(0.50,1.0);
	b->addFunctionPoint(0.75,0.0);

	list[n].update();
	
}


void iPaletteList::setPalette_Haze(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Haze";
	
	r->movFunctionPoint(0,0.0,1.0);
	g->movFunctionPoint(0,0.0,0.8);
	b->movFunctionPoint(0,0.0,1.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,0.8);
	b->movFunctionPoint(1,1.0,0.0);

	r->addFunctionPoint(0.5,0.0);
	g->addFunctionPoint(0.5,0.1);
	b->addFunctionPoint(0.5,0.5);

	list[n].update();
	
}


void iPaletteList::setPalette_Starlight(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "Starlight";
	
	r->movFunctionPoint(0,0.0,0.5);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,1.0);
	g->movFunctionPoint(1,1.0,1.0);
	b->movFunctionPoint(1,1.0,0.7);

	r->addFunctionPoint(0.5,0.9);
	g->addFunctionPoint(0.5,0.7);
	b->addFunctionPoint(0.5,0.2);

	list[n].update();
	
}


void iPaletteList::setPalette_3color(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "3 color";
	
	r->movFunctionPoint(0,0.0,1.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,0.0);
	g->movFunctionPoint(1,1.0,0.0);
	b->movFunctionPoint(1,1.0,1.0);

	r->addFunctionPoint(0.3333,1.0);
	g->addFunctionPoint(0.3333,0.0);
	b->addFunctionPoint(0.3333,0.0);
	r->addFunctionPoint(0.3334,0.0);
	g->addFunctionPoint(0.3334,1.0);
	b->addFunctionPoint(0.3334,0.0);

	r->addFunctionPoint(0.6666,0.0);
	g->addFunctionPoint(0.6666,1.0);
	b->addFunctionPoint(0.6666,0.0);
	r->addFunctionPoint(0.6667,0.0);
	g->addFunctionPoint(0.6667,0.0);
	b->addFunctionPoint(0.6667,1.0);

	list[n].update();
	
}


void iPaletteList::setPalette_4color(int n)
{
	iPiecewiseFunction *r, *g, *b;
	list[n].getPiecewiseFunctions(r,g,b);
	names[n] = "4 color";
	
	r->movFunctionPoint(0,0.0,1.0);
	g->movFunctionPoint(0,0.0,0.0);
	b->movFunctionPoint(0,0.0,0.0);

	r->movFunctionPoint(1,1.0,0.0);
	g->movFunctionPoint(1,1.0,0.0);
	b->movFunctionPoint(1,1.0,1.0);

	r->addFunctionPoint(0.24,1.0);
	g->addFunctionPoint(0.24,0.0);
	b->addFunctionPoint(0.24,0.0);
	r->addFunctionPoint(0.25,1.0);
	g->addFunctionPoint(0.25,1.0);
	b->addFunctionPoint(0.25,0.0);

	r->addFunctionPoint(0.49,1.0);
	g->addFunctionPoint(0.49,1.0);
	b->addFunctionPoint(0.49,0.0);
	r->addFunctionPoint(0.50,0.0);
	g->addFunctionPoint(0.50,1.0);
	b->addFunctionPoint(0.50,0.0);

	r->addFunctionPoint(0.74,0.0);
	g->addFunctionPoint(0.74,1.0);
	b->addFunctionPoint(0.74,0.0);
	r->addFunctionPoint(0.75,0.0);
	g->addFunctionPoint(0.75,0.0);
	b->addFunctionPoint(0.75,1.0);

	list[n].update();
	
}


