/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparallelizator.h"

#include "ivtk.h"
#include "iqt.h"
#include "iqtwindow.h"

#include <vtkMultiProcessController.h>
#include <vtkTimerLog.h>


vtkMultiProcessController* iParallelizator::controller = 0;
int iParallelizator::minNumberOfProcessors = 1;
int iParallelizator::maxNumberOfProcessors = 1;
vtkTimerLog **iParallelizator::timer = NULL;
vtkTimerLog *iParallelizator::timer2 = NULL;
void (*iParallelizator::function)(vtkMultiProcessController *,void *) = NULL;


void reportNullPointer(int);


void iParallelizator::construct()
{
	if(controller == 0)
	{
		controller = vtkMultiProcessController::New();
		if(controller == 0) reportNullPointer(3001);
		int argc = 0;
		char **argv = 0;
		controller->Initialize(&argc,&argv);

		if(controller->IsA("vtkThreadedController"))
		{
			minNumberOfProcessors = maxNumberOfProcessors = 1;
		}
		
		if(controller->IsA("vtkMPIController"))
		{
			minNumberOfProcessors = maxNumberOfProcessors = controller->GetNumberOfProcesses();
		}

		timer2 = vtkTimerLog::New();

		timer = new vtkTimerLog*[maxNumberOfProcessors];
		for(int i=0; i<maxNumberOfProcessors; i++) timer[i] = vtkTimerLog::New();
		function = NULL;
	}
}


void iParallelizator::destruct()
{
	if(controller != 0) 
	{
		controller->Finalize();
		controller->Delete();
		timer2->Delete();
		for(int i=0; i<maxNumberOfProcessors; i++) timer[i]->Delete();
		delete [] timer;
	}
}


void iParallelizator::execute(void (*f)(vtkMultiProcessController *,void *), void *data)
{
	function = f;

	timer2->StartTimer();

	if(controller->GetNumberOfProcesses() > 1)
	{
		controller->SetSingleMethod(iParallelizator::timedSingleMethod,data);
		controller->SingleMethodExecute();
	}
	else
	{
		f(NULL,data);
	}

	timer2->StopTimer();

	iQTWindow::getCurrentWindow()->updateParallelDisplay();

}


int iParallelizator::getMinNumberOfProcessors()
{ 
	return minNumberOfProcessors; 
}


int iParallelizator::getMaxNumberOfProcessors()
{ 
	return maxNumberOfProcessors; 
}


int iParallelizator::getNumberOfProcessors()
{ 
	int n = controller->GetNumberOfProcesses(); 
	if(n <= 0) return 1; else return n;
}


void iParallelizator::setNumberOfProcessors(int n)
{ 
	int i;
	if(n > maxNumberOfProcessors) 
	{
		for(i=0; i<maxNumberOfProcessors; i++) timer[i]->Delete();
		delete [] timer;
		maxNumberOfProcessors = n;
		timer = new vtkTimerLog*[maxNumberOfProcessors];
		for(i=0; i<maxNumberOfProcessors; i++) timer[i] = vtkTimerLog::New();
	}
	controller->SetNumberOfProcesses(n); 
}


void iParallelizator::timedSingleMethod(vtkMultiProcessController *controller, void *data)
{
	if(function == NULL) return;

	int myId;
	if(controller != NULL)
	{
		myId = controller->GetLocalProcessId();
	}
	else
	{
		myId = 0;
	}

	if(function != NULL)
	{
		timer[myId]->StartTimer();
		function(controller,data);
		timer[myId]->StopTimer();
	}

}


float iParallelizator::getLastProcessorExecutionTime(int n)
{
	if(controller->GetNumberOfProcesses() > 1)
	{
		if(n<maxNumberOfProcessors && timer!=NULL && timer[n]!=NULL) return timer[n]->GetElapsedTime(); else return 0.0;
	}
	else
	{
		return timer2->GetElapsedTime();
	}
}


float iParallelizator::getLastTotalExecutionTime()
{
	return timer2->GetElapsedTime();
}

