/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of iparticlessplitter.h
  
=========================================================================*/

#include "iparticlesconnector.h"
#include "imath.h"

#include <vtkObjectFactory.h>
#include <vtkFloatArray.h>
#include <vtkIdTypeArray.h>
#include <vtkPolyData.h>
#include <vtkCellArray.h>
#include <vtkPointData.h>


//------------------------------------------------------------------------------
iParticlesConnector* iParticlesConnector::New()
{
	return new iParticlesConnector;
}


// Construct object to extract all of the input data.
iParticlesConnector::iParticlesConnector()
{
	attCon = attSep = 0;
	connected = false;
}


iParticlesConnector::~iParticlesConnector()
{
}

	
void iParticlesConnector::setAttributeToConnect(int a)
{
	if(a>=0 && a!=attCon)
	{
		attCon = a;
		this->Modified();
	}
}


void iParticlesConnector::setAttributeToSeparate(int a)
{
	if(a>=0 && a!=attSep)
	{
		attSep = a;
		this->Modified();
	}
}


void iParticlesConnector::setConnected(bool s)
{
	connected = s;
	this->Modified();
}


#define EQTOL  1.001

void iParticlesConnector::ExecuteData(vtkDataObject *)
{

	vtkPolyData *input = this->GetInput();
	vtkPolyData *output = (vtkPolyData *)this->GetOutput();

	if(input == 0) return;

	output->Initialize();
	output->ShallowCopy(input);

	vtkFloatArray *inAtts = (vtkFloatArray *)input->GetPointData()->GetScalars();
	if(!connected || inAtts==0 || (attSep==attCon && attSep!=0)) return;

	vtkIdType np = input->GetNumberOfPoints();
	vtkIdType nc = inAtts->GetNumberOfComponents();

	if(attCon<0 || attCon>=nc || attSep<0 || attSep>=nc) return;

	vtkCellArray *olin = vtkCellArray::New();
	vtkIdType id[2];

	vtkIdType l;
	vtkIdType *ind = new vtkIdType[np];
	float *data = inAtts->GetPointer(0);

	if(attSep == 0)
	{
		this->index1(np,nc,attCon,data,ind);
		for(l=0; l<np-1; l++)
		{
			if(l%1000 == 0)
			{
				this->UpdateProgress((float)l/np);
				if(this->GetAbortExecute()) break;
			}
			id[0] = ind[l];
			id[1] = ind[l+1];
			olin->InsertNextCell(2,id);
		}
	}
	else
	{
		this->index2(np,nc,attSep,attCon,data,ind);
		for(l=0; l<np-1; l++)
		{
			if(l%1000 == 0)
			{
				this->UpdateProgress((float)l/np);
				if(this->GetAbortExecute()) break;
			}
			if(data[attSep+nc*ind[l]]*EQTOL > data[attSep+nc*ind[l+1]])
			{
				id[0] = ind[l];
				id[1] = ind[l+1];
				olin->InsertNextCell(2,id);
			}
		}

	}

	delete [] ind;
	output->SetLines(olin);
	olin->Delete();
	
}


#define SWAP(a,b) itemp=(a);(a)=(b);(b)=itemp;
#define M 7
#define NSTACK 50

void iParticlesConnector::index1(vtkIdType n, int nc, int att, float *arr, vtkIdType *indx)
{
	vtkIdType i,indxt,ir=n,itemp,j,k,l=1;
	int jstack=0,*istack;
	float a;

	indx -= 1; // NR arrays start with 1

	istack = new int[1+NSTACK];
	for (j=1;j<=n;j++) indx[j]=j-1;
	for (;;) 
	{
		if(this->GetAbortExecute()) break;
		if (ir-l < M) 
		{
			for (j=l+1;j<=ir;j++) 
			{
				indxt=indx[j];
				a=arr[att+nc*indxt];
				for (i=j-1;i>=l;i--) 
				{
					if (arr[att+nc*indx[i]] <= a) break;
					indx[i+1]=indx[i];
				}
				indx[i+1]=indxt;
			}
			if (jstack == 0) break;
			ir=istack[jstack--];
			l=istack[jstack--];
		} 
		else 
		{
			k=(l+ir) >> 1;
			SWAP(indx[k],indx[l+1]);
			if (arr[att+nc*indx[l]] > arr[att+nc*indx[ir]]) 
			{
				SWAP(indx[l],indx[ir])
			}
			if (arr[att+nc*indx[l+1]] > arr[att+nc*indx[ir]]) 
			{
				SWAP(indx[l+1],indx[ir])
			}
			if (arr[att+nc*indx[l]] > arr[att+nc*indx[l+1]]) 
			{
				SWAP(indx[l],indx[l+1])
			}
			i=l+1;
			j=ir;
			indxt=indx[l+1];
			a=arr[att+nc*indxt];
			for (;;) 
			{
				do i++; while (arr[att+nc*indx[i]] < a);
				do j--; while (arr[att+nc*indx[j]] > a);
				if (j < i) break;
				SWAP(indx[i],indx[j])
			}
			indx[l+1]=indx[j];
			indx[j]=indxt;
			jstack += 2;
			if (jstack > NSTACK) vtkErrorMacro("NSTACK too small in iParticlesConnector::index1.");
			if (ir-i+1 >= j-l) 
			{
				istack[jstack]=ir;
				istack[jstack-1]=i;
				ir=j-1;
			} 
			else 
			{
				istack[jstack]=j-1;
				istack[jstack-1]=l;
				l=i;
			}
		}
	}
	delete [] istack;
}


#define GREATER(ind1,ind2)  (arr[att1+nc*ind1]>EQTOL*arr[att1+nc*ind2]||(arr[att1+nc*ind1]>arr[att1+nc*ind2]/EQTOL&&arr[att2+nc*ind1]>arr[att2+nc*ind2]))


void iParticlesConnector::index2(vtkIdType n, int nc, int att1, int att2, float *arr, vtkIdType *indx)
{
	vtkIdType i,indxt,ir=n,itemp,j,k,l=1;
	int jstack=0,*istack;

	indx -= 1; // NR arrays start with 1

	istack = new int[1+NSTACK];
	for (j=1;j<=n;j++) indx[j]=j-1;
	for (;;) 
	{
		if(this->GetAbortExecute()) break;
		if (ir-l < M) 
		{
			for (j=l+1;j<=ir;j++) 
			{
				indxt=indx[j];
				for (i=j-1;i>=l;i--) 
				{
					if (!GREATER(indx[i],indxt)) break;
					indx[i+1]=indx[i];
				}
				indx[i+1]=indxt;
			}
			if (jstack == 0) break;
			ir=istack[jstack--];
			l=istack[jstack--];
		} 
		else 
		{
			k=(l+ir) >> 1;
			SWAP(indx[k],indx[l+1]);
			if (GREATER(indx[l],indx[ir])) 
			{
				SWAP(indx[l],indx[ir])
			}
			if (GREATER(indx[l+1],indx[ir])) 
			{
				SWAP(indx[l+1],indx[ir])
			}
			if (GREATER(indx[l],indx[l+1])) 
			{
				SWAP(indx[l],indx[l+1])
			}
			i=l+1;
			j=ir;
			indxt=indx[l+1];
			for (;;) 
			{
				do i++; while (!GREATER(indx[i],indxt));
				do j--; while (GREATER(indx[j],indxt));
				if (j < i) break;
				SWAP(indx[i],indx[j])
			}
			indx[l+1]=indx[j];
			indx[j]=indxt;
			jstack += 2;
			if (jstack > NSTACK) vtkErrorMacro("NSTACK too small in iParticlesConnector::index2.");
			if (ir-i+1 >= j-l) 
			{
				istack[jstack]=ir;
				istack[jstack-1]=i;
				ir=j-1;
			} 
			else 
			{
				istack[jstack]=j-1;
				istack[jstack-1]=l;
				l=i;
			}
		}
	}
	delete [] istack;
}
