/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*=========================================================================

  Implementation of ipolydataextender.h
  
=========================================================================*/


#include "ilimits.h"
#include "ivisualobject.h"

#include "ipolygonaldataextender.h"
#include "ipolygonaldata.h"

#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkCellArray.h>


void reportNullPointer(int);


//------------------------------------------------------------------------------
iPolygonalDataExtender* iPolygonalDataExtender::New(iVisualObject *m)
{
	return new iPolygonalDataExtender(m);
}


iPolygonalDataExtender::iPolygonalDataExtender(iVisualObject *m) : iVisualObjectSource(m)
{
	active = false;
}


void iPolygonalDataExtender::ExecuteData(vtkDataObject *)
{
	iPolygonalData *input = (iPolygonalData *)this->GetInput();
	iPolygonalData *output = (iPolygonalData *)this->GetOutput();
	int ext[3], extDown[3], extUp[3];
	int i,i1,j1,k1;

	if(input == 0) return;

	output->ShallowCopy(input);

	bool work = false;
	for(i=0; i<3; i++)
	{
		extDown[i] = myObject->getLimits()->getExtendDown(i);
		extUp[i] = myObject->getLimits()->getExtendUp(i);
		if(extDown[i] != 0) work = true;
		if(extUp[i] != 0) work = true;
	}

	if(!work || !active) 
	{
		output->ShallowCopy(input);
		return;
	}

	int next = 1;
	for(i=0; i<3; i++)
	{
		ext[i] = extUp[i] - extDown[i] + 1;
		next *= ext[i];
	}

	vtkPoints *ipoi = input->GetPoints();
	if(ipoi == 0) return;
	vtkCellArray *iver = input->GetVerts();
	vtkCellArray *ilin = input->GetLines();
	vtkCellArray *ipol = input->GetPolys();
	vtkCellArray *isrp = input->GetStrips();
	vtkDataArray *isca = input->GetPointData()->GetScalars();
	vtkDataArray *inor = input->GetPointData()->GetNormals();

	float  *iptrpoiF = 0;
	double *iptrpoiD = 0;
//	char *iptrver = (iver == 0) ? 0 : (char *)iver->GetPointer();
//	char *iptrlin = (ilin == 0) ? 0 : (char *)ilin->GetPointer();
//	char *iptrpol = (ipol == 0) ? 0 : (char *)ipol->GetPointer();
//	char *iptrsrp = (isrp == 0) ? 0 : (char *)isrp->GetPointer();
	float *iptrsca = (isca == 0) ? 0 : (float *)isca->GetVoidPointer(0);
	float *iptrnor = (inor == 0) ? 0 : (float *)inor->GetVoidPointer(0);

	vtkIdType npoi = ipoi->GetNumberOfPoints();
	if(npoi == 0) return;
	vtkIdType nver = (iver == 0) ? 0 : iver->GetNumberOfCells();
	vtkIdType nlin = (ilin == 0) ? 0 : ilin->GetNumberOfCells();
	vtkIdType npol = (ipol == 0) ? 0 : ipol->GetNumberOfCells();
	vtkIdType nsrp = (isrp == 0) ? 0 : isrp->GetNumberOfCells();
	vtkIdType nsca = (isca == 0) ? 0 : isca->GetNumberOfComponents()*isca->GetNumberOfTuples();
	vtkIdType nnor = (inor == 0) ? 0 : inor->GetNumberOfComponents()*inor->GetNumberOfTuples();
	
	vtkPoints *opoi = vtkPoints::New(ipoi->GetDataType());
	if(opoi == 0) reportNullPointer(7101);
	opoi->SetNumberOfPoints(next*npoi);

	float  *optrpoiF = 0;
	double *optrpoiD = 0;
	float *optrsca = 0, *optrnor = 0;
	vtkCellArray *over = 0, *olin = 0, *opol = 0, *osrp = 0;
	vtkFloatArray *osca = 0, *onor = 0;

	bool pointsAreFloat;
	switch(ipoi->GetDataType())
	{
	case VTK_FLOAT:
		{
			pointsAreFloat = true;
			iptrpoiF = (float  *)ipoi->GetVoidPointer(0);
			optrpoiF = (float  *)opoi->GetVoidPointer(0);
			break;
		}
	case VTK_DOUBLE:
		{
			pointsAreFloat = false;
			iptrpoiD = (double *)ipoi->GetVoidPointer(0);
			optrpoiD = (double *)opoi->GetVoidPointer(0);
			break;
		}
	default: 
		{
			vtkErrorMacro("Incorrect Points type");
			return;
		}
	}

	if(nver > 0) 
	{
		over = vtkCellArray::New();
		if(over == 0) reportNullPointer(7102);
	}
	if(nlin > 0) 
	{
		olin = vtkCellArray::New();
		if(olin == 0) reportNullPointer(7103);
	}
	if(npol > 0) 
	{
		opol = vtkCellArray::New();
		if(opol == 0) reportNullPointer(7104);
	}
	if(nsrp > 0) 
	{
		osrp = vtkCellArray::New();
		if(osrp == 0) reportNullPointer(7105);
	}
	if(nsca > 0) 
	{
		osca = vtkFloatArray::New();
		if(osca == 0) reportNullPointer(7106);
		osca->SetNumberOfComponents(isca->GetNumberOfComponents());
		osca->SetNumberOfTuples(next*isca->GetNumberOfTuples());
		optrsca = (float *)osca->GetVoidPointer(0);
	}
	if(nnor > 0) 
	{
		onor = vtkFloatArray::New();
		if(onor == 0) reportNullPointer(7107);
		onor->SetNumberOfComponents(3);
		onor->SetNumberOfTuples(next*inor->GetNumberOfTuples());
		optrnor = (float *)onor->GetVoidPointer(0);
	}

	vtkIdType npts, *pts, *pts1;

	int maxCellSize = 0, curmax;
	if(nver > 0)
	{
		curmax = iver->GetMaxCellSize();
		if(curmax > maxCellSize) maxCellSize = curmax;
	}
	if(nlin > 0)
	{
		curmax = ilin->GetMaxCellSize();
		if(curmax > maxCellSize) maxCellSize = curmax;
	}
	if(npol > 0)
	{
		curmax = ipol->GetMaxCellSize();
		if(curmax > maxCellSize) maxCellSize = curmax;
	}
	if(nsrp > 0)
	{
		curmax = isrp->GetMaxCellSize();
		if(curmax > maxCellSize) maxCellSize = curmax;
	}

	pts = new vtkIdType[maxCellSize];
	if(pts == 0) reportNullPointer(7108);

	int off = 0;
	vtkIdType l, loff1, loff2, noff;
	int done = 0;
	for(i1=extDown[0]; i1<=extUp[0]; i1++)
	{
		for(j1=extDown[1]; j1<=extUp[1]; j1++)
		{
			for(k1=extDown[2]; k1<=extUp[2]; k1++)
			{
				
				done++;
				this->UpdateProgress((float)done/next);
				if(this->GetAbortExecute()) break;

				noff = off*npoi;
				if(pointsAreFloat)
				{
					for(l=0; l<npoi; l++)
					{
						loff2 = 3*l;
						loff1 = 3*noff + loff2;
						optrpoiF[loff1+0] = iptrpoiF[loff2+0] + 2.0*i1;
						optrpoiF[loff1+1] = iptrpoiF[loff2+1] + 2.0*j1;
						optrpoiF[loff1+2] = iptrpoiF[loff2+2] + 2.0*k1;
					}
				}
				else
				{
					for(l=0; l<npoi; l++)
					{
						loff2 = 3*l;
						loff1 = 3*noff + loff2;
						optrpoiD[loff1+0] = iptrpoiD[loff2+0] + 2.0*i1;
						optrpoiD[loff1+1] = iptrpoiD[loff2+1] + 2.0*j1;
						optrpoiD[loff1+2] = iptrpoiD[loff2+2] + 2.0*k1;
					}
				}
				if(nver > 0) for(iver->InitTraversal(); iver->GetNextCell(npts,pts1)!=0; )
				{
					pts[0] = pts1[0] + noff;
					over->InsertNextCell(1,pts);
				}
				if(nlin > 0) for(ilin->InitTraversal(); ilin->GetNextCell(npts,pts1)!=0; )
				{
					for(i=0; i<npts && i<maxCellSize; i++) pts[i] = pts1[i] + noff;
					olin->InsertNextCell(npts,pts);
				}
				if(npol > 0) for(ipol->InitTraversal(); ipol->GetNextCell(npts,pts1)!=0; )
				{
					for(i=0; i<npts && i<maxCellSize; i++) pts[i] = pts1[i] + noff;
					opol->InsertNextCell(npts,pts);
				}
				if(nsrp > 0) for(isrp->InitTraversal(); isrp->GetNextCell(npts,pts1)!=0; )
				{
					for(i=0; i<npts && i<maxCellSize; i++) pts[i] = pts1[i] + noff;
					osrp->InsertNextCell(npts,pts);
				}
				if(nsca > 0) memcpy(optrsca+off*nsca,iptrsca,nsca*sizeof(float));
				if(nnor > 0) memcpy(optrnor+off*nnor,iptrnor,nnor*sizeof(float));
				off++;			}
		}
	}

	delete [] pts;

	output->SetPoints(opoi);
	opoi->Delete();

	if(nver > 0)
	{
		output->SetVerts(over);
		over->Delete();
	}
	if(nlin > 0)
	{
		output->SetLines(olin);
		olin->Delete();
	}
	if(npol > 0)
	{
		output->SetPolys(opol);
		opol->Delete();
	}
	if(nsrp > 0)
	{
		output->SetStrips(osrp);
		osrp->Delete();
	}
	if(nsca > 0)
	{
		output->GetPointData()->SetScalars(osca);
		osca->Delete();
	}
	if(nnor > 0)
	{
		output->GetPointData()->SetNormals(onor);
		onor->Delete();
	}

	this->Modified();

}


float iPolygonalDataExtender::getMemorySize()
{
	int i;
	bool work = false;
	//
	// Could be zero if the pipeline has not executed yet.
	//
	if(myObject->getLimits() == 0)
	{
		return 0.0;
	}

	for(i=0; i<3; i++)
	{
		if(myObject->getLimits()->getExtendDown(i) != 0) work = true;
		if(myObject->getLimits()->getExtendUp(i) != 0) work = true;
	}

	if(!work || !active) return 0.0;
	return this->GetOutput()->GetActualMemorySize();

}
