/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iqt.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iqtextension.h"

#include "iqtdefs.h"
#include "iqt_surf.h"
#include "iqt_xsec.h"
#include "iqt_volv.h"
#include "iqt_part.h"
#include "iqt_vect.h"
#include "iqt_tens.h"
#include "iqt_data.h"
#include "iqt_view.h"
#include "iqt_wins.h"

#include "idialogabout.h"
#include "idialoghelp.h"
#include "idatareader.h"
#include "idatainfo.h"
#include "ierror.h"
#include "iparticlessplitter.h"
#include "iuniformmeshdata.h"
#include "isurface.h"
#include "itensor.h"
#include "ivector.h"
#include "ivolume.h"
#include "ixsection.h"
#include "iparticles.h"
#include "idatagateway.h"
#include "iuniformmeshdataconverter.h"
#include "istreamline.h"
#include "iprogressbar.h"
#include "ipostscriptwriter.h"
#include "idialogloadfile.h"
#include "ivtkcallback.h"

#include <qaction.h>
#include <qapplication.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qfiledialog.h>
#include <qlayout.h>
#include <qprogressbar.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qtabwidget.h>
#include <qtextedit.h> 
#include <qwidgetstack.h>
#include <qworkspace.h>


#define CURRENT_LIMITS	iVTKWindow::getCurrentWindow()->getReader()->getLimits(iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataChannel())

#define CALL_VTK_FUNCTION1(fun,arg) \
{ \
	if(this->optionsAffectAll) \
	{ \
		int k; \
		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->fun(arg); \
	} \
	else \
	{ \
		iVTKWindow::getCurrentWindow()->fun(arg); \
	} \
}


QPixmap image(char *s);

// *********************************************************************************
//
//  
//   Menu actions
//
//
// *********************************************************************************
//
#include <qtooltip.h>

void iQT::fileOpenMesh()
{
	
    QString s = QFileDialog::getOpenFileName(iVTKWindow::getCurrentWindow()->getReader()->getCurrentMeshFileDir(),
		"IFRIT mesh files (*.bin *.txt)",
		this,
		"open IFRIT mesh file",
		"Choose a file" );

	this->fileOpenMesh(s);

}

	
void iQT::fileOpenMesh(const QString &s)
{
    int nold = CURRENT_LIMITS->getNvar();
	
    if(!s.isEmpty()) 
	{
		//	 
		// Filename is read. Send the event to VTK and write into the log.
		//
#if defined(IFRIT_NO_THREADED_FILE_LOAD) || !defined(QT_THREAD_SUPPORT)
		this->startTask();
		ProgressBar->reset();
		iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_MESH,ProgressBar);
		int res = iVTKWindow::getCurrentWindow()->getReader()->loadMeshFile(iString(s),false,false,true); 
		ProgressBar->reset();
		this->afterMeshFileRead(iVTKWindow::getCurrentWindowIndex(),nold,res,s);
		this->endTask(); 
#else
		iDialogLoadFile *lf = new iDialogLoadFile;
		connect(lf, SIGNAL( finished(iDataReader*,int,QString,int,int) ), this, SLOT( updateAfterFileRead(iDataReader*,int,QString,int,int) ) );
		lf->show(iVTKWindow::getCurrentWindow()->getReader(),IDATAREADER_MESH,s,false,false,true,nold);
#endif		
    }
	
}


void iQT::fileOpenPart()
{
	
    QString s = QFileDialog::getOpenFileName(iVTKWindow::getCurrentWindow()->getReader()->getCurrentPartFileDir(),
		"IFRIT particle files (*.bin *.txt)",
		this,
		"open IFRIT particle file",
		"Choose a file" );

	this->fileOpenPart(s);

}


void iQT::fileOpenPart(const QString &s)
{
    int mold = CURRENT_LIMITS->getNatt();
	
    if(!s.isEmpty()) 
	{
		//	 
		// Filename is read. Send the event to VTK and write into the log.
		//
#if defined(IFRIT_NO_THREADED_FILE_LOAD) || !defined(QT_THREAD_SUPPORT)
		this->startTask();
		ProgressBar->reset();
		iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_PART,ProgressBar);
		int res = iVTKWindow::getCurrentWindow()->getReader()->loadPartFile(iString(s),false,false,true); 
		ProgressBar->reset();
		this->afterPartFileRead(iVTKWindow::getCurrentWindowIndex(),mold,res,s);
		this->endTask();
#else
		iDialogLoadFile *lf = new iDialogLoadFile;
		connect(lf, SIGNAL( finished(iDataReader*,int,QString,int,int) ), this, SLOT( updateAfterFileRead(iDataReader*,int,QString,int,int) ) );
		lf->show(iVTKWindow::getCurrentWindow()->getReader(),IDATAREADER_PART,s,false,false,true,mold);
#endif		
    }
	
}


void iQT::fileOpenVect()
{
	
    QString s = QFileDialog::getOpenFileName(iVTKWindow::getCurrentWindow()->getReader()->getCurrentVectFileDir(),
		"IFRIT vector field files (*.bin *.txt)",
		this,
		"open IFRIT vector field file",
		"Choose a file" );

	this->fileOpenVect(s);

}

	
void iQT::fileOpenVect(const QString &s)
{
	
    if(!s.isEmpty()) 
	{
		//	 
		// Filename is read. Send the event to VTK and write into the log.
		//
#if defined(IFRIT_NO_THREADED_FILE_LOAD) || !defined(QT_THREAD_SUPPORT)
		this->startTask();
		ProgressBar->reset();
		iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_VECT,ProgressBar);
		int res = iVTKWindow::getCurrentWindow()->getReader()->loadVectFile(iString(s),false,false,true); 
		ProgressBar->reset();
		this->afterVectFileRead(iVTKWindow::getCurrentWindowIndex(),0,res,s);
		this->endTask();
#else		
		iDialogLoadFile *lf = new iDialogLoadFile;
		connect(lf, SIGNAL( finished(iDataReader*,int,QString,int,int) ), this, SLOT( updateAfterFileRead(iDataReader*,int,QString,int,int) ) );
		lf->show(iVTKWindow::getCurrentWindow()->getReader(),IDATAREADER_VECT,s,false,false,true,0);
#endif
    }
	
}


void iQT::fileOpenTens()
{
	
    QString s = QFileDialog::getOpenFileName(iVTKWindow::getCurrentWindow()->getReader()->getCurrentTensFileDir(),
		"IFRIT tensor field files (*.bin *.txt)",
		this,
		"open IFRIT tensor field file",
		"Choose a file" );
	
	this->fileOpenTens(s);

}


void iQT::fileOpenTens(const QString &s)
{

    if(!s.isEmpty()) 
	{
		//	 
		// Filename is read. Send the event to VTK and write into the log.
		//
#if defined(IFRIT_NO_THREADED_FILE_LOAD) || !defined(QT_THREAD_SUPPORT)
		this->startTask();
		ProgressBar->reset();
		iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_TENS,ProgressBar);
		int res = iVTKWindow::getCurrentWindow()->getReader()->loadTensFile(iString(s),false,false,true); 
		ProgressBar->reset();
		this->afterTensFileRead(iVTKWindow::getCurrentWindowIndex(),0,res,s);
		this->endTask();
#else		
		iDialogLoadFile *lf = new iDialogLoadFile;
		connect(lf, SIGNAL( finished(iDataReader*,int,QString,int,int) ), this, SLOT( updateAfterFileRead(iDataReader*,int,QString,int,int) ) );
		lf->show(iVTKWindow::getCurrentWindow()->getReader(),IDATAREADER_TENS,s,false,false,true,0);
#endif
    }
	
}


void iQT::fileOpenSet()
{
	
	QString dir;
	char setLeader = iVTKWindow::getCurrentWindow()->getReader()->getSetLeader();
	switch (setLeader) 
	{
	case 'a':	{ dir = iVTKWindow::getCurrentWindow()->getReader()->getCurrentMeshFileDir(); break; }
	case 'd':	{ dir = iVTKWindow::getCurrentWindow()->getReader()->getCurrentPartFileDir(); break; }
	case 'b':	{ dir = iVTKWindow::getCurrentWindow()->getReader()->getCurrentVectFileDir(); break; }
	case 'c':	{ dir = iVTKWindow::getCurrentWindow()->getReader()->getCurrentTensFileDir(); break; }
	}
    QString s = QFileDialog::getOpenFileName(dir,
		"IFRIT files (*.bin *.txt)",
		this,
		"open IFRIT file",
		"Choose a file" );

	this->fileOpenSet(s);

}


void iQT::fileOpenSet(const QString &s)
{
    int nold = CURRENT_LIMITS->getNvar();
    int mold = CURRENT_LIMITS->getNatt();

    if(!s.isEmpty() && QFile::exists(s)) 
	{
		//	 
		// Filename is read. Send the event to VTK and write into the log.
		//
#if defined(IFRIT_NO_THREADED_FILE_LOAD) || !defined(QT_THREAD_SUPPORT)
		this->startTask();
		ProgressBar->reset();
		iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_ALL,ProgressBar);
		int res = iVTKWindow::getCurrentWindow()->getReader()->loadFileSet(s,true); 
		ProgressBar->reset();
		
		if(res == 0) this->updateVisitedFilesList(false);
		
		TextLog->append("<hr>");
		
		switch (res) 
		{
		case 0: { logLine("File set read",s); break; }
		case 11: { logLine("Error in reading file set","no file set is established.",QColor(255,0,0)); break; }
		case 12: { logLine("Error in reading file set","file does not belong to a set.",QColor(255,0,0)); break; }
		case -1: { logLine("Error in reading file set","incorrect file type.",QColor(255,0,0)); break; }
		default: { logLine("Error in reading file set","data file is corrupted.",QColor(255,0,0)); break; }
		}
		
		if(iVTKWindow::getCurrentWindow()->getReader()->isMeshFileBelongToSet()) this->afterMeshFileRead(iVTKWindow::getCurrentWindowIndex(),nold,res,s);
		if(iVTKWindow::getCurrentWindow()->getReader()->isPartFileBelongToSet()) this->afterPartFileRead(iVTKWindow::getCurrentWindowIndex(),mold,res,s);
		if(iVTKWindow::getCurrentWindow()->getReader()->isVectFileBelongToSet()) this->afterVectFileRead(iVTKWindow::getCurrentWindowIndex(),0,res,s);
		if(iVTKWindow::getCurrentWindow()->getReader()->isTensFileBelongToSet()) this->afterTensFileRead(iVTKWindow::getCurrentWindowIndex(),0,res,s);
		this->endTask(); 
#else
		iDialogLoadFile *lf = new iDialogLoadFile;
		connect(lf, SIGNAL( finished(iDataReader*,int,QString,int,int) ), this, SLOT( updateAfterFileRead(iDataReader*,int,QString,int,int) ) );
		lf->show(iVTKWindow::getCurrentWindow()->getReader(),IDATAREADER_ALL,s,false,false,true,0);
#endif		
    }
    
}


void iQT::fileOpenVisited(const QString &ss)
{
	
	QString s = ss;
	char m = ' ';
	//
	//  If this is a string from the CurrentFile combo box, replace it with the respective string from the
	//  CurrentFileFull combo box.
	//
	int i;
	for(i=0; i<CurrentFile->count(); i++)
	{
		if(ss == CurrentFile->text(i)) 
		{
			s = iVTKWindow::getCurrentWindow()->getReader()->getVisitedFilesList()[i].mid(1);
			m = iVTKWindow::getCurrentWindow()->getReader()->getVisitedFilesList()[i][0];
			break;
		}
	}

	switch(m)
	{
	case 'a': { this->fileOpenMesh(s); break; }
	case 'b': { this->fileOpenVect(s); break; }
	case 'c': { this->fileOpenTens(s); break; }
	case 'd': { this->fileOpenPart(s); break; }
	case 'A': 
	case 'B': 
	case 'C': 
	case 'D': { this->fileOpenSet(s); break; }
	default:
		{ 
			if(ext != NULL) ext->fileOpen(m-'d',s); 
		}
	}

}


void iQT::fileExit()
{
	this->close();    
}


void iQT::styleDock(bool t)
{
	int k;
	QPoint p;

	if(remoteControlled || t==dockState || iVTKWindow::getMaxWindowIndex()>1000) return;

    inUpdate = true;
	
    if(!this->isVisible())
	{
		dockState = false;
		CALL_FUNCTION1(styleDockAction,setOn,false);
		inUpdate = false;
		return;
    } 
	
	this->startTask();
	
	dockState = !dockState;

	this->hide();

    if(dockState)
	{

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->hide();
		if(ext != 0) ext->hide();

		this->placeWindowsInDockedPositions(iVTKWindow::getWindow(0)->pos(),iVTKWindow::getWindow(0)->size());

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->show();
		if(ext != 0) ext->show();

		qApp->processEvents(30000);
		if(qApp->hasPendingEvents())
		{
			this->endTask();
			this->popupError("Unable to process all events.\n IFRIT will now exit.",IFRITERROR_UNABLE_PROCESS_EVENTS);
			for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
			{
				iVTKWindow::getWindow(k)->setCloseable();
				iVTKWindow::getWindow(k)->close();
			}
			this->close();
		}

		if(iVTKWindow::getMaxWindowIndex() == 0) iVTKWindow::getWindow(0)->showMaximized();

    } 
	else 
	{
		
		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTKWindow::getWindow(k)->showNormal();
			iVTKWindow::getWindow(k)->hide();
		}
		if(ext != 0) ext->hide();

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTKWindow::getWindow(k)->reparent(0,0,p,false);
			iVTKWindow::getWindow(k)->setIcon(dockVTKIcon);
			if(k < dockVTKnum)
			{
				iVTKWindow::getWindow(k)->resize(dockVTKSize[k]);
				iVTKWindow::getWindow(k)->move(dockVTKPos[k]);
				p = dockVTKPos[k];
			}
			else
			{
				p += QPoint(20,20);
				iVTKWindow::getWindow(k)->resize(dockVTKSize[0]);
				iVTKWindow::getWindow(k)->move(p);
			}

		}

		delete [] dockVTKPos;
		delete [] dockVTKSize;

		if(ext != 0) 
		{
			ext->reparent(0,0,p,false);
			ext->setMaximumSize(32767,32767);
			ext->resize(dockExtSize);
			ext->move(dockExtPos);
			ext->addDockWindow(ext->toolBar);
			ext->setIcon(dockExtIcon);
		}

		FrameLeft->hide();
//		FrameRight->setMinimumSize(0,0);
//		FrameRight->setMaximumSize(32767,32767);
		
		this->setMaximumSize(dockQTSize);  // not clear why simple resize does not work
		this->move(dockQTPos);

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->show();
		if(ext != 0) ext->show();

    }
	
	this->show();
	this->setMaximumSize(32767,32767);

	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->resetGL();
	
    inUpdate = false;
	this->endTask();
    
}
//
//  Helper function
//
void iQT::placeWindowsInDockedPositions(QPoint vtkPos, QSize vtkSize)
{
	int k;
	QPoint p;

	FrameLeft->show();
		
	dockVTKnum = iVTKWindow::getMaxWindowIndex() + 1;
	dockVTKPos = new QPoint[dockVTKnum];
	dockVTKSize = new QSize[dockVTKnum];
	
	dockQTPos = this->pos();
	dockQTSize = this->size();
	
	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
	{
		dockVTKPos[k] = iVTKWindow::getWindow(k)->pos();
		dockVTKSize[k] = iVTKWindow::getWindow(k)->size();
	}
	dockVTKIcon = *iVTKWindow::getWindow(0)->icon();
	if(ext != NULL)
	{
		dockExtPos = ext->pos();
		dockExtSize = ext->size();
		dockExtIcon = *ext->icon();
	}

	this->move(vtkPos);
	this->resize(this->width()+vtkSize.width(),this->height());
				
	if(ext != NULL) 
	{
		ext->reparent(FrameLeft,0,p,false);
		ext->resize(0,0);
		ext->setFixedHeight(ext->height());
		this->addDockWindow(ext->toolBar);
	}
	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
	{
		iVTKWindow::getWindow(k)->reparent(workspace,0,p,false);
		iVTKWindow::getWindow(k)->setFocusPolicy(ClickFocus);
		p += QPoint(10,10);
	}
	
	if(ext!=NULL) FrameLeft->layout()->add(ext);

	dockState = true;
	CALL_FUNCTION1(styleDockAction,setOn,dockState);

}


void iQT::styleStyle(QAction *a)
{
    
    if(a == styleStyleWinAction) qApp->setStyle("windows");
    if(a == styleStyleCDEAction) qApp->setStyle("cde");
    if(a == styleStyleMotAction) qApp->setStyle("motif");
    if(a == styleStylePlaAction) qApp->setStyle("platinum");
    if(a == styleStyleSGIAction) qApp->setStyle("sgi");
	
}


void iQT::styleBox(QAction *a)
{
	this->startTask();
	if(a == styleBoxIfritAction) CALL_VTK_FUNCTION1(setBoxType,0);
	if(a == styleBoxClassicAction) CALL_VTK_FUNCTION1(setBoxType,1);
	if(a == styleBoxHairAction) CALL_VTK_FUNCTION1(setBoxType,2);
	if(a == styleBoxAxesAction)
	{
		CALL_VTK_FUNCTION1(setBoxType,3);
		this->loadAxesBoxLabels();
	}
	this->render(true);
	this->endTask();
}


void iQT::styleFontType( QAction * a )
{
	this->startTask();
	int ft = 0;
	if(a == styleFontTypeBitmapAction) ft = 0;
	if(a == styleFontTypeVectorAction) ft = 1;
	CALL_VTK_FUNCTION1(setFontType,ft);
    this->render(true);
	this->endTask();
}


void iQT::styleFontSize( QAction * a )
{
    int fs = 0;
    if(a == styleFontSize10Action) fs = 10;
    if(a == styleFontSize12Action) fs = 12;
    if(a == styleFontSize14Action) fs = 14;
    if(a == styleFontSize16Action) fs = 16;
    if(a == styleFontSize20Action) fs = 20;
    if(a == styleFontSize24Action) fs = 24;
    if(a == styleFontSize32Action) fs = 32;
    if(a == styleFontSize40Action) fs = 40;
    if(a == styleFontSize48Action) fs = 48;
    if(a == styleFontSize64Action) fs = 64;
    if(fs > 0)
	{
		this->startTask();
		CALL_VTK_FUNCTION1(setFontSize,fs);
		this->render(true);
		this->endTask();
    }   
}


void iQT::styleTab( QAction *a )
{
	
    if(a == styleTabTextAction)
	{
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VIEW),QPixmap(),QString("View"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_SURF),QPixmap(),QString("Surface"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_XSEC),QPixmap(),QString("X-section"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VOLV),QPixmap(),QString("Volume"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_PART),QPixmap(),QString("Particles"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VECT),QPixmap(),QString("Vector"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_TENS),QPixmap(),QString("Tensor"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_DATA),QPixmap(),QString("Data"));
		if(ext != NULL) ext->styleTab(1);
	}
    if(a == styleTabIconAction)
	{
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VIEW),image("view.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_SURF),image("surf.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_XSEC),image("xsec.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VOLV),image("volv.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_PART),image("part.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VECT),image("vect.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_TENS),image("tens.png"),QString(""));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_DATA),image("data.png"),QString(""));
		if(ext != NULL) ext->styleTab(2);
	}
    if(a == styleTabBothAction)
	{
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VIEW),image("view.png"),QString("View"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_SURF),image("surf.png"),QString("Surface"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_XSEC),image("xsec.png"),QString("X-section"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VOLV),image("volv.png"),QString("Volume"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_PART),image("part.png"),QString("Particles"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_VECT),image("vect.png"),QString("Vector"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_TENS),image("tens.png"),QString("Tensor"));
		TabWidget->changeTab(TabWidget->page(DISPLAYPAGE_DATA),image("data.png"),QString("Data"));
		if(ext != NULL) ext->styleTab(3);
	}
	
}


void iQT::styleSlidersRenderTracking(bool s)
{
    this->sliderRenderTracking = s;
}


void iQT::styleAllowAntialiasing(bool s)
{
	styleAllowAntialiasingAction->blockSignals(true);
	iVTKWindow::getCurrentWindow()->setAntialiasingOn(s); 
	this->render(false,true);
	styleAllowAntialiasingAction->blockSignals(false);
}


void iQT::styleAllowTooltips( bool s )
{
	
	QToolTip::setGloballyEnabled(s);
	
}


void iQT::styleInteractorHelp( bool s )
{
	int i;

    if(s) 
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->setBottomWidget(FrameRight); 
	}
	else 
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->setBottomWidget((QWidget *)0);     
	}
}


void iQT::optionsImageZoom(QAction *v)
{
    int m = 1;
	
    if(v == optionsImageZoomX002Action) m = 2;
    if(v == optionsImageZoomX003Action) m = 3;
    if(v == optionsImageZoomX004Action) m = 4;
    if(v == optionsImageZoomX005Action) m = 5;
    if(v == optionsImageZoomX006Action) m = 6;
    if(v == optionsImageZoomX008Action) m = 8;
    if(v == optionsImageZoomX010Action) m = 10;
    if(v == optionsImageZoomX015Action) m = 15;
    if(v == optionsImageZoomX020Action) m = 20;
    if(v == optionsImageZoomX030Action) m = 30;
    if(v == optionsImageZoomX040Action) m = 40;
    if(v == optionsImageZoomX050Action) m = 50;
    if(v == optionsImageZoomX060Action) m = 60;
    if(v == optionsImageZoomX080Action) m = 80;
    if(v == optionsImageZoomX100Action) m = 100;
	
	CALL_VTK_FUNCTION1(setImageMagnification,m);

	this->updateImageComposer(true);
    
}


void iQT::optionsImageFormat(QAction *v)
{
    int m = 0;
    
    if(v == optionsImageFormatPNMAction) m = 1;
    if(v == optionsImageFormatBMPAction) m = 2;
    if(v == optionsImageFormatPNGAction) m = 3;
    if(v == optionsImageFormatTIFAction) m = 4;
    if(v == optionsImageFormatPSAction) m = 5;
	
	CALL_VTK_FUNCTION1(setImageType,m);
    
	optionsImageFormatPSPaperFormatActionGroup->setEnabled((v == optionsImageFormatPSAction));
	optionsImageFormatPSOrientationActionGroup->setEnabled((v == optionsImageFormatPSAction));

}


void iQT::optionsImageFormatPSPaperFormat(QAction * v)
{
	int i, k;
	for(i=0; i<NFORMATS; i++)
	{
		if(v == psPaperFormatList[i])
		{
			if(optionsAffectAll)
			{
				for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) if(iVTKWindow::getWindow(k)->getWriter()->IsA("vtkPostScriptWriter")) ((iPostScriptWriter *)iVTKWindow::getWindow(k)->getWriter())->setPaperFormat(i);
			}
			else
			{
				if(iVTKWindow::getCurrentWindow()->getWriter()->IsA("vtkPostScriptWriter")) ((iPostScriptWriter *)iVTKWindow::getCurrentWindow()->getWriter())->setPaperFormat(i);
			}
			break;
		}
	}
}


void iQT::optionsImageFormatPSOrientation(QAction * v)
{
    int k, m = 0;
    if(v == optionsImageFormatPSOrientationLandscapeAction) m = 1;
	if(optionsAffectAll)
	{
		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) if(iVTKWindow::getWindow(k)->getWriter()->IsA("vtkPostScriptWriter"))	((iPostScriptWriter *)iVTKWindow::getWindow(k)->getWriter())->setOrientation(m);
	}
	else
	{
		if(iVTKWindow::getCurrentWindow()->getWriter()->IsA("vtkPostScriptWriter"))	((iPostScriptWriter *)iVTKWindow::getCurrentWindow()->getWriter())->setOrientation(m);
	}
}


void iQT::optionsVariableLimits(QAction *vtkNotUsed(v))
{
	int k;

    variableLimits = 0;
    if(optionsVarLimitsAdjustableAction->isOn()) variableLimits = 1;
    if(optionsVarLimitsMixedAction->isOn()) variableLimits = 2;
	if(optionsAffectAll)
	{
	    for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) iVTKWindow::getWindow(k)->getReader()->setAdjustableBounds((variableLimits>0));
	}
	else
	{
	    iVTKWindow::getCurrentWindow()->getReader()->setAdjustableBounds((variableLimits>0));
	}
	this->TabWidgetData->updateData_Widgets();
	this->startTask();
	if(!blockRenderer) this->reloadLastSet(IDATAREADER_MESH);
	this->endTask();
	this->updateWidgets(-1);
	
}


void iQT::optionsDataEndiness( QAction * v )
{
    if(v == optionsDataEndinessBigAction) 
	{
		iVTKWindow::getCurrentWindow()->getReader()->setEndinessOfData(true);
	}
    if(v == optionsDataEndinessLittleAction) 
	{
		iVTKWindow::getCurrentWindow()->getReader()->setEndinessOfData(false);
    }
}


void iQT::optionsReportMemory( bool s )
{
    reportMemory = s;
}


void iQT::helpContents()
{
	static QString s;
	s = "vtkhelp.ihf";
	this->displayHelp(s);
}


void iQT::displayHelp(const QString &ref )
{
	Help->setSource(ref);
	Help->show();
}


void iQT::helpAbout()
{
    iDialogAbout q;
    q.exec();
}
//
// *********************************************************************************
//
//  
//   Control buttons
//
//
// *********************************************************************************
//
void iQT::dumpImage()
{
	
	this->startTask();
    iVTKWindow::getCurrentWindow()->dumpImage();
	this->endTask();
	ProgressBar->reset();
	
}


void iQT::loadNextSetAll()
{
	this->loadNextSet(true,0,true);
}


bool iQT::loadNextSet(bool all, int skip, bool saveInTheList)
{
	int i, ret, retall;

	if(iVTKWindow::getCurrentWindow()->isClone())
	{
		popupInfo("Cannot load set from a clone.\n");
		return false;
	}

	int nold = CURRENT_LIMITS->getNvar();
	int mold = CURRENT_LIMITS->getNatt();
	
	Button_LoadNextSet->setEnabled(true);

	int rec1 = iVTKWindow::getCurrentWindow()->getReader()->getRecordNumber();
	QString s;
	int pos = iVTKWindow::getCurrentWindow()->getReader()->getLastFileSetName().find(s.sprintf("%04d",rec1),0);
	if(pos == -1)
	{
		this->popupError("Corrupted lastFileSetName variable.",IFRITERROR_CORRUPTED_LASTFILESETNAME);
		Button_LoadNextSet->setEnabled(false);
		return false;
	}
	
	retall = 0;

	iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_ALL,ProgressBar);
	ret = iVTKWindow::getCurrentWindow()->getReader()->loadNextRecord(skip,false,saveInTheList);
	if(ret == -9) retall = -9;
	int rec2 = iVTKWindow::getCurrentWindow()->getReader()->getRecordNumber();

	if(iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataChannel()==0)
	{
		if(ret == 0)
		{
			if(iVTKWindow::getCurrentWindow()->getReader()->isMeshFileBelongToSet()) this->afterMeshFileRead(iVTKWindow::getCurrentWindowIndex(),nold,ret,iVTKWindow::getCurrentWindow()->getReader()->getLastMeshFileName());
			if(iVTKWindow::getCurrentWindow()->getReader()->isPartFileBelongToSet()) this->afterPartFileRead(iVTKWindow::getCurrentWindowIndex(),mold,ret,iVTKWindow::getCurrentWindow()->getReader()->getLastPartFileName());
			if(iVTKWindow::getCurrentWindow()->getReader()->isVectFileBelongToSet()) this->afterVectFileRead(iVTKWindow::getCurrentWindowIndex(),0,ret,iVTKWindow::getCurrentWindow()->getReader()->getLastVectFileName());
			if(iVTKWindow::getCurrentWindow()->getReader()->isTensFileBelongToSet()) this->afterTensFileRead(iVTKWindow::getCurrentWindowIndex(),0,ret,iVTKWindow::getCurrentWindow()->getReader()->getLastTensFileName());
		}
		else
		{
			if(iVTKWindow::getCurrentWindow()->getReader()->isMeshFileBelongToSet()) this->afterFileReadError(FILEREADERRORMODE_MESH,iVTKWindow::getCurrentWindowIndex());
			if(iVTKWindow::getCurrentWindow()->getReader()->isPartFileBelongToSet()) this->afterFileReadError(FILEREADERRORMODE_PART,iVTKWindow::getCurrentWindowIndex());
			if(iVTKWindow::getCurrentWindow()->getReader()->isVectFileBelongToSet()) this->afterFileReadError(FILEREADERRORMODE_VECT,iVTKWindow::getCurrentWindowIndex());
			if(iVTKWindow::getCurrentWindow()->getReader()->isTensFileBelongToSet()) this->afterFileReadError(FILEREADERRORMODE_TENS,iVTKWindow::getCurrentWindowIndex());
		}
	}
	else if(ext != NULL) ext->afterFileRead(iVTKWindow::getCurrentWindowIndex(),0,ret,iVTKWindow::getCurrentWindow()->getReader()->getLastFileSetName());
	
	if(ret != 0 ) this->logLine("Error in reading file","next set cannot be loaded (load it manually for more info)",red);

	if(ret==0 && saveInTheList) this->updateVisitedFilesList(false);
	
	if(ret == 0) iVTKWindow::getCurrentWindow()->showLabel(iVTKWindow::getCurrentWindow()->isLabelVisible()); 

	for(i=0; i<=iVTKWindow::getMaxWindowIndex() && all; i++) 
	{
		if(i!=iVTKWindow::getCurrentWindowIndex() && !iVTKWindow::getWindow(i)->isClone())
		{
			ret = iVTKWindow::getWindow(i)->getReader()->loadNextRecord(skip,false,saveInTheList);
			if(ret == -9) retall = -9;
			if(ret==0 && saveInTheList) this->updateVisitedFilesList(i,false);
			if(ret==0 && abs(iVTKWindow::getWindow(i)->getReader()->getRecordNumber()-rec2)>abs(rec2-rec1)/5)
			{
				s = s.setNum(i+1);
				s = "Window #" + s + ": record read is not similar to \n that of the current window";
				popupInfo((char *)s.latin1());
			}
			if(ret == 0) iVTKWindow::getWindow(i)->showLabel(iVTKWindow::getWindow(i)->isLabelVisible()); 
		}
		else ret = 0;
	}
		
	if(retall == -9)
	{
		this->popupInfo("There are no more sets to load.");
		Button_LoadNextSet->setEnabled(false);
	}
	
	ProgressBar->reset();
	this->updateVariablesExplorer();

	this->render(true,all);

	return (retall == 0);

}
//
// *********************************************************************************
//
//  
//   reset whatever is required after a file read
//
//
// *********************************************************************************
//
void iQT::updateAfterFileRead(iDataReader *reader, int mode, QString filename, int flags, int res)
{
	if(res == -999) return;

	int i;
	for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) if(iVTKWindow::getWindow(i)->getReader()==reader && !iVTKWindow::getWindow(i)->isClone()) break;

	if(i <= iVTKWindow::getMaxWindowIndex())
	{
		switch (mode)
		{
		case IDATAREADER_MESH: 
			{
				this->afterMeshFileRead(i,flags,res,filename);
				break;
			}
		case IDATAREADER_PART: 
			{
				this->afterPartFileRead(i,flags,res,filename);
				break;
			}
		case IDATAREADER_VECT: 
			{
				this->afterVectFileRead(i,flags,res,filename);
				break;
			}
		case IDATAREADER_TENS: 
			{
				this->afterTensFileRead(i,flags,res,filename);
				break;
			}
		default: res = -999;
		}
	}
}


#define IQT_AFTERFILEREAD_COMMONSTEP_1(label) \
	iDataInfo info; \
	iVTKWindow::getCurrentWindow()->getReader()->getDataInfo(&info); \
	if(info.mode == "ba") logLine("File type","binary mesh file."); \
	if(info.mode == "ta") logLine("File type","ascii mesh file."); \
	if(info.mode == "bd") logLine("File type","binary particle file."); \
	if(info.mode == "td") logLine("File type","ascii particle file."); \
	if(info.mode == "bb") logLine("File type","binary vector field file."); \
	if(info.mode == "tb") logLine("File type","ascii vector field file."); \
	if(info.mode == "bc") logLine("File type","binary tensor field file."); \
	if(info.mode == "tc") logLine("File type","ascii tensor field file."); \
	if(info.isSet) fileOpenSetAction->setEnabled(true); else fileOpenSetAction->setEnabled(false); \
	if(iVTKWindow::getCurrentWindow()->getReader()->is##label##FileAnimatable()) \
	{ \
		logLine("Animation status","animatable"); \
	} \
	else \
	{ \
		logLine("Animation status","NOT animatable"); \
	} \
	anim = info.animatable; \
	Button_Animate->setEnabled(anim); \
	Button_LoadNextSet->setEnabled(anim); 


#define IQT_AFTERFILEREAD_COMMONSTEP_2 \
	if(anim) \
	{ \
		QString s1; \
		char setLeader = iVTKWindow::getCurrentWindow()->getReader()->getSetLeader(); \
		switch (setLeader) \
		{ \
		case 'a': { s1 = "Mesh (leader)"; break; } \
		case 'd': { s1 = "Particles (leader)"; break; } \
		case 'b': { s1 = "Vector field (leader)"; break; } \
		case 'c': { s1 = "Tensor field (leader)"; break; } \
		} \
		if(iVTKWindow::getCurrentWindow()->getReader()->isMeshFileBelongToSet() && setLeader!='a') s1 += " + Mesh"; \
		if(iVTKWindow::getCurrentWindow()->getReader()->isPartFileBelongToSet() && setLeader!='d') s1 += " + Particles"; \
		if(iVTKWindow::getCurrentWindow()->getReader()->isVectFileBelongToSet() && setLeader!='b') s1 += " + Vector field"; \
		if(iVTKWindow::getCurrentWindow()->getReader()->isTensFileBelongToSet() && setLeader!='c') s1 += " + Tensor field"; \
		logLine("Current set",s1); \
	} \
	iVTKWindow::getCurrentWindow()->showLabel(iVTKWindow::getCurrentWindow()->isLabelVisible()); \
	TabWidgetData->updateData_Widgets(); 


void iQT::afterMeshFileRead(int wi, int nold, int res, QString fname)
{
	
    int i,j;
    int nnew = CURRENT_LIMITS->getNvar();
	
    bool anim = false;
    
	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		
		if(res == 0) this->updateVisitedFilesList(false);
		
		TextLog->append("<hr>");
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) TextLog->append("<br><br>");
		
		switch (res) 
		{
		case 0: { logLine("File read",fname); break; }
		case 1: { logLine("Error in reading file","file does not exist.",QColor(255,0,0)); break; }
		case 2: { logLine("Error in reading file","no header or wrong endiness.",QColor(255,0,0)); break; }
		case 3: { logLine("Error in reading file","file is corrupted.",QColor(255,0,0)); break; }
		case 4: { logLine("Error in reading file","file is corrupted, but some data are read.",QColor(255,0,0)); break; }
		case 10: { logLine("Error in reading file","not enough memory to load the data.",QColor(255,0,0)); break; }
		case -1: { logLine("Error in reading file","incorrect file type.",QColor(255,0,0)); break; }
		case -2: { logLine("Error in reading file","system error.",QColor(255,0,0)); break; }
		case -99: { logLine("Error in reading file","reading was aborted.",QColor(255,0,0)); break; }
		}
		
		if(res == 0) 
		{
			
			IQT_AFTERFILEREAD_COMMONSTEP_1(Mesh)
				//
				//  Disable creation of mesh data from particles & enable array calculator
				//
				CALL_FUNCTION1(TabWidgetData->Data_Tran_PartToMeshCheckBox,setChecked,false);
			//		TabWidgetData->Data_Tran_PartToMeshGroupBox->setEnabled(false);
			TabWidgetData->TabWidgetTransform->page(0)->setEnabled(true);
			
			QString s1,s2;
			s1.setNum(CURRENT_LIMITS->getNvar());
			logLine("Number of variables in the file",s1);
			for(i=0; i<CURRENT_LIMITS->getNvar(); i++)
			{
				info.varMin[i] = iVTKWindow::getCurrentWindow()->getReader()->getLimits(0)->unstretchVar(info.varMin[i],i+1);
				info.varMax[i] = iVTKWindow::getCurrentWindow()->getReader()->getLimits(0)->unstretchVar(info.varMax[i],i+1);
				float ql = log10(1.0e-30+fabs(info.varMin[i]));
				float qu = log10(1.0e-30+fabs(info.varMax[i]));
				s1 = QString("from ");
				s2.setNum(ql,'f',3);
				s1 += s2;
				s1 += QString("dex (");
				s2.setNum(info.varMin[i],'g',3);
				s1 += s2;
				s1 += QString(") to ");
				s2.setNum(qu,'f',3);
				s1 += s2;
				s1 += QString("dex (");
				s2.setNum(info.varMax[i],'g',3);
				s1 += s2;
				s1 += QString(")."); 
				s2.setNum(i+1);
				s2 = "    Variable " + s2;
				logLine(s2,s1);
			}
			s1 = QString("");
			s2.setNum(info.meshDims[0]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.meshDims[1]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.meshDims[2]); s1 += s2; 
			logLine("Dimensions",s1);
			
		} 
		else 
		{
			
			nnew = 0;
			
		}
		
		for(i=nAttButtons; i<nButtons; i++)
		{
			for(j=0; j<nnew; j++) if(button[i][j]) button[i][j]->setEnabled(true);
			for(j=nnew; j<nold; j++) if(button[i][j]) button[i][j]->setEnabled(false);
		}
		
		this->fixVariableSettings(nold,nnew);
		
		IQT_AFTERFILEREAD_COMMONSTEP_2
			
	}
		
    if(nnew > 0) 
	{
		this->afterFileReadConnectMeshAndVector(wi);
		this->afterFileReadConnectMeshAndTensor(wi);
		iVTKWindow::getWindow(wi)->getGateway()->Modified();
    }
	else
	{
		this->afterFileReadError(FILEREADERRORMODE_MESH,wi);
	}
	
	if(!iVTKWindow::getWindow(wi)->getSurfaceFamily()->getCurrentMember()->isInitialized()) { iVTKWindow::getWindow(wi)->getSurfaceFamily()->getCurrentMember()->setDataChannel(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel()); }
	if(!iVTKWindow::getWindow(wi)->getVolumeFamily()->getCurrentMember()->isInitialized()) { iVTKWindow::getWindow(wi)->getVolumeFamily()->getCurrentMember()->setDataChannel(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel()); }
	if(!iVTKWindow::getWindow(wi)->getXsectionFamily()->getCurrentMember()->isInitialized()) { iVTKWindow::getWindow(wi)->getXsectionFamily()->getCurrentMember()->setDataChannel(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel()); }

	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		TabWidgetSurf->updateSurf_Widgets();
		TabWidgetXsec->updateXsec_Widgets();
		TabWidgetVolv->updateVolv_Widgets();
	}

	this->updateVariablesExplorer();

	this->render(true,true); // render all windows so the clones are also updated

}


void iQT::afterPartFileRead(int wi, int mold, int res, QString fname)
{
	
    int i, j;
	int att = 0;
    int mnew = CURRENT_LIMITS->getNatt();
	
    bool anim = false;
    
	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		
		if(res == 0) this->updateVisitedFilesList(false);
		
		TextLog->append("<hr>");
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) TextLog->append("<br><br>");
		
		switch (res) 
		{
		case 0: { logLine("File read",fname); break; }
		case 1: { logLine("Error in reading file","file does not exist.",QColor(255,0,0)); break; }
		case 2: { logLine("Error in reading file","no header or wrong endiness.",QColor(255,0,0)); break; }
		case 3: { logLine("Error in reading file","file is corrupted.",QColor(255,0,0)); break; }
		case 4: { logLine("Error in reading file","file is corrupted, but some data are read.",QColor(255,0,0)); break; }
		case 10: { logLine("Error in reading file","not enough memory to load the data.",QColor(255,0,0)); break; }
		case -1: { logLine("Error in reading file","incorrect file type.",QColor(255,0,0)); break; }
		case -2: { logLine("Error in reading file","system error.",QColor(255,0,0)); break; }
		case -99: { logLine("Error in reading file","reading was aborted.",QColor(255,0,0)); break; }
		}
		
		if(res == 0) 
		{
			
			IQT_AFTERFILEREAD_COMMONSTEP_1(Part)
			//
			//  Particle file
			//	    
			QString s1,s2;
			s1.setNum(info.npart);
			logLine("Number of particles in the file",s1);
			s1.setNum(CURRENT_LIMITS->getNatt());
			logLine("Number of attributes per particle",s1);
			for(i=0; i<CURRENT_LIMITS->getNatt(); i++)
			{
				s1 = QString("from ");
				s2.setNum(info.attMin[i],'g',3);
				s1 += s2;
				s1 += QString(" to ");
				s2.setNum(info.attMax[i],'g',3);
				s1 += s2;
				s1 += QString("."); 
				s2.setNum(i+1);
				s2 = "    Attribute " + s2;
				logLine(s2,s1);
			}
			
			//
			//  Enable mesh creation from particles if there are attributes in the data
			//
			if(CURRENT_LIMITS->getNatt() > 0)
			{
				TabWidgetData->TabWidgetTransform->page(1)->setEnabled(true);
				TabWidgetData->Data_Tran_PartToMeshApplyNow->setEnabled(false);
			}
			else
			{
				TabWidgetData->TabWidgetTransform->page(1)->setEnabled(false);
			}
			
			if(CURRENT_LIMITS->periodicBC)
			{
				QString s1;
				s1 = QString("");
				if(CURRENT_LIMITS->periodic[0]) s1 += "X, ";
				if(CURRENT_LIMITS->periodic[1]) s1 += "Y, ";
				if(CURRENT_LIMITS->periodic[2]) s1 += "Z";
				logLine("Data periodic in",s1);
			}
			
		} 
		else 
		{
			
			mnew = 0;
			
		}
		
		int imax = nAttButtons;
		if(iVTKWindow::getCurrentWindow()->getReader()->getMeshFromParticles()) imax = nButtons;
		for(i=0; i<imax; i++)
		{
			for(j=0; j<mnew; j++) if(button[i][j]) button[i][j]->setEnabled(true);
			for(j=mnew; j<mold; j++) if(button[i][j]) button[i][j]->setEnabled(false);
		}
		
		if(iVTKWindow::getCurrentWindow()->getReader()->getMeshFromParticles()) this->fixVariableSettings(mold,mnew);
		
		if(mnew > 0) 
		{
			iVTKWindow::getCurrentWindow()->showLabel(TabWidgetView->View_TimeLabelButton->isChecked()&&anim); 
		}
		
		IQT_AFTERFILEREAD_COMMONSTEP_2
		
	}

	if(res == 0) 
	{
		
		iVTKWindow::getWindow(wi)->setCurrentParticlesFamily(0);
		
		//
		//  Update lookup tables for all pieces
		//
		int i;
		for(i=0; i<=iVTKWindow::getWindow(wi)->getParticlesFamily()->getMaxMemberIndex(); i++) iVTKWindow::getWindow(wi)->getParticlesFamily()->getMember(i)->updateColorLookupTable();
		//
		//  Reset attribute limits if particles are split into pieces
		//
		att = iVTKWindow::getWindow(wi)->getParticlesFamily()->getCurrentMember()->getParticlesSplitter()->getAttribute();
		if(att > 0)
		{
			iVTKWindow::getWindow(wi)->getParticlesFamily()->getCurrentMember()->getParticlesSplitter()->resetLimits();
			iVTKWindow::getWindow(wi)->getParticlesFamily()->getCurrentMember()->getParticlesSplitter()->setAttributeMinMax(iVTKWindow::getWindow(wi)->getReader()->getLimits(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel())->getAttMin(att),iVTKWindow::getWindow(wi)->getReader()->getLimits(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel())->getAttMax(att));
		}

		iVTKWindow::getWindow(wi)->getParticlesFamily()->getCurrentMember()->getParticlesSplitter()->Modified();
		
	}
	else
	{
		this->afterFileReadError(FILEREADERRORMODE_PART,wi);
	}

	this->render(true,true); // render all windows so the clones are also updated

	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		//
		//  Must do it at the end after the pipeline is executed so that all data are up to date.
		//
		TabWidgetPart->updatePart_Widgets();
		TabWidgetPart->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);
	}

}


void iQT::afterVectFileRead(int wi, int, int res, QString fname)
{
	
    bool anim = false;
    
	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		
		if(res == 0) this->updateVisitedFilesList(false);
		
		TextLog->append("<hr>");
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) TextLog->append("<br><br>");
		
		switch (res) 
		{
		case 0: { logLine("File read",fname); break; }
		case 1: { logLine("Error in reading file","file does not exist.",QColor(255,0,0)); break; }
		case 2: { logLine("Error in reading file","no header or wrong endiness.",QColor(255,0,0)); break; }
		case 3: { logLine("Error in reading file","file is corrupted.",QColor(255,0,0)); break; }
		case 10: { logLine("Error in reading file","not enough memory to load the data.",QColor(255,0,0)); break; }
		case -1: { logLine("Error in reading file","incorrect file type.",QColor(255,0,0)); break; }
		case -2: { logLine("Error in reading file","system error.",QColor(255,0,0)); break; }
		case -99: { logLine("Error in reading file","reading was aborted.",QColor(255,0,0)); break; }
		}
		
		if(res == 0) 
		{
			
			IQT_AFTERFILEREAD_COMMONSTEP_1(Vect);
			//
			//  Vector field file
			//	    
			QString s1,s2;
			s1.setNum(info.vecMax,'g',3);
			logLine("Maximum amplitude",s1);
			s1 = QString("");
			s2.setNum(info.vectDims[0]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.vectDims[1]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.vectDims[2]); s1 += s2; 
			logLine("Dimensions",s1);
			
			if(CURRENT_LIMITS->periodicBC)
			{
				QString s1;
				s1 = QString("");
				if(CURRENT_LIMITS->periodic[0]) s1 += "X, ";
				if(CURRENT_LIMITS->periodic[1]) s1 += "Y, ";
				if(CURRENT_LIMITS->periodic[2]) s1 += "Z";
				logLine("Data periodic in",s1);
			}
			
		} 
		
		IQT_AFTERFILEREAD_COMMONSTEP_2
			
	}
	
	if(!iVTKWindow::getWindow(wi)->getVectorFamily()->getCurrentMember()->isInitialized()) { iVTKWindow::getWindow(wi)->getVectorFamily()->setDataChannel(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel()); }
	
	if(res == 0) 
	{
		this->afterFileReadConnectMeshAndVector(wi);
	}	
	else
	{
		this->afterFileReadError(FILEREADERRORMODE_VECT,wi);
	}

	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		TabWidgetVect->updateVect_Widgets();
	}
	
	this->render(true,true); // render all windows so the clones are also updated
	
}


void iQT::afterTensFileRead(int wi, int, int res, QString fname)
{
	
    bool anim = false;

	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		
		if(res == 0) this->updateVisitedFilesList(false);
		
		TextLog->append("<hr>");
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) TextLog->append("<br><br>");
		
		switch (res) 
		{
		case 0: { logLine("File read",fname); break; }
		case 1: { logLine("Error in reading file","file does not exist.",QColor(255,0,0)); break; }
		case 2: { logLine("Error in reading file","no header or wrong endiness.",QColor(255,0,0)); break; }
		case 3: { logLine("Error in reading file","file is corrupted.",QColor(255,0,0)); break; }
		case 10: { logLine("Error in reading file","not enough memory to load the data.",QColor(255,0,0)); break; }
		case -1: { logLine("Error in reading file","incorrect file type.",QColor(255,0,0)); break; }
		case -2: { logLine("Error in reading file","system error.",QColor(255,0,0)); break; }
		case -99: { logLine("Error in reading file","reading was aborted.",QColor(255,0,0)); break; }
		}
		
		if(res == 0) 
		{
			
			IQT_AFTERFILEREAD_COMMONSTEP_1(Tens);
			//
			//  Tensor field file
			//	    
			QString s1,s2;
			s1.setNum(info.tenMax,'g',3);
			logLine("Maximum amplitude",s1);
			s1 = QString("");
			s2.setNum(info.tensDims[0]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.tensDims[1]); s1 += s2; s1 += QString(" x ");
			s2.setNum(info.tensDims[2]); s1 += s2; 
			logLine("Dimensions",s1);
			
			if(CURRENT_LIMITS->periodicBC)
			{
				QString s1;
				s1 = QString("");
				if(CURRENT_LIMITS->periodic[0]) s1 += "X, ";
				if(CURRENT_LIMITS->periodic[1]) s1 += "Y, ";
				if(CURRENT_LIMITS->periodic[2]) s1 += "Z";
				logLine("Data periodic in",s1);
			}
			
		} 
		
		IQT_AFTERFILEREAD_COMMONSTEP_2
			
	}
	
	if(!iVTKWindow::getWindow(wi)->getTensorFamily()->getCurrentMember()->isInitialized()) { iVTKWindow::getWindow(wi)->getTensorFamily()->setDataChannel(iVTKWindow::getWindow(wi)->getReader()->getCurrentDataChannel()); }
	
	if(res == 0) 
	{
		this->afterFileReadConnectMeshAndTensor(wi);
	}
	else
	{
		this->afterFileReadError(FILEREADERRORMODE_TENS,wi);
	}

	if(wi == iVTKWindow::getCurrentWindowIndex())
	{
		TabWidgetTens->updateTens_Widgets();
	}
	
	this->render(true,true); // render all windows so the clones are also updated

}


void iQT::afterFileReadConnectMeshAndVector(int wi)
{
	int meshDims[3], vectDims[3];
	
	bool connected = false;
	if(iVTKWindow::getWindow(wi)->getReader()->getMeshOutput()!=0 && iVTKWindow::getWindow(wi)->getReader()->getVectorOutput()!=0)
	{
		
		iVTKWindow::getWindow(wi)->getReader()->getMeshOutput()->GetDimensions(meshDims);
		iVTKWindow::getWindow(wi)->getReader()->getVectorOutput()->GetDimensions(vectDims);
		
		if(meshDims[0]==vectDims[0] && meshDims[1]==vectDims[1] && meshDims[2]==vectDims[2] && meshDims[0]!=0 && meshDims[1]!=0 && meshDims[2]!=0)
		{
			
			connected = true;
			CURRENT_VECTOR->getStreamLine()->setScalars(iVTKWindow::getWindow(wi)->getReader()->getMeshOutput());
			
			if(CURRENT_LIMITS->getNvar() > 0) 
			{
				logLine("Vector field","streamlines can be colored by mesh variables");
				TabWidgetVect->Vect_Paint_PaintVar1->setEnabled(true);
			}
			if(CURRENT_LIMITS->getNvar() > 1) 
			{
				TabWidgetVect->Vect_Paint_PaintVar2->setEnabled(true);
			}
			if(CURRENT_LIMITS->getNvar() > 2) 
			{
				TabWidgetVect->Vect_Paint_PaintVar3->setEnabled(true);
			}
			
		}
	}
	
	if(!connected)
	{
		CALL_FUNCTION1(TabWidgetVect->Vect_Paint_PaintWithButtons,setButton,0);
		CURRENT_VECTOR->setColorBy(0);

		TabWidgetVect->Vect_Paint_PaintVar1->setEnabled(false);
		TabWidgetVect->Vect_Paint_PaintVar2->setEnabled(false);
		TabWidgetVect->Vect_Paint_PaintVar3->setEnabled(false);
		CURRENT_VECTOR->getStreamLine()->setScalars(0);
	}
	
}


void iQT::afterFileReadConnectMeshAndTensor(int wi)
{
	int meshDims[3], tensDims[3];
	
	bool connected = false;
	if(iVTKWindow::getWindow(wi)->getReader()->getMeshOutput()!=0 && iVTKWindow::getWindow(wi)->getReader()->getTensorOutput()!=0)
	{
		
		iVTKWindow::getWindow(wi)->getReader()->getMeshOutput()->GetDimensions(meshDims);
		iVTKWindow::getWindow(wi)->getReader()->getTensorOutput()->GetDimensions(tensDims);
		
		if(meshDims[0]==tensDims[0] && meshDims[1]==tensDims[1] && meshDims[2]==tensDims[2] && meshDims[0]!=0 && meshDims[1]!=0 && meshDims[2]!=0)
		{
			
			connected = true;
			CURRENT_TENSOR->setColorBy(TabWidgetTens->Tens_Paint_PaintWithButtons->id(TabWidgetTens->Tens_Paint_PaintWithButtons->selected()));			

			if(CURRENT_LIMITS->getNvar() > 0) 
			{
				logLine("Tensor field","glyphs can be colored by mesh variables");
				TabWidgetTens->Tens_Paint_PaintVar1->setEnabled(true);
			}
			if(CURRENT_LIMITS->getNvar() > 1) 
			{
				TabWidgetTens->Tens_Paint_PaintVar2->setEnabled(true);
			}
			if(CURRENT_LIMITS->getNvar() > 2) 
			{
				TabWidgetTens->Tens_Paint_PaintVar3->setEnabled(true);
			}
			
		}
	}
	
	if(!connected)
	{
		CALL_FUNCTION1(TabWidgetTens->Tens_Paint_PaintWithButtons,setButton,0);
		CURRENT_TENSOR->setColorBy(0);

		TabWidgetTens->Tens_Paint_PaintVar1->setEnabled(false);
		TabWidgetTens->Tens_Paint_PaintVar2->setEnabled(false);
		TabWidgetTens->Tens_Paint_PaintVar3->setEnabled(false);
	}
	
}


void iQT::afterFileReadError(int mode, int wi)
{
	int k;
	
	iDataReader *sr = iVTKWindow::getWindow(wi)->getReader();

	if(mode & FILEREADERRORMODE_MESH)
	{
		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			CALL_FUNCTION1(TabWidgetView->View_Surf,setChecked,false);
			CALL_FUNCTION1(TabWidgetSurf->View_SurfLocal,setChecked,false);
			CALL_FUNCTION1(actionShowSurf,setOn,false);
			
			CALL_FUNCTION1(TabWidgetView->View_Xsec,setChecked,false);
			CALL_FUNCTION1(TabWidgetXsec->View_XsecLocal,setChecked,false);
			CALL_FUNCTION1(actionShowXsec,setOn,false);
			
			CALL_FUNCTION1(TabWidgetView->View_Volv,setChecked,false);
			CALL_FUNCTION1(TabWidgetVolv->View_VolvLocal,setChecked,false);
			CALL_FUNCTION1(actionShowVolv,setOn,false);
		}

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTK *s = iVTKWindow::getWindow(k);
			if(s->getReader() == sr) // all clones
			{
				s->getSurfaceFamily()->show(false);
				s->getXsectionFamily()->show(false);
				s->getVolumeFamily()->show(false);
			}
		}

		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			TabWidgetView->updateSwitchesSurf();
			TabWidgetView->updateSwitchesXsec();
			TabWidgetView->updateSwitchesVolv();
		}
	}
	
	if(mode & FILEREADERRORMODE_PART)
	{
		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			CALL_FUNCTION1(TabWidgetView->View_Part,setChecked,false);
			CALL_FUNCTION1(TabWidgetPart->View_PartLocal,setChecked,false);
			CALL_FUNCTION1(actionShowPart,setOn,false);
		}
		
		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTK *s = iVTKWindow::getWindow(k);
			if(s->getReader() == sr) // all clones
			{
				s->getParticlesFamily()->show(false);
			}
		}

		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			TabWidgetView->updateSwitchesPart();
		}
	}
	
	if(mode & FILEREADERRORMODE_VECT)
	{
		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			CALL_FUNCTION1(TabWidgetView->View_Vect,setChecked,false);
			CALL_FUNCTION1(TabWidgetVect->View_VectLocal,setChecked,false);
			CALL_FUNCTION1(actionShowVect,setOn,false);
		}

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTK *s = iVTKWindow::getWindow(k);
			if(s->getReader() == sr) // all clones
			{
				s->getVectorFamily()->show(false);
			}
		}

		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			TabWidgetView->updateSwitchesVect();
		}
	}
	
	if(mode & FILEREADERRORMODE_TENS)
	{
		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			CALL_FUNCTION1(TabWidgetView->View_Tens,setChecked,false);
			CALL_FUNCTION1(TabWidgetTens->View_TensLocal,setChecked,false);
			CALL_FUNCTION1(actionShowTens,setOn,false);
		}

		for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
		{
			iVTK *s = iVTKWindow::getWindow(k);
			if(s->getReader() == sr) // all clones
			{
				s->getTensorFamily()->show(false);
			}
		}

		if(wi == iVTKWindow::getCurrentWindowIndex())
		{
			TabWidgetView->updateSwitchesTens();
		}
	}

}


void iQT::reloadLastSet(int mode)
{
	reloadLastSetMode |= mode;
	this->reloadLastSet();
}


void iQT::setReloadLastSetMode(int mode)
{
	reloadLastSetMode |= mode;
}


void iQT::reloadLastSet()
{
	int ret;

	iVTKWindow::getCurrentWindow()->getReader()->setProgressBar(IDATAREADER_ALL,ProgressBar);
	ret = iVTKWindow::getCurrentWindow()->getReader()->reloadCurrentSet(reloadLastSetMode);
	reloadLastSetMode = 0;
	ProgressBar->reset();

	if(ret == 0)
	{
		this->updateVariablesExplorer();
		this->render(true,true);
	}
	else
	{
		switch(ret)
		{
		case 1:
			{
				this->popupInfo("Reloading of the current file set failed.\n The mesh file has been removed from the disk since the last loading.");
				break;
			}
		case 100:
			{
				this->popupInfo("Reloading of the current file set failed.\n The particle file has been removed from the disk since the last loading.");
				break;
			}
		case 10000:
			{
				this->popupInfo("Reloading of the current file set failed.\n The vector file has been removed from the disk since the last loading.");
				break;
			}
		case 1000000:
			{
				this->popupInfo("Reloading of the current file set failed.\n The tensor file has been removed from the disk since the last loading.");
				break;
			}
		default:
			{
				this->popupError("Reloading of the current file set failed.\n Please report IFRIT bug with error code:",ret);
				break;
			}
		}
	}

}


void iQT::fixVariableSettings(int nold, int nnew)
{
	int i, k;
	
	iVTKWindow::getCurrentWindow()->getGateway()->Modified();
	for(i=0; i<=iVTKWindow::getCurrentWindow()->getVolumeFamily()->getMaxMemberIndex(); i++) iVTKWindow::getCurrentWindow()->getVolumeFamily()->getMember(i)->getHistogramConverter()->Update();

	if(nold <= nnew) return; 

	iDataReader *sr = iVTKWindow::getCurrentWindow()->getReader();
	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) if(iVTKWindow::getWindow(k)->getReader() == sr) // all clones
	{
		for(i=0; i<=iVTKWindow::getCurrentWindow()->getSurfaceFamily()->getMaxMemberIndex(); i++)
		{
			iSurface *sf = iVTKWindow::getCurrentWindow()->getSurfaceFamily()->getMember(i);
			if(sf->getVar() > nnew) sf->setVar(1);
			if(sf->getVarPaint(false) > nnew) sf->setVarPaint(false,0);
			if(sf->getVarPaint(true) > nnew) sf->setVarPaint(true,0);
		}
		
		for(i=0; i<iVTKWindow::getCurrentWindow()->getXsectionFamily()->getMaxMemberIndex(); i++)
		{
			iXsection *sf = iVTKWindow::getCurrentWindow()->getXsectionFamily()->getMember(i);
			if(sf->getVar() > nnew) sf->setVar(1);
		}
		
		for(i=0; i<=iVTKWindow::getCurrentWindow()->getVolumeFamily()->getMaxMemberIndex(); i++)
		{
			iVolume *sf = iVTKWindow::getCurrentWindow()->getVolumeFamily()->getMember(i);
			if(sf->getVar() > nnew) sf->setVar(1);
		}
	}

}




