/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iqt.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iglobals.h"

#include "iqtdefs.h"
#include "iqt_part.h"
#include "iqt_view.h"
#include "iqtextension.h"
#include "ierror.h"
#include "ihistogrammaker.h"
#include "ilabel.h"
#include "imath.h"
#include "iparticles.h"
#include "iparticlesdataconverter.h"
#include "iparticlesconnector.h"
#include "iparticlessplitter.h"
#include "ipiecewisefunction.h"
#include "ilimits.h"
#include "ipalette.h"
#include "idatareader.h"

#include <vtkLODActor.h>

#include <qaction.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcolordialog.h> 
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qlineedit.h>
#include <qmemarray.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qtabwidget.h>
#include <qvalidator.h>


#define CURRENT_LIMITS	iVTKWindow::getCurrentWindow()->getReader()->getLimits(CURRENT_PARTICLES->getDataChannel(),CURRENT_PARTICLES->getDataStream())


QPixmap image(char *s);
void reportNullPointer(int ec);


int ifrit_compare_float__(const void *elem1, const void *elem2)
{
	float f1 = *(float *)elem1;
	float f2 = *(float *)elem2;
	if(f1 < f2) return -1; else if(f1 > f2) return 1; else return 0;
}

//
// *********************************************************************************
//
//  
//   Particle tool panel widgets
//
//
// *********************************************************************************
//
void iQT_Part::init()
{
	
	myQT = (iQT *)parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget();
	if(strcmp(myQT->name(),"iQT") != 0)
	{
		iQTWindow::getCurrentWindow()->popupError("Incorrect parenting in Tab Pages",IFRITERROR_INCORRECT_PARENTING_IN_TABS);
		exit(1);
	}

	iconsShown = 0;
    cpPart_Size_Function = 0;
	cpPart_Piece_Splitter = 0;
	//
	//  Make sliders look better
	//
	LayoutPartOpacity->setStretchFactor(LayoutPartOpacityLeft,3);
	LayoutPartOpacity->addStretch(1);
	
	LayoutPartPaintMinMax->setStretchFactor(Part_Paint_MinMaxFrame,3);
	LayoutPartPaintMinMax->addStretch(1);
	
	LayoutPartSizeMinMax->setStretchFactor(Part_Size_MinMaxFrame,3);
	LayoutPartSizeMinMax->addStretch(1);
	
	LayoutPartMaterial->setStretchFactor(Part_Paint_MaterialGroupBox,3);
	LayoutPartMaterial->addStretch(1);
	//
	//  Disable splitting without data present
	//
	TabWidget->page(3)->setEnabled(false);
	//
	//  Set validator for the extra scale factor line edit
	//
	QDoubleValidator *v = new QDoubleValidator(Part_Size_ExtraFactorLineEdit);
	v->setBottom(1.0e-100);
	v->setTop(1.0e100);
	Part_Size_ExtraFactorLineEdit->setValidator(v);

}


void iQT_Part::doRender()
{
        myQT->doRender();
}


void iQT_Part::updatePart_Widgets()
{
	static int nInstance = -1;
	static iVTK *myVTK = 0;
	int i;
	QString s1;
    
	//
	//  update instance lists if VTK or the # of instances changed
	//
	if(iVTKWindow::getCurrentWindow()!=myVTK || nInstance!=iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMaxMemberIndex())
	{
		if(iVTKWindow::getCurrentWindow() != myVTK)
		{
			CALL_FUNCTION1(Part_Piece_HistogramResolutionSlider,setValue,0);
		}

		myVTK = iVTKWindow::getCurrentWindow();
		nInstance = iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMaxMemberIndex();

		Part_Main_PieceList->blockSignals(true);
		Part_Main_PieceList->clear();
		for(i=0; i<=nInstance; i++)
		{
			s1 = s1.setNum(i+1);
			Part_Main_PieceList->insertItem("Piece "+s1,-1);
		}
		Part_Main_PieceList->blockSignals(false);
		
		Part_Piece_DeleteCurrentPieceButton->setEnabled((nInstance > 0));
		Part_Piece_CreateNewPieceButton->setEnabled((nInstance < MAXINSTANCE-1));

		cpPart_Piece_Splitter = iVTKWindow::getCurrentWindow()->getParticlesFamily()->getCurrentMemberIndex();
		CALL_FUNCTION1(Part_Main_PieceList,setCurrentItem,cpPart_Piece_Splitter);

	}

    int v;
    float f;
    iParticles* s = CURRENT_PARTICLES;
	QPixmap p;
	resetNames = true;
//
//  Update tab icons
//
	if(s->getDataChannel() == 0)
	{
		if(iconsShown != 0)
		{
			iconsShown = 0;
			resetNames = true;
			p = image("part.png");
			TabWidget->changeTab(TabWidget->page(0),p,QString("Main"));
			TabWidget->changeTab(TabWidget->page(1),p,QString("Paint"));
			TabWidget->changeTab(TabWidget->page(2),p,QString("Size"));
			TabWidget->changeTab(TabWidget->page(3),p,QString("Advanced"));
			myQT->actionShowPart->setIconSet(QIconSet(p));
		}
	}
	//
	//  Enable/disable attribute buttons
	//
	if(iVTKWindow::getCurrentWindow()->getReader()->isTherePartData(s->getDataChannel(),s->getDataStream()))
	{
		v = CURRENT_LIMITS->getNatt();
	}
	else v = 0;

	CALL_FUNCTION1(Part_Paint_PaintAtt1,setEnabled,(v>0));
	CALL_FUNCTION1(Part_Size_SizeAtt1,setEnabled,(v>0));
	CALL_FUNCTION1(Part_Piece_SplitWithAtt1,setEnabled,(v>0));
	CALL_FUNCTION1(Part_Connt_ConnectWithAtt1,setEnabled,(v>0));
	CALL_FUNCTION1(Part_Connt_SeparateWithAtt1,setEnabled,(v>0));

	CALL_FUNCTION1(Part_Paint_PaintAtt2,setEnabled,(v>1));
	CALL_FUNCTION1(Part_Size_SizeAtt2,setEnabled,(v>1));
	CALL_FUNCTION1(Part_Piece_SplitWithAtt2,setEnabled,(v>1));
	CALL_FUNCTION1(Part_Connt_ConnectWithAtt2,setEnabled,(v>1));
	CALL_FUNCTION1(Part_Connt_SeparateWithAtt2,setEnabled,(v>1));

	CALL_FUNCTION1(Part_Paint_PaintAtt3,setEnabled,(v>2));
	CALL_FUNCTION1(Part_Size_SizeAtt3,setEnabled,(v>2));
	CALL_FUNCTION1(Part_Piece_SplitWithAtt3,setEnabled,(v>2));
	CALL_FUNCTION1(Part_Connt_ConnectWithAtt3,setEnabled,(v>2));
	CALL_FUNCTION1(Part_Connt_SeparateWithAtt3,setEnabled,(v>2));
//
//  Main panel widgets
//    
	myQT->TabWidgetView->updateSwitchesPart();

    CALL_FUNCTION1(Part_Main_Type,setButton,s->getType()); 

    TabWidget->setTabEnabled(TabWidget->page(2),s->getType()>0);

    CALL_FUNCTION1(Part_Main_ColorLabel,setPaletteBackgroundColor,s->getColor());
    
    CALL_FUNCTION1(Part_Main_Opacity,setValue,round(40.0+10.0*log10(s->getOpacity())));
    
    CALL_FUNCTION1(Part_Main_Size,setValue,round(10.0*log10(s->getFixedSize())));
//
//  Paint panel widgets
//  
    v = s->getAttToColor();

    CALL_FUNCTION1(Part_Paint_PaintWithBox,setButton,v);
    if(s->getLogScaleToColor()) 
	{
		CALL_FUNCTION1(Part_Paint_Scale,setButton,1); 
	}
	else
	{
		CALL_FUNCTION1(Part_Paint_Scale,setButton,0);
	}
    
    Part_Paint_PaintFrame->setEnabled((v>0));
    
    CALL_FUNCTION1(Part_Paint_PaletteList,setCurrentItem,s->getPal());
    myQT->updatePalettePixmap(iPaletteList::getInstance()->getPalette(s->getPal()),&myQT->pixmapPaint);
    Part_Paint_PalettePixmap->setPixmap(myQT->pixmapPaint);
        
    float fmin, fmax, flo, fhi;
    fmin = CURRENT_LIMITS->getAttMin(v);
    fmax = CURRENT_LIMITS->getAttMax(v);
    flo = CURRENT_LIMITS->getAttLow(v);
    fhi = CURRENT_LIMITS->getAttHigh(v);
    
    if(s->getLogScaleToColor() && v>0) 
	{
		
		fmin = log10(1.0e-30+fabs(fmin));
		fmax = log10(1.0e-30+fabs(fmax));
		flo = log10(1.0e-30+fabs(flo));
		fhi = log10(1.0e-30+fabs(fhi));
		
		int lmin = round(myQT->scaleSlider1*fmin);
		int lmax = round(myQT->scaleSlider1*fmax);
		int llo = round(myQT->scaleSlider1*flo);
		int lhi = round(myQT->scaleSlider1*fhi);
		
		Part_Paint_MinLCD->display((double)llo/myQT->scaleSlider1);    
		CALL_FUNCTION1(Part_Paint_MinSlider,setMinValue,lmin);
		CALL_FUNCTION1(Part_Paint_MinSlider,setMaxValue,lmax);
		CALL_FUNCTION1(Part_Paint_MinSlider,setValue,llo);
		
		Part_Paint_MaxLCD->display((double)lhi/myQT->scaleSlider1);    
		CALL_FUNCTION1(Part_Paint_MaxSlider,setMinValue,lmin);
		CALL_FUNCTION1(Part_Paint_MaxSlider,setMaxValue,lmax);
		CALL_FUNCTION1(Part_Paint_MaxSlider,setValue,lhi);
		
    } 
	else 
	{
		
		Part_Paint_MinLCD->display((double)flo);    
		CALL_FUNCTION1(Part_Paint_MinSlider,setMinValue,0);
		CALL_FUNCTION1(Part_Paint_MinSlider,setMaxValue,100);
		CALL_FUNCTION1(Part_Paint_MinSlider,setValue,round(100.0*(flo-fmin)/(fmax-fmin)));
		
		Part_Paint_MaxLCD->display((double)fhi);    
		CALL_FUNCTION1(Part_Paint_MaxSlider,setMinValue,0);
		CALL_FUNCTION1(Part_Paint_MaxSlider,setMaxValue,100);
		CALL_FUNCTION1(Part_Paint_MaxSlider,setValue,round(100.0*(fhi-fmin)/(fmax-fmin)));
		
    }
//
//  Materials panel
//    
    CALL_FUNCTION1(Part_Paint_MaterialShade,setChecked,CURRENT_PARTICLES->getShade());
    
    f = CURRENT_PARTICLES->getDiffuse();
    CALL_FUNCTION1(Part_Paint_MaterialDiffuseSlider,setValue,round(100.0*f));
    Part_Paint_MaterialDiffuseLCD->display((double)f);
    
    f = CURRENT_PARTICLES->getAmbient();
    CALL_FUNCTION1(Part_Paint_MaterialAmbientSlider,setValue,round(100.0*f));
    Part_Paint_MaterialAmbientLCD->display((double)f);
    
    f = CURRENT_PARTICLES->getSpecular();
    CALL_FUNCTION1(Part_Paint_MaterialSpecularSlider,setValue,round(100.0*f));
    Part_Paint_MaterialSpecularLCD->display((double)f);
    
    f = CURRENT_PARTICLES->getSpecularPower();
    CALL_FUNCTION1(Part_Paint_MaterialSpecularPowerSlider,setValue,round(f));
    Part_Paint_MaterialSpecularPowerLCD->display((double)f);
//
//  Size panel 
//    
    v = s->getAttToSize();

    CALL_FUNCTION1(Part_Size_SizeWithBox,setButton,v);
    if(s->getLogScaleToSize()) 
	{
		CALL_FUNCTION1(Part_Size_Scale,setButton,1); 
	}
	else
	{
		CALL_FUNCTION1(Part_Size_Scale,setButton,0);
	}

	TabWidgetPartSize->setTabEnabled(TabWidgetPartSize->page(1),!CURRENT_PARTICLES->getAttSizeDirect());
    CALL_FUNCTION1(Part_Size_DirectSizeCheckBox,setChecked,CURRENT_PARTICLES->getAttSizeDirect());
	CALL_FUNCTION1(Part_Size_ExtraFactorLineEdit,setText,QString::number(CURRENT_PARTICLES->getAttSizeExtraFactor()));

    fmin = CURRENT_LIMITS->getAttMin(v);
    fmax = CURRENT_LIMITS->getAttMax(v);
    flo = CURRENT_LIMITS->getAttLow(v);
    fhi = CURRENT_LIMITS->getAttHigh(v);
    
    if(s->getLogScaleToSize() && v>0) 
	{
		
		fmin = log10(1.0e-30+fabs(fmin));
		fmax = log10(1.0e-30+fabs(fmax));
		flo = log10(1.0e-30+fabs(flo));
		fhi = log10(1.0e-30+fabs(fhi));
		
		int lmin = round(myQT->scaleSlider1*fmin);
		int lmax = round(myQT->scaleSlider1*fmax);
		int llo = round(myQT->scaleSlider1*flo);
		int lhi = round(myQT->scaleSlider1*fhi);
		
		Part_Size_MinLCD->display((double)llo/myQT->scaleSlider1);    
		CALL_FUNCTION1(Part_Size_MinSlider,setMinValue,lmin);
		CALL_FUNCTION1(Part_Size_MinSlider,setMaxValue,lmax);
		CALL_FUNCTION1(Part_Size_MinSlider,setValue,llo);
		
		Part_Size_MaxLCD->display((double)lhi/myQT->scaleSlider1);    
		CALL_FUNCTION1(Part_Size_MaxSlider,setMinValue,lmin);
		CALL_FUNCTION1(Part_Size_MaxSlider,setMaxValue,lmax);
		CALL_FUNCTION1(Part_Size_MaxSlider,setValue,lhi);
		
    } 
	else 
	{
		
		Part_Size_MinLCD->display((double)flo);    
		CALL_FUNCTION1(Part_Size_MinSlider,setMinValue,0);
		CALL_FUNCTION1(Part_Size_MinSlider,setMaxValue,100);
		CALL_FUNCTION1(Part_Size_MinSlider,setValue,round(100.0*(flo-fmin)/(fmax-fmin)));
		
		Part_Size_MaxLCD->display((double)fhi);    
		CALL_FUNCTION1(Part_Size_MaxSlider,setMinValue,0);
		CALL_FUNCTION1(Part_Size_MaxSlider,setMaxValue,100);
		CALL_FUNCTION1(Part_Size_MaxSlider,setValue,round(100.0*(fhi-fmin)/(fmax-fmin)));
		
    }
    
	updatePart_Size_FunctionPixmap(0);
//
//  Pieces panel
//    
	TabWidget->page(3)->setEnabled(CURRENT_LIMITS->getNatt()>0);

	CALL_FUNCTION1(Part_Piece_Activate,setChecked,(CURRENT_PARTICLES->getParticlesSplitter()->getAttribute()>0));

	CALL_FUNCTION1(Part_Piece_SplitWithBox,setButton,CURRENT_PARTICLES->getParticlesSplitter()->getAttribute()-1);

    if(CURRENT_PARTICLES->getParticlesSplitter()->getLogScale()) 
	{
		CALL_FUNCTION1(Part_Piece_ScaleBox,setButton,1); 
	}
	else
	{
		CALL_FUNCTION1(Part_Piece_ScaleBox,setButton,0);
	}
//
//  Connect panel
//
    CALL_FUNCTION1(Part_Connt_Activate,setChecked,CURRENT_PARTICLES->getConnector()->isConnected());
	CALL_FUNCTION1(Part_Connt_ConnectWith,setButton,CURRENT_PARTICLES->getConnector()->getAttributeToConnect());
	CALL_FUNCTION1(Part_Connt_SeparateWith,setButton,CURRENT_PARTICLES->getConnector()->getAttributeToSeparate());
    CALL_FUNCTION1(Part_Connt_LineWidthSlider,setValue,round(CURRENT_PARTICLES->getLineWidth()));

    if(myQT->ext != NULL) myQT->ext->updateWidgets(DISPLAYPAGE_PART);

	//
	//  Reset attribute names if needed
	//
	if(resetNames)
	{
		CALL_FUNCTION1(Part_Paint_PaintAtt1,setText,CURRENT_LIMITS->getAttName(1));
		CALL_FUNCTION1(Part_Size_SizeAtt1,setText,CURRENT_LIMITS->getAttName(1));
		CALL_FUNCTION1(Part_Piece_SplitWithAtt1,setText,CURRENT_LIMITS->getAttName(1));
		CALL_FUNCTION1(Part_Connt_ConnectWithAtt1,setText,CURRENT_LIMITS->getAttName(1));
		CALL_FUNCTION1(Part_Connt_SeparateWithAtt1,setText,CURRENT_LIMITS->getAttName(1));
		CALL_FUNCTION1(Part_Paint_PaintAtt2,setText,CURRENT_LIMITS->getAttName(2));
		CALL_FUNCTION1(Part_Size_SizeAtt2,setText,CURRENT_LIMITS->getAttName(2));
		CALL_FUNCTION1(Part_Piece_SplitWithAtt2,setText,CURRENT_LIMITS->getAttName(2));
		CALL_FUNCTION1(Part_Connt_ConnectWithAtt2,setText,CURRENT_LIMITS->getAttName(2));
		CALL_FUNCTION1(Part_Connt_SeparateWithAtt2,setText,CURRENT_LIMITS->getAttName(2));
		CALL_FUNCTION1(Part_Paint_PaintAtt3,setText,CURRENT_LIMITS->getAttName(3));
		CALL_FUNCTION1(Part_Size_SizeAtt3,setText,CURRENT_LIMITS->getAttName(3));
		CALL_FUNCTION1(Part_Piece_SplitWithAtt3,setText,CURRENT_LIMITS->getAttName(3));
		CALL_FUNCTION1(Part_Connt_ConnectWithAtt3,setText,CURRENT_LIMITS->getAttName(3));
		CALL_FUNCTION1(Part_Connt_SeparateWithAtt3,setText,CURRENT_LIMITS->getAttName(3));
	}

	myQT->updateDataChannelComboBox(DISPLAYPAGE_PART);

}


void iQT_Part::updatePart_Main_Piece( int vtkNotUsed(n) )
{

	cpPart_Piece_Splitter = Part_Main_PieceList->currentItem();
	iVTKWindow::getCurrentWindow()->getParticlesFamily()->setCurrentMemberIndex(cpPart_Piece_Splitter);
	this->updatePart_Widgets();
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

}


void iQT_Part::updatePart_Main_Type( int v )
{
    
	myQT->startTask();
    CURRENT_PARTICLES->setType(v);
    updatePart_Widgets();
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Main_ColorLabel()
{
    
    QColor c = QColorDialog::getColor(); 
	if(c.isValid())
	{
		myQT->startTask();
		CALL_FUNCTION1(Part_Main_ColorLabel,setPaletteBackgroundColor,c);
		CURRENT_PARTICLES->setColor(c);
		myQT->render(true);
		myQT->endTask();
	}

}


void iQT_Part::updatePart_Main_Opacity( int v )
{
    
    CURRENT_PARTICLES->setOpacity(pow(10.0,-4.0+0.1*v));
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Main_Size( int v )
{

    CURRENT_PARTICLES->setFixedSize(pow(10.0,0.1*v));
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Paint_PaintWithBox( int v )
{
    
	myQT->startTask();
    CURRENT_PARTICLES->setAttToColor(v);
    updatePart_Widgets();
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Paint_ScaleBox( int v )
{
    
	myQT->startTask();
    CURRENT_PARTICLES->setLogScaleToColor((v==1));
    updatePart_Widgets();
    myQT->render(true);    
	myQT->endTask();

}


void iQT_Part::updatePart_Paint_PalettePixmap( int n )
{
    
    myQT->updatePalettePixmap(iPaletteList::getInstance()->getPalette(n),&myQT->pixmapPaint);
    Part_Paint_PalettePixmap->setPixmap(myQT->pixmapPaint);

}


void iQT_Part::updatePart_Paint_PalettePixmapFinal( int n )
{

    int no = CURRENT_PARTICLES->getPal();
    if(n != no ) 
	{
		myQT->startTask();
		updatePart_Paint_PalettePixmap(n);
		CURRENT_PARTICLES->setPal(n);
		myQT->render(true);
		myQT->endTask();
    }
    
}


void iQT_Part::updatePart_Paint_MinSlider( int l )
{
    
    float f;
    int v = CURRENT_PARTICLES->getAttToColor();
    if(CURRENT_PARTICLES->getLogScaleToColor() && v>0) 
	{
		f = pow10(l/myQT->scaleSlider1);
    } 
	else 
	{
		f = CURRENT_LIMITS->getAttMin(v) + 0.01*l*(CURRENT_LIMITS->getAttMax(v)-CURRENT_LIMITS->getAttMin(v));
    }
    CURRENT_LIMITS->setAttLow(v,f);
    updatePart_Widgets();
    CURRENT_PARTICLES->setLogScaleToColor(CURRENT_PARTICLES->getLogScaleToColor());
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Paint_MaxSlider( int l )
{
    
    float f;
    int v = CURRENT_PARTICLES->getAttToColor();
    if(CURRENT_PARTICLES->getLogScaleToColor() && v>0) 
	{
		f = pow10(l/myQT->scaleSlider1);
    } 
	else 
	{
		f = CURRENT_LIMITS->getAttMin(v) + 0.01*l*(CURRENT_LIMITS->getAttMax(v)-CURRENT_LIMITS->getAttMin(v));
    }
    CURRENT_LIMITS->setAttHigh(v,f);
    updatePart_Widgets();
    CURRENT_PARTICLES->setLogScaleToColor(CURRENT_PARTICLES->getLogScaleToColor());
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Size_SizeWithBox( int v )
{
    
	myQT->startTask();
    CURRENT_PARTICLES->setAttToSize(v);
    updatePart_Widgets();
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Size_ScaleBox( int v )
{
    
	myQT->startTask();
    CURRENT_PARTICLES->setLogScaleToSize((v==1));
    updatePart_Widgets();
    myQT->render(true);    
	myQT->endTask();

}


void iQT_Part::updatePart_Size_MinSlider( int l )
{
    
    float f;
    int v = CURRENT_PARTICLES->getAttToSize();
    if(CURRENT_PARTICLES->getLogScaleToSize() && v>0) 
	{
		f = pow10(l/myQT->scaleSlider1);
    } 
	else 
	{
		f = CURRENT_LIMITS->getAttMin(v) + 0.01*l*(CURRENT_LIMITS->getAttMax(v)-CURRENT_LIMITS->getAttMin(v));
    }
    CURRENT_LIMITS->setAttLow(v,f);
    updatePart_Widgets();
    CURRENT_PARTICLES->setLogScaleToSize(CURRENT_PARTICLES->getLogScaleToSize());
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Size_MaxSlider( int l )
{
    
    float f;
    int v = CURRENT_PARTICLES->getAttToSize();
    if(CURRENT_PARTICLES->getLogScaleToSize() && v>0) 
	{
		f = pow10(l/myQT->scaleSlider1);
    } 
	else 
	{
		f = CURRENT_LIMITS->getAttMin(v) + 0.01*l*(CURRENT_LIMITS->getAttMax(v)-CURRENT_LIMITS->getAttMin(v));
    }
    CURRENT_LIMITS->setAttHigh(v,f);
    updatePart_Widgets();
    CURRENT_PARTICLES->setLogScaleToSize(CURRENT_PARTICLES->getLogScaleToSize());
    if(myQT->sliderRenderTracking) myQT->render(false);

}


void iQT_Part::updatePart_Size_FunctionPixmap( QMouseEvent * e )
{
    
	int catt = CURRENT_PARTICLES->getHistogramMaker()->getInputComponent();
    myQT->updateFunctionPixmap(e,CURRENT_PARTICLES->getSizeFunction(),Part_Size_FunctionPixmap,
		cpPart_Size_Function,CURRENT_PARTICLES->getHistogramMaker()->getHistogram(0,CURRENT_PARTICLES->getLogScaleToSize(),CURRENT_LIMITS->getAttLow(catt),CURRENT_LIMITS->getAttHigh(catt)));
	
}


void iQT_Part::updatePart_Size_FunctionResButton()
{

    int n = CURRENT_PARTICLES->getSizeFunction()->getFunctionN();
    for(int i=0; i<n-2; i++) CURRENT_PARTICLES->getSizeFunction()->delFunctionPoint(1);
    CURRENT_PARTICLES->getSizeFunction()->movFunctionPoint(0,0.0,0.1);
    CURRENT_PARTICLES->getSizeFunction()->movFunctionPoint(1,1.0,0.1);
    cpPart_Size_Function=0;
    updatePart_Size_FunctionPixmap(0);
    
}

void iQT_Part::updatePart_Size_FunctionAddButton()
{

    CURRENT_PARTICLES->getSizeFunction()->addFunctionPoint(cpPart_Size_Function);
    if(cpPart_Size_Function <  CURRENT_PARTICLES->getSizeFunction()->getFunctionN()-1) cpPart_Size_Function++;
    updatePart_Size_FunctionPixmap(0);
    
}

void iQT_Part::updatePart_Size_FunctionDelButton()
{

    CURRENT_PARTICLES->getSizeFunction()->delFunctionPoint(cpPart_Size_Function);
    if(cpPart_Size_Function > 0) cpPart_Size_Function--;
    updatePart_Size_FunctionPixmap(0);
    
}


void iQT_Part::updatePart_Size_FunctionApply()
{

    CURRENT_PARTICLES->getConverter()->Modified();
	myQT->startTask();
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_Part::updatePart_Paint_MaterialShade( bool s )
{

    CURRENT_PARTICLES->setShade(s);
	myQT->startTask();
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Paint_MaterialAmbientSlider( int n )
{

    CURRENT_PARTICLES->setAmbient(0.01*n);
    Part_Paint_MaterialAmbientLCD->display((double)0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}


void iQT_Part::updatePart_Paint_MaterialDiffuseSlide( int n )
{

    CURRENT_PARTICLES->setDiffuse(0.01*n);
    Part_Paint_MaterialDiffuseLCD->display((double)0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}


void iQT_Part::updatePart_Paint_MaterialSpecularSlider( int n )
{

    CURRENT_PARTICLES->setSpecular(0.01*n);
    Part_Paint_MaterialSpecularLCD->display((double)0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}


void iQT_Part::updatePart_Paint_MaterialSpecularPower( int n )
{
    
    CURRENT_PARTICLES->setSpecularPower((float)n);
    Part_Paint_MaterialSpecularPowerLCD->display((double)n);
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}


void iQT_Part::launchPaletteEditor()
{
	
	myQT->launchPaletteEditor(Part_Paint_PaletteList->currentItem());
	
}


void iQT_Part::updatePart_Piece_Activate(bool s)
{

	myQT->startTask();
//	CURRENT_PARTICLES->getParticlesSplitter()->setActive(s);
	if(s)
	{
		CURRENT_PARTICLES->getParticlesSplitter()->setAttribute(1+Part_Piece_SplitWithBox->id(Part_Piece_SplitWithBox->selected()));
	}
	else
	{
		CURRENT_PARTICLES->getParticlesSplitter()->setAttribute(0);
		this->updatePart_Piece_RestoreToOnePiece();
	}
    myQT->render(true);
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);
	myQT->endTask();

}


void iQT_Part::updatePart_Piece_SplitWithBox(int vtkNotUsed(n))
{

	myQT->startTask();
	CURRENT_PARTICLES->getParticlesSplitter()->setAttribute(1+Part_Piece_SplitWithBox->id(Part_Piece_SplitWithBox->selected()));
    myQT->render(true);
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);
	myQT->endTask();
	
}


void iQT_Part::updatePart_Piece_ScaleBox(int n)
{

	CURRENT_PARTICLES->getParticlesSplitter()->setLogScale((n==1));
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

}


void iQT_Part::updatePart_Piece_Tile(bool s)
{

	if(CURRENT_PARTICLES->getParticlesSplitter()->getAttribute()>0 && s)
	{
		
		float attMin, attMax, aMin, aMax;
		int npic = CURRENT_PARTICLES->getParticlesSplitter()->getNumberOfPieces();
		CURRENT_PARTICLES->getParticlesSplitter()->getAttributeMinMax(attMin,attMax);

		float *aLo = new float[npic];
		if(aLo == 0) reportNullPointer(1302);

		int i;
		for(i=0; i<npic; i++)
		{
			CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(i,aMin,aMax);
			aLo[i] = aMin;
		}
		//
		//  STDLIB qsort - since the number of pieces is likely to be small
		//
		qsort(aLo,npic,sizeof(float),ifrit_compare_float__);
		//
		//  Tile pieces
		//
		for(i=1; i<npic-1; i++)
		{
			aMin = aLo[i];
			aMax = aLo[i+1];
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(i,aMin,aMax);
		}

		if(npic > 1)
		{
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(0,attMin,aLo[1]);
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(npic-1,aLo[npic-1],attMax);
		}
		else
		{
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(0,attMin,attMax);
		}

		this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

		delete [] aLo;

	}

}


void iQT_Part::updatePart_Piece_CreateNewPiece()
{
    int i;
    QString s;
    
	i = iVTKWindow::getCurrentWindow()->getParticlesFamily()->createMember();
    if(i != -1) 
	{
		//
		//  Create a new piece
		//
		int newPiece = CURRENT_PARTICLES->getParticlesSplitter()->createNewPiece(0.0,1.0);
		if(newPiece == 0)
		{
			myQT->popupError("Cannot create a new piece",IFRITERROR_CANNOT_CREATE_PARTICLES_PIECE);
			iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMember(i)->getActor()->VisibilityOff();
		}
		//
		//  Split the current piece into two
		//
		if(newPiece>0 && cpPart_Piece_Splitter>=0 && cpPart_Piece_Splitter<CURRENT_PARTICLES->getParticlesSplitter()->getNumberOfPieces())
		{
			float amin, amax, acen;
			CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(cpPart_Piece_Splitter,amin,amax);
			if(Part_Piece_ScaleBox->id(Part_Piece_ScaleBox->selected()) == 1)
			{
				acen = sqrt(fabs(amin*amax));
			}
			else
			{
				acen = 0.5*(amin+amax);
			}
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter,amin,acen);
			CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(newPiece,acen,amax);
		}
		//
		//  Copy the state
		//
		iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMember(i)->copyState(CURRENT_PARTICLES);
		//
		//  Set the correct piece number
		//
		iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMember(i)->setParticlesSplitterPiece(newPiece);
		//
		//  update widgets
		//
		iVTKWindow::getCurrentWindow()->getParticlesFamily()->setCurrentMemberIndex(i);
        CURRENT_PARTICLES->show(myQT->TabWidgetView->View_Part->isChecked());

		this->updatePart_Widgets();
		this->updatePart_Piece_Tile(Part_Piece_Tile->isChecked());
		this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

    }
    else Part_Piece_CreateNewPieceButton->setEnabled(false);


}


void iQT_Part::updatePart_Piece_DeleteCurrentPiece()
{
    
    if(iVTKWindow::getCurrentWindow()->getParticlesFamily()->deleteMember(iVTKWindow::getCurrentWindow()->getParticlesFamily()->getCurrentMemberIndex()))
	{
        CURRENT_PARTICLES->show(myQT->TabWidgetView->View_Part->isChecked());
		this->updatePart_Widgets();
		this->updatePart_Piece_Tile(Part_Piece_Tile->isChecked());
		this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

    } 
    
}


void iQT_Part::updatePart_Piece_RestoreToOnePiece()
{
    int i,n;
    QString s;
    
	n = iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMaxMemberIndex();
	if(n > 0)
	{
		for(i=0; i<n; i++) iVTKWindow::getCurrentWindow()->getParticlesFamily()->deleteMember(1);
        CURRENT_PARTICLES->show(myQT->TabWidgetView->View_Part->isChecked());
    } 
	//
	//  Reset splitter by setting the attribute to 0 and setting it back to what it was
	//
	int att = CURRENT_PARTICLES->getParticlesSplitter()->getAttribute();
	CURRENT_PARTICLES->getParticlesSplitter()->setAttribute(0);
	CURRENT_PARTICLES->getParticlesSplitter()->setAttribute(att);
	cpPart_Piece_Splitter = 0;

	this->updatePart_Widgets();
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);

}


void iQT_Part::updatePart_Piece_Apply()
{

	myQT->startTask();
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Piece_SplitterPixmap( QMouseEvent * e )
{
	static const int lineWidth = 3;
    static int mouseXold=0, mouseYold=0;
	static QPixmap pix = QPixmap(256,256);
	static QPainter z;
	static QPen p;
	static int tol = 15;
    float dx;
    int i, i1, ixl = 0, iyl = 0, idx, idxmin, indmin, ixmax, iymax;
    int ix1, iy1, ix2;
	bool inEvent, inClick;
	static bool isMax = false;

    ixmax = Part_Piece_SplitterPixmap->geometry().width()-1; iymax = Part_Piece_SplitterPixmap->geometry().height()-1;
	
	if((1+ixmax)!=pix.width() || (1+iymax)!=pix.height())
	{
		pix = QPixmap(ixmax+1,iymax+1);
	}
	
	z.begin(&pix);
    z.eraseRect(0,0,pix.width(),pix.height());
    
	if(CURRENT_PARTICLES->getParticlesSplitter()->getAttribute()>0)
	{
		
		bool inLog = CURRENT_PARTICLES->getParticlesSplitter()->getLogScale();
		int nBins = CURRENT_PARTICLES->getParticlesSplitter()->getHistogramMaker()->getEstimatedNumberOfBins();
		if(nBins > 0)
		{
			nBins = round(nBins*pow10(0.1*Part_Piece_HistogramResolutionSlider->value()));
			if(nBins < 2) nBins = 2;
			myQT->drawHistogram(z,CURRENT_PARTICLES->getParticlesSplitter()->getHistogramMaker()->getHistogram(nBins,inLog),myQT->histoColor,1.0,true,false,0);
		}

		p.setWidth(lineWidth);
		p.setColor(QColor(255-z.backgroundColor().red(),255-z.backgroundColor().green(),255-z.backgroundColor().blue()));
		z.setPen(p);
		
		float attMin, attMax, aMin, aMax, a;
		int npic = CURRENT_PARTICLES->getParticlesSplitter()->getNumberOfPieces();
		CURRENT_PARTICLES->getParticlesSplitter()->getAttributeMinMax(attMin,attMax);

		int old_cpPart_Piece_Splitter = cpPart_Piece_Splitter;

		if(inLog)
		{
			attMin = log10(1.0e-30+fabs(attMin));
			attMax = log10(1.0e-30+fabs(attMax));
		}

		dx = ixmax/(attMax-attMin);

		if(e != 0 ) 
		{
			ixl = e->x();
			iyl = iymax - e->y();
		}
		inEvent = (e != 0);
		inClick = inEvent && (e->button()==Qt::LeftButton || e->button()==Qt::RightButton);

		idxmin = ixmax;
		indmin = -1;
		for(i=0; i<=npic; i++)
		{
			if(i == npic)
			{
				if(inClick && indmin>=0 && idxmin<tol)
				{
					cpPart_Piece_Splitter = indmin;
				}
				i1 = cpPart_Piece_Splitter;
				p.setColor(QColor(z.backgroundColor().red(),255-z.backgroundColor().green(),255-z.backgroundColor().blue()));
				z.setPen(p);
				CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(cpPart_Piece_Splitter,aMin,aMax);
			}
			else
			{
				i1 = i;
				CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(i,aMin,aMax);
			}

			if(inLog)
			{
				aMin = log10(1.0e-30+fabs(aMin));
				aMax = log10(1.0e-30+fabs(aMax));
			}

			ix1 = round(dx*(aMin-attMin));
			if(ix1>=0 && ix1<lineWidth/2) ix1 = lineWidth/2;
			ix2 = round(dx*(aMax-attMin));
			if(ix2<=ixmax && ix2>ixmax-lineWidth/2) ix2 = ixmax-lineWidth/2;
			z.drawLine(ix1,0,ix1,iymax);
			z.drawLine(ix2,0,ix2,iymax);
			iy1 = round(iymax*(0.05+0.4*i1/npic));
			z.drawLine(ix1,iy1,ix2,iy1);

			if(inClick && i<npic)
			{
				idx = ixl - ix1;
				if(idx>=0 && idxmin>idx)
				{
					indmin = i;
					idxmin = idx;
					isMax = false;
				}
				idx = ix2 - ixl;
				if(idx>=0 && idxmin>idx)
				{
					indmin = i;
					idxmin = idx;
					isMax = true;
				}
			}

		}					

		if(inEvent) 
		{
			if(ixl!=mouseXold && e->state()!=Qt::NoButton && e->stateAfter()!=Qt::NoButton)
			{
				a = ixl/dx + attMin;
				if(inLog) a = pow10(a);
				CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(cpPart_Piece_Splitter,aMin,aMax);
				CURRENT_PARTICLES->getParticlesSplitter()->getAttributeMinMax(attMin,attMax);

				if(Part_Piece_Tile->isChecked())
				{
					if(npic > 1)
					{
						if(cpPart_Piece_Splitter == 0)
						{
							if(isMax)
							{
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(0,attMin,a);
								CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(1,aMin,aMax);
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(1,a,aMax);
							}
						}
						else if(cpPart_Piece_Splitter == npic-1)
						{
							if(!isMax)
							{
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(npic-1,a,attMax);
								CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(npic-2,aMin,aMax);
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(npic-2,aMin,a);
							}
						}
						else
						{
							if(isMax)
							{	
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter,aMin,a);
								CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(cpPart_Piece_Splitter+1,aMin,aMax);
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter+1,a,aMax);
							}
							else
							{
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter,a,aMax);
								CURRENT_PARTICLES->getParticlesSplitter()->getAttributeLimits(cpPart_Piece_Splitter-1,aMin,aMax);
								CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter-1,aMin,a);
							}
						}
					}
				}
				else
				{
					if(isMax) aMax = a; else aMin = a;	
					CURRENT_PARTICLES->getParticlesSplitter()->setAttributeLimits(cpPart_Piece_Splitter,aMin,aMax);
				}
			
			}
			mouseXold = ixl;
			mouseYold = iyl;
		}
		
		if(cpPart_Piece_Splitter != old_cpPart_Piece_Splitter)
		{
			CALL_FUNCTION1(Part_Main_PieceList,setCurrentItem,cpPart_Piece_Splitter);
			iVTKWindow::getCurrentWindow()->getParticlesFamily()->setCurrentMemberIndex(cpPart_Piece_Splitter);
			this->updatePart_Widgets();
		}

	    Part_Piece_SplitterPixmap->setPixmap(pix);
		if(e!=0 && e->state()==Qt::RightButton) myQT->render(true);
    
	}
	else
	{
	    Part_Piece_SplitterPixmap->setPixmap(pix);
	}

    z.end();
    
}



void iQT_Part::updatePart_Piece_HistogramResolution( int vtkNotUsed(n) )
{
	this->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);
}


void iQT_Part::updatePart_Connt_Activate(bool s)
{

	myQT->startTask();
	CURRENT_PARTICLES->getConnector()->setConnected(s);
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Connt_SeparateWith(int n)
{

	myQT->startTask();
	CURRENT_PARTICLES->getConnector()->setAttributeToSeparate(n);
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Connt_ConnectWith(int n)
{

	myQT->startTask();
	CURRENT_PARTICLES->getConnector()->setAttributeToConnect(n);
    myQT->render(true);
	myQT->endTask();

}


void iQT_Part::updatePart_Connt_LineWidthSlider( int n )
{

    CURRENT_PARTICLES->setLineWidth(n);
    if(myQT->sliderRenderTracking) myQT->render(false);
    
}

void iQT_Part::updatePart_Size_DirectSize(bool s)
{
	TabWidgetPartSize->setTabEnabled(TabWidgetPartSize->page(1),!s);
	myQT->startTask();
	CURRENT_PARTICLES->setAttSizeDirect(s);
    myQT->render(true);
	myQT->endTask();
}


void iQT_Part::updatePart_Size_ExtraFactor(const QString &s)
{
	bool ok = false;
	double f = s.toDouble(&ok);

	if(ok)
	{
		myQT->startTask();
		CURRENT_PARTICLES->setAttSizeExtraFactor(f);
		myQT->render(true);
		myQT->endTask();
	}
}
