/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "iqt.h"
#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "iqtextension.h"
#include "iqtdefs.h"
#include "iqt_vect.h"
#include "iqt_view.h"

#include "imath.h"
#include "ivector.h"
#include "ilimits.h"
#include "ipalette.h"
#include "idatareader.h"
#include "ierror.h"

#include <qaction.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcolordialog.h> 
#include <qcombobox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qtabwidget.h>


#define CURRENT_LIMITS	iVTKWindow::getCurrentWindow()->getReader()->getLimits(CURRENT_VECTOR->getDataChannel())


//
// *********************************************************************************
//
//  
//   Vector field panel widgets
//
//
// *********************************************************************************
//
void iQT_Vect::init()
{

	myQT = (iQT *)parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget()->parentWidget();
	if(strcmp(myQT->name(),"iQT") != 0)
	{
		iQTWindow::getCurrentWindow()->popupError("Incorrect parenting in Tab Pages",IFRITERROR_INCORRECT_PARENTING_IN_TABS);
		exit(1);
	}

	//
	//  Make sliders look better
	//
	LayoutVectMagnitude->setStretchFactor(LayoutVectMagnitudeLeft,3);
	LayoutVectMagnitude->addStretch(1);
	
	LayoutVectMaterial->setStretchFactor(Vect_Paint_MaterialGroupBox,3);
	LayoutVectMaterial->addStretch(1);
	
}


void iQT_Vect::doRender()
{
	myQT->doRender();
}


void iQT_Vect::updateVect_Widgets()
{
	int n;
	float f;
	
	//
	//  Variable names
	//
	Vect_Paint_PaintVar1->setText(CURRENT_LIMITS->getVarName(1));
    Vect_Paint_PaintVar2->setText(CURRENT_LIMITS->getVarName(2));
    Vect_Paint_PaintVar3->setText(CURRENT_LIMITS->getVarName(3));
	//
	//  Main panel wisgets
	//
	myQT->TabWidgetView->updateSwitchesVect();
	
	n = CURRENT_VECTOR->getMode();
	CALL_FUNCTION1(Vect_Main_Mode,setButton,n);
	
    CALL_FUNCTION1(Vect_Main_HedgeColorLabel,setPaletteBackgroundColor,CURRENT_VECTOR->getColor());
	
	CALL_FUNCTION1(Vect_Main_HedgeSizeSlider,setValue,round(10*log10((double)CURRENT_VECTOR->getHedgeSize())));
	
	f = CURRENT_LIMITS->getVecHigh();
	CALL_FUNCTION1(Vect_Main_MagnitudeSlider,setValue,round(10.0*log10(f/CURRENT_LIMITS->getVecMax())));
	Vect_Main_MagnitudeLCD->display(f);
	
	CALL_FUNCTION1(Vect_Main_ResampleSlider,setValue,round(log10((double)CURRENT_VECTOR->getSampleRate())/log10(2.0)));
	Vect_Main_ResampleLCD->display(CURRENT_VECTOR->getSampleRate());
	//
	//  Paint panel
	//
	CALL_FUNCTION1(Vect_Paint_PaintWithButtons,setButton,CURRENT_VECTOR->getColorBy());
	
    CALL_FUNCTION1(Vect_Paint_PaletteList,setCurrentItem,CURRENT_VECTOR->getPal());
    myQT->updatePalettePixmap(iPaletteList::getInstance()->getPalette(CURRENT_VECTOR->getPal()),&myQT->pixmapPaint);
    Vect_Paint_PalettePixmap->setPixmap(myQT->pixmapPaint);
	//
	//  Material panel widgets
	//    
    f = CURRENT_VECTOR->getDiffuse();
    CALL_FUNCTION1(Vect_Paint_MaterialDiffuseSlider,setValue,round(100.0*f));
    Vect_Paint_MaterialDiffuseLCD->display((double)f);
    
    f = CURRENT_VECTOR->getAmbient();
    CALL_FUNCTION1(Vect_Paint_MaterialAmbientSlider,setValue,round(100.0*f));
    Vect_Paint_MaterialAmbientLCD->display((double)f);
    
    f = CURRENT_VECTOR->getSpecular();
    CALL_FUNCTION1(Vect_Paint_MaterialSpecularSlider,setValue,round(100.0*f));
    Vect_Paint_MaterialSpecularLCD->display((double)f);
    
    f = CURRENT_VECTOR->getSpecularPower();
    CALL_FUNCTION1(Vect_Paint_MaterialSpecularPowerSlider,setValue,round(f));
    Vect_Paint_MaterialSpecularPowerLCD->display((double)f);
	//
	//  Streamlines panel
	//
	CALL_FUNCTION1(Vect_Lines_EdgeButtons,setButton,CURRENT_VECTOR->getStartSide());
	
	CALL_FUNCTION1(Vect_Lines_SourceTypeButtonGroup,setButton,CURRENT_VECTOR->getSourceType());

	CALL_FUNCTION1(Vect_Lines_DirectionButtons,setButton,CURRENT_VECTOR->getDir());
	
	this->updateCoordinateDisplays();
	
	CALL_FUNCTION1(Vect_Lines_QualitySlider,setValue,CURRENT_VECTOR->getQuality());
	
	Vect_Lines_LengthLCD->display(CURRENT_VECTOR->getLength());
	CALL_FUNCTION1(Vect_Lines_LengthSlider,setValue,round(10.0*log10((double)CURRENT_VECTOR->getLength())));
	
	CALL_FUNCTION1(Vect_Lines_TubeSizeSlider,setValue,CURRENT_VECTOR->getTubeSize());
	
	CALL_FUNCTION1(Vect_Lines_TubeRangeSlider,setValue,round(CURRENT_VECTOR->getTubeRangeFactor()));

	CALL_FUNCTION1(Vect_Lines_TubeVariationSlider,setValue,round(10.0*log10(CURRENT_VECTOR->getTubeVariationFactor())));

    if(myQT->ext != NULL) myQT->ext->updateWidgets(DISPLAYPAGE_VECT);
	myQT->updateDataChannelComboBox(DISPLAYPAGE_VECT);
	
}


void iQT_Vect::updateCoordinateDisplays()
{
	double x = CURRENT_VECTOR->getPos();
    CALL_FUNCTION1(Vect_Lines_PosSlider,setMinValue,round(myQT->scaleSlider2*(-1.0)));
    CALL_FUNCTION1(Vect_Lines_PosSlider,setMaxValue,round(myQT->scaleSlider2*1.0));
    CALL_FUNCTION1(Vect_Lines_PosSlider,setValue,round(myQT->scaleSlider2*x));
    Vect_Lines_PosLCD->display(iVTKWindow::getCurrentWindow()->transformToExternalCoordinate(x));
}


void iQT_Vect::updateVect_Main_Mode( int n )
{
	
	if(n != CURRENT_VECTOR->getMode()){
		myQT->startTask();
		CURRENT_VECTOR->setMode(n);
		myQT->render(true);
		myQT->endTask();
	}
	
}


void iQT_Vect::updateVect_Main_Resample( int n )
{
	
	int rr = round(pow(2.0,n));
	Vect_Main_ResampleLCD->display(rr);
	CURRENT_VECTOR->setSampleRate(rr);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Main_HedgeSize( int n )
{
	
	CURRENT_VECTOR->setHedgeSize(pow10(0.1*n));
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Main_HedgeColor()
{
	
    QColor c = QColorDialog::getColor(); 
	if(c.isValid())
	{
		myQT->startTask();
		CALL_FUNCTION1(Vect_Main_HedgeColorLabel,setPaletteBackgroundColor,c);
		CURRENT_VECTOR->setColor(c);
		myQT->render(true);
		myQT->endTask();
	}
	
}


void iQT_Vect::updateVect_Main_Magnitude( int n )
{
	
	float f = CURRENT_LIMITS->getVecMax()*pow10(0.1*n);
	CURRENT_LIMITS->setVecHigh(1,f);
	Vect_Main_MagnitudeLCD->display(f);
	CURRENT_VECTOR->updatePipeline();
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Paint_PalettePixmap(int n)
{
    
    myQT->updatePalettePixmap(iPaletteList::getInstance()->getPalette(n),&myQT->pixmapPaint);
    Vect_Paint_PalettePixmap->setPixmap(myQT->pixmapPaint);
    
}


void iQT_Vect::updateVect_Paint_PalettePixmapFinal( int n )
{
	
	myQT->startTask();
    updateVect_Paint_PalettePixmap(n);
    CURRENT_VECTOR->setPal(n);
	updateVect_Widgets();  // needed for the colorbars to work properly
    myQT->render(true);
	myQT->endTask();
    
}


void iQT_Vect::updateVect_Lines_Edge( int n )
{
	
	myQT->startTask();
	CURRENT_VECTOR->setStartSide(n);
	myQT->render(true);
	myQT->endTask();
	
}


void iQT_Vect::updateVect_Lines_Direction( int n )
{
	
	myQT->startTask();
	CURRENT_VECTOR->setDir(n);
	myQT->render(true);
	myQT->endTask();
	
}


void iQT_Vect::updateVect_Lines_Position( int n )
{
	
    float f = n/myQT->scaleSlider2;
    Vect_Lines_PosLCD->display(iVTKWindow::getCurrentWindow()->transformToExternalCoordinate((double)f));
	CURRENT_VECTOR->setPos(f);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Lines_Quality( int n )
{
	
	CURRENT_VECTOR->setQuality(n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}



void iQT_Vect::updateVect_Lines_TubeSize( int n )
{
	
	CURRENT_VECTOR->setTubeSize(n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Lines_TubeRangeFactor( int n )
{
	
	CURRENT_VECTOR->setTubeRangeFactor((float)n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Lines_TubeVariationFactor( int n )
{
	
	CURRENT_VECTOR->setTubeVariationFactor(pow(10.0,0.1*n));
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Paint_PaintWithButtons( int n )
{
	
	myQT->startTask();
	CURRENT_VECTOR->setColorBy(n);
    myQT->render(true);
	myQT->endTask();
	
}


void iQT_Vect::updateVect_Lines_Length( int n )
{
	
	float l = pow(10.0,0.1*n);
	CURRENT_VECTOR->setLength(l);
	Vect_Lines_LengthLCD->display((float)l);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Lines_SourceTypeButtonGroup( int n )
{

	myQT->startTask();
	CURRENT_VECTOR->setSourceType(n);
    myQT->render(true);
	myQT->endTask();

}


void iQT_Vect::updateVect_Paint_MaterialAmbientSlider( int n )
{
	
	CURRENT_VECTOR->setAmbient(0.01*n);
	Vect_Paint_MaterialAmbientLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Paint_MaterialDiffuseSlider( int n )
{
	
	CURRENT_VECTOR->setDiffuse(0.01*n);
	Vect_Paint_MaterialDiffuseLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Paint_MaterialSpecularSlider( int n )
{
	
	CURRENT_VECTOR->setSpecular(0.01*n);
	Vect_Paint_MaterialSpecularLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::updateVect_Paint_MaterialSpecularPower( int n )
{
	
	CURRENT_VECTOR->setSpecularPower(0.01*n);
	Vect_Paint_MaterialSpecularPowerLCD->display(0.01*n);
    if(myQT->sliderRenderTracking) myQT->render(false);
	
}


void iQT_Vect::launchPaletteEditor()
{
	
	myQT->launchPaletteEditor(Vect_Paint_PaletteList->currentItem());
	
}


void iQT_Vect::updateTabPageChanged(QWidget *)
{
	myQT->updateDataChannelComboBox(DISPLAYPAGE_VECT);
}
