/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/
  
//
//  This class is based on original vtkQGLRenderWindow class written by Manish P. Pagey [pagey@drcsdca.com ]
//

#include "iglobals.h"
#include "irenderwindowinteractor.h"
#include "irenderwindow.h"

//#include <stdio.h>
//#include <stdlib.h>
//#include <string>
//#include <cmath>


#include <qkeycode.h>

#include <vtkActor.h>
#include <vtkInteractorStyle.h>
#include <vtkCommand.h>


iRenderWindowInteractor* iRenderWindowInteractor::New() 
{
	return new iRenderWindowInteractor;
}

iRenderWindowInteractor::iRenderWindowInteractor() 
{
	this->Enabled = 0;
	this->mTimer = new QTimer( this );
	connect(mTimer, SIGNAL(timeout()), this, SLOT(TimerFunc()));
	my_render_win = 0;
}


iRenderWindowInteractor::~iRenderWindowInteractor() 
{
	delete mTimer;
}


void iRenderWindowInteractor::SetRenderWindow(iRenderWindow *win)
{
	vtkRenderWindowInteractor::SetRenderWindow(win);
	my_render_win = win;
}
//
// We never allow the iRenderWindowInteractor to control 
// the event loop. The application always has the control. 
//
void iRenderWindowInteractor::Initialize() 
{
	//
	// We cannot do much unless there is a render window 
	// associated with this interactor. 
	//
	if(!RenderWindow ) 
	{
		vtkErrorMacro(<< "iRenderWindowInteractor::Initialize(): No render window attached!") ;
		return ;
	}
	
	//
	// If the render window has zero size, then set it to a default 
	// value of 300x300.
	// 
	int *size = my_render_win->GetSize();
	size[0] = ((size[0] > 0) ? size[0] : 300);
	size[1] = ((size[1] > 0) ? size[1] : 300);
	
	//
	// Enable the interactor. 
	//
	this->Enable();
	
	//
	// Start the rendering of the window. 
	//
	my_render_win->Start();
	
	//
	// Set our size to that of the render window. 
	//
	this->Size[0] = size[0];
	this->Size[1] = size[1];
	
	//
	// The interactor has been initialized.
	//
	this->Initialized = 1;
	return;
}


void iRenderWindowInteractor::Enable() 
{
	//
	// Do not need to do anything if already enabled.
	//
	if(this->Enabled) return;
	
	//
	// Attach slots to every useful signal of the render window. 
	//
	this->ConnectSlots();
	
	this->Enabled = 1;
	this->Modified();
}


void iRenderWindowInteractor::Disable() 
{
	if(!this->Enabled) return;
	
	this->DisconnectSlots();
	this->Enabled = 0;
	this->Modified();
}


void iRenderWindowInteractor::Start() 
{
	//
	// We do not allow this interactor to control the 
	// event loop. Only the QtApplication objects are
	// allowed to do that. 
	//
	vtkErrorMacro(<<"iRenderWindowInteractor::Start() not allowed to start event loop.");
	return;
}


int iRenderWindowInteractor::CreateTimer(int vtkNotUsed(timertype)) 
{
	//
	// Start a one-shot timer for 10ms. 
	//
	mTimer->start(10, true);
	return 1;
}


int iRenderWindowInteractor::DestroyTimer(void) 
{
	//
	// :TRICKY: Tue May  2 00:17:32 2000 Pagey
	//
	// QTimer will automatically expire after 10ms. So 
	// we do not need to do anything here. In fact, we 
	// should not even Stop() the QTimer here because doing 
	// this will skip some of the processing that the TimerFunc()
	// does and will result in undesirable effects. For 
	// example, this will result in vtkLODActor to leave
	// the models in low-res mode after the mouse stops
	// moving. 
	//
	return 1;
}


void iRenderWindowInteractor::TimerFunc() 
{
	if(!this->Enabled) return;

	((vtkInteractorStyle *)this->InteractorStyle)->OnTimer();
	emit RenderWindowModified();
}

//
// Dummy definitions for a few slots.
//

void iRenderWindowInteractor::ConnectSlots() 
{
	ProcessSlotConnections(true);
}


void iRenderWindowInteractor::DisconnectSlots() 
{
	ProcessSlotConnections(false);
}


void iRenderWindowInteractor::ProcessSlotConnections(bool conn) 
{
	//
	// We cannot do much unless there is a render window 
	// associated with this interactor. 
	//
	if(!RenderWindow) 
	{
		vtkErrorMacro(<< "iRenderWindowInteractor::Initialize(): No render window attached!");
		return;
	}
	
	bool (*slot_func) ( const QObject * sender, 
		      const char * signal, 
			  const QObject * receiver, 
			  const char * member );
	if( conn ) 
	{
		slot_func = &QObject::connect;
	} 
	else 
	{
		slot_func = &QObject::disconnect;
	}
	
	slot_func( my_render_win, SIGNAL(LeftButtonPressed(const QMouseEvent *)), 
		this,          SLOT(LeftButtonPressed(const QMouseEvent *)) ) ;
	slot_func( my_render_win, SIGNAL(LeftButtonReleased(const QMouseEvent *)), 
		this,          SLOT(LeftButtonReleased(const QMouseEvent *)) ) ;
	
	slot_func( my_render_win, SIGNAL(MiddleButtonPressed(const QMouseEvent *)), 
		this,          SLOT(MiddleButtonPressed(const QMouseEvent *)) ) ;
	slot_func( my_render_win, SIGNAL(MiddleButtonReleased(const QMouseEvent *)), 
		this,          SLOT(MiddleButtonReleased(const QMouseEvent *)) ) ;
	
	slot_func( my_render_win, SIGNAL(RightButtonPressed(const QMouseEvent *)), 
		this,          SLOT(RightButtonPressed(const QMouseEvent *)) ) ;
	slot_func( my_render_win, SIGNAL(RightButtonReleased(const QMouseEvent *)), 
		this,          SLOT(RightButtonReleased(const QMouseEvent *)) ) ;
	
	slot_func( my_render_win, SIGNAL(MouseMove(const QMouseEvent *)), 
		this,          SLOT(MouseMove(const QMouseEvent *)) ) ;
	
	slot_func( my_render_win, SIGNAL(KeyPressed(QKeyEvent *)),
		this,          SLOT(KeyPressed(QKeyEvent *)) ) ;
	
	slot_func( this,          SIGNAL(RenderWindowModified()), 
		my_render_win, SLOT(updateGL()) ) ;
	
}
//
//
//
#ifndef I_VTK_VERSION
#error ERROR: iglobals.h is not included
$TERMINATE$
#endif

void iRenderWindowInteractor::MouseMove(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
		
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnMouseMove();
	this->InvokeEvent(vtkCommand::MouseMoveEvent, NULL);
		
#else
	
	this->InteractorStyle->OnMouseMove(event->state() & ControlButton, 
		event->state() & ShiftButton,
		event->x(), 
		this->Size[1] - event->y() - 1) ;
	
#endif
		
	emit RenderWindowModified() ;
	
}
//
//
//
void iRenderWindowInteractor::LeftButtonPressed(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)(this->InteractorStyle))->OnLeftButtonDown();
	this->InvokeEvent(vtkCommand::LeftButtonPressEvent, NULL);
		
#else
	
	this->InteractorStyle->OnLeftButtonDown( (event->state() & ControlButton), 
					   (event->state() & ShiftButton), 
					   event->x(),
					   this->Size[1] - event->y() - 1 ) ;
	
#endif
			
}
//
//
//
void iRenderWindowInteractor::LeftButtonReleased(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnLeftButtonUp();
	this->InvokeEvent(vtkCommand::LeftButtonReleaseEvent, NULL);
		
#else
	
	this->InteractorStyle->OnLeftButtonUp( (event->state() & ControlButton), 
		(event->state() & ShiftButton), 
		event->x(),
		this->Size[1] - event->y() - 1 ) ;
	
#endif
			
}
//
//
//
void iRenderWindowInteractor::MiddleButtonPressed(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnMiddleButtonDown();
	this->InvokeEvent(vtkCommand::MiddleButtonPressEvent, NULL);
		
#else
	
	this->InteractorStyle->OnMiddleButtonDown( (event->state() & ControlButton), 
		(event->state() & ShiftButton), 
		event->x(),
		this->Size[1] - event->y() - 1 ) ;
	
#endif
	
		
}
//
//
//
void iRenderWindowInteractor::MiddleButtonReleased(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnMiddleButtonUp();
	this->InvokeEvent(vtkCommand::MiddleButtonReleaseEvent, NULL);
		
#else
	
	this->InteractorStyle->OnMiddleButtonUp( (event->state() & ControlButton), 
					   (event->state() & ShiftButton), 
					   event->x(),
					   this->Size[1] - event->y() - 1 ) ;
	
#endif
			
}
//
//
//
void iRenderWindowInteractor::RightButtonPressed(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnRightButtonDown();
	this->InvokeEvent(vtkCommand::RightButtonPressEvent, NULL);
		
#else
	
	this->InteractorStyle->OnRightButtonDown( (event->state() & ControlButton), 
		(event->state() & ShiftButton), 
		event->x(),
		this->Size[1] - event->y() - 1 ) ;
	
#endif
	
		
}
//
//
//
void iRenderWindowInteractor::RightButtonReleased(const QMouseEvent *event) 
{
	if(!this->Enabled) return;
	
#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetEventPosition(event->x(),this->Size[1]-event->y()-1);
	this->SetKeySym("0");
//	((vtkInteractorStyle *)this->InteractorStyle)->OnRightButtonUp();
	this->InvokeEvent(vtkCommand::RightButtonReleaseEvent, NULL);
		
#else
	
	this->InteractorStyle->OnRightButtonUp( (event->state() & ControlButton), 
		(event->state() & ShiftButton), 
		event->x(),
		this->Size[1] - event->y() - 1 ) ;
	
#endif
			
}
//
//
//
void iRenderWindowInteractor::KeyPressed(QKeyEvent *event) 
{

	if(!this->Enabled) return;
	
	if(event->key()==Qt::Key_F || (event->key()==Qt::Key_Escape&&RenderWindow->GetFullScreen()))
	{
		RenderWindow->SetFullScreen(!RenderWindow->GetFullScreen());
		emit RenderWindowModified() ;
		return;
	}

#ifndef I_VTK_VERSION_40
	
	this->SetControlKey(event->state()&ControlButton);
	this->SetShiftKey(event->state()&ShiftButton);
	this->SetKeyCode(event->key());
	this->SetKeySym("0");
//	((vtkInteractorStyle *)(this->InteractorStyle))->OnChar();
	this->InvokeEvent(vtkCommand::CharEvent, NULL);
		
#else
	
	this->InteractorStyle->OnChar( event->state() & ControlButton, 
		event->state() & ShiftButton, 
		event->key(), 
		1 ) ;
	
#endif
			
	//
	// :TODO: Fri Apr 28 23:21:24 2000 Pagey
	//
	// Its a waste to emit this signal unless the key pressed was
	// one of the keys recognized by the InteractorStyle object.
	// I will change this once I figure out how to do this. 
	//
	emit RenderWindowModified() ;
}


