/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of ivector.h
*/

#include "iglobals.h"
#include "ivector.h"

#include "ipalette.h"
#include "ivtk.h"
#include "ilimits.h"
#include "iglobals.h"
#include "iobjectfactory.h"

#include "imath.h"
#include "istreamline.h"
#include "ivtkcallback.h"
#include "idatareader.h"
#include "ipolygonaldata.h"
#include "iuniformmeshdata.h"
#include "ipolygonaldataextender.h"
#include "ivisualobjectobserver.h"
#include "icolorbars.h"
#include "imarker.h"

#include <vtkCommand.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkActor.h>
#include <vtkProperty.h>
#include <vtkHedgeHog.h>
#include <vtkTubeFilter.h>
#include <vtkExtractVOI.h>
#include <vtkIdTypeArray.h>
#include <vtkRuledSurfaceFilter.h>
#include <vtkLookupTable.h>


void reportNullPointer(int ec);


//
// iVector class
//

iVector* iVector::New(iVTK *mVTK)
{
	return (iVector *)iObjectFactory::createObject("Vector",mVTK);
}


iVector::iVector(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	pal = 0;
	mode = 0;
	tubeSize = 1;
	quality = 0;
	startSide = 0;
	sampleRate = 16;
	hedgeSize = 1.0;
	length = 1.0;
	pos = -1.0;
	colorBy = 0;
	dir = 2;
	tubeRangeFactor = 10;
	tubeVariationFactor = 0.01; 
	sourceType = 0;

	color = NICE_DEFAULT_COLOR;
	//
	//  Do VTK stuff
	//	
	vectorActor = vtkLODActor::New();
	if(vectorActor == 0) reportNullPointer(6201);
	vectorMapper = vtkPolyDataMapper::New();
	if(vectorMapper == 0) reportNullPointer(6202);
	vectorLine = iStreamLine::New();
	if(vectorLine == 0) reportNullPointer(6203);
	vectorTube = vtkTubeFilter::New();
	if(vectorTube == 0) reportNullPointer(6205);
	vectorExtender = iPolygonalDataExtender::New(this);
	if(vectorExtender == 0) reportNullPointer(6206);
	vectorSource = vtkPoints::New(VTK_DOUBLE);
	if(vectorSource == 0) reportNullPointer(6207);
	vectorHedge = vtkHedgeHog::New();
	if(vectorHedge == 0) reportNullPointer(6208);
	vectorFilter = vtkExtractVOI::New();
	if(vectorFilter == 0) reportNullPointer(6209);
	vectorRibbons = vtkRuledSurfaceFilter::New();
	if(vectorRibbons == 0) reportNullPointer(6210);
	
	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
	vectorMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorLine->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorTube->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorExtender->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorHedge->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorFilter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	vectorRibbons->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	
	vectorFilter->SetInput(myVTK->getReader()->getVectorOutput());

	vectorLine->SetInput(myVTK->getReader()->getVectorOutput());
	vectorLine->setSource(vectorSource);
	vectorLine->setDirection(dir);

	vectorHedge->SetInput(vectorFilter->GetOutput());

	vectorTube->SetInput(vectorLine->GetOutput());
	vectorTube->SetNumberOfSides(6);
	vectorTube->SetVaryRadiusToVaryRadiusByScalar();

	vectorRibbons->SetDistanceFactor(15.0);
	vectorRibbons->SetRuledModeToPointWalk();
	vectorRibbons->SetOffset(0);
	vectorRibbons->SetOnRatio(2);
	vectorRibbons->PassLinesOn();
	vectorRibbons->SetInput(vectorLine->GetOutput());

	vectorMapper->SetInput(vectorExtender->GetOutput());
    vectorMapper->ScalarVisibilityOff();
	vectorMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
	vectorMapper->SetScalarModeToDefault();
	vectorMapper->UseLookupTableScalarRangeOff();
	
	vectorActor->SetMapper(vectorMapper);
	vectorActor->SetPosition(0.0,0.0,0.0);

	vectorProperty = vectorActor->GetProperty();
	vectorProperty->SetInterpolationToGouraud();
	vectorProperty->SetAmbient(ambient);
	vectorProperty->SetDiffuse(diffuse);
	vectorProperty->SetSpecular(specular);
	vectorProperty->SetSpecularPower(specularPower);
	//
	//  Add observer to keep information about this object
	//
	vectorActor->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_VECT,IDATATYPE_VECT);

	vectorActor->VisibilityOff();

}


iVector::~iVector()
{
	
	if(!init) myVTK->removeProp(vectorActor);
	vectorActor->Delete();	
	vectorMapper->Delete();
	vectorLine->Delete();
	vectorTube->Delete();
	vectorExtender->Delete();
	vectorSource->Delete();
	vectorHedge->Delete();
	vectorFilter->Delete();
	vectorRibbons->Delete();

}


void iVector::reset()
{
	vectorFilter->SetInput(myVTK->getReader()->getVectorOutput());
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(vectorActor);
		init = true;
	}
}


void iVector::computeSpacing()
{
	curSpacing = myVTK->getReader()->getVectorOutput()->GetSpacing()[0];
}


void iVector::updatePipeline()
{

	float vmax = this->getLimits()->getVecHigh();
	this->computeSpacing();
	vectorHedge->SetScaleFactor(hedgeSize*sampleRate*curSpacing/vmax);
	vectorLine->setMinimumSpeed(tubeVariationFactor*vmax);

	if(colorBy == 0) 
	{
		vectorMapper->ScalarVisibilityOff();
	} 
	else 
	{
		vectorMapper->ScalarVisibilityOn();
		switch (colorBy) 
		{
		case 1:	
			{ 
				vectorMapper->ColorByArrayComponent(0,1); 
				vectorMapper->SetScalarRange(this->getLimits()->getVecLow(),this->getLimits()->getVecHigh());
				break;
			}
		case 2: 
			{
				vectorMapper->ColorByArrayComponent(0,2); 
				vectorMapper->SetScalarRange(this->getLimits()->getVecLow()/curSpacing,this->getLimits()->getVecHigh()/curSpacing);
				break;
			}
		case 3: 
			{
				vectorMapper->ColorByArrayComponent(0,3); 
				vectorMapper->SetScalarRange(this->getLimits()->getVecLow()/curSpacing,this->getLimits()->getVecHigh()/curSpacing);
				break;
			}
		}

		if(colorBy > 3 && colorBy <= 3+this->getLimits()->getNvar()) 
		{
			vectorMapper->ColorByArrayComponent(0,colorBy); 
			vectorMapper->SetScalarRange(this->getLimits()->getVarLow(colorBy-3),this->getLimits()->getVarHigh(colorBy-3));
		}
	
	}

	if(mode == 0)
	{
		vectorExtender->SetInput(vectorHedge->GetOutput()); 
		if(sampleRate == 1)
		{
			vectorHedge->SetInput(myVTK->getReader()->getVectorOutput());
		} 
		else 
		{
			vectorFilter->SetSampleRate(sampleRate,sampleRate,sampleRate);
			vectorHedge->SetInput(vectorFilter->GetOutput());
		}
		return;
	}

	if(mode==1 || mode==2 || mode==3)
	{

		vectorLine->setQuality(quality);
		vectorLine->setLength(2.0*length);
		vectorLine->setSplitLines(false);

		if(mode == 1)
		{
			vectorExtender->SetInput(vectorLine->GetOutput()); 
			vectorActor->GetProperty()->SetLineWidth(tubeSize);
		}

		if(mode == 2)
		{
			vectorExtender->SetInput(vectorTube->GetOutput());
			float r = 0.005*tubeSize;
			vectorTube->SetRadius(r);
			vectorTube->SetRadiusFactor(tubeRangeFactor);
		}

		if(mode == 3)
		{
			vectorLine->setSplitLines(true);
			vectorExtender->SetInput(vectorRibbons->GetOutput());
		}

		return;
	}

}


void iVector::computeFilterInputParameters(int* dims, vtkFloat* spa, vtkFloat* org)
{
	vectorFilter->GetInput()->GetDimensions(dims); 
	vectorFilter->GetInput()->GetSpacing(spa); 
	vectorFilter->GetInput()->GetOrigin(org); 
}


void iVector::updateSource()
{ 
	int i, j, k, dir1, dir2, dims[3];
	vtkIdType n, l;
	vtkFloat spa[3], org[3];
	double x[3];

	int dir = startSide;
	dir1 = (dir+1)%3; dir2 = (dir+2)%3;

	vectorLine->setAxisDirection(dir);
	vectorSource->Initialize();

	this->computeFilterInputParameters(dims,spa,org);
	
	switch(sourceType)
	{
	case 1:
		{
			vectorSource->SetNumberOfPoints(myVTK->getMarkerFamily()->getMaxMemberIndex()+1);
			
			for(i=0; i<=myVTK->getMarkerFamily()->getMaxMemberIndex(); i++)
			{
				vectorSource->SetPoint(i,myVTK->getMarkerFamily()->getMember(i)->getPosition());
			}
			break;
		}
	default:
		{

			n = dims[dir1]*dims[dir2]/sampleRate/sampleRate;
			if(n < 1) return;

			vectorSource->SetNumberOfPoints(n);

			k = round(0.5*(pos+1.0)*(dims[dir]-1));
			x[dir] = org[dir] + spa[dir]*k;
			l = 0;

			for(i=sampleRate/2; i<dims[dir2]; i+=sampleRate)
			{
				for(j=sampleRate/2; j<dims[dir1]; j+=sampleRate)
				{
					x[dir1] = org[dir1] + spa[dir1]*j;
					x[dir2] = org[dir2] + spa[dir2]*i;

					vectorSource->SetPoint(l,x); l++;
				}
			}
			break;
		}
	}

	vectorLine->Modified();
}


void iVector::showClipPlane(bool s)
{

	if(s && !clipPlaneOn)
	{
		vectorMapper->AddClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = true;
	} 
	if(!s && clipPlaneOn)
	{
		vectorMapper->RemoveClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = false;
	}

}


void iVector::showColorBars(bool show)
{
	if(vectorActor->GetVisibility() == 0) return;
	if(colorBy > 3) myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_VECTOR,colorBy-3,dataChannel,pal,show);
}


void iVector::setSourceType(int v)
{ 
	if(v>=0 && v<2 && v!=sourceType) 
	{
		sourceType = v;
		updateSource();
	}
}


void iVector::setMode(int v)
{ 
	if(v>=0 && v<4 && v!=mode) 
	{
		mode = v;
		updatePipeline();
		updateSource();
	}
}


void iVector::setStartSide(int v)
{ 
	if(v>=0 && v<3 && v!=startSide) 
	{
		startSide = v;
		updateSource();
	}
}


void iVector::setDir(int v)
{ 
	if(v>=0 && v<5 && v!=dir) 
	{
		dir = v;
		vectorLine->setDirection(dir);
	}
}


void iVector::setPos(double v)
{ 
	if(v>=-1.0 && v<=1.0 && v!=pos) 
	{
		pos = v;
		updateSource();
	}
}


void iVector::setHedgeSize(float v)
{ 
	if(v>0.0 && v<=1.0e5 && v!=hedgeSize)
	{
		hedgeSize = v;
		updatePipeline();
	}
}


void iVector::setLength(float v)
{ 
	if(v>0.0 && v<100.0 && v!=length)
	{
		length = v;
		updatePipeline();
	}
}


void iVector::setColor(iColor c)
{ 
	color = c;
	vectorProperty->SetColor(color.red()/255.0,color.green()/255.0,color.blue()/255.0);
}


void iVector::setSampleRate(int v)
{ 
	if(v>0 && v<1600 && v!=sampleRate) 
	{
		sampleRate = v;
		vectorFilter->SetSampleRate(sampleRate,sampleRate,sampleRate);
		updateSource();
	}
}


void iVector::setQuality(int v)
{ 
	if(v>=0 && v<10 && v!=quality) 
	{
		quality = v;
		vectorLine->setQuality(v);
	}
}


void iVector::setTubeSize(int v)
{ 
	if(v>=1 && v!=tubeSize && v<=10) 
	{
		tubeSize = v;
		this->updatePipeline();
	}
}


void iVector::setTubeRangeFactor(float q)
{ 
	if(q>1.0 && q<10.1 && fabs(q/tubeRangeFactor-1)>0.1 ) 
	{
		tubeRangeFactor = q;
		this->updatePipeline();
	}
}


void iVector::setTubeVariationFactor(float q)
{ 
	if(q>1.0e-7 && q<1.0 && fabs(q/tubeVariationFactor-1)>0.1 ) 
	{
		tubeVariationFactor = q;
		this->updatePipeline();
	}
}


void iVector::setColorBy(int v)
{ 
	if(v>=0 && v!=colorBy) 
	{
		this->showColorBars(false);
		colorBy = v;
		this->showColorBars(true);
		this->updatePipeline();
	}
}


void iVector::setPal(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	this->showColorBars(false);
	pal = p;
	this->showColorBars(true);
	vectorMapper->SetLookupTable(iPaletteList::getInstance()->getPalette(pal)->getTable());
}


void iVector::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	vectorProperty->SetAmbient(v);
}


void iVector::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	vectorProperty->SetDiffuse(v);
}


void iVector::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	vectorProperty->SetSpecular(v);
}


void iVector::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	vectorProperty->SetSpecularPower(v);
}


void iVector::show(bool show)
{
	  
	if(show)
	{
		if(init)
		{
			this->updateDataChannel();
			this->updatePipeline();
			this->updateSource();
			this->setColor(color);
			myVTK->addProp(vectorActor);
			init = false;
		}
		if(!init)
		{
			updatePipeline();
			updateSource();
			vectorActor->VisibilityOn();
			this->showColorBars(true);
		}
	} 
	else 
	{
		if(!init) 
		{
			this->showColorBars(false);
			vectorActor->VisibilityOff();
		}
	}
	
}


float iVector::getMemorySize()
{
	float s = 0.0;

	s += vectorFilter->GetOutput()->GetActualMemorySize();
	s += vectorSource->GetActualMemorySize();
	s += vectorHedge->GetOutput()->GetActualMemorySize();
	s += vectorLine->getMemorySize();
	s += vectorTube->GetOutput()->GetActualMemorySize();
	s += vectorRibbons->GetOutput()->GetActualMemorySize();
	s += vectorExtender->getMemorySize();
	return s;

}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVector::packState(iString &s)
{

	iVisualObject::packState(s);

	this->packValue(s,"iVector::mode",mode);
	this->packValue(s,"iVector::color",color);
	this->packValue(s,"iVector::pal",pal);
	this->packValue(s,"iVector::tubeSize",tubeSize);
	this->packValue(s,"iVector::quality",quality);
	this->packValue(s,"iVector::colorBy",colorBy);
	this->packValue(s,"iVector::sampleRate",sampleRate);
	this->packValue(s,"iVector::dir",dir);
	this->packValue(s,"iVector::startSide",startSide);

	this->packValue(s,"iVector::pos",pos);
	this->packValue(s,"iVector::length",length);
	this->packValue(s,"iVector::hedgeSize",hedgeSize);

}


void iVector::unpackState(iString s)
{
	int i; float f; iColor c;

	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iVector::mode",i)) this->setMode(i);
	if(this->unpackValue(s,"iVector::color",c)) this->setColor(c);
	if(this->unpackValue(s,"iVector::pal",i)) this->setPal(i);
	if(this->unpackValue(s,"iVector::tubeSize",i)) this->setTubeSize(i);
	if(this->unpackValue(s,"iVector::quality",i)) this->setQuality(i);
	if(this->unpackValue(s,"iVector::colorBy",i)) this->setColorBy(i);
	if(this->unpackValue(s,"iVector::sampleRate",i)) this->setSampleRate(i);
	if(this->unpackValue(s,"iVector::dir",i)) this->setDir(i);
	if(this->unpackValue(s,"iVector::startSide",i)) this->setStartSide(i);
	
	if(this->unpackValue(s,"iVector::pos",f)) this->setPos(f);
	if(this->unpackValue(s,"iVector::length",f)) this->setLength(f);
	if(this->unpackValue(s,"iVector::hedgeSize",f)) this->setHedgeSize(f);

}

