/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
**  Part of implementation of ivtk.h that explicitly depended on QT and does not depend on VTK
*/
#include <qnamespace.h>
#include <qevent.h>
#include <qbuttongroup.h>
#include <qapplication.h>
#include <qimage.h>
#include <qcursor.h>

#include <vtkInteractorStyleTrackballCamera.h>
#include <vtkInteractorStyleFlight.h>
#include <vtkImageData.h>
#include <vtkImageReader2.h>
#include <vtkJPEGReader.h>
#include <vtkPNMReader.h>
#include <vtkBMPReader.h>
#include <vtkTIFFReader.h>
#include <vtkPNGReader.h>
#include <vtkRenderer.h>

#include "ivtk.h"
#include "iqt.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"
#include "ianimator.h"
#include "ixsection.h"

#include "iqtdefs.h"
#include "ierror.h"

#include "imath.h"
#include "imeasuringboxinteractorstyle.h"
#include "idialogpopup.h"
#include "idatareader.h"


bool iVTK::event(QEvent *e)
{
	if(eventsBlocked) 
	{
		return false;
	}
	else
	{
		return QGLWidget::event(e);
	}
}

	
void iVTK::closeEvent(QCloseEvent *e)
{

	if(closeable) e->accept();

}

	
void iVTK::enterEvent(QEvent *e)
{
	iRenderWindow::enterEvent(e);
	//
	//  Set the window under focus as current
	//
	if(focusCurrent && this!=iVTKWindow::getCurrentWindow()) 
	{
		iQTWindow::getCurrentWindow()->setCurrentWindow(this);
	}
	if(autoRaise) 
	{ 
		this->raise(); 
		this->setFocus(); 
	}

	if(bottomWidget == 0) return;

	QRect rb = bottomWidget->geometry();
	QRect r;
	r.setRect(round(rb.x()+0.05*rb.width()),round(rb.y()+0.05*rb.height()),round(0.9*rb.width()),round(0.9*rb.height()));
	popupDialog->setGeometry(r);
	QPoint p(round(0.05*rb.width()),round(0.05*rb.height()));

	int pdMode = 0;
	if(iren->GetInteractorStyle() == istyleView) pdMode = 0;
	if(iren->GetInteractorStyle() == istyleFlight) pdMode = 1;
	if(iren->GetInteractorStyle() == istyleMeasuringBox) pdMode = 2;
	if(animator->getMode() == 4) pdMode = 3;
	popupDialog->showHelp(pdMode);

	popupDialog->reparent(bottomWidget,0,p,true);

}

	
void iVTK::leaveEvent(QEvent *e)
{
	iRenderWindow::leaveEvent(e);

	if(bottomWidget == 0) return;

	popupDialog->reparent(0,0,QPoint(0,0));
	popupDialog->close();

}

	
void iVTK::resizeEvent(QResizeEvent *e)
{

    iRenderWindow::resizeEvent(e);
    if(windowSizeButtonGroup != 0) CALL_FUNCTION1(windowSizeButtonGroup,setButton,0);
    if(windowTileButtonGroup != 0) CALL_FUNCTION1(windowTileButtonGroup,setButton,0);
	iQTWindow::getCurrentWindow()->updateImageComposer(true);

}


void iVTK::moveEvent(QMoveEvent *e)
{

    iRenderWindow::moveEvent(e);
    if(windowTileButtonGroup != 0) CALL_FUNCTION1(windowTileButtonGroup,setButton,0);

}


void iVTK::mouseMoveEvent(QMouseEvent *e)
{
	
	if(inBroadcast) return;
	inBroadcast = true;
	
	if(broadcastMouseEvents)
	{
		int i;
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->iRenderWindow::mouseMoveEvent(e);
	}
	else
	{
		iRenderWindow::mouseMoveEvent(e);
	}
	
	inBroadcast = false;
	
}


void iVTK::mousePressEvent(QMouseEvent *e)
{
	if(broadcastMouseEvents && iVTKWindow::getMaxWindowIndex()>0)
	{
		int i;
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->iRenderWindow::mousePressEvent(e);
	}
	else
	{
		iRenderWindow::mousePressEvent(e);
	}
}


void iVTK::mouseReleaseEvent(QMouseEvent *e)
{
	if(broadcastMouseEvents && iVTKWindow::getMaxWindowIndex()>0)
	{
		int i;
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->iRenderWindow::mouseReleaseEvent(e);
	}
	else
	{
		iRenderWindow::mouseReleaseEvent(e);
	}
}


void iVTK::startRender()
{
	lastRenderTime = ren->GetLastRenderTimeInSeconds();
	if(lastRenderTime > 0.1) 
	{
		iQTWindow::getCurrentWindow()->showBusy(true);
		qApp->flush();
	}
}


void iVTK::endRender()
{
	if(lastRenderTime > 0.1) 
	{
		iQTWindow::getCurrentWindow()->showBusy(false);
		qApp->flush();
	}
}


int iVTK::isReadyForDumpingImage()
{

	if(this->isMinimized()) 
	{
		this->showNormal();
		if(this->isMinimized()) 
		{
			iQTWindow::getCurrentWindow()->popupInfo("The visualization window must be visible.\n Please, restore the visualization window before creating an image.");
			return 2;
		}
	}
	//
	//  Make sure we are ready for the action...
	//
	this->raise();
	qApp->flush();
	this->updateGL();
	qApp->processEvents();

	if(this->isVisible()) return 0; else return 1;

}


void iVTK::reportDumpImageError()
{
	iQTWindow::getCurrentWindow()->popupError("Failed to create image for unknown reason.",IFRITERROR_FAILED_TO_CREATE_IMAGE);
}


void iVTK::createImageByMethod1()
{
	//
	// This does not work under magnification yet.
	//
	int *s = this->GetSize();

	//
	//  QGLWidget::renderPixmap does nor render textures under Unix - force polygobal mode for xsections
	//
	int k;
	for(k=0; k<=xsectionFamily->getMaxMemberIndex(); k++) xsectionFamily->getMember(k)->forcePolygonalMode(true);

	//
	//  render unto a pixmap
	//
	this->vtkRenderWindow::DoubleBuffer = 0; // QGL renders pixmaps in the front buffer
	QImage q = this->renderPixmap(s[0],s[1],false).convertToImage();
	this->vtkRenderWindow::DoubleBuffer = 1; 
	this->convertImage(q,imageData);

	for(k=0; k<=xsectionFamily->getMaxMemberIndex(); k++) xsectionFamily->getMember(k)->forcePolygonalMode(false);

}


void iVTK::setImageComposer(iDialogImageComposer *ic)
{
	imageComposer = ic;
}

//
//  Image manipulation helpers
//

bool iVTK::loadImageFromFile(iString filename, vtkImageData* data)
{
	iString suf = filename.section('.',-1).lower();

	vtkImageReader2 *ir = 0;

	if(suf=="jpg" || suf=="jpeg")
	{
		ir = vtkJPEGReader::New();
	}
	
	if(suf == "pnm")
	{
		ir = vtkPNMReader::New();
	}
	
	if(suf == "bmp")
	{
		ir = vtkBMPReader::New();
	}
	
	if(suf=="tif" || suf=="tiff")
	{
		ir = vtkTIFFReader::New();
	}
	
	if(suf == "png")
	{
		ir = vtkPNGReader::New();
	}
	
	if(ir!=0 && ir->CanReadFile(filename.latin1())==3)
	{
		ir->SetFileName(filename.latin1());
		ir->SetDataScalarTypeToUnsignedChar();
		ir->SetNumberOfScalarComponents(3);
		ir->Update();
		data->DeepCopy(ir->GetOutput());
		ir->Delete();
		return true;
	}
	else return false;

}


QImage iVTK::loadImageFromFile(iString filename)
{
	vtkImageData *d = vtkImageData::New();
	QImage q;
	if(iVTK::loadImageFromFile(filename,d))
	{
		iVTK::convertImage(d,q);
	}
	d->Delete();
	return q;
}


void iVTK::scaleImage(vtkImageData* data, int w, int h)
{
	if(w<2 || h<2)
	{
		data->Initialize();
		return;
	}
	//
	//  Use Qt as the best available implementation
	//
	QImage q;
	iVTK::convertImage(data,q);
	iVTK::convertImage(q.smoothScale(w,h,QImage::ScaleMin),data);
}


void iVTK::convertImage(vtkImageData *d, QImage &s)
{
	if(d==0 || d->GetNumberOfScalarComponents()!=3 || d->GetScalarType()!=VTK_UNSIGNED_CHAR) return;

	int dims[3];
	d->GetDimensions(dims);
	int w = dims[0];
	int h = dims[1];

	s.create(w,h,32);
	unsigned char *sPtr = s.bits();
	unsigned char *dPtr = (unsigned char *)d->GetScalarPointer();

	sPtr += 4*w*(h-1);

	int j, i;
	for(j=0; j<h; j++)
	{
		for(i=0; i<w; i++)
		{
			sPtr[0] = dPtr[2];
			sPtr[1] = dPtr[1];
			sPtr[2] = dPtr[0];
			sPtr[3] = 255;
			dPtr += 3;
			sPtr += 4;
		}
		sPtr -= 4*2*w;
	}
}


void iVTK::convertImage(QImage s, vtkImageData *d)
{
	if(d==0 || s.depth()!=32) return;

	int w = s.width();
	int h = s.height();

	d->SetDimensions(w,h,1);
	d->SetScalarType(VTK_UNSIGNED_CHAR);
	d->SetNumberOfScalarComponents(3);
	d->SetOrigin(0.0,0.0,0.0);
	d->SetSpacing(1.0,1.0,1.0);
	d->AllocateScalars();

	unsigned char *sPtr = s.bits();
	unsigned char *dPtr = (unsigned char *)d->GetScalarPointer();

	if(endiness)
	{
		iDataReader::Swap4BytesRange((char *)sPtr,w*h);
	}

	sPtr += 4*w*(h-1);

	int j, i;
	for(j=0; j<h; j++)
	{
		for(i=0; i<w; i++)
		{
			dPtr[0] = sPtr[2];
			dPtr[1] = sPtr[1];
			dPtr[2] = sPtr[0];
			dPtr += 3;
			sPtr += 4;
		}
		sPtr -= 4*2*w;
	}

}
