/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icrosssectionparallelpipeline.h"


#include "iarray.h"
#include "icrosssectionserialpipeline.h"
#include "icrosssectionviewobject.h"
#include "idatadistributor.h"
#include "ierror.h"
#include "ifamily.h"
#include "iserialpipeline.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"
#include "iparallelpipelinetemplate.h"

//
//  Macros
//
#include "iparallelpipelinemacro.h"

//
// iCrossSectionParallelPipeline class
//
iCrossSectionParallelPipeline::iCrossSectionParallelPipeline(iCrossSectionViewObject *s) : iParallelGridDataToPolyDataPipeline(1,s)
{
	mParent = s;

	mTextureData = vtkImageData::New(); IERROR_ASSERT_NULL_POINTER(mTextureData);
	mTextureData->SetScalarTypeToFloat();
	//
	//  We shouldn't patch the seams since it messes up textures and is incompatible with
	//  optimization iOrthoPolygonPlaneFilter makes.
	//
	mPolish = false;
}


iCrossSectionParallelPipeline::~iCrossSectionParallelPipeline()
{
	mTextureData->Delete();
}


void iCrossSectionParallelPipeline::InitExecution(vtkImageData *input, vtkPolyData *)
{
	int Axis = mParent->GetDir();
	int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;

	double org[3], orgOut[3], spa[3], spaOut[3];
	int dims[3], dimsOut[3], oldDimsOut[3];

	input->GetOrigin(org);
	input->GetSpacing(spa);
	input->GetDimensions(dims);

	//
	//  Texture dimensions
	//
	orgOut[2] = org[Axis];
	orgOut[0] = org[Uidx];
	orgOut[1] = org[Vidx];

	spaOut[2] = spa[Axis];
	spaOut[0] = spa[Uidx];
	spaOut[1] = spa[Vidx];

	dimsOut[2] = 1;
	dimsOut[0] = dims[Uidx] - 1;
	dimsOut[1] = dims[Vidx] - 1;

	mTextureData->GetDimensions(oldDimsOut);
	if(oldDimsOut[0]!=dimsOut[0] || oldDimsOut[1]!=dimsOut[1] || oldDimsOut[2]!=dimsOut[2])
	{
		mTextureData->Initialize();
		mTextureData->SetDimensions(dimsOut);
		mTextureData->AllocateScalars();
	}
	mTextureData->SetOrigin(orgOut);
	mTextureData->SetSpacing(spaOut);

	//
	//  Assign texture offsets
	//
	if(mParent->GetMethod() == 1)
	{
		int i, Axis = mParent->GetDir();
		int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;

		iGridDataDistributor *md = dynamic_cast<iGridDataDistributor*>(mDistributor);
		if(md == 0)
		{
			vtkErrorMacro("Invalid data distributor");
			return;
		}

		int dims[3];
		this->GetInput()->GetDimensions(dims);

		if(Axis == md->GetSplitDimension())
		{
			for(i=0; i<mWorkPipelines.Size(); i++)
			{
				iPointerCast<iCrossSectionSerialPipeline,iGridDataToPolyDataPipeline>(mWorkPipelines[i])->SetTexturePiece(mTextureData,0,0);
			}
		}
		else
		{
			int offx = 0, offy = 0;
				
			for(i=0; i<mWorkPipelines.Size(); i++)
			{
				iPointerCast<iCrossSectionSerialPipeline,iGridDataToPolyDataPipeline>(mWorkPipelines[i])->SetTexturePiece(mTextureData,offx,offy);
				mWorkPipelines[i]->GetInput()->GetDimensions(dims);
				if(md->GetSplitDimension() == Uidx)
				{
					offx += dims[Uidx] - 1;
				}
				else offy += dims[Vidx] - 1;
			}
		}
	}
}


int iCrossSectionParallelPipeline::ParallelExecute(int step)
{
	return iParallelGridDataToPolyDataPipeline::ParallelExecute(step);
}


IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,Method);
IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,Dir);
IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,Var);
IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,Pos);
IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,ResampleFactor);
IPARALLELPIPELINE_DEFINE_UPDATE_FUNCTION(iCrossSection,GridDataToPolyData,InterpolateData);


void iCrossSectionParallelPipeline::Append()
{
	switch(mParent->GetMethod())
	{
	case 0:
		{
			//
			//  Append quad pieces
			//
			iParallelGridDataToPolyDataPipeline::Append();
			break;
		}
	case 1:
		{
			//
			//  Nothing to append in texture mode, the texture is filled automatically.
			//  Just need to create the quad on which the texture is projected.
			//
			this->CreateTextureSupport();
			break;
		}
	}
}


iGridDataToPolyDataPipeline* iCrossSectionParallelPipeline::CreateSerialPipeline()
{
	return mParent->CreateSerialPipeline();
}


float iCrossSectionParallelPipeline::GetMemorySize() const
{
	float s = iParallelGridDataToPolyDataPipeline::GetMemorySize();

	s += mTextureData->GetActualMemorySize();

	return s;
}


void iCrossSectionParallelPipeline::CreateTextureSupport()
{
	//
	//  Prepare texture support
	//
	int dims[3];
	double org[3], spa[3];

	this->GetInput()->GetOrigin(org);
	this->GetInput()->GetSpacing(spa);
	this->GetInput()->GetDimensions(dims);

	int Axis = mParent->GetDir();
	int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;
	float tc[2];
	static vtkIdType pts[4]={0,1,2,3};
	//
	// We'll create the building blocks of polydata including data attributes.
	//
	vtkPoints *points;
	points = vtkPoints::New(VTK_FLOAT); IERROR_ASSERT_NULL_POINTER(points);

	vtkCellArray *polys;
	polys = vtkCellArray::New(); IERROR_ASSERT_NULL_POINTER(polys);

	vtkFloatArray *tcoords;
	tcoords = vtkFloatArray::New(); IERROR_ASSERT_NULL_POINTER(tcoords);

	points->SetNumberOfPoints(4);
	tcoords->SetNumberOfComponents(2);
	tcoords->SetNumberOfTuples(4);
	//
	// Load the cell, and data attributes.
	//
	polys->InsertNextCell(4,pts);
	//
	//  Place the support plane
	//
	double x1[3];
	x1[Axis] = mParent->GetPos();
	//
	//  LL point 
	//
	x1[Uidx] = org[Uidx];
	x1[Vidx] = org[Vidx];
	points->SetPoint(0,x1);
	tc[0] = 0.0f;
	tc[1] = 0.0f;
	tcoords->SetTuple(0,tc);
	//
	//  LR point 
	//
	x1[Uidx] = org[Uidx] + spa[Uidx]*(dims[Uidx]-1);
	x1[Vidx] = org[Vidx];
	points->SetPoint(1,x1);
	tc[0] = 1.0f;
	tc[1] = 0.0f;
	tcoords->SetTuple(1,tc);
	//
	//  UR point 
	//
	x1[Uidx] = org[Uidx] + spa[Uidx]*(dims[Uidx]-1);
	x1[Vidx] = org[Vidx] + spa[Vidx]*(dims[Vidx]-1);
	points->SetPoint(2,x1);
	tc[0] = 1.0f;
	tc[1] = 1.0f;
	tcoords->SetTuple(2,tc);
	//
	//  UL point 
	//
	x1[Uidx] = org[Uidx];
	x1[Vidx] = org[Vidx] + spa[Vidx]*(dims[Vidx]-1);
	points->SetPoint(3,x1);
	tc[0] = 0.0f;
	tc[1] = 1.0f;
	tcoords->SetTuple(3,tc);
	//
	// We now assign the pieces to the vtkPolyData.
	//
	vtkPolyData *output = this->GetOutput();
	if(output != 0)
	{
		output->Initialize();
		output->SetPolys(polys);
		output->SetPoints(points);
		output->GetPointData()->SetTCoords(tcoords);
	}
	polys->Delete();
	points->Delete();
	tcoords->Delete();
}

