/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idataconsumer.h"


#include "idata.h"
#include "idatahandler.h"
#include "idatareader.h"
#include "ierror.h"
#include "iviewmodule.h"
#include "ivtk.h"


//
//  Templates
//
#include "iarraytemplate.h"


using namespace iParameter;


int iDataConsumer::mModeGlobal = _DataConsumerOptimizeForSpeed;


iDataConsumer::iDataConsumer(iViewModule *vm, const iDataInfo &info) : iViewModuleComponent(vm), mPrimaryDataInfo(new iDataInfo(info))
{
	IERROR_ASSERT_NULL_POINTER(mPrimaryDataInfo);
	mSecondaryDataInfo = new iDataInfo; IERROR_ASSERT_NULL_POINTER(mSecondaryDataInfo);
	mModeLocal = mModeGlobal;
	mOverrideGlobal = false;
	
	mActiveDataTypeIndex = 0;

	mViewModule->RegisterDataConsumer(this);
}


iDataConsumer::~iDataConsumer()
{
	mViewModule->UnRegisterDataConsumer(this);
	delete mPrimaryDataInfo;
	delete mSecondaryDataInfo;

	//
	//  Check that all DataHandlers have been deleted too. 
	//  Fails for VTK 5.0.0 - Garbage collection is more convoluted.
	//
	if(mDataHandlers.Size() > 0)
	{
#ifdef I_DEBUG
#ifdef IVTK_4
		IERROR_REPORT_ERROR("Not all DataHandlers have been deleted.");
#endif
#endif
	}
}


void iDataConsumer::AddSecondaryDataType(const iDataType &type)
{
	*mSecondaryDataInfo += type;
}


bool iDataConsumer::IsUsingData(const iDataType &type, bool onlyprimary) const
{
	if(onlyprimary)
	{
		return mPrimaryDataInfo->Includes(type);
	}
	else
	{
		return mPrimaryDataInfo->Includes(type) || mSecondaryDataInfo->Includes(type);
	}
}


bool iDataConsumer::IsDataPresent() const
{
	return this->GetViewModule()->GetReader()->IsThereData(this->GetActiveDataType());
}


void iDataConsumer::SetActiveDataType(const iDataType &type)
{
	this->SetActiveDataTypeIndex(mPrimaryDataInfo->Index(type));
}


const iDataType& iDataConsumer::GetActiveDataType() const
{
	return mPrimaryDataInfo->Type(mActiveDataTypeIndex);
}


bool iDataConsumer::IsActiveDataType(const iDataType &type) const
{
	return type == mPrimaryDataInfo->Type(mActiveDataTypeIndex);
}


void iDataConsumer::SetActiveDataTypeIndex(int v)
{
	if(v!=mActiveDataTypeIndex && v>=0 && v<mPrimaryDataInfo->Count())
	{
		mActiveDataTypeIndex = v;
		this->UpdateDataType();
		this->SyncWithData(0,0);
	}
}


vtkDataSet* iDataConsumer::GetData() const
{
	return this->GetViewModule()->GetReader()->GetOutput(this->GetActiveDataType());
}


void iDataConsumer::SetGlobalOptimizationMode(int mode)
{
	if(mode>=0 && mode<__NumDataConsumerOptimizationModes)
	{
		mModeGlobal = mode;
	}
}


void iDataConsumer::SetOptimizationMode(int mode)
{
	//
	//  Reset the global mode;
	//
	if(mode>=0 && mode<__NumDataConsumerOptimizationModes)
	{
		mModeLocal = mode;
		mOverrideGlobal = true;
	}
	else if(mode==_DataConsumerResetToGlobal)
	{
		mModeLocal = mModeGlobal;
		mOverrideGlobal = false;
	}
}


int iDataConsumer::GetOptimizationMode() const 
{
	if(mOverrideGlobal) return mModeLocal; else return mModeGlobal;
}


//
//  DataHandler registry functionality
//
void iDataConsumer::RegisterDataHandler(iDataHandler *c)
{
	mDataHandlers.AddUnique(c);
}


void iDataConsumer::UnRegisterDataHandler(iDataHandler *c)
{
	mDataHandlers.Remove(c);
}


void iDataConsumer::SyncWithData(int info, void *data)
{
	int i;
	for(i=0; i<mDataHandlers.Size(); i++) mDataHandlers[i]->SyncWithData(info,data);
	this->SyncWithDataBody(info,data);
}

