/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idatadistributor.h"


#include "iarray.h"
#include "ierror.h"
#include "ireplicatedpolydata.h"
#include "iviewobject.h"
#include "iviewobjectreplicator.h"

#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkImageData.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkStructuredPoints.h>

//
//  Templates
//
#include "iarraytemplate.h"


//
//  iGridDataDistributor class
//
iGridDataDistributor::iGridDataDistributor()
{
	mDim = -1;
	mNumEdges = 0;  
	mEdges = 0;
}


iGridDataDistributor::~iGridDataDistributor()
{
	if(mEdges != 0) delete [] mEdges;
}


void iGridDataDistributor::DistributeData(vtkImageData *input, iArray<vtkImageData*> &parallelInputs)
{
	int i, j, nc;
	//
	//  Default implementation: split the data in uniform chunks along the last non-trivial dimensions
	//
	if(input == 0) return;

	//
	//  Read input structure and choose the splitted direction
	//
    int dims[3], dimsPiece[3];
    double org[3], orgPiece[3], spa[3];

    input->GetOrigin(org);
    input->GetSpacing(spa);
    input->GetDimensions(dims);

	mDim = 2;
	while(mDim>=0 && dims[mDim]<=1) mDim--;

	//
	//  Set the number of processors
	//
	mNumEdges = parallelInputs.Size() - 1;
	if(mNumEdges >= dims[mDim]) mNumEdges = dims[mDim] - 1;

	if(mNumEdges > 0)
	{
		mEdges = new double[mNumEdges]; IERROR_ASSERT_NULL_POINTER(mEdges);
	}
	else
	{
		//
		//  Nothing to do for 1 proc
		//
		parallelInputs[0]->ShallowCopy(input);
		return;
	}

	//
	//  Split the data between processors
	//
	int at, kstp, kbeg, kend;
	
	for(i=0; i<3; i++)
	{
		dimsPiece[i] = dims[i];
		orgPiece[i] = org[i];
	}
		
	kstp = (dims[mDim]+mNumEdges-1)/(mNumEdges+1); //  This is point data, hence -2 rather than -1 in kstp
	long dataOffset = 1L;
	for(i=0; i<mDim; i++) dataOffset *= dims[i];
	
	vtkFloatArray *array;

	for(i=0; i<=mNumEdges; i++)
	{
		kbeg = kstp*i;
		kend = kbeg + kstp + 1;  //  This is point data, so the array boundaries must overlap, hence +1 in kend
		if(i == mNumEdges) kend = dims[mDim]; else
		{
			mEdges[i] = org[mDim] + spa[mDim]*(kend-1);
		}
		
		dimsPiece[mDim] = kend - kbeg;
		orgPiece[mDim] = org[mDim] + spa[mDim]*kbeg;
		
		parallelInputs[i]->Initialize();
		parallelInputs[i]->SetDimensions(dimsPiece);
		parallelInputs[i]->SetOrigin(orgPiece);
		parallelInputs[i]->SetSpacing(spa);
		parallelInputs[i]->GetPointData()->CopyStructure(input->GetPointData());

		for(j=0; j<input->GetPointData()->GetNumberOfArrays(); j++)
		{
			nc = input->GetPointData()->GetArray(j)->GetNumberOfComponents();
			array = dynamic_cast<vtkFloatArray*>(parallelInputs[i]->GetPointData()->GetArray(j));
			if(array!=0 && nc>0)
			{
				if((at=input->GetPointData()->IsArrayAnAttribute(j)) != -1)
				{
					//
					// If this array is an attribute in the source, make it so
					// in the target as well - CopyStructure does not copy attributes
					//
					parallelInputs[i]->GetPointData()->SetActiveAttribute(j,at);
				}
				array->SetArray((float *)input->GetPointData()->GetArray(j)->GetVoidPointer(0)+nc*dataOffset*kbeg,(long)nc*(long)dimsPiece[0]*(long)dimsPiece[1]*(long)dimsPiece[2],1);
				array->SetName(input->GetPointData()->GetArray(j)->GetName());
			}
		}
	}
}


void iGridDataDistributor::PolishOutput(iViewObject *vo, vtkPolyData *output)
{
	int i, j;
	int extDown = 0;
	int next = 1;
	if(vo->GetReplicator() != 0)
	{
		extDown = vo->GetReplicator()->GetDataReplicated()->GetNumReplicas(2*mDim);
		next += vo->GetReplicator()->GetDataReplicated()->GetNumReplicas(2*mDim+1) + extDown;
	}

	if(mNumEdges > 0) 
	{
		int ntot = mNumEdges*next;
		double *edges = new double[ntot];
		if(edges == 0) return;

		for(j=0; j<next; j++)
		{
			for(i=0; i<mNumEdges; i++)
			{
				edges[i+mNumEdges*j] = mEdges[i] + 2*(j-extDown);
			}
		}
		
		iReplicatedPolyData::PatchStitches(output,mDim,edges,ntot);

		delete [] edges;
	}
}


void iGridDataDistributor::PolishOutput(iViewObject *, vtkStructuredPoints *)
{
	//
	//  Nothing to polish here
	//
}


//
//  iPolyDataDistributor class
//
iPolyDataDistributor::iPolyDataDistributor()
{
}


iPolyDataDistributor::~iPolyDataDistributor()
{
}


void iPolyDataDistributor::DistributeData(vtkPolyData *input, iArray<vtkPolyData*> &parallelInputs)
{
	int i;
	//
	//  Default implementation: split the data in uniform chunks by the particle index.
	//  Only split Vert data, assign Line, Poly, and Strip data to the master thread
	//
	if(input == 0) return;

	int nPieces = parallelInputs.Size();
	vtkIdType nParticles = input->GetNumberOfPoints();

	if(nPieces > nParticles) nPieces = nParticles;
	if(nPieces < 1) nPieces = 1;
	vtkIdType kbeg, kstp = (nParticles+nPieces-1)/nPieces;

	vtkIdType *inPtr = input->GetVerts()->GetData()->GetPointer(0);
	vtkCellArray *tmpCells;
	vtkIdTypeArray *tmpData;

	for(i=0; i<nPieces; i++)
	{
		kbeg = kstp*i;
		if(i == nPieces-1) kstp = nParticles - kbeg;
		
		parallelInputs[i]->Initialize();
		parallelInputs[i]->ShallowCopy(input);
		if(i > 0)
		{
			parallelInputs[i]->SetVerts(0);
			parallelInputs[i]->SetLines(0);
			parallelInputs[i]->SetPolys(0);
			parallelInputs[i]->SetStrips(0);
		}

		tmpCells = vtkCellArray::New();
		if(tmpCells != 0)
		{
			tmpData = vtkIdTypeArray::New();
			if(tmpData != 0)
			{
				tmpData->SetArray(inPtr+2*kbeg,2*kstp,1);
				tmpCells->SetCells(kstp,tmpData);
				parallelInputs[i]->SetVerts(tmpCells);
				tmpData->Delete();
			}
			tmpCells->Delete();
		}
	}

	for(i=nPieces; i<parallelInputs.Size(); i++)
	{
		parallelInputs[i]->Initialize();
	}
}


void iPolyDataDistributor::PolishOutput(iViewObject* /*vo*/, vtkPolyData* /*output*/)
{
	//
	//  Nothing to polish here
	//
}


void iPolyDataDistributor::PolishOutput(iViewObject *, vtkStructuredPoints *)
{
	//
	//  Nothing to polish here
	//
}
